/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.index.schema;

import java.io.Closeable;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.CopyOption;
import java.nio.file.Path;
import java.util.Comparator;
import java.util.function.IntConsumer;
import org.eclipse.collections.api.list.MutableList;
import org.eclipse.collections.impl.factory.Lists;
import org.neo4j.index.internal.gbptree.Layout;
import org.neo4j.io.IOUtils;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.StoreChannel;
import org.neo4j.io.memory.ByteBufferFactory;
import org.neo4j.io.memory.ScopedBuffer;
import org.neo4j.io.pagecache.ByteArrayPageCursor;
import org.neo4j.io.pagecache.PageCursor;
import org.neo4j.kernel.impl.index.schema.BlockEntry;
import org.neo4j.kernel.impl.index.schema.BlockEntryCursor;
import org.neo4j.kernel.impl.index.schema.BlockEntryReader;
import org.neo4j.kernel.impl.index.schema.BlockReader;
import org.neo4j.kernel.impl.index.schema.ListBasedBlockEntryCursor;
import org.neo4j.kernel.impl.index.schema.MergingBlockEntryReader;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.util.Preconditions;

class BlockStorage<KEY, VALUE>
implements Closeable {
    static final int BLOCK_HEADER_SIZE = 16;
    private final Layout<KEY, VALUE> layout;
    private final FileSystemAbstraction fs;
    private final MutableList<BlockEntry<KEY, VALUE>> bufferedEntries;
    private final Comparator<BlockEntry<KEY, VALUE>> comparator;
    private final StoreChannel storeChannel;
    private final Monitor monitor;
    private final int blockSize;
    private final MemoryTracker memoryTracker;
    private final ByteBufferFactory bufferFactory;
    private final Path blockFile;
    private long numberOfBlocksInCurrentFile;
    private int currentBufferSize;
    private boolean doneAdding;
    private long entryCount;

    BlockStorage(Layout<KEY, VALUE> layout, ByteBufferFactory bufferFactory, FileSystemAbstraction fs, Path blockFile, Monitor monitor, MemoryTracker memoryTracker) throws IOException {
        this.layout = layout;
        this.fs = fs;
        this.blockFile = blockFile;
        this.monitor = monitor;
        this.blockSize = bufferFactory.bufferSize();
        this.memoryTracker = memoryTracker;
        this.bufferedEntries = Lists.mutable.empty();
        this.bufferFactory = bufferFactory;
        this.comparator = (e0, e1) -> layout.compare(e0.key(), e1.key());
        this.storeChannel = fs.write(blockFile);
        this.resetBufferedEntries();
    }

    public void add(KEY key, VALUE value) throws IOException {
        Preconditions.checkState((!this.doneAdding ? 1 : 0) != 0, (String)"Cannot add more after done adding");
        int entrySize = BlockEntry.entrySize(this.layout, key, value);
        if (this.currentBufferSize + entrySize > this.blockSize) {
            this.flushAndResetBuffer();
            ++this.numberOfBlocksInCurrentFile;
        }
        this.bufferedEntries.add(new BlockEntry<KEY, VALUE>(key, value));
        this.currentBufferSize += entrySize;
        this.monitor.entryAdded(entrySize);
    }

    void doneAdding() throws IOException {
        if (!this.bufferedEntries.isEmpty()) {
            this.flushAndResetBuffer();
            ++this.numberOfBlocksInCurrentFile;
        }
        this.doneAdding = true;
        this.storeChannel.close();
    }

    private void resetBufferedEntries() {
        this.bufferedEntries.clear();
        this.currentBufferSize = 16;
    }

    private void flushAndResetBuffer() throws IOException {
        this.bufferedEntries.sortThis(this.comparator);
        ListBasedBlockEntryCursor<KEY, VALUE> entries = new ListBasedBlockEntryCursor<KEY, VALUE>(this.bufferedEntries);
        ByteBuffer byteBuffer = this.bufferFactory.acquireThreadLocalBuffer(this.memoryTracker);
        try {
            this.writeBlock(this.storeChannel, entries, this.blockSize, this.bufferedEntries.size(), Cancellation.NOT_CANCELLABLE, count -> this.entryCount += (long)count, byteBuffer);
        }
        finally {
            this.bufferFactory.releaseThreadLocalBuffer();
        }
        this.monitor.blockFlushed(this.bufferedEntries.size(), this.currentBufferSize, this.storeChannel.position());
        this.resetBufferedEntries();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void merge(int mergeFactor, Cancellation cancellation) throws IOException {
        Path tempFile;
        this.monitor.mergeStarted(this.entryCount, BlockStorage.calculateNumberOfEntriesWrittenDuringMerges(this.entryCount, this.numberOfBlocksInCurrentFile, mergeFactor));
        Path sourceFile = this.blockFile;
        Path targetFile = tempFile = this.blockFile.resolveSibling(String.valueOf(this.blockFile.getFileName()) + ".b");
        int bufferSize = this.bufferFactory.bufferSize();
        try (ByteBufferFactory.Allocator mergeBufferAllocator = this.bufferFactory.newLocalAllocator();
             ScopedBuffer writeBuffer = mergeBufferAllocator.allocate(bufferSize, this.memoryTracker);
             CompositeScopedBuffer readBuffers = new CompositeScopedBuffer(mergeFactor, bufferSize, mergeBufferAllocator, this.memoryTracker);){
            while (this.numberOfBlocksInCurrentFile > 1L) {
                try (BlockReader<KEY, VALUE> reader = this.reader(sourceFile, false);
                     StoreChannel targetChannel = this.fs.write(targetFile);){
                    long blocksMergedSoFar = 0L;
                    long blocksInMergedFile = 0L;
                    while (!cancellation.cancelled() && blocksMergedSoFar < this.numberOfBlocksInCurrentFile) {
                        blocksMergedSoFar += (long)this.performSingleMerge(mergeFactor, reader, targetChannel, cancellation, readBuffers.getScopedBuffers(), writeBuffer.getBuffer());
                        ++blocksInMergedFile;
                    }
                    this.numberOfBlocksInCurrentFile = blocksInMergedFile;
                    this.monitor.mergeIterationFinished(blocksMergedSoFar, blocksInMergedFile);
                }
                Path tmpSourceFile = sourceFile;
                sourceFile = targetFile;
                targetFile = tmpSourceFile;
            }
        }
        finally {
            if (sourceFile == this.blockFile) {
                if (this.fs.fileExists(tempFile)) {
                    this.fs.deleteFile(tempFile);
                }
            } else {
                this.fs.deleteFile(this.blockFile);
                this.fs.renameFile(tempFile, this.blockFile, new CopyOption[0]);
            }
        }
    }

    static long calculateNumberOfEntriesWrittenDuringMerges(long entryCount, long numberOfBlocks, int mergeFactor) {
        int singleMerges = 0;
        long blocks = numberOfBlocks;
        while (blocks > 1L) {
            ++singleMerges;
            blocks = (long)Math.ceil((double)blocks / (double)mergeFactor);
        }
        return (long)singleMerges * entryCount;
    }

    private int performSingleMerge(int mergeFactor, BlockReader<KEY, VALUE> reader, StoreChannel targetChannel, Cancellation cancellation, ScopedBuffer[] readBuffers, ByteBuffer writeBuffer) throws IOException {
        try (MergingBlockEntryReader<KEY, VALUE> merger = new MergingBlockEntryReader<KEY, VALUE>(this.layout);){
            long blockSize = 0L;
            long entryCount = 0L;
            int blocksMerged = 0;
            for (int i = 0; i < mergeFactor; ++i) {
                readBuffers[i].getBuffer().clear();
                BlockEntryReader<KEY, VALUE> source = reader.nextBlock(readBuffers[i]);
                if (source == null) break;
                blockSize += source.blockSize();
                entryCount += source.entryCount();
                ++blocksMerged;
                merger.addSource(source);
            }
            writeBuffer.clear();
            this.writeBlock(targetChannel, merger, blockSize, entryCount, cancellation, this.monitor::entriesMerged, writeBuffer);
            this.monitor.mergedBlocks(blockSize, entryCount, blocksMerged);
            int n = blocksMerged;
            return n;
        }
    }

    private void writeBlock(StoreChannel targetChannel, BlockEntryCursor<KEY, VALUE> blockEntryCursor, long blockSize, long entryCount, Cancellation cancellation, IntConsumer entryCountReporter, ByteBuffer byteBuffer) throws IOException {
        BlockStorage.writeHeader(byteBuffer, blockSize, entryCount);
        long actualDataSize = BlockStorage.writeEntries(targetChannel, byteBuffer, this.layout, blockEntryCursor, cancellation, entryCountReporter);
        BlockStorage.writeLastEntriesWithPadding(targetChannel, byteBuffer, blockSize - actualDataSize);
    }

    private static void writeHeader(ByteBuffer byteBuffer, long blockSize, long entryCount) {
        byteBuffer.putLong(blockSize);
        byteBuffer.putLong(entryCount);
    }

    private static <KEY, VALUE> long writeEntries(StoreChannel targetChannel, ByteBuffer byteBuffer, Layout<KEY, VALUE> layout, BlockEntryCursor<KEY, VALUE> blockEntryCursor, Cancellation cancellation, IntConsumer entryCountReporter) throws IOException {
        long actualDataSize = 16L;
        ByteArrayPageCursor pageCursor = new ByteArrayPageCursor(byteBuffer);
        int entryCountToReport = 0;
        while (blockEntryCursor.next()) {
            KEY key = blockEntryCursor.key();
            VALUE value = blockEntryCursor.value();
            int entrySize = BlockEntry.entrySize(layout, key, value);
            actualDataSize += (long)entrySize;
            ++entryCountToReport;
            if (byteBuffer.remaining() < entrySize) {
                if (cancellation.cancelled()) break;
                byteBuffer.flip();
                targetChannel.writeAll(byteBuffer);
                byteBuffer.clear();
                entryCountReporter.accept(entryCountToReport);
                entryCountToReport = 0;
            }
            BlockEntry.write((PageCursor)pageCursor, layout, key, value);
        }
        if (entryCountToReport > 0) {
            entryCountReporter.accept(entryCountToReport);
        }
        return actualDataSize;
    }

    private static void writeLastEntriesWithPadding(StoreChannel channel, ByteBuffer byteBuffer, long padding) throws IOException {
        boolean didWrite;
        do {
            int toPadThisTime = (int)Math.min((long)byteBuffer.remaining(), padding);
            byte[] padArray = new byte[toPadThisTime];
            byteBuffer.put(padArray);
            padding -= (long)toPadThisTime;
            boolean bl = didWrite = byteBuffer.position() > 0;
            if (!didWrite) continue;
            byteBuffer.flip();
            channel.writeAll(byteBuffer);
            byteBuffer.clear();
        } while (didWrite);
    }

    @Override
    public void close() throws IOException {
        IOUtils.closeAll((AutoCloseable[])new AutoCloseable[]{this.storeChannel});
        if (this.fs.fileExists(this.blockFile)) {
            this.fs.deleteFile(this.blockFile);
        }
    }

    BlockReader<KEY, VALUE> reader(boolean produceNewKeyAndValueInstances) throws IOException {
        return this.reader(this.blockFile, produceNewKeyAndValueInstances);
    }

    private BlockReader<KEY, VALUE> reader(Path file, boolean produceNewKeyAndValueInstances) throws IOException {
        return new BlockReader<KEY, VALUE>(this.fs, file, this.layout, produceNewKeyAndValueInstances);
    }

    public static interface Monitor {
        public static final Monitor NO_MONITOR = new Adapter();

        public void entryAdded(int var1);

        public void blockFlushed(long var1, int var3, long var4);

        public void mergeStarted(long var1, long var3);

        public void entriesMerged(int var1);

        public void mergeIterationFinished(long var1, long var3);

        public void mergedBlocks(long var1, long var3, long var5);

        public static class Adapter
        implements Monitor {
            @Override
            public void entryAdded(int entrySize) {
            }

            @Override
            public void blockFlushed(long keyCount, int numberOfBytes, long positionAfterFlush) {
            }

            @Override
            public void mergeStarted(long entryCount, long totalEntriesToWriteDuringMerge) {
            }

            @Override
            public void entriesMerged(int entries) {
            }

            @Override
            public void mergeIterationFinished(long numberOfBlocksBefore, long numberOfBlocksAfter) {
            }

            @Override
            public void mergedBlocks(long resultingBlockSize, long resultingEntryCount, long numberOfBlocks) {
            }
        }

        public static class Delegate
        implements Monitor {
            private final Monitor actual;

            Delegate(Monitor actual) {
                this.actual = actual;
            }

            @Override
            public void entryAdded(int entrySize) {
                this.actual.entryAdded(entrySize);
            }

            @Override
            public void blockFlushed(long keyCount, int numberOfBytes, long positionAfterFlush) {
                this.actual.blockFlushed(keyCount, numberOfBytes, positionAfterFlush);
            }

            @Override
            public void mergeStarted(long entryCount, long totalEntriesToWriteDuringMerge) {
                this.actual.mergeStarted(entryCount, totalEntriesToWriteDuringMerge);
            }

            @Override
            public void entriesMerged(int entries) {
                this.actual.entriesMerged(entries);
            }

            @Override
            public void mergeIterationFinished(long numberOfBlocksBefore, long numberOfBlocksAfter) {
                this.actual.mergeIterationFinished(numberOfBlocksBefore, numberOfBlocksAfter);
            }

            @Override
            public void mergedBlocks(long resultingBlockSize, long resultingEntryCount, long numberOfBlocks) {
                this.actual.mergedBlocks(resultingBlockSize, resultingEntryCount, numberOfBlocks);
            }
        }
    }

    @FunctionalInterface
    public static interface Cancellation {
        public static final Cancellation NOT_CANCELLABLE = () -> false;

        public boolean cancelled();
    }

    private static class CompositeScopedBuffer
    implements AutoCloseable {
        private final ScopedBuffer[] scopedBuffers;

        CompositeScopedBuffer(int numberOfBuffers, int bufferSize, ByteBufferFactory.Allocator allocator, MemoryTracker memoryTracker) {
            this.scopedBuffers = new ScopedBuffer[numberOfBuffers];
            for (int i = 0; i < this.scopedBuffers.length; ++i) {
                this.scopedBuffers[i] = allocator.allocate(bufferSize, memoryTracker);
            }
        }

        public ScopedBuffer[] getScopedBuffers() {
            return this.scopedBuffers;
        }

        @Override
        public void close() {
            IOUtils.closeAllSilently((AutoCloseable[])this.scopedBuffers);
        }
    }
}

