/* -*- Mode: java; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

package org.mozilla.javascript;

public class NativeSet extends ScriptableObject {
    private static final long serialVersionUID = -8442212766987072986L;
    private static final String CLASS_NAME = "Set";
    static final String ITERATOR_TAG = "Set Iterator";

    static final SymbolKey GETSIZE = new SymbolKey("[Symbol.getSize]", Symbol.Kind.REGULAR);

    private final Hashtable entries = new Hashtable();

    private boolean instanceOfSet = false;

    static Object init(Context cx, Scriptable scope, boolean sealed) {
        LambdaConstructor constructor =
                new LambdaConstructor(
                        scope,
                        CLASS_NAME,
                        0,
                        LambdaConstructor.CONSTRUCTOR_NEW,
                        NativeSet::jsConstructor);
        constructor.setPrototypePropertyAttributes(DONTENUM | READONLY | PERMANENT);

        var propAttrs = DONTENUM | READONLY;
        constructor.definePrototypeMethod(scope, "add", 1, NativeSet::js_add, DONTENUM, propAttrs);
        constructor.definePrototypeMethod(
                scope, "delete", 1, NativeSet::js_delete, DONTENUM, propAttrs);
        constructor.definePrototypeMethod(scope, "has", 1, NativeSet::js_has, DONTENUM, propAttrs);
        constructor.definePrototypeMethod(
                scope, "clear", 0, NativeSet::js_clear, DONTENUM, propAttrs);
        constructor.definePrototypeMethod(
                scope, "values", 0, NativeSet::js_values, DONTENUM, propAttrs);
        constructor.definePrototypeAlias("values", "keys", propAttrs);
        constructor.definePrototypeAlias("values", SymbolKey.ITERATOR, DONTENUM);

        constructor.definePrototypeMethod(
                scope, "forEach", 1, NativeSet::js_forEach, DONTENUM, propAttrs);

        constructor.definePrototypeMethod(
                scope, "entries", 0, NativeSet::js_entries, DONTENUM, propAttrs);

        // ES2025 Set methods
        constructor.definePrototypeMethod(
                scope, "intersection", 1, NativeSet::js_intersection, DONTENUM, propAttrs);
        constructor.definePrototypeMethod(
                scope, "union", 1, NativeSet::js_union, DONTENUM, propAttrs);
        constructor.definePrototypeMethod(
                scope, "difference", 1, NativeSet::js_difference, DONTENUM, propAttrs);
        constructor.definePrototypeMethod(
                scope,
                "symmetricDifference",
                1,
                NativeSet::js_symmetricDifference,
                DONTENUM,
                propAttrs);
        constructor.definePrototypeMethod(
                scope, "isSubsetOf", 1, NativeSet::js_isSubsetOf, DONTENUM, propAttrs);
        constructor.definePrototypeMethod(
                scope, "isSupersetOf", 1, NativeSet::js_isSupersetOf, DONTENUM, propAttrs);
        constructor.definePrototypeMethod(
                scope, "isDisjointFrom", 1, NativeSet::js_isDisjointFrom, DONTENUM, propAttrs);

        // The spec requires very specific handling of the "size" prototype
        // property that's not like other things that we already do.
        ScriptableObject desc = (ScriptableObject) cx.newObject(scope);
        desc.put("enumerable", desc, Boolean.FALSE);
        desc.put("configurable", desc, Boolean.TRUE);
        LambdaFunction sizeFunc = new LambdaFunction(scope, "get size", 0, NativeSet::js_getSize);
        sizeFunc.setPrototypeProperty(Undefined.instance);
        desc.put("get", desc, sizeFunc);
        constructor.definePrototypeProperty(cx, "size", desc);
        constructor.definePrototypeProperty(cx, NativeSet.GETSIZE, desc);

        constructor.definePrototypeProperty(
                SymbolKey.TO_STRING_TAG, CLASS_NAME, DONTENUM | READONLY);

        ScriptRuntimeES6.addSymbolSpecies(cx, scope, constructor);
        if (sealed) {
            constructor.sealObject();
            ((ScriptableObject) constructor.getPrototypeProperty()).sealObject();
        }
        return constructor;
    }

    @Override
    public String getClassName() {
        return CLASS_NAME;
    }

    private static Scriptable jsConstructor(Context cx, Scriptable scope, Object[] args) {
        NativeSet ns = new NativeSet();
        ns.instanceOfSet = true;
        if (args.length > 0) {
            loadFromIterable(cx, scope, ns, NativeMap.key(args));
        }
        return ns;
    }

    private static Object js_add(Context cx, Scriptable scope, Scriptable thisObj, Object[] args) {
        NativeSet realThis = realThis(thisObj, "add");
        var k = NativeMap.key(args);
        return realThis.js_add(k);
    }

    private Object js_add(Object k) {
        // Special handling of "negative zero" from the spec.
        if ((k instanceof Number) && ((Number) k).doubleValue() == ScriptRuntime.negativeZero) {
            entries.put(ScriptRuntime.zeroObj, ScriptRuntime.zeroObj);
            return this;
        }
        entries.put(k, k);
        return this;
    }

    private static Object js_delete(
            Context cx, Scriptable scope, Scriptable thisObj, Object[] args) {
        NativeSet realThis = realThis(thisObj, "add");
        var arg = NativeMap.key(args);
        return realThis.js_delete(arg);
    }

    private Object js_delete(Object arg) {
        return entries.deleteEntry(arg);
    }

    private static Object js_has(Context cx, Scriptable scope, Scriptable thisObj, Object[] args) {
        NativeSet realThis = realThis(thisObj, "add");
        var arg = NativeMap.key(args);
        return realThis.js_has(arg);
    }

    private Object js_has(Object arg) {
        // Special handling of "negative zero" from the spec.
        if ((arg instanceof Number) && ((Number) arg).doubleValue() == ScriptRuntime.negativeZero) {
            return entries.has(ScriptRuntime.zeroObj);
        }
        return entries.has(arg);
    }

    private static Object js_clear(
            Context cx, Scriptable scope, Scriptable thisObj, Object[] args) {
        NativeSet realThis = realThis(thisObj, "add");
        return realThis.js_clear();
    }

    private Object js_clear() {
        entries.clear();
        return Undefined.instance;
    }

    private static Object js_getSize(
            Context cx, Scriptable scope, Scriptable thisObj, Object[] args) {
        NativeSet realThis = realThis(thisObj, "add");
        return realThis.js_getSize();
    }

    private Object js_getSize() {
        return entries.size();
    }

    private static Object js_values(
            Context cx, Scriptable scope, Scriptable thisObj, Object[] args) {
        NativeSet realThis = realThis(thisObj, "add");
        return realThis(thisObj, "values").js_iterator(scope, NativeCollectionIterator.Type.VALUES);
    }

    private static Object js_entries(
            Context cx, Scriptable scope, Scriptable thisObj, Object[] args) {
        NativeSet realThis = realThis(thisObj, "add");
        return realThis(thisObj, "values").js_iterator(scope, NativeCollectionIterator.Type.BOTH);
    }

    private Object js_iterator(Scriptable scope, NativeCollectionIterator.Type type) {
        return new NativeCollectionIterator(scope, ITERATOR_TAG, type, entries.iterator());
    }

    private static Object js_forEach(
            Context cx, Scriptable scope, Scriptable thisObj, Object[] args) {
        return realThis(thisObj, "forEach")
                .js_forEach(
                        cx,
                        scope,
                        NativeMap.key(args),
                        args.length > 1 ? args[1] : Undefined.instance);
    }

    private Object js_forEach(Context cx, Scriptable scope, Object arg1, Object arg2) {
        if (!(arg1 instanceof Callable)) {
            throw ScriptRuntime.notFunctionError(arg1);
        }
        final Callable f = (Callable) arg1;

        boolean isStrict = cx.isStrictMode();
        for (Hashtable.Entry entry : entries) {
            // Per spec must convert every time so that primitives are always regenerated...
            Scriptable thisObj = ScriptRuntime.toObjectOrNull(cx, arg2, scope);

            if (thisObj == null && !isStrict) {
                thisObj = scope;
            }
            if (thisObj == null) {
                thisObj = Undefined.SCRIPTABLE_UNDEFINED;
            }

            final Hashtable.Entry e = entry;
            f.call(cx, scope, thisObj, new Object[] {e.value, e.value, this});
        }
        return Undefined.instance;
    }

    /**
     * If an "iterable" object was passed to the constructor, there are many many things to do. This
     * is common code with NativeWeakSet.
     */
    static void loadFromIterable(Context cx, Scriptable scope, ScriptableObject set, Object arg1) {
        if ((arg1 == null) || Undefined.instance.equals(arg1)) {
            return;
        }

        // Call the "[Symbol.iterator]" property as a function.
        Object ito = ScriptRuntime.callIterator(arg1, cx, scope);
        if (Undefined.instance.equals(ito)) {
            // Per spec, ignore if the iterator returns undefined
            return;
        }

        // Find the "add" function of our own prototype, since it might have
        // been replaced. Since we're not fully constructed yet, create a dummy instance
        // so that we can get our own prototype.
        ScriptableObject dummy = ensureScriptableObject(cx.newObject(scope, set.getClassName()));
        var addCall = ScriptRuntime.getPropAndThis(dummy.getPrototype(), "add", cx, scope);
        Callable add = addCall.getCallable();

        // Finally, run through all the iterated values and add them!
        try (IteratorLikeIterable it = new IteratorLikeIterable(cx, scope, ito)) {
            for (Object val : it) {
                final Object finalVal = val == Scriptable.NOT_FOUND ? Undefined.instance : val;
                add.call(cx, scope, set, new Object[] {finalVal});
            }
        }
    }

    private static NativeSet realThis(Scriptable thisObj, String name) {
        NativeSet ns = LambdaConstructor.convertThisObject(thisObj, NativeSet.class);
        if (!ns.instanceOfSet) {
            // If we get here, then this object doesn't have the "Set internal data slot."
            throw ScriptRuntime.typeErrorById("msg.incompat.call", name);
        }
        return ns;
    }

    // ES2025 Set Methods Implementation

    private static Object js_intersection(
            Context cx, Scriptable scope, Scriptable thisObj, Object[] args) {
        return realThis(thisObj, "intersection").js_intersection(cx, scope, args);
    }

    private Object js_intersection(Context cx, Scriptable scope, Object[] args) {
        Object otherObj = args.length > 0 ? args[0] : Undefined.instance;

        NativeSet result = (NativeSet) cx.newObject(scope, CLASS_NAME);
        result.instanceOfSet = true;

        // ES2025: GetSetRecord requires size, has, and keys properties
        Scriptable scriptable = ScriptableObject.ensureScriptable(otherObj);
        Object sizeVal = ScriptableObject.getProperty(scriptable, "size");
        Object hasVal = ScriptableObject.getProperty(scriptable, "has");
        Object keysVal = ScriptableObject.getProperty(scriptable, "keys");

        // Validate all required properties exist
        validateSetLike(sizeVal, hasVal, keysVal);

        return js_intersectionSetLike(cx, scope, otherObj, result, sizeVal, hasVal, keysVal);
    }

    private Object js_intersectionSetLike(
            Context cx,
            Scriptable scope,
            Object otherObj,
            NativeSet result,
            Object sizeVal,
            Object hasVal,
            Object keysVal) {
        // Validate has and keys are callable
        if (!(hasVal instanceof Callable)) {
            throw ScriptRuntime.typeErrorById(
                    "msg.isnt.function", "has", ScriptRuntime.typeof(hasVal));
        }
        if (!(keysVal instanceof Callable)) {
            throw ScriptRuntime.typeErrorById(
                    "msg.isnt.function", "keys", ScriptRuntime.typeof(keysVal));
        }

        Callable hasMethod = (Callable) hasVal;
        Callable keysMethod = (Callable) keysVal;

        // ES2025: Compare sizes to determine iteration strategy
        double otherSizeDouble = ScriptRuntime.toNumber(sizeVal);
        if (Double.isNaN(otherSizeDouble)) {
            throw ScriptRuntime.typeError("size is not a number");
        }
        int otherSize =
                Double.isInfinite(otherSizeDouble)
                        ? Integer.MAX_VALUE
                        : (int) Math.floor(otherSizeDouble);
        int thisSize = entries.size();

        if (thisSize <= otherSize) {
            // When this.size <= other.size: iterate through this, call other.has()
            for (Hashtable.Entry entry : entries) {
                Object key = entry.key;
                Object inOther = callHas(cx, scope, otherObj, hasMethod, key);
                if (ScriptRuntime.toBoolean(inOther)) {
                    result.js_add(key);
                }
            }
        } else {
            // When this.size > other.size: iterate through other.keys(), call this.has()
            Object iterator =
                    ScriptRuntime.callIterator(
                            keysMethod.call(
                                    cx,
                                    scope,
                                    ScriptableObject.ensureScriptable(otherObj),
                                    ScriptRuntime.emptyArgs),
                            cx,
                            scope);
            try (IteratorLikeIterable it = new IteratorLikeIterable(cx, scope, iterator)) {
                for (Object key : it) {
                    if (js_has(key) == Boolean.TRUE) {
                        result.js_add(key);
                    }
                }
            }
        }

        return result;
    }

    private static Object js_union(
            Context cx, Scriptable scope, Scriptable thisObj, Object[] args) {
        return realThis(thisObj, "union").js_union(cx, scope, args);
    }

    private Object js_union(Context cx, Scriptable scope, Object[] args) {
        Object otherObj = args.length > 0 ? args[0] : Undefined.instance;

        NativeSet result = (NativeSet) cx.newObject(scope, CLASS_NAME);
        result.instanceOfSet = true;

        // Add all elements from this set
        for (Hashtable.Entry entry : entries) {
            result.js_add(entry.key);
        }

        // ES2025: GetSetRecord requires size, has, and keys properties
        Scriptable scriptable = ScriptableObject.ensureScriptable(otherObj);
        Object sizeVal = ScriptableObject.getProperty(scriptable, "size");
        Object hasVal = ScriptableObject.getProperty(scriptable, "has");
        Object keysVal = ScriptableObject.getProperty(scriptable, "keys");

        // Validate all required properties exist
        validateSetLike(sizeVal, hasVal, keysVal);

        // Validate size is a number (required by GetSetRecord)
        double otherSizeDouble = ScriptRuntime.toNumber(sizeVal);
        if (Double.isNaN(otherSizeDouble)) {
            throw ScriptRuntime.typeError("size is not a number");
        }

        // Validate has and keys are callable (GetSetRecord requires both even if union only uses
        // keys)
        if (!(hasVal instanceof Callable)) {
            throw ScriptRuntime.typeErrorById(
                    "msg.isnt.function", "has", ScriptRuntime.typeof(hasVal));
        }
        if (!(keysVal instanceof Callable)) {
            throw ScriptRuntime.typeErrorById(
                    "msg.isnt.function", "keys", ScriptRuntime.typeof(keysVal));
        }

        // Set-like object - use keys method
        Callable keysMethod = (Callable) keysVal;
        Object iterator =
                ScriptRuntime.callIterator(
                        keysMethod.call(cx, scope, scriptable, ScriptRuntime.emptyArgs), cx, scope);
        try (IteratorLikeIterable it = new IteratorLikeIterable(cx, scope, iterator)) {
            for (Object key : it) {
                result.js_add(key);
            }
        }

        return result;
    }

    private static Object js_difference(
            Context cx, Scriptable scope, Scriptable thisObj, Object[] args) {
        return realThis(thisObj, "difference").js_difference(cx, scope, args);
    }

    private Object js_difference(Context cx, Scriptable scope, Object[] args) {
        Object otherObj = args.length > 0 ? args[0] : Undefined.instance;

        NativeSet result = (NativeSet) cx.newObject(scope, CLASS_NAME);
        result.instanceOfSet = true;

        // ES2025: GetSetRecord requires size, has, and keys properties
        Scriptable scriptable = ScriptableObject.ensureScriptable(otherObj);
        Object sizeVal = ScriptableObject.getProperty(scriptable, "size");
        Object hasVal = ScriptableObject.getProperty(scriptable, "has");
        Object keysVal = ScriptableObject.getProperty(scriptable, "keys");

        // Validate all required properties exist
        validateSetLike(sizeVal, hasVal, keysVal);

        return js_differenceSetLike(cx, scope, otherObj, result, sizeVal, hasVal, keysVal);
    }

    private Object js_differenceSetLike(
            Context cx,
            Scriptable scope,
            Object otherObj,
            NativeSet result,
            Object sizeVal,
            Object hasVal,
            Object keysVal) {
        // Validate has and keys are callable
        if (!(hasVal instanceof Callable)) {
            throw ScriptRuntime.typeErrorById(
                    "msg.isnt.function", "has", ScriptRuntime.typeof(hasVal));
        }
        if (!(keysVal instanceof Callable)) {
            throw ScriptRuntime.typeErrorById(
                    "msg.isnt.function", "keys", ScriptRuntime.typeof(keysVal));
        }

        Callable hasMethod = (Callable) hasVal;
        Callable keysMethod = (Callable) keysVal;

        double otherSizeDouble = ScriptRuntime.toNumber(sizeVal);
        if (Double.isNaN(otherSizeDouble)) {
            throw ScriptRuntime.typeError("size is not a number");
        }
        int otherSize =
                Double.isInfinite(otherSizeDouble)
                        ? Integer.MAX_VALUE
                        : (int) Math.floor(otherSizeDouble);
        int thisSize = entries.size();

        // According to the spec and test converts-negative-zero.js:
        // When this.size > other.size, we should iterate through other.keys()
        // and remove matching elements, NOT call other.has()
        if (thisSize > otherSize) {

            // First, add all elements from this set
            for (Hashtable.Entry entry : entries) {
                result.js_add(entry.key);
            }

            // Then iterate through other and remove matching elements
            Object iterator =
                    ScriptRuntime.callIterator(
                            keysMethod.call(
                                    cx,
                                    scope,
                                    ScriptableObject.ensureScriptable(otherObj),
                                    ScriptRuntime.emptyArgs),
                            cx,
                            scope);
            try (IteratorLikeIterable it = new IteratorLikeIterable(cx, scope, iterator)) {
                for (Object key : it) {
                    // Convert -0 to +0 as the spec requires
                    if (key instanceof Number
                            && ((Number) key).doubleValue() == ScriptRuntime.negativeZero) {
                        result.js_delete(ScriptRuntime.zeroObj);
                    } else {
                        result.js_delete(key);
                    }
                }
            }
        } else {
            // When this.size <= other.size, iterate through this and check with has()

            for (Hashtable.Entry entry : entries) {
                Object key = entry.key;
                Object inOther = callHas(cx, scope, otherObj, hasMethod, key);
                if (!ScriptRuntime.toBoolean(inOther)) {
                    result.js_add(key);
                }
            }
        }

        return result;
    }

    private static Object js_symmetricDifference(
            Context cx, Scriptable scope, Scriptable thisObj, Object[] args) {
        return realThis(thisObj, "symmetricDifference").js_symmetricDifference(cx, scope, args);
    }

    private Object js_symmetricDifference(Context cx, Scriptable scope, Object[] args) {
        Object otherObj = args.length > 0 ? args[0] : Undefined.instance;

        NativeSet result = (NativeSet) cx.newObject(scope, CLASS_NAME);
        result.instanceOfSet = true;

        // ES2025: GetSetRecord requires size, has, and keys properties
        Scriptable scriptable = ScriptableObject.ensureScriptable(otherObj);
        Object sizeVal = ScriptableObject.getProperty(scriptable, "size");
        Object hasVal = ScriptableObject.getProperty(scriptable, "has");
        Object keysVal = ScriptableObject.getProperty(scriptable, "keys");

        // Validate all required properties exist
        validateSetLike(sizeVal, hasVal, keysVal);

        // Validate size is a number (required by GetSetRecord)
        double otherSizeDouble = ScriptRuntime.toNumber(sizeVal);
        if (Double.isNaN(otherSizeDouble)) {
            throw ScriptRuntime.typeError("size is not a number");
        }

        // Validate has and keys are callable
        if (!(hasVal instanceof Callable)) {
            throw ScriptRuntime.typeErrorById(
                    "msg.isnt.function", "has", ScriptRuntime.typeof(hasVal));
        }
        if (!(keysVal instanceof Callable)) {
            throw ScriptRuntime.typeErrorById(
                    "msg.isnt.function", "keys", ScriptRuntime.typeof(keysVal));
        }

        // Set-like object path
        Callable hasMethod = (Callable) hasVal;
        Callable keysMethod = (Callable) keysVal;

        // Add elements from this that are not in other
        for (Hashtable.Entry entry : entries) {
            Object key = entry.key;
            Object inOther = callHas(cx, scope, otherObj, hasMethod, key);
            if (!ScriptRuntime.toBoolean(inOther)) {
                result.js_add(key);
            }
        }

        // Add elements from other that are not in this
        Object iterator =
                ScriptRuntime.callIterator(
                        keysMethod.call(cx, scope, scriptable, ScriptRuntime.emptyArgs), cx, scope);
        try (IteratorLikeIterable it = new IteratorLikeIterable(cx, scope, iterator)) {
            for (Object key : it) {
                if (js_has(key) != Boolean.TRUE) {
                    result.js_add(key);
                }
            }
        }

        return result;
    }

    private static Object js_isSubsetOf(
            Context cx, Scriptable scope, Scriptable thisObj, Object[] args) {
        return realThis(thisObj, "isSubsetOf").js_isSubsetOf(cx, scope, args);
    }

    private Object js_isSubsetOf(Context cx, Scriptable scope, Object[] args) {
        Object otherObj = args.length > 0 ? args[0] : Undefined.instance;

        // ES2025: GetSetRecord requires size, has, and keys properties
        Scriptable scriptable = ScriptableObject.ensureScriptable(otherObj);
        Object sizeVal = ScriptableObject.getProperty(scriptable, "size");
        Object hasVal = ScriptableObject.getProperty(scriptable, "has");
        Object keysVal = ScriptableObject.getProperty(scriptable, "keys");

        // Validate all required properties exist
        validateSetLike(sizeVal, hasVal, keysVal);

        // Validate has and keys are callable (even though isSubsetOf only uses has)
        if (!(hasVal instanceof Callable)) {
            throw ScriptRuntime.typeErrorById(
                    "msg.isnt.function", "has", ScriptRuntime.typeof(hasVal));
        }
        if (!(keysVal instanceof Callable)) {
            throw ScriptRuntime.typeErrorById(
                    "msg.isnt.function", "keys", ScriptRuntime.typeof(keysVal));
        }

        // Set-like object - use has method for efficiency
        Callable hasMethod = (Callable) hasVal;

        // Check size optimization
        double otherSizeDouble = ScriptRuntime.toNumber(sizeVal);
        if (Double.isNaN(otherSizeDouble)) {
            throw ScriptRuntime.typeError("size is not a number");
        }
        int otherSize =
                Double.isInfinite(otherSizeDouble)
                        ? Integer.MAX_VALUE
                        : (int) Math.floor(otherSizeDouble);
        int thisSize = entries.size();

        // If this set is larger than other, it cannot be a subset
        if (thisSize > otherSize) {
            return Boolean.FALSE;
        }

        // Check if all elements of this are in other
        for (Hashtable.Entry entry : entries) {
            Object key = entry.key;
            Object inOther = callHas(cx, scope, otherObj, hasMethod, key);
            if (!ScriptRuntime.toBoolean(inOther)) {
                return Boolean.FALSE;
            }
        }

        return Boolean.TRUE;
    }

    private static Object js_isSupersetOf(
            Context cx, Scriptable scope, Scriptable thisObj, Object[] args) {
        return realThis(thisObj, "isSupersetOf").js_isSupersetOf(cx, scope, args);
    }

    private Object js_isSupersetOf(Context cx, Scriptable scope, Object[] args) {
        Object otherObj = args.length > 0 ? args[0] : Undefined.instance;

        // ES2025: GetSetRecord requires size, has, and keys properties
        Scriptable scriptable = ScriptableObject.ensureScriptable(otherObj);
        Object sizeVal = ScriptableObject.getProperty(scriptable, "size");
        Object hasVal = ScriptableObject.getProperty(scriptable, "has");
        Object keysVal = ScriptableObject.getProperty(scriptable, "keys");

        // Validate all required properties exist
        validateSetLike(sizeVal, hasVal, keysVal);

        // Validate size is a number (required by GetSetRecord)
        double otherSizeDouble = ScriptRuntime.toNumber(sizeVal);
        if (Double.isNaN(otherSizeDouble)) {
            throw ScriptRuntime.typeError("size is not a number");
        }

        // Validate has and keys are callable (GetSetRecord requires both even if isSupersetOf only
        // uses keys)
        if (!(hasVal instanceof Callable)) {
            throw ScriptRuntime.typeErrorById(
                    "msg.isnt.function", "has", ScriptRuntime.typeof(hasVal));
        }
        if (!(keysVal instanceof Callable)) {
            throw ScriptRuntime.typeErrorById(
                    "msg.isnt.function", "keys", ScriptRuntime.typeof(keysVal));
        }

        // Iterate through other.keys() and check if all elements are in this
        Callable keysMethod = (Callable) keysVal;
        Object iterator =
                ScriptRuntime.callIterator(
                        keysMethod.call(cx, scope, scriptable, ScriptRuntime.emptyArgs), cx, scope);
        try (IteratorLikeIterable it = new IteratorLikeIterable(cx, scope, iterator)) {
            for (Object value : it) {
                if (js_has(value) != Boolean.TRUE) {
                    return Boolean.FALSE;
                }
            }
        }
        return Boolean.TRUE;
    }

    private static Object js_isDisjointFrom(
            Context cx, Scriptable scope, Scriptable thisObj, Object[] args) {
        return realThis(thisObj, "isDisjointFrom").js_isDisjointFrom(cx, scope, args);
    }

    private Object js_isDisjointFrom(Context cx, Scriptable scope, Object[] args) {
        Object otherObj = args.length > 0 ? args[0] : Undefined.instance;

        // ES2025: GetSetRecord requires size, has, and keys properties
        Scriptable scriptable = ScriptableObject.ensureScriptable(otherObj);
        Object sizeVal = ScriptableObject.getProperty(scriptable, "size");
        Object hasVal = ScriptableObject.getProperty(scriptable, "has");
        Object keysVal = ScriptableObject.getProperty(scriptable, "keys");

        // Validate all required properties exist
        validateSetLike(sizeVal, hasVal, keysVal);

        // Validate has and keys are callable
        if (!(hasVal instanceof Callable)) {
            throw ScriptRuntime.typeErrorById(
                    "msg.isnt.function", "has", ScriptRuntime.typeof(hasVal));
        }
        if (!(keysVal instanceof Callable)) {
            throw ScriptRuntime.typeErrorById(
                    "msg.isnt.function", "keys", ScriptRuntime.typeof(keysVal));
        }

        // Set-like object with size optimization
        Callable hasMethod = (Callable) hasVal;
        Callable keysMethod = (Callable) keysVal;

        double otherSizeDouble = ScriptRuntime.toNumber(sizeVal);
        if (Double.isNaN(otherSizeDouble)) {
            throw ScriptRuntime.typeError("size is not a number");
        }
        int otherSize =
                Double.isInfinite(otherSizeDouble)
                        ? Integer.MAX_VALUE
                        : (int) Math.floor(otherSizeDouble);
        int thisSize = entries.size();

        if (thisSize <= otherSize) {
            // Iterate through this set
            for (Hashtable.Entry entry : entries) {
                Object key = entry.key;
                Object inOther = callHas(cx, scope, otherObj, hasMethod, key);
                if (ScriptRuntime.toBoolean(inOther)) {
                    return Boolean.FALSE;
                }
            }
        } else {
            // Iterate through other
            Object iterator =
                    ScriptRuntime.callIterator(
                            keysMethod.call(cx, scope, scriptable, ScriptRuntime.emptyArgs),
                            cx,
                            scope);
            try (IteratorLikeIterable it = new IteratorLikeIterable(cx, scope, iterator)) {
                for (Object key : it) {
                    if (js_has(key) == Boolean.TRUE) {
                        return Boolean.FALSE;
                    }
                }
            }
        }

        return Boolean.TRUE;
    }

    // Helper methods for Set operations

    private static Object callHas(
            Context cx, Scriptable scope, Object obj, Object hasMethod, Object key) {
        return ((Callable) hasMethod)
                .call(cx, scope, ScriptableObject.ensureScriptable(obj), new Object[] {key});
    }

    private static void validateSetLike(Object sizeVal, Object hasVal, Object keysVal) {
        if (sizeVal == Scriptable.NOT_FOUND) {
            throw ScriptRuntime.typeError("Set-like object must have a 'size' property");
        }
        if (hasVal == Scriptable.NOT_FOUND) {
            throw ScriptRuntime.typeError("Set-like object must have a 'has' method");
        }
        if (keysVal == Scriptable.NOT_FOUND) {
            throw ScriptRuntime.typeError("Set-like object must have a 'keys' method");
        }
    }
}
