/*
 * Decompiled with CFR 0.152.
 */
package org.kiwiproject.consul.util.failover.strategy;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.net.HostAndPort;
import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import okhttp3.HttpUrl;
import okhttp3.Request;
import okhttp3.Response;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.kiwiproject.consul.util.HostAndPorts;
import org.kiwiproject.consul.util.failover.strategy.ConsulFailoverStrategy;

public class BlacklistingConsulFailoverStrategy
implements ConsulFailoverStrategy {
    @VisibleForTesting
    final Map<HostAndPort, Instant> blacklist = new ConcurrentHashMap<HostAndPort, Instant>();
    private final List<HostAndPort> targets;
    private final int numberOfTargets;
    private final long timeout;

    public BlacklistingConsulFailoverStrategy(Collection<HostAndPort> targets, long timeout) {
        this(targets, Duration.ofMillis(timeout));
    }

    public BlacklistingConsulFailoverStrategy(Collection<HostAndPort> targets, Duration timeout) {
        Preconditions.checkArgument((Objects.nonNull(targets) && !targets.isEmpty() ? 1 : 0) != 0, (Object)"targets must not be null or empty");
        this.targets = List.copyOf(targets);
        this.numberOfTargets = this.targets.size();
        Preconditions.checkArgument((boolean)Objects.nonNull(timeout), (Object)"timeout must not be null");
        long timeoutMillis = timeout.toMillis();
        Preconditions.checkArgument((timeoutMillis > 0L ? 1 : 0) != 0, (Object)"timeout must be positive");
        this.timeout = timeoutMillis;
    }

    @Override
    public @NonNull Optional<Request> computeNextStage(Request previousRequest) {
        return this.computeNextStage(previousRequest, null);
    }

    @Override
    public @NonNull Optional<Request> computeNextStage(@NonNull Request previousRequest, @Nullable Response previousResponse) {
        HostAndPort nextTarget = HostAndPorts.hostAndPortFromOkHttpRequest(previousRequest);
        if (BlacklistingConsulFailoverStrategy.previousResponseFailedAndWasNot404(previousResponse)) {
            this.addToBlacklist(nextTarget);
        }
        if (this.isBlacklisted(nextTarget)) {
            Optional<HostAndPort> optionalNext = this.findTargetNotInBlacklist();
            if (optionalNext.isEmpty()) {
                return Optional.empty();
            }
            nextTarget = optionalNext.get();
        }
        HttpUrl nextURL = previousRequest.url().newBuilder().host(nextTarget.getHost()).port(nextTarget.getPort()).build();
        return Optional.of(previousRequest.newBuilder().url(nextURL).build());
    }

    private Optional<HostAndPort> findTargetNotInBlacklist() {
        return this.targets.stream().filter(target -> {
            if (this.isNotBlacklisted((HostAndPort)target)) {
                return true;
            }
            if (this.isPastBlacklistDuration((HostAndPort)target)) {
                this.blacklist.remove(target);
                return true;
            }
            return false;
        }).findAny();
    }

    @VisibleForTesting
    boolean isPastBlacklistDuration(HostAndPort target) {
        Instant blacklistedAt = this.blacklist.get(target);
        if (Objects.isNull(blacklistedAt)) {
            return true;
        }
        Instant timeoutExpiration = blacklistedAt.plusMillis(this.timeout);
        Instant now = Instant.now();
        return now.equals(timeoutExpiration) || now.isAfter(timeoutExpiration);
    }

    private static boolean previousResponseFailedAndWasNot404(Response previousResponse) {
        return Objects.nonNull(previousResponse) && !previousResponse.isSuccessful() && previousResponse.code() != 404;
    }

    @Override
    public boolean isRequestViable(@NonNull Request request) {
        return this.atLeastOneTargetIsAvailable() || this.isNotBlacklisted(HostAndPorts.hostAndPortFromOkHttpRequest(request)) || this.findTargetNotInBlacklist().isPresent();
    }

    private boolean atLeastOneTargetIsAvailable() {
        return this.numberOfTargets > this.blacklist.size();
    }

    private boolean isNotBlacklisted(HostAndPort target) {
        return !this.isBlacklisted(target);
    }

    private boolean isBlacklisted(HostAndPort target) {
        return this.blacklist.containsKey(target);
    }

    @Override
    public void markRequestFailed(@NonNull Request request) {
        HostAndPort hostAndPort = HostAndPorts.hostAndPortFromOkHttpRequest(request);
        this.addToBlacklist(hostAndPort);
    }

    @VisibleForTesting
    void addToBlacklist(HostAndPort target) {
        this.blacklist.put(target, Instant.now());
    }
}

