/*
 * Decompiled with CFR 0.152.
 */
package org.jline.reader.impl.history;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.time.DateTimeException;
import java.time.Instant;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Objects;
import java.util.Spliterator;
import org.jline.reader.History;
import org.jline.reader.LineReader;
import org.jline.reader.impl.ReaderUtils;
import org.jline.utils.Log;

public class DefaultHistory
implements History {
    public static final int DEFAULT_HISTORY_SIZE = 500;
    public static final int DEFAULT_HISTORY_FILE_SIZE = 10000;
    private final LinkedList<History.Entry> items = new LinkedList();
    private LineReader reader;
    private int lastLoaded = 0;
    private int nbEntriesInFile = 0;
    private int offset = 0;
    private int index = 0;

    public DefaultHistory() {
    }

    public DefaultHistory(LineReader reader) {
        this.attach(reader);
    }

    private Path getPath() {
        Object obj;
        Object object = obj = this.reader != null ? this.reader.getVariables().get("history-file") : null;
        if (obj instanceof Path) {
            return (Path)obj;
        }
        if (obj instanceof File) {
            return ((File)obj).toPath();
        }
        if (obj != null) {
            return Paths.get(obj.toString(), new String[0]);
        }
        return null;
    }

    @Override
    public void attach(LineReader reader) {
        if (this.reader != reader) {
            this.reader = reader;
            try {
                this.load();
            }
            catch (IOException | IllegalArgumentException e) {
                Log.warn((Object[])new Object[]{"Failed to load history", e});
            }
        }
    }

    @Override
    public void load() throws IOException {
        block15: {
            Path path = this.getPath();
            if (path != null) {
                try {
                    if (!Files.exists(path, new LinkOption[0])) break block15;
                    Log.trace((Object[])new Object[]{"Loading history from: ", path});
                    try (BufferedReader reader = Files.newBufferedReader(path);){
                        this.internalClear();
                        reader.lines().forEach(line -> this.addHistoryLine(path, (String)line));
                        this.nbEntriesInFile = this.lastLoaded = this.items.size();
                        this.maybeResize();
                    }
                }
                catch (IOException | IllegalArgumentException e) {
                    Log.debug((Object[])new Object[]{"Failed to load history; clearing", e});
                    this.internalClear();
                    throw e;
                }
            }
        }
    }

    protected void addHistoryLine(Path path, String line) {
        if (this.reader.isSet(LineReader.Option.HISTORY_TIMESTAMPED)) {
            Instant time;
            int idx = line.indexOf(58);
            String badHistoryFileSyntax = "Bad history file syntax! The history file `" + path + "` may be an older history: please remove it or use a different history file.";
            if (idx < 0) {
                throw new IllegalArgumentException(badHistoryFileSyntax);
            }
            try {
                time = Instant.ofEpochMilli(Long.parseLong(line.substring(0, idx)));
            }
            catch (NumberFormatException | DateTimeException e) {
                throw new IllegalArgumentException(badHistoryFileSyntax);
            }
            String unescaped = DefaultHistory.unescape(line.substring(idx + 1));
            this.internalAdd(time, unescaped);
        } else {
            this.internalAdd(Instant.now(), DefaultHistory.unescape(line));
        }
    }

    @Override
    public void purge() throws IOException {
        this.internalClear();
        Path path = this.getPath();
        if (path != null) {
            Log.trace((Object[])new Object[]{"Purging history from: ", path});
            Files.deleteIfExists(path);
        }
    }

    @Override
    public void save() throws IOException {
        Path path = this.getPath();
        if (path != null) {
            Log.trace((Object[])new Object[]{"Saving history to: ", path});
            Files.createDirectories(path.toAbsolutePath().getParent(), new FileAttribute[0]);
            try (BufferedWriter writer = Files.newBufferedWriter(path.toAbsolutePath(), StandardOpenOption.WRITE, StandardOpenOption.APPEND, StandardOpenOption.CREATE);){
                for (History.Entry entry : this.items.subList(this.lastLoaded, this.items.size())) {
                    if (!this.isPersistable(entry)) continue;
                    writer.append(this.format(entry));
                }
            }
            this.nbEntriesInFile += this.items.size() - this.lastLoaded;
            int max = ReaderUtils.getInt(this.reader, "history-file-size", 10000);
            if (this.nbEntriesInFile > max + max / 4) {
                this.trimHistory(path, max);
            }
        }
        this.lastLoaded = this.items.size();
    }

    protected void trimHistory(Path path, int max) throws IOException {
        Log.trace((Object[])new Object[]{"Trimming history path: ", path});
        LinkedList<History.Entry> allItems = new LinkedList<History.Entry>();
        try (BufferedReader reader = Files.newBufferedReader(path);){
            reader.lines().forEach(l -> {
                int idx = l.indexOf(58);
                Instant time = Instant.ofEpochMilli(Long.parseLong(l.substring(0, idx)));
                String line = DefaultHistory.unescape(l.substring(idx + 1));
                allItems.add(this.createEntry(allItems.size(), time, line));
            });
        }
        DefaultHistory.doTrimHistory(allItems, max);
        Path temp = Files.createTempFile(path.toAbsolutePath().getParent(), path.getFileName().toString(), ".tmp", new FileAttribute[0]);
        try (BufferedWriter writer = Files.newBufferedWriter(temp, StandardOpenOption.WRITE);){
            for (History.Entry entry : allItems) {
                writer.append(this.format(entry));
            }
        }
        Files.move(temp, path, StandardCopyOption.REPLACE_EXISTING);
        this.internalClear();
        this.offset = allItems.get(0).index();
        this.items.addAll(allItems);
        this.lastLoaded = this.items.size();
        this.nbEntriesInFile = this.items.size();
        this.maybeResize();
    }

    protected EntryImpl createEntry(int index, Instant time, String line) {
        return new EntryImpl(index, time, line);
    }

    private void internalClear() {
        this.offset = 0;
        this.index = 0;
        this.lastLoaded = 0;
        this.nbEntriesInFile = 0;
        this.items.clear();
    }

    static void doTrimHistory(List<History.Entry> allItems, int max) {
        for (int idx = 0; idx < allItems.size(); ++idx) {
            int ridx = allItems.size() - idx - 1;
            String line = allItems.get(ridx).line().trim();
            ListIterator<History.Entry> iterator = allItems.listIterator(ridx);
            while (iterator.hasPrevious()) {
                String l = iterator.previous().line();
                if (!line.equals(l.trim())) continue;
                iterator.remove();
            }
        }
        while (allItems.size() > max) {
            allItems.remove(0);
        }
    }

    @Override
    public int size() {
        return this.items.size();
    }

    @Override
    public boolean isEmpty() {
        return this.items.isEmpty();
    }

    @Override
    public int index() {
        return this.offset + this.index;
    }

    @Override
    public int first() {
        return this.offset;
    }

    @Override
    public int last() {
        return this.offset + this.items.size() - 1;
    }

    private String format(History.Entry entry) {
        if (this.reader.isSet(LineReader.Option.HISTORY_TIMESTAMPED)) {
            return Long.toString(entry.time().toEpochMilli()) + ":" + DefaultHistory.escape(entry.line()) + "\n";
        }
        return DefaultHistory.escape(entry.line()) + "\n";
    }

    @Override
    public String get(int index) {
        return this.items.get(index - this.offset).line();
    }

    @Override
    public void add(Instant time, String line) {
        Objects.requireNonNull(time);
        Objects.requireNonNull(line);
        if (ReaderUtils.getBoolean(this.reader, "disable-history", false)) {
            return;
        }
        if (ReaderUtils.isSet(this.reader, LineReader.Option.HISTORY_IGNORE_SPACE) && line.startsWith(" ")) {
            return;
        }
        if (ReaderUtils.isSet(this.reader, LineReader.Option.HISTORY_REDUCE_BLANKS)) {
            line = line.trim();
        }
        if (ReaderUtils.isSet(this.reader, LineReader.Option.HISTORY_IGNORE_DUPS) && !this.items.isEmpty() && line.equals(this.items.getLast().line())) {
            return;
        }
        if (this.matchPatterns(ReaderUtils.getString(this.reader, "history-ignore", ""), line)) {
            return;
        }
        this.internalAdd(time, line);
        if (ReaderUtils.isSet(this.reader, LineReader.Option.HISTORY_INCREMENTAL)) {
            try {
                this.save();
            }
            catch (IOException e) {
                Log.warn((Object[])new Object[]{"Failed to save history", e});
            }
        }
    }

    protected boolean matchPatterns(String patterns, String line) {
        if (patterns == null || patterns.isEmpty()) {
            return false;
        }
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < patterns.length(); ++i) {
            char ch = patterns.charAt(i);
            if (ch == '\\') {
                ch = patterns.charAt(++i);
                sb.append(ch);
                continue;
            }
            if (ch == ':') {
                sb.append('|');
                continue;
            }
            if (ch != '*') continue;
            sb.append('.').append('*');
        }
        return line.matches(sb.toString());
    }

    protected void internalAdd(Instant time, String line) {
        EntryImpl entry = new EntryImpl(this.offset + this.items.size(), time, line);
        this.items.add(entry);
        this.maybeResize();
    }

    private void maybeResize() {
        while (this.size() > ReaderUtils.getInt(this.reader, "history-size", 500)) {
            this.items.removeFirst();
            --this.lastLoaded;
            ++this.offset;
        }
        this.index = this.size();
    }

    @Override
    public ListIterator<History.Entry> iterator(int index) {
        return this.items.listIterator(index - this.offset);
    }

    @Override
    public Spliterator<History.Entry> spliterator() {
        return this.items.spliterator();
    }

    @Override
    public boolean moveToLast() {
        int lastEntry = this.size() - 1;
        if (lastEntry >= 0 && lastEntry != this.index) {
            this.index = this.size() - 1;
            return true;
        }
        return false;
    }

    @Override
    public boolean moveTo(int index) {
        if ((index -= this.offset) >= 0 && index < this.size()) {
            this.index = index;
            return true;
        }
        return false;
    }

    @Override
    public boolean moveToFirst() {
        if (this.size() > 0 && this.index != 0) {
            this.index = 0;
            return true;
        }
        return false;
    }

    @Override
    public void moveToEnd() {
        this.index = this.size();
    }

    @Override
    public String current() {
        if (this.index >= this.size()) {
            return "";
        }
        return this.items.get(this.index).line();
    }

    @Override
    public boolean previous() {
        if (this.index <= 0) {
            return false;
        }
        --this.index;
        return true;
    }

    @Override
    public boolean next() {
        if (this.index >= this.size()) {
            return false;
        }
        ++this.index;
        return true;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        for (History.Entry e : this) {
            sb.append(e.toString()).append("\n");
        }
        return sb.toString();
    }

    private static String escape(String s) {
        StringBuilder sb = new StringBuilder();
        block5: for (int i = 0; i < s.length(); ++i) {
            char ch = s.charAt(i);
            switch (ch) {
                case '\n': {
                    sb.append('\\');
                    sb.append('n');
                    continue block5;
                }
                case '\r': {
                    sb.append('\\');
                    sb.append('r');
                    continue block5;
                }
                case '\\': {
                    sb.append('\\');
                    sb.append('\\');
                    continue block5;
                }
                default: {
                    sb.append(ch);
                }
            }
        }
        return sb.toString();
    }

    static String unescape(String s) {
        StringBuilder sb = new StringBuilder();
        block3: for (int i = 0; i < s.length(); ++i) {
            char ch = s.charAt(i);
            switch (ch) {
                case '\\': {
                    ch = s.charAt(++i);
                    if (ch == 'n') {
                        sb.append('\n');
                        continue block3;
                    }
                    if (ch == 'r') {
                        sb.append('\r');
                        continue block3;
                    }
                    sb.append(ch);
                    continue block3;
                }
                default: {
                    sb.append(ch);
                }
            }
        }
        return sb.toString();
    }

    protected static class EntryImpl
    implements History.Entry {
        private final int index;
        private final Instant time;
        private final String line;

        public EntryImpl(int index, Instant time, String line) {
            this.index = index;
            this.time = time;
            this.line = line;
        }

        @Override
        public int index() {
            return this.index;
        }

        @Override
        public Instant time() {
            return this.time;
        }

        @Override
        public String line() {
            return this.line;
        }

        public String toString() {
            return String.format("%d: %s", this.index, this.line);
        }
    }
}

