/*
 * Copyright 2010-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package kotlin.reflect.jvm.internal.impl.builtins

import com.google.protobuf.ExtensionRegistryLite
import kotlin.reflect.jvm.internal.impl.name.ClassId
import kotlin.reflect.jvm.internal.impl.name.FqName
import kotlin.reflect.jvm.internal.impl.serialization.SerializedResourcePaths
import kotlin.reflect.jvm.internal.impl.serialization.builtins.BuiltInsProtoBuf

object BuiltInsSerializedResourcePaths : SerializedResourcePaths {
    override val extensionRegistry: ExtensionRegistryLite

    init {
        extensionRegistry = ExtensionRegistryLite.newInstance()
        BuiltInsProtoBuf.registerAllExtensions(extensionRegistry)
    }

    val CLASS_METADATA_FILE_EXTENSION = "kotlin_class"
    val PACKAGE_FILE_EXTENSION = "kotlin_package"
    val STRING_TABLE_FILE_EXTENSION = "kotlin_string_table"
    val BUILTINS_FILE_EXTENSION = "kotlin_builtins"

    override fun getClassMetadataPath(classId: ClassId): String {
        return packageFqNameToPath(classId.packageFqName) + "/" + classId.relativeClassName.asString() +
               "." + CLASS_METADATA_FILE_EXTENSION
    }

    override fun getPackageFilePath(fqName: FqName): String =
            packageFqNameToPath(fqName) + "/" + shortName(fqName) + "." + PACKAGE_FILE_EXTENSION

    override fun getStringTableFilePath(fqName: FqName): String =
            packageFqNameToPath(fqName) + "/" + shortName(fqName) + "." + STRING_TABLE_FILE_EXTENSION

    fun getBuiltInsFilePath(fqName: FqName): String =
            packageFqNameToPath(fqName) + "/" + shortName(fqName) + "." + BUILTINS_FILE_EXTENSION


    private fun packageFqNameToPath(fqName: FqName): String =
            fqName.asString().replace('.', '/')

    private fun shortName(fqName: FqName): String =
            if (fqName.isRoot) "default-package" else fqName.shortName().asString()
}
