/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.server.profileservice;

import java.util.Collection;
import java.util.concurrent.atomic.AtomicBoolean;

import org.jboss.bootstrap.spi.Bootstrap;
import org.jboss.dependency.spi.ControllerContext;
import org.jboss.deployers.client.spi.IncompleteDeploymentException;
import org.jboss.deployers.client.spi.main.MainDeployer;
import org.jboss.deployers.plugins.main.MainDeployerImpl;
import org.jboss.deployers.structure.spi.DeploymentContext;
import org.jboss.deployers.vfs.spi.client.VFSDeployment;
import org.jboss.kernel.Kernel;
import org.jboss.kernel.spi.dependency.KernelController;
import org.jboss.logging.Logger;
import org.jboss.managed.api.ManagedDeployment.DeploymentPhase;
import org.jboss.profileservice.spi.Profile;
import org.jboss.profileservice.spi.ProfileKey;
import org.jboss.profileservice.spi.ProfileService;
import org.jboss.system.server.Server;

/**
 * Bootstraps the profile service
 * 
 * @author Scott.Stark@jboss.org
 * @author adrian@jboss.org
 * @version $Revision: 76814 $
 */
public class ProfileServiceBootstrap implements Bootstrap
{
   /** The log */
   private static final Logger log = Logger.getLogger(ProfileServiceBootstrap.class);
   
   /** The name of the profile that is being booted */
   protected String profileName = "default";
   
   /** The server MainDeployer */
   protected MainDeployer mainDeployer;

   /** The server ProfileService */
   protected ProfileService profileService;

   /** The kernel */
   protected Kernel kernel;
   
   /** Whether we are shutdown */
   private AtomicBoolean shutdown = new AtomicBoolean(false);
   
   /**
    * Create a new ProfileServiceBootstrap.
    */
   public ProfileServiceBootstrap()
   {
   }

   /**
    * Return the MainDeployer bean.
    * 
    * @return the MainDeployer bean if bootstrap succeeded, null otherwise.
    */
   public MainDeployer getMainDeployer()
   {
      return mainDeployer;
   }
   public void setMainDeployer(MainDeployer mainDeployer)
   {
      this.mainDeployer = mainDeployer;
   }

   /**
    * Return the ProfileService bean.
    * 
    * @return the ProfileService bean if bootstrap succeeded, null otherwise
    */
   public ProfileService getProfileService()
   {
      return profileService;
   }
   public void setProfileService(ProfileService profileService)
   {
      this.profileService = profileService;
   }

   /**
    * Get the kernel.
    * 
    * @return the kernel.
    */
   public Kernel getKernel()
   {
      return kernel;
   }

   /**
    * Set the kernel.
    * 
    * @param kernel the kernel.
    */
   public void setKernel(Kernel kernel)
   {
      this.kernel = kernel;
   }

   public void start(Server server) throws Exception
   {
      shutdown.set(false);
      
      if(profileService == null)
         throw new IllegalStateException("The ProfileService has not been injected"); 
      log.debug("Using ProfileService: " + profileService);
      if(profileService == null)
         throw new IllegalStateException("The MainDeployer has not been injected"); 
      log.debug("Using MainDeployer: " + mainDeployer);

      // Validate that everything is ok
      mainDeployer.checkComplete();


      // Load the profile beans
      try
      {
         loadProfile(profileName);
      }
      catch (IncompleteDeploymentException e)
      {
         log.error("Failed to load profile: " + e.getMessage());
      }
      catch (Exception e)
      {
         log.error("Failed to load profile: ", e);
      }
      // Mark the profile as ready for hotdeployment if supported
      Profile profile = profileService.getActiveProfile();
      if( profile != null )
         profile.enableModifiedDeploymentChecks(true);
   }

   public void prepareShutdown(Server server)
   {
      shutdown.set(true);
      if (mainDeployer != null)
         mainDeployer.prepareShutdown();
   }

   public void shutdown(Server server)
   {
      unloadProfile(profileName);
      try
      {
         mainDeployer.shutdown();
      }
      catch (Throwable t)
      {
         log.warn("Error shutting down the main deployer", t);
      }
   }

   /**
    * Get a bean
    * 
    * @param <T> the expected type 
    * @param controller the controller 
    * @param name the bean name
    * @param expectedType the expected type
    * @return the bean
    * @throws IllegalStateException if the bean is not installed or has the wrong type
    */
   protected <T> T getBean(KernelController controller, Object name, Class<T> expectedType)
   {
      ControllerContext context = controller.getInstalledContext(name);
      if (context == null)
         throw new IllegalStateException("Context not installed: " + name);
      Object result = context.getTarget();
      if (result == null)
         throw new IllegalStateException("No target for " + name);
      if (expectedType.isInstance(result) == false)
         throw new IllegalStateException(name + " expected " + expectedType.getName() + " was " + result.getClass().getName());
      return expectedType.cast(result);
   }

   /**
    * Load the deployments associated with the named profile and deploy them
    * using the MainDeployer.
    * 
    * @param name
    * @throws Exception for any error
    * @throws NullPointerException if either the MainDeployer or ProfileService
    * have not been injected.
    */
   protected void loadProfile(String name) throws Exception
   {
      MainDeployer deployer = getMainDeployer();
      if (deployer == null)
         throw new NullPointerException("MainDeployer has not been set");
      ProfileService ps = getProfileService();
      if (ps == null)
         throw new NullPointerException("ProfileService has not been set");

      // Load the named profile
      ProfileKey key = new ProfileKey(name);
      Profile profile = ps.getProfile(key);

      // HACK
      VFSDeployment first = null;
      
      // Deploy the bootstrap
      Collection<VFSDeployment> boostraps = profile.getDeployments(DeploymentPhase.BOOTSTRAP);
      for (VFSDeployment d : boostraps)
      {
         deployer.addDeployment(d);
         if (first == null)
            first = d;
      }
      deployer.process();
      deployer.checkComplete();

      Thread thread = Thread.currentThread();
      ClassLoader old = thread.getContextClassLoader();
      // FIXME remove this hack
      MainDeployerImpl hack = (MainDeployerImpl) deployer;
      ClassLoader cl = null;
      if (first != null)
      {
         DeploymentContext ctx = hack.getDeploymentContext(first.getName());
         if (ctx != null)
            cl = ctx.getClassLoader();
      }
      //if (cl != null)
      //   thread.setContextClassLoader(cl);
      try
      {
         
         // Deploy the profile deployers
         Collection<VFSDeployment> profileDeployers = profile.getDeployments(DeploymentPhase.DEPLOYER);
         for (VFSDeployment d : profileDeployers)
            deployer.addDeployment(d);
         deployer.process();
         deployer.checkComplete();

         // Deploy the profile applications
         Collection<VFSDeployment> profileDeployments = profile.getDeployments(DeploymentPhase.APPLICATION);
         for (VFSDeployment d : profileDeployments)
            deployer.addDeployment(d);
         deployer.process();
         deployer.checkComplete();
      }
      finally
      {
         thread.setContextClassLoader(old);
      }
   }

   /**
    * Unload the deployments associated with the named profile and undeploy them
    * using the MainDeployer in reverse phase order.
    * 
    * @param name the profile name
    * @throws NullPointerException if either the MainDeployer or ProfileService
    * have not been injected.
    */
   protected void unloadProfile(String name)
   {
      MainDeployer deployer = getMainDeployer();
      if (deployer == null)
      {
         log.warn("MainDeployer has not been set");
         return;
      }
      ProfileService ps = getProfileService();
      if (ps == null)
      {
         log.warn("ProfileService has not been set");
         return;
      }

      try
      {
         // Load the named profile
         ProfileKey key = new ProfileKey(name);
         Profile profile = ps.getProfile(key);

         // HACK
         VFSDeployment first = null;
         
         // Deploy the bootstrap
         Collection<VFSDeployment> boostraps = profile.getDeployments(DeploymentPhase.BOOTSTRAP);
         for (VFSDeployment d : boostraps)
         {
            if (first == null)
            {
               first = d;
               break;
            }
         }

         Thread thread = Thread.currentThread();
         ClassLoader old = thread.getContextClassLoader();
         // FIXME remove this hack
         MainDeployerImpl hack = (MainDeployerImpl) deployer;
         ClassLoader cl = null;
         if (first != null)
         {
            try
            {
               DeploymentContext ctx = hack.getDeploymentContext(first.getName());
               if (ctx != null)
                  cl = ctx.getClassLoader();
            }
            catch (Exception e)
            {
               log.debug("Unable to get first deployment", e);
            }
         }
         //if (cl != null)
         //   thread.setContextClassLoader(cl);
         try
         {
            // Undeploy the applications
            unload(deployer, profile.getDeployments(DeploymentPhase.APPLICATION));
            // Undeploy the deployers
            unload(deployer, profile.getDeployments(DeploymentPhase.DEPLOYER));
            // Undeploy the bootstrap
            unload(deployer, profile.getDeployments(DeploymentPhase.BOOTSTRAP));
         }
         finally
         {
            thread.setContextClassLoader(old);
         }
      }
      catch (Throwable t)
      {
         log.warn("Error unloading profile", t);
      }
   }

   /**
    * Unload a set of deployments
    * 
    * @param deployer the main deployer
    * @param deployments the deployments
    */
   protected void unload(MainDeployer deployer, Collection<VFSDeployment> deployments)
   {
      if (deployments == null || deployments.isEmpty())
         return;
      
      for (VFSDeployment d : deployments)
      {
         try
         {
            deployer.removeDeployment(d);
         }
         catch (Exception e)
         {
            log.warn("Unable to remove deployment: " + d);
         }
      }
      deployer.process();
   }
}
