/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.server.profile.basic;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jboss.deployers.vfs.spi.client.VFSDeployment;
import org.jboss.deployers.vfs.spi.client.VFSDeploymentFactory;
import org.jboss.managed.api.ManagedDeployment.DeploymentPhase;
import org.jboss.profileservice.spi.ModificationInfo;
import org.jboss.profileservice.spi.NoSuchDeploymentException;
import org.jboss.profileservice.spi.Profile;
import org.jboss.profileservice.spi.ProfileKey;
import org.jboss.profileservice.spi.ModificationInfo.ModifyStatus;
import org.jboss.system.server.profileservice.VFSScanner;
import org.jboss.util.JBossObject;
import org.jboss.virtual.VFS;
import org.jboss.virtual.VirtualFile;

/**
 * A basic profile implementation that uses in memory store for the applications.
 * This has to be populated by some other components.
 * 
 * @see VFSScanner and subclasses for an example usage.
 * 
 * @author Scott.Stark@jboss.org
 * @author adrian@jboss.org
 * @version $Revision: 75112 $
 */
public class ProfileImpl extends JBossObject implements Profile
{
   /** The deployment factory */
   private VFSDeploymentFactory deploymentFactory = VFSDeploymentFactory.getInstance();
   
   private ProfileKey key;
   /** The directory containing the profiles */
   private File profileRoot;
   private LinkedHashMap<String, VFSDeployment> bootstraps = new LinkedHashMap<String, VFSDeployment>();
   private LinkedHashMap<String, VFSDeployment> applications = new LinkedHashMap<String, VFSDeployment>();
   private LinkedHashMap<String, VFSDeployment> deployers = new LinkedHashMap<String, VFSDeployment>();
   /** Is hot deployment checking enabled */
   private volatile boolean hotdeployEnabled;
   /** The last time the profile was modified */
   private long lastModified;

   public ProfileImpl(String profileRoot, ProfileKey key)
   {
      this.key = key;
      this.profileRoot = new File(profileRoot);
      log.info("Using profile root:"+this.profileRoot.getAbsolutePath());
   }

   public String getName()
   {
      return key.getName();
   }

   public ProfileKey getKey()
   {
      return key;
   }

   public String getVersion()
   {
      return null;
   }

   
   public long getLastModified()
   {
      return this.lastModified;
   }

   /**
    * Get the names of the applications in the profile
    * @return names of applications
    */
   public Set<String> getDeploymentNames()
   {
      HashSet<String> names = new HashSet<String>();
      names.addAll(this.bootstraps.keySet());
      names.addAll(this.deployers.keySet());
      names.addAll(this.applications.keySet());
      return names;
   }

   public VirtualFile getRootFile(DeploymentPhase phase)
      throws Exception
   {
      VirtualFile root = VFS.getRoot(profileRoot.toURI());
      switch( phase )
      {
         case BOOTSTRAP:
            root = root.findChild("conf");
            break;
         case DEPLOYER:
            root = root.findChild("deployers");
            break;
         case APPLICATION:
            root = root.findChild("deploy");
            break;
      }      
      return root;
   }

   public void addDeployment(VFSDeployment d, DeploymentPhase phase)
      throws Exception
   {
      switch( phase )
      {
         case BOOTSTRAP:
            this.addBootstrap(d);
            break;
         case DEPLOYER:
            this.addDeployer(d);
            break;
         case APPLICATION:
            this.addApplication(d);
            break;
      }
      this.lastModified = System.currentTimeMillis();
   }
   public void updateDeployment(VFSDeployment d, DeploymentPhase phase)
      throws Exception
   {
   
   }

   /**
    * Find a deployment context from the basic profile service.
    * 
    * @param    name    the name of the deployment
    * @param    phase   optionally a deployment phase, if null search all
    * @return   the deployment or null if not found
    */
   private VFSDeployment findDeployment(String name, DeploymentPhase phase)
   {
      VFSDeployment d = null;
      if( phase == null )
      {
         // Try all phases
         d = this.getBootstrap(name);
         if( d == null )
            d = this.getDeployer(name);
         if( d == null )
            d = this.getApplication(name);
      }
      else
      {
         switch( phase )
         {
            case BOOTSTRAP:
               d = this.getBootstrap(name);
               break;
            case DEPLOYER:
               d = this.getDeployer(name);
               break;
            case APPLICATION:
               d = this.getApplication(name);
               break;
         }
      }
      return d;
   }
   
   /**
    * Get a deployment context from the basic profile service.
    * 
    * @param    name    the name of the deployment
    * @param    phase   optionally a deployment phase, if null search all
    * @return   the deployment or NoSuchDeploymentException
    * @throws   NoSuchDeploymentException   if the deployment can't be found
    */
   public VFSDeployment getDeployment(String name, DeploymentPhase phase)
      throws NoSuchDeploymentException
   {
      VFSDeployment d = findDeployment(name, phase);
      // Make sure we don't return null
      if( d == null )
         throw new NoSuchDeploymentException("name="+name+", phase="+phase);
      return d;
   }

   /**
    * Checks whether a deployment context is available in the basic profile service.
    * 
    * @param    name    the name of the deployment
    * @param    phase   optionally a deployment phase, if null search all
    * @return   true if the deployment is found or false otherwise
    */
   public boolean hasDeployment(String name, DeploymentPhase phase)
   {
      return findDeployment(name, phase) != null;
   }
   
   public Set<String> getDeploymentNames(DeploymentPhase phase)
   {
      HashSet<String> names = new HashSet<String>();
      switch( phase )
      {
         case BOOTSTRAP:
            names.addAll(this.bootstraps.keySet());
            break;
         case DEPLOYER:
            names.addAll(this.deployers.keySet());
            break;
         case APPLICATION:
            names.addAll(this.applications.keySet());
            break;
      }
      return names;
   }

   public Set<String> getDeploymentNamesForType(String type)
   {
      throw new UnsupportedOperationException("getDeploymentNamesForType");
      /** TODO Deployment.getTypes()
      HashSet<String> names = new HashSet<String>();
      for(VFSDeployment d : bootstraps.values())
      {
         if( d.getTypes().contains(type) )
            names.add(d.getName());
      }
      for(VFSDeployment d : deployers.values())
      {
         if( d.getTypes().contains(type) )
            names.add(d.getName());
      }
      for(VFSDeployment d : applications.values())
      {
         if( d.getTypes().contains(type) )
            names.add(d.getName());
      }
      return names;
      */
   }

   public Collection<VFSDeployment> getDeployments()
   {
      HashSet<VFSDeployment> deployments = new HashSet<VFSDeployment>();
      deployments.addAll(this.bootstraps.values());
      deployments.addAll(this.deployers.values());
      deployments.addAll(this.applications.values());
      return Collections.unmodifiableCollection(deployments);
   }

   /**
    * Go through the applications looking for modifications. This also
    * picks the last application parent directory as the basis for
    * scanning for new deployments.
    * TODO: to handle multiple application/hotdeployment directories a
    * set of parent dirs needs to be built.
    */
   public Collection<ModificationInfo> getModifiedDeployments()
      throws Exception
   {
      if( hotdeployEnabled == false )
         return Collections.emptyList();

      ArrayList<ModificationInfo> modified = new ArrayList<ModificationInfo>();
      Collection<VFSDeployment> apps = applications.values();
      boolean trace = log.isTraceEnabled();
      if( trace )
         log.trace("Checking applications for modifications");
      VirtualFile applicationDir = null;
      if( apps != null )
      {
         Iterator<VFSDeployment> iter = apps.iterator();
         while( iter.hasNext() )
         {
            VFSDeployment d = iter.next();
            VirtualFile root = d.getRoot();
            // Save the containing parent dir to scan for new deployments
            VirtualFile parent = root.getParent();
            if( applicationDir == null || parent.getPathName().compareTo(applicationDir.getPathName()) < 0 )
               applicationDir = parent;
            // Check for removal
            if( root.exists() == false )
            {
               long rootLastModified = root.getLastModified();
               ModificationInfo info = new ModificationInfo(d, rootLastModified, ModifyStatus.REMOVED);
               modified.add(info);
               iter.remove();
               if( trace )
                  log.trace(root.getPathName() + " was removed");
               postRemove(d);
            }
            // Check for modification
            else if(hasBeenModified(root))
            {
               long rootLastModified = root.getLastModified();
               if( trace )
                  log.trace(root.getPathName() + " was modified: " + rootLastModified);
               // Need to create a duplicate ctx
               VFSDeployment deployment2 = deploymentFactory.createVFSDeployment(root);
               ModificationInfo info = new ModificationInfo(deployment2, rootLastModified, ModifyStatus.MODIFIED);
               modified.add(info);
            }
            // TODO: this could check metadata files modifications as well
         }

         // If there are no apps, use the default appplication root
         if( applicationDir == null )
            applicationDir = getRootFile(DeploymentPhase.APPLICATION);
         // Now check for additions
         ArrayList<VirtualFile> added = new ArrayList<VirtualFile>();
         addDeployments(added, applicationDir);
         for(VirtualFile vf : added)
         {
            VFSDeployment d = deploymentFactory.createVFSDeployment(vf);
            ModificationInfo info = new ModificationInfo(d, vf.getLastModified(), ModifyStatus.ADDED);
            modified.add(info);
            applications.put(d.getName(), d);
         }
      }
      if(modified.size() > 0)
         lastModified = System.currentTimeMillis();
      return modified;
   }

   /**
    * Has the root been modified.
    *
    * @param root the virtual file root
    * @return true if modifed
    * @throws Exception for any error
    */
   protected boolean hasBeenModified(VirtualFile root) throws Exception
   {
      return root.hasBeenModified();
   }

   /**
    * Apply post remove operation.
    *
    * @param deployment the vfs deployment
    * @throws Exception for any error
    */
   protected void postRemove(VFSDeployment deployment) throws Exception
   {
      // noop here
   }

   /**
    * Noop as basic profile does not support hot deployment currently
    */
   public void enableModifiedDeploymentChecks(boolean flag)
   {
      this.hotdeployEnabled = flag;
   }

   public Collection<VFSDeployment> getDeployments(DeploymentPhase phase)
   {
      Collection<VFSDeployment> deployments = null;
      switch( phase )
      {
         case BOOTSTRAP:
            deployments = this.getBootstraps();
            break;
         case DEPLOYER:
            deployments = this.getDeployers();
            break;
         case APPLICATION:
            deployments = this.getApplications();
            break;
      }
      return deployments;
   }

   public VFSDeployment removeDeployment(String name, DeploymentPhase phase)
      throws Exception
   {
      VFSDeployment d = null;
      switch( phase )
      {
         case BOOTSTRAP:
            d = this.removeBootstrap(name);
            break;
         case DEPLOYER:
            d = this.removeDeployer(name);
            break;
         case APPLICATION:
            d = this.removeApplication(name);
            break;
      }
      return d;
   }

   public Map<String, Object> getConfig()
   {
      // TODO Auto-generated method stub
      return null;
   }

   protected void addBootstrap(VFSDeployment d)
   {
      bootstraps.put(d.getName(), d);
   }

   protected VFSDeployment removeBootstrap(String name)
   {
      return bootstraps.remove(name);
   }

   protected VFSDeployment getBootstrap(String name)
   {
      return bootstraps.get(name);
   }

   protected Collection<VFSDeployment> getBootstraps()
   {
      return Collections.unmodifiableCollection(bootstraps.values());
   }

   protected void addDeployer(VFSDeployment d)
   {
      deployers.put(d.getName(), d);
   }

   protected VFSDeployment removeDeployer(String name)
   {
      return deployers.remove(name);
   }

   protected VFSDeployment getDeployer(String name)
   {
      return deployers.get(name);
   }

   protected Collection<VFSDeployment> getDeployers()
   {
      return Collections.unmodifiableCollection(deployers.values());
   }

   protected void addApplication(VFSDeployment d)
   {
      applications.put(d.getName(), d);
   }

   protected VFSDeployment removeApplication(String name)
   {
      return applications.remove(name);
   }

   protected VFSDeployment getApplication(String name)
   {
      return applications.get(name);
   }

   protected Collection<VFSDeployment> getApplications()
   {
      return Collections.unmodifiableCollection(applications.values());
   }

   private void addDeployments(List<VirtualFile> list, VirtualFile root)
      throws Exception
   {
      List<VirtualFile> components = root.getChildren();
      
      for (VirtualFile component : components)
      {
         String key = component.toURI().toString();
         if( applications.containsKey(key) == true )
            continue;

         if (component.isLeaf())
         {
            list.add(component);
         }
         else if (component.getName().indexOf('.') == -1)
         {
            // recurse if not '.' in name and recursive search is enabled
            addDeployments(list, component);
         }
         else
         {
            list.add(component);
         }
      }
   }

}
