/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.deployers.spi.management.deploy;

import java.net.URL;

import org.jboss.managed.api.ManagedDeployment.DeploymentPhase;
import org.jboss.profileservice.spi.ProfileKey;

/**
 * An interface for uploading and controlling deployments in a profile.
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision:$
 */
public interface DeploymentManager
{
   /**
    * Load and associate the given profile with the DeploymentManager
    * for future operations.
    * 
    * @param key - the profile to load
    * @param allowHotDeployments - Can be used to suspend hot deployment
    * processing while working against a live server to avoid conflicts.
    * @throws Exception for any error
    */
   public void loadProfile(ProfileKey key, boolean allowHotDeployments) throws Exception;
   /**
    * Release a profile. This frees any resources and resets the hot deployment
    * processing.
    * 
    * @param key - the profile to load
    * @param allowHotDeployments - Can be used to suspend hot deployment
    * processing while working against a live server to avoid conflicts.
    * @throws Exception
    */
   public void releaseProfile(ProfileKey key, boolean allowHotDeployments) throws Exception;

   /**
    * Add raw deployment content to the profile.
    * 
    * @param name phase unique name
    * @param phase the phase this deployment applies to
    * @param contentURL - URL for obtaining the deployment archive
    * @return DeploymentProgress progress object for the deployment
    */
   @Deprecated
   public DeploymentProgress distribute(String name, DeploymentPhase phase, URL contentURL)
      throws Exception;

   /**
    * Add raw deployment content to the profile.
    *
    * @param name phase unique name
    * @param phase the phase this deployment applies to
    * @param contentURL - URL for obtaining the deployment archive
    * @param copyContent should we copy content or just reference it
    * @return DeploymentProgress progress object for the deployment
    */
   public DeploymentProgress distribute(String name, DeploymentPhase phase,
         URL contentURL, boolean copyContent)
      throws Exception;

   /**
    * Map from the external name passed to distribute to the uploaded
    * deployment repository name.
    * 
    * @param names - the list of external deployment names
    * @param phase the deployment phase
    * @return the mapped list of unique names in the deployment repository.
    * @throws Exception for any error
    */
   public String[] getRepositoryNames(String[] names, DeploymentPhase phase) throws Exception;

   /**
    * Is redeploy supported.
    *
    * @return true if redeploy suported
    */
   public boolean isRedeploySupported();

   /**
    * Do redeploy.
    *
    * @param name the deployment to redeploy
    * @param phase the deployment phase
    * @param contentURL deployment's url
    * @return a DeploymentProgress used to run and track the redeploy progress.
    */
   public DeploymentProgress redeploy(String name, DeploymentPhase phase, URL contentURL)
      throws Exception;

   /**
    * Prepare a set of deployments through their dependency analysis phase.
    *
    * @param phase the deployment phase
    * @param names the unique names of the deployments within the phase to prepare
    * @return a DeploymentProgress used to run and track the prepare progress.
    * @throws Exception for any error
    */
   public DeploymentProgress prepare(DeploymentPhase phase, String... names)
      throws Exception;

   /**
    * Start a previously distributed deployment.
    *
    * @param phase the deployment phase
    * @param names the unique names of the deployments within the phase to start
    * @return a DeploymentProgress used to run and track the start progress.
    * @throws Exception for any error
    */
   public DeploymentProgress start(DeploymentPhase phase, String... names)
      throws Exception;

   /**
    * Stop a previously started deployment.
    *
    * @param phase the deployment phase
    * @param names the unique names of the deployments within the phase to stop
    * @return a DeploymentProgress used to run and track the stop progress.
    * @throws Exception for any error
    */
   public DeploymentProgress stop(DeploymentPhase phase, String... names)
      throws Exception;

   /**
    * Undeploy a previously distributed deployment.
    *
    * @param phase the deployment phase
    * @param names the unique names of the deployments within the phase to prepare
    * @return a DeploymentProgress used to run and track the prepare progress.
    * @throws Exception for any error
    */
   public DeploymentProgress undeploy(DeploymentPhase phase, String... names)
      throws Exception;
}
