/*
 * Copyright 2002-2009 Andy Clark, Marc Guillemot
 * Copyright 2017-2023 Ronald Brill
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.htmlunit.cyberneko;

/**
 * Parser for the Pre-defined named HTML entities.
 * <a href="https://html.spec.whatwg.org/multipage/parsing.html#character-reference-state">12.2.5.72 Character reference state</a>
 * <p>
 * From the spec:<br>
 * Consume the maximum number of characters possible, with the consumed characters matching
 * one of the identifiers in the first column of the named character references table
 * (in a case-sensitive manner).
 * Append each character to the temporary buffer when it's consumed.
 *
 * @author Ronald Brill
 */
public class HTMLEntitiesParser {
    public static final int STATE_START = 0;
    private static final int STATE_ENDS_WITH_SEMICOLON = -2;

    private static final int STATE_HEXADECIMAL_CHAR = -102;
    private static final int STATE_DECIMAL_CHAR = -104;

    private static final int STATE_NUMERIC_CHAR_END_SEMICOLON_MISSING = -105;

    private int state;
    private int consumedCount;
    private String match;
    private int code;
    private int matchLength;

    public String getMatch() {
        return match;
    }

    public int getMatchLength() {
        return matchLength;
    }

    public int getRewindCount() {
        return consumedCount - matchLength;
    }

    public boolean endsWithSemicolon() {
        return STATE_ENDS_WITH_SEMICOLON == state;
    }

    public HTMLEntitiesParser() {
        state = STATE_START;
    }

    public void setMatchFromCode() {
        // If the number is 0x00, then this is a null-character-reference parse error. Set the character reference code to 0xFFFD.
        // If the number is greater than 0x10FFFF, then this is a character-reference-outside-unicode-range parse error. Set the character reference code to 0xFFFD.
        if ((0x00 == code) || (code > 0x10FFFF)) {
            match = "\uFFFD";
            matchLength = consumedCount;
            return;
        }

        // If the number is a surrogate, then this is a surrogate-character-reference parse error. Set the character reference code to 0xFFFD
        if (Character.isSurrogate((char) code)) {
            match = "\uFFFD";
            return;
        }

        // If the number is a noncharacter, then this is a noncharacter-character-reference parse error.

        // If the number is 0x0D, or a control that's not ASCII whitespace, then this is a control-character-reference parse error.

        // If the number is one of the numbers in the first column of the following table, then find the row with that number in the first column,
        // and set the character reference code to the number in the second column of that row.
        switch (code) {
            case 0x80:
                match = "\u20AC";
                matchLength = consumedCount;
                return;

            case 0x82:
                match = "\u201A";
                matchLength = consumedCount;
                return;

            case 0x83:
                match = "\u0192";
                matchLength = consumedCount;
                return;

            case 0x84:
                match = "\u201E";
                matchLength = consumedCount;
                return;

            case 0x85:
                match = "\u2026";
                matchLength = consumedCount;
                return;

            case 0x86:
                match = "\u2020";
                matchLength = consumedCount;
                return;

            case 0x87:
                match = "\u2021";
                matchLength = consumedCount;
                return;

            case 0x88:
                match = "\u02C6";
                matchLength = consumedCount;
                return;

            case 0x89:
                match = "\u2030";
                matchLength = consumedCount;
                return;

            case 0x8A:
                match = "\u0160";
                matchLength = consumedCount;
                return;

            case 0x8B:
                match = "\u2039";
                matchLength = consumedCount;
                return;

            case 0x8C:
                match = "\u0152";
                matchLength = consumedCount;
                return;

            case 0x8E:
                match = "\u017D";
                matchLength = consumedCount;
                return;

            case 0x91:
                match = "\u2018";
                matchLength = consumedCount;
                return;

            case 0x92:
                match = "\u2019";
                matchLength = consumedCount;
                return;

            case 0x93:
                match = "\u201C";
                matchLength = consumedCount;
                return;

            case 0x94:
                match = "\u201D";
                matchLength = consumedCount;
                return;

            case 0x95:
                match = "\u2022";
                matchLength = consumedCount;
                return;

            case 0x96:
                match = "\u2013";
                matchLength = consumedCount;
                return;

            case 0x97:
                match = "\u2014";
                matchLength = consumedCount;
                return;

            case 0x98:
                match = "\u20DC";
                matchLength = consumedCount;
                return;

            case 0x99:
                match = "\u2122";
                matchLength = consumedCount;
                return;

            case 0x9A:
                match = "\u0161";
                matchLength = consumedCount;
                return;

            case 0x9B:
                match = "\u203A";
                matchLength = consumedCount;
                return;

            case 0x9C:
                match = "\u0153";
                matchLength = consumedCount;
                return;

            case 0x9E:
                match = "\u017E";
                matchLength = consumedCount;
                return;

            case 0x9F:
                match = "\u0178";
                matchLength = consumedCount;
                return;

            default:
                break;
        }
        match = new String(Character.toChars(code));
        matchLength = consumedCount;
    }

    public boolean parseNumeric(final int current) {
        consumedCount++;
        switch (state) {
            case STATE_START:
                if ('X' == current || 'x' == current) {
                    state = STATE_HEXADECIMAL_CHAR;
                    return true;
                }
                if ('0' <= current && current <= '9') {
                    state = STATE_DECIMAL_CHAR;
                    code = (code * 10) + current - 0x30;
                    return true;
                }
                break;
            case STATE_HEXADECIMAL_CHAR:
                if ('0' <= current && current <= '9') {
                    code = (code * 16) + current - 0x30;
                    return true;
                }
                if ('A' <= current && current <= 'F') {
                    code = (code * 16) + current - 0x37;
                    return true;
                }
                if ('a' <= current && current <= 'f') {
                    code = (code * 16) + current - 0x57;
                    return true;
                }
                if (';' == current) {
                    setMatchFromCode();
                    return false;
                }

                state = STATE_NUMERIC_CHAR_END_SEMICOLON_MISSING;
                setMatchFromCode();
                matchLength = consumedCount - 1;
                break;
            case STATE_DECIMAL_CHAR:
                if ('0' <= current && current <= '9') {
                    code = (code * 10) + current - 0x30;
                    return true;
                }
                if (';' == current) {
                    setMatchFromCode();
                    return false;
                }

                state = STATE_NUMERIC_CHAR_END_SEMICOLON_MISSING;
                setMatchFromCode();
                matchLength = consumedCount - 1;
                break;
        }
        return false;
    }

    private boolean parse1(final int current) {
        consumedCount++;
        switch (state) {
            case 0:
                switch (current) {
                    case 'A' :
                        state = 1;
                        return true;
                    case 'B' :
                        state = 77;
                        return true;
                    case 'C' :
                        state = 126;
                        return true;
                    case 'D' :
                        state = 342;
                        return true;
                    case 'E' :
                        state = 666;
                        return true;
                    case 'F' :
                        state = 788;
                        return true;
                    case 'G' :
                        state = 842;
                        return true;
                    case 'H' :
                        state = 929;
                        return true;
                    case 'I' :
                        state = 995;
                        return true;
                    case 'J' :
                        state = 1096;
                        return true;
                    case 'K' :
                        state = 1118;
                        return true;
                    case 'L' :
                        state = 1143;
                        return true;
                    case 'M' :
                        state = 1514;
                        return true;
                    case 'N' :
                        state = 1553;
                        return true;
                    case 'O' :
                        state = 2064;
                        return true;
                    case 'P' :
                        state = 2173;
                        return true;
                    case 'Q' :
                        state = 2258;
                        return true;
                    case 'R' :
                        state = 2270;
                        return true;
                    case 'S' :
                        state = 2525;
                        return true;
                    case 'T' :
                        state = 2722;
                        return true;
                    case 'U' :
                        state = 2817;
                        return true;
                    case 'V' :
                        state = 2997;
                        return true;
                    case 'W' :
                        state = 3067;
                        return true;
                    case 'X' :
                        state = 3084;
                        return true;
                    case 'Y' :
                        state = 3094;
                        return true;
                    case 'Z' :
                        state = 3125;
                        return true;
                    case 'a' :
                        state = 3166;
                        return true;
                    case 'b' :
                        state = 3311;
                        return true;
                    case 'c' :
                        state = 3623;
                        return true;
                    case 'd' :
                        state = 3915;
                        return true;
                    case 'e' :
                        state = 4156;
                        return true;
                    case 'f' :
                        state = 4342;
                        return true;
                    case 'g' :
                        state = 4442;
                        return true;
                    case 'h' :
                        state = 4588;
                        return true;
                    case 'i' :
                        state = 4704;
                        return true;
                    case 'j' :
                        state = 4847;
                        return true;
                    case 'k' :
                        state = 4873;
                        return true;
                    case 'l' :
                        state = 4904;
                        return true;
                    case 'm' :
                        state = 5397;
                        return true;
                    case 'n' :
                        state = 5510;
                        return true;
                    case 'o' :
                        state = 5945;
                        return true;
                    case 'p' :
                        state = 6092;
                        return true;
                    case 'q' :
                        state = 6272;
                        return true;
                    case 'r' :
                        state = 6309;
                        return true;
                    case 's' :
                        state = 6642;
                        return true;
                    case 't' :
                        state = 7043;
                        return true;
                    case 'u' :
                        state = 7238;
                        return true;
                    case 'v' :
                        state = 7408;
                        return true;
                    case 'w' :
                        state = 7562;
                        return true;
                    case 'x' :
                        state = 7593;
                        return true;
                    case 'y' :
                        state = 7672;
                        return true;
                    case 'z' :
                        state = 7702;
                        return true;
                }
                break;
            case 1:
                switch (current) {
                    case 'E' :
                        state = 2;
                        return true;
                    case 'M' :
                        state = 6;
                        return true;
                    case 'a' :
                        state = 8;
                        return true;
                    case 'b' :
                        state = 13;
                        return true;
                    case 'c' :
                        state = 18;
                        return true;
                    case 'f' :
                        state = 23;
                        return true;
                    case 'g' :
                        state = 25;
                        return true;
                    case 'l' :
                        state = 30;
                        return true;
                    case 'm' :
                        state = 34;
                        return true;
                    case 'n' :
                        state = 38;
                        return true;
                    case 'o' :
                        state = 40;
                        return true;
                    case 'p' :
                        state = 46;
                        return true;
                    case 'r' :
                        state = 58;
                        return true;
                    case 's' :
                        state = 62;
                        return true;
                    case 't' :
                        state = 69;
                        return true;
                    case 'u' :
                        state = 74;
                        return true;
                }
                break;
            case 2:
                if ('l' == current) {
                    state = 3;
                    return true;
                }
                break;
            case 3:
                if ('i' == current) {
                    state = 4;
                    return true;
                }
                break;
            case 4:
                // AElig
                if ('g' == current) {
                    match = "\u00C6";
                    matchLength = consumedCount;
                    state = 5;
                    return true;
                }
                break;
            case 5:
                // AElig;
                if (';' == current) {
                    match = "\u00C6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 6:
                // AMP
                if ('P' == current) {
                    match = "&";
                    matchLength = consumedCount;
                    state = 7;
                    return true;
                }
                break;
            case 7:
                // AMP;
                if (';' == current) {
                    match = "&";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 8:
                if ('c' == current) {
                    state = 9;
                    return true;
                }
                break;
            case 9:
                if ('u' == current) {
                    state = 10;
                    return true;
                }
                break;
            case 10:
                if ('t' == current) {
                    state = 11;
                    return true;
                }
                break;
            case 11:
                // Aacute
                if ('e' == current) {
                    match = "\u00C1";
                    matchLength = consumedCount;
                    state = 12;
                    return true;
                }
                break;
            case 12:
                // Aacute;
                if (';' == current) {
                    match = "\u00C1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 13:
                if ('r' == current) {
                    state = 14;
                    return true;
                }
                break;
            case 14:
                if ('e' == current) {
                    state = 15;
                    return true;
                }
                break;
            case 15:
                if ('v' == current) {
                    state = 16;
                    return true;
                }
                break;
            case 16:
                if ('e' == current) {
                    state = 17;
                    return true;
                }
                break;
            case 17:
                // Abreve;
                if (';' == current) {
                    match = "\u0102";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 18:
                if ('i' == current) {
                    state = 19;
                    return true;
                }
                else if ('y' == current) {
                    state = 22;
                    return true;
                }
                break;
            case 19:
                if ('r' == current) {
                    state = 20;
                    return true;
                }
                break;
            case 20:
                // Acirc
                if ('c' == current) {
                    match = "\u00C2";
                    matchLength = consumedCount;
                    state = 21;
                    return true;
                }
                break;
            case 21:
                // Acirc;
                if (';' == current) {
                    match = "\u00C2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 22:
                // Acy;
                if (';' == current) {
                    match = "\u0410";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 23:
                if ('r' == current) {
                    state = 24;
                    return true;
                }
                break;
            case 24:
                // Afr;
                if (';' == current) {
                    match = "\uD835\uDD04";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 25:
                if ('r' == current) {
                    state = 26;
                    return true;
                }
                break;
            case 26:
                if ('a' == current) {
                    state = 27;
                    return true;
                }
                break;
            case 27:
                if ('v' == current) {
                    state = 28;
                    return true;
                }
                break;
            case 28:
                // Agrave
                if ('e' == current) {
                    match = "\u00C0";
                    matchLength = consumedCount;
                    state = 29;
                    return true;
                }
                break;
            case 29:
                // Agrave;
                if (';' == current) {
                    match = "\u00C0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 30:
                if ('p' == current) {
                    state = 31;
                    return true;
                }
                break;
            case 31:
                if ('h' == current) {
                    state = 32;
                    return true;
                }
                break;
            case 32:
                if ('a' == current) {
                    state = 33;
                    return true;
                }
                break;
            case 33:
                // Alpha;
                if (';' == current) {
                    match = "\u0391";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 34:
                if ('a' == current) {
                    state = 35;
                    return true;
                }
                break;
            case 35:
                if ('c' == current) {
                    state = 36;
                    return true;
                }
                break;
            case 36:
                if ('r' == current) {
                    state = 37;
                    return true;
                }
                break;
            case 37:
                // Amacr;
                if (';' == current) {
                    match = "\u0100";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 38:
                if ('d' == current) {
                    state = 39;
                    return true;
                }
                break;
            case 39:
                // And;
                if (';' == current) {
                    match = "\u2A53";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 40:
                if ('g' == current) {
                    state = 41;
                    return true;
                }
                else if ('p' == current) {
                    state = 44;
                    return true;
                }
                break;
            case 41:
                if ('o' == current) {
                    state = 42;
                    return true;
                }
                break;
            case 42:
                if ('n' == current) {
                    state = 43;
                    return true;
                }
                break;
            case 43:
                // Aogon;
                if (';' == current) {
                    match = "\u0104";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 44:
                if ('f' == current) {
                    state = 45;
                    return true;
                }
                break;
            case 45:
                // Aopf;
                if (';' == current) {
                    match = "\uD835\uDD38";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 46:
                if ('p' == current) {
                    state = 47;
                    return true;
                }
                break;
            case 47:
                if ('l' == current) {
                    state = 48;
                    return true;
                }
                break;
            case 48:
                if ('y' == current) {
                    state = 49;
                    return true;
                }
                break;
            case 49:
                if ('F' == current) {
                    state = 50;
                    return true;
                }
                break;
            case 50:
                if ('u' == current) {
                    state = 51;
                    return true;
                }
                break;
            case 51:
                if ('n' == current) {
                    state = 52;
                    return true;
                }
                break;
            case 52:
                if ('c' == current) {
                    state = 53;
                    return true;
                }
                break;
            case 53:
                if ('t' == current) {
                    state = 54;
                    return true;
                }
                break;
            case 54:
                if ('i' == current) {
                    state = 55;
                    return true;
                }
                break;
            case 55:
                if ('o' == current) {
                    state = 56;
                    return true;
                }
                break;
            case 56:
                if ('n' == current) {
                    state = 57;
                    return true;
                }
                break;
            case 57:
                // ApplyFunction;
                if (';' == current) {
                    match = "\u2061";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 58:
                if ('i' == current) {
                    state = 59;
                    return true;
                }
                break;
            case 59:
                if ('n' == current) {
                    state = 60;
                    return true;
                }
                break;
            case 60:
                // Aring
                if ('g' == current) {
                    match = "\u00C5";
                    matchLength = consumedCount;
                    state = 61;
                    return true;
                }
                break;
            case 61:
                // Aring;
                if (';' == current) {
                    match = "\u00C5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 62:
                if ('c' == current) {
                    state = 63;
                    return true;
                }
                else if ('s' == current) {
                    state = 65;
                    return true;
                }
                break;
            case 63:
                if ('r' == current) {
                    state = 64;
                    return true;
                }
                break;
            case 64:
                // Ascr;
                if (';' == current) {
                    match = "\uD835\uDC9C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 65:
                if ('i' == current) {
                    state = 66;
                    return true;
                }
                break;
            case 66:
                if ('g' == current) {
                    state = 67;
                    return true;
                }
                break;
            case 67:
                if ('n' == current) {
                    state = 68;
                    return true;
                }
                break;
            case 68:
                // Assign;
                if (';' == current) {
                    match = "\u2254";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 69:
                if ('i' == current) {
                    state = 70;
                    return true;
                }
                break;
            case 70:
                if ('l' == current) {
                    state = 71;
                    return true;
                }
                break;
            case 71:
                if ('d' == current) {
                    state = 72;
                    return true;
                }
                break;
            case 72:
                // Atilde
                if ('e' == current) {
                    match = "\u00C3";
                    matchLength = consumedCount;
                    state = 73;
                    return true;
                }
                break;
            case 73:
                // Atilde;
                if (';' == current) {
                    match = "\u00C3";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 74:
                if ('m' == current) {
                    state = 75;
                    return true;
                }
                break;
            case 75:
                // Auml
                if ('l' == current) {
                    match = "\u00C4";
                    matchLength = consumedCount;
                    state = 76;
                    return true;
                }
                break;
            case 76:
                // Auml;
                if (';' == current) {
                    match = "\u00C4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 77:
                switch (current) {
                    case 'a' :
                        state = 78;
                        return true;
                    case 'c' :
                        state = 91;
                        return true;
                    case 'e' :
                        state = 93;
                        return true;
                    case 'f' :
                        state = 109;
                        return true;
                    case 'o' :
                        state = 111;
                        return true;
                    case 'r' :
                        state = 114;
                        return true;
                    case 's' :
                        state = 118;
                        return true;
                    case 'u' :
                        state = 121;
                        return true;
                }
                break;
            case 78:
                if ('c' == current) {
                    state = 79;
                    return true;
                }
                else if ('r' == current) {
                    state = 86;
                    return true;
                }
                break;
            case 79:
                if ('k' == current) {
                    state = 80;
                    return true;
                }
                break;
            case 80:
                if ('s' == current) {
                    state = 81;
                    return true;
                }
                break;
            case 81:
                if ('l' == current) {
                    state = 82;
                    return true;
                }
                break;
            case 82:
                if ('a' == current) {
                    state = 83;
                    return true;
                }
                break;
            case 83:
                if ('s' == current) {
                    state = 84;
                    return true;
                }
                break;
            case 84:
                if ('h' == current) {
                    state = 85;
                    return true;
                }
                break;
            case 85:
                // Backslash;
                if (';' == current) {
                    match = "\u2216";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 86:
                if ('v' == current) {
                    state = 87;
                    return true;
                }
                else if ('w' == current) {
                    state = 88;
                    return true;
                }
                break;
            case 87:
                // Barv;
                if (';' == current) {
                    match = "\u2AE7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 88:
                if ('e' == current) {
                    state = 89;
                    return true;
                }
                break;
            case 89:
                if ('d' == current) {
                    state = 90;
                    return true;
                }
                break;
            case 90:
                // Barwed;
                if (';' == current) {
                    match = "\u2306";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 91:
                if ('y' == current) {
                    state = 92;
                    return true;
                }
                break;
            case 92:
                // Bcy;
                if (';' == current) {
                    match = "\u0411";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 93:
                switch (current) {
                    case 'c' :
                        state = 94;
                        return true;
                    case 'r' :
                        state = 99;
                        return true;
                    case 't' :
                        state = 107;
                        return true;
                }
                break;
            case 94:
                if ('a' == current) {
                    state = 95;
                    return true;
                }
                break;
            case 95:
                if ('u' == current) {
                    state = 96;
                    return true;
                }
                break;
            case 96:
                if ('s' == current) {
                    state = 97;
                    return true;
                }
                break;
            case 97:
                if ('e' == current) {
                    state = 98;
                    return true;
                }
                break;
            case 98:
                // Because;
                if (';' == current) {
                    match = "\u2235";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 99:
                if ('n' == current) {
                    state = 100;
                    return true;
                }
                break;
            case 100:
                if ('o' == current) {
                    state = 101;
                    return true;
                }
                break;
            case 101:
                if ('u' == current) {
                    state = 102;
                    return true;
                }
                break;
            case 102:
                if ('l' == current) {
                    state = 103;
                    return true;
                }
                break;
            case 103:
                if ('l' == current) {
                    state = 104;
                    return true;
                }
                break;
            case 104:
                if ('i' == current) {
                    state = 105;
                    return true;
                }
                break;
            case 105:
                if ('s' == current) {
                    state = 106;
                    return true;
                }
                break;
            case 106:
                // Bernoullis;
                if (';' == current) {
                    match = "\u212C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 107:
                if ('a' == current) {
                    state = 108;
                    return true;
                }
                break;
            case 108:
                // Beta;
                if (';' == current) {
                    match = "\u0392";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 109:
                if ('r' == current) {
                    state = 110;
                    return true;
                }
                break;
            case 110:
                // Bfr;
                if (';' == current) {
                    match = "\uD835\uDD05";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 111:
                if ('p' == current) {
                    state = 112;
                    return true;
                }
                break;
            case 112:
                if ('f' == current) {
                    state = 113;
                    return true;
                }
                break;
            case 113:
                // Bopf;
                if (';' == current) {
                    match = "\uD835\uDD39";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 114:
                if ('e' == current) {
                    state = 115;
                    return true;
                }
                break;
            case 115:
                if ('v' == current) {
                    state = 116;
                    return true;
                }
                break;
            case 116:
                if ('e' == current) {
                    state = 117;
                    return true;
                }
                break;
            case 117:
                // Breve;
                if (';' == current) {
                    match = "\u02D8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 118:
                if ('c' == current) {
                    state = 119;
                    return true;
                }
                break;
            case 119:
                if ('r' == current) {
                    state = 120;
                    return true;
                }
                break;
            case 120:
                // Bscr;
                if (';' == current) {
                    match = "\u212C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 121:
                if ('m' == current) {
                    state = 122;
                    return true;
                }
                break;
            case 122:
                if ('p' == current) {
                    state = 123;
                    return true;
                }
                break;
            case 123:
                if ('e' == current) {
                    state = 124;
                    return true;
                }
                break;
            case 124:
                if ('q' == current) {
                    state = 125;
                    return true;
                }
                break;
            case 125:
                // Bumpeq;
                if (';' == current) {
                    match = "\u224E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 126:
                switch (current) {
                    case 'H' :
                        state = 127;
                        return true;
                    case 'O' :
                        state = 130;
                        return true;
                    case 'a' :
                        state = 133;
                        return true;
                    case 'c' :
                        state = 161;
                        return true;
                    case 'd' :
                        state = 178;
                        return true;
                    case 'e' :
                        state = 181;
                        return true;
                    case 'f' :
                        state = 194;
                        return true;
                    case 'h' :
                        state = 196;
                        return true;
                    case 'i' :
                        state = 198;
                        return true;
                    case 'l' :
                        state = 220;
                        return true;
                    case 'o' :
                        state = 266;
                        return true;
                    case 'r' :
                        state = 330;
                        return true;
                    case 's' :
                        state = 334;
                        return true;
                    case 'u' :
                        state = 337;
                        return true;
                }
                break;
            case 127:
                if ('c' == current) {
                    state = 128;
                    return true;
                }
                break;
            case 128:
                if ('y' == current) {
                    state = 129;
                    return true;
                }
                break;
            case 129:
                // CHcy;
                if (';' == current) {
                    match = "\u0427";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 130:
                if ('P' == current) {
                    state = 131;
                    return true;
                }
                break;
            case 131:
                // COPY
                if ('Y' == current) {
                    match = "\u00A9";
                    matchLength = consumedCount;
                    state = 132;
                    return true;
                }
                break;
            case 132:
                // COPY;
                if (';' == current) {
                    match = "\u00A9";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 133:
                switch (current) {
                    case 'c' :
                        state = 134;
                        return true;
                    case 'p' :
                        state = 138;
                        return true;
                    case 'y' :
                        state = 156;
                        return true;
                }
                break;
            case 134:
                if ('u' == current) {
                    state = 135;
                    return true;
                }
                break;
            case 135:
                if ('t' == current) {
                    state = 136;
                    return true;
                }
                break;
            case 136:
                if ('e' == current) {
                    state = 137;
                    return true;
                }
                break;
            case 137:
                // Cacute;
                if (';' == current) {
                    match = "\u0106";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 138:
                // Cap;
                if (';' == current) {
                    match = "\u22D2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('i' == current) {
                    state = 139;
                    return true;
                }
                break;
            case 139:
                if ('t' == current) {
                    state = 140;
                    return true;
                }
                break;
            case 140:
                if ('a' == current) {
                    state = 141;
                    return true;
                }
                break;
            case 141:
                if ('l' == current) {
                    state = 142;
                    return true;
                }
                break;
            case 142:
                if ('D' == current) {
                    state = 143;
                    return true;
                }
                break;
            case 143:
                if ('i' == current) {
                    state = 144;
                    return true;
                }
                break;
            case 144:
                if ('f' == current) {
                    state = 145;
                    return true;
                }
                break;
            case 145:
                if ('f' == current) {
                    state = 146;
                    return true;
                }
                break;
            case 146:
                if ('e' == current) {
                    state = 147;
                    return true;
                }
                break;
            case 147:
                if ('r' == current) {
                    state = 148;
                    return true;
                }
                break;
            case 148:
                if ('e' == current) {
                    state = 149;
                    return true;
                }
                break;
            case 149:
                if ('n' == current) {
                    state = 150;
                    return true;
                }
                break;
            case 150:
                if ('t' == current) {
                    state = 151;
                    return true;
                }
                break;
            case 151:
                if ('i' == current) {
                    state = 152;
                    return true;
                }
                break;
            case 152:
                if ('a' == current) {
                    state = 153;
                    return true;
                }
                break;
            case 153:
                if ('l' == current) {
                    state = 154;
                    return true;
                }
                break;
            case 154:
                if ('D' == current) {
                    state = 155;
                    return true;
                }
                break;
            case 155:
                // CapitalDifferentialD;
                if (';' == current) {
                    match = "\u2145";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 156:
                if ('l' == current) {
                    state = 157;
                    return true;
                }
                break;
            case 157:
                if ('e' == current) {
                    state = 158;
                    return true;
                }
                break;
            case 158:
                if ('y' == current) {
                    state = 159;
                    return true;
                }
                break;
            case 159:
                if ('s' == current) {
                    state = 160;
                    return true;
                }
                break;
            case 160:
                // Cayleys;
                if (';' == current) {
                    match = "\u212D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 161:
                switch (current) {
                    case 'a' :
                        state = 162;
                        return true;
                    case 'e' :
                        state = 166;
                        return true;
                    case 'i' :
                        state = 170;
                        return true;
                    case 'o' :
                        state = 173;
                        return true;
                }
                break;
            case 162:
                if ('r' == current) {
                    state = 163;
                    return true;
                }
                break;
            case 163:
                if ('o' == current) {
                    state = 164;
                    return true;
                }
                break;
            case 164:
                if ('n' == current) {
                    state = 165;
                    return true;
                }
                break;
            case 165:
                // Ccaron;
                if (';' == current) {
                    match = "\u010C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 166:
                if ('d' == current) {
                    state = 167;
                    return true;
                }
                break;
            case 167:
                if ('i' == current) {
                    state = 168;
                    return true;
                }
                break;
            case 168:
                // Ccedil
                if ('l' == current) {
                    match = "\u00C7";
                    matchLength = consumedCount;
                    state = 169;
                    return true;
                }
                break;
            case 169:
                // Ccedil;
                if (';' == current) {
                    match = "\u00C7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 170:
                if ('r' == current) {
                    state = 171;
                    return true;
                }
                break;
            case 171:
                if ('c' == current) {
                    state = 172;
                    return true;
                }
                break;
            case 172:
                // Ccirc;
                if (';' == current) {
                    match = "\u0108";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 173:
                if ('n' == current) {
                    state = 174;
                    return true;
                }
                break;
            case 174:
                if ('i' == current) {
                    state = 175;
                    return true;
                }
                break;
            case 175:
                if ('n' == current) {
                    state = 176;
                    return true;
                }
                break;
            case 176:
                if ('t' == current) {
                    state = 177;
                    return true;
                }
                break;
            case 177:
                // Cconint;
                if (';' == current) {
                    match = "\u2230";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 178:
                if ('o' == current) {
                    state = 179;
                    return true;
                }
                break;
            case 179:
                if ('t' == current) {
                    state = 180;
                    return true;
                }
                break;
            case 180:
                // Cdot;
                if (';' == current) {
                    match = "\u010A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 181:
                if ('d' == current) {
                    state = 182;
                    return true;
                }
                else if ('n' == current) {
                    state = 187;
                    return true;
                }
                break;
            case 182:
                if ('i' == current) {
                    state = 183;
                    return true;
                }
                break;
            case 183:
                if ('l' == current) {
                    state = 184;
                    return true;
                }
                break;
            case 184:
                if ('l' == current) {
                    state = 185;
                    return true;
                }
                break;
            case 185:
                if ('a' == current) {
                    state = 186;
                    return true;
                }
                break;
            case 186:
                // Cedilla;
                if (';' == current) {
                    match = "\u00B8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 187:
                if ('t' == current) {
                    state = 188;
                    return true;
                }
                break;
            case 188:
                if ('e' == current) {
                    state = 189;
                    return true;
                }
                break;
            case 189:
                if ('r' == current) {
                    state = 190;
                    return true;
                }
                break;
            case 190:
                if ('D' == current) {
                    state = 191;
                    return true;
                }
                break;
            case 191:
                if ('o' == current) {
                    state = 192;
                    return true;
                }
                break;
            case 192:
                if ('t' == current) {
                    state = 193;
                    return true;
                }
                break;
            case 193:
                // CenterDot;
                if (';' == current) {
                    match = "\u00B7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 194:
                if ('r' == current) {
                    state = 195;
                    return true;
                }
                break;
            case 195:
                // Cfr;
                if (';' == current) {
                    match = "\u212D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 196:
                if ('i' == current) {
                    state = 197;
                    return true;
                }
                break;
            case 197:
                // Chi;
                if (';' == current) {
                    match = "\u03A7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 198:
                if ('r' == current) {
                    state = 199;
                    return true;
                }
                break;
            case 199:
                if ('c' == current) {
                    state = 200;
                    return true;
                }
                break;
            case 200:
                if ('l' == current) {
                    state = 201;
                    return true;
                }
                break;
            case 201:
                if ('e' == current) {
                    state = 202;
                    return true;
                }
                break;
            case 202:
                switch (current) {
                    case 'D' :
                        state = 203;
                        return true;
                    case 'M' :
                        state = 206;
                        return true;
                    case 'P' :
                        state = 211;
                        return true;
                    case 'T' :
                        state = 215;
                        return true;
                }
                break;
            case 203:
                if ('o' == current) {
                    state = 204;
                    return true;
                }
                break;
            case 204:
                if ('t' == current) {
                    state = 205;
                    return true;
                }
                break;
            case 205:
                // CircleDot;
                if (';' == current) {
                    match = "\u2299";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 206:
                if ('i' == current) {
                    state = 207;
                    return true;
                }
                break;
            case 207:
                if ('n' == current) {
                    state = 208;
                    return true;
                }
                break;
            case 208:
                if ('u' == current) {
                    state = 209;
                    return true;
                }
                break;
            case 209:
                if ('s' == current) {
                    state = 210;
                    return true;
                }
                break;
            case 210:
                // CircleMinus;
                if (';' == current) {
                    match = "\u2296";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 211:
                if ('l' == current) {
                    state = 212;
                    return true;
                }
                break;
            case 212:
                if ('u' == current) {
                    state = 213;
                    return true;
                }
                break;
            case 213:
                if ('s' == current) {
                    state = 214;
                    return true;
                }
                break;
            case 214:
                // CirclePlus;
                if (';' == current) {
                    match = "\u2295";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 215:
                if ('i' == current) {
                    state = 216;
                    return true;
                }
                break;
            case 216:
                if ('m' == current) {
                    state = 217;
                    return true;
                }
                break;
            case 217:
                if ('e' == current) {
                    state = 218;
                    return true;
                }
                break;
            case 218:
                if ('s' == current) {
                    state = 219;
                    return true;
                }
                break;
            case 219:
                // CircleTimes;
                if (';' == current) {
                    match = "\u2297";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 220:
                if ('o' == current) {
                    state = 221;
                    return true;
                }
                break;
            case 221:
                if ('c' == current) {
                    state = 222;
                    return true;
                }
                else if ('s' == current) {
                    state = 243;
                    return true;
                }
                break;
            case 222:
                if ('k' == current) {
                    state = 223;
                    return true;
                }
                break;
            case 223:
                if ('w' == current) {
                    state = 224;
                    return true;
                }
                break;
            case 224:
                if ('i' == current) {
                    state = 225;
                    return true;
                }
                break;
            case 225:
                if ('s' == current) {
                    state = 226;
                    return true;
                }
                break;
            case 226:
                if ('e' == current) {
                    state = 227;
                    return true;
                }
                break;
            case 227:
                if ('C' == current) {
                    state = 228;
                    return true;
                }
                break;
            case 228:
                if ('o' == current) {
                    state = 229;
                    return true;
                }
                break;
            case 229:
                if ('n' == current) {
                    state = 230;
                    return true;
                }
                break;
            case 230:
                if ('t' == current) {
                    state = 231;
                    return true;
                }
                break;
            case 231:
                if ('o' == current) {
                    state = 232;
                    return true;
                }
                break;
            case 232:
                if ('u' == current) {
                    state = 233;
                    return true;
                }
                break;
            case 233:
                if ('r' == current) {
                    state = 234;
                    return true;
                }
                break;
            case 234:
                if ('I' == current) {
                    state = 235;
                    return true;
                }
                break;
            case 235:
                if ('n' == current) {
                    state = 236;
                    return true;
                }
                break;
            case 236:
                if ('t' == current) {
                    state = 237;
                    return true;
                }
                break;
            case 237:
                if ('e' == current) {
                    state = 238;
                    return true;
                }
                break;
            case 238:
                if ('g' == current) {
                    state = 239;
                    return true;
                }
                break;
            case 239:
                if ('r' == current) {
                    state = 240;
                    return true;
                }
                break;
            case 240:
                if ('a' == current) {
                    state = 241;
                    return true;
                }
                break;
            case 241:
                if ('l' == current) {
                    state = 242;
                    return true;
                }
                break;
            case 242:
                // ClockwiseContourIntegral;
                if (';' == current) {
                    match = "\u2232";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 243:
                if ('e' == current) {
                    state = 244;
                    return true;
                }
                break;
            case 244:
                if ('C' == current) {
                    state = 245;
                    return true;
                }
                break;
            case 245:
                if ('u' == current) {
                    state = 246;
                    return true;
                }
                break;
            case 246:
                if ('r' == current) {
                    state = 247;
                    return true;
                }
                break;
            case 247:
                if ('l' == current) {
                    state = 248;
                    return true;
                }
                break;
            case 248:
                if ('y' == current) {
                    state = 249;
                    return true;
                }
                break;
            case 249:
                if ('D' == current) {
                    state = 250;
                    return true;
                }
                else if ('Q' == current) {
                    state = 261;
                    return true;
                }
                break;
            case 250:
                if ('o' == current) {
                    state = 251;
                    return true;
                }
                break;
            case 251:
                if ('u' == current) {
                    state = 252;
                    return true;
                }
                break;
            case 252:
                if ('b' == current) {
                    state = 253;
                    return true;
                }
                break;
            case 253:
                if ('l' == current) {
                    state = 254;
                    return true;
                }
                break;
            case 254:
                if ('e' == current) {
                    state = 255;
                    return true;
                }
                break;
            case 255:
                if ('Q' == current) {
                    state = 256;
                    return true;
                }
                break;
            case 256:
                if ('u' == current) {
                    state = 257;
                    return true;
                }
                break;
            case 257:
                if ('o' == current) {
                    state = 258;
                    return true;
                }
                break;
            case 258:
                if ('t' == current) {
                    state = 259;
                    return true;
                }
                break;
            case 259:
                if ('e' == current) {
                    state = 260;
                    return true;
                }
                break;
            case 260:
                // CloseCurlyDoubleQuote;
                if (';' == current) {
                    match = "\u201D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 261:
                if ('u' == current) {
                    state = 262;
                    return true;
                }
                break;
            case 262:
                if ('o' == current) {
                    state = 263;
                    return true;
                }
                break;
            case 263:
                if ('t' == current) {
                    state = 264;
                    return true;
                }
                break;
            case 264:
                if ('e' == current) {
                    state = 265;
                    return true;
                }
                break;
            case 265:
                // CloseCurlyQuote;
                if (';' == current) {
                    match = "\u2019";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 266:
                switch (current) {
                    case 'l' :
                        state = 267;
                        return true;
                    case 'n' :
                        state = 271;
                        return true;
                    case 'p' :
                        state = 293;
                        return true;
                    case 'u' :
                        state = 301;
                        return true;
                }
                break;
            case 267:
                if ('o' == current) {
                    state = 268;
                    return true;
                }
                break;
            case 268:
                if ('n' == current) {
                    state = 269;
                    return true;
                }
                break;
            case 269:
                // Colon;
                if (';' == current) {
                    match = "\u2237";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state = 270;
                    return true;
                }
                break;
            case 270:
                // Colone;
                if (';' == current) {
                    match = "\u2A74";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 271:
                switch (current) {
                    case 'g' :
                        state = 272;
                        return true;
                    case 'i' :
                        state = 278;
                        return true;
                    case 't' :
                        state = 281;
                        return true;
                }
                break;
            case 272:
                if ('r' == current) {
                    state = 273;
                    return true;
                }
                break;
            case 273:
                if ('u' == current) {
                    state = 274;
                    return true;
                }
                break;
            case 274:
                if ('e' == current) {
                    state = 275;
                    return true;
                }
                break;
            case 275:
                if ('n' == current) {
                    state = 276;
                    return true;
                }
                break;
            case 276:
                if ('t' == current) {
                    state = 277;
                    return true;
                }
                break;
            case 277:
                // Congruent;
                if (';' == current) {
                    match = "\u2261";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 278:
                if ('n' == current) {
                    state = 279;
                    return true;
                }
                break;
            case 279:
                if ('t' == current) {
                    state = 280;
                    return true;
                }
                break;
            case 280:
                // Conint;
                if (';' == current) {
                    match = "\u222F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 281:
                if ('o' == current) {
                    state = 282;
                    return true;
                }
                break;
            case 282:
                if ('u' == current) {
                    state = 283;
                    return true;
                }
                break;
            case 283:
                if ('r' == current) {
                    state = 284;
                    return true;
                }
                break;
            case 284:
                if ('I' == current) {
                    state = 285;
                    return true;
                }
                break;
            case 285:
                if ('n' == current) {
                    state = 286;
                    return true;
                }
                break;
            case 286:
                if ('t' == current) {
                    state = 287;
                    return true;
                }
                break;
            case 287:
                if ('e' == current) {
                    state = 288;
                    return true;
                }
                break;
            case 288:
                if ('g' == current) {
                    state = 289;
                    return true;
                }
                break;
            case 289:
                if ('r' == current) {
                    state = 290;
                    return true;
                }
                break;
            case 290:
                if ('a' == current) {
                    state = 291;
                    return true;
                }
                break;
            case 291:
                if ('l' == current) {
                    state = 292;
                    return true;
                }
                break;
            case 292:
                // ContourIntegral;
                if (';' == current) {
                    match = "\u222E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 293:
                if ('f' == current) {
                    state = 294;
                    return true;
                }
                else if ('r' == current) {
                    state = 295;
                    return true;
                }
                break;
            case 294:
                // Copf;
                if (';' == current) {
                    match = "\u2102";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 295:
                if ('o' == current) {
                    state = 296;
                    return true;
                }
                break;
            case 296:
                if ('d' == current) {
                    state = 297;
                    return true;
                }
                break;
            case 297:
                if ('u' == current) {
                    state = 298;
                    return true;
                }
                break;
            case 298:
                if ('c' == current) {
                    state = 299;
                    return true;
                }
                break;
            case 299:
                if ('t' == current) {
                    state = 300;
                    return true;
                }
                break;
            case 300:
                // Coproduct;
                if (';' == current) {
                    match = "\u2210";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 301:
                if ('n' == current) {
                    state = 302;
                    return true;
                }
                break;
            case 302:
                if ('t' == current) {
                    state = 303;
                    return true;
                }
                break;
            case 303:
                if ('e' == current) {
                    state = 304;
                    return true;
                }
                break;
            case 304:
                if ('r' == current) {
                    state = 305;
                    return true;
                }
                break;
            case 305:
                if ('C' == current) {
                    state = 306;
                    return true;
                }
                break;
            case 306:
                if ('l' == current) {
                    state = 307;
                    return true;
                }
                break;
            case 307:
                if ('o' == current) {
                    state = 308;
                    return true;
                }
                break;
            case 308:
                if ('c' == current) {
                    state = 309;
                    return true;
                }
                break;
            case 309:
                if ('k' == current) {
                    state = 310;
                    return true;
                }
                break;
            case 310:
                if ('w' == current) {
                    state = 311;
                    return true;
                }
                break;
            case 311:
                if ('i' == current) {
                    state = 312;
                    return true;
                }
                break;
            case 312:
                if ('s' == current) {
                    state = 313;
                    return true;
                }
                break;
            case 313:
                if ('e' == current) {
                    state = 314;
                    return true;
                }
                break;
            case 314:
                if ('C' == current) {
                    state = 315;
                    return true;
                }
                break;
            case 315:
                if ('o' == current) {
                    state = 316;
                    return true;
                }
                break;
            case 316:
                if ('n' == current) {
                    state = 317;
                    return true;
                }
                break;
            case 317:
                if ('t' == current) {
                    state = 318;
                    return true;
                }
                break;
            case 318:
                if ('o' == current) {
                    state = 319;
                    return true;
                }
                break;
            case 319:
                if ('u' == current) {
                    state = 320;
                    return true;
                }
                break;
            case 320:
                if ('r' == current) {
                    state = 321;
                    return true;
                }
                break;
            case 321:
                if ('I' == current) {
                    state = 322;
                    return true;
                }
                break;
            case 322:
                if ('n' == current) {
                    state = 323;
                    return true;
                }
                break;
            case 323:
                if ('t' == current) {
                    state = 324;
                    return true;
                }
                break;
            case 324:
                if ('e' == current) {
                    state = 325;
                    return true;
                }
                break;
            case 325:
                if ('g' == current) {
                    state = 326;
                    return true;
                }
                break;
            case 326:
                if ('r' == current) {
                    state = 327;
                    return true;
                }
                break;
            case 327:
                if ('a' == current) {
                    state = 328;
                    return true;
                }
                break;
            case 328:
                if ('l' == current) {
                    state = 329;
                    return true;
                }
                break;
            case 329:
                // CounterClockwiseContourIntegral;
                if (';' == current) {
                    match = "\u2233";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 330:
                if ('o' == current) {
                    state = 331;
                    return true;
                }
                break;
            case 331:
                if ('s' == current) {
                    state = 332;
                    return true;
                }
                break;
            case 332:
                if ('s' == current) {
                    state = 333;
                    return true;
                }
                break;
            case 333:
                // Cross;
                if (';' == current) {
                    match = "\u2A2F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 334:
                if ('c' == current) {
                    state = 335;
                    return true;
                }
                break;
            case 335:
                if ('r' == current) {
                    state = 336;
                    return true;
                }
                break;
            case 336:
                // Cscr;
                if (';' == current) {
                    match = "\uD835\uDC9E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 337:
                if ('p' == current) {
                    state = 338;
                    return true;
                }
                break;
            case 338:
                // Cup;
                if (';' == current) {
                    match = "\u22D3";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('C' == current) {
                    state = 339;
                    return true;
                }
                break;
            case 339:
                if ('a' == current) {
                    state = 340;
                    return true;
                }
                break;
            case 340:
                if ('p' == current) {
                    state = 341;
                    return true;
                }
                break;
            case 341:
                // CupCap;
                if (';' == current) {
                    match = "\u224D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 342:
                switch (current) {
                    case 'D' :
                        state = 343;
                        return true;
                    case 'J' :
                        state = 350;
                        return true;
                    case 'S' :
                        state = 353;
                        return true;
                    case 'Z' :
                        state = 356;
                        return true;
                    case 'a' :
                        state = 359;
                        return true;
                    case 'c' :
                        state = 369;
                        return true;
                    case 'e' :
                        state = 375;
                        return true;
                    case 'f' :
                        state = 379;
                        return true;
                    case 'i' :
                        state = 381;
                        return true;
                    case 'o' :
                        state = 433;
                        return true;
                    case 's' :
                        state = 659;
                        return true;
                }
                break;
            case 343:
                // DD;
                if (';' == current) {
                    match = "\u2145";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('o' == current) {
                    state = 344;
                    return true;
                }
                break;
            case 344:
                if ('t' == current) {
                    state = 345;
                    return true;
                }
                break;
            case 345:
                if ('r' == current) {
                    state = 346;
                    return true;
                }
                break;
            case 346:
                if ('a' == current) {
                    state = 347;
                    return true;
                }
                break;
            case 347:
                if ('h' == current) {
                    state = 348;
                    return true;
                }
                break;
            case 348:
                if ('d' == current) {
                    state = 349;
                    return true;
                }
                break;
            case 349:
                // DDotrahd;
                if (';' == current) {
                    match = "\u2911";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 350:
                if ('c' == current) {
                    state = 351;
                    return true;
                }
                break;
            case 351:
                if ('y' == current) {
                    state = 352;
                    return true;
                }
                break;
            case 352:
                // DJcy;
                if (';' == current) {
                    match = "\u0402";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 353:
                if ('c' == current) {
                    state = 354;
                    return true;
                }
                break;
            case 354:
                if ('y' == current) {
                    state = 355;
                    return true;
                }
                break;
            case 355:
                // DScy;
                if (';' == current) {
                    match = "\u0405";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 356:
                if ('c' == current) {
                    state = 357;
                    return true;
                }
                break;
            case 357:
                if ('y' == current) {
                    state = 358;
                    return true;
                }
                break;
            case 358:
                // DZcy;
                if (';' == current) {
                    match = "\u040F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 359:
                switch (current) {
                    case 'g' :
                        state = 360;
                        return true;
                    case 'r' :
                        state = 364;
                        return true;
                    case 's' :
                        state = 366;
                        return true;
                }
                break;
            case 360:
                if ('g' == current) {
                    state = 361;
                    return true;
                }
                break;
            case 361:
                if ('e' == current) {
                    state = 362;
                    return true;
                }
                break;
            case 362:
                if ('r' == current) {
                    state = 363;
                    return true;
                }
                break;
            case 363:
                // Dagger;
                if (';' == current) {
                    match = "\u2021";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 364:
                if ('r' == current) {
                    state = 365;
                    return true;
                }
                break;
            case 365:
                // Darr;
                if (';' == current) {
                    match = "\u21A1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 366:
                if ('h' == current) {
                    state = 367;
                    return true;
                }
                break;
            case 367:
                if ('v' == current) {
                    state = 368;
                    return true;
                }
                break;
            case 368:
                // Dashv;
                if (';' == current) {
                    match = "\u2AE4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 369:
                if ('a' == current) {
                    state = 370;
                    return true;
                }
                else if ('y' == current) {
                    state = 374;
                    return true;
                }
                break;
            case 370:
                if ('r' == current) {
                    state = 371;
                    return true;
                }
                break;
            case 371:
                if ('o' == current) {
                    state = 372;
                    return true;
                }
                break;
            case 372:
                if ('n' == current) {
                    state = 373;
                    return true;
                }
                break;
            case 373:
                // Dcaron;
                if (';' == current) {
                    match = "\u010E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 374:
                // Dcy;
                if (';' == current) {
                    match = "\u0414";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 375:
                if ('l' == current) {
                    state = 376;
                    return true;
                }
                break;
            case 376:
                // Del;
                if (';' == current) {
                    match = "\u2207";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('t' == current) {
                    state = 377;
                    return true;
                }
                break;
            case 377:
                if ('a' == current) {
                    state = 378;
                    return true;
                }
                break;
            case 378:
                // Delta;
                if (';' == current) {
                    match = "\u0394";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 379:
                if ('r' == current) {
                    state = 380;
                    return true;
                }
                break;
            case 380:
                // Dfr;
                if (';' == current) {
                    match = "\uD835\uDD07";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 381:
                if ('a' == current) {
                    state = 382;
                    return true;
                }
                else if ('f' == current) {
                    state = 422;
                    return true;
                }
                break;
            case 382:
                if ('c' == current) {
                    state = 383;
                    return true;
                }
                else if ('m' == current) {
                    state = 418;
                    return true;
                }
                break;
            case 383:
                if ('r' == current) {
                    state = 384;
                    return true;
                }
                break;
            case 384:
                if ('i' == current) {
                    state = 385;
                    return true;
                }
                break;
            case 385:
                if ('t' == current) {
                    state = 386;
                    return true;
                }
                break;
            case 386:
                if ('i' == current) {
                    state = 387;
                    return true;
                }
                break;
            case 387:
                if ('c' == current) {
                    state = 388;
                    return true;
                }
                break;
            case 388:
                if ('a' == current) {
                    state = 389;
                    return true;
                }
                break;
            case 389:
                if ('l' == current) {
                    state = 390;
                    return true;
                }
                break;
            case 390:
                switch (current) {
                    case 'A' :
                        state = 391;
                        return true;
                    case 'D' :
                        state = 396;
                        return true;
                    case 'G' :
                        state = 408;
                        return true;
                    case 'T' :
                        state = 413;
                        return true;
                }
                break;
            case 391:
                if ('c' == current) {
                    state = 392;
                    return true;
                }
                break;
            case 392:
                if ('u' == current) {
                    state = 393;
                    return true;
                }
                break;
            case 393:
                if ('t' == current) {
                    state = 394;
                    return true;
                }
                break;
            case 394:
                if ('e' == current) {
                    state = 395;
                    return true;
                }
                break;
            case 395:
                // DiacriticalAcute;
                if (';' == current) {
                    match = "\u00B4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 396:
                if ('o' == current) {
                    state = 397;
                    return true;
                }
                break;
            case 397:
                if ('t' == current) {
                    state = 398;
                    return true;
                }
                else if ('u' == current) {
                    state = 399;
                    return true;
                }
                break;
            case 398:
                // DiacriticalDot;
                if (';' == current) {
                    match = "\u02D9";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 399:
                if ('b' == current) {
                    state = 400;
                    return true;
                }
                break;
            case 400:
                if ('l' == current) {
                    state = 401;
                    return true;
                }
                break;
            case 401:
                if ('e' == current) {
                    state = 402;
                    return true;
                }
                break;
            case 402:
                if ('A' == current) {
                    state = 403;
                    return true;
                }
                break;
            case 403:
                if ('c' == current) {
                    state = 404;
                    return true;
                }
                break;
            case 404:
                if ('u' == current) {
                    state = 405;
                    return true;
                }
                break;
            case 405:
                if ('t' == current) {
                    state = 406;
                    return true;
                }
                break;
            case 406:
                if ('e' == current) {
                    state = 407;
                    return true;
                }
                break;
            case 407:
                // DiacriticalDoubleAcute;
                if (';' == current) {
                    match = "\u02DD";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 408:
                if ('r' == current) {
                    state = 409;
                    return true;
                }
                break;
            case 409:
                if ('a' == current) {
                    state = 410;
                    return true;
                }
                break;
            case 410:
                if ('v' == current) {
                    state = 411;
                    return true;
                }
                break;
            case 411:
                if ('e' == current) {
                    state = 412;
                    return true;
                }
                break;
            case 412:
                // DiacriticalGrave;
                if (';' == current) {
                    match = "`";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 413:
                if ('i' == current) {
                    state = 414;
                    return true;
                }
                break;
            case 414:
                if ('l' == current) {
                    state = 415;
                    return true;
                }
                break;
            case 415:
                if ('d' == current) {
                    state = 416;
                    return true;
                }
                break;
            case 416:
                if ('e' == current) {
                    state = 417;
                    return true;
                }
                break;
            case 417:
                // DiacriticalTilde;
                if (';' == current) {
                    match = "\u02DC";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 418:
                if ('o' == current) {
                    state = 419;
                    return true;
                }
                break;
            case 419:
                if ('n' == current) {
                    state = 420;
                    return true;
                }
                break;
            case 420:
                if ('d' == current) {
                    state = 421;
                    return true;
                }
                break;
            case 421:
                // Diamond;
                if (';' == current) {
                    match = "\u22C4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 422:
                if ('f' == current) {
                    state = 423;
                    return true;
                }
                break;
            case 423:
                if ('e' == current) {
                    state = 424;
                    return true;
                }
                break;
            case 424:
                if ('r' == current) {
                    state = 425;
                    return true;
                }
                break;
            case 425:
                if ('e' == current) {
                    state = 426;
                    return true;
                }
                break;
            case 426:
                if ('n' == current) {
                    state = 427;
                    return true;
                }
                break;
            case 427:
                if ('t' == current) {
                    state = 428;
                    return true;
                }
                break;
            case 428:
                if ('i' == current) {
                    state = 429;
                    return true;
                }
                break;
            case 429:
                if ('a' == current) {
                    state = 430;
                    return true;
                }
                break;
            case 430:
                if ('l' == current) {
                    state = 431;
                    return true;
                }
                break;
            case 431:
                if ('D' == current) {
                    state = 432;
                    return true;
                }
                break;
            case 432:
                // DifferentialD;
                if (';' == current) {
                    match = "\u2146";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 433:
                switch (current) {
                    case 'p' :
                        state = 434;
                        return true;
                    case 't' :
                        state = 436;
                        return true;
                    case 'u' :
                        state = 445;
                        return true;
                    case 'w' :
                        state = 568;
                        return true;
                }
                break;
            case 434:
                if ('f' == current) {
                    state = 435;
                    return true;
                }
                break;
            case 435:
                // Dopf;
                if (';' == current) {
                    match = "\uD835\uDD3B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 436:
                switch (current) {
                    case ';' : // Dot;
                        match = "\u00A8";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'D' :
                        state = 437;
                        return true;
                    case 'E' :
                        state = 440;
                        return true;
                }
                break;
            case 437:
                if ('o' == current) {
                    state = 438;
                    return true;
                }
                break;
            case 438:
                if ('t' == current) {
                    state = 439;
                    return true;
                }
                break;
            case 439:
                // DotDot;
                if (';' == current) {
                    match = "\u20DC";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 440:
                if ('q' == current) {
                    state = 441;
                    return true;
                }
                break;
            case 441:
                if ('u' == current) {
                    state = 442;
                    return true;
                }
                break;
            case 442:
                if ('a' == current) {
                    state = 443;
                    return true;
                }
                break;
            case 443:
                if ('l' == current) {
                    state = 444;
                    return true;
                }
                break;
            case 444:
                // DotEqual;
                if (';' == current) {
                    match = "\u2250";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 445:
                if ('b' == current) {
                    state = 446;
                    return true;
                }
                break;
            case 446:
                if ('l' == current) {
                    state = 447;
                    return true;
                }
                break;
            case 447:
                if ('e' == current) {
                    state = 448;
                    return true;
                }
                break;
            case 448:
                switch (current) {
                    case 'C' :
                        state = 449;
                        return true;
                    case 'D' :
                        state = 464;
                        return true;
                    case 'L' :
                        state = 474;
                        return true;
                    case 'R' :
                        state = 528;
                        return true;
                    case 'U' :
                        state = 541;
                        return true;
                    case 'V' :
                        state = 557;
                        return true;
                }
                break;
            case 449:
                if ('o' == current) {
                    state = 450;
                    return true;
                }
                break;
            case 450:
                if ('n' == current) {
                    state = 451;
                    return true;
                }
                break;
            case 451:
                if ('t' == current) {
                    state = 452;
                    return true;
                }
                break;
            case 452:
                if ('o' == current) {
                    state = 453;
                    return true;
                }
                break;
            case 453:
                if ('u' == current) {
                    state = 454;
                    return true;
                }
                break;
            case 454:
                if ('r' == current) {
                    state = 455;
                    return true;
                }
                break;
            case 455:
                if ('I' == current) {
                    state = 456;
                    return true;
                }
                break;
            case 456:
                if ('n' == current) {
                    state = 457;
                    return true;
                }
                break;
            case 457:
                if ('t' == current) {
                    state = 458;
                    return true;
                }
                break;
            case 458:
                if ('e' == current) {
                    state = 459;
                    return true;
                }
                break;
            case 459:
                if ('g' == current) {
                    state = 460;
                    return true;
                }
                break;
            case 460:
                if ('r' == current) {
                    state = 461;
                    return true;
                }
                break;
            case 461:
                if ('a' == current) {
                    state = 462;
                    return true;
                }
                break;
            case 462:
                if ('l' == current) {
                    state = 463;
                    return true;
                }
                break;
            case 463:
                // DoubleContourIntegral;
                if (';' == current) {
                    match = "\u222F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 464:
                if ('o' == current) {
                    state = 465;
                    return true;
                }
                break;
            case 465:
                if ('t' == current) {
                    state = 466;
                    return true;
                }
                else if ('w' == current) {
                    state = 467;
                    return true;
                }
                break;
            case 466:
                // DoubleDot;
                if (';' == current) {
                    match = "\u00A8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 467:
                if ('n' == current) {
                    state = 468;
                    return true;
                }
                break;
            case 468:
                if ('A' == current) {
                    state = 469;
                    return true;
                }
                break;
            case 469:
                if ('r' == current) {
                    state = 470;
                    return true;
                }
                break;
            case 470:
                if ('r' == current) {
                    state = 471;
                    return true;
                }
                break;
            case 471:
                if ('o' == current) {
                    state = 472;
                    return true;
                }
                break;
            case 472:
                if ('w' == current) {
                    state = 473;
                    return true;
                }
                break;
            case 473:
                // DoubleDownArrow;
                if (';' == current) {
                    match = "\u21D3";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 474:
                if ('e' == current) {
                    state = 475;
                    return true;
                }
                else if ('o' == current) {
                    state = 496;
                    return true;
                }
                break;
            case 475:
                if ('f' == current) {
                    state = 476;
                    return true;
                }
                break;
            case 476:
                if ('t' == current) {
                    state = 477;
                    return true;
                }
                break;
            case 477:
                switch (current) {
                    case 'A' :
                        state = 478;
                        return true;
                    case 'R' :
                        state = 483;
                        return true;
                    case 'T' :
                        state = 493;
                        return true;
                }
                break;
            case 478:
                if ('r' == current) {
                    state = 479;
                    return true;
                }
                break;
            case 479:
                if ('r' == current) {
                    state = 480;
                    return true;
                }
                break;
            case 480:
                if ('o' == current) {
                    state = 481;
                    return true;
                }
                break;
            case 481:
                if ('w' == current) {
                    state = 482;
                    return true;
                }
                break;
            case 482:
                // DoubleLeftArrow;
                if (';' == current) {
                    match = "\u21D0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 483:
                if ('i' == current) {
                    state = 484;
                    return true;
                }
                break;
            case 484:
                if ('g' == current) {
                    state = 485;
                    return true;
                }
                break;
            case 485:
                if ('h' == current) {
                    state = 486;
                    return true;
                }
                break;
            case 486:
                if ('t' == current) {
                    state = 487;
                    return true;
                }
                break;
            case 487:
                if ('A' == current) {
                    state = 488;
                    return true;
                }
                break;
            case 488:
                if ('r' == current) {
                    state = 489;
                    return true;
                }
                break;
            case 489:
                if ('r' == current) {
                    state = 490;
                    return true;
                }
                break;
            case 490:
                if ('o' == current) {
                    state = 491;
                    return true;
                }
                break;
            case 491:
                if ('w' == current) {
                    state = 492;
                    return true;
                }
                break;
            case 492:
                // DoubleLeftRightArrow;
                if (';' == current) {
                    match = "\u21D4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 493:
                if ('e' == current) {
                    state = 494;
                    return true;
                }
                break;
            case 494:
                if ('e' == current) {
                    state = 495;
                    return true;
                }
                break;
            case 495:
                // DoubleLeftTee;
                if (';' == current) {
                    match = "\u2AE4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 496:
                if ('n' == current) {
                    state = 497;
                    return true;
                }
                break;
            case 497:
                if ('g' == current) {
                    state = 498;
                    return true;
                }
                break;
            case 498:
                if ('L' == current) {
                    state = 499;
                    return true;
                }
                else if ('R' == current) {
                    state = 518;
                    return true;
                }
                break;
            case 499:
                if ('e' == current) {
                    state = 500;
                    return true;
                }
                break;
            case 500:
                if ('f' == current) {
                    state = 501;
                    return true;
                }
                break;
            case 501:
                if ('t' == current) {
                    state = 502;
                    return true;
                }
                break;
            case 502:
                if ('A' == current) {
                    state = 503;
                    return true;
                }
                else if ('R' == current) {
                    state = 508;
                    return true;
                }
                break;
            case 503:
                if ('r' == current) {
                    state = 504;
                    return true;
                }
                break;
            case 504:
                if ('r' == current) {
                    state = 505;
                    return true;
                }
                break;
            case 505:
                if ('o' == current) {
                    state = 506;
                    return true;
                }
                break;
            case 506:
                if ('w' == current) {
                    state = 507;
                    return true;
                }
                break;
            case 507:
                // DoubleLongLeftArrow;
                if (';' == current) {
                    match = "\u27F8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 508:
                if ('i' == current) {
                    state = 509;
                    return true;
                }
                break;
            case 509:
                if ('g' == current) {
                    state = 510;
                    return true;
                }
                break;
            case 510:
                if ('h' == current) {
                    state = 511;
                    return true;
                }
                break;
            case 511:
                if ('t' == current) {
                    state = 512;
                    return true;
                }
                break;
            case 512:
                if ('A' == current) {
                    state = 513;
                    return true;
                }
                break;
            case 513:
                if ('r' == current) {
                    state = 514;
                    return true;
                }
                break;
            case 514:
                if ('r' == current) {
                    state = 515;
                    return true;
                }
                break;
            case 515:
                if ('o' == current) {
                    state = 516;
                    return true;
                }
                break;
            case 516:
                if ('w' == current) {
                    state = 517;
                    return true;
                }
                break;
            case 517:
                // DoubleLongLeftRightArrow;
                if (';' == current) {
                    match = "\u27FA";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 518:
                if ('i' == current) {
                    state = 519;
                    return true;
                }
                break;
            case 519:
                if ('g' == current) {
                    state = 520;
                    return true;
                }
                break;
            case 520:
                if ('h' == current) {
                    state = 521;
                    return true;
                }
                break;
            case 521:
                if ('t' == current) {
                    state = 522;
                    return true;
                }
                break;
            case 522:
                if ('A' == current) {
                    state = 523;
                    return true;
                }
                break;
            case 523:
                if ('r' == current) {
                    state = 524;
                    return true;
                }
                break;
            case 524:
                if ('r' == current) {
                    state = 525;
                    return true;
                }
                break;
            case 525:
                if ('o' == current) {
                    state = 526;
                    return true;
                }
                break;
            case 526:
                if ('w' == current) {
                    state = 527;
                    return true;
                }
                break;
            case 527:
                // DoubleLongRightArrow;
                if (';' == current) {
                    match = "\u27F9";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 528:
                if ('i' == current) {
                    state = 529;
                    return true;
                }
                break;
            case 529:
                if ('g' == current) {
                    state = 530;
                    return true;
                }
                break;
            case 530:
                if ('h' == current) {
                    state = 531;
                    return true;
                }
                break;
            case 531:
                if ('t' == current) {
                    state = 532;
                    return true;
                }
                break;
            case 532:
                if ('A' == current) {
                    state = 533;
                    return true;
                }
                else if ('T' == current) {
                    state = 538;
                    return true;
                }
                break;
            case 533:
                if ('r' == current) {
                    state = 534;
                    return true;
                }
                break;
            case 534:
                if ('r' == current) {
                    state = 535;
                    return true;
                }
                break;
            case 535:
                if ('o' == current) {
                    state = 536;
                    return true;
                }
                break;
            case 536:
                if ('w' == current) {
                    state = 537;
                    return true;
                }
                break;
            case 537:
                // DoubleRightArrow;
                if (';' == current) {
                    match = "\u21D2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 538:
                if ('e' == current) {
                    state = 539;
                    return true;
                }
                break;
            case 539:
                if ('e' == current) {
                    state = 540;
                    return true;
                }
                break;
            case 540:
                // DoubleRightTee;
                if (';' == current) {
                    match = "\u22A8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 541:
                if ('p' == current) {
                    state = 542;
                    return true;
                }
                break;
            case 542:
                if ('A' == current) {
                    state = 543;
                    return true;
                }
                else if ('D' == current) {
                    state = 548;
                    return true;
                }
                break;
            case 543:
                if ('r' == current) {
                    state = 544;
                    return true;
                }
                break;
            case 544:
                if ('r' == current) {
                    state = 545;
                    return true;
                }
                break;
            case 545:
                if ('o' == current) {
                    state = 546;
                    return true;
                }
                break;
            case 546:
                if ('w' == current) {
                    state = 547;
                    return true;
                }
                break;
            case 547:
                // DoubleUpArrow;
                if (';' == current) {
                    match = "\u21D1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 548:
                if ('o' == current) {
                    state = 549;
                    return true;
                }
                break;
            case 549:
                if ('w' == current) {
                    state = 550;
                    return true;
                }
                break;
            case 550:
                if ('n' == current) {
                    state = 551;
                    return true;
                }
                break;
            case 551:
                if ('A' == current) {
                    state = 552;
                    return true;
                }
                break;
            case 552:
                if ('r' == current) {
                    state = 553;
                    return true;
                }
                break;
            case 553:
                if ('r' == current) {
                    state = 554;
                    return true;
                }
                break;
            case 554:
                if ('o' == current) {
                    state = 555;
                    return true;
                }
                break;
            case 555:
                if ('w' == current) {
                    state = 556;
                    return true;
                }
                break;
            case 556:
                // DoubleUpDownArrow;
                if (';' == current) {
                    match = "\u21D5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 557:
                if ('e' == current) {
                    state = 558;
                    return true;
                }
                break;
            case 558:
                if ('r' == current) {
                    state = 559;
                    return true;
                }
                break;
            case 559:
                if ('t' == current) {
                    state = 560;
                    return true;
                }
                break;
            case 560:
                if ('i' == current) {
                    state = 561;
                    return true;
                }
                break;
            case 561:
                if ('c' == current) {
                    state = 562;
                    return true;
                }
                break;
            case 562:
                if ('a' == current) {
                    state = 563;
                    return true;
                }
                break;
            case 563:
                if ('l' == current) {
                    state = 564;
                    return true;
                }
                break;
            case 564:
                if ('B' == current) {
                    state = 565;
                    return true;
                }
                break;
            case 565:
                if ('a' == current) {
                    state = 566;
                    return true;
                }
                break;
            case 566:
                if ('r' == current) {
                    state = 567;
                    return true;
                }
                break;
            case 567:
                // DoubleVerticalBar;
                if (';' == current) {
                    match = "\u2225";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 568:
                if ('n' == current) {
                    state = 569;
                    return true;
                }
                break;
            case 569:
                switch (current) {
                    case 'A' :
                        state = 570;
                        return true;
                    case 'B' :
                        state = 585;
                        return true;
                    case 'L' :
                        state = 590;
                        return true;
                    case 'R' :
                        state = 623;
                        return true;
                    case 'T' :
                        state = 646;
                        return true;
                    case 'a' :
                        state = 654;
                        return true;
                }
                break;
            case 570:
                if ('r' == current) {
                    state = 571;
                    return true;
                }
                break;
            case 571:
                if ('r' == current) {
                    state = 572;
                    return true;
                }
                break;
            case 572:
                if ('o' == current) {
                    state = 573;
                    return true;
                }
                break;
            case 573:
                if ('w' == current) {
                    state = 574;
                    return true;
                }
                break;
            case 574:
                switch (current) {
                    case ';' : // DownArrow;
                        match = "\u2193";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 575;
                        return true;
                    case 'U' :
                        state = 578;
                        return true;
                }
                break;
            case 575:
                if ('a' == current) {
                    state = 576;
                    return true;
                }
                break;
            case 576:
                if ('r' == current) {
                    state = 577;
                    return true;
                }
                break;
            case 577:
                // DownArrowBar;
                if (';' == current) {
                    match = "\u2913";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 578:
                if ('p' == current) {
                    state = 579;
                    return true;
                }
                break;
            case 579:
                if ('A' == current) {
                    state = 580;
                    return true;
                }
                break;
            case 580:
                if ('r' == current) {
                    state = 581;
                    return true;
                }
                break;
            case 581:
                if ('r' == current) {
                    state = 582;
                    return true;
                }
                break;
            case 582:
                if ('o' == current) {
                    state = 583;
                    return true;
                }
                break;
            case 583:
                if ('w' == current) {
                    state = 584;
                    return true;
                }
                break;
            case 584:
                // DownArrowUpArrow;
                if (';' == current) {
                    match = "\u21F5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 585:
                if ('r' == current) {
                    state = 586;
                    return true;
                }
                break;
            case 586:
                if ('e' == current) {
                    state = 587;
                    return true;
                }
                break;
            case 587:
                if ('v' == current) {
                    state = 588;
                    return true;
                }
                break;
            case 588:
                if ('e' == current) {
                    state = 589;
                    return true;
                }
                break;
            case 589:
                // DownBreve;
                if (';' == current) {
                    match = "\u0311";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 590:
                if ('e' == current) {
                    state = 591;
                    return true;
                }
                break;
            case 591:
                if ('f' == current) {
                    state = 592;
                    return true;
                }
                break;
            case 592:
                if ('t' == current) {
                    state = 593;
                    return true;
                }
                break;
            case 593:
                switch (current) {
                    case 'R' :
                        state = 594;
                        return true;
                    case 'T' :
                        state = 605;
                        return true;
                    case 'V' :
                        state = 614;
                        return true;
                }
                break;
            case 594:
                if ('i' == current) {
                    state = 595;
                    return true;
                }
                break;
            case 595:
                if ('g' == current) {
                    state = 596;
                    return true;
                }
                break;
            case 596:
                if ('h' == current) {
                    state = 597;
                    return true;
                }
                break;
            case 597:
                if ('t' == current) {
                    state = 598;
                    return true;
                }
                break;
            case 598:
                if ('V' == current) {
                    state = 599;
                    return true;
                }
                break;
            case 599:
                if ('e' == current) {
                    state = 600;
                    return true;
                }
                break;
            case 600:
                if ('c' == current) {
                    state = 601;
                    return true;
                }
                break;
            case 601:
                if ('t' == current) {
                    state = 602;
                    return true;
                }
                break;
            case 602:
                if ('o' == current) {
                    state = 603;
                    return true;
                }
                break;
            case 603:
                if ('r' == current) {
                    state = 604;
                    return true;
                }
                break;
            case 604:
                // DownLeftRightVector;
                if (';' == current) {
                    match = "\u2950";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 605:
                if ('e' == current) {
                    state = 606;
                    return true;
                }
                break;
            case 606:
                if ('e' == current) {
                    state = 607;
                    return true;
                }
                break;
            case 607:
                if ('V' == current) {
                    state = 608;
                    return true;
                }
                break;
            case 608:
                if ('e' == current) {
                    state = 609;
                    return true;
                }
                break;
            case 609:
                if ('c' == current) {
                    state = 610;
                    return true;
                }
                break;
            case 610:
                if ('t' == current) {
                    state = 611;
                    return true;
                }
                break;
            case 611:
                if ('o' == current) {
                    state = 612;
                    return true;
                }
                break;
            case 612:
                if ('r' == current) {
                    state = 613;
                    return true;
                }
                break;
            case 613:
                // DownLeftTeeVector;
                if (';' == current) {
                    match = "\u295E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 614:
                if ('e' == current) {
                    state = 615;
                    return true;
                }
                break;
            case 615:
                if ('c' == current) {
                    state = 616;
                    return true;
                }
                break;
            case 616:
                if ('t' == current) {
                    state = 617;
                    return true;
                }
                break;
            case 617:
                if ('o' == current) {
                    state = 618;
                    return true;
                }
                break;
            case 618:
                if ('r' == current) {
                    state = 619;
                    return true;
                }
                break;
            case 619:
                // DownLeftVector;
                if (';' == current) {
                    match = "\u21BD";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('B' == current) {
                    state = 620;
                    return true;
                }
                break;
            case 620:
                if ('a' == current) {
                    state = 621;
                    return true;
                }
                break;
            case 621:
                if ('r' == current) {
                    state = 622;
                    return true;
                }
                break;
            case 622:
                // DownLeftVectorBar;
                if (';' == current) {
                    match = "\u2956";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 623:
                if ('i' == current) {
                    state = 624;
                    return true;
                }
                break;
            case 624:
                if ('g' == current) {
                    state = 625;
                    return true;
                }
                break;
            case 625:
                if ('h' == current) {
                    state = 626;
                    return true;
                }
                break;
            case 626:
                if ('t' == current) {
                    state = 627;
                    return true;
                }
                break;
            case 627:
                if ('T' == current) {
                    state = 628;
                    return true;
                }
                else if ('V' == current) {
                    state = 637;
                    return true;
                }
                break;
            case 628:
                if ('e' == current) {
                    state = 629;
                    return true;
                }
                break;
            case 629:
                if ('e' == current) {
                    state = 630;
                    return true;
                }
                break;
            case 630:
                if ('V' == current) {
                    state = 631;
                    return true;
                }
                break;
            case 631:
                if ('e' == current) {
                    state = 632;
                    return true;
                }
                break;
            case 632:
                if ('c' == current) {
                    state = 633;
                    return true;
                }
                break;
            case 633:
                if ('t' == current) {
                    state = 634;
                    return true;
                }
                break;
            case 634:
                if ('o' == current) {
                    state = 635;
                    return true;
                }
                break;
            case 635:
                if ('r' == current) {
                    state = 636;
                    return true;
                }
                break;
            case 636:
                // DownRightTeeVector;
                if (';' == current) {
                    match = "\u295F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 637:
                if ('e' == current) {
                    state = 638;
                    return true;
                }
                break;
            case 638:
                if ('c' == current) {
                    state = 639;
                    return true;
                }
                break;
            case 639:
                if ('t' == current) {
                    state = 640;
                    return true;
                }
                break;
            case 640:
                if ('o' == current) {
                    state = 641;
                    return true;
                }
                break;
            case 641:
                if ('r' == current) {
                    state = 642;
                    return true;
                }
                break;
            case 642:
                // DownRightVector;
                if (';' == current) {
                    match = "\u21C1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('B' == current) {
                    state = 643;
                    return true;
                }
                break;
            case 643:
                if ('a' == current) {
                    state = 644;
                    return true;
                }
                break;
            case 644:
                if ('r' == current) {
                    state = 645;
                    return true;
                }
                break;
            case 645:
                // DownRightVectorBar;
                if (';' == current) {
                    match = "\u2957";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 646:
                if ('e' == current) {
                    state = 647;
                    return true;
                }
                break;
            case 647:
                if ('e' == current) {
                    state = 648;
                    return true;
                }
                break;
            case 648:
                // DownTee;
                if (';' == current) {
                    match = "\u22A4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('A' == current) {
                    state = 649;
                    return true;
                }
                break;
            case 649:
                if ('r' == current) {
                    state = 650;
                    return true;
                }
                break;
            case 650:
                if ('r' == current) {
                    state = 651;
                    return true;
                }
                break;
            case 651:
                if ('o' == current) {
                    state = 652;
                    return true;
                }
                break;
            case 652:
                if ('w' == current) {
                    state = 653;
                    return true;
                }
                break;
            case 653:
                // DownTeeArrow;
                if (';' == current) {
                    match = "\u21A7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 654:
                if ('r' == current) {
                    state = 655;
                    return true;
                }
                break;
            case 655:
                if ('r' == current) {
                    state = 656;
                    return true;
                }
                break;
            case 656:
                if ('o' == current) {
                    state = 657;
                    return true;
                }
                break;
            case 657:
                if ('w' == current) {
                    state = 658;
                    return true;
                }
                break;
            case 658:
                // Downarrow;
                if (';' == current) {
                    match = "\u21D3";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 659:
                if ('c' == current) {
                    state = 660;
                    return true;
                }
                else if ('t' == current) {
                    state = 662;
                    return true;
                }
                break;
            case 660:
                if ('r' == current) {
                    state = 661;
                    return true;
                }
                break;
            case 661:
                // Dscr;
                if (';' == current) {
                    match = "\uD835\uDC9F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 662:
                if ('r' == current) {
                    state = 663;
                    return true;
                }
                break;
            case 663:
                if ('o' == current) {
                    state = 664;
                    return true;
                }
                break;
            case 664:
                if ('k' == current) {
                    state = 665;
                    return true;
                }
                break;
            case 665:
                // Dstrok;
                if (';' == current) {
                    match = "\u0110";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 666:
                switch (current) {
                    case 'N' :
                        state = 667;
                        return true;
                    case 'T' :
                        state = 669;
                        return true;
                    case 'a' :
                        state = 671;
                        return true;
                    case 'c' :
                        state = 676;
                        return true;
                    case 'd' :
                        state = 685;
                        return true;
                    case 'f' :
                        state = 688;
                        return true;
                    case 'g' :
                        state = 690;
                        return true;
                    case 'l' :
                        state = 695;
                        return true;
                    case 'm' :
                        state = 701;
                        return true;
                    case 'o' :
                        state = 734;
                        return true;
                    case 'p' :
                        state = 740;
                        return true;
                    case 'q' :
                        state = 746;
                        return true;
                    case 's' :
                        state = 763;
                        return true;
                    case 't' :
                        state = 768;
                        return true;
                    case 'u' :
                        state = 770;
                        return true;
                    case 'x' :
                        state = 773;
                        return true;
                }
                break;
            case 667:
                if ('G' == current) {
                    state = 668;
                    return true;
                }
                break;
            case 668:
                // ENG;
                if (';' == current) {
                    match = "\u014A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 669:
                // ETH
                if ('H' == current) {
                    match = "\u00D0";
                    matchLength = consumedCount;
                    state = 670;
                    return true;
                }
                break;
            case 670:
                // ETH;
                if (';' == current) {
                    match = "\u00D0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 671:
                if ('c' == current) {
                    state = 672;
                    return true;
                }
                break;
            case 672:
                if ('u' == current) {
                    state = 673;
                    return true;
                }
                break;
            case 673:
                if ('t' == current) {
                    state = 674;
                    return true;
                }
                break;
            case 674:
                // Eacute
                if ('e' == current) {
                    match = "\u00C9";
                    matchLength = consumedCount;
                    state = 675;
                    return true;
                }
                break;
            case 675:
                // Eacute;
                if (';' == current) {
                    match = "\u00C9";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 676:
                switch (current) {
                    case 'a' :
                        state = 677;
                        return true;
                    case 'i' :
                        state = 681;
                        return true;
                    case 'y' :
                        state = 684;
                        return true;
                }
                break;
            case 677:
                if ('r' == current) {
                    state = 678;
                    return true;
                }
                break;
            case 678:
                if ('o' == current) {
                    state = 679;
                    return true;
                }
                break;
            case 679:
                if ('n' == current) {
                    state = 680;
                    return true;
                }
                break;
            case 680:
                // Ecaron;
                if (';' == current) {
                    match = "\u011A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 681:
                if ('r' == current) {
                    state = 682;
                    return true;
                }
                break;
            case 682:
                // Ecirc
                if ('c' == current) {
                    match = "\u00CA";
                    matchLength = consumedCount;
                    state = 683;
                    return true;
                }
                break;
            case 683:
                // Ecirc;
                if (';' == current) {
                    match = "\u00CA";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 684:
                // Ecy;
                if (';' == current) {
                    match = "\u042D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 685:
                if ('o' == current) {
                    state = 686;
                    return true;
                }
                break;
            case 686:
                if ('t' == current) {
                    state = 687;
                    return true;
                }
                break;
            case 687:
                // Edot;
                if (';' == current) {
                    match = "\u0116";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 688:
                if ('r' == current) {
                    state = 689;
                    return true;
                }
                break;
            case 689:
                // Efr;
                if (';' == current) {
                    match = "\uD835\uDD08";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 690:
                if ('r' == current) {
                    state = 691;
                    return true;
                }
                break;
            case 691:
                if ('a' == current) {
                    state = 692;
                    return true;
                }
                break;
            case 692:
                if ('v' == current) {
                    state = 693;
                    return true;
                }
                break;
            case 693:
                // Egrave
                if ('e' == current) {
                    match = "\u00C8";
                    matchLength = consumedCount;
                    state = 694;
                    return true;
                }
                break;
            case 694:
                // Egrave;
                if (';' == current) {
                    match = "\u00C8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 695:
                if ('e' == current) {
                    state = 696;
                    return true;
                }
                break;
            case 696:
                if ('m' == current) {
                    state = 697;
                    return true;
                }
                break;
            case 697:
                if ('e' == current) {
                    state = 698;
                    return true;
                }
                break;
            case 698:
                if ('n' == current) {
                    state = 699;
                    return true;
                }
                break;
            case 699:
                if ('t' == current) {
                    state = 700;
                    return true;
                }
                break;
            case 700:
                // Element;
                if (';' == current) {
                    match = "\u2208";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 701:
                if ('a' == current) {
                    state = 702;
                    return true;
                }
                else if ('p' == current) {
                    state = 705;
                    return true;
                }
                break;
            case 702:
                if ('c' == current) {
                    state = 703;
                    return true;
                }
                break;
            case 703:
                if ('r' == current) {
                    state = 704;
                    return true;
                }
                break;
            case 704:
                // Emacr;
                if (';' == current) {
                    match = "\u0112";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 705:
                if ('t' == current) {
                    state = 706;
                    return true;
                }
                break;
            case 706:
                if ('y' == current) {
                    state = 707;
                    return true;
                }
                break;
            case 707:
                if ('S' == current) {
                    state = 708;
                    return true;
                }
                else if ('V' == current) {
                    state = 719;
                    return true;
                }
                break;
            case 708:
                if ('m' == current) {
                    state = 709;
                    return true;
                }
                break;
            case 709:
                if ('a' == current) {
                    state = 710;
                    return true;
                }
                break;
            case 710:
                if ('l' == current) {
                    state = 711;
                    return true;
                }
                break;
            case 711:
                if ('l' == current) {
                    state = 712;
                    return true;
                }
                break;
            case 712:
                if ('S' == current) {
                    state = 713;
                    return true;
                }
                break;
            case 713:
                if ('q' == current) {
                    state = 714;
                    return true;
                }
                break;
            case 714:
                if ('u' == current) {
                    state = 715;
                    return true;
                }
                break;
            case 715:
                if ('a' == current) {
                    state = 716;
                    return true;
                }
                break;
            case 716:
                if ('r' == current) {
                    state = 717;
                    return true;
                }
                break;
            case 717:
                if ('e' == current) {
                    state = 718;
                    return true;
                }
                break;
            case 718:
                // EmptySmallSquare;
                if (';' == current) {
                    match = "\u25FB";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 719:
                if ('e' == current) {
                    state = 720;
                    return true;
                }
                break;
            case 720:
                if ('r' == current) {
                    state = 721;
                    return true;
                }
                break;
            case 721:
                if ('y' == current) {
                    state = 722;
                    return true;
                }
                break;
            case 722:
                if ('S' == current) {
                    state = 723;
                    return true;
                }
                break;
            case 723:
                if ('m' == current) {
                    state = 724;
                    return true;
                }
                break;
            case 724:
                if ('a' == current) {
                    state = 725;
                    return true;
                }
                break;
            case 725:
                if ('l' == current) {
                    state = 726;
                    return true;
                }
                break;
            case 726:
                if ('l' == current) {
                    state = 727;
                    return true;
                }
                break;
            case 727:
                if ('S' == current) {
                    state = 728;
                    return true;
                }
                break;
            case 728:
                if ('q' == current) {
                    state = 729;
                    return true;
                }
                break;
            case 729:
                if ('u' == current) {
                    state = 730;
                    return true;
                }
                break;
            case 730:
                if ('a' == current) {
                    state = 731;
                    return true;
                }
                break;
            case 731:
                if ('r' == current) {
                    state = 732;
                    return true;
                }
                break;
            case 732:
                if ('e' == current) {
                    state = 733;
                    return true;
                }
                break;
            case 733:
                // EmptyVerySmallSquare;
                if (';' == current) {
                    match = "\u25AB";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 734:
                if ('g' == current) {
                    state = 735;
                    return true;
                }
                else if ('p' == current) {
                    state = 738;
                    return true;
                }
                break;
            case 735:
                if ('o' == current) {
                    state = 736;
                    return true;
                }
                break;
            case 736:
                if ('n' == current) {
                    state = 737;
                    return true;
                }
                break;
            case 737:
                // Eogon;
                if (';' == current) {
                    match = "\u0118";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 738:
                if ('f' == current) {
                    state = 739;
                    return true;
                }
                break;
            case 739:
                // Eopf;
                if (';' == current) {
                    match = "\uD835\uDD3C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 740:
                if ('s' == current) {
                    state = 741;
                    return true;
                }
                break;
            case 741:
                if ('i' == current) {
                    state = 742;
                    return true;
                }
                break;
            case 742:
                if ('l' == current) {
                    state = 743;
                    return true;
                }
                break;
            case 743:
                if ('o' == current) {
                    state = 744;
                    return true;
                }
                break;
            case 744:
                if ('n' == current) {
                    state = 745;
                    return true;
                }
                break;
            case 745:
                // Epsilon;
                if (';' == current) {
                    match = "\u0395";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 746:
                if ('u' == current) {
                    state = 747;
                    return true;
                }
                break;
            case 747:
                if ('a' == current) {
                    state = 748;
                    return true;
                }
                else if ('i' == current) {
                    state = 755;
                    return true;
                }
                break;
            case 748:
                if ('l' == current) {
                    state = 749;
                    return true;
                }
                break;
            case 749:
                // Equal;
                if (';' == current) {
                    match = "\u2A75";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('T' == current) {
                    state = 750;
                    return true;
                }
                break;
            case 750:
                if ('i' == current) {
                    state = 751;
                    return true;
                }
                break;
            case 751:
                if ('l' == current) {
                    state = 752;
                    return true;
                }
                break;
            case 752:
                if ('d' == current) {
                    state = 753;
                    return true;
                }
                break;
            case 753:
                if ('e' == current) {
                    state = 754;
                    return true;
                }
                break;
            case 754:
                // EqualTilde;
                if (';' == current) {
                    match = "\u2242";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 755:
                if ('l' == current) {
                    state = 756;
                    return true;
                }
                break;
            case 756:
                if ('i' == current) {
                    state = 757;
                    return true;
                }
                break;
            case 757:
                if ('b' == current) {
                    state = 758;
                    return true;
                }
                break;
            case 758:
                if ('r' == current) {
                    state = 759;
                    return true;
                }
                break;
            case 759:
                if ('i' == current) {
                    state = 760;
                    return true;
                }
                break;
            case 760:
                if ('u' == current) {
                    state = 761;
                    return true;
                }
                break;
            case 761:
                if ('m' == current) {
                    state = 762;
                    return true;
                }
                break;
            case 762:
                // Equilibrium;
                if (';' == current) {
                    match = "\u21CC";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 763:
                if ('c' == current) {
                    state = 764;
                    return true;
                }
                else if ('i' == current) {
                    state = 766;
                    return true;
                }
                break;
            case 764:
                if ('r' == current) {
                    state = 765;
                    return true;
                }
                break;
            case 765:
                // Escr;
                if (';' == current) {
                    match = "\u2130";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 766:
                if ('m' == current) {
                    state = 767;
                    return true;
                }
                break;
            case 767:
                // Esim;
                if (';' == current) {
                    match = "\u2A73";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 768:
                if ('a' == current) {
                    state = 769;
                    return true;
                }
                break;
            case 769:
                // Eta;
                if (';' == current) {
                    match = "\u0397";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 770:
                if ('m' == current) {
                    state = 771;
                    return true;
                }
                break;
            case 771:
                // Euml
                if ('l' == current) {
                    match = "\u00CB";
                    matchLength = consumedCount;
                    state = 772;
                    return true;
                }
                break;
            case 772:
                // Euml;
                if (';' == current) {
                    match = "\u00CB";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 773:
                if ('i' == current) {
                    state = 774;
                    return true;
                }
                else if ('p' == current) {
                    state = 778;
                    return true;
                }
                break;
            case 774:
                if ('s' == current) {
                    state = 775;
                    return true;
                }
                break;
            case 775:
                if ('t' == current) {
                    state = 776;
                    return true;
                }
                break;
            case 776:
                if ('s' == current) {
                    state = 777;
                    return true;
                }
                break;
            case 777:
                // Exists;
                if (';' == current) {
                    match = "\u2203";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 778:
                if ('o' == current) {
                    state = 779;
                    return true;
                }
                break;
            case 779:
                if ('n' == current) {
                    state = 780;
                    return true;
                }
                break;
            case 780:
                if ('e' == current) {
                    state = 781;
                    return true;
                }
                break;
            case 781:
                if ('n' == current) {
                    state = 782;
                    return true;
                }
                break;
            case 782:
                if ('t' == current) {
                    state = 783;
                    return true;
                }
                break;
            case 783:
                if ('i' == current) {
                    state = 784;
                    return true;
                }
                break;
            case 784:
                if ('a' == current) {
                    state = 785;
                    return true;
                }
                break;
            case 785:
                if ('l' == current) {
                    state = 786;
                    return true;
                }
                break;
            case 786:
                if ('E' == current) {
                    state = 787;
                    return true;
                }
                break;
            case 787:
                // ExponentialE;
                if (';' == current) {
                    match = "\u2147";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 788:
                switch (current) {
                    case 'c' :
                        state = 789;
                        return true;
                    case 'f' :
                        state = 791;
                        return true;
                    case 'i' :
                        state = 793;
                        return true;
                    case 'o' :
                        state = 824;
                        return true;
                    case 's' :
                        state = 839;
                        return true;
                }
                break;
            case 789:
                if ('y' == current) {
                    state = 790;
                    return true;
                }
                break;
            case 790:
                // Fcy;
                if (';' == current) {
                    match = "\u0424";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 791:
                if ('r' == current) {
                    state = 792;
                    return true;
                }
                break;
            case 792:
                // Ffr;
                if (';' == current) {
                    match = "\uD835\uDD09";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 793:
                if ('l' == current) {
                    state = 794;
                    return true;
                }
                break;
            case 794:
                if ('l' == current) {
                    state = 795;
                    return true;
                }
                break;
            case 795:
                if ('e' == current) {
                    state = 796;
                    return true;
                }
                break;
            case 796:
                if ('d' == current) {
                    state = 797;
                    return true;
                }
                break;
            case 797:
                if ('S' == current) {
                    state = 798;
                    return true;
                }
                else if ('V' == current) {
                    state = 809;
                    return true;
                }
                break;
            case 798:
                if ('m' == current) {
                    state = 799;
                    return true;
                }
                break;
            case 799:
                if ('a' == current) {
                    state = 800;
                    return true;
                }
                break;
            case 800:
                if ('l' == current) {
                    state = 801;
                    return true;
                }
                break;
            case 801:
                if ('l' == current) {
                    state = 802;
                    return true;
                }
                break;
            case 802:
                if ('S' == current) {
                    state = 803;
                    return true;
                }
                break;
            case 803:
                if ('q' == current) {
                    state = 804;
                    return true;
                }
                break;
            case 804:
                if ('u' == current) {
                    state = 805;
                    return true;
                }
                break;
            case 805:
                if ('a' == current) {
                    state = 806;
                    return true;
                }
                break;
            case 806:
                if ('r' == current) {
                    state = 807;
                    return true;
                }
                break;
            case 807:
                if ('e' == current) {
                    state = 808;
                    return true;
                }
                break;
            case 808:
                // FilledSmallSquare;
                if (';' == current) {
                    match = "\u25FC";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 809:
                if ('e' == current) {
                    state = 810;
                    return true;
                }
                break;
            case 810:
                if ('r' == current) {
                    state = 811;
                    return true;
                }
                break;
            case 811:
                if ('y' == current) {
                    state = 812;
                    return true;
                }
                break;
            case 812:
                if ('S' == current) {
                    state = 813;
                    return true;
                }
                break;
            case 813:
                if ('m' == current) {
                    state = 814;
                    return true;
                }
                break;
            case 814:
                if ('a' == current) {
                    state = 815;
                    return true;
                }
                break;
            case 815:
                if ('l' == current) {
                    state = 816;
                    return true;
                }
                break;
            case 816:
                if ('l' == current) {
                    state = 817;
                    return true;
                }
                break;
            case 817:
                if ('S' == current) {
                    state = 818;
                    return true;
                }
                break;
            case 818:
                if ('q' == current) {
                    state = 819;
                    return true;
                }
                break;
            case 819:
                if ('u' == current) {
                    state = 820;
                    return true;
                }
                break;
            case 820:
                if ('a' == current) {
                    state = 821;
                    return true;
                }
                break;
            case 821:
                if ('r' == current) {
                    state = 822;
                    return true;
                }
                break;
            case 822:
                if ('e' == current) {
                    state = 823;
                    return true;
                }
                break;
            case 823:
                // FilledVerySmallSquare;
                if (';' == current) {
                    match = "\u25AA";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 824:
                switch (current) {
                    case 'p' :
                        state = 825;
                        return true;
                    case 'r' :
                        state = 827;
                        return true;
                    case 'u' :
                        state = 831;
                        return true;
                }
                break;
            case 825:
                if ('f' == current) {
                    state = 826;
                    return true;
                }
                break;
            case 826:
                // Fopf;
                if (';' == current) {
                    match = "\uD835\uDD3D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 827:
                if ('A' == current) {
                    state = 828;
                    return true;
                }
                break;
            case 828:
                if ('l' == current) {
                    state = 829;
                    return true;
                }
                break;
            case 829:
                if ('l' == current) {
                    state = 830;
                    return true;
                }
                break;
            case 830:
                // ForAll;
                if (';' == current) {
                    match = "\u2200";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 831:
                if ('r' == current) {
                    state = 832;
                    return true;
                }
                break;
            case 832:
                if ('i' == current) {
                    state = 833;
                    return true;
                }
                break;
            case 833:
                if ('e' == current) {
                    state = 834;
                    return true;
                }
                break;
            case 834:
                if ('r' == current) {
                    state = 835;
                    return true;
                }
                break;
            case 835:
                if ('t' == current) {
                    state = 836;
                    return true;
                }
                break;
            case 836:
                if ('r' == current) {
                    state = 837;
                    return true;
                }
                break;
            case 837:
                if ('f' == current) {
                    state = 838;
                    return true;
                }
                break;
            case 838:
                // Fouriertrf;
                if (';' == current) {
                    match = "\u2131";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 839:
                if ('c' == current) {
                    state = 840;
                    return true;
                }
                break;
            case 840:
                if ('r' == current) {
                    state = 841;
                    return true;
                }
                break;
            case 841:
                // Fscr;
                if (';' == current) {
                    match = "\u2131";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 842:
                switch (current) {
                    case 'J' :
                        state = 843;
                        return true;
                    case 'T' : // GT
                        match = ">";
                        matchLength = consumedCount;
                        state = 846;
                        return true;
                    case 'a' :
                        state = 847;
                        return true;
                    case 'b' :
                        state = 852;
                        return true;
                    case 'c' :
                        state = 857;
                        return true;
                    case 'd' :
                        state = 866;
                        return true;
                    case 'f' :
                        state = 869;
                        return true;
                    case 'g' :
                        state = 871;
                        return true;
                    case 'o' :
                        state = 872;
                        return true;
                    case 'r' :
                        state = 875;
                        return true;
                    case 's' :
                        state = 925;
                        return true;
                    case 't' :
                        state = 928;
                        return true;
                }
                break;
            case 843:
                if ('c' == current) {
                    state = 844;
                    return true;
                }
                break;
            case 844:
                if ('y' == current) {
                    state = 845;
                    return true;
                }
                break;
            case 845:
                // GJcy;
                if (';' == current) {
                    match = "\u0403";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 846:
                // GT;
                if (';' == current) {
                    match = ">";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 847:
                if ('m' == current) {
                    state = 848;
                    return true;
                }
                break;
            case 848:
                if ('m' == current) {
                    state = 849;
                    return true;
                }
                break;
            case 849:
                if ('a' == current) {
                    state = 850;
                    return true;
                }
                break;
            case 850:
                // Gamma;
                if (';' == current) {
                    match = "\u0393";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('d' == current) {
                    state = 851;
                    return true;
                }
                break;
            case 851:
                // Gammad;
                if (';' == current) {
                    match = "\u03DC";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 852:
                if ('r' == current) {
                    state = 853;
                    return true;
                }
                break;
            case 853:
                if ('e' == current) {
                    state = 854;
                    return true;
                }
                break;
            case 854:
                if ('v' == current) {
                    state = 855;
                    return true;
                }
                break;
            case 855:
                if ('e' == current) {
                    state = 856;
                    return true;
                }
                break;
            case 856:
                // Gbreve;
                if (';' == current) {
                    match = "\u011E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 857:
                switch (current) {
                    case 'e' :
                        state = 858;
                        return true;
                    case 'i' :
                        state = 862;
                        return true;
                    case 'y' :
                        state = 865;
                        return true;
                }
                break;
            case 858:
                if ('d' == current) {
                    state = 859;
                    return true;
                }
                break;
            case 859:
                if ('i' == current) {
                    state = 860;
                    return true;
                }
                break;
            case 860:
                if ('l' == current) {
                    state = 861;
                    return true;
                }
                break;
            case 861:
                // Gcedil;
                if (';' == current) {
                    match = "\u0122";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 862:
                if ('r' == current) {
                    state = 863;
                    return true;
                }
                break;
            case 863:
                if ('c' == current) {
                    state = 864;
                    return true;
                }
                break;
            case 864:
                // Gcirc;
                if (';' == current) {
                    match = "\u011C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 865:
                // Gcy;
                if (';' == current) {
                    match = "\u0413";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 866:
                if ('o' == current) {
                    state = 867;
                    return true;
                }
                break;
            case 867:
                if ('t' == current) {
                    state = 868;
                    return true;
                }
                break;
            case 868:
                // Gdot;
                if (';' == current) {
                    match = "\u0120";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 869:
                if ('r' == current) {
                    state = 870;
                    return true;
                }
                break;
            case 870:
                // Gfr;
                if (';' == current) {
                    match = "\uD835\uDD0A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 871:
                // Gg;
                if (';' == current) {
                    match = "\u22D9";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 872:
                if ('p' == current) {
                    state = 873;
                    return true;
                }
                break;
            case 873:
                if ('f' == current) {
                    state = 874;
                    return true;
                }
                break;
            case 874:
                // Gopf;
                if (';' == current) {
                    match = "\uD835\uDD3E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 875:
                if ('e' == current) {
                    state = 876;
                    return true;
                }
                break;
            case 876:
                if ('a' == current) {
                    state = 877;
                    return true;
                }
                break;
            case 877:
                if ('t' == current) {
                    state = 878;
                    return true;
                }
                break;
            case 878:
                if ('e' == current) {
                    state = 879;
                    return true;
                }
                break;
            case 879:
                if ('r' == current) {
                    state = 880;
                    return true;
                }
                break;
            case 880:
                switch (current) {
                    case 'E' :
                        state = 881;
                        return true;
                    case 'F' :
                        state = 890;
                        return true;
                    case 'G' :
                        state = 899;
                        return true;
                    case 'L' :
                        state = 906;
                        return true;
                    case 'S' :
                        state = 910;
                        return true;
                    case 'T' :
                        state = 920;
                        return true;
                }
                break;
            case 881:
                if ('q' == current) {
                    state = 882;
                    return true;
                }
                break;
            case 882:
                if ('u' == current) {
                    state = 883;
                    return true;
                }
                break;
            case 883:
                if ('a' == current) {
                    state = 884;
                    return true;
                }
                break;
            case 884:
                if ('l' == current) {
                    state = 885;
                    return true;
                }
                break;
            case 885:
                // GreaterEqual;
                if (';' == current) {
                    match = "\u2265";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('L' == current) {
                    state = 886;
                    return true;
                }
                break;
            case 886:
                if ('e' == current) {
                    state = 887;
                    return true;
                }
                break;
            case 887:
                if ('s' == current) {
                    state = 888;
                    return true;
                }
                break;
            case 888:
                if ('s' == current) {
                    state = 889;
                    return true;
                }
                break;
            case 889:
                // GreaterEqualLess;
                if (';' == current) {
                    match = "\u22DB";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 890:
                if ('u' == current) {
                    state = 891;
                    return true;
                }
                break;
            case 891:
                if ('l' == current) {
                    state = 892;
                    return true;
                }
                break;
            case 892:
                if ('l' == current) {
                    state = 893;
                    return true;
                }
                break;
            case 893:
                if ('E' == current) {
                    state = 894;
                    return true;
                }
                break;
            case 894:
                if ('q' == current) {
                    state = 895;
                    return true;
                }
                break;
            case 895:
                if ('u' == current) {
                    state = 896;
                    return true;
                }
                break;
            case 896:
                if ('a' == current) {
                    state = 897;
                    return true;
                }
                break;
            case 897:
                if ('l' == current) {
                    state = 898;
                    return true;
                }
                break;
            case 898:
                // GreaterFullEqual;
                if (';' == current) {
                    match = "\u2267";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 899:
                if ('r' == current) {
                    state = 900;
                    return true;
                }
                break;
            case 900:
                if ('e' == current) {
                    state = 901;
                    return true;
                }
                break;
            case 901:
                if ('a' == current) {
                    state = 902;
                    return true;
                }
                break;
            case 902:
                if ('t' == current) {
                    state = 903;
                    return true;
                }
                break;
            case 903:
                if ('e' == current) {
                    state = 904;
                    return true;
                }
                break;
            case 904:
                if ('r' == current) {
                    state = 905;
                    return true;
                }
                break;
            case 905:
                // GreaterGreater;
                if (';' == current) {
                    match = "\u2AA2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 906:
                if ('e' == current) {
                    state = 907;
                    return true;
                }
                break;
            case 907:
                if ('s' == current) {
                    state = 908;
                    return true;
                }
                break;
            case 908:
                if ('s' == current) {
                    state = 909;
                    return true;
                }
                break;
            case 909:
                // GreaterLess;
                if (';' == current) {
                    match = "\u2277";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 910:
                if ('l' == current) {
                    state = 911;
                    return true;
                }
                break;
            case 911:
                if ('a' == current) {
                    state = 912;
                    return true;
                }
                break;
            case 912:
                if ('n' == current) {
                    state = 913;
                    return true;
                }
                break;
            case 913:
                if ('t' == current) {
                    state = 914;
                    return true;
                }
                break;
            case 914:
                if ('E' == current) {
                    state = 915;
                    return true;
                }
                break;
            case 915:
                if ('q' == current) {
                    state = 916;
                    return true;
                }
                break;
            case 916:
                if ('u' == current) {
                    state = 917;
                    return true;
                }
                break;
            case 917:
                if ('a' == current) {
                    state = 918;
                    return true;
                }
                break;
            case 918:
                if ('l' == current) {
                    state = 919;
                    return true;
                }
                break;
            case 919:
                // GreaterSlantEqual;
                if (';' == current) {
                    match = "\u2A7E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 920:
                if ('i' == current) {
                    state = 921;
                    return true;
                }
                break;
            case 921:
                if ('l' == current) {
                    state = 922;
                    return true;
                }
                break;
            case 922:
                if ('d' == current) {
                    state = 923;
                    return true;
                }
                break;
            case 923:
                if ('e' == current) {
                    state = 924;
                    return true;
                }
                break;
            case 924:
                // GreaterTilde;
                if (';' == current) {
                    match = "\u2273";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 925:
                if ('c' == current) {
                    state = 926;
                    return true;
                }
                break;
            case 926:
                if ('r' == current) {
                    state = 927;
                    return true;
                }
                break;
            case 927:
                // Gscr;
                if (';' == current) {
                    match = "\uD835\uDCA2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 928:
                // Gt;
                if (';' == current) {
                    match = "\u226B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 929:
                switch (current) {
                    case 'A' :
                        state = 930;
                        return true;
                    case 'a' :
                        state = 935;
                        return true;
                    case 'c' :
                        state = 940;
                        return true;
                    case 'f' :
                        state = 944;
                        return true;
                    case 'i' :
                        state = 946;
                        return true;
                    case 'o' :
                        state = 957;
                        return true;
                    case 's' :
                        state = 972;
                        return true;
                    case 'u' :
                        state = 979;
                        return true;
                }
                break;
            case 930:
                if ('R' == current) {
                    state = 931;
                    return true;
                }
                break;
            case 931:
                if ('D' == current) {
                    state = 932;
                    return true;
                }
                break;
            case 932:
                if ('c' == current) {
                    state = 933;
                    return true;
                }
                break;
            case 933:
                if ('y' == current) {
                    state = 934;
                    return true;
                }
                break;
            case 934:
                // HARDcy;
                if (';' == current) {
                    match = "\u042A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 935:
                if ('c' == current) {
                    state = 936;
                    return true;
                }
                else if ('t' == current) {
                    state = 939;
                    return true;
                }
                break;
            case 936:
                if ('e' == current) {
                    state = 937;
                    return true;
                }
                break;
            case 937:
                if ('k' == current) {
                    state = 938;
                    return true;
                }
                break;
            case 938:
                // Hacek;
                if (';' == current) {
                    match = "\u02C7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 939:
                // Hat;
                if (';' == current) {
                    match = "^";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 940:
                if ('i' == current) {
                    state = 941;
                    return true;
                }
                break;
            case 941:
                if ('r' == current) {
                    state = 942;
                    return true;
                }
                break;
            case 942:
                if ('c' == current) {
                    state = 943;
                    return true;
                }
                break;
            case 943:
                // Hcirc;
                if (';' == current) {
                    match = "\u0124";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 944:
                if ('r' == current) {
                    state = 945;
                    return true;
                }
                break;
            case 945:
                // Hfr;
                if (';' == current) {
                    match = "\u210C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 946:
                if ('l' == current) {
                    state = 947;
                    return true;
                }
                break;
            case 947:
                if ('b' == current) {
                    state = 948;
                    return true;
                }
                break;
            case 948:
                if ('e' == current) {
                    state = 949;
                    return true;
                }
                break;
            case 949:
                if ('r' == current) {
                    state = 950;
                    return true;
                }
                break;
            case 950:
                if ('t' == current) {
                    state = 951;
                    return true;
                }
                break;
            case 951:
                if ('S' == current) {
                    state = 952;
                    return true;
                }
                break;
            case 952:
                if ('p' == current) {
                    state = 953;
                    return true;
                }
                break;
            case 953:
                if ('a' == current) {
                    state = 954;
                    return true;
                }
                break;
            case 954:
                if ('c' == current) {
                    state = 955;
                    return true;
                }
                break;
            case 955:
                if ('e' == current) {
                    state = 956;
                    return true;
                }
                break;
            case 956:
                // HilbertSpace;
                if (';' == current) {
                    match = "\u210B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 957:
                if ('p' == current) {
                    state = 958;
                    return true;
                }
                else if ('r' == current) {
                    state = 960;
                    return true;
                }
                break;
            case 958:
                if ('f' == current) {
                    state = 959;
                    return true;
                }
                break;
            case 959:
                // Hopf;
                if (';' == current) {
                    match = "\u210D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 960:
                if ('i' == current) {
                    state = 961;
                    return true;
                }
                break;
            case 961:
                if ('z' == current) {
                    state = 962;
                    return true;
                }
                break;
            case 962:
                if ('o' == current) {
                    state = 963;
                    return true;
                }
                break;
            case 963:
                if ('n' == current) {
                    state = 964;
                    return true;
                }
                break;
            case 964:
                if ('t' == current) {
                    state = 965;
                    return true;
                }
                break;
            case 965:
                if ('a' == current) {
                    state = 966;
                    return true;
                }
                break;
            case 966:
                if ('l' == current) {
                    state = 967;
                    return true;
                }
                break;
            case 967:
                if ('L' == current) {
                    state = 968;
                    return true;
                }
                break;
            case 968:
                if ('i' == current) {
                    state = 969;
                    return true;
                }
                break;
            case 969:
                if ('n' == current) {
                    state = 970;
                    return true;
                }
                break;
            case 970:
                if ('e' == current) {
                    state = 971;
                    return true;
                }
                break;
            case 971:
                // HorizontalLine;
                if (';' == current) {
                    match = "\u2500";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 972:
                if ('c' == current) {
                    state = 973;
                    return true;
                }
                else if ('t' == current) {
                    state = 975;
                    return true;
                }
                break;
            case 973:
                if ('r' == current) {
                    state = 974;
                    return true;
                }
                break;
            case 974:
                // Hscr;
                if (';' == current) {
                    match = "\u210B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 975:
                if ('r' == current) {
                    state = 976;
                    return true;
                }
                break;
            case 976:
                if ('o' == current) {
                    state = 977;
                    return true;
                }
                break;
            case 977:
                if ('k' == current) {
                    state = 978;
                    return true;
                }
                break;
            case 978:
                // Hstrok;
                if (';' == current) {
                    match = "\u0126";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 979:
                if ('m' == current) {
                    state = 980;
                    return true;
                }
                break;
            case 980:
                if ('p' == current) {
                    state = 981;
                    return true;
                }
                break;
            case 981:
                if ('D' == current) {
                    state = 982;
                    return true;
                }
                else if ('E' == current) {
                    state = 990;
                    return true;
                }
                break;
            case 982:
                if ('o' == current) {
                    state = 983;
                    return true;
                }
                break;
            case 983:
                if ('w' == current) {
                    state = 984;
                    return true;
                }
                break;
            case 984:
                if ('n' == current) {
                    state = 985;
                    return true;
                }
                break;
            case 985:
                if ('H' == current) {
                    state = 986;
                    return true;
                }
                break;
            case 986:
                if ('u' == current) {
                    state = 987;
                    return true;
                }
                break;
            case 987:
                if ('m' == current) {
                    state = 988;
                    return true;
                }
                break;
            case 988:
                if ('p' == current) {
                    state = 989;
                    return true;
                }
                break;
            case 989:
                // HumpDownHump;
                if (';' == current) {
                    match = "\u224E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 990:
                if ('q' == current) {
                    state = 991;
                    return true;
                }
                break;
            case 991:
                if ('u' == current) {
                    state = 992;
                    return true;
                }
                break;
            case 992:
                if ('a' == current) {
                    state = 993;
                    return true;
                }
                break;
            case 993:
                if ('l' == current) {
                    state = 994;
                    return true;
                }
                break;
            case 994:
                // HumpEqual;
                if (';' == current) {
                    match = "\u224F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 995:
                switch (current) {
                    case 'E' :
                        state = 996;
                        return true;
                    case 'J' :
                        state = 999;
                        return true;
                    case 'O' :
                        state = 1003;
                        return true;
                    case 'a' :
                        state = 1006;
                        return true;
                    case 'c' :
                        state = 1011;
                        return true;
                    case 'd' :
                        state = 1016;
                        return true;
                    case 'f' :
                        state = 1019;
                        return true;
                    case 'g' :
                        state = 1021;
                        return true;
                    case 'm' :
                        state = 1026;
                        return true;
                    case 'n' :
                        state = 1042;
                        return true;
                    case 'o' :
                        state = 1074;
                        return true;
                    case 's' :
                        state = 1082;
                        return true;
                    case 't' :
                        state = 1085;
                        return true;
                    case 'u' :
                        state = 1090;
                        return true;
                }
                break;
            case 996:
                if ('c' == current) {
                    state = 997;
                    return true;
                }
                break;
            case 997:
                if ('y' == current) {
                    state = 998;
                    return true;
                }
                break;
            case 998:
                // IEcy;
                if (';' == current) {
                    match = "\u0415";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 999:
                if ('l' == current) {
                    state = 1000;
                    return true;
                }
                break;
        }
        return false;
    }

    private boolean parse2(final int current) {
        consumedCount++;
        switch (state) {
            case 1000:
                if ('i' == current) {
                    state = 1001;
                    return true;
                }
                break;
            case 1001:
                if ('g' == current) {
                    state = 1002;
                    return true;
                }
                break;
            case 1002:
                // IJlig;
                if (';' == current) {
                    match = "\u0132";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1003:
                if ('c' == current) {
                    state = 1004;
                    return true;
                }
                break;
            case 1004:
                if ('y' == current) {
                    state = 1005;
                    return true;
                }
                break;
            case 1005:
                // IOcy;
                if (';' == current) {
                    match = "\u0401";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1006:
                if ('c' == current) {
                    state = 1007;
                    return true;
                }
                break;
            case 1007:
                if ('u' == current) {
                    state = 1008;
                    return true;
                }
                break;
            case 1008:
                if ('t' == current) {
                    state = 1009;
                    return true;
                }
                break;
            case 1009:
                // Iacute
                if ('e' == current) {
                    match = "\u00CD";
                    matchLength = consumedCount;
                    state = 1010;
                    return true;
                }
                break;
            case 1010:
                // Iacute;
                if (';' == current) {
                    match = "\u00CD";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1011:
                if ('i' == current) {
                    state = 1012;
                    return true;
                }
                else if ('y' == current) {
                    state = 1015;
                    return true;
                }
                break;
            case 1012:
                if ('r' == current) {
                    state = 1013;
                    return true;
                }
                break;
            case 1013:
                // Icirc
                if ('c' == current) {
                    match = "\u00CE";
                    matchLength = consumedCount;
                    state = 1014;
                    return true;
                }
                break;
            case 1014:
                // Icirc;
                if (';' == current) {
                    match = "\u00CE";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1015:
                // Icy;
                if (';' == current) {
                    match = "\u0418";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1016:
                if ('o' == current) {
                    state = 1017;
                    return true;
                }
                break;
            case 1017:
                if ('t' == current) {
                    state = 1018;
                    return true;
                }
                break;
            case 1018:
                // Idot;
                if (';' == current) {
                    match = "\u0130";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1019:
                if ('r' == current) {
                    state = 1020;
                    return true;
                }
                break;
            case 1020:
                // Ifr;
                if (';' == current) {
                    match = "\u2111";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1021:
                if ('r' == current) {
                    state = 1022;
                    return true;
                }
                break;
            case 1022:
                if ('a' == current) {
                    state = 1023;
                    return true;
                }
                break;
            case 1023:
                if ('v' == current) {
                    state = 1024;
                    return true;
                }
                break;
            case 1024:
                // Igrave
                if ('e' == current) {
                    match = "\u00CC";
                    matchLength = consumedCount;
                    state = 1025;
                    return true;
                }
                break;
            case 1025:
                // Igrave;
                if (';' == current) {
                    match = "\u00CC";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1026:
                switch (current) {
                    case ';' : // Im;
                        match = "\u2111";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'a' :
                        state = 1027;
                        return true;
                    case 'p' :
                        state = 1037;
                        return true;
                }
                break;
            case 1027:
                if ('c' == current) {
                    state = 1028;
                    return true;
                }
                else if ('g' == current) {
                    state = 1030;
                    return true;
                }
                break;
            case 1028:
                if ('r' == current) {
                    state = 1029;
                    return true;
                }
                break;
            case 1029:
                // Imacr;
                if (';' == current) {
                    match = "\u012A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1030:
                if ('i' == current) {
                    state = 1031;
                    return true;
                }
                break;
            case 1031:
                if ('n' == current) {
                    state = 1032;
                    return true;
                }
                break;
            case 1032:
                if ('a' == current) {
                    state = 1033;
                    return true;
                }
                break;
            case 1033:
                if ('r' == current) {
                    state = 1034;
                    return true;
                }
                break;
            case 1034:
                if ('y' == current) {
                    state = 1035;
                    return true;
                }
                break;
            case 1035:
                if ('I' == current) {
                    state = 1036;
                    return true;
                }
                break;
            case 1036:
                // ImaginaryI;
                if (';' == current) {
                    match = "\u2148";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1037:
                if ('l' == current) {
                    state = 1038;
                    return true;
                }
                break;
            case 1038:
                if ('i' == current) {
                    state = 1039;
                    return true;
                }
                break;
            case 1039:
                if ('e' == current) {
                    state = 1040;
                    return true;
                }
                break;
            case 1040:
                if ('s' == current) {
                    state = 1041;
                    return true;
                }
                break;
            case 1041:
                // Implies;
                if (';' == current) {
                    match = "\u21D2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1042:
                if ('t' == current) {
                    state = 1043;
                    return true;
                }
                else if ('v' == current) {
                    state = 1057;
                    return true;
                }
                break;
            case 1043:
                // Int;
                if (';' == current) {
                    match = "\u222C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state = 1044;
                    return true;
                }
                break;
            case 1044:
                if ('g' == current) {
                    state = 1045;
                    return true;
                }
                else if ('r' == current) {
                    state = 1049;
                    return true;
                }
                break;
            case 1045:
                if ('r' == current) {
                    state = 1046;
                    return true;
                }
                break;
            case 1046:
                if ('a' == current) {
                    state = 1047;
                    return true;
                }
                break;
            case 1047:
                if ('l' == current) {
                    state = 1048;
                    return true;
                }
                break;
            case 1048:
                // Integral;
                if (';' == current) {
                    match = "\u222B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1049:
                if ('s' == current) {
                    state = 1050;
                    return true;
                }
                break;
            case 1050:
                if ('e' == current) {
                    state = 1051;
                    return true;
                }
                break;
            case 1051:
                if ('c' == current) {
                    state = 1052;
                    return true;
                }
                break;
            case 1052:
                if ('t' == current) {
                    state = 1053;
                    return true;
                }
                break;
            case 1053:
                if ('i' == current) {
                    state = 1054;
                    return true;
                }
                break;
            case 1054:
                if ('o' == current) {
                    state = 1055;
                    return true;
                }
                break;
            case 1055:
                if ('n' == current) {
                    state = 1056;
                    return true;
                }
                break;
            case 1056:
                // Intersection;
                if (';' == current) {
                    match = "\u22C2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1057:
                if ('i' == current) {
                    state = 1058;
                    return true;
                }
                break;
            case 1058:
                if ('s' == current) {
                    state = 1059;
                    return true;
                }
                break;
            case 1059:
                if ('i' == current) {
                    state = 1060;
                    return true;
                }
                break;
            case 1060:
                if ('b' == current) {
                    state = 1061;
                    return true;
                }
                break;
            case 1061:
                if ('l' == current) {
                    state = 1062;
                    return true;
                }
                break;
            case 1062:
                if ('e' == current) {
                    state = 1063;
                    return true;
                }
                break;
            case 1063:
                if ('C' == current) {
                    state = 1064;
                    return true;
                }
                else if ('T' == current) {
                    state = 1069;
                    return true;
                }
                break;
            case 1064:
                if ('o' == current) {
                    state = 1065;
                    return true;
                }
                break;
            case 1065:
                if ('m' == current) {
                    state = 1066;
                    return true;
                }
                break;
            case 1066:
                if ('m' == current) {
                    state = 1067;
                    return true;
                }
                break;
            case 1067:
                if ('a' == current) {
                    state = 1068;
                    return true;
                }
                break;
            case 1068:
                // InvisibleComma;
                if (';' == current) {
                    match = "\u2063";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1069:
                if ('i' == current) {
                    state = 1070;
                    return true;
                }
                break;
            case 1070:
                if ('m' == current) {
                    state = 1071;
                    return true;
                }
                break;
            case 1071:
                if ('e' == current) {
                    state = 1072;
                    return true;
                }
                break;
            case 1072:
                if ('s' == current) {
                    state = 1073;
                    return true;
                }
                break;
            case 1073:
                // InvisibleTimes;
                if (';' == current) {
                    match = "\u2062";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1074:
                switch (current) {
                    case 'g' :
                        state = 1075;
                        return true;
                    case 'p' :
                        state = 1078;
                        return true;
                    case 't' :
                        state = 1080;
                        return true;
                }
                break;
            case 1075:
                if ('o' == current) {
                    state = 1076;
                    return true;
                }
                break;
            case 1076:
                if ('n' == current) {
                    state = 1077;
                    return true;
                }
                break;
            case 1077:
                // Iogon;
                if (';' == current) {
                    match = "\u012E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1078:
                if ('f' == current) {
                    state = 1079;
                    return true;
                }
                break;
            case 1079:
                // Iopf;
                if (';' == current) {
                    match = "\uD835\uDD40";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1080:
                if ('a' == current) {
                    state = 1081;
                    return true;
                }
                break;
            case 1081:
                // Iota;
                if (';' == current) {
                    match = "\u0399";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1082:
                if ('c' == current) {
                    state = 1083;
                    return true;
                }
                break;
            case 1083:
                if ('r' == current) {
                    state = 1084;
                    return true;
                }
                break;
            case 1084:
                // Iscr;
                if (';' == current) {
                    match = "\u2110";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1085:
                if ('i' == current) {
                    state = 1086;
                    return true;
                }
                break;
            case 1086:
                if ('l' == current) {
                    state = 1087;
                    return true;
                }
                break;
            case 1087:
                if ('d' == current) {
                    state = 1088;
                    return true;
                }
                break;
            case 1088:
                if ('e' == current) {
                    state = 1089;
                    return true;
                }
                break;
            case 1089:
                // Itilde;
                if (';' == current) {
                    match = "\u0128";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1090:
                if ('k' == current) {
                    state = 1091;
                    return true;
                }
                else if ('m' == current) {
                    state = 1094;
                    return true;
                }
                break;
            case 1091:
                if ('c' == current) {
                    state = 1092;
                    return true;
                }
                break;
            case 1092:
                if ('y' == current) {
                    state = 1093;
                    return true;
                }
                break;
            case 1093:
                // Iukcy;
                if (';' == current) {
                    match = "\u0406";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1094:
                // Iuml
                if ('l' == current) {
                    match = "\u00CF";
                    matchLength = consumedCount;
                    state = 1095;
                    return true;
                }
                break;
            case 1095:
                // Iuml;
                if (';' == current) {
                    match = "\u00CF";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1096:
                switch (current) {
                    case 'c' :
                        state = 1097;
                        return true;
                    case 'f' :
                        state = 1102;
                        return true;
                    case 'o' :
                        state = 1104;
                        return true;
                    case 's' :
                        state = 1107;
                        return true;
                    case 'u' :
                        state = 1114;
                        return true;
                }
                break;
            case 1097:
                if ('i' == current) {
                    state = 1098;
                    return true;
                }
                else if ('y' == current) {
                    state = 1101;
                    return true;
                }
                break;
            case 1098:
                if ('r' == current) {
                    state = 1099;
                    return true;
                }
                break;
            case 1099:
                if ('c' == current) {
                    state = 1100;
                    return true;
                }
                break;
            case 1100:
                // Jcirc;
                if (';' == current) {
                    match = "\u0134";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1101:
                // Jcy;
                if (';' == current) {
                    match = "\u0419";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1102:
                if ('r' == current) {
                    state = 1103;
                    return true;
                }
                break;
            case 1103:
                // Jfr;
                if (';' == current) {
                    match = "\uD835\uDD0D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1104:
                if ('p' == current) {
                    state = 1105;
                    return true;
                }
                break;
            case 1105:
                if ('f' == current) {
                    state = 1106;
                    return true;
                }
                break;
            case 1106:
                // Jopf;
                if (';' == current) {
                    match = "\uD835\uDD41";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1107:
                if ('c' == current) {
                    state = 1108;
                    return true;
                }
                else if ('e' == current) {
                    state = 1110;
                    return true;
                }
                break;
            case 1108:
                if ('r' == current) {
                    state = 1109;
                    return true;
                }
                break;
            case 1109:
                // Jscr;
                if (';' == current) {
                    match = "\uD835\uDCA5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1110:
                if ('r' == current) {
                    state = 1111;
                    return true;
                }
                break;
            case 1111:
                if ('c' == current) {
                    state = 1112;
                    return true;
                }
                break;
            case 1112:
                if ('y' == current) {
                    state = 1113;
                    return true;
                }
                break;
            case 1113:
                // Jsercy;
                if (';' == current) {
                    match = "\u0408";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1114:
                if ('k' == current) {
                    state = 1115;
                    return true;
                }
                break;
            case 1115:
                if ('c' == current) {
                    state = 1116;
                    return true;
                }
                break;
            case 1116:
                if ('y' == current) {
                    state = 1117;
                    return true;
                }
                break;
            case 1117:
                // Jukcy;
                if (';' == current) {
                    match = "\u0404";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1118:
                switch (current) {
                    case 'H' :
                        state = 1119;
                        return true;
                    case 'J' :
                        state = 1122;
                        return true;
                    case 'a' :
                        state = 1125;
                        return true;
                    case 'c' :
                        state = 1129;
                        return true;
                    case 'f' :
                        state = 1135;
                        return true;
                    case 'o' :
                        state = 1137;
                        return true;
                    case 's' :
                        state = 1140;
                        return true;
                }
                break;
            case 1119:
                if ('c' == current) {
                    state = 1120;
                    return true;
                }
                break;
            case 1120:
                if ('y' == current) {
                    state = 1121;
                    return true;
                }
                break;
            case 1121:
                // KHcy;
                if (';' == current) {
                    match = "\u0425";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1122:
                if ('c' == current) {
                    state = 1123;
                    return true;
                }
                break;
            case 1123:
                if ('y' == current) {
                    state = 1124;
                    return true;
                }
                break;
            case 1124:
                // KJcy;
                if (';' == current) {
                    match = "\u040C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1125:
                if ('p' == current) {
                    state = 1126;
                    return true;
                }
                break;
            case 1126:
                if ('p' == current) {
                    state = 1127;
                    return true;
                }
                break;
            case 1127:
                if ('a' == current) {
                    state = 1128;
                    return true;
                }
                break;
            case 1128:
                // Kappa;
                if (';' == current) {
                    match = "\u039A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1129:
                if ('e' == current) {
                    state = 1130;
                    return true;
                }
                else if ('y' == current) {
                    state = 1134;
                    return true;
                }
                break;
            case 1130:
                if ('d' == current) {
                    state = 1131;
                    return true;
                }
                break;
            case 1131:
                if ('i' == current) {
                    state = 1132;
                    return true;
                }
                break;
            case 1132:
                if ('l' == current) {
                    state = 1133;
                    return true;
                }
                break;
            case 1133:
                // Kcedil;
                if (';' == current) {
                    match = "\u0136";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1134:
                // Kcy;
                if (';' == current) {
                    match = "\u041A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1135:
                if ('r' == current) {
                    state = 1136;
                    return true;
                }
                break;
            case 1136:
                // Kfr;
                if (';' == current) {
                    match = "\uD835\uDD0E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1137:
                if ('p' == current) {
                    state = 1138;
                    return true;
                }
                break;
            case 1138:
                if ('f' == current) {
                    state = 1139;
                    return true;
                }
                break;
            case 1139:
                // Kopf;
                if (';' == current) {
                    match = "\uD835\uDD42";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1140:
                if ('c' == current) {
                    state = 1141;
                    return true;
                }
                break;
            case 1141:
                if ('r' == current) {
                    state = 1142;
                    return true;
                }
                break;
            case 1142:
                // Kscr;
                if (';' == current) {
                    match = "\uD835\uDCA6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1143:
                switch (current) {
                    case 'J' :
                        state = 1144;
                        return true;
                    case 'T' : // LT
                        match = "<";
                        matchLength = consumedCount;
                        state = 1147;
                        return true;
                    case 'a' :
                        state = 1148;
                        return true;
                    case 'c' :
                        state = 1169;
                        return true;
                    case 'e' :
                        state = 1179;
                        return true;
                    case 'f' :
                        state = 1404;
                        return true;
                    case 'l' :
                        state = 1406;
                        return true;
                    case 'm' :
                        state = 1415;
                        return true;
                    case 'o' :
                        state = 1420;
                        return true;
                    case 's' :
                        state = 1505;
                        return true;
                    case 't' :
                        state = 1513;
                        return true;
                }
                break;
            case 1144:
                if ('c' == current) {
                    state = 1145;
                    return true;
                }
                break;
            case 1145:
                if ('y' == current) {
                    state = 1146;
                    return true;
                }
                break;
            case 1146:
                // LJcy;
                if (';' == current) {
                    match = "\u0409";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1147:
                // LT;
                if (';' == current) {
                    match = "<";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1148:
                switch (current) {
                    case 'c' :
                        state = 1149;
                        return true;
                    case 'm' :
                        state = 1153;
                        return true;
                    case 'n' :
                        state = 1157;
                        return true;
                    case 'p' :
                        state = 1159;
                        return true;
                    case 'r' :
                        state = 1167;
                        return true;
                }
                break;
            case 1149:
                if ('u' == current) {
                    state = 1150;
                    return true;
                }
                break;
            case 1150:
                if ('t' == current) {
                    state = 1151;
                    return true;
                }
                break;
            case 1151:
                if ('e' == current) {
                    state = 1152;
                    return true;
                }
                break;
            case 1152:
                // Lacute;
                if (';' == current) {
                    match = "\u0139";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1153:
                if ('b' == current) {
                    state = 1154;
                    return true;
                }
                break;
            case 1154:
                if ('d' == current) {
                    state = 1155;
                    return true;
                }
                break;
            case 1155:
                if ('a' == current) {
                    state = 1156;
                    return true;
                }
                break;
            case 1156:
                // Lambda;
                if (';' == current) {
                    match = "\u039B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1157:
                if ('g' == current) {
                    state = 1158;
                    return true;
                }
                break;
            case 1158:
                // Lang;
                if (';' == current) {
                    match = "\u27EA";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1159:
                if ('l' == current) {
                    state = 1160;
                    return true;
                }
                break;
            case 1160:
                if ('a' == current) {
                    state = 1161;
                    return true;
                }
                break;
            case 1161:
                if ('c' == current) {
                    state = 1162;
                    return true;
                }
                break;
            case 1162:
                if ('e' == current) {
                    state = 1163;
                    return true;
                }
                break;
            case 1163:
                if ('t' == current) {
                    state = 1164;
                    return true;
                }
                break;
            case 1164:
                if ('r' == current) {
                    state = 1165;
                    return true;
                }
                break;
            case 1165:
                if ('f' == current) {
                    state = 1166;
                    return true;
                }
                break;
            case 1166:
                // Laplacetrf;
                if (';' == current) {
                    match = "\u2112";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1167:
                if ('r' == current) {
                    state = 1168;
                    return true;
                }
                break;
            case 1168:
                // Larr;
                if (';' == current) {
                    match = "\u219E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1169:
                switch (current) {
                    case 'a' :
                        state = 1170;
                        return true;
                    case 'e' :
                        state = 1174;
                        return true;
                    case 'y' :
                        state = 1178;
                        return true;
                }
                break;
            case 1170:
                if ('r' == current) {
                    state = 1171;
                    return true;
                }
                break;
            case 1171:
                if ('o' == current) {
                    state = 1172;
                    return true;
                }
                break;
            case 1172:
                if ('n' == current) {
                    state = 1173;
                    return true;
                }
                break;
            case 1173:
                // Lcaron;
                if (';' == current) {
                    match = "\u013D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1174:
                if ('d' == current) {
                    state = 1175;
                    return true;
                }
                break;
            case 1175:
                if ('i' == current) {
                    state = 1176;
                    return true;
                }
                break;
            case 1176:
                if ('l' == current) {
                    state = 1177;
                    return true;
                }
                break;
            case 1177:
                // Lcedil;
                if (';' == current) {
                    match = "\u013B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1178:
                // Lcy;
                if (';' == current) {
                    match = "\u041B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1179:
                if ('f' == current) {
                    state = 1180;
                    return true;
                }
                else if ('s' == current) {
                    state = 1355;
                    return true;
                }
                break;
            case 1180:
                if ('t' == current) {
                    state = 1181;
                    return true;
                }
                break;
            case 1181:
                switch (current) {
                    case 'A' :
                        state = 1182;
                        return true;
                    case 'C' :
                        state = 1211;
                        return true;
                    case 'D' :
                        state = 1218;
                        return true;
                    case 'F' :
                        state = 1251;
                        return true;
                    case 'R' :
                        state = 1256;
                        return true;
                    case 'T' :
                        state = 1272;
                        return true;
                    case 'U' :
                        state = 1301;
                        return true;
                    case 'V' :
                        state = 1331;
                        return true;
                    case 'a' :
                        state = 1340;
                        return true;
                    case 'r' :
                        state = 1345;
                        return true;
                }
                break;
            case 1182:
                if ('n' == current) {
                    state = 1183;
                    return true;
                }
                else if ('r' == current) {
                    state = 1194;
                    return true;
                }
                break;
            case 1183:
                if ('g' == current) {
                    state = 1184;
                    return true;
                }
                break;
            case 1184:
                if ('l' == current) {
                    state = 1185;
                    return true;
                }
                break;
            case 1185:
                if ('e' == current) {
                    state = 1186;
                    return true;
                }
                break;
            case 1186:
                if ('B' == current) {
                    state = 1187;
                    return true;
                }
                break;
            case 1187:
                if ('r' == current) {
                    state = 1188;
                    return true;
                }
                break;
            case 1188:
                if ('a' == current) {
                    state = 1189;
                    return true;
                }
                break;
            case 1189:
                if ('c' == current) {
                    state = 1190;
                    return true;
                }
                break;
            case 1190:
                if ('k' == current) {
                    state = 1191;
                    return true;
                }
                break;
            case 1191:
                if ('e' == current) {
                    state = 1192;
                    return true;
                }
                break;
            case 1192:
                if ('t' == current) {
                    state = 1193;
                    return true;
                }
                break;
            case 1193:
                // LeftAngleBracket;
                if (';' == current) {
                    match = "\u27E8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1194:
                if ('r' == current) {
                    state = 1195;
                    return true;
                }
                break;
            case 1195:
                if ('o' == current) {
                    state = 1196;
                    return true;
                }
                break;
            case 1196:
                if ('w' == current) {
                    state = 1197;
                    return true;
                }
                break;
            case 1197:
                switch (current) {
                    case ';' : // LeftArrow;
                        match = "\u2190";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 1198;
                        return true;
                    case 'R' :
                        state = 1201;
                        return true;
                }
                break;
            case 1198:
                if ('a' == current) {
                    state = 1199;
                    return true;
                }
                break;
            case 1199:
                if ('r' == current) {
                    state = 1200;
                    return true;
                }
                break;
            case 1200:
                // LeftArrowBar;
                if (';' == current) {
                    match = "\u21E4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1201:
                if ('i' == current) {
                    state = 1202;
                    return true;
                }
                break;
            case 1202:
                if ('g' == current) {
                    state = 1203;
                    return true;
                }
                break;
            case 1203:
                if ('h' == current) {
                    state = 1204;
                    return true;
                }
                break;
            case 1204:
                if ('t' == current) {
                    state = 1205;
                    return true;
                }
                break;
            case 1205:
                if ('A' == current) {
                    state = 1206;
                    return true;
                }
                break;
            case 1206:
                if ('r' == current) {
                    state = 1207;
                    return true;
                }
                break;
            case 1207:
                if ('r' == current) {
                    state = 1208;
                    return true;
                }
                break;
            case 1208:
                if ('o' == current) {
                    state = 1209;
                    return true;
                }
                break;
            case 1209:
                if ('w' == current) {
                    state = 1210;
                    return true;
                }
                break;
            case 1210:
                // LeftArrowRightArrow;
                if (';' == current) {
                    match = "\u21C6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1211:
                if ('e' == current) {
                    state = 1212;
                    return true;
                }
                break;
            case 1212:
                if ('i' == current) {
                    state = 1213;
                    return true;
                }
                break;
            case 1213:
                if ('l' == current) {
                    state = 1214;
                    return true;
                }
                break;
            case 1214:
                if ('i' == current) {
                    state = 1215;
                    return true;
                }
                break;
            case 1215:
                if ('n' == current) {
                    state = 1216;
                    return true;
                }
                break;
            case 1216:
                if ('g' == current) {
                    state = 1217;
                    return true;
                }
                break;
            case 1217:
                // LeftCeiling;
                if (';' == current) {
                    match = "\u2308";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1218:
                if ('o' == current) {
                    state = 1219;
                    return true;
                }
                break;
            case 1219:
                if ('u' == current) {
                    state = 1220;
                    return true;
                }
                else if ('w' == current) {
                    state = 1231;
                    return true;
                }
                break;
            case 1220:
                if ('b' == current) {
                    state = 1221;
                    return true;
                }
                break;
            case 1221:
                if ('l' == current) {
                    state = 1222;
                    return true;
                }
                break;
            case 1222:
                if ('e' == current) {
                    state = 1223;
                    return true;
                }
                break;
            case 1223:
                if ('B' == current) {
                    state = 1224;
                    return true;
                }
                break;
            case 1224:
                if ('r' == current) {
                    state = 1225;
                    return true;
                }
                break;
            case 1225:
                if ('a' == current) {
                    state = 1226;
                    return true;
                }
                break;
            case 1226:
                if ('c' == current) {
                    state = 1227;
                    return true;
                }
                break;
            case 1227:
                if ('k' == current) {
                    state = 1228;
                    return true;
                }
                break;
            case 1228:
                if ('e' == current) {
                    state = 1229;
                    return true;
                }
                break;
            case 1229:
                if ('t' == current) {
                    state = 1230;
                    return true;
                }
                break;
            case 1230:
                // LeftDoubleBracket;
                if (';' == current) {
                    match = "\u27E6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1231:
                if ('n' == current) {
                    state = 1232;
                    return true;
                }
                break;
            case 1232:
                if ('T' == current) {
                    state = 1233;
                    return true;
                }
                else if ('V' == current) {
                    state = 1242;
                    return true;
                }
                break;
            case 1233:
                if ('e' == current) {
                    state = 1234;
                    return true;
                }
                break;
            case 1234:
                if ('e' == current) {
                    state = 1235;
                    return true;
                }
                break;
            case 1235:
                if ('V' == current) {
                    state = 1236;
                    return true;
                }
                break;
            case 1236:
                if ('e' == current) {
                    state = 1237;
                    return true;
                }
                break;
            case 1237:
                if ('c' == current) {
                    state = 1238;
                    return true;
                }
                break;
            case 1238:
                if ('t' == current) {
                    state = 1239;
                    return true;
                }
                break;
            case 1239:
                if ('o' == current) {
                    state = 1240;
                    return true;
                }
                break;
            case 1240:
                if ('r' == current) {
                    state = 1241;
                    return true;
                }
                break;
            case 1241:
                // LeftDownTeeVector;
                if (';' == current) {
                    match = "\u2961";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1242:
                if ('e' == current) {
                    state = 1243;
                    return true;
                }
                break;
            case 1243:
                if ('c' == current) {
                    state = 1244;
                    return true;
                }
                break;
            case 1244:
                if ('t' == current) {
                    state = 1245;
                    return true;
                }
                break;
            case 1245:
                if ('o' == current) {
                    state = 1246;
                    return true;
                }
                break;
            case 1246:
                if ('r' == current) {
                    state = 1247;
                    return true;
                }
                break;
            case 1247:
                // LeftDownVector;
                if (';' == current) {
                    match = "\u21C3";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('B' == current) {
                    state = 1248;
                    return true;
                }
                break;
            case 1248:
                if ('a' == current) {
                    state = 1249;
                    return true;
                }
                break;
            case 1249:
                if ('r' == current) {
                    state = 1250;
                    return true;
                }
                break;
            case 1250:
                // LeftDownVectorBar;
                if (';' == current) {
                    match = "\u2959";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1251:
                if ('l' == current) {
                    state = 1252;
                    return true;
                }
                break;
            case 1252:
                if ('o' == current) {
                    state = 1253;
                    return true;
                }
                break;
            case 1253:
                if ('o' == current) {
                    state = 1254;
                    return true;
                }
                break;
            case 1254:
                if ('r' == current) {
                    state = 1255;
                    return true;
                }
                break;
            case 1255:
                // LeftFloor;
                if (';' == current) {
                    match = "\u230A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1256:
                if ('i' == current) {
                    state = 1257;
                    return true;
                }
                break;
            case 1257:
                if ('g' == current) {
                    state = 1258;
                    return true;
                }
                break;
            case 1258:
                if ('h' == current) {
                    state = 1259;
                    return true;
                }
                break;
            case 1259:
                if ('t' == current) {
                    state = 1260;
                    return true;
                }
                break;
            case 1260:
                if ('A' == current) {
                    state = 1261;
                    return true;
                }
                else if ('V' == current) {
                    state = 1266;
                    return true;
                }
                break;
            case 1261:
                if ('r' == current) {
                    state = 1262;
                    return true;
                }
                break;
            case 1262:
                if ('r' == current) {
                    state = 1263;
                    return true;
                }
                break;
            case 1263:
                if ('o' == current) {
                    state = 1264;
                    return true;
                }
                break;
            case 1264:
                if ('w' == current) {
                    state = 1265;
                    return true;
                }
                break;
            case 1265:
                // LeftRightArrow;
                if (';' == current) {
                    match = "\u2194";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1266:
                if ('e' == current) {
                    state = 1267;
                    return true;
                }
                break;
            case 1267:
                if ('c' == current) {
                    state = 1268;
                    return true;
                }
                break;
            case 1268:
                if ('t' == current) {
                    state = 1269;
                    return true;
                }
                break;
            case 1269:
                if ('o' == current) {
                    state = 1270;
                    return true;
                }
                break;
            case 1270:
                if ('r' == current) {
                    state = 1271;
                    return true;
                }
                break;
            case 1271:
                // LeftRightVector;
                if (';' == current) {
                    match = "\u294E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1272:
                if ('e' == current) {
                    state = 1273;
                    return true;
                }
                else if ('r' == current) {
                    state = 1286;
                    return true;
                }
                break;
            case 1273:
                if ('e' == current) {
                    state = 1274;
                    return true;
                }
                break;
            case 1274:
                switch (current) {
                    case ';' : // LeftTee;
                        match = "\u22A3";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'A' :
                        state = 1275;
                        return true;
                    case 'V' :
                        state = 1280;
                        return true;
                }
                break;
            case 1275:
                if ('r' == current) {
                    state = 1276;
                    return true;
                }
                break;
            case 1276:
                if ('r' == current) {
                    state = 1277;
                    return true;
                }
                break;
            case 1277:
                if ('o' == current) {
                    state = 1278;
                    return true;
                }
                break;
            case 1278:
                if ('w' == current) {
                    state = 1279;
                    return true;
                }
                break;
            case 1279:
                // LeftTeeArrow;
                if (';' == current) {
                    match = "\u21A4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1280:
                if ('e' == current) {
                    state = 1281;
                    return true;
                }
                break;
            case 1281:
                if ('c' == current) {
                    state = 1282;
                    return true;
                }
                break;
            case 1282:
                if ('t' == current) {
                    state = 1283;
                    return true;
                }
                break;
            case 1283:
                if ('o' == current) {
                    state = 1284;
                    return true;
                }
                break;
            case 1284:
                if ('r' == current) {
                    state = 1285;
                    return true;
                }
                break;
            case 1285:
                // LeftTeeVector;
                if (';' == current) {
                    match = "\u295A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1286:
                if ('i' == current) {
                    state = 1287;
                    return true;
                }
                break;
            case 1287:
                if ('a' == current) {
                    state = 1288;
                    return true;
                }
                break;
            case 1288:
                if ('n' == current) {
                    state = 1289;
                    return true;
                }
                break;
            case 1289:
                if ('g' == current) {
                    state = 1290;
                    return true;
                }
                break;
            case 1290:
                if ('l' == current) {
                    state = 1291;
                    return true;
                }
                break;
            case 1291:
                if ('e' == current) {
                    state = 1292;
                    return true;
                }
                break;
            case 1292:
                switch (current) {
                    case ';' : // LeftTriangle;
                        match = "\u22B2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 1293;
                        return true;
                    case 'E' :
                        state = 1296;
                        return true;
                }
                break;
            case 1293:
                if ('a' == current) {
                    state = 1294;
                    return true;
                }
                break;
            case 1294:
                if ('r' == current) {
                    state = 1295;
                    return true;
                }
                break;
            case 1295:
                // LeftTriangleBar;
                if (';' == current) {
                    match = "\u29CF";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1296:
                if ('q' == current) {
                    state = 1297;
                    return true;
                }
                break;
            case 1297:
                if ('u' == current) {
                    state = 1298;
                    return true;
                }
                break;
            case 1298:
                if ('a' == current) {
                    state = 1299;
                    return true;
                }
                break;
            case 1299:
                if ('l' == current) {
                    state = 1300;
                    return true;
                }
                break;
            case 1300:
                // LeftTriangleEqual;
                if (';' == current) {
                    match = "\u22B4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1301:
                if ('p' == current) {
                    state = 1302;
                    return true;
                }
                break;
            case 1302:
                switch (current) {
                    case 'D' :
                        state = 1303;
                        return true;
                    case 'T' :
                        state = 1313;
                        return true;
                    case 'V' :
                        state = 1322;
                        return true;
                }
                break;
            case 1303:
                if ('o' == current) {
                    state = 1304;
                    return true;
                }
                break;
            case 1304:
                if ('w' == current) {
                    state = 1305;
                    return true;
                }
                break;
            case 1305:
                if ('n' == current) {
                    state = 1306;
                    return true;
                }
                break;
            case 1306:
                if ('V' == current) {
                    state = 1307;
                    return true;
                }
                break;
            case 1307:
                if ('e' == current) {
                    state = 1308;
                    return true;
                }
                break;
            case 1308:
                if ('c' == current) {
                    state = 1309;
                    return true;
                }
                break;
            case 1309:
                if ('t' == current) {
                    state = 1310;
                    return true;
                }
                break;
            case 1310:
                if ('o' == current) {
                    state = 1311;
                    return true;
                }
                break;
            case 1311:
                if ('r' == current) {
                    state = 1312;
                    return true;
                }
                break;
            case 1312:
                // LeftUpDownVector;
                if (';' == current) {
                    match = "\u2951";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1313:
                if ('e' == current) {
                    state = 1314;
                    return true;
                }
                break;
            case 1314:
                if ('e' == current) {
                    state = 1315;
                    return true;
                }
                break;
            case 1315:
                if ('V' == current) {
                    state = 1316;
                    return true;
                }
                break;
            case 1316:
                if ('e' == current) {
                    state = 1317;
                    return true;
                }
                break;
            case 1317:
                if ('c' == current) {
                    state = 1318;
                    return true;
                }
                break;
            case 1318:
                if ('t' == current) {
                    state = 1319;
                    return true;
                }
                break;
            case 1319:
                if ('o' == current) {
                    state = 1320;
                    return true;
                }
                break;
            case 1320:
                if ('r' == current) {
                    state = 1321;
                    return true;
                }
                break;
            case 1321:
                // LeftUpTeeVector;
                if (';' == current) {
                    match = "\u2960";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1322:
                if ('e' == current) {
                    state = 1323;
                    return true;
                }
                break;
            case 1323:
                if ('c' == current) {
                    state = 1324;
                    return true;
                }
                break;
            case 1324:
                if ('t' == current) {
                    state = 1325;
                    return true;
                }
                break;
            case 1325:
                if ('o' == current) {
                    state = 1326;
                    return true;
                }
                break;
            case 1326:
                if ('r' == current) {
                    state = 1327;
                    return true;
                }
                break;
            case 1327:
                // LeftUpVector;
                if (';' == current) {
                    match = "\u21BF";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('B' == current) {
                    state = 1328;
                    return true;
                }
                break;
            case 1328:
                if ('a' == current) {
                    state = 1329;
                    return true;
                }
                break;
            case 1329:
                if ('r' == current) {
                    state = 1330;
                    return true;
                }
                break;
            case 1330:
                // LeftUpVectorBar;
                if (';' == current) {
                    match = "\u2958";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1331:
                if ('e' == current) {
                    state = 1332;
                    return true;
                }
                break;
            case 1332:
                if ('c' == current) {
                    state = 1333;
                    return true;
                }
                break;
            case 1333:
                if ('t' == current) {
                    state = 1334;
                    return true;
                }
                break;
            case 1334:
                if ('o' == current) {
                    state = 1335;
                    return true;
                }
                break;
            case 1335:
                if ('r' == current) {
                    state = 1336;
                    return true;
                }
                break;
            case 1336:
                // LeftVector;
                if (';' == current) {
                    match = "\u21BC";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('B' == current) {
                    state = 1337;
                    return true;
                }
                break;
            case 1337:
                if ('a' == current) {
                    state = 1338;
                    return true;
                }
                break;
            case 1338:
                if ('r' == current) {
                    state = 1339;
                    return true;
                }
                break;
            case 1339:
                // LeftVectorBar;
                if (';' == current) {
                    match = "\u2952";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1340:
                if ('r' == current) {
                    state = 1341;
                    return true;
                }
                break;
            case 1341:
                if ('r' == current) {
                    state = 1342;
                    return true;
                }
                break;
            case 1342:
                if ('o' == current) {
                    state = 1343;
                    return true;
                }
                break;
            case 1343:
                if ('w' == current) {
                    state = 1344;
                    return true;
                }
                break;
            case 1344:
                // Leftarrow;
                if (';' == current) {
                    match = "\u21D0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1345:
                if ('i' == current) {
                    state = 1346;
                    return true;
                }
                break;
            case 1346:
                if ('g' == current) {
                    state = 1347;
                    return true;
                }
                break;
            case 1347:
                if ('h' == current) {
                    state = 1348;
                    return true;
                }
                break;
            case 1348:
                if ('t' == current) {
                    state = 1349;
                    return true;
                }
                break;
            case 1349:
                if ('a' == current) {
                    state = 1350;
                    return true;
                }
                break;
            case 1350:
                if ('r' == current) {
                    state = 1351;
                    return true;
                }
                break;
            case 1351:
                if ('r' == current) {
                    state = 1352;
                    return true;
                }
                break;
            case 1352:
                if ('o' == current) {
                    state = 1353;
                    return true;
                }
                break;
            case 1353:
                if ('w' == current) {
                    state = 1354;
                    return true;
                }
                break;
            case 1354:
                // Leftrightarrow;
                if (';' == current) {
                    match = "\u21D4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1355:
                if ('s' == current) {
                    state = 1356;
                    return true;
                }
                break;
            case 1356:
                switch (current) {
                    case 'E' :
                        state = 1357;
                        return true;
                    case 'F' :
                        state = 1369;
                        return true;
                    case 'G' :
                        state = 1378;
                        return true;
                    case 'L' :
                        state = 1385;
                        return true;
                    case 'S' :
                        state = 1389;
                        return true;
                    case 'T' :
                        state = 1399;
                        return true;
                }
                break;
            case 1357:
                if ('q' == current) {
                    state = 1358;
                    return true;
                }
                break;
            case 1358:
                if ('u' == current) {
                    state = 1359;
                    return true;
                }
                break;
            case 1359:
                if ('a' == current) {
                    state = 1360;
                    return true;
                }
                break;
            case 1360:
                if ('l' == current) {
                    state = 1361;
                    return true;
                }
                break;
            case 1361:
                if ('G' == current) {
                    state = 1362;
                    return true;
                }
                break;
            case 1362:
                if ('r' == current) {
                    state = 1363;
                    return true;
                }
                break;
            case 1363:
                if ('e' == current) {
                    state = 1364;
                    return true;
                }
                break;
            case 1364:
                if ('a' == current) {
                    state = 1365;
                    return true;
                }
                break;
            case 1365:
                if ('t' == current) {
                    state = 1366;
                    return true;
                }
                break;
            case 1366:
                if ('e' == current) {
                    state = 1367;
                    return true;
                }
                break;
            case 1367:
                if ('r' == current) {
                    state = 1368;
                    return true;
                }
                break;
            case 1368:
                // LessEqualGreater;
                if (';' == current) {
                    match = "\u22DA";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1369:
                if ('u' == current) {
                    state = 1370;
                    return true;
                }
                break;
            case 1370:
                if ('l' == current) {
                    state = 1371;
                    return true;
                }
                break;
            case 1371:
                if ('l' == current) {
                    state = 1372;
                    return true;
                }
                break;
            case 1372:
                if ('E' == current) {
                    state = 1373;
                    return true;
                }
                break;
            case 1373:
                if ('q' == current) {
                    state = 1374;
                    return true;
                }
                break;
            case 1374:
                if ('u' == current) {
                    state = 1375;
                    return true;
                }
                break;
            case 1375:
                if ('a' == current) {
                    state = 1376;
                    return true;
                }
                break;
            case 1376:
                if ('l' == current) {
                    state = 1377;
                    return true;
                }
                break;
            case 1377:
                // LessFullEqual;
                if (';' == current) {
                    match = "\u2266";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1378:
                if ('r' == current) {
                    state = 1379;
                    return true;
                }
                break;
            case 1379:
                if ('e' == current) {
                    state = 1380;
                    return true;
                }
                break;
            case 1380:
                if ('a' == current) {
                    state = 1381;
                    return true;
                }
                break;
            case 1381:
                if ('t' == current) {
                    state = 1382;
                    return true;
                }
                break;
            case 1382:
                if ('e' == current) {
                    state = 1383;
                    return true;
                }
                break;
            case 1383:
                if ('r' == current) {
                    state = 1384;
                    return true;
                }
                break;
            case 1384:
                // LessGreater;
                if (';' == current) {
                    match = "\u2276";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1385:
                if ('e' == current) {
                    state = 1386;
                    return true;
                }
                break;
            case 1386:
                if ('s' == current) {
                    state = 1387;
                    return true;
                }
                break;
            case 1387:
                if ('s' == current) {
                    state = 1388;
                    return true;
                }
                break;
            case 1388:
                // LessLess;
                if (';' == current) {
                    match = "\u2AA1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1389:
                if ('l' == current) {
                    state = 1390;
                    return true;
                }
                break;
            case 1390:
                if ('a' == current) {
                    state = 1391;
                    return true;
                }
                break;
            case 1391:
                if ('n' == current) {
                    state = 1392;
                    return true;
                }
                break;
            case 1392:
                if ('t' == current) {
                    state = 1393;
                    return true;
                }
                break;
            case 1393:
                if ('E' == current) {
                    state = 1394;
                    return true;
                }
                break;
            case 1394:
                if ('q' == current) {
                    state = 1395;
                    return true;
                }
                break;
            case 1395:
                if ('u' == current) {
                    state = 1396;
                    return true;
                }
                break;
            case 1396:
                if ('a' == current) {
                    state = 1397;
                    return true;
                }
                break;
            case 1397:
                if ('l' == current) {
                    state = 1398;
                    return true;
                }
                break;
            case 1398:
                // LessSlantEqual;
                if (';' == current) {
                    match = "\u2A7D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1399:
                if ('i' == current) {
                    state = 1400;
                    return true;
                }
                break;
            case 1400:
                if ('l' == current) {
                    state = 1401;
                    return true;
                }
                break;
            case 1401:
                if ('d' == current) {
                    state = 1402;
                    return true;
                }
                break;
            case 1402:
                if ('e' == current) {
                    state = 1403;
                    return true;
                }
                break;
            case 1403:
                // LessTilde;
                if (';' == current) {
                    match = "\u2272";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1404:
                if ('r' == current) {
                    state = 1405;
                    return true;
                }
                break;
            case 1405:
                // Lfr;
                if (';' == current) {
                    match = "\uD835\uDD0F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1406:
                // Ll;
                if (';' == current) {
                    match = "\u22D8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state = 1407;
                    return true;
                }
                break;
            case 1407:
                if ('f' == current) {
                    state = 1408;
                    return true;
                }
                break;
            case 1408:
                if ('t' == current) {
                    state = 1409;
                    return true;
                }
                break;
            case 1409:
                if ('a' == current) {
                    state = 1410;
                    return true;
                }
                break;
            case 1410:
                if ('r' == current) {
                    state = 1411;
                    return true;
                }
                break;
            case 1411:
                if ('r' == current) {
                    state = 1412;
                    return true;
                }
                break;
            case 1412:
                if ('o' == current) {
                    state = 1413;
                    return true;
                }
                break;
            case 1413:
                if ('w' == current) {
                    state = 1414;
                    return true;
                }
                break;
            case 1414:
                // Lleftarrow;
                if (';' == current) {
                    match = "\u21DA";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1415:
                if ('i' == current) {
                    state = 1416;
                    return true;
                }
                break;
            case 1416:
                if ('d' == current) {
                    state = 1417;
                    return true;
                }
                break;
            case 1417:
                if ('o' == current) {
                    state = 1418;
                    return true;
                }
                break;
            case 1418:
                if ('t' == current) {
                    state = 1419;
                    return true;
                }
                break;
            case 1419:
                // Lmidot;
                if (';' == current) {
                    match = "\u013F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1420:
                switch (current) {
                    case 'n' :
                        state = 1421;
                        return true;
                    case 'p' :
                        state = 1481;
                        return true;
                    case 'w' :
                        state = 1483;
                        return true;
                }
                break;
            case 1421:
                if ('g' == current) {
                    state = 1422;
                    return true;
                }
                break;
            case 1422:
                switch (current) {
                    case 'L' :
                        state = 1423;
                        return true;
                    case 'R' :
                        state = 1442;
                        return true;
                    case 'l' :
                        state = 1452;
                        return true;
                    case 'r' :
                        state = 1471;
                        return true;
                }
                break;
            case 1423:
                if ('e' == current) {
                    state = 1424;
                    return true;
                }
                break;
            case 1424:
                if ('f' == current) {
                    state = 1425;
                    return true;
                }
                break;
            case 1425:
                if ('t' == current) {
                    state = 1426;
                    return true;
                }
                break;
            case 1426:
                if ('A' == current) {
                    state = 1427;
                    return true;
                }
                else if ('R' == current) {
                    state = 1432;
                    return true;
                }
                break;
            case 1427:
                if ('r' == current) {
                    state = 1428;
                    return true;
                }
                break;
            case 1428:
                if ('r' == current) {
                    state = 1429;
                    return true;
                }
                break;
            case 1429:
                if ('o' == current) {
                    state = 1430;
                    return true;
                }
                break;
            case 1430:
                if ('w' == current) {
                    state = 1431;
                    return true;
                }
                break;
            case 1431:
                // LongLeftArrow;
                if (';' == current) {
                    match = "\u27F5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1432:
                if ('i' == current) {
                    state = 1433;
                    return true;
                }
                break;
            case 1433:
                if ('g' == current) {
                    state = 1434;
                    return true;
                }
                break;
            case 1434:
                if ('h' == current) {
                    state = 1435;
                    return true;
                }
                break;
            case 1435:
                if ('t' == current) {
                    state = 1436;
                    return true;
                }
                break;
            case 1436:
                if ('A' == current) {
                    state = 1437;
                    return true;
                }
                break;
            case 1437:
                if ('r' == current) {
                    state = 1438;
                    return true;
                }
                break;
            case 1438:
                if ('r' == current) {
                    state = 1439;
                    return true;
                }
                break;
            case 1439:
                if ('o' == current) {
                    state = 1440;
                    return true;
                }
                break;
            case 1440:
                if ('w' == current) {
                    state = 1441;
                    return true;
                }
                break;
            case 1441:
                // LongLeftRightArrow;
                if (';' == current) {
                    match = "\u27F7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1442:
                if ('i' == current) {
                    state = 1443;
                    return true;
                }
                break;
            case 1443:
                if ('g' == current) {
                    state = 1444;
                    return true;
                }
                break;
            case 1444:
                if ('h' == current) {
                    state = 1445;
                    return true;
                }
                break;
            case 1445:
                if ('t' == current) {
                    state = 1446;
                    return true;
                }
                break;
            case 1446:
                if ('A' == current) {
                    state = 1447;
                    return true;
                }
                break;
            case 1447:
                if ('r' == current) {
                    state = 1448;
                    return true;
                }
                break;
            case 1448:
                if ('r' == current) {
                    state = 1449;
                    return true;
                }
                break;
            case 1449:
                if ('o' == current) {
                    state = 1450;
                    return true;
                }
                break;
            case 1450:
                if ('w' == current) {
                    state = 1451;
                    return true;
                }
                break;
            case 1451:
                // LongRightArrow;
                if (';' == current) {
                    match = "\u27F6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1452:
                if ('e' == current) {
                    state = 1453;
                    return true;
                }
                break;
            case 1453:
                if ('f' == current) {
                    state = 1454;
                    return true;
                }
                break;
            case 1454:
                if ('t' == current) {
                    state = 1455;
                    return true;
                }
                break;
            case 1455:
                if ('a' == current) {
                    state = 1456;
                    return true;
                }
                else if ('r' == current) {
                    state = 1461;
                    return true;
                }
                break;
            case 1456:
                if ('r' == current) {
                    state = 1457;
                    return true;
                }
                break;
            case 1457:
                if ('r' == current) {
                    state = 1458;
                    return true;
                }
                break;
            case 1458:
                if ('o' == current) {
                    state = 1459;
                    return true;
                }
                break;
            case 1459:
                if ('w' == current) {
                    state = 1460;
                    return true;
                }
                break;
            case 1460:
                // Longleftarrow;
                if (';' == current) {
                    match = "\u27F8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1461:
                if ('i' == current) {
                    state = 1462;
                    return true;
                }
                break;
            case 1462:
                if ('g' == current) {
                    state = 1463;
                    return true;
                }
                break;
            case 1463:
                if ('h' == current) {
                    state = 1464;
                    return true;
                }
                break;
            case 1464:
                if ('t' == current) {
                    state = 1465;
                    return true;
                }
                break;
            case 1465:
                if ('a' == current) {
                    state = 1466;
                    return true;
                }
                break;
            case 1466:
                if ('r' == current) {
                    state = 1467;
                    return true;
                }
                break;
            case 1467:
                if ('r' == current) {
                    state = 1468;
                    return true;
                }
                break;
            case 1468:
                if ('o' == current) {
                    state = 1469;
                    return true;
                }
                break;
            case 1469:
                if ('w' == current) {
                    state = 1470;
                    return true;
                }
                break;
            case 1470:
                // Longleftrightarrow;
                if (';' == current) {
                    match = "\u27FA";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1471:
                if ('i' == current) {
                    state = 1472;
                    return true;
                }
                break;
            case 1472:
                if ('g' == current) {
                    state = 1473;
                    return true;
                }
                break;
            case 1473:
                if ('h' == current) {
                    state = 1474;
                    return true;
                }
                break;
            case 1474:
                if ('t' == current) {
                    state = 1475;
                    return true;
                }
                break;
            case 1475:
                if ('a' == current) {
                    state = 1476;
                    return true;
                }
                break;
            case 1476:
                if ('r' == current) {
                    state = 1477;
                    return true;
                }
                break;
            case 1477:
                if ('r' == current) {
                    state = 1478;
                    return true;
                }
                break;
            case 1478:
                if ('o' == current) {
                    state = 1479;
                    return true;
                }
                break;
            case 1479:
                if ('w' == current) {
                    state = 1480;
                    return true;
                }
                break;
            case 1480:
                // Longrightarrow;
                if (';' == current) {
                    match = "\u27F9";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1481:
                if ('f' == current) {
                    state = 1482;
                    return true;
                }
                break;
            case 1482:
                // Lopf;
                if (';' == current) {
                    match = "\uD835\uDD43";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1483:
                if ('e' == current) {
                    state = 1484;
                    return true;
                }
                break;
            case 1484:
                if ('r' == current) {
                    state = 1485;
                    return true;
                }
                break;
            case 1485:
                if ('L' == current) {
                    state = 1486;
                    return true;
                }
                else if ('R' == current) {
                    state = 1495;
                    return true;
                }
                break;
            case 1486:
                if ('e' == current) {
                    state = 1487;
                    return true;
                }
                break;
            case 1487:
                if ('f' == current) {
                    state = 1488;
                    return true;
                }
                break;
            case 1488:
                if ('t' == current) {
                    state = 1489;
                    return true;
                }
                break;
            case 1489:
                if ('A' == current) {
                    state = 1490;
                    return true;
                }
                break;
            case 1490:
                if ('r' == current) {
                    state = 1491;
                    return true;
                }
                break;
            case 1491:
                if ('r' == current) {
                    state = 1492;
                    return true;
                }
                break;
            case 1492:
                if ('o' == current) {
                    state = 1493;
                    return true;
                }
                break;
            case 1493:
                if ('w' == current) {
                    state = 1494;
                    return true;
                }
                break;
            case 1494:
                // LowerLeftArrow;
                if (';' == current) {
                    match = "\u2199";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1495:
                if ('i' == current) {
                    state = 1496;
                    return true;
                }
                break;
            case 1496:
                if ('g' == current) {
                    state = 1497;
                    return true;
                }
                break;
            case 1497:
                if ('h' == current) {
                    state = 1498;
                    return true;
                }
                break;
            case 1498:
                if ('t' == current) {
                    state = 1499;
                    return true;
                }
                break;
            case 1499:
                if ('A' == current) {
                    state = 1500;
                    return true;
                }
                break;
            case 1500:
                if ('r' == current) {
                    state = 1501;
                    return true;
                }
                break;
            case 1501:
                if ('r' == current) {
                    state = 1502;
                    return true;
                }
                break;
            case 1502:
                if ('o' == current) {
                    state = 1503;
                    return true;
                }
                break;
            case 1503:
                if ('w' == current) {
                    state = 1504;
                    return true;
                }
                break;
            case 1504:
                // LowerRightArrow;
                if (';' == current) {
                    match = "\u2198";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1505:
                switch (current) {
                    case 'c' :
                        state = 1506;
                        return true;
                    case 'h' :
                        state = 1508;
                        return true;
                    case 't' :
                        state = 1509;
                        return true;
                }
                break;
            case 1506:
                if ('r' == current) {
                    state = 1507;
                    return true;
                }
                break;
            case 1507:
                // Lscr;
                if (';' == current) {
                    match = "\u2112";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1508:
                // Lsh;
                if (';' == current) {
                    match = "\u21B0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1509:
                if ('r' == current) {
                    state = 1510;
                    return true;
                }
                break;
            case 1510:
                if ('o' == current) {
                    state = 1511;
                    return true;
                }
                break;
            case 1511:
                if ('k' == current) {
                    state = 1512;
                    return true;
                }
                break;
            case 1512:
                // Lstrok;
                if (';' == current) {
                    match = "\u0141";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1513:
                // Lt;
                if (';' == current) {
                    match = "\u226A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1514:
                switch (current) {
                    case 'a' :
                        state = 1515;
                        return true;
                    case 'c' :
                        state = 1517;
                        return true;
                    case 'e' :
                        state = 1519;
                        return true;
                    case 'f' :
                        state = 1536;
                        return true;
                    case 'i' :
                        state = 1538;
                        return true;
                    case 'o' :
                        state = 1546;
                        return true;
                    case 's' :
                        state = 1549;
                        return true;
                    case 'u' :
                        state = 1552;
                        return true;
                }
                break;
            case 1515:
                if ('p' == current) {
                    state = 1516;
                    return true;
                }
                break;
            case 1516:
                // Map;
                if (';' == current) {
                    match = "\u2905";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1517:
                if ('y' == current) {
                    state = 1518;
                    return true;
                }
                break;
            case 1518:
                // Mcy;
                if (';' == current) {
                    match = "\u041C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1519:
                if ('d' == current) {
                    state = 1520;
                    return true;
                }
                else if ('l' == current) {
                    state = 1529;
                    return true;
                }
                break;
            case 1520:
                if ('i' == current) {
                    state = 1521;
                    return true;
                }
                break;
            case 1521:
                if ('u' == current) {
                    state = 1522;
                    return true;
                }
                break;
            case 1522:
                if ('m' == current) {
                    state = 1523;
                    return true;
                }
                break;
            case 1523:
                if ('S' == current) {
                    state = 1524;
                    return true;
                }
                break;
            case 1524:
                if ('p' == current) {
                    state = 1525;
                    return true;
                }
                break;
            case 1525:
                if ('a' == current) {
                    state = 1526;
                    return true;
                }
                break;
            case 1526:
                if ('c' == current) {
                    state = 1527;
                    return true;
                }
                break;
            case 1527:
                if ('e' == current) {
                    state = 1528;
                    return true;
                }
                break;
            case 1528:
                // MediumSpace;
                if (';' == current) {
                    match = "\u205F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1529:
                if ('l' == current) {
                    state = 1530;
                    return true;
                }
                break;
            case 1530:
                if ('i' == current) {
                    state = 1531;
                    return true;
                }
                break;
            case 1531:
                if ('n' == current) {
                    state = 1532;
                    return true;
                }
                break;
            case 1532:
                if ('t' == current) {
                    state = 1533;
                    return true;
                }
                break;
            case 1533:
                if ('r' == current) {
                    state = 1534;
                    return true;
                }
                break;
            case 1534:
                if ('f' == current) {
                    state = 1535;
                    return true;
                }
                break;
            case 1535:
                // Mellintrf;
                if (';' == current) {
                    match = "\u2133";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1536:
                if ('r' == current) {
                    state = 1537;
                    return true;
                }
                break;
            case 1537:
                // Mfr;
                if (';' == current) {
                    match = "\uD835\uDD10";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1538:
                if ('n' == current) {
                    state = 1539;
                    return true;
                }
                break;
            case 1539:
                if ('u' == current) {
                    state = 1540;
                    return true;
                }
                break;
            case 1540:
                if ('s' == current) {
                    state = 1541;
                    return true;
                }
                break;
            case 1541:
                if ('P' == current) {
                    state = 1542;
                    return true;
                }
                break;
            case 1542:
                if ('l' == current) {
                    state = 1543;
                    return true;
                }
                break;
            case 1543:
                if ('u' == current) {
                    state = 1544;
                    return true;
                }
                break;
            case 1544:
                if ('s' == current) {
                    state = 1545;
                    return true;
                }
                break;
            case 1545:
                // MinusPlus;
                if (';' == current) {
                    match = "\u2213";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1546:
                if ('p' == current) {
                    state = 1547;
                    return true;
                }
                break;
            case 1547:
                if ('f' == current) {
                    state = 1548;
                    return true;
                }
                break;
            case 1548:
                // Mopf;
                if (';' == current) {
                    match = "\uD835\uDD44";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1549:
                if ('c' == current) {
                    state = 1550;
                    return true;
                }
                break;
            case 1550:
                if ('r' == current) {
                    state = 1551;
                    return true;
                }
                break;
            case 1551:
                // Mscr;
                if (';' == current) {
                    match = "\u2133";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1552:
                // Mu;
                if (';' == current) {
                    match = "\u039C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1553:
                switch (current) {
                    case 'J' :
                        state = 1554;
                        return true;
                    case 'a' :
                        state = 1557;
                        return true;
                    case 'c' :
                        state = 1562;
                        return true;
                    case 'e' :
                        state = 1572;
                        return true;
                    case 'f' :
                        state = 1650;
                        return true;
                    case 'o' :
                        state = 1652;
                        return true;
                    case 's' :
                        state = 2055;
                        return true;
                    case 't' :
                        state = 2058;
                        return true;
                    case 'u' :
                        state = 2063;
                        return true;
                }
                break;
            case 1554:
                if ('c' == current) {
                    state = 1555;
                    return true;
                }
                break;
            case 1555:
                if ('y' == current) {
                    state = 1556;
                    return true;
                }
                break;
            case 1556:
                // NJcy;
                if (';' == current) {
                    match = "\u040A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1557:
                if ('c' == current) {
                    state = 1558;
                    return true;
                }
                break;
            case 1558:
                if ('u' == current) {
                    state = 1559;
                    return true;
                }
                break;
            case 1559:
                if ('t' == current) {
                    state = 1560;
                    return true;
                }
                break;
            case 1560:
                if ('e' == current) {
                    state = 1561;
                    return true;
                }
                break;
            case 1561:
                // Nacute;
                if (';' == current) {
                    match = "\u0143";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1562:
                switch (current) {
                    case 'a' :
                        state = 1563;
                        return true;
                    case 'e' :
                        state = 1567;
                        return true;
                    case 'y' :
                        state = 1571;
                        return true;
                }
                break;
            case 1563:
                if ('r' == current) {
                    state = 1564;
                    return true;
                }
                break;
            case 1564:
                if ('o' == current) {
                    state = 1565;
                    return true;
                }
                break;
            case 1565:
                if ('n' == current) {
                    state = 1566;
                    return true;
                }
                break;
            case 1566:
                // Ncaron;
                if (';' == current) {
                    match = "\u0147";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1567:
                if ('d' == current) {
                    state = 1568;
                    return true;
                }
                break;
            case 1568:
                if ('i' == current) {
                    state = 1569;
                    return true;
                }
                break;
            case 1569:
                if ('l' == current) {
                    state = 1570;
                    return true;
                }
                break;
            case 1570:
                // Ncedil;
                if (';' == current) {
                    match = "\u0145";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1571:
                // Ncy;
                if (';' == current) {
                    match = "\u041D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1572:
                switch (current) {
                    case 'g' :
                        state = 1573;
                        return true;
                    case 's' :
                        state = 1619;
                        return true;
                    case 'w' :
                        state = 1645;
                        return true;
                }
                break;
            case 1573:
                if ('a' == current) {
                    state = 1574;
                    return true;
                }
                break;
            case 1574:
                if ('t' == current) {
                    state = 1575;
                    return true;
                }
                break;
            case 1575:
                if ('i' == current) {
                    state = 1576;
                    return true;
                }
                break;
            case 1576:
                if ('v' == current) {
                    state = 1577;
                    return true;
                }
                break;
            case 1577:
                if ('e' == current) {
                    state = 1578;
                    return true;
                }
                break;
            case 1578:
                switch (current) {
                    case 'M' :
                        state = 1579;
                        return true;
                    case 'T' :
                        state = 1590;
                        return true;
                    case 'V' :
                        state = 1606;
                        return true;
                }
                break;
            case 1579:
                if ('e' == current) {
                    state = 1580;
                    return true;
                }
                break;
            case 1580:
                if ('d' == current) {
                    state = 1581;
                    return true;
                }
                break;
            case 1581:
                if ('i' == current) {
                    state = 1582;
                    return true;
                }
                break;
            case 1582:
                if ('u' == current) {
                    state = 1583;
                    return true;
                }
                break;
            case 1583:
                if ('m' == current) {
                    state = 1584;
                    return true;
                }
                break;
            case 1584:
                if ('S' == current) {
                    state = 1585;
                    return true;
                }
                break;
            case 1585:
                if ('p' == current) {
                    state = 1586;
                    return true;
                }
                break;
            case 1586:
                if ('a' == current) {
                    state = 1587;
                    return true;
                }
                break;
            case 1587:
                if ('c' == current) {
                    state = 1588;
                    return true;
                }
                break;
            case 1588:
                if ('e' == current) {
                    state = 1589;
                    return true;
                }
                break;
            case 1589:
                // NegativeMediumSpace;
                if (';' == current) {
                    match = "\u200B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1590:
                if ('h' == current) {
                    state = 1591;
                    return true;
                }
                break;
            case 1591:
                if ('i' == current) {
                    state = 1592;
                    return true;
                }
                break;
            case 1592:
                if ('c' == current) {
                    state = 1593;
                    return true;
                }
                else if ('n' == current) {
                    state = 1600;
                    return true;
                }
                break;
            case 1593:
                if ('k' == current) {
                    state = 1594;
                    return true;
                }
                break;
            case 1594:
                if ('S' == current) {
                    state = 1595;
                    return true;
                }
                break;
            case 1595:
                if ('p' == current) {
                    state = 1596;
                    return true;
                }
                break;
            case 1596:
                if ('a' == current) {
                    state = 1597;
                    return true;
                }
                break;
            case 1597:
                if ('c' == current) {
                    state = 1598;
                    return true;
                }
                break;
            case 1598:
                if ('e' == current) {
                    state = 1599;
                    return true;
                }
                break;
            case 1599:
                // NegativeThickSpace;
                if (';' == current) {
                    match = "\u200B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1600:
                if ('S' == current) {
                    state = 1601;
                    return true;
                }
                break;
            case 1601:
                if ('p' == current) {
                    state = 1602;
                    return true;
                }
                break;
            case 1602:
                if ('a' == current) {
                    state = 1603;
                    return true;
                }
                break;
            case 1603:
                if ('c' == current) {
                    state = 1604;
                    return true;
                }
                break;
            case 1604:
                if ('e' == current) {
                    state = 1605;
                    return true;
                }
                break;
            case 1605:
                // NegativeThinSpace;
                if (';' == current) {
                    match = "\u200B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1606:
                if ('e' == current) {
                    state = 1607;
                    return true;
                }
                break;
            case 1607:
                if ('r' == current) {
                    state = 1608;
                    return true;
                }
                break;
            case 1608:
                if ('y' == current) {
                    state = 1609;
                    return true;
                }
                break;
            case 1609:
                if ('T' == current) {
                    state = 1610;
                    return true;
                }
                break;
            case 1610:
                if ('h' == current) {
                    state = 1611;
                    return true;
                }
                break;
            case 1611:
                if ('i' == current) {
                    state = 1612;
                    return true;
                }
                break;
            case 1612:
                if ('n' == current) {
                    state = 1613;
                    return true;
                }
                break;
            case 1613:
                if ('S' == current) {
                    state = 1614;
                    return true;
                }
                break;
            case 1614:
                if ('p' == current) {
                    state = 1615;
                    return true;
                }
                break;
            case 1615:
                if ('a' == current) {
                    state = 1616;
                    return true;
                }
                break;
            case 1616:
                if ('c' == current) {
                    state = 1617;
                    return true;
                }
                break;
            case 1617:
                if ('e' == current) {
                    state = 1618;
                    return true;
                }
                break;
            case 1618:
                // NegativeVeryThinSpace;
                if (';' == current) {
                    match = "\u200B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1619:
                if ('t' == current) {
                    state = 1620;
                    return true;
                }
                break;
            case 1620:
                if ('e' == current) {
                    state = 1621;
                    return true;
                }
                break;
            case 1621:
                if ('d' == current) {
                    state = 1622;
                    return true;
                }
                break;
            case 1622:
                if ('G' == current) {
                    state = 1623;
                    return true;
                }
                else if ('L' == current) {
                    state = 1637;
                    return true;
                }
                break;
            case 1623:
                if ('r' == current) {
                    state = 1624;
                    return true;
                }
                break;
            case 1624:
                if ('e' == current) {
                    state = 1625;
                    return true;
                }
                break;
            case 1625:
                if ('a' == current) {
                    state = 1626;
                    return true;
                }
                break;
            case 1626:
                if ('t' == current) {
                    state = 1627;
                    return true;
                }
                break;
            case 1627:
                if ('e' == current) {
                    state = 1628;
                    return true;
                }
                break;
            case 1628:
                if ('r' == current) {
                    state = 1629;
                    return true;
                }
                break;
            case 1629:
                if ('G' == current) {
                    state = 1630;
                    return true;
                }
                break;
            case 1630:
                if ('r' == current) {
                    state = 1631;
                    return true;
                }
                break;
            case 1631:
                if ('e' == current) {
                    state = 1632;
                    return true;
                }
                break;
            case 1632:
                if ('a' == current) {
                    state = 1633;
                    return true;
                }
                break;
            case 1633:
                if ('t' == current) {
                    state = 1634;
                    return true;
                }
                break;
            case 1634:
                if ('e' == current) {
                    state = 1635;
                    return true;
                }
                break;
            case 1635:
                if ('r' == current) {
                    state = 1636;
                    return true;
                }
                break;
            case 1636:
                // NestedGreaterGreater;
                if (';' == current) {
                    match = "\u226B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1637:
                if ('e' == current) {
                    state = 1638;
                    return true;
                }
                break;
            case 1638:
                if ('s' == current) {
                    state = 1639;
                    return true;
                }
                break;
            case 1639:
                if ('s' == current) {
                    state = 1640;
                    return true;
                }
                break;
            case 1640:
                if ('L' == current) {
                    state = 1641;
                    return true;
                }
                break;
            case 1641:
                if ('e' == current) {
                    state = 1642;
                    return true;
                }
                break;
            case 1642:
                if ('s' == current) {
                    state = 1643;
                    return true;
                }
                break;
            case 1643:
                if ('s' == current) {
                    state = 1644;
                    return true;
                }
                break;
            case 1644:
                // NestedLessLess;
                if (';' == current) {
                    match = "\u226A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1645:
                if ('L' == current) {
                    state = 1646;
                    return true;
                }
                break;
            case 1646:
                if ('i' == current) {
                    state = 1647;
                    return true;
                }
                break;
            case 1647:
                if ('n' == current) {
                    state = 1648;
                    return true;
                }
                break;
            case 1648:
                if ('e' == current) {
                    state = 1649;
                    return true;
                }
                break;
            case 1649:
                // NewLine;
                if (';' == current) {
                    match = "\n";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1650:
                if ('r' == current) {
                    state = 1651;
                    return true;
                }
                break;
            case 1651:
                // Nfr;
                if (';' == current) {
                    match = "\uD835\uDD11";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1652:
                switch (current) {
                    case 'B' :
                        state = 1653;
                        return true;
                    case 'n' :
                        state = 1658;
                        return true;
                    case 'p' :
                        state = 1672;
                        return true;
                    case 't' :
                        state = 1674;
                        return true;
                }
                break;
            case 1653:
                if ('r' == current) {
                    state = 1654;
                    return true;
                }
                break;
            case 1654:
                if ('e' == current) {
                    state = 1655;
                    return true;
                }
                break;
            case 1655:
                if ('a' == current) {
                    state = 1656;
                    return true;
                }
                break;
            case 1656:
                if ('k' == current) {
                    state = 1657;
                    return true;
                }
                break;
            case 1657:
                // NoBreak;
                if (';' == current) {
                    match = "\u2060";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1658:
                if ('B' == current) {
                    state = 1659;
                    return true;
                }
                break;
            case 1659:
                if ('r' == current) {
                    state = 1660;
                    return true;
                }
                break;
            case 1660:
                if ('e' == current) {
                    state = 1661;
                    return true;
                }
                break;
            case 1661:
                if ('a' == current) {
                    state = 1662;
                    return true;
                }
                break;
            case 1662:
                if ('k' == current) {
                    state = 1663;
                    return true;
                }
                break;
            case 1663:
                if ('i' == current) {
                    state = 1664;
                    return true;
                }
                break;
            case 1664:
                if ('n' == current) {
                    state = 1665;
                    return true;
                }
                break;
            case 1665:
                if ('g' == current) {
                    state = 1666;
                    return true;
                }
                break;
            case 1666:
                if ('S' == current) {
                    state = 1667;
                    return true;
                }
                break;
            case 1667:
                if ('p' == current) {
                    state = 1668;
                    return true;
                }
                break;
            case 1668:
                if ('a' == current) {
                    state = 1669;
                    return true;
                }
                break;
            case 1669:
                if ('c' == current) {
                    state = 1670;
                    return true;
                }
                break;
            case 1670:
                if ('e' == current) {
                    state = 1671;
                    return true;
                }
                break;
            case 1671:
                // NonBreakingSpace;
                if (';' == current) {
                    match = "\u00A0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1672:
                if ('f' == current) {
                    state = 1673;
                    return true;
                }
                break;
            case 1673:
                // Nopf;
                if (';' == current) {
                    match = "\u2115";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1674:
                switch (current) {
                    case ';' : // Not;
                        match = "\u2AEC";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'C' :
                        state = 1675;
                        return true;
                    case 'D' :
                        state = 1689;
                        return true;
                    case 'E' :
                        state = 1706;
                        return true;
                    case 'G' :
                        state = 1727;
                        return true;
                    case 'H' :
                        state = 1780;
                        return true;
                    case 'L' :
                        state = 1801;
                        return true;
                    case 'N' :
                        state = 1860;
                        return true;
                    case 'P' :
                        state = 1888;
                        return true;
                    case 'R' :
                        state = 1911;
                        return true;
                    case 'S' :
                        state = 1945;
                        return true;
                    case 'T' :
                        state = 2020;
                        return true;
                    case 'V' :
                        state = 2044;
                        return true;
                }
                break;
            case 1675:
                if ('o' == current) {
                    state = 1676;
                    return true;
                }
                else if ('u' == current) {
                    state = 1684;
                    return true;
                }
                break;
            case 1676:
                if ('n' == current) {
                    state = 1677;
                    return true;
                }
                break;
            case 1677:
                if ('g' == current) {
                    state = 1678;
                    return true;
                }
                break;
            case 1678:
                if ('r' == current) {
                    state = 1679;
                    return true;
                }
                break;
            case 1679:
                if ('u' == current) {
                    state = 1680;
                    return true;
                }
                break;
            case 1680:
                if ('e' == current) {
                    state = 1681;
                    return true;
                }
                break;
            case 1681:
                if ('n' == current) {
                    state = 1682;
                    return true;
                }
                break;
            case 1682:
                if ('t' == current) {
                    state = 1683;
                    return true;
                }
                break;
            case 1683:
                // NotCongruent;
                if (';' == current) {
                    match = "\u2262";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1684:
                if ('p' == current) {
                    state = 1685;
                    return true;
                }
                break;
            case 1685:
                if ('C' == current) {
                    state = 1686;
                    return true;
                }
                break;
            case 1686:
                if ('a' == current) {
                    state = 1687;
                    return true;
                }
                break;
            case 1687:
                if ('p' == current) {
                    state = 1688;
                    return true;
                }
                break;
            case 1688:
                // NotCupCap;
                if (';' == current) {
                    match = "\u226D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1689:
                if ('o' == current) {
                    state = 1690;
                    return true;
                }
                break;
            case 1690:
                if ('u' == current) {
                    state = 1691;
                    return true;
                }
                break;
            case 1691:
                if ('b' == current) {
                    state = 1692;
                    return true;
                }
                break;
            case 1692:
                if ('l' == current) {
                    state = 1693;
                    return true;
                }
                break;
            case 1693:
                if ('e' == current) {
                    state = 1694;
                    return true;
                }
                break;
            case 1694:
                if ('V' == current) {
                    state = 1695;
                    return true;
                }
                break;
            case 1695:
                if ('e' == current) {
                    state = 1696;
                    return true;
                }
                break;
            case 1696:
                if ('r' == current) {
                    state = 1697;
                    return true;
                }
                break;
            case 1697:
                if ('t' == current) {
                    state = 1698;
                    return true;
                }
                break;
            case 1698:
                if ('i' == current) {
                    state = 1699;
                    return true;
                }
                break;
            case 1699:
                if ('c' == current) {
                    state = 1700;
                    return true;
                }
                break;
            case 1700:
                if ('a' == current) {
                    state = 1701;
                    return true;
                }
                break;
            case 1701:
                if ('l' == current) {
                    state = 1702;
                    return true;
                }
                break;
            case 1702:
                if ('B' == current) {
                    state = 1703;
                    return true;
                }
                break;
            case 1703:
                if ('a' == current) {
                    state = 1704;
                    return true;
                }
                break;
            case 1704:
                if ('r' == current) {
                    state = 1705;
                    return true;
                }
                break;
            case 1705:
                // NotDoubleVerticalBar;
                if (';' == current) {
                    match = "\u2226";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1706:
                switch (current) {
                    case 'l' :
                        state = 1707;
                        return true;
                    case 'q' :
                        state = 1713;
                        return true;
                    case 'x' :
                        state = 1722;
                        return true;
                }
                break;
            case 1707:
                if ('e' == current) {
                    state = 1708;
                    return true;
                }
                break;
            case 1708:
                if ('m' == current) {
                    state = 1709;
                    return true;
                }
                break;
            case 1709:
                if ('e' == current) {
                    state = 1710;
                    return true;
                }
                break;
            case 1710:
                if ('n' == current) {
                    state = 1711;
                    return true;
                }
                break;
            case 1711:
                if ('t' == current) {
                    state = 1712;
                    return true;
                }
                break;
            case 1712:
                // NotElement;
                if (';' == current) {
                    match = "\u2209";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1713:
                if ('u' == current) {
                    state = 1714;
                    return true;
                }
                break;
            case 1714:
                if ('a' == current) {
                    state = 1715;
                    return true;
                }
                break;
            case 1715:
                if ('l' == current) {
                    state = 1716;
                    return true;
                }
                break;
            case 1716:
                // NotEqual;
                if (';' == current) {
                    match = "\u2260";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('T' == current) {
                    state = 1717;
                    return true;
                }
                break;
            case 1717:
                if ('i' == current) {
                    state = 1718;
                    return true;
                }
                break;
            case 1718:
                if ('l' == current) {
                    state = 1719;
                    return true;
                }
                break;
            case 1719:
                if ('d' == current) {
                    state = 1720;
                    return true;
                }
                break;
            case 1720:
                if ('e' == current) {
                    state = 1721;
                    return true;
                }
                break;
            case 1721:
                // NotEqualTilde;
                if (';' == current) {
                    match = "\u2242\u0338";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1722:
                if ('i' == current) {
                    state = 1723;
                    return true;
                }
                break;
            case 1723:
                if ('s' == current) {
                    state = 1724;
                    return true;
                }
                break;
            case 1724:
                if ('t' == current) {
                    state = 1725;
                    return true;
                }
                break;
            case 1725:
                if ('s' == current) {
                    state = 1726;
                    return true;
                }
                break;
            case 1726:
                // NotExists;
                if (';' == current) {
                    match = "\u2204";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1727:
                if ('r' == current) {
                    state = 1728;
                    return true;
                }
                break;
            case 1728:
                if ('e' == current) {
                    state = 1729;
                    return true;
                }
                break;
            case 1729:
                if ('a' == current) {
                    state = 1730;
                    return true;
                }
                break;
            case 1730:
                if ('t' == current) {
                    state = 1731;
                    return true;
                }
                break;
            case 1731:
                if ('e' == current) {
                    state = 1732;
                    return true;
                }
                break;
            case 1732:
                if ('r' == current) {
                    state = 1733;
                    return true;
                }
                break;
            case 1733:
                switch (current) {
                    case ';' : // NotGreater;
                        match = "\u226F";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 1734;
                        return true;
                    case 'F' :
                        state = 1739;
                        return true;
                    case 'G' :
                        state = 1750;
                        return true;
                    case 'L' :
                        state = 1759;
                        return true;
                    case 'S' :
                        state = 1763;
                        return true;
                    case 'T' :
                        state = 1775;
                        return true;
                }
                break;
            case 1734:
                if ('q' == current) {
                    state = 1735;
                    return true;
                }
                break;
            case 1735:
                if ('u' == current) {
                    state = 1736;
                    return true;
                }
                break;
            case 1736:
                if ('a' == current) {
                    state = 1737;
                    return true;
                }
                break;
            case 1737:
                if ('l' == current) {
                    state = 1738;
                    return true;
                }
                break;
            case 1738:
                // NotGreaterEqual;
                if (';' == current) {
                    match = "\u2271";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1739:
                if ('u' == current) {
                    state = 1740;
                    return true;
                }
                break;
            case 1740:
                if ('l' == current) {
                    state = 1741;
                    return true;
                }
                break;
            case 1741:
                if ('l' == current) {
                    state = 1742;
                    return true;
                }
                break;
            case 1742:
                if ('E' == current) {
                    state = 1743;
                    return true;
                }
                break;
            case 1743:
                if ('q' == current) {
                    state = 1744;
                    return true;
                }
                break;
            case 1744:
                if ('u' == current) {
                    state = 1745;
                    return true;
                }
                break;
            case 1745:
                if ('a' == current) {
                    state = 1746;
                    return true;
                }
                break;
            case 1746:
                if ('l' == current) {
                    state = 1747;
                    return true;
                }
                break;
            case 1747:
                if (';' == current) {
                    state = 1748;
                    return true;
                }
                break;
            case 1748:
                if ('≧' == current) {
                    state = 1749;
                    return true;
                }
                break;
            case 1749:
                // NotGreaterFullEqual;≧̸
                if ('̸' == current) {
                    match = "";
                    matchLength = consumedCount;
                    return false;
                }
                break;
            case 1750:
                if ('r' == current) {
                    state = 1751;
                    return true;
                }
                break;
            case 1751:
                if ('e' == current) {
                    state = 1752;
                    return true;
                }
                break;
            case 1752:
                if ('a' == current) {
                    state = 1753;
                    return true;
                }
                break;
            case 1753:
                if ('t' == current) {
                    state = 1754;
                    return true;
                }
                break;
            case 1754:
                if ('e' == current) {
                    state = 1755;
                    return true;
                }
                break;
            case 1755:
                if ('r' == current) {
                    state = 1756;
                    return true;
                }
                break;
            case 1756:
                if (';' == current) {
                    state = 1757;
                    return true;
                }
                break;
            case 1757:
                if ('≫' == current) {
                    state = 1758;
                    return true;
                }
                break;
            case 1758:
                // NotGreaterGreater;≫̸
                if ('̸' == current) {
                    match = "";
                    matchLength = consumedCount;
                    return false;
                }
                break;
            case 1759:
                if ('e' == current) {
                    state = 1760;
                    return true;
                }
                break;
            case 1760:
                if ('s' == current) {
                    state = 1761;
                    return true;
                }
                break;
            case 1761:
                if ('s' == current) {
                    state = 1762;
                    return true;
                }
                break;
            case 1762:
                // NotGreaterLess;
                if (';' == current) {
                    match = "\u2279";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1763:
                if ('l' == current) {
                    state = 1764;
                    return true;
                }
                break;
            case 1764:
                if ('a' == current) {
                    state = 1765;
                    return true;
                }
                break;
            case 1765:
                if ('n' == current) {
                    state = 1766;
                    return true;
                }
                break;
            case 1766:
                if ('t' == current) {
                    state = 1767;
                    return true;
                }
                break;
            case 1767:
                if ('E' == current) {
                    state = 1768;
                    return true;
                }
                break;
            case 1768:
                if ('q' == current) {
                    state = 1769;
                    return true;
                }
                break;
            case 1769:
                if ('u' == current) {
                    state = 1770;
                    return true;
                }
                break;
            case 1770:
                if ('a' == current) {
                    state = 1771;
                    return true;
                }
                break;
            case 1771:
                if ('l' == current) {
                    state = 1772;
                    return true;
                }
                break;
            case 1772:
                if (';' == current) {
                    state = 1773;
                    return true;
                }
                break;
            case 1773:
                if ('⩾' == current) {
                    state = 1774;
                    return true;
                }
                break;
            case 1774:
                // NotGreaterSlantEqual;⩾̸
                if ('̸' == current) {
                    match = "";
                    matchLength = consumedCount;
                    return false;
                }
                break;
            case 1775:
                if ('i' == current) {
                    state = 1776;
                    return true;
                }
                break;
            case 1776:
                if ('l' == current) {
                    state = 1777;
                    return true;
                }
                break;
            case 1777:
                if ('d' == current) {
                    state = 1778;
                    return true;
                }
                break;
            case 1778:
                if ('e' == current) {
                    state = 1779;
                    return true;
                }
                break;
            case 1779:
                // NotGreaterTilde;
                if (';' == current) {
                    match = "\u2275";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1780:
                if ('u' == current) {
                    state = 1781;
                    return true;
                }
                break;
            case 1781:
                if ('m' == current) {
                    state = 1782;
                    return true;
                }
                break;
            case 1782:
                if ('p' == current) {
                    state = 1783;
                    return true;
                }
                break;
            case 1783:
                if ('D' == current) {
                    state = 1784;
                    return true;
                }
                else if ('E' == current) {
                    state = 1794;
                    return true;
                }
                break;
            case 1784:
                if ('o' == current) {
                    state = 1785;
                    return true;
                }
                break;
            case 1785:
                if ('w' == current) {
                    state = 1786;
                    return true;
                }
                break;
            case 1786:
                if ('n' == current) {
                    state = 1787;
                    return true;
                }
                break;
            case 1787:
                if ('H' == current) {
                    state = 1788;
                    return true;
                }
                break;
            case 1788:
                if ('u' == current) {
                    state = 1789;
                    return true;
                }
                break;
            case 1789:
                if ('m' == current) {
                    state = 1790;
                    return true;
                }
                break;
            case 1790:
                if ('p' == current) {
                    state = 1791;
                    return true;
                }
                break;
            case 1791:
                if (';' == current) {
                    state = 1792;
                    return true;
                }
                break;
            case 1792:
                if ('≎' == current) {
                    state = 1793;
                    return true;
                }
                break;
            case 1793:
                // NotHumpDownHump;≎̸
                if ('̸' == current) {
                    match = "";
                    matchLength = consumedCount;
                    return false;
                }
                break;
            case 1794:
                if ('q' == current) {
                    state = 1795;
                    return true;
                }
                break;
            case 1795:
                if ('u' == current) {
                    state = 1796;
                    return true;
                }
                break;
            case 1796:
                if ('a' == current) {
                    state = 1797;
                    return true;
                }
                break;
            case 1797:
                if ('l' == current) {
                    state = 1798;
                    return true;
                }
                break;
            case 1798:
                if (';' == current) {
                    state = 1799;
                    return true;
                }
                break;
            case 1799:
                if ('≏' == current) {
                    state = 1800;
                    return true;
                }
                break;
            case 1800:
                // NotHumpEqual;≏̸
                if ('̸' == current) {
                    match = "";
                    matchLength = consumedCount;
                    return false;
                }
                break;
            case 1801:
                if ('e' == current) {
                    state = 1802;
                    return true;
                }
                break;
            case 1802:
                if ('f' == current) {
                    state = 1803;
                    return true;
                }
                else if ('s' == current) {
                    state = 1823;
                    return true;
                }
                break;
            case 1803:
                if ('t' == current) {
                    state = 1804;
                    return true;
                }
                break;
            case 1804:
                if ('T' == current) {
                    state = 1805;
                    return true;
                }
                break;
            case 1805:
                if ('r' == current) {
                    state = 1806;
                    return true;
                }
                break;
            case 1806:
                if ('i' == current) {
                    state = 1807;
                    return true;
                }
                break;
            case 1807:
                if ('a' == current) {
                    state = 1808;
                    return true;
                }
                break;
            case 1808:
                if ('n' == current) {
                    state = 1809;
                    return true;
                }
                break;
            case 1809:
                if ('g' == current) {
                    state = 1810;
                    return true;
                }
                break;
            case 1810:
                if ('l' == current) {
                    state = 1811;
                    return true;
                }
                break;
            case 1811:
                if ('e' == current) {
                    state = 1812;
                    return true;
                }
                break;
            case 1812:
                switch (current) {
                    case ';' : // NotLeftTriangle;
                        match = "\u22EA";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 1813;
                        return true;
                    case 'E' :
                        state = 1818;
                        return true;
                }
                break;
            case 1813:
                if ('a' == current) {
                    state = 1814;
                    return true;
                }
                break;
            case 1814:
                if ('r' == current) {
                    state = 1815;
                    return true;
                }
                break;
            case 1815:
                if (';' == current) {
                    state = 1816;
                    return true;
                }
                break;
            case 1816:
                if ('⧏' == current) {
                    state = 1817;
                    return true;
                }
                break;
            case 1817:
                // NotLeftTriangleBar;⧏̸
                if ('̸' == current) {
                    match = "";
                    matchLength = consumedCount;
                    return false;
                }
                break;
            case 1818:
                if ('q' == current) {
                    state = 1819;
                    return true;
                }
                break;
            case 1819:
                if ('u' == current) {
                    state = 1820;
                    return true;
                }
                break;
            case 1820:
                if ('a' == current) {
                    state = 1821;
                    return true;
                }
                break;
            case 1821:
                if ('l' == current) {
                    state = 1822;
                    return true;
                }
                break;
            case 1822:
                // NotLeftTriangleEqual;
                if (';' == current) {
                    match = "\u22EC";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1823:
                if ('s' == current) {
                    state = 1824;
                    return true;
                }
                break;
            case 1824:
                switch (current) {
                    case ';' : // NotLess;
                        match = "\u226E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 1825;
                        return true;
                    case 'G' :
                        state = 1830;
                        return true;
                    case 'L' :
                        state = 1837;
                        return true;
                    case 'S' :
                        state = 1843;
                        return true;
                    case 'T' :
                        state = 1855;
                        return true;
                }
                break;
            case 1825:
                if ('q' == current) {
                    state = 1826;
                    return true;
                }
                break;
            case 1826:
                if ('u' == current) {
                    state = 1827;
                    return true;
                }
                break;
            case 1827:
                if ('a' == current) {
                    state = 1828;
                    return true;
                }
                break;
            case 1828:
                if ('l' == current) {
                    state = 1829;
                    return true;
                }
                break;
            case 1829:
                // NotLessEqual;
                if (';' == current) {
                    match = "\u2270";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1830:
                if ('r' == current) {
                    state = 1831;
                    return true;
                }
                break;
            case 1831:
                if ('e' == current) {
                    state = 1832;
                    return true;
                }
                break;
            case 1832:
                if ('a' == current) {
                    state = 1833;
                    return true;
                }
                break;
            case 1833:
                if ('t' == current) {
                    state = 1834;
                    return true;
                }
                break;
            case 1834:
                if ('e' == current) {
                    state = 1835;
                    return true;
                }
                break;
            case 1835:
                if ('r' == current) {
                    state = 1836;
                    return true;
                }
                break;
            case 1836:
                // NotLessGreater;
                if (';' == current) {
                    match = "\u2278";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1837:
                if ('e' == current) {
                    state = 1838;
                    return true;
                }
                break;
            case 1838:
                if ('s' == current) {
                    state = 1839;
                    return true;
                }
                break;
            case 1839:
                if ('s' == current) {
                    state = 1840;
                    return true;
                }
                break;
            case 1840:
                if (';' == current) {
                    state = 1841;
                    return true;
                }
                break;
            case 1841:
                if ('≪' == current) {
                    state = 1842;
                    return true;
                }
                break;
            case 1842:
                // NotLessLess;≪̸
                if ('̸' == current) {
                    match = "";
                    matchLength = consumedCount;
                    return false;
                }
                break;
            case 1843:
                if ('l' == current) {
                    state = 1844;
                    return true;
                }
                break;
            case 1844:
                if ('a' == current) {
                    state = 1845;
                    return true;
                }
                break;
            case 1845:
                if ('n' == current) {
                    state = 1846;
                    return true;
                }
                break;
            case 1846:
                if ('t' == current) {
                    state = 1847;
                    return true;
                }
                break;
            case 1847:
                if ('E' == current) {
                    state = 1848;
                    return true;
                }
                break;
            case 1848:
                if ('q' == current) {
                    state = 1849;
                    return true;
                }
                break;
            case 1849:
                if ('u' == current) {
                    state = 1850;
                    return true;
                }
                break;
            case 1850:
                if ('a' == current) {
                    state = 1851;
                    return true;
                }
                break;
            case 1851:
                if ('l' == current) {
                    state = 1852;
                    return true;
                }
                break;
            case 1852:
                if (';' == current) {
                    state = 1853;
                    return true;
                }
                break;
            case 1853:
                if ('⩽' == current) {
                    state = 1854;
                    return true;
                }
                break;
            case 1854:
                // NotLessSlantEqual;⩽̸
                if ('̸' == current) {
                    match = "";
                    matchLength = consumedCount;
                    return false;
                }
                break;
            case 1855:
                if ('i' == current) {
                    state = 1856;
                    return true;
                }
                break;
            case 1856:
                if ('l' == current) {
                    state = 1857;
                    return true;
                }
                break;
            case 1857:
                if ('d' == current) {
                    state = 1858;
                    return true;
                }
                break;
            case 1858:
                if ('e' == current) {
                    state = 1859;
                    return true;
                }
                break;
            case 1859:
                // NotLessTilde;
                if (';' == current) {
                    match = "\u2274";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1860:
                if ('e' == current) {
                    state = 1861;
                    return true;
                }
                break;
            case 1861:
                if ('s' == current) {
                    state = 1862;
                    return true;
                }
                break;
            case 1862:
                if ('t' == current) {
                    state = 1863;
                    return true;
                }
                break;
            case 1863:
                if ('e' == current) {
                    state = 1864;
                    return true;
                }
                break;
            case 1864:
                if ('d' == current) {
                    state = 1865;
                    return true;
                }
                break;
            case 1865:
                if ('G' == current) {
                    state = 1866;
                    return true;
                }
                else if ('L' == current) {
                    state = 1880;
                    return true;
                }
                break;
            case 1866:
                if ('r' == current) {
                    state = 1867;
                    return true;
                }
                break;
            case 1867:
                if ('e' == current) {
                    state = 1868;
                    return true;
                }
                break;
            case 1868:
                if ('a' == current) {
                    state = 1869;
                    return true;
                }
                break;
            case 1869:
                if ('t' == current) {
                    state = 1870;
                    return true;
                }
                break;
            case 1870:
                if ('e' == current) {
                    state = 1871;
                    return true;
                }
                break;
            case 1871:
                if ('r' == current) {
                    state = 1872;
                    return true;
                }
                break;
            case 1872:
                if ('G' == current) {
                    state = 1873;
                    return true;
                }
                break;
            case 1873:
                if ('r' == current) {
                    state = 1874;
                    return true;
                }
                break;
            case 1874:
                if ('e' == current) {
                    state = 1875;
                    return true;
                }
                break;
            case 1875:
                if ('a' == current) {
                    state = 1876;
                    return true;
                }
                break;
            case 1876:
                if ('t' == current) {
                    state = 1877;
                    return true;
                }
                break;
            case 1877:
                if ('e' == current) {
                    state = 1878;
                    return true;
                }
                break;
            case 1878:
                if ('r' == current) {
                    state = 1879;
                    return true;
                }
                break;
            case 1879:
                // NotNestedGreaterGreater;
                if (';' == current) {
                    match = "\u2AA2\u0338";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1880:
                if ('e' == current) {
                    state = 1881;
                    return true;
                }
                break;
            case 1881:
                if ('s' == current) {
                    state = 1882;
                    return true;
                }
                break;
            case 1882:
                if ('s' == current) {
                    state = 1883;
                    return true;
                }
                break;
            case 1883:
                if ('L' == current) {
                    state = 1884;
                    return true;
                }
                break;
            case 1884:
                if ('e' == current) {
                    state = 1885;
                    return true;
                }
                break;
            case 1885:
                if ('s' == current) {
                    state = 1886;
                    return true;
                }
                break;
            case 1886:
                if ('s' == current) {
                    state = 1887;
                    return true;
                }
                break;
            case 1887:
                // NotNestedLessLess;
                if (';' == current) {
                    match = "\u2AA1\u0338";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1888:
                if ('r' == current) {
                    state = 1889;
                    return true;
                }
                break;
            case 1889:
                if ('e' == current) {
                    state = 1890;
                    return true;
                }
                break;
            case 1890:
                if ('c' == current) {
                    state = 1891;
                    return true;
                }
                break;
            case 1891:
                if ('e' == current) {
                    state = 1892;
                    return true;
                }
                break;
            case 1892:
                if ('d' == current) {
                    state = 1893;
                    return true;
                }
                break;
            case 1893:
                if ('e' == current) {
                    state = 1894;
                    return true;
                }
                break;
            case 1894:
                if ('s' == current) {
                    state = 1895;
                    return true;
                }
                break;
            case 1895:
                switch (current) {
                    case ';' : // NotPrecedes;
                        match = "\u2280";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 1896;
                        return true;
                    case 'S' :
                        state = 1901;
                        return true;
                }
                break;
            case 1896:
                if ('q' == current) {
                    state = 1897;
                    return true;
                }
                break;
            case 1897:
                if ('u' == current) {
                    state = 1898;
                    return true;
                }
                break;
            case 1898:
                if ('a' == current) {
                    state = 1899;
                    return true;
                }
                break;
            case 1899:
                if ('l' == current) {
                    state = 1900;
                    return true;
                }
                break;
            case 1900:
                // NotPrecedesEqual;
                if (';' == current) {
                    match = "\u2AAF\u0338";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1901:
                if ('l' == current) {
                    state = 1902;
                    return true;
                }
                break;
            case 1902:
                if ('a' == current) {
                    state = 1903;
                    return true;
                }
                break;
            case 1903:
                if ('n' == current) {
                    state = 1904;
                    return true;
                }
                break;
            case 1904:
                if ('t' == current) {
                    state = 1905;
                    return true;
                }
                break;
            case 1905:
                if ('E' == current) {
                    state = 1906;
                    return true;
                }
                break;
            case 1906:
                if ('q' == current) {
                    state = 1907;
                    return true;
                }
                break;
            case 1907:
                if ('u' == current) {
                    state = 1908;
                    return true;
                }
                break;
            case 1908:
                if ('a' == current) {
                    state = 1909;
                    return true;
                }
                break;
            case 1909:
                if ('l' == current) {
                    state = 1910;
                    return true;
                }
                break;
            case 1910:
                // NotPrecedesSlantEqual;
                if (';' == current) {
                    match = "\u22E0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1911:
                if ('e' == current) {
                    state = 1912;
                    return true;
                }
                else if ('i' == current) {
                    state = 1925;
                    return true;
                }
                break;
            case 1912:
                if ('v' == current) {
                    state = 1913;
                    return true;
                }
                break;
            case 1913:
                if ('e' == current) {
                    state = 1914;
                    return true;
                }
                break;
            case 1914:
                if ('r' == current) {
                    state = 1915;
                    return true;
                }
                break;
            case 1915:
                if ('s' == current) {
                    state = 1916;
                    return true;
                }
                break;
            case 1916:
                if ('e' == current) {
                    state = 1917;
                    return true;
                }
                break;
            case 1917:
                if ('E' == current) {
                    state = 1918;
                    return true;
                }
                break;
            case 1918:
                if ('l' == current) {
                    state = 1919;
                    return true;
                }
                break;
            case 1919:
                if ('e' == current) {
                    state = 1920;
                    return true;
                }
                break;
            case 1920:
                if ('m' == current) {
                    state = 1921;
                    return true;
                }
                break;
            case 1921:
                if ('e' == current) {
                    state = 1922;
                    return true;
                }
                break;
            case 1922:
                if ('n' == current) {
                    state = 1923;
                    return true;
                }
                break;
            case 1923:
                if ('t' == current) {
                    state = 1924;
                    return true;
                }
                break;
            case 1924:
                // NotReverseElement;
                if (';' == current) {
                    match = "\u220C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1925:
                if ('g' == current) {
                    state = 1926;
                    return true;
                }
                break;
            case 1926:
                if ('h' == current) {
                    state = 1927;
                    return true;
                }
                break;
            case 1927:
                if ('t' == current) {
                    state = 1928;
                    return true;
                }
                break;
            case 1928:
                if ('T' == current) {
                    state = 1929;
                    return true;
                }
                break;
            case 1929:
                if ('r' == current) {
                    state = 1930;
                    return true;
                }
                break;
            case 1930:
                if ('i' == current) {
                    state = 1931;
                    return true;
                }
                break;
            case 1931:
                if ('a' == current) {
                    state = 1932;
                    return true;
                }
                break;
            case 1932:
                if ('n' == current) {
                    state = 1933;
                    return true;
                }
                break;
            case 1933:
                if ('g' == current) {
                    state = 1934;
                    return true;
                }
                break;
            case 1934:
                if ('l' == current) {
                    state = 1935;
                    return true;
                }
                break;
            case 1935:
                if ('e' == current) {
                    state = 1936;
                    return true;
                }
                break;
            case 1936:
                switch (current) {
                    case ';' : // NotRightTriangle;
                        match = "\u22EB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 1937;
                        return true;
                    case 'E' :
                        state = 1940;
                        return true;
                }
                break;
            case 1937:
                if ('a' == current) {
                    state = 1938;
                    return true;
                }
                break;
            case 1938:
                if ('r' == current) {
                    state = 1939;
                    return true;
                }
                break;
            case 1939:
                // NotRightTriangleBar;
                if (';' == current) {
                    match = "\u29D0\u0338";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1940:
                if ('q' == current) {
                    state = 1941;
                    return true;
                }
                break;
            case 1941:
                if ('u' == current) {
                    state = 1942;
                    return true;
                }
                break;
            case 1942:
                if ('a' == current) {
                    state = 1943;
                    return true;
                }
                break;
            case 1943:
                if ('l' == current) {
                    state = 1944;
                    return true;
                }
                break;
            case 1944:
                // NotRightTriangleEqual;
                if (';' == current) {
                    match = "\u22ED";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1945:
                if ('q' == current) {
                    state = 1946;
                    return true;
                }
                else if ('u' == current) {
                    state = 1973;
                    return true;
                }
                break;
            case 1946:
                if ('u' == current) {
                    state = 1947;
                    return true;
                }
                break;
            case 1947:
                if ('a' == current) {
                    state = 1948;
                    return true;
                }
                break;
            case 1948:
                if ('r' == current) {
                    state = 1949;
                    return true;
                }
                break;
            case 1949:
                if ('e' == current) {
                    state = 1950;
                    return true;
                }
                break;
            case 1950:
                if ('S' == current) {
                    state = 1951;
                    return true;
                }
                break;
            case 1951:
                if ('u' == current) {
                    state = 1952;
                    return true;
                }
                break;
            case 1952:
                if ('b' == current) {
                    state = 1953;
                    return true;
                }
                else if ('p' == current) {
                    state = 1962;
                    return true;
                }
                break;
            case 1953:
                if ('s' == current) {
                    state = 1954;
                    return true;
                }
                break;
            case 1954:
                if ('e' == current) {
                    state = 1955;
                    return true;
                }
                break;
            case 1955:
                if ('t' == current) {
                    state = 1956;
                    return true;
                }
                break;
            case 1956:
                // NotSquareSubset;
                if (';' == current) {
                    match = "\u228F\u0338";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('E' == current) {
                    state = 1957;
                    return true;
                }
                break;
            case 1957:
                if ('q' == current) {
                    state = 1958;
                    return true;
                }
                break;
            case 1958:
                if ('u' == current) {
                    state = 1959;
                    return true;
                }
                break;
            case 1959:
                if ('a' == current) {
                    state = 1960;
                    return true;
                }
                break;
            case 1960:
                if ('l' == current) {
                    state = 1961;
                    return true;
                }
                break;
            case 1961:
                // NotSquareSubsetEqual;
                if (';' == current) {
                    match = "\u22E2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1962:
                if ('e' == current) {
                    state = 1963;
                    return true;
                }
                break;
            case 1963:
                if ('r' == current) {
                    state = 1964;
                    return true;
                }
                break;
            case 1964:
                if ('s' == current) {
                    state = 1965;
                    return true;
                }
                break;
            case 1965:
                if ('e' == current) {
                    state = 1966;
                    return true;
                }
                break;
            case 1966:
                if ('t' == current) {
                    state = 1967;
                    return true;
                }
                break;
            case 1967:
                // NotSquareSuperset;
                if (';' == current) {
                    match = "\u2290\u0338";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('E' == current) {
                    state = 1968;
                    return true;
                }
                break;
            case 1968:
                if ('q' == current) {
                    state = 1969;
                    return true;
                }
                break;
            case 1969:
                if ('u' == current) {
                    state = 1970;
                    return true;
                }
                break;
            case 1970:
                if ('a' == current) {
                    state = 1971;
                    return true;
                }
                break;
            case 1971:
                if ('l' == current) {
                    state = 1972;
                    return true;
                }
                break;
            case 1972:
                // NotSquareSupersetEqual;
                if (';' == current) {
                    match = "\u22E3";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1973:
                switch (current) {
                    case 'b' :
                        state = 1974;
                        return true;
                    case 'c' :
                        state = 1983;
                        return true;
                    case 'p' :
                        state = 2009;
                        return true;
                }
                break;
            case 1974:
                if ('s' == current) {
                    state = 1975;
                    return true;
                }
                break;
            case 1975:
                if ('e' == current) {
                    state = 1976;
                    return true;
                }
                break;
            case 1976:
                if ('t' == current) {
                    state = 1977;
                    return true;
                }
                break;
            case 1977:
                // NotSubset;
                if (';' == current) {
                    match = "\u2282\u20D2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('E' == current) {
                    state = 1978;
                    return true;
                }
                break;
            case 1978:
                if ('q' == current) {
                    state = 1979;
                    return true;
                }
                break;
            case 1979:
                if ('u' == current) {
                    state = 1980;
                    return true;
                }
                break;
            case 1980:
                if ('a' == current) {
                    state = 1981;
                    return true;
                }
                break;
            case 1981:
                if ('l' == current) {
                    state = 1982;
                    return true;
                }
                break;
            case 1982:
                // NotSubsetEqual;
                if (';' == current) {
                    match = "\u2288";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1983:
                if ('c' == current) {
                    state = 1984;
                    return true;
                }
                break;
            case 1984:
                if ('e' == current) {
                    state = 1985;
                    return true;
                }
                break;
            case 1985:
                if ('e' == current) {
                    state = 1986;
                    return true;
                }
                break;
            case 1986:
                if ('d' == current) {
                    state = 1987;
                    return true;
                }
                break;
            case 1987:
                if ('s' == current) {
                    state = 1988;
                    return true;
                }
                break;
            case 1988:
                switch (current) {
                    case ';' : // NotSucceeds;
                        match = "\u2281";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 1989;
                        return true;
                    case 'S' :
                        state = 1994;
                        return true;
                    case 'T' :
                        state = 2004;
                        return true;
                }
                break;
            case 1989:
                if ('q' == current) {
                    state = 1990;
                    return true;
                }
                break;
            case 1990:
                if ('u' == current) {
                    state = 1991;
                    return true;
                }
                break;
            case 1991:
                if ('a' == current) {
                    state = 1992;
                    return true;
                }
                break;
            case 1992:
                if ('l' == current) {
                    state = 1993;
                    return true;
                }
                break;
            case 1993:
                // NotSucceedsEqual;
                if (';' == current) {
                    match = "\u2AB0\u0338";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1994:
                if ('l' == current) {
                    state = 1995;
                    return true;
                }
                break;
            case 1995:
                if ('a' == current) {
                    state = 1996;
                    return true;
                }
                break;
            case 1996:
                if ('n' == current) {
                    state = 1997;
                    return true;
                }
                break;
            case 1997:
                if ('t' == current) {
                    state = 1998;
                    return true;
                }
                break;
            case 1998:
                if ('E' == current) {
                    state = 1999;
                    return true;
                }
                break;
            case 1999:
                if ('q' == current) {
                    state = 2000;
                    return true;
                }
                break;
        }
        return false;
    }

    private boolean parse3(final int current) {
        consumedCount++;
        switch (state) {
            case 2000:
                if ('u' == current) {
                    state = 2001;
                    return true;
                }
                break;
            case 2001:
                if ('a' == current) {
                    state = 2002;
                    return true;
                }
                break;
            case 2002:
                if ('l' == current) {
                    state = 2003;
                    return true;
                }
                break;
            case 2003:
                // NotSucceedsSlantEqual;
                if (';' == current) {
                    match = "\u22E1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2004:
                if ('i' == current) {
                    state = 2005;
                    return true;
                }
                break;
            case 2005:
                if ('l' == current) {
                    state = 2006;
                    return true;
                }
                break;
            case 2006:
                if ('d' == current) {
                    state = 2007;
                    return true;
                }
                break;
            case 2007:
                if ('e' == current) {
                    state = 2008;
                    return true;
                }
                break;
            case 2008:
                // NotSucceedsTilde;
                if (';' == current) {
                    match = "\u227F\u0338";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2009:
                if ('e' == current) {
                    state = 2010;
                    return true;
                }
                break;
            case 2010:
                if ('r' == current) {
                    state = 2011;
                    return true;
                }
                break;
            case 2011:
                if ('s' == current) {
                    state = 2012;
                    return true;
                }
                break;
            case 2012:
                if ('e' == current) {
                    state = 2013;
                    return true;
                }
                break;
            case 2013:
                if ('t' == current) {
                    state = 2014;
                    return true;
                }
                break;
            case 2014:
                // NotSuperset;
                if (';' == current) {
                    match = "\u2283\u20D2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('E' == current) {
                    state = 2015;
                    return true;
                }
                break;
            case 2015:
                if ('q' == current) {
                    state = 2016;
                    return true;
                }
                break;
            case 2016:
                if ('u' == current) {
                    state = 2017;
                    return true;
                }
                break;
            case 2017:
                if ('a' == current) {
                    state = 2018;
                    return true;
                }
                break;
            case 2018:
                if ('l' == current) {
                    state = 2019;
                    return true;
                }
                break;
            case 2019:
                // NotSupersetEqual;
                if (';' == current) {
                    match = "\u2289";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2020:
                if ('i' == current) {
                    state = 2021;
                    return true;
                }
                break;
            case 2021:
                if ('l' == current) {
                    state = 2022;
                    return true;
                }
                break;
            case 2022:
                if ('d' == current) {
                    state = 2023;
                    return true;
                }
                break;
            case 2023:
                if ('e' == current) {
                    state = 2024;
                    return true;
                }
                break;
            case 2024:
                switch (current) {
                    case ';' : // NotTilde;
                        match = "\u2241";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 2025;
                        return true;
                    case 'F' :
                        state = 2030;
                        return true;
                    case 'T' :
                        state = 2039;
                        return true;
                }
                break;
            case 2025:
                if ('q' == current) {
                    state = 2026;
                    return true;
                }
                break;
            case 2026:
                if ('u' == current) {
                    state = 2027;
                    return true;
                }
                break;
            case 2027:
                if ('a' == current) {
                    state = 2028;
                    return true;
                }
                break;
            case 2028:
                if ('l' == current) {
                    state = 2029;
                    return true;
                }
                break;
            case 2029:
                // NotTildeEqual;
                if (';' == current) {
                    match = "\u2244";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2030:
                if ('u' == current) {
                    state = 2031;
                    return true;
                }
                break;
            case 2031:
                if ('l' == current) {
                    state = 2032;
                    return true;
                }
                break;
            case 2032:
                if ('l' == current) {
                    state = 2033;
                    return true;
                }
                break;
            case 2033:
                if ('E' == current) {
                    state = 2034;
                    return true;
                }
                break;
            case 2034:
                if ('q' == current) {
                    state = 2035;
                    return true;
                }
                break;
            case 2035:
                if ('u' == current) {
                    state = 2036;
                    return true;
                }
                break;
            case 2036:
                if ('a' == current) {
                    state = 2037;
                    return true;
                }
                break;
            case 2037:
                if ('l' == current) {
                    state = 2038;
                    return true;
                }
                break;
            case 2038:
                // NotTildeFullEqual;
                if (';' == current) {
                    match = "\u2247";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2039:
                if ('i' == current) {
                    state = 2040;
                    return true;
                }
                break;
            case 2040:
                if ('l' == current) {
                    state = 2041;
                    return true;
                }
                break;
            case 2041:
                if ('d' == current) {
                    state = 2042;
                    return true;
                }
                break;
            case 2042:
                if ('e' == current) {
                    state = 2043;
                    return true;
                }
                break;
            case 2043:
                // NotTildeTilde;
                if (';' == current) {
                    match = "\u2249";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2044:
                if ('e' == current) {
                    state = 2045;
                    return true;
                }
                break;
            case 2045:
                if ('r' == current) {
                    state = 2046;
                    return true;
                }
                break;
            case 2046:
                if ('t' == current) {
                    state = 2047;
                    return true;
                }
                break;
            case 2047:
                if ('i' == current) {
                    state = 2048;
                    return true;
                }
                break;
            case 2048:
                if ('c' == current) {
                    state = 2049;
                    return true;
                }
                break;
            case 2049:
                if ('a' == current) {
                    state = 2050;
                    return true;
                }
                break;
            case 2050:
                if ('l' == current) {
                    state = 2051;
                    return true;
                }
                break;
            case 2051:
                if ('B' == current) {
                    state = 2052;
                    return true;
                }
                break;
            case 2052:
                if ('a' == current) {
                    state = 2053;
                    return true;
                }
                break;
            case 2053:
                if ('r' == current) {
                    state = 2054;
                    return true;
                }
                break;
            case 2054:
                // NotVerticalBar;
                if (';' == current) {
                    match = "\u2224";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2055:
                if ('c' == current) {
                    state = 2056;
                    return true;
                }
                break;
            case 2056:
                if ('r' == current) {
                    state = 2057;
                    return true;
                }
                break;
            case 2057:
                // Nscr;
                if (';' == current) {
                    match = "\uD835\uDCA9";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2058:
                if ('i' == current) {
                    state = 2059;
                    return true;
                }
                break;
            case 2059:
                if ('l' == current) {
                    state = 2060;
                    return true;
                }
                break;
            case 2060:
                if ('d' == current) {
                    state = 2061;
                    return true;
                }
                break;
            case 2061:
                // Ntilde
                if ('e' == current) {
                    match = "\u00D1";
                    matchLength = consumedCount;
                    state = 2062;
                    return true;
                }
                break;
            case 2062:
                // Ntilde;
                if (';' == current) {
                    match = "\u00D1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2063:
                // Nu;
                if (';' == current) {
                    match = "\u039D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2064:
                switch (current) {
                    case 'E' :
                        state = 2065;
                        return true;
                    case 'a' :
                        state = 2069;
                        return true;
                    case 'c' :
                        state = 2074;
                        return true;
                    case 'd' :
                        state = 2079;
                        return true;
                    case 'f' :
                        state = 2084;
                        return true;
                    case 'g' :
                        state = 2086;
                        return true;
                    case 'm' :
                        state = 2091;
                        return true;
                    case 'o' :
                        state = 2103;
                        return true;
                    case 'p' :
                        state = 2106;
                        return true;
                    case 'r' :
                        state = 2130;
                        return true;
                    case 's' :
                        state = 2131;
                        return true;
                    case 't' :
                        state = 2138;
                        return true;
                    case 'u' :
                        state = 2146;
                        return true;
                    case 'v' :
                        state = 2149;
                        return true;
                }
                break;
            case 2065:
                if ('l' == current) {
                    state = 2066;
                    return true;
                }
                break;
            case 2066:
                if ('i' == current) {
                    state = 2067;
                    return true;
                }
                break;
            case 2067:
                if ('g' == current) {
                    state = 2068;
                    return true;
                }
                break;
            case 2068:
                // OElig;
                if (';' == current) {
                    match = "\u0152";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2069:
                if ('c' == current) {
                    state = 2070;
                    return true;
                }
                break;
            case 2070:
                if ('u' == current) {
                    state = 2071;
                    return true;
                }
                break;
            case 2071:
                if ('t' == current) {
                    state = 2072;
                    return true;
                }
                break;
            case 2072:
                // Oacute
                if ('e' == current) {
                    match = "\u00D3";
                    matchLength = consumedCount;
                    state = 2073;
                    return true;
                }
                break;
            case 2073:
                // Oacute;
                if (';' == current) {
                    match = "\u00D3";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2074:
                if ('i' == current) {
                    state = 2075;
                    return true;
                }
                else if ('y' == current) {
                    state = 2078;
                    return true;
                }
                break;
            case 2075:
                if ('r' == current) {
                    state = 2076;
                    return true;
                }
                break;
            case 2076:
                // Ocirc
                if ('c' == current) {
                    match = "\u00D4";
                    matchLength = consumedCount;
                    state = 2077;
                    return true;
                }
                break;
            case 2077:
                // Ocirc;
                if (';' == current) {
                    match = "\u00D4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2078:
                // Ocy;
                if (';' == current) {
                    match = "\u041E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2079:
                if ('b' == current) {
                    state = 2080;
                    return true;
                }
                break;
            case 2080:
                if ('l' == current) {
                    state = 2081;
                    return true;
                }
                break;
            case 2081:
                if ('a' == current) {
                    state = 2082;
                    return true;
                }
                break;
            case 2082:
                if ('c' == current) {
                    state = 2083;
                    return true;
                }
                break;
            case 2083:
                // Odblac;
                if (';' == current) {
                    match = "\u0150";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2084:
                if ('r' == current) {
                    state = 2085;
                    return true;
                }
                break;
            case 2085:
                // Ofr;
                if (';' == current) {
                    match = "\uD835\uDD12";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2086:
                if ('r' == current) {
                    state = 2087;
                    return true;
                }
                break;
            case 2087:
                if ('a' == current) {
                    state = 2088;
                    return true;
                }
                break;
            case 2088:
                if ('v' == current) {
                    state = 2089;
                    return true;
                }
                break;
            case 2089:
                // Ograve
                if ('e' == current) {
                    match = "\u00D2";
                    matchLength = consumedCount;
                    state = 2090;
                    return true;
                }
                break;
            case 2090:
                // Ograve;
                if (';' == current) {
                    match = "\u00D2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2091:
                switch (current) {
                    case 'a' :
                        state = 2092;
                        return true;
                    case 'e' :
                        state = 2095;
                        return true;
                    case 'i' :
                        state = 2098;
                        return true;
                }
                break;
            case 2092:
                if ('c' == current) {
                    state = 2093;
                    return true;
                }
                break;
            case 2093:
                if ('r' == current) {
                    state = 2094;
                    return true;
                }
                break;
            case 2094:
                // Omacr;
                if (';' == current) {
                    match = "\u014C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2095:
                if ('g' == current) {
                    state = 2096;
                    return true;
                }
                break;
            case 2096:
                if ('a' == current) {
                    state = 2097;
                    return true;
                }
                break;
            case 2097:
                // Omega;
                if (';' == current) {
                    match = "\u03A9";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2098:
                if ('c' == current) {
                    state = 2099;
                    return true;
                }
                break;
            case 2099:
                if ('r' == current) {
                    state = 2100;
                    return true;
                }
                break;
            case 2100:
                if ('o' == current) {
                    state = 2101;
                    return true;
                }
                break;
            case 2101:
                if ('n' == current) {
                    state = 2102;
                    return true;
                }
                break;
            case 2102:
                // Omicron;
                if (';' == current) {
                    match = "\u039F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2103:
                if ('p' == current) {
                    state = 2104;
                    return true;
                }
                break;
            case 2104:
                if ('f' == current) {
                    state = 2105;
                    return true;
                }
                break;
            case 2105:
                // Oopf;
                if (';' == current) {
                    match = "\uD835\uDD46";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2106:
                if ('e' == current) {
                    state = 2107;
                    return true;
                }
                break;
            case 2107:
                if ('n' == current) {
                    state = 2108;
                    return true;
                }
                break;
            case 2108:
                if ('C' == current) {
                    state = 2109;
                    return true;
                }
                break;
            case 2109:
                if ('u' == current) {
                    state = 2110;
                    return true;
                }
                break;
            case 2110:
                if ('r' == current) {
                    state = 2111;
                    return true;
                }
                break;
            case 2111:
                if ('l' == current) {
                    state = 2112;
                    return true;
                }
                break;
            case 2112:
                if ('y' == current) {
                    state = 2113;
                    return true;
                }
                break;
            case 2113:
                if ('D' == current) {
                    state = 2114;
                    return true;
                }
                else if ('Q' == current) {
                    state = 2125;
                    return true;
                }
                break;
            case 2114:
                if ('o' == current) {
                    state = 2115;
                    return true;
                }
                break;
            case 2115:
                if ('u' == current) {
                    state = 2116;
                    return true;
                }
                break;
            case 2116:
                if ('b' == current) {
                    state = 2117;
                    return true;
                }
                break;
            case 2117:
                if ('l' == current) {
                    state = 2118;
                    return true;
                }
                break;
            case 2118:
                if ('e' == current) {
                    state = 2119;
                    return true;
                }
                break;
            case 2119:
                if ('Q' == current) {
                    state = 2120;
                    return true;
                }
                break;
            case 2120:
                if ('u' == current) {
                    state = 2121;
                    return true;
                }
                break;
            case 2121:
                if ('o' == current) {
                    state = 2122;
                    return true;
                }
                break;
            case 2122:
                if ('t' == current) {
                    state = 2123;
                    return true;
                }
                break;
            case 2123:
                if ('e' == current) {
                    state = 2124;
                    return true;
                }
                break;
            case 2124:
                // OpenCurlyDoubleQuote;
                if (';' == current) {
                    match = "\u201C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2125:
                if ('u' == current) {
                    state = 2126;
                    return true;
                }
                break;
            case 2126:
                if ('o' == current) {
                    state = 2127;
                    return true;
                }
                break;
            case 2127:
                if ('t' == current) {
                    state = 2128;
                    return true;
                }
                break;
            case 2128:
                if ('e' == current) {
                    state = 2129;
                    return true;
                }
                break;
            case 2129:
                // OpenCurlyQuote;
                if (';' == current) {
                    match = "\u2018";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2130:
                // Or;
                if (';' == current) {
                    match = "\u2A54";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2131:
                if ('c' == current) {
                    state = 2132;
                    return true;
                }
                else if ('l' == current) {
                    state = 2134;
                    return true;
                }
                break;
            case 2132:
                if ('r' == current) {
                    state = 2133;
                    return true;
                }
                break;
            case 2133:
                // Oscr;
                if (';' == current) {
                    match = "\uD835\uDCAA";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2134:
                if ('a' == current) {
                    state = 2135;
                    return true;
                }
                break;
            case 2135:
                if ('s' == current) {
                    state = 2136;
                    return true;
                }
                break;
            case 2136:
                // Oslash
                if ('h' == current) {
                    match = "\u00D8";
                    matchLength = consumedCount;
                    state = 2137;
                    return true;
                }
                break;
            case 2137:
                // Oslash;
                if (';' == current) {
                    match = "\u00D8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2138:
                if ('i' == current) {
                    state = 2139;
                    return true;
                }
                break;
            case 2139:
                if ('l' == current) {
                    state = 2140;
                    return true;
                }
                else if ('m' == current) {
                    state = 2143;
                    return true;
                }
                break;
            case 2140:
                if ('d' == current) {
                    state = 2141;
                    return true;
                }
                break;
            case 2141:
                // Otilde
                if ('e' == current) {
                    match = "\u00D5";
                    matchLength = consumedCount;
                    state = 2142;
                    return true;
                }
                break;
            case 2142:
                // Otilde;
                if (';' == current) {
                    match = "\u00D5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2143:
                if ('e' == current) {
                    state = 2144;
                    return true;
                }
                break;
            case 2144:
                if ('s' == current) {
                    state = 2145;
                    return true;
                }
                break;
            case 2145:
                // Otimes;
                if (';' == current) {
                    match = "\u2A37";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2146:
                if ('m' == current) {
                    state = 2147;
                    return true;
                }
                break;
            case 2147:
                // Ouml
                if ('l' == current) {
                    match = "\u00D6";
                    matchLength = consumedCount;
                    state = 2148;
                    return true;
                }
                break;
            case 2148:
                // Ouml;
                if (';' == current) {
                    match = "\u00D6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2149:
                if ('e' == current) {
                    state = 2150;
                    return true;
                }
                break;
            case 2150:
                if ('r' == current) {
                    state = 2151;
                    return true;
                }
                break;
            case 2151:
                if ('B' == current) {
                    state = 2152;
                    return true;
                }
                else if ('P' == current) {
                    state = 2162;
                    return true;
                }
                break;
            case 2152:
                if ('a' == current) {
                    state = 2153;
                    return true;
                }
                else if ('r' == current) {
                    state = 2155;
                    return true;
                }
                break;
            case 2153:
                if ('r' == current) {
                    state = 2154;
                    return true;
                }
                break;
            case 2154:
                // OverBar;
                if (';' == current) {
                    match = "\u203E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2155:
                if ('a' == current) {
                    state = 2156;
                    return true;
                }
                break;
            case 2156:
                if ('c' == current) {
                    state = 2157;
                    return true;
                }
                break;
            case 2157:
                if ('e' == current) {
                    state = 2158;
                    return true;
                }
                else if ('k' == current) {
                    state = 2159;
                    return true;
                }
                break;
            case 2158:
                // OverBrace;
                if (';' == current) {
                    match = "\u23DE";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2159:
                if ('e' == current) {
                    state = 2160;
                    return true;
                }
                break;
            case 2160:
                if ('t' == current) {
                    state = 2161;
                    return true;
                }
                break;
            case 2161:
                // OverBracket;
                if (';' == current) {
                    match = "\u23B4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2162:
                if ('a' == current) {
                    state = 2163;
                    return true;
                }
                break;
            case 2163:
                if ('r' == current) {
                    state = 2164;
                    return true;
                }
                break;
            case 2164:
                if ('e' == current) {
                    state = 2165;
                    return true;
                }
                break;
            case 2165:
                if ('n' == current) {
                    state = 2166;
                    return true;
                }
                break;
            case 2166:
                if ('t' == current) {
                    state = 2167;
                    return true;
                }
                break;
            case 2167:
                if ('h' == current) {
                    state = 2168;
                    return true;
                }
                break;
            case 2168:
                if ('e' == current) {
                    state = 2169;
                    return true;
                }
                break;
            case 2169:
                if ('s' == current) {
                    state = 2170;
                    return true;
                }
                break;
            case 2170:
                if ('i' == current) {
                    state = 2171;
                    return true;
                }
                break;
            case 2171:
                if ('s' == current) {
                    state = 2172;
                    return true;
                }
                break;
            case 2172:
                // OverParenthesis;
                if (';' == current) {
                    match = "\u23DC";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2173:
                switch (current) {
                    case 'a' :
                        state = 2174;
                        return true;
                    case 'c' :
                        state = 2181;
                        return true;
                    case 'f' :
                        state = 2183;
                        return true;
                    case 'h' :
                        state = 2185;
                        return true;
                    case 'i' :
                        state = 2187;
                        return true;
                    case 'l' :
                        state = 2188;
                        return true;
                    case 'o' :
                        state = 2196;
                        return true;
                    case 'r' :
                        state = 2210;
                        return true;
                    case 's' :
                        state = 2254;
                        return true;
                }
                break;
            case 2174:
                if ('r' == current) {
                    state = 2175;
                    return true;
                }
                break;
            case 2175:
                if ('t' == current) {
                    state = 2176;
                    return true;
                }
                break;
            case 2176:
                if ('i' == current) {
                    state = 2177;
                    return true;
                }
                break;
            case 2177:
                if ('a' == current) {
                    state = 2178;
                    return true;
                }
                break;
            case 2178:
                if ('l' == current) {
                    state = 2179;
                    return true;
                }
                break;
            case 2179:
                if ('D' == current) {
                    state = 2180;
                    return true;
                }
                break;
            case 2180:
                // PartialD;
                if (';' == current) {
                    match = "\u2202";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2181:
                if ('y' == current) {
                    state = 2182;
                    return true;
                }
                break;
            case 2182:
                // Pcy;
                if (';' == current) {
                    match = "\u041F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2183:
                if ('r' == current) {
                    state = 2184;
                    return true;
                }
                break;
            case 2184:
                // Pfr;
                if (';' == current) {
                    match = "\uD835\uDD13";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2185:
                if ('i' == current) {
                    state = 2186;
                    return true;
                }
                break;
            case 2186:
                // Phi;
                if (';' == current) {
                    match = "\u03A6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2187:
                // Pi;
                if (';' == current) {
                    match = "\u03A0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2188:
                if ('u' == current) {
                    state = 2189;
                    return true;
                }
                break;
            case 2189:
                if ('s' == current) {
                    state = 2190;
                    return true;
                }
                break;
            case 2190:
                if ('M' == current) {
                    state = 2191;
                    return true;
                }
                break;
            case 2191:
                if ('i' == current) {
                    state = 2192;
                    return true;
                }
                break;
            case 2192:
                if ('n' == current) {
                    state = 2193;
                    return true;
                }
                break;
            case 2193:
                if ('u' == current) {
                    state = 2194;
                    return true;
                }
                break;
            case 2194:
                if ('s' == current) {
                    state = 2195;
                    return true;
                }
                break;
            case 2195:
                // PlusMinus;
                if (';' == current) {
                    match = "\u00B1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2196:
                if ('i' == current) {
                    state = 2197;
                    return true;
                }
                else if ('p' == current) {
                    state = 2208;
                    return true;
                }
                break;
            case 2197:
                if ('n' == current) {
                    state = 2198;
                    return true;
                }
                break;
            case 2198:
                if ('c' == current) {
                    state = 2199;
                    return true;
                }
                break;
            case 2199:
                if ('a' == current) {
                    state = 2200;
                    return true;
                }
                break;
            case 2200:
                if ('r' == current) {
                    state = 2201;
                    return true;
                }
                break;
            case 2201:
                if ('e' == current) {
                    state = 2202;
                    return true;
                }
                break;
            case 2202:
                if ('p' == current) {
                    state = 2203;
                    return true;
                }
                break;
            case 2203:
                if ('l' == current) {
                    state = 2204;
                    return true;
                }
                break;
            case 2204:
                if ('a' == current) {
                    state = 2205;
                    return true;
                }
                break;
            case 2205:
                if ('n' == current) {
                    state = 2206;
                    return true;
                }
                break;
            case 2206:
                if ('e' == current) {
                    state = 2207;
                    return true;
                }
                break;
            case 2207:
                // Poincareplane;
                if (';' == current) {
                    match = "\u210C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2208:
                if ('f' == current) {
                    state = 2209;
                    return true;
                }
                break;
            case 2209:
                // Popf;
                if (';' == current) {
                    match = "\u2119";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2210:
                switch (current) {
                    case ';' : // Pr;
                        match = "\u2ABB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 2211;
                        return true;
                    case 'i' :
                        state = 2237;
                        return true;
                    case 'o' :
                        state = 2240;
                        return true;
                }
                break;
            case 2211:
                if ('c' == current) {
                    state = 2212;
                    return true;
                }
                break;
            case 2212:
                if ('e' == current) {
                    state = 2213;
                    return true;
                }
                break;
            case 2213:
                if ('d' == current) {
                    state = 2214;
                    return true;
                }
                break;
            case 2214:
                if ('e' == current) {
                    state = 2215;
                    return true;
                }
                break;
            case 2215:
                if ('s' == current) {
                    state = 2216;
                    return true;
                }
                break;
            case 2216:
                switch (current) {
                    case ';' : // Precedes;
                        match = "\u227A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 2217;
                        return true;
                    case 'S' :
                        state = 2222;
                        return true;
                    case 'T' :
                        state = 2232;
                        return true;
                }
                break;
            case 2217:
                if ('q' == current) {
                    state = 2218;
                    return true;
                }
                break;
            case 2218:
                if ('u' == current) {
                    state = 2219;
                    return true;
                }
                break;
            case 2219:
                if ('a' == current) {
                    state = 2220;
                    return true;
                }
                break;
            case 2220:
                if ('l' == current) {
                    state = 2221;
                    return true;
                }
                break;
            case 2221:
                // PrecedesEqual;
                if (';' == current) {
                    match = "\u2AAF";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2222:
                if ('l' == current) {
                    state = 2223;
                    return true;
                }
                break;
            case 2223:
                if ('a' == current) {
                    state = 2224;
                    return true;
                }
                break;
            case 2224:
                if ('n' == current) {
                    state = 2225;
                    return true;
                }
                break;
            case 2225:
                if ('t' == current) {
                    state = 2226;
                    return true;
                }
                break;
            case 2226:
                if ('E' == current) {
                    state = 2227;
                    return true;
                }
                break;
            case 2227:
                if ('q' == current) {
                    state = 2228;
                    return true;
                }
                break;
            case 2228:
                if ('u' == current) {
                    state = 2229;
                    return true;
                }
                break;
            case 2229:
                if ('a' == current) {
                    state = 2230;
                    return true;
                }
                break;
            case 2230:
                if ('l' == current) {
                    state = 2231;
                    return true;
                }
                break;
            case 2231:
                // PrecedesSlantEqual;
                if (';' == current) {
                    match = "\u227C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2232:
                if ('i' == current) {
                    state = 2233;
                    return true;
                }
                break;
            case 2233:
                if ('l' == current) {
                    state = 2234;
                    return true;
                }
                break;
            case 2234:
                if ('d' == current) {
                    state = 2235;
                    return true;
                }
                break;
            case 2235:
                if ('e' == current) {
                    state = 2236;
                    return true;
                }
                break;
            case 2236:
                // PrecedesTilde;
                if (';' == current) {
                    match = "\u227E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2237:
                if ('m' == current) {
                    state = 2238;
                    return true;
                }
                break;
            case 2238:
                if ('e' == current) {
                    state = 2239;
                    return true;
                }
                break;
            case 2239:
                // Prime;
                if (';' == current) {
                    match = "\u2033";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2240:
                if ('d' == current) {
                    state = 2241;
                    return true;
                }
                else if ('p' == current) {
                    state = 2245;
                    return true;
                }
                break;
            case 2241:
                if ('u' == current) {
                    state = 2242;
                    return true;
                }
                break;
            case 2242:
                if ('c' == current) {
                    state = 2243;
                    return true;
                }
                break;
            case 2243:
                if ('t' == current) {
                    state = 2244;
                    return true;
                }
                break;
            case 2244:
                // Product;
                if (';' == current) {
                    match = "\u220F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2245:
                if ('o' == current) {
                    state = 2246;
                    return true;
                }
                break;
            case 2246:
                if ('r' == current) {
                    state = 2247;
                    return true;
                }
                break;
            case 2247:
                if ('t' == current) {
                    state = 2248;
                    return true;
                }
                break;
            case 2248:
                if ('i' == current) {
                    state = 2249;
                    return true;
                }
                break;
            case 2249:
                if ('o' == current) {
                    state = 2250;
                    return true;
                }
                break;
            case 2250:
                if ('n' == current) {
                    state = 2251;
                    return true;
                }
                break;
            case 2251:
                // Proportion;
                if (';' == current) {
                    match = "\u2237";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('a' == current) {
                    state = 2252;
                    return true;
                }
                break;
            case 2252:
                if ('l' == current) {
                    state = 2253;
                    return true;
                }
                break;
            case 2253:
                // Proportional;
                if (';' == current) {
                    match = "\u221D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2254:
                if ('c' == current) {
                    state = 2255;
                    return true;
                }
                else if ('i' == current) {
                    state = 2257;
                    return true;
                }
                break;
            case 2255:
                if ('r' == current) {
                    state = 2256;
                    return true;
                }
                break;
            case 2256:
                // Pscr;
                if (';' == current) {
                    match = "\uD835\uDCAB";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2257:
                // Psi;
                if (';' == current) {
                    match = "\u03A8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2258:
                switch (current) {
                    case 'U' :
                        state = 2259;
                        return true;
                    case 'f' :
                        state = 2262;
                        return true;
                    case 'o' :
                        state = 2264;
                        return true;
                    case 's' :
                        state = 2267;
                        return true;
                }
                break;
            case 2259:
                if ('O' == current) {
                    state = 2260;
                    return true;
                }
                break;
            case 2260:
                // QUOT
                if ('T' == current) {
                    match = "\"";
                    matchLength = consumedCount;
                    state = 2261;
                    return true;
                }
                break;
            case 2261:
                // QUOT;
                if (';' == current) {
                    match = "\"";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2262:
                if ('r' == current) {
                    state = 2263;
                    return true;
                }
                break;
            case 2263:
                // Qfr;
                if (';' == current) {
                    match = "\uD835\uDD14";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2264:
                if ('p' == current) {
                    state = 2265;
                    return true;
                }
                break;
            case 2265:
                if ('f' == current) {
                    state = 2266;
                    return true;
                }
                break;
            case 2266:
                // Qopf;
                if (';' == current) {
                    match = "\u211A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2267:
                if ('c' == current) {
                    state = 2268;
                    return true;
                }
                break;
            case 2268:
                if ('r' == current) {
                    state = 2269;
                    return true;
                }
                break;
            case 2269:
                // Qscr;
                if (';' == current) {
                    match = "\uD835\uDCAC";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2270:
                switch (current) {
                    case 'B' :
                        state = 2271;
                        return true;
                    case 'E' :
                        state = 2275;
                        return true;
                    case 'a' :
                        state = 2277;
                        return true;
                    case 'c' :
                        state = 2288;
                        return true;
                    case 'e' :
                        state = 2298;
                        return true;
                    case 'f' :
                        state = 2334;
                        return true;
                    case 'h' :
                        state = 2336;
                        return true;
                    case 'i' :
                        state = 2338;
                        return true;
                    case 'o' :
                        state = 2488;
                        return true;
                    case 'r' :
                        state = 2501;
                        return true;
                    case 's' :
                        state = 2511;
                        return true;
                    case 'u' :
                        state = 2515;
                        return true;
                }
                break;
            case 2271:
                if ('a' == current) {
                    state = 2272;
                    return true;
                }
                break;
            case 2272:
                if ('r' == current) {
                    state = 2273;
                    return true;
                }
                break;
            case 2273:
                if ('r' == current) {
                    state = 2274;
                    return true;
                }
                break;
            case 2274:
                // RBarr;
                if (';' == current) {
                    match = "\u2910";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2275:
                // REG
                if ('G' == current) {
                    match = "\u00AE";
                    matchLength = consumedCount;
                    state = 2276;
                    return true;
                }
                break;
            case 2276:
                // REG;
                if (';' == current) {
                    match = "\u00AE";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2277:
                switch (current) {
                    case 'c' :
                        state = 2278;
                        return true;
                    case 'n' :
                        state = 2282;
                        return true;
                    case 'r' :
                        state = 2284;
                        return true;
                }
                break;
            case 2278:
                if ('u' == current) {
                    state = 2279;
                    return true;
                }
                break;
            case 2279:
                if ('t' == current) {
                    state = 2280;
                    return true;
                }
                break;
            case 2280:
                if ('e' == current) {
                    state = 2281;
                    return true;
                }
                break;
            case 2281:
                // Racute;
                if (';' == current) {
                    match = "\u0154";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2282:
                if ('g' == current) {
                    state = 2283;
                    return true;
                }
                break;
            case 2283:
                // Rang;
                if (';' == current) {
                    match = "\u27EB";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2284:
                if ('r' == current) {
                    state = 2285;
                    return true;
                }
                break;
            case 2285:
                // Rarr;
                if (';' == current) {
                    match = "\u21A0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('t' == current) {
                    state = 2286;
                    return true;
                }
                break;
            case 2286:
                if ('l' == current) {
                    state = 2287;
                    return true;
                }
                break;
            case 2287:
                // Rarrtl;
                if (';' == current) {
                    match = "\u2916";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2288:
                switch (current) {
                    case 'a' :
                        state = 2289;
                        return true;
                    case 'e' :
                        state = 2293;
                        return true;
                    case 'y' :
                        state = 2297;
                        return true;
                }
                break;
            case 2289:
                if ('r' == current) {
                    state = 2290;
                    return true;
                }
                break;
            case 2290:
                if ('o' == current) {
                    state = 2291;
                    return true;
                }
                break;
            case 2291:
                if ('n' == current) {
                    state = 2292;
                    return true;
                }
                break;
            case 2292:
                // Rcaron;
                if (';' == current) {
                    match = "\u0158";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2293:
                if ('d' == current) {
                    state = 2294;
                    return true;
                }
                break;
            case 2294:
                if ('i' == current) {
                    state = 2295;
                    return true;
                }
                break;
            case 2295:
                if ('l' == current) {
                    state = 2296;
                    return true;
                }
                break;
            case 2296:
                // Rcedil;
                if (';' == current) {
                    match = "\u0156";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2297:
                // Rcy;
                if (';' == current) {
                    match = "\u0420";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2298:
                // Re;
                if (';' == current) {
                    match = "\u211C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('v' == current) {
                    state = 2299;
                    return true;
                }
                break;
            case 2299:
                if ('e' == current) {
                    state = 2300;
                    return true;
                }
                break;
            case 2300:
                if ('r' == current) {
                    state = 2301;
                    return true;
                }
                break;
            case 2301:
                if ('s' == current) {
                    state = 2302;
                    return true;
                }
                break;
            case 2302:
                if ('e' == current) {
                    state = 2303;
                    return true;
                }
                break;
            case 2303:
                if ('E' == current) {
                    state = 2304;
                    return true;
                }
                else if ('U' == current) {
                    state = 2321;
                    return true;
                }
                break;
            case 2304:
                if ('l' == current) {
                    state = 2305;
                    return true;
                }
                else if ('q' == current) {
                    state = 2311;
                    return true;
                }
                break;
            case 2305:
                if ('e' == current) {
                    state = 2306;
                    return true;
                }
                break;
            case 2306:
                if ('m' == current) {
                    state = 2307;
                    return true;
                }
                break;
            case 2307:
                if ('e' == current) {
                    state = 2308;
                    return true;
                }
                break;
            case 2308:
                if ('n' == current) {
                    state = 2309;
                    return true;
                }
                break;
            case 2309:
                if ('t' == current) {
                    state = 2310;
                    return true;
                }
                break;
            case 2310:
                // ReverseElement;
                if (';' == current) {
                    match = "\u220B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2311:
                if ('u' == current) {
                    state = 2312;
                    return true;
                }
                break;
            case 2312:
                if ('i' == current) {
                    state = 2313;
                    return true;
                }
                break;
            case 2313:
                if ('l' == current) {
                    state = 2314;
                    return true;
                }
                break;
            case 2314:
                if ('i' == current) {
                    state = 2315;
                    return true;
                }
                break;
            case 2315:
                if ('b' == current) {
                    state = 2316;
                    return true;
                }
                break;
            case 2316:
                if ('r' == current) {
                    state = 2317;
                    return true;
                }
                break;
            case 2317:
                if ('i' == current) {
                    state = 2318;
                    return true;
                }
                break;
            case 2318:
                if ('u' == current) {
                    state = 2319;
                    return true;
                }
                break;
            case 2319:
                if ('m' == current) {
                    state = 2320;
                    return true;
                }
                break;
            case 2320:
                // ReverseEquilibrium;
                if (';' == current) {
                    match = "\u21CB";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2321:
                if ('p' == current) {
                    state = 2322;
                    return true;
                }
                break;
            case 2322:
                if ('E' == current) {
                    state = 2323;
                    return true;
                }
                break;
            case 2323:
                if ('q' == current) {
                    state = 2324;
                    return true;
                }
                break;
            case 2324:
                if ('u' == current) {
                    state = 2325;
                    return true;
                }
                break;
            case 2325:
                if ('i' == current) {
                    state = 2326;
                    return true;
                }
                break;
            case 2326:
                if ('l' == current) {
                    state = 2327;
                    return true;
                }
                break;
            case 2327:
                if ('i' == current) {
                    state = 2328;
                    return true;
                }
                break;
            case 2328:
                if ('b' == current) {
                    state = 2329;
                    return true;
                }
                break;
            case 2329:
                if ('r' == current) {
                    state = 2330;
                    return true;
                }
                break;
            case 2330:
                if ('i' == current) {
                    state = 2331;
                    return true;
                }
                break;
            case 2331:
                if ('u' == current) {
                    state = 2332;
                    return true;
                }
                break;
            case 2332:
                if ('m' == current) {
                    state = 2333;
                    return true;
                }
                break;
            case 2333:
                // ReverseUpEquilibrium;
                if (';' == current) {
                    match = "\u296F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2334:
                if ('r' == current) {
                    state = 2335;
                    return true;
                }
                break;
            case 2335:
                // Rfr;
                if (';' == current) {
                    match = "\u211C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2336:
                if ('o' == current) {
                    state = 2337;
                    return true;
                }
                break;
            case 2337:
                // Rho;
                if (';' == current) {
                    match = "\u03A1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2338:
                if ('g' == current) {
                    state = 2339;
                    return true;
                }
                break;
            case 2339:
                if ('h' == current) {
                    state = 2340;
                    return true;
                }
                break;
            case 2340:
                if ('t' == current) {
                    state = 2341;
                    return true;
                }
                break;
            case 2341:
                switch (current) {
                    case 'A' :
                        state = 2342;
                        return true;
                    case 'C' :
                        state = 2370;
                        return true;
                    case 'D' :
                        state = 2377;
                        return true;
                    case 'F' :
                        state = 2410;
                        return true;
                    case 'T' :
                        state = 2415;
                        return true;
                    case 'U' :
                        state = 2444;
                        return true;
                    case 'V' :
                        state = 2474;
                        return true;
                    case 'a' :
                        state = 2483;
                        return true;
                }
                break;
            case 2342:
                if ('n' == current) {
                    state = 2343;
                    return true;
                }
                else if ('r' == current) {
                    state = 2354;
                    return true;
                }
                break;
            case 2343:
                if ('g' == current) {
                    state = 2344;
                    return true;
                }
                break;
            case 2344:
                if ('l' == current) {
                    state = 2345;
                    return true;
                }
                break;
            case 2345:
                if ('e' == current) {
                    state = 2346;
                    return true;
                }
                break;
            case 2346:
                if ('B' == current) {
                    state = 2347;
                    return true;
                }
                break;
            case 2347:
                if ('r' == current) {
                    state = 2348;
                    return true;
                }
                break;
            case 2348:
                if ('a' == current) {
                    state = 2349;
                    return true;
                }
                break;
            case 2349:
                if ('c' == current) {
                    state = 2350;
                    return true;
                }
                break;
            case 2350:
                if ('k' == current) {
                    state = 2351;
                    return true;
                }
                break;
            case 2351:
                if ('e' == current) {
                    state = 2352;
                    return true;
                }
                break;
            case 2352:
                if ('t' == current) {
                    state = 2353;
                    return true;
                }
                break;
            case 2353:
                // RightAngleBracket;
                if (';' == current) {
                    match = "\u27E9";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2354:
                if ('r' == current) {
                    state = 2355;
                    return true;
                }
                break;
            case 2355:
                if ('o' == current) {
                    state = 2356;
                    return true;
                }
                break;
            case 2356:
                if ('w' == current) {
                    state = 2357;
                    return true;
                }
                break;
            case 2357:
                switch (current) {
                    case ';' : // RightArrow;
                        match = "\u2192";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 2358;
                        return true;
                    case 'L' :
                        state = 2361;
                        return true;
                }
                break;
            case 2358:
                if ('a' == current) {
                    state = 2359;
                    return true;
                }
                break;
            case 2359:
                if ('r' == current) {
                    state = 2360;
                    return true;
                }
                break;
            case 2360:
                // RightArrowBar;
                if (';' == current) {
                    match = "\u21E5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2361:
                if ('e' == current) {
                    state = 2362;
                    return true;
                }
                break;
            case 2362:
                if ('f' == current) {
                    state = 2363;
                    return true;
                }
                break;
            case 2363:
                if ('t' == current) {
                    state = 2364;
                    return true;
                }
                break;
            case 2364:
                if ('A' == current) {
                    state = 2365;
                    return true;
                }
                break;
            case 2365:
                if ('r' == current) {
                    state = 2366;
                    return true;
                }
                break;
            case 2366:
                if ('r' == current) {
                    state = 2367;
                    return true;
                }
                break;
            case 2367:
                if ('o' == current) {
                    state = 2368;
                    return true;
                }
                break;
            case 2368:
                if ('w' == current) {
                    state = 2369;
                    return true;
                }
                break;
            case 2369:
                // RightArrowLeftArrow;
                if (';' == current) {
                    match = "\u21C4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2370:
                if ('e' == current) {
                    state = 2371;
                    return true;
                }
                break;
            case 2371:
                if ('i' == current) {
                    state = 2372;
                    return true;
                }
                break;
            case 2372:
                if ('l' == current) {
                    state = 2373;
                    return true;
                }
                break;
            case 2373:
                if ('i' == current) {
                    state = 2374;
                    return true;
                }
                break;
            case 2374:
                if ('n' == current) {
                    state = 2375;
                    return true;
                }
                break;
            case 2375:
                if ('g' == current) {
                    state = 2376;
                    return true;
                }
                break;
            case 2376:
                // RightCeiling;
                if (';' == current) {
                    match = "\u2309";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2377:
                if ('o' == current) {
                    state = 2378;
                    return true;
                }
                break;
            case 2378:
                if ('u' == current) {
                    state = 2379;
                    return true;
                }
                else if ('w' == current) {
                    state = 2390;
                    return true;
                }
                break;
            case 2379:
                if ('b' == current) {
                    state = 2380;
                    return true;
                }
                break;
            case 2380:
                if ('l' == current) {
                    state = 2381;
                    return true;
                }
                break;
            case 2381:
                if ('e' == current) {
                    state = 2382;
                    return true;
                }
                break;
            case 2382:
                if ('B' == current) {
                    state = 2383;
                    return true;
                }
                break;
            case 2383:
                if ('r' == current) {
                    state = 2384;
                    return true;
                }
                break;
            case 2384:
                if ('a' == current) {
                    state = 2385;
                    return true;
                }
                break;
            case 2385:
                if ('c' == current) {
                    state = 2386;
                    return true;
                }
                break;
            case 2386:
                if ('k' == current) {
                    state = 2387;
                    return true;
                }
                break;
            case 2387:
                if ('e' == current) {
                    state = 2388;
                    return true;
                }
                break;
            case 2388:
                if ('t' == current) {
                    state = 2389;
                    return true;
                }
                break;
            case 2389:
                // RightDoubleBracket;
                if (';' == current) {
                    match = "\u27E7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2390:
                if ('n' == current) {
                    state = 2391;
                    return true;
                }
                break;
            case 2391:
                if ('T' == current) {
                    state = 2392;
                    return true;
                }
                else if ('V' == current) {
                    state = 2401;
                    return true;
                }
                break;
            case 2392:
                if ('e' == current) {
                    state = 2393;
                    return true;
                }
                break;
            case 2393:
                if ('e' == current) {
                    state = 2394;
                    return true;
                }
                break;
            case 2394:
                if ('V' == current) {
                    state = 2395;
                    return true;
                }
                break;
            case 2395:
                if ('e' == current) {
                    state = 2396;
                    return true;
                }
                break;
            case 2396:
                if ('c' == current) {
                    state = 2397;
                    return true;
                }
                break;
            case 2397:
                if ('t' == current) {
                    state = 2398;
                    return true;
                }
                break;
            case 2398:
                if ('o' == current) {
                    state = 2399;
                    return true;
                }
                break;
            case 2399:
                if ('r' == current) {
                    state = 2400;
                    return true;
                }
                break;
            case 2400:
                // RightDownTeeVector;
                if (';' == current) {
                    match = "\u295D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2401:
                if ('e' == current) {
                    state = 2402;
                    return true;
                }
                break;
            case 2402:
                if ('c' == current) {
                    state = 2403;
                    return true;
                }
                break;
            case 2403:
                if ('t' == current) {
                    state = 2404;
                    return true;
                }
                break;
            case 2404:
                if ('o' == current) {
                    state = 2405;
                    return true;
                }
                break;
            case 2405:
                if ('r' == current) {
                    state = 2406;
                    return true;
                }
                break;
            case 2406:
                // RightDownVector;
                if (';' == current) {
                    match = "\u21C2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('B' == current) {
                    state = 2407;
                    return true;
                }
                break;
            case 2407:
                if ('a' == current) {
                    state = 2408;
                    return true;
                }
                break;
            case 2408:
                if ('r' == current) {
                    state = 2409;
                    return true;
                }
                break;
            case 2409:
                // RightDownVectorBar;
                if (';' == current) {
                    match = "\u2955";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2410:
                if ('l' == current) {
                    state = 2411;
                    return true;
                }
                break;
            case 2411:
                if ('o' == current) {
                    state = 2412;
                    return true;
                }
                break;
            case 2412:
                if ('o' == current) {
                    state = 2413;
                    return true;
                }
                break;
            case 2413:
                if ('r' == current) {
                    state = 2414;
                    return true;
                }
                break;
            case 2414:
                // RightFloor;
                if (';' == current) {
                    match = "\u230B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2415:
                if ('e' == current) {
                    state = 2416;
                    return true;
                }
                else if ('r' == current) {
                    state = 2429;
                    return true;
                }
                break;
            case 2416:
                if ('e' == current) {
                    state = 2417;
                    return true;
                }
                break;
            case 2417:
                switch (current) {
                    case ';' : // RightTee;
                        match = "\u22A2";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'A' :
                        state = 2418;
                        return true;
                    case 'V' :
                        state = 2423;
                        return true;
                }
                break;
            case 2418:
                if ('r' == current) {
                    state = 2419;
                    return true;
                }
                break;
            case 2419:
                if ('r' == current) {
                    state = 2420;
                    return true;
                }
                break;
            case 2420:
                if ('o' == current) {
                    state = 2421;
                    return true;
                }
                break;
            case 2421:
                if ('w' == current) {
                    state = 2422;
                    return true;
                }
                break;
            case 2422:
                // RightTeeArrow;
                if (';' == current) {
                    match = "\u21A6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2423:
                if ('e' == current) {
                    state = 2424;
                    return true;
                }
                break;
            case 2424:
                if ('c' == current) {
                    state = 2425;
                    return true;
                }
                break;
            case 2425:
                if ('t' == current) {
                    state = 2426;
                    return true;
                }
                break;
            case 2426:
                if ('o' == current) {
                    state = 2427;
                    return true;
                }
                break;
            case 2427:
                if ('r' == current) {
                    state = 2428;
                    return true;
                }
                break;
            case 2428:
                // RightTeeVector;
                if (';' == current) {
                    match = "\u295B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2429:
                if ('i' == current) {
                    state = 2430;
                    return true;
                }
                break;
            case 2430:
                if ('a' == current) {
                    state = 2431;
                    return true;
                }
                break;
            case 2431:
                if ('n' == current) {
                    state = 2432;
                    return true;
                }
                break;
            case 2432:
                if ('g' == current) {
                    state = 2433;
                    return true;
                }
                break;
            case 2433:
                if ('l' == current) {
                    state = 2434;
                    return true;
                }
                break;
            case 2434:
                if ('e' == current) {
                    state = 2435;
                    return true;
                }
                break;
            case 2435:
                switch (current) {
                    case ';' : // RightTriangle;
                        match = "\u22B3";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 2436;
                        return true;
                    case 'E' :
                        state = 2439;
                        return true;
                }
                break;
            case 2436:
                if ('a' == current) {
                    state = 2437;
                    return true;
                }
                break;
            case 2437:
                if ('r' == current) {
                    state = 2438;
                    return true;
                }
                break;
            case 2438:
                // RightTriangleBar;
                if (';' == current) {
                    match = "\u29D0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2439:
                if ('q' == current) {
                    state = 2440;
                    return true;
                }
                break;
            case 2440:
                if ('u' == current) {
                    state = 2441;
                    return true;
                }
                break;
            case 2441:
                if ('a' == current) {
                    state = 2442;
                    return true;
                }
                break;
            case 2442:
                if ('l' == current) {
                    state = 2443;
                    return true;
                }
                break;
            case 2443:
                // RightTriangleEqual;
                if (';' == current) {
                    match = "\u22B5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2444:
                if ('p' == current) {
                    state = 2445;
                    return true;
                }
                break;
            case 2445:
                switch (current) {
                    case 'D' :
                        state = 2446;
                        return true;
                    case 'T' :
                        state = 2456;
                        return true;
                    case 'V' :
                        state = 2465;
                        return true;
                }
                break;
            case 2446:
                if ('o' == current) {
                    state = 2447;
                    return true;
                }
                break;
            case 2447:
                if ('w' == current) {
                    state = 2448;
                    return true;
                }
                break;
            case 2448:
                if ('n' == current) {
                    state = 2449;
                    return true;
                }
                break;
            case 2449:
                if ('V' == current) {
                    state = 2450;
                    return true;
                }
                break;
            case 2450:
                if ('e' == current) {
                    state = 2451;
                    return true;
                }
                break;
            case 2451:
                if ('c' == current) {
                    state = 2452;
                    return true;
                }
                break;
            case 2452:
                if ('t' == current) {
                    state = 2453;
                    return true;
                }
                break;
            case 2453:
                if ('o' == current) {
                    state = 2454;
                    return true;
                }
                break;
            case 2454:
                if ('r' == current) {
                    state = 2455;
                    return true;
                }
                break;
            case 2455:
                // RightUpDownVector;
                if (';' == current) {
                    match = "\u294F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2456:
                if ('e' == current) {
                    state = 2457;
                    return true;
                }
                break;
            case 2457:
                if ('e' == current) {
                    state = 2458;
                    return true;
                }
                break;
            case 2458:
                if ('V' == current) {
                    state = 2459;
                    return true;
                }
                break;
            case 2459:
                if ('e' == current) {
                    state = 2460;
                    return true;
                }
                break;
            case 2460:
                if ('c' == current) {
                    state = 2461;
                    return true;
                }
                break;
            case 2461:
                if ('t' == current) {
                    state = 2462;
                    return true;
                }
                break;
            case 2462:
                if ('o' == current) {
                    state = 2463;
                    return true;
                }
                break;
            case 2463:
                if ('r' == current) {
                    state = 2464;
                    return true;
                }
                break;
            case 2464:
                // RightUpTeeVector;
                if (';' == current) {
                    match = "\u295C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2465:
                if ('e' == current) {
                    state = 2466;
                    return true;
                }
                break;
            case 2466:
                if ('c' == current) {
                    state = 2467;
                    return true;
                }
                break;
            case 2467:
                if ('t' == current) {
                    state = 2468;
                    return true;
                }
                break;
            case 2468:
                if ('o' == current) {
                    state = 2469;
                    return true;
                }
                break;
            case 2469:
                if ('r' == current) {
                    state = 2470;
                    return true;
                }
                break;
            case 2470:
                // RightUpVector;
                if (';' == current) {
                    match = "\u21BE";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('B' == current) {
                    state = 2471;
                    return true;
                }
                break;
            case 2471:
                if ('a' == current) {
                    state = 2472;
                    return true;
                }
                break;
            case 2472:
                if ('r' == current) {
                    state = 2473;
                    return true;
                }
                break;
            case 2473:
                // RightUpVectorBar;
                if (';' == current) {
                    match = "\u2954";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2474:
                if ('e' == current) {
                    state = 2475;
                    return true;
                }
                break;
            case 2475:
                if ('c' == current) {
                    state = 2476;
                    return true;
                }
                break;
            case 2476:
                if ('t' == current) {
                    state = 2477;
                    return true;
                }
                break;
            case 2477:
                if ('o' == current) {
                    state = 2478;
                    return true;
                }
                break;
            case 2478:
                if ('r' == current) {
                    state = 2479;
                    return true;
                }
                break;
            case 2479:
                // RightVector;
                if (';' == current) {
                    match = "\u21C0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('B' == current) {
                    state = 2480;
                    return true;
                }
                break;
            case 2480:
                if ('a' == current) {
                    state = 2481;
                    return true;
                }
                break;
            case 2481:
                if ('r' == current) {
                    state = 2482;
                    return true;
                }
                break;
            case 2482:
                // RightVectorBar;
                if (';' == current) {
                    match = "\u2953";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2483:
                if ('r' == current) {
                    state = 2484;
                    return true;
                }
                break;
            case 2484:
                if ('r' == current) {
                    state = 2485;
                    return true;
                }
                break;
            case 2485:
                if ('o' == current) {
                    state = 2486;
                    return true;
                }
                break;
            case 2486:
                if ('w' == current) {
                    state = 2487;
                    return true;
                }
                break;
            case 2487:
                // Rightarrow;
                if (';' == current) {
                    match = "\u21D2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2488:
                if ('p' == current) {
                    state = 2489;
                    return true;
                }
                else if ('u' == current) {
                    state = 2491;
                    return true;
                }
                break;
            case 2489:
                if ('f' == current) {
                    state = 2490;
                    return true;
                }
                break;
            case 2490:
                // Ropf;
                if (';' == current) {
                    match = "\u211D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2491:
                if ('n' == current) {
                    state = 2492;
                    return true;
                }
                break;
            case 2492:
                if ('d' == current) {
                    state = 2493;
                    return true;
                }
                break;
            case 2493:
                if ('I' == current) {
                    state = 2494;
                    return true;
                }
                break;
            case 2494:
                if ('m' == current) {
                    state = 2495;
                    return true;
                }
                break;
            case 2495:
                if ('p' == current) {
                    state = 2496;
                    return true;
                }
                break;
            case 2496:
                if ('l' == current) {
                    state = 2497;
                    return true;
                }
                break;
            case 2497:
                if ('i' == current) {
                    state = 2498;
                    return true;
                }
                break;
            case 2498:
                if ('e' == current) {
                    state = 2499;
                    return true;
                }
                break;
            case 2499:
                if ('s' == current) {
                    state = 2500;
                    return true;
                }
                break;
            case 2500:
                // RoundImplies;
                if (';' == current) {
                    match = "\u2970";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2501:
                if ('i' == current) {
                    state = 2502;
                    return true;
                }
                break;
            case 2502:
                if ('g' == current) {
                    state = 2503;
                    return true;
                }
                break;
            case 2503:
                if ('h' == current) {
                    state = 2504;
                    return true;
                }
                break;
            case 2504:
                if ('t' == current) {
                    state = 2505;
                    return true;
                }
                break;
            case 2505:
                if ('a' == current) {
                    state = 2506;
                    return true;
                }
                break;
            case 2506:
                if ('r' == current) {
                    state = 2507;
                    return true;
                }
                break;
            case 2507:
                if ('r' == current) {
                    state = 2508;
                    return true;
                }
                break;
            case 2508:
                if ('o' == current) {
                    state = 2509;
                    return true;
                }
                break;
            case 2509:
                if ('w' == current) {
                    state = 2510;
                    return true;
                }
                break;
            case 2510:
                // Rrightarrow;
                if (';' == current) {
                    match = "\u21DB";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2511:
                if ('c' == current) {
                    state = 2512;
                    return true;
                }
                else if ('h' == current) {
                    state = 2514;
                    return true;
                }
                break;
            case 2512:
                if ('r' == current) {
                    state = 2513;
                    return true;
                }
                break;
            case 2513:
                // Rscr;
                if (';' == current) {
                    match = "\u211B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2514:
                // Rsh;
                if (';' == current) {
                    match = "\u21B1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2515:
                if ('l' == current) {
                    state = 2516;
                    return true;
                }
                break;
            case 2516:
                if ('e' == current) {
                    state = 2517;
                    return true;
                }
                break;
            case 2517:
                if ('D' == current) {
                    state = 2518;
                    return true;
                }
                break;
            case 2518:
                if ('e' == current) {
                    state = 2519;
                    return true;
                }
                break;
            case 2519:
                if ('l' == current) {
                    state = 2520;
                    return true;
                }
                break;
            case 2520:
                if ('a' == current) {
                    state = 2521;
                    return true;
                }
                break;
            case 2521:
                if ('y' == current) {
                    state = 2522;
                    return true;
                }
                break;
            case 2522:
                if ('e' == current) {
                    state = 2523;
                    return true;
                }
                break;
            case 2523:
                if ('d' == current) {
                    state = 2524;
                    return true;
                }
                break;
            case 2524:
                // RuleDelayed;
                if (';' == current) {
                    match = "\u29F4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2525:
                switch (current) {
                    case 'H' :
                        state = 2526;
                        return true;
                    case 'O' :
                        state = 2533;
                        return true;
                    case 'a' :
                        state = 2538;
                        return true;
                    case 'c' :
                        state = 2543;
                        return true;
                    case 'f' :
                        state = 2556;
                        return true;
                    case 'h' :
                        state = 2558;
                        return true;
                    case 'i' :
                        state = 2597;
                        return true;
                    case 'm' :
                        state = 2601;
                        return true;
                    case 'o' :
                        state = 2611;
                        return true;
                    case 'q' :
                        state = 2614;
                        return true;
                    case 's' :
                        state = 2660;
                        return true;
                    case 't' :
                        state = 2663;
                        return true;
                    case 'u' :
                        state = 2666;
                        return true;
                }
                break;
            case 2526:
                if ('C' == current) {
                    state = 2527;
                    return true;
                }
                else if ('c' == current) {
                    state = 2531;
                    return true;
                }
                break;
            case 2527:
                if ('H' == current) {
                    state = 2528;
                    return true;
                }
                break;
            case 2528:
                if ('c' == current) {
                    state = 2529;
                    return true;
                }
                break;
            case 2529:
                if ('y' == current) {
                    state = 2530;
                    return true;
                }
                break;
            case 2530:
                // SHCHcy;
                if (';' == current) {
                    match = "\u0429";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2531:
                if ('y' == current) {
                    state = 2532;
                    return true;
                }
                break;
            case 2532:
                // SHcy;
                if (';' == current) {
                    match = "\u0428";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2533:
                if ('F' == current) {
                    state = 2534;
                    return true;
                }
                break;
            case 2534:
                if ('T' == current) {
                    state = 2535;
                    return true;
                }
                break;
            case 2535:
                if ('c' == current) {
                    state = 2536;
                    return true;
                }
                break;
            case 2536:
                if ('y' == current) {
                    state = 2537;
                    return true;
                }
                break;
            case 2537:
                // SOFTcy;
                if (';' == current) {
                    match = "\u042C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2538:
                if ('c' == current) {
                    state = 2539;
                    return true;
                }
                break;
            case 2539:
                if ('u' == current) {
                    state = 2540;
                    return true;
                }
                break;
            case 2540:
                if ('t' == current) {
                    state = 2541;
                    return true;
                }
                break;
            case 2541:
                if ('e' == current) {
                    state = 2542;
                    return true;
                }
                break;
            case 2542:
                // Sacute;
                if (';' == current) {
                    match = "\u015A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2543:
                switch (current) {
                    case ';' : // Sc;
                        match = "\u2ABC";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'a' :
                        state = 2544;
                        return true;
                    case 'e' :
                        state = 2548;
                        return true;
                    case 'i' :
                        state = 2552;
                        return true;
                    case 'y' :
                        state = 2555;
                        return true;
                }
                break;
            case 2544:
                if ('r' == current) {
                    state = 2545;
                    return true;
                }
                break;
            case 2545:
                if ('o' == current) {
                    state = 2546;
                    return true;
                }
                break;
            case 2546:
                if ('n' == current) {
                    state = 2547;
                    return true;
                }
                break;
            case 2547:
                // Scaron;
                if (';' == current) {
                    match = "\u0160";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2548:
                if ('d' == current) {
                    state = 2549;
                    return true;
                }
                break;
            case 2549:
                if ('i' == current) {
                    state = 2550;
                    return true;
                }
                break;
            case 2550:
                if ('l' == current) {
                    state = 2551;
                    return true;
                }
                break;
            case 2551:
                // Scedil;
                if (';' == current) {
                    match = "\u015E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2552:
                if ('r' == current) {
                    state = 2553;
                    return true;
                }
                break;
            case 2553:
                if ('c' == current) {
                    state = 2554;
                    return true;
                }
                break;
            case 2554:
                // Scirc;
                if (';' == current) {
                    match = "\u015C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2555:
                // Scy;
                if (';' == current) {
                    match = "\u0421";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2556:
                if ('r' == current) {
                    state = 2557;
                    return true;
                }
                break;
            case 2557:
                // Sfr;
                if (';' == current) {
                    match = "\uD835\uDD16";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2558:
                if ('o' == current) {
                    state = 2559;
                    return true;
                }
                break;
            case 2559:
                if ('r' == current) {
                    state = 2560;
                    return true;
                }
                break;
            case 2560:
                if ('t' == current) {
                    state = 2561;
                    return true;
                }
                break;
            case 2561:
                switch (current) {
                    case 'D' :
                        state = 2562;
                        return true;
                    case 'L' :
                        state = 2571;
                        return true;
                    case 'R' :
                        state = 2580;
                        return true;
                    case 'U' :
                        state = 2590;
                        return true;
                }
                break;
            case 2562:
                if ('o' == current) {
                    state = 2563;
                    return true;
                }
                break;
            case 2563:
                if ('w' == current) {
                    state = 2564;
                    return true;
                }
                break;
            case 2564:
                if ('n' == current) {
                    state = 2565;
                    return true;
                }
                break;
            case 2565:
                if ('A' == current) {
                    state = 2566;
                    return true;
                }
                break;
            case 2566:
                if ('r' == current) {
                    state = 2567;
                    return true;
                }
                break;
            case 2567:
                if ('r' == current) {
                    state = 2568;
                    return true;
                }
                break;
            case 2568:
                if ('o' == current) {
                    state = 2569;
                    return true;
                }
                break;
            case 2569:
                if ('w' == current) {
                    state = 2570;
                    return true;
                }
                break;
            case 2570:
                // ShortDownArrow;
                if (';' == current) {
                    match = "\u2193";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2571:
                if ('e' == current) {
                    state = 2572;
                    return true;
                }
                break;
            case 2572:
                if ('f' == current) {
                    state = 2573;
                    return true;
                }
                break;
            case 2573:
                if ('t' == current) {
                    state = 2574;
                    return true;
                }
                break;
            case 2574:
                if ('A' == current) {
                    state = 2575;
                    return true;
                }
                break;
            case 2575:
                if ('r' == current) {
                    state = 2576;
                    return true;
                }
                break;
            case 2576:
                if ('r' == current) {
                    state = 2577;
                    return true;
                }
                break;
            case 2577:
                if ('o' == current) {
                    state = 2578;
                    return true;
                }
                break;
            case 2578:
                if ('w' == current) {
                    state = 2579;
                    return true;
                }
                break;
            case 2579:
                // ShortLeftArrow;
                if (';' == current) {
                    match = "\u2190";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2580:
                if ('i' == current) {
                    state = 2581;
                    return true;
                }
                break;
            case 2581:
                if ('g' == current) {
                    state = 2582;
                    return true;
                }
                break;
            case 2582:
                if ('h' == current) {
                    state = 2583;
                    return true;
                }
                break;
            case 2583:
                if ('t' == current) {
                    state = 2584;
                    return true;
                }
                break;
            case 2584:
                if ('A' == current) {
                    state = 2585;
                    return true;
                }
                break;
            case 2585:
                if ('r' == current) {
                    state = 2586;
                    return true;
                }
                break;
            case 2586:
                if ('r' == current) {
                    state = 2587;
                    return true;
                }
                break;
            case 2587:
                if ('o' == current) {
                    state = 2588;
                    return true;
                }
                break;
            case 2588:
                if ('w' == current) {
                    state = 2589;
                    return true;
                }
                break;
            case 2589:
                // ShortRightArrow;
                if (';' == current) {
                    match = "\u2192";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2590:
                if ('p' == current) {
                    state = 2591;
                    return true;
                }
                break;
            case 2591:
                if ('A' == current) {
                    state = 2592;
                    return true;
                }
                break;
            case 2592:
                if ('r' == current) {
                    state = 2593;
                    return true;
                }
                break;
            case 2593:
                if ('r' == current) {
                    state = 2594;
                    return true;
                }
                break;
            case 2594:
                if ('o' == current) {
                    state = 2595;
                    return true;
                }
                break;
            case 2595:
                if ('w' == current) {
                    state = 2596;
                    return true;
                }
                break;
            case 2596:
                // ShortUpArrow;
                if (';' == current) {
                    match = "\u2191";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2597:
                if ('g' == current) {
                    state = 2598;
                    return true;
                }
                break;
            case 2598:
                if ('m' == current) {
                    state = 2599;
                    return true;
                }
                break;
            case 2599:
                if ('a' == current) {
                    state = 2600;
                    return true;
                }
                break;
            case 2600:
                // Sigma;
                if (';' == current) {
                    match = "\u03A3";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2601:
                if ('a' == current) {
                    state = 2602;
                    return true;
                }
                break;
            case 2602:
                if ('l' == current) {
                    state = 2603;
                    return true;
                }
                break;
            case 2603:
                if ('l' == current) {
                    state = 2604;
                    return true;
                }
                break;
            case 2604:
                if ('C' == current) {
                    state = 2605;
                    return true;
                }
                break;
            case 2605:
                if ('i' == current) {
                    state = 2606;
                    return true;
                }
                break;
            case 2606:
                if ('r' == current) {
                    state = 2607;
                    return true;
                }
                break;
            case 2607:
                if ('c' == current) {
                    state = 2608;
                    return true;
                }
                break;
            case 2608:
                if ('l' == current) {
                    state = 2609;
                    return true;
                }
                break;
            case 2609:
                if ('e' == current) {
                    state = 2610;
                    return true;
                }
                break;
            case 2610:
                // SmallCircle;
                if (';' == current) {
                    match = "\u2218";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2611:
                if ('p' == current) {
                    state = 2612;
                    return true;
                }
                break;
            case 2612:
                if ('f' == current) {
                    state = 2613;
                    return true;
                }
                break;
            case 2613:
                // Sopf;
                if (';' == current) {
                    match = "\uD835\uDD4A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2614:
                if ('r' == current) {
                    state = 2615;
                    return true;
                }
                else if ('u' == current) {
                    state = 2617;
                    return true;
                }
                break;
            case 2615:
                if ('t' == current) {
                    state = 2616;
                    return true;
                }
                break;
            case 2616:
                // Sqrt;
                if (';' == current) {
                    match = "\u221A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2617:
                if ('a' == current) {
                    state = 2618;
                    return true;
                }
                break;
            case 2618:
                if ('r' == current) {
                    state = 2619;
                    return true;
                }
                break;
            case 2619:
                if ('e' == current) {
                    state = 2620;
                    return true;
                }
                break;
            case 2620:
                switch (current) {
                    case ';' : // Square;
                        match = "\u25A1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'I' :
                        state = 2621;
                        return true;
                    case 'S' :
                        state = 2633;
                        return true;
                    case 'U' :
                        state = 2655;
                        return true;
                }
                break;
            case 2621:
                if ('n' == current) {
                    state = 2622;
                    return true;
                }
                break;
            case 2622:
                if ('t' == current) {
                    state = 2623;
                    return true;
                }
                break;
            case 2623:
                if ('e' == current) {
                    state = 2624;
                    return true;
                }
                break;
            case 2624:
                if ('r' == current) {
                    state = 2625;
                    return true;
                }
                break;
            case 2625:
                if ('s' == current) {
                    state = 2626;
                    return true;
                }
                break;
            case 2626:
                if ('e' == current) {
                    state = 2627;
                    return true;
                }
                break;
            case 2627:
                if ('c' == current) {
                    state = 2628;
                    return true;
                }
                break;
            case 2628:
                if ('t' == current) {
                    state = 2629;
                    return true;
                }
                break;
            case 2629:
                if ('i' == current) {
                    state = 2630;
                    return true;
                }
                break;
            case 2630:
                if ('o' == current) {
                    state = 2631;
                    return true;
                }
                break;
            case 2631:
                if ('n' == current) {
                    state = 2632;
                    return true;
                }
                break;
            case 2632:
                // SquareIntersection;
                if (';' == current) {
                    match = "\u2293";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2633:
                if ('u' == current) {
                    state = 2634;
                    return true;
                }
                break;
            case 2634:
                if ('b' == current) {
                    state = 2635;
                    return true;
                }
                else if ('p' == current) {
                    state = 2644;
                    return true;
                }
                break;
            case 2635:
                if ('s' == current) {
                    state = 2636;
                    return true;
                }
                break;
            case 2636:
                if ('e' == current) {
                    state = 2637;
                    return true;
                }
                break;
            case 2637:
                if ('t' == current) {
                    state = 2638;
                    return true;
                }
                break;
            case 2638:
                // SquareSubset;
                if (';' == current) {
                    match = "\u228F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('E' == current) {
                    state = 2639;
                    return true;
                }
                break;
            case 2639:
                if ('q' == current) {
                    state = 2640;
                    return true;
                }
                break;
            case 2640:
                if ('u' == current) {
                    state = 2641;
                    return true;
                }
                break;
            case 2641:
                if ('a' == current) {
                    state = 2642;
                    return true;
                }
                break;
            case 2642:
                if ('l' == current) {
                    state = 2643;
                    return true;
                }
                break;
            case 2643:
                // SquareSubsetEqual;
                if (';' == current) {
                    match = "\u2291";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2644:
                if ('e' == current) {
                    state = 2645;
                    return true;
                }
                break;
            case 2645:
                if ('r' == current) {
                    state = 2646;
                    return true;
                }
                break;
            case 2646:
                if ('s' == current) {
                    state = 2647;
                    return true;
                }
                break;
            case 2647:
                if ('e' == current) {
                    state = 2648;
                    return true;
                }
                break;
            case 2648:
                if ('t' == current) {
                    state = 2649;
                    return true;
                }
                break;
            case 2649:
                // SquareSuperset;
                if (';' == current) {
                    match = "\u2290";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('E' == current) {
                    state = 2650;
                    return true;
                }
                break;
            case 2650:
                if ('q' == current) {
                    state = 2651;
                    return true;
                }
                break;
            case 2651:
                if ('u' == current) {
                    state = 2652;
                    return true;
                }
                break;
            case 2652:
                if ('a' == current) {
                    state = 2653;
                    return true;
                }
                break;
            case 2653:
                if ('l' == current) {
                    state = 2654;
                    return true;
                }
                break;
            case 2654:
                // SquareSupersetEqual;
                if (';' == current) {
                    match = "\u2292";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2655:
                if ('n' == current) {
                    state = 2656;
                    return true;
                }
                break;
            case 2656:
                if ('i' == current) {
                    state = 2657;
                    return true;
                }
                break;
            case 2657:
                if ('o' == current) {
                    state = 2658;
                    return true;
                }
                break;
            case 2658:
                if ('n' == current) {
                    state = 2659;
                    return true;
                }
                break;
            case 2659:
                // SquareUnion;
                if (';' == current) {
                    match = "\u2294";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2660:
                if ('c' == current) {
                    state = 2661;
                    return true;
                }
                break;
            case 2661:
                if ('r' == current) {
                    state = 2662;
                    return true;
                }
                break;
            case 2662:
                // Sscr;
                if (';' == current) {
                    match = "\uD835\uDCAE";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2663:
                if ('a' == current) {
                    state = 2664;
                    return true;
                }
                break;
            case 2664:
                if ('r' == current) {
                    state = 2665;
                    return true;
                }
                break;
            case 2665:
                // Star;
                if (';' == current) {
                    match = "\u22C6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2666:
                switch (current) {
                    case 'b' :
                        state = 2667;
                        return true;
                    case 'c' :
                        state = 2676;
                        return true;
                    case 'm' :
                        state = 2707;
                        return true;
                    case 'p' :
                        state = 2708;
                        return true;
                }
                break;
            case 2667:
                // Sub;
                if (';' == current) {
                    match = "\u22D0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('s' == current) {
                    state = 2668;
                    return true;
                }
                break;
            case 2668:
                if ('e' == current) {
                    state = 2669;
                    return true;
                }
                break;
            case 2669:
                if ('t' == current) {
                    state = 2670;
                    return true;
                }
                break;
            case 2670:
                // Subset;
                if (';' == current) {
                    match = "\u22D0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('E' == current) {
                    state = 2671;
                    return true;
                }
                break;
            case 2671:
                if ('q' == current) {
                    state = 2672;
                    return true;
                }
                break;
            case 2672:
                if ('u' == current) {
                    state = 2673;
                    return true;
                }
                break;
            case 2673:
                if ('a' == current) {
                    state = 2674;
                    return true;
                }
                break;
            case 2674:
                if ('l' == current) {
                    state = 2675;
                    return true;
                }
                break;
            case 2675:
                // SubsetEqual;
                if (';' == current) {
                    match = "\u2286";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2676:
                if ('c' == current) {
                    state = 2677;
                    return true;
                }
                else if ('h' == current) {
                    state = 2702;
                    return true;
                }
                break;
            case 2677:
                if ('e' == current) {
                    state = 2678;
                    return true;
                }
                break;
            case 2678:
                if ('e' == current) {
                    state = 2679;
                    return true;
                }
                break;
            case 2679:
                if ('d' == current) {
                    state = 2680;
                    return true;
                }
                break;
            case 2680:
                if ('s' == current) {
                    state = 2681;
                    return true;
                }
                break;
            case 2681:
                switch (current) {
                    case ';' : // Succeeds;
                        match = "\u227B";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 2682;
                        return true;
                    case 'S' :
                        state = 2687;
                        return true;
                    case 'T' :
                        state = 2697;
                        return true;
                }
                break;
            case 2682:
                if ('q' == current) {
                    state = 2683;
                    return true;
                }
                break;
            case 2683:
                if ('u' == current) {
                    state = 2684;
                    return true;
                }
                break;
            case 2684:
                if ('a' == current) {
                    state = 2685;
                    return true;
                }
                break;
            case 2685:
                if ('l' == current) {
                    state = 2686;
                    return true;
                }
                break;
            case 2686:
                // SucceedsEqual;
                if (';' == current) {
                    match = "\u2AB0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2687:
                if ('l' == current) {
                    state = 2688;
                    return true;
                }
                break;
            case 2688:
                if ('a' == current) {
                    state = 2689;
                    return true;
                }
                break;
            case 2689:
                if ('n' == current) {
                    state = 2690;
                    return true;
                }
                break;
            case 2690:
                if ('t' == current) {
                    state = 2691;
                    return true;
                }
                break;
            case 2691:
                if ('E' == current) {
                    state = 2692;
                    return true;
                }
                break;
            case 2692:
                if ('q' == current) {
                    state = 2693;
                    return true;
                }
                break;
            case 2693:
                if ('u' == current) {
                    state = 2694;
                    return true;
                }
                break;
            case 2694:
                if ('a' == current) {
                    state = 2695;
                    return true;
                }
                break;
            case 2695:
                if ('l' == current) {
                    state = 2696;
                    return true;
                }
                break;
            case 2696:
                // SucceedsSlantEqual;
                if (';' == current) {
                    match = "\u227D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2697:
                if ('i' == current) {
                    state = 2698;
                    return true;
                }
                break;
            case 2698:
                if ('l' == current) {
                    state = 2699;
                    return true;
                }
                break;
            case 2699:
                if ('d' == current) {
                    state = 2700;
                    return true;
                }
                break;
            case 2700:
                if ('e' == current) {
                    state = 2701;
                    return true;
                }
                break;
            case 2701:
                // SucceedsTilde;
                if (';' == current) {
                    match = "\u227F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2702:
                if ('T' == current) {
                    state = 2703;
                    return true;
                }
                break;
            case 2703:
                if ('h' == current) {
                    state = 2704;
                    return true;
                }
                break;
            case 2704:
                if ('a' == current) {
                    state = 2705;
                    return true;
                }
                break;
            case 2705:
                if ('t' == current) {
                    state = 2706;
                    return true;
                }
                break;
            case 2706:
                // SuchThat;
                if (';' == current) {
                    match = "\u220B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2707:
                // Sum;
                if (';' == current) {
                    match = "\u2211";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2708:
                switch (current) {
                    case ';' : // Sup;
                        match = "\u22D1";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 2709;
                        return true;
                    case 's' :
                        state = 2719;
                        return true;
                }
                break;
            case 2709:
                if ('r' == current) {
                    state = 2710;
                    return true;
                }
                break;
            case 2710:
                if ('s' == current) {
                    state = 2711;
                    return true;
                }
                break;
            case 2711:
                if ('e' == current) {
                    state = 2712;
                    return true;
                }
                break;
            case 2712:
                if ('t' == current) {
                    state = 2713;
                    return true;
                }
                break;
            case 2713:
                // Superset;
                if (';' == current) {
                    match = "\u2283";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('E' == current) {
                    state = 2714;
                    return true;
                }
                break;
            case 2714:
                if ('q' == current) {
                    state = 2715;
                    return true;
                }
                break;
            case 2715:
                if ('u' == current) {
                    state = 2716;
                    return true;
                }
                break;
            case 2716:
                if ('a' == current) {
                    state = 2717;
                    return true;
                }
                break;
            case 2717:
                if ('l' == current) {
                    state = 2718;
                    return true;
                }
                break;
            case 2718:
                // SupersetEqual;
                if (';' == current) {
                    match = "\u2287";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2719:
                if ('e' == current) {
                    state = 2720;
                    return true;
                }
                break;
            case 2720:
                if ('t' == current) {
                    state = 2721;
                    return true;
                }
                break;
            case 2721:
                // Supset;
                if (';' == current) {
                    match = "\u22D1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2722:
                switch (current) {
                    case 'H' :
                        state = 2723;
                        return true;
                    case 'R' :
                        state = 2727;
                        return true;
                    case 'S' :
                        state = 2731;
                        return true;
                    case 'a' :
                        state = 2737;
                        return true;
                    case 'c' :
                        state = 2740;
                        return true;
                    case 'f' :
                        state = 2750;
                        return true;
                    case 'h' :
                        state = 2752;
                        return true;
                    case 'i' :
                        state = 2776;
                        return true;
                    case 'o' :
                        state = 2799;
                        return true;
                    case 'r' :
                        state = 2802;
                        return true;
                    case 's' :
                        state = 2810;
                        return true;
                }
                break;
            case 2723:
                if ('O' == current) {
                    state = 2724;
                    return true;
                }
                break;
            case 2724:
                if ('R' == current) {
                    state = 2725;
                    return true;
                }
                break;
            case 2725:
                // THORN
                if ('N' == current) {
                    match = "\u00DE";
                    matchLength = consumedCount;
                    state = 2726;
                    return true;
                }
                break;
            case 2726:
                // THORN;
                if (';' == current) {
                    match = "\u00DE";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2727:
                if ('A' == current) {
                    state = 2728;
                    return true;
                }
                break;
            case 2728:
                if ('D' == current) {
                    state = 2729;
                    return true;
                }
                break;
            case 2729:
                if ('E' == current) {
                    state = 2730;
                    return true;
                }
                break;
            case 2730:
                // TRADE;
                if (';' == current) {
                    match = "\u2122";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2731:
                if ('H' == current) {
                    state = 2732;
                    return true;
                }
                else if ('c' == current) {
                    state = 2735;
                    return true;
                }
                break;
            case 2732:
                if ('c' == current) {
                    state = 2733;
                    return true;
                }
                break;
            case 2733:
                if ('y' == current) {
                    state = 2734;
                    return true;
                }
                break;
            case 2734:
                // TSHcy;
                if (';' == current) {
                    match = "\u040B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2735:
                if ('y' == current) {
                    state = 2736;
                    return true;
                }
                break;
            case 2736:
                // TScy;
                if (';' == current) {
                    match = "\u0426";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2737:
                if ('b' == current) {
                    state = 2738;
                    return true;
                }
                else if ('u' == current) {
                    state = 2739;
                    return true;
                }
                break;
            case 2738:
                // Tab;
                if (';' == current) {
                    match = "\u0009";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2739:
                // Tau;
                if (';' == current) {
                    match = "\u03A4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2740:
                switch (current) {
                    case 'a' :
                        state = 2741;
                        return true;
                    case 'e' :
                        state = 2745;
                        return true;
                    case 'y' :
                        state = 2749;
                        return true;
                }
                break;
            case 2741:
                if ('r' == current) {
                    state = 2742;
                    return true;
                }
                break;
            case 2742:
                if ('o' == current) {
                    state = 2743;
                    return true;
                }
                break;
            case 2743:
                if ('n' == current) {
                    state = 2744;
                    return true;
                }
                break;
            case 2744:
                // Tcaron;
                if (';' == current) {
                    match = "\u0164";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2745:
                if ('d' == current) {
                    state = 2746;
                    return true;
                }
                break;
            case 2746:
                if ('i' == current) {
                    state = 2747;
                    return true;
                }
                break;
            case 2747:
                if ('l' == current) {
                    state = 2748;
                    return true;
                }
                break;
            case 2748:
                // Tcedil;
                if (';' == current) {
                    match = "\u0162";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2749:
                // Tcy;
                if (';' == current) {
                    match = "\u0422";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2750:
                if ('r' == current) {
                    state = 2751;
                    return true;
                }
                break;
            case 2751:
                // Tfr;
                if (';' == current) {
                    match = "\uD835\uDD17";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2752:
                if ('e' == current) {
                    state = 2753;
                    return true;
                }
                else if ('i' == current) {
                    state = 2762;
                    return true;
                }
                break;
            case 2753:
                if ('r' == current) {
                    state = 2754;
                    return true;
                }
                else if ('t' == current) {
                    state = 2760;
                    return true;
                }
                break;
            case 2754:
                if ('e' == current) {
                    state = 2755;
                    return true;
                }
                break;
            case 2755:
                if ('f' == current) {
                    state = 2756;
                    return true;
                }
                break;
            case 2756:
                if ('o' == current) {
                    state = 2757;
                    return true;
                }
                break;
            case 2757:
                if ('r' == current) {
                    state = 2758;
                    return true;
                }
                break;
            case 2758:
                if ('e' == current) {
                    state = 2759;
                    return true;
                }
                break;
            case 2759:
                // Therefore;
                if (';' == current) {
                    match = "\u2234";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2760:
                if ('a' == current) {
                    state = 2761;
                    return true;
                }
                break;
            case 2761:
                // Theta;
                if (';' == current) {
                    match = "\u0398";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2762:
                if ('c' == current) {
                    state = 2763;
                    return true;
                }
                else if ('n' == current) {
                    state = 2770;
                    return true;
                }
                break;
            case 2763:
                if ('k' == current) {
                    state = 2764;
                    return true;
                }
                break;
            case 2764:
                if ('S' == current) {
                    state = 2765;
                    return true;
                }
                break;
            case 2765:
                if ('p' == current) {
                    state = 2766;
                    return true;
                }
                break;
            case 2766:
                if ('a' == current) {
                    state = 2767;
                    return true;
                }
                break;
            case 2767:
                if ('c' == current) {
                    state = 2768;
                    return true;
                }
                break;
            case 2768:
                if ('e' == current) {
                    state = 2769;
                    return true;
                }
                break;
            case 2769:
                // ThickSpace;
                if (';' == current) {
                    match = "\u205F\u200A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2770:
                if ('S' == current) {
                    state = 2771;
                    return true;
                }
                break;
            case 2771:
                if ('p' == current) {
                    state = 2772;
                    return true;
                }
                break;
            case 2772:
                if ('a' == current) {
                    state = 2773;
                    return true;
                }
                break;
            case 2773:
                if ('c' == current) {
                    state = 2774;
                    return true;
                }
                break;
            case 2774:
                if ('e' == current) {
                    state = 2775;
                    return true;
                }
                break;
            case 2775:
                // ThinSpace;
                if (';' == current) {
                    match = "\u2009";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2776:
                if ('l' == current) {
                    state = 2777;
                    return true;
                }
                break;
            case 2777:
                if ('d' == current) {
                    state = 2778;
                    return true;
                }
                break;
            case 2778:
                if ('e' == current) {
                    state = 2779;
                    return true;
                }
                break;
            case 2779:
                switch (current) {
                    case ';' : // Tilde;
                        match = "\u223C";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 2780;
                        return true;
                    case 'F' :
                        state = 2785;
                        return true;
                    case 'T' :
                        state = 2794;
                        return true;
                }
                break;
            case 2780:
                if ('q' == current) {
                    state = 2781;
                    return true;
                }
                break;
            case 2781:
                if ('u' == current) {
                    state = 2782;
                    return true;
                }
                break;
            case 2782:
                if ('a' == current) {
                    state = 2783;
                    return true;
                }
                break;
            case 2783:
                if ('l' == current) {
                    state = 2784;
                    return true;
                }
                break;
            case 2784:
                // TildeEqual;
                if (';' == current) {
                    match = "\u2243";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2785:
                if ('u' == current) {
                    state = 2786;
                    return true;
                }
                break;
            case 2786:
                if ('l' == current) {
                    state = 2787;
                    return true;
                }
                break;
            case 2787:
                if ('l' == current) {
                    state = 2788;
                    return true;
                }
                break;
            case 2788:
                if ('E' == current) {
                    state = 2789;
                    return true;
                }
                break;
            case 2789:
                if ('q' == current) {
                    state = 2790;
                    return true;
                }
                break;
            case 2790:
                if ('u' == current) {
                    state = 2791;
                    return true;
                }
                break;
            case 2791:
                if ('a' == current) {
                    state = 2792;
                    return true;
                }
                break;
            case 2792:
                if ('l' == current) {
                    state = 2793;
                    return true;
                }
                break;
            case 2793:
                // TildeFullEqual;
                if (';' == current) {
                    match = "\u2245";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2794:
                if ('i' == current) {
                    state = 2795;
                    return true;
                }
                break;
            case 2795:
                if ('l' == current) {
                    state = 2796;
                    return true;
                }
                break;
            case 2796:
                if ('d' == current) {
                    state = 2797;
                    return true;
                }
                break;
            case 2797:
                if ('e' == current) {
                    state = 2798;
                    return true;
                }
                break;
            case 2798:
                // TildeTilde;
                if (';' == current) {
                    match = "\u2248";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2799:
                if ('p' == current) {
                    state = 2800;
                    return true;
                }
                break;
            case 2800:
                if ('f' == current) {
                    state = 2801;
                    return true;
                }
                break;
            case 2801:
                // Topf;
                if (';' == current) {
                    match = "\uD835\uDD4B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2802:
                if ('i' == current) {
                    state = 2803;
                    return true;
                }
                break;
            case 2803:
                if ('p' == current) {
                    state = 2804;
                    return true;
                }
                break;
            case 2804:
                if ('l' == current) {
                    state = 2805;
                    return true;
                }
                break;
            case 2805:
                if ('e' == current) {
                    state = 2806;
                    return true;
                }
                break;
            case 2806:
                if ('D' == current) {
                    state = 2807;
                    return true;
                }
                break;
            case 2807:
                if ('o' == current) {
                    state = 2808;
                    return true;
                }
                break;
            case 2808:
                if ('t' == current) {
                    state = 2809;
                    return true;
                }
                break;
            case 2809:
                // TripleDot;
                if (';' == current) {
                    match = "\u20DB";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2810:
                if ('c' == current) {
                    state = 2811;
                    return true;
                }
                else if ('t' == current) {
                    state = 2813;
                    return true;
                }
                break;
            case 2811:
                if ('r' == current) {
                    state = 2812;
                    return true;
                }
                break;
            case 2812:
                // Tscr;
                if (';' == current) {
                    match = "\uD835\uDCAF";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2813:
                if ('r' == current) {
                    state = 2814;
                    return true;
                }
                break;
            case 2814:
                if ('o' == current) {
                    state = 2815;
                    return true;
                }
                break;
            case 2815:
                if ('k' == current) {
                    state = 2816;
                    return true;
                }
                break;
            case 2816:
                // Tstrok;
                if (';' == current) {
                    match = "\u0166";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2817:
                switch (current) {
                    case 'a' :
                        state = 2818;
                        return true;
                    case 'b' :
                        state = 2829;
                        return true;
                    case 'c' :
                        state = 2836;
                        return true;
                    case 'd' :
                        state = 2841;
                        return true;
                    case 'f' :
                        state = 2846;
                        return true;
                    case 'g' :
                        state = 2848;
                        return true;
                    case 'm' :
                        state = 2853;
                        return true;
                    case 'n' :
                        state = 2857;
                        return true;
                    case 'o' :
                        state = 2889;
                        return true;
                    case 'p' :
                        state = 2895;
                        return true;
                    case 'r' :
                        state = 2982;
                        return true;
                    case 's' :
                        state = 2986;
                        return true;
                    case 't' :
                        state = 2989;
                        return true;
                    case 'u' :
                        state = 2994;
                        return true;
                }
                break;
            case 2818:
                if ('c' == current) {
                    state = 2819;
                    return true;
                }
                else if ('r' == current) {
                    state = 2823;
                    return true;
                }
                break;
            case 2819:
                if ('u' == current) {
                    state = 2820;
                    return true;
                }
                break;
            case 2820:
                if ('t' == current) {
                    state = 2821;
                    return true;
                }
                break;
            case 2821:
                // Uacute
                if ('e' == current) {
                    match = "\u00DA";
                    matchLength = consumedCount;
                    state = 2822;
                    return true;
                }
                break;
            case 2822:
                // Uacute;
                if (';' == current) {
                    match = "\u00DA";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2823:
                if ('r' == current) {
                    state = 2824;
                    return true;
                }
                break;
            case 2824:
                // Uarr;
                if (';' == current) {
                    match = "\u219F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('o' == current) {
                    state = 2825;
                    return true;
                }
                break;
            case 2825:
                if ('c' == current) {
                    state = 2826;
                    return true;
                }
                break;
            case 2826:
                if ('i' == current) {
                    state = 2827;
                    return true;
                }
                break;
            case 2827:
                if ('r' == current) {
                    state = 2828;
                    return true;
                }
                break;
            case 2828:
                // Uarrocir;
                if (';' == current) {
                    match = "\u2949";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2829:
                if ('r' == current) {
                    state = 2830;
                    return true;
                }
                break;
            case 2830:
                if ('c' == current) {
                    state = 2831;
                    return true;
                }
                else if ('e' == current) {
                    state = 2833;
                    return true;
                }
                break;
            case 2831:
                if ('y' == current) {
                    state = 2832;
                    return true;
                }
                break;
            case 2832:
                // Ubrcy;
                if (';' == current) {
                    match = "\u040E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2833:
                if ('v' == current) {
                    state = 2834;
                    return true;
                }
                break;
            case 2834:
                if ('e' == current) {
                    state = 2835;
                    return true;
                }
                break;
            case 2835:
                // Ubreve;
                if (';' == current) {
                    match = "\u016C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2836:
                if ('i' == current) {
                    state = 2837;
                    return true;
                }
                else if ('y' == current) {
                    state = 2840;
                    return true;
                }
                break;
            case 2837:
                if ('r' == current) {
                    state = 2838;
                    return true;
                }
                break;
            case 2838:
                // Ucirc
                if ('c' == current) {
                    match = "\u00DB";
                    matchLength = consumedCount;
                    state = 2839;
                    return true;
                }
                break;
            case 2839:
                // Ucirc;
                if (';' == current) {
                    match = "\u00DB";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2840:
                // Ucy;
                if (';' == current) {
                    match = "\u0423";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2841:
                if ('b' == current) {
                    state = 2842;
                    return true;
                }
                break;
            case 2842:
                if ('l' == current) {
                    state = 2843;
                    return true;
                }
                break;
            case 2843:
                if ('a' == current) {
                    state = 2844;
                    return true;
                }
                break;
            case 2844:
                if ('c' == current) {
                    state = 2845;
                    return true;
                }
                break;
            case 2845:
                // Udblac;
                if (';' == current) {
                    match = "\u0170";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2846:
                if ('r' == current) {
                    state = 2847;
                    return true;
                }
                break;
            case 2847:
                // Ufr;
                if (';' == current) {
                    match = "\uD835\uDD18";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2848:
                if ('r' == current) {
                    state = 2849;
                    return true;
                }
                break;
            case 2849:
                if ('a' == current) {
                    state = 2850;
                    return true;
                }
                break;
            case 2850:
                if ('v' == current) {
                    state = 2851;
                    return true;
                }
                break;
            case 2851:
                // Ugrave
                if ('e' == current) {
                    match = "\u00D9";
                    matchLength = consumedCount;
                    state = 2852;
                    return true;
                }
                break;
            case 2852:
                // Ugrave;
                if (';' == current) {
                    match = "\u00D9";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2853:
                if ('a' == current) {
                    state = 2854;
                    return true;
                }
                break;
            case 2854:
                if ('c' == current) {
                    state = 2855;
                    return true;
                }
                break;
            case 2855:
                if ('r' == current) {
                    state = 2856;
                    return true;
                }
                break;
            case 2856:
                // Umacr;
                if (';' == current) {
                    match = "\u016A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2857:
                if ('d' == current) {
                    state = 2858;
                    return true;
                }
                else if ('i' == current) {
                    state = 2882;
                    return true;
                }
                break;
            case 2858:
                if ('e' == current) {
                    state = 2859;
                    return true;
                }
                break;
            case 2859:
                if ('r' == current) {
                    state = 2860;
                    return true;
                }
                break;
            case 2860:
                if ('B' == current) {
                    state = 2861;
                    return true;
                }
                else if ('P' == current) {
                    state = 2871;
                    return true;
                }
                break;
            case 2861:
                if ('a' == current) {
                    state = 2862;
                    return true;
                }
                else if ('r' == current) {
                    state = 2864;
                    return true;
                }
                break;
            case 2862:
                if ('r' == current) {
                    state = 2863;
                    return true;
                }
                break;
            case 2863:
                // UnderBar;
                if (';' == current) {
                    match = "_";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2864:
                if ('a' == current) {
                    state = 2865;
                    return true;
                }
                break;
            case 2865:
                if ('c' == current) {
                    state = 2866;
                    return true;
                }
                break;
            case 2866:
                if ('e' == current) {
                    state = 2867;
                    return true;
                }
                else if ('k' == current) {
                    state = 2868;
                    return true;
                }
                break;
            case 2867:
                // UnderBrace;
                if (';' == current) {
                    match = "\u23DF";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2868:
                if ('e' == current) {
                    state = 2869;
                    return true;
                }
                break;
            case 2869:
                if ('t' == current) {
                    state = 2870;
                    return true;
                }
                break;
            case 2870:
                // UnderBracket;
                if (';' == current) {
                    match = "\u23B5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2871:
                if ('a' == current) {
                    state = 2872;
                    return true;
                }
                break;
            case 2872:
                if ('r' == current) {
                    state = 2873;
                    return true;
                }
                break;
            case 2873:
                if ('e' == current) {
                    state = 2874;
                    return true;
                }
                break;
            case 2874:
                if ('n' == current) {
                    state = 2875;
                    return true;
                }
                break;
            case 2875:
                if ('t' == current) {
                    state = 2876;
                    return true;
                }
                break;
            case 2876:
                if ('h' == current) {
                    state = 2877;
                    return true;
                }
                break;
            case 2877:
                if ('e' == current) {
                    state = 2878;
                    return true;
                }
                break;
            case 2878:
                if ('s' == current) {
                    state = 2879;
                    return true;
                }
                break;
            case 2879:
                if ('i' == current) {
                    state = 2880;
                    return true;
                }
                break;
            case 2880:
                if ('s' == current) {
                    state = 2881;
                    return true;
                }
                break;
            case 2881:
                // UnderParenthesis;
                if (';' == current) {
                    match = "\u23DD";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2882:
                if ('o' == current) {
                    state = 2883;
                    return true;
                }
                break;
            case 2883:
                if ('n' == current) {
                    state = 2884;
                    return true;
                }
                break;
            case 2884:
                // Union;
                if (';' == current) {
                    match = "\u22C3";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('P' == current) {
                    state = 2885;
                    return true;
                }
                break;
            case 2885:
                if ('l' == current) {
                    state = 2886;
                    return true;
                }
                break;
            case 2886:
                if ('u' == current) {
                    state = 2887;
                    return true;
                }
                break;
            case 2887:
                if ('s' == current) {
                    state = 2888;
                    return true;
                }
                break;
            case 2888:
                // UnionPlus;
                if (';' == current) {
                    match = "\u228E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2889:
                if ('g' == current) {
                    state = 2890;
                    return true;
                }
                else if ('p' == current) {
                    state = 2893;
                    return true;
                }
                break;
            case 2890:
                if ('o' == current) {
                    state = 2891;
                    return true;
                }
                break;
            case 2891:
                if ('n' == current) {
                    state = 2892;
                    return true;
                }
                break;
            case 2892:
                // Uogon;
                if (';' == current) {
                    match = "\u0172";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2893:
                if ('f' == current) {
                    state = 2894;
                    return true;
                }
                break;
            case 2894:
                // Uopf;
                if (';' == current) {
                    match = "\uD835\uDD4C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2895:
                switch (current) {
                    case 'A' :
                        state = 2896;
                        return true;
                    case 'D' :
                        state = 2913;
                        return true;
                    case 'E' :
                        state = 2922;
                        return true;
                    case 'T' :
                        state = 2933;
                        return true;
                    case 'a' :
                        state = 2941;
                        return true;
                    case 'd' :
                        state = 2946;
                        return true;
                    case 'p' :
                        state = 2955;
                        return true;
                    case 's' :
                        state = 2977;
                        return true;
                }
                break;
            case 2896:
                if ('r' == current) {
                    state = 2897;
                    return true;
                }
                break;
            case 2897:
                if ('r' == current) {
                    state = 2898;
                    return true;
                }
                break;
            case 2898:
                if ('o' == current) {
                    state = 2899;
                    return true;
                }
                break;
            case 2899:
                if ('w' == current) {
                    state = 2900;
                    return true;
                }
                break;
            case 2900:
                switch (current) {
                    case ';' : // UpArrow;
                        match = "\u2191";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B' :
                        state = 2901;
                        return true;
                    case 'D' :
                        state = 2904;
                        return true;
                }
                break;
            case 2901:
                if ('a' == current) {
                    state = 2902;
                    return true;
                }
                break;
            case 2902:
                if ('r' == current) {
                    state = 2903;
                    return true;
                }
                break;
            case 2903:
                // UpArrowBar;
                if (';' == current) {
                    match = "\u2912";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2904:
                if ('o' == current) {
                    state = 2905;
                    return true;
                }
                break;
            case 2905:
                if ('w' == current) {
                    state = 2906;
                    return true;
                }
                break;
            case 2906:
                if ('n' == current) {
                    state = 2907;
                    return true;
                }
                break;
            case 2907:
                if ('A' == current) {
                    state = 2908;
                    return true;
                }
                break;
            case 2908:
                if ('r' == current) {
                    state = 2909;
                    return true;
                }
                break;
            case 2909:
                if ('r' == current) {
                    state = 2910;
                    return true;
                }
                break;
            case 2910:
                if ('o' == current) {
                    state = 2911;
                    return true;
                }
                break;
            case 2911:
                if ('w' == current) {
                    state = 2912;
                    return true;
                }
                break;
            case 2912:
                // UpArrowDownArrow;
                if (';' == current) {
                    match = "\u21C5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2913:
                if ('o' == current) {
                    state = 2914;
                    return true;
                }
                break;
            case 2914:
                if ('w' == current) {
                    state = 2915;
                    return true;
                }
                break;
            case 2915:
                if ('n' == current) {
                    state = 2916;
                    return true;
                }
                break;
            case 2916:
                if ('A' == current) {
                    state = 2917;
                    return true;
                }
                break;
            case 2917:
                if ('r' == current) {
                    state = 2918;
                    return true;
                }
                break;
            case 2918:
                if ('r' == current) {
                    state = 2919;
                    return true;
                }
                break;
            case 2919:
                if ('o' == current) {
                    state = 2920;
                    return true;
                }
                break;
            case 2920:
                if ('w' == current) {
                    state = 2921;
                    return true;
                }
                break;
            case 2921:
                // UpDownArrow;
                if (';' == current) {
                    match = "\u2195";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2922:
                if ('q' == current) {
                    state = 2923;
                    return true;
                }
                break;
            case 2923:
                if ('u' == current) {
                    state = 2924;
                    return true;
                }
                break;
            case 2924:
                if ('i' == current) {
                    state = 2925;
                    return true;
                }
                break;
            case 2925:
                if ('l' == current) {
                    state = 2926;
                    return true;
                }
                break;
            case 2926:
                if ('i' == current) {
                    state = 2927;
                    return true;
                }
                break;
            case 2927:
                if ('b' == current) {
                    state = 2928;
                    return true;
                }
                break;
            case 2928:
                if ('r' == current) {
                    state = 2929;
                    return true;
                }
                break;
            case 2929:
                if ('i' == current) {
                    state = 2930;
                    return true;
                }
                break;
            case 2930:
                if ('u' == current) {
                    state = 2931;
                    return true;
                }
                break;
            case 2931:
                if ('m' == current) {
                    state = 2932;
                    return true;
                }
                break;
            case 2932:
                // UpEquilibrium;
                if (';' == current) {
                    match = "\u296E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2933:
                if ('e' == current) {
                    state = 2934;
                    return true;
                }
                break;
            case 2934:
                if ('e' == current) {
                    state = 2935;
                    return true;
                }
                break;
            case 2935:
                // UpTee;
                if (';' == current) {
                    match = "\u22A5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('A' == current) {
                    state = 2936;
                    return true;
                }
                break;
            case 2936:
                if ('r' == current) {
                    state = 2937;
                    return true;
                }
                break;
            case 2937:
                if ('r' == current) {
                    state = 2938;
                    return true;
                }
                break;
            case 2938:
                if ('o' == current) {
                    state = 2939;
                    return true;
                }
                break;
            case 2939:
                if ('w' == current) {
                    state = 2940;
                    return true;
                }
                break;
            case 2940:
                // UpTeeArrow;
                if (';' == current) {
                    match = "\u21A5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2941:
                if ('r' == current) {
                    state = 2942;
                    return true;
                }
                break;
            case 2942:
                if ('r' == current) {
                    state = 2943;
                    return true;
                }
                break;
            case 2943:
                if ('o' == current) {
                    state = 2944;
                    return true;
                }
                break;
            case 2944:
                if ('w' == current) {
                    state = 2945;
                    return true;
                }
                break;
            case 2945:
                // Uparrow;
                if (';' == current) {
                    match = "\u21D1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2946:
                if ('o' == current) {
                    state = 2947;
                    return true;
                }
                break;
            case 2947:
                if ('w' == current) {
                    state = 2948;
                    return true;
                }
                break;
            case 2948:
                if ('n' == current) {
                    state = 2949;
                    return true;
                }
                break;
            case 2949:
                if ('a' == current) {
                    state = 2950;
                    return true;
                }
                break;
            case 2950:
                if ('r' == current) {
                    state = 2951;
                    return true;
                }
                break;
            case 2951:
                if ('r' == current) {
                    state = 2952;
                    return true;
                }
                break;
            case 2952:
                if ('o' == current) {
                    state = 2953;
                    return true;
                }
                break;
            case 2953:
                if ('w' == current) {
                    state = 2954;
                    return true;
                }
                break;
            case 2954:
                // Updownarrow;
                if (';' == current) {
                    match = "\u21D5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2955:
                if ('e' == current) {
                    state = 2956;
                    return true;
                }
                break;
            case 2956:
                if ('r' == current) {
                    state = 2957;
                    return true;
                }
                break;
            case 2957:
                if ('L' == current) {
                    state = 2958;
                    return true;
                }
                else if ('R' == current) {
                    state = 2967;
                    return true;
                }
                break;
            case 2958:
                if ('e' == current) {
                    state = 2959;
                    return true;
                }
                break;
            case 2959:
                if ('f' == current) {
                    state = 2960;
                    return true;
                }
                break;
            case 2960:
                if ('t' == current) {
                    state = 2961;
                    return true;
                }
                break;
            case 2961:
                if ('A' == current) {
                    state = 2962;
                    return true;
                }
                break;
            case 2962:
                if ('r' == current) {
                    state = 2963;
                    return true;
                }
                break;
            case 2963:
                if ('r' == current) {
                    state = 2964;
                    return true;
                }
                break;
            case 2964:
                if ('o' == current) {
                    state = 2965;
                    return true;
                }
                break;
            case 2965:
                if ('w' == current) {
                    state = 2966;
                    return true;
                }
                break;
            case 2966:
                // UpperLeftArrow;
                if (';' == current) {
                    match = "\u2196";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2967:
                if ('i' == current) {
                    state = 2968;
                    return true;
                }
                break;
            case 2968:
                if ('g' == current) {
                    state = 2969;
                    return true;
                }
                break;
            case 2969:
                if ('h' == current) {
                    state = 2970;
                    return true;
                }
                break;
            case 2970:
                if ('t' == current) {
                    state = 2971;
                    return true;
                }
                break;
            case 2971:
                if ('A' == current) {
                    state = 2972;
                    return true;
                }
                break;
            case 2972:
                if ('r' == current) {
                    state = 2973;
                    return true;
                }
                break;
            case 2973:
                if ('r' == current) {
                    state = 2974;
                    return true;
                }
                break;
            case 2974:
                if ('o' == current) {
                    state = 2975;
                    return true;
                }
                break;
            case 2975:
                if ('w' == current) {
                    state = 2976;
                    return true;
                }
                break;
            case 2976:
                // UpperRightArrow;
                if (';' == current) {
                    match = "\u2197";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2977:
                if ('i' == current) {
                    state = 2978;
                    return true;
                }
                break;
            case 2978:
                // Upsi;
                if (';' == current) {
                    match = "\u03D2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('l' == current) {
                    state = 2979;
                    return true;
                }
                break;
            case 2979:
                if ('o' == current) {
                    state = 2980;
                    return true;
                }
                break;
            case 2980:
                if ('n' == current) {
                    state = 2981;
                    return true;
                }
                break;
            case 2981:
                // Upsilon;
                if (';' == current) {
                    match = "\u03A5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2982:
                if ('i' == current) {
                    state = 2983;
                    return true;
                }
                break;
            case 2983:
                if ('n' == current) {
                    state = 2984;
                    return true;
                }
                break;
            case 2984:
                if ('g' == current) {
                    state = 2985;
                    return true;
                }
                break;
            case 2985:
                // Uring;
                if (';' == current) {
                    match = "\u016E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2986:
                if ('c' == current) {
                    state = 2987;
                    return true;
                }
                break;
            case 2987:
                if ('r' == current) {
                    state = 2988;
                    return true;
                }
                break;
            case 2988:
                // Uscr;
                if (';' == current) {
                    match = "\uD835\uDCB0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2989:
                if ('i' == current) {
                    state = 2990;
                    return true;
                }
                break;
            case 2990:
                if ('l' == current) {
                    state = 2991;
                    return true;
                }
                break;
            case 2991:
                if ('d' == current) {
                    state = 2992;
                    return true;
                }
                break;
            case 2992:
                if ('e' == current) {
                    state = 2993;
                    return true;
                }
                break;
            case 2993:
                // Utilde;
                if (';' == current) {
                    match = "\u0168";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2994:
                if ('m' == current) {
                    state = 2995;
                    return true;
                }
                break;
            case 2995:
                // Uuml
                if ('l' == current) {
                    match = "\u00DC";
                    matchLength = consumedCount;
                    state = 2996;
                    return true;
                }
                break;
            case 2996:
                // Uuml;
                if (';' == current) {
                    match = "\u00DC";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2997:
                switch (current) {
                    case 'D' :
                        state = 2998;
                        return true;
                    case 'b' :
                        state = 3002;
                        return true;
                    case 'c' :
                        state = 3005;
                        return true;
                    case 'd' :
                        state = 3007;
                        return true;
                    case 'e' :
                        state = 3012;
                        return true;
                    case 'f' :
                        state = 3054;
                        return true;
                    case 'o' :
                        state = 3056;
                        return true;
                    case 's' :
                        state = 3059;
                        return true;
                    case 'v' :
                        state = 3062;
                        return true;
                }
                break;
            case 2998:
                if ('a' == current) {
                    state = 2999;
                    return true;
                }
                break;
            case 2999:
                if ('s' == current) {
                    state = 3000;
                    return true;
                }
                break;
        }
        return false;
    }

    private boolean parse4(final int current) {
        consumedCount++;
        switch (state) {
            case 3000:
                if ('h' == current) {
                    state = 3001;
                    return true;
                }
                break;
            case 3001:
                // VDash;
                if (';' == current) {
                    match = "\u22AB";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3002:
                if ('a' == current) {
                    state = 3003;
                    return true;
                }
                break;
            case 3003:
                if ('r' == current) {
                    state = 3004;
                    return true;
                }
                break;
            case 3004:
                // Vbar;
                if (';' == current) {
                    match = "\u2AEB";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3005:
                if ('y' == current) {
                    state = 3006;
                    return true;
                }
                break;
            case 3006:
                // Vcy;
                if (';' == current) {
                    match = "\u0412";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3007:
                if ('a' == current) {
                    state = 3008;
                    return true;
                }
                break;
            case 3008:
                if ('s' == current) {
                    state = 3009;
                    return true;
                }
                break;
            case 3009:
                if ('h' == current) {
                    state = 3010;
                    return true;
                }
                break;
            case 3010:
                // Vdash;
                if (';' == current) {
                    match = "\u22A9";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('l' == current) {
                    state = 3011;
                    return true;
                }
                break;
            case 3011:
                // Vdashl;
                if (';' == current) {
                    match = "\u2AE6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3012:
                if ('e' == current) {
                    state = 3013;
                    return true;
                }
                else if ('r' == current) {
                    state = 3014;
                    return true;
                }
                break;
            case 3013:
                // Vee;
                if (';' == current) {
                    match = "\u22C1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3014:
                switch (current) {
                    case 'b' :
                        state = 3015;
                        return true;
                    case 't' :
                        state = 3018;
                        return true;
                    case 'y' :
                        state = 3044;
                        return true;
                }
                break;
            case 3015:
                if ('a' == current) {
                    state = 3016;
                    return true;
                }
                break;
            case 3016:
                if ('r' == current) {
                    state = 3017;
                    return true;
                }
                break;
            case 3017:
                // Verbar;
                if (';' == current) {
                    match = "\u2016";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3018:
                // Vert;
                if (';' == current) {
                    match = "\u2016";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('i' == current) {
                    state = 3019;
                    return true;
                }
                break;
            case 3019:
                if ('c' == current) {
                    state = 3020;
                    return true;
                }
                break;
            case 3020:
                if ('a' == current) {
                    state = 3021;
                    return true;
                }
                break;
            case 3021:
                if ('l' == current) {
                    state = 3022;
                    return true;
                }
                break;
            case 3022:
                switch (current) {
                    case 'B' :
                        state = 3023;
                        return true;
                    case 'L' :
                        state = 3026;
                        return true;
                    case 'S' :
                        state = 3030;
                        return true;
                    case 'T' :
                        state = 3039;
                        return true;
                }
                break;
            case 3023:
                if ('a' == current) {
                    state = 3024;
                    return true;
                }
                break;
            case 3024:
                if ('r' == current) {
                    state = 3025;
                    return true;
                }
                break;
            case 3025:
                // VerticalBar;
                if (';' == current) {
                    match = "\u2223";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3026:
                if ('i' == current) {
                    state = 3027;
                    return true;
                }
                break;
            case 3027:
                if ('n' == current) {
                    state = 3028;
                    return true;
                }
                break;
            case 3028:
                if ('e' == current) {
                    state = 3029;
                    return true;
                }
                break;
            case 3029:
                // VerticalLine;
                if (';' == current) {
                    match = "|";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3030:
                if ('e' == current) {
                    state = 3031;
                    return true;
                }
                break;
            case 3031:
                if ('p' == current) {
                    state = 3032;
                    return true;
                }
                break;
            case 3032:
                if ('a' == current) {
                    state = 3033;
                    return true;
                }
                break;
            case 3033:
                if ('r' == current) {
                    state = 3034;
                    return true;
                }
                break;
            case 3034:
                if ('a' == current) {
                    state = 3035;
                    return true;
                }
                break;
            case 3035:
                if ('t' == current) {
                    state = 3036;
                    return true;
                }
                break;
            case 3036:
                if ('o' == current) {
                    state = 3037;
                    return true;
                }
                break;
            case 3037:
                if ('r' == current) {
                    state = 3038;
                    return true;
                }
                break;
            case 3038:
                // VerticalSeparator;
                if (';' == current) {
                    match = "\u2758";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3039:
                if ('i' == current) {
                    state = 3040;
                    return true;
                }
                break;
            case 3040:
                if ('l' == current) {
                    state = 3041;
                    return true;
                }
                break;
            case 3041:
                if ('d' == current) {
                    state = 3042;
                    return true;
                }
                break;
            case 3042:
                if ('e' == current) {
                    state = 3043;
                    return true;
                }
                break;
            case 3043:
                // VerticalTilde;
                if (';' == current) {
                    match = "\u2240";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3044:
                if ('T' == current) {
                    state = 3045;
                    return true;
                }
                break;
            case 3045:
                if ('h' == current) {
                    state = 3046;
                    return true;
                }
                break;
            case 3046:
                if ('i' == current) {
                    state = 3047;
                    return true;
                }
                break;
            case 3047:
                if ('n' == current) {
                    state = 3048;
                    return true;
                }
                break;
            case 3048:
                if ('S' == current) {
                    state = 3049;
                    return true;
                }
                break;
            case 3049:
                if ('p' == current) {
                    state = 3050;
                    return true;
                }
                break;
            case 3050:
                if ('a' == current) {
                    state = 3051;
                    return true;
                }
                break;
            case 3051:
                if ('c' == current) {
                    state = 3052;
                    return true;
                }
                break;
            case 3052:
                if ('e' == current) {
                    state = 3053;
                    return true;
                }
                break;
            case 3053:
                // VeryThinSpace;
                if (';' == current) {
                    match = "\u200A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3054:
                if ('r' == current) {
                    state = 3055;
                    return true;
                }
                break;
            case 3055:
                // Vfr;
                if (';' == current) {
                    match = "\uD835\uDD19";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3056:
                if ('p' == current) {
                    state = 3057;
                    return true;
                }
                break;
            case 3057:
                if ('f' == current) {
                    state = 3058;
                    return true;
                }
                break;
            case 3058:
                // Vopf;
                if (';' == current) {
                    match = "\uD835\uDD4D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3059:
                if ('c' == current) {
                    state = 3060;
                    return true;
                }
                break;
            case 3060:
                if ('r' == current) {
                    state = 3061;
                    return true;
                }
                break;
            case 3061:
                // Vscr;
                if (';' == current) {
                    match = "\uD835\uDCB1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3062:
                if ('d' == current) {
                    state = 3063;
                    return true;
                }
                break;
            case 3063:
                if ('a' == current) {
                    state = 3064;
                    return true;
                }
                break;
            case 3064:
                if ('s' == current) {
                    state = 3065;
                    return true;
                }
                break;
            case 3065:
                if ('h' == current) {
                    state = 3066;
                    return true;
                }
                break;
            case 3066:
                // Vvdash;
                if (';' == current) {
                    match = "\u22AA";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3067:
                switch (current) {
                    case 'c' :
                        state = 3068;
                        return true;
                    case 'e' :
                        state = 3072;
                        return true;
                    case 'f' :
                        state = 3076;
                        return true;
                    case 'o' :
                        state = 3078;
                        return true;
                    case 's' :
                        state = 3081;
                        return true;
                }
                break;
            case 3068:
                if ('i' == current) {
                    state = 3069;
                    return true;
                }
                break;
            case 3069:
                if ('r' == current) {
                    state = 3070;
                    return true;
                }
                break;
            case 3070:
                if ('c' == current) {
                    state = 3071;
                    return true;
                }
                break;
            case 3071:
                // Wcirc;
                if (';' == current) {
                    match = "\u0174";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3072:
                if ('d' == current) {
                    state = 3073;
                    return true;
                }
                break;
            case 3073:
                if ('g' == current) {
                    state = 3074;
                    return true;
                }
                break;
            case 3074:
                if ('e' == current) {
                    state = 3075;
                    return true;
                }
                break;
            case 3075:
                // Wedge;
                if (';' == current) {
                    match = "\u22C0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3076:
                if ('r' == current) {
                    state = 3077;
                    return true;
                }
                break;
            case 3077:
                // Wfr;
                if (';' == current) {
                    match = "\uD835\uDD1A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3078:
                if ('p' == current) {
                    state = 3079;
                    return true;
                }
                break;
            case 3079:
                if ('f' == current) {
                    state = 3080;
                    return true;
                }
                break;
            case 3080:
                // Wopf;
                if (';' == current) {
                    match = "\uD835\uDD4E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3081:
                if ('c' == current) {
                    state = 3082;
                    return true;
                }
                break;
            case 3082:
                if ('r' == current) {
                    state = 3083;
                    return true;
                }
                break;
            case 3083:
                // Wscr;
                if (';' == current) {
                    match = "\uD835\uDCB2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3084:
                switch (current) {
                    case 'f' :
                        state = 3085;
                        return true;
                    case 'i' :
                        state = 3087;
                        return true;
                    case 'o' :
                        state = 3088;
                        return true;
                    case 's' :
                        state = 3091;
                        return true;
                }
                break;
            case 3085:
                if ('r' == current) {
                    state = 3086;
                    return true;
                }
                break;
            case 3086:
                // Xfr;
                if (';' == current) {
                    match = "\uD835\uDD1B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3087:
                // Xi;
                if (';' == current) {
                    match = "\u039E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3088:
                if ('p' == current) {
                    state = 3089;
                    return true;
                }
                break;
            case 3089:
                if ('f' == current) {
                    state = 3090;
                    return true;
                }
                break;
            case 3090:
                // Xopf;
                if (';' == current) {
                    match = "\uD835\uDD4F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3091:
                if ('c' == current) {
                    state = 3092;
                    return true;
                }
                break;
            case 3092:
                if ('r' == current) {
                    state = 3093;
                    return true;
                }
                break;
            case 3093:
                // Xscr;
                if (';' == current) {
                    match = "\uD835\uDCB3";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3094:
                switch (current) {
                    case 'A' :
                        state = 3095;
                        return true;
                    case 'I' :
                        state = 3098;
                        return true;
                    case 'U' :
                        state = 3101;
                        return true;
                    case 'a' :
                        state = 3104;
                        return true;
                    case 'c' :
                        state = 3109;
                        return true;
                    case 'f' :
                        state = 3114;
                        return true;
                    case 'o' :
                        state = 3116;
                        return true;
                    case 's' :
                        state = 3119;
                        return true;
                    case 'u' :
                        state = 3122;
                        return true;
                }
                break;
            case 3095:
                if ('c' == current) {
                    state = 3096;
                    return true;
                }
                break;
            case 3096:
                if ('y' == current) {
                    state = 3097;
                    return true;
                }
                break;
            case 3097:
                // YAcy;
                if (';' == current) {
                    match = "\u042F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3098:
                if ('c' == current) {
                    state = 3099;
                    return true;
                }
                break;
            case 3099:
                if ('y' == current) {
                    state = 3100;
                    return true;
                }
                break;
            case 3100:
                // YIcy;
                if (';' == current) {
                    match = "\u0407";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3101:
                if ('c' == current) {
                    state = 3102;
                    return true;
                }
                break;
            case 3102:
                if ('y' == current) {
                    state = 3103;
                    return true;
                }
                break;
            case 3103:
                // YUcy;
                if (';' == current) {
                    match = "\u042E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3104:
                if ('c' == current) {
                    state = 3105;
                    return true;
                }
                break;
            case 3105:
                if ('u' == current) {
                    state = 3106;
                    return true;
                }
                break;
            case 3106:
                if ('t' == current) {
                    state = 3107;
                    return true;
                }
                break;
            case 3107:
                // Yacute
                if ('e' == current) {
                    match = "\u00DD";
                    matchLength = consumedCount;
                    state = 3108;
                    return true;
                }
                break;
            case 3108:
                // Yacute;
                if (';' == current) {
                    match = "\u00DD";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3109:
                if ('i' == current) {
                    state = 3110;
                    return true;
                }
                else if ('y' == current) {
                    state = 3113;
                    return true;
                }
                break;
            case 3110:
                if ('r' == current) {
                    state = 3111;
                    return true;
                }
                break;
            case 3111:
                if ('c' == current) {
                    state = 3112;
                    return true;
                }
                break;
            case 3112:
                // Ycirc;
                if (';' == current) {
                    match = "\u0176";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3113:
                // Ycy;
                if (';' == current) {
                    match = "\u042B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3114:
                if ('r' == current) {
                    state = 3115;
                    return true;
                }
                break;
            case 3115:
                // Yfr;
                if (';' == current) {
                    match = "\uD835\uDD1C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3116:
                if ('p' == current) {
                    state = 3117;
                    return true;
                }
                break;
            case 3117:
                if ('f' == current) {
                    state = 3118;
                    return true;
                }
                break;
            case 3118:
                // Yopf;
                if (';' == current) {
                    match = "\uD835\uDD50";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3119:
                if ('c' == current) {
                    state = 3120;
                    return true;
                }
                break;
            case 3120:
                if ('r' == current) {
                    state = 3121;
                    return true;
                }
                break;
            case 3121:
                // Yscr;
                if (';' == current) {
                    match = "\uD835\uDCB4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3122:
                if ('m' == current) {
                    state = 3123;
                    return true;
                }
                break;
            case 3123:
                if ('l' == current) {
                    state = 3124;
                    return true;
                }
                break;
            case 3124:
                // Yuml;
                if (';' == current) {
                    match = "\u0178";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3125:
                switch (current) {
                    case 'H' :
                        state = 3126;
                        return true;
                    case 'a' :
                        state = 3129;
                        return true;
                    case 'c' :
                        state = 3134;
                        return true;
                    case 'd' :
                        state = 3140;
                        return true;
                    case 'e' :
                        state = 3143;
                        return true;
                    case 'f' :
                        state = 3158;
                        return true;
                    case 'o' :
                        state = 3160;
                        return true;
                    case 's' :
                        state = 3163;
                        return true;
                }
                break;
            case 3126:
                if ('c' == current) {
                    state = 3127;
                    return true;
                }
                break;
            case 3127:
                if ('y' == current) {
                    state = 3128;
                    return true;
                }
                break;
            case 3128:
                // ZHcy;
                if (';' == current) {
                    match = "\u0416";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3129:
                if ('c' == current) {
                    state = 3130;
                    return true;
                }
                break;
            case 3130:
                if ('u' == current) {
                    state = 3131;
                    return true;
                }
                break;
            case 3131:
                if ('t' == current) {
                    state = 3132;
                    return true;
                }
                break;
            case 3132:
                if ('e' == current) {
                    state = 3133;
                    return true;
                }
                break;
            case 3133:
                // Zacute;
                if (';' == current) {
                    match = "\u0179";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3134:
                if ('a' == current) {
                    state = 3135;
                    return true;
                }
                else if ('y' == current) {
                    state = 3139;
                    return true;
                }
                break;
            case 3135:
                if ('r' == current) {
                    state = 3136;
                    return true;
                }
                break;
            case 3136:
                if ('o' == current) {
                    state = 3137;
                    return true;
                }
                break;
            case 3137:
                if ('n' == current) {
                    state = 3138;
                    return true;
                }
                break;
            case 3138:
                // Zcaron;
                if (';' == current) {
                    match = "\u017D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3139:
                // Zcy;
                if (';' == current) {
                    match = "\u0417";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3140:
                if ('o' == current) {
                    state = 3141;
                    return true;
                }
                break;
            case 3141:
                if ('t' == current) {
                    state = 3142;
                    return true;
                }
                break;
            case 3142:
                // Zdot;
                if (';' == current) {
                    match = "\u017B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3143:
                if ('r' == current) {
                    state = 3144;
                    return true;
                }
                else if ('t' == current) {
                    state = 3156;
                    return true;
                }
                break;
            case 3144:
                if ('o' == current) {
                    state = 3145;
                    return true;
                }
                break;
            case 3145:
                if ('W' == current) {
                    state = 3146;
                    return true;
                }
                break;
            case 3146:
                if ('i' == current) {
                    state = 3147;
                    return true;
                }
                break;
            case 3147:
                if ('d' == current) {
                    state = 3148;
                    return true;
                }
                break;
            case 3148:
                if ('t' == current) {
                    state = 3149;
                    return true;
                }
                break;
            case 3149:
                if ('h' == current) {
                    state = 3150;
                    return true;
                }
                break;
            case 3150:
                if ('S' == current) {
                    state = 3151;
                    return true;
                }
                break;
            case 3151:
                if ('p' == current) {
                    state = 3152;
                    return true;
                }
                break;
            case 3152:
                if ('a' == current) {
                    state = 3153;
                    return true;
                }
                break;
            case 3153:
                if ('c' == current) {
                    state = 3154;
                    return true;
                }
                break;
            case 3154:
                if ('e' == current) {
                    state = 3155;
                    return true;
                }
                break;
            case 3155:
                // ZeroWidthSpace;
                if (';' == current) {
                    match = "\u200B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3156:
                if ('a' == current) {
                    state = 3157;
                    return true;
                }
                break;
            case 3157:
                // Zeta;
                if (';' == current) {
                    match = "\u0396";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3158:
                if ('r' == current) {
                    state = 3159;
                    return true;
                }
                break;
            case 3159:
                // Zfr;
                if (';' == current) {
                    match = "\u2128";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3160:
                if ('p' == current) {
                    state = 3161;
                    return true;
                }
                break;
            case 3161:
                if ('f' == current) {
                    state = 3162;
                    return true;
                }
                break;
            case 3162:
                // Zopf;
                if (';' == current) {
                    match = "\u2124";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3163:
                if ('c' == current) {
                    state = 3164;
                    return true;
                }
                break;
            case 3164:
                if ('r' == current) {
                    state = 3165;
                    return true;
                }
                break;
            case 3165:
                // Zscr;
                if (';' == current) {
                    match = "\uD835\uDCB5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3166:
                switch (current) {
                    case 'a' :
                        state = 3167;
                        return true;
                    case 'b' :
                        state = 3172;
                        return true;
                    case 'c' :
                        state = 3177;
                        return true;
                    case 'e' :
                        state = 3187;
                        return true;
                    case 'f' :
                        state = 3191;
                        return true;
                    case 'g' :
                        state = 3193;
                        return true;
                    case 'l' :
                        state = 3198;
                        return true;
                    case 'm' :
                        state = 3209;
                        return true;
                    case 'n' :
                        state = 3216;
                        return true;
                    case 'o' :
                        state = 3257;
                        return true;
                    case 'p' :
                        state = 3263;
                        return true;
                    case 'r' :
                        state = 3280;
                        return true;
                    case 's' :
                        state = 3284;
                        return true;
                    case 't' :
                        state = 3293;
                        return true;
                    case 'u' :
                        state = 3298;
                        return true;
                    case 'w' :
                        state = 3301;
                        return true;
                }
                break;
            case 3167:
                if ('c' == current) {
                    state = 3168;
                    return true;
                }
                break;
            case 3168:
                if ('u' == current) {
                    state = 3169;
                    return true;
                }
                break;
            case 3169:
                if ('t' == current) {
                    state = 3170;
                    return true;
                }
                break;
            case 3170:
                // aacute
                if ('e' == current) {
                    match = "\u00E1";
                    matchLength = consumedCount;
                    state = 3171;
                    return true;
                }
                break;
            case 3171:
                // aacute;
                if (';' == current) {
                    match = "\u00E1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3172:
                if ('r' == current) {
                    state = 3173;
                    return true;
                }
                break;
            case 3173:
                if ('e' == current) {
                    state = 3174;
                    return true;
                }
                break;
            case 3174:
                if ('v' == current) {
                    state = 3175;
                    return true;
                }
                break;
            case 3175:
                if ('e' == current) {
                    state = 3176;
                    return true;
                }
                break;
            case 3176:
                // abreve;
                if (';' == current) {
                    match = "\u0103";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3177:
                switch (current) {
                    case ';' : // ac;
                        match = "\u223E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 3178;
                        return true;
                    case 'd' :
                        state = 3179;
                        return true;
                    case 'i' :
                        state = 3180;
                        return true;
                    case 'u' :
                        state = 3183;
                        return true;
                    case 'y' :
                        state = 3186;
                        return true;
                }
                break;
            case 3178:
                // acE;
                if (';' == current) {
                    match = "\u223E\u0333";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3179:
                // acd;
                if (';' == current) {
                    match = "\u223F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3180:
                if ('r' == current) {
                    state = 3181;
                    return true;
                }
                break;
            case 3181:
                // acirc
                if ('c' == current) {
                    match = "\u00E2";
                    matchLength = consumedCount;
                    state = 3182;
                    return true;
                }
                break;
            case 3182:
                // acirc;
                if (';' == current) {
                    match = "\u00E2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3183:
                if ('t' == current) {
                    state = 3184;
                    return true;
                }
                break;
            case 3184:
                // acute
                if ('e' == current) {
                    match = "\u00B4";
                    matchLength = consumedCount;
                    state = 3185;
                    return true;
                }
                break;
            case 3185:
                // acute;
                if (';' == current) {
                    match = "\u00B4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3186:
                // acy;
                if (';' == current) {
                    match = "\u0430";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3187:
                if ('l' == current) {
                    state = 3188;
                    return true;
                }
                break;
            case 3188:
                if ('i' == current) {
                    state = 3189;
                    return true;
                }
                break;
            case 3189:
                // aelig
                if ('g' == current) {
                    match = "\u00E6";
                    matchLength = consumedCount;
                    state = 3190;
                    return true;
                }
                break;
            case 3190:
                // aelig;
                if (';' == current) {
                    match = "\u00E6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3191:
                // af;
                if (';' == current) {
                    match = "\u2061";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('r' == current) {
                    state = 3192;
                    return true;
                }
                break;
            case 3192:
                // afr;
                if (';' == current) {
                    match = "\uD835\uDD1E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3193:
                if ('r' == current) {
                    state = 3194;
                    return true;
                }
                break;
            case 3194:
                if ('a' == current) {
                    state = 3195;
                    return true;
                }
                break;
            case 3195:
                if ('v' == current) {
                    state = 3196;
                    return true;
                }
                break;
            case 3196:
                // agrave
                if ('e' == current) {
                    match = "\u00E0";
                    matchLength = consumedCount;
                    state = 3197;
                    return true;
                }
                break;
            case 3197:
                // agrave;
                if (';' == current) {
                    match = "\u00E0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3198:
                if ('e' == current) {
                    state = 3199;
                    return true;
                }
                else if ('p' == current) {
                    state = 3206;
                    return true;
                }
                break;
            case 3199:
                if ('f' == current) {
                    state = 3200;
                    return true;
                }
                else if ('p' == current) {
                    state = 3204;
                    return true;
                }
                break;
            case 3200:
                if ('s' == current) {
                    state = 3201;
                    return true;
                }
                break;
            case 3201:
                if ('y' == current) {
                    state = 3202;
                    return true;
                }
                break;
            case 3202:
                if ('m' == current) {
                    state = 3203;
                    return true;
                }
                break;
            case 3203:
                // alefsym;
                if (';' == current) {
                    match = "\u2135";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3204:
                if ('h' == current) {
                    state = 3205;
                    return true;
                }
                break;
            case 3205:
                // aleph;
                if (';' == current) {
                    match = "\u2135";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3206:
                if ('h' == current) {
                    state = 3207;
                    return true;
                }
                break;
            case 3207:
                if ('a' == current) {
                    state = 3208;
                    return true;
                }
                break;
            case 3208:
                // alpha;
                if (';' == current) {
                    match = "\u03B1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3209:
                if ('a' == current) {
                    state = 3210;
                    return true;
                }
                // amp
                else if ('p' == current) {
                    match = "&";
                    matchLength = consumedCount;
                    state = 3215;
                    return true;
                }
                break;
            case 3210:
                if ('c' == current) {
                    state = 3211;
                    return true;
                }
                else if ('l' == current) {
                    state = 3213;
                    return true;
                }
                break;
            case 3211:
                if ('r' == current) {
                    state = 3212;
                    return true;
                }
                break;
            case 3212:
                // amacr;
                if (';' == current) {
                    match = "\u0101";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3213:
                if ('g' == current) {
                    state = 3214;
                    return true;
                }
                break;
            case 3214:
                // amalg;
                if (';' == current) {
                    match = "\u2A3F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3215:
                // amp;
                if (';' == current) {
                    match = "&";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3216:
                if ('d' == current) {
                    state = 3217;
                    return true;
                }
                else if ('g' == current) {
                    state = 3228;
                    return true;
                }
                break;
            case 3217:
                switch (current) {
                    case ';' : // and;
                        match = "\u2227";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'a' :
                        state = 3218;
                        return true;
                    case 'd' :
                        state = 3221;
                        return true;
                    case 's' :
                        state = 3222;
                        return true;
                    case 'v' :
                        state = 3227;
                        return true;
                }
                break;
            case 3218:
                if ('n' == current) {
                    state = 3219;
                    return true;
                }
                break;
            case 3219:
                if ('d' == current) {
                    state = 3220;
                    return true;
                }
                break;
            case 3220:
                // andand;
                if (';' == current) {
                    match = "\u2A55";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3221:
                // andd;
                if (';' == current) {
                    match = "\u2A5C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3222:
                if ('l' == current) {
                    state = 3223;
                    return true;
                }
                break;
            case 3223:
                if ('o' == current) {
                    state = 3224;
                    return true;
                }
                break;
            case 3224:
                if ('p' == current) {
                    state = 3225;
                    return true;
                }
                break;
            case 3225:
                if ('e' == current) {
                    state = 3226;
                    return true;
                }
                break;
            case 3226:
                // andslope;
                if (';' == current) {
                    match = "\u2A58";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3227:
                // andv;
                if (';' == current) {
                    match = "\u2A5A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3228:
                switch (current) {
                    case ';' : // ang;
                        match = "\u2220";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 3229;
                        return true;
                    case 'l' :
                        state = 3230;
                        return true;
                    case 'm' :
                        state = 3232;
                        return true;
                    case 'r' :
                        state = 3244;
                        return true;
                    case 's' :
                        state = 3249;
                        return true;
                    case 'z' :
                        state = 3253;
                        return true;
                }
                break;
            case 3229:
                // ange;
                if (';' == current) {
                    match = "\u29A4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3230:
                if ('e' == current) {
                    state = 3231;
                    return true;
                }
                break;
            case 3231:
                // angle;
                if (';' == current) {
                    match = "\u2220";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3232:
                if ('s' == current) {
                    state = 3233;
                    return true;
                }
                break;
            case 3233:
                if ('d' == current) {
                    state = 3234;
                    return true;
                }
                break;
            case 3234:
                // angmsd;
                if (';' == current) {
                    match = "\u2221";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('a' == current) {
                    state = 3235;
                    return true;
                }
                break;
            case 3235:
                switch (current) {
                    case 'a' :
                        state = 3236;
                        return true;
                    case 'b' :
                        state = 3237;
                        return true;
                    case 'c' :
                        state = 3238;
                        return true;
                    case 'd' :
                        state = 3239;
                        return true;
                    case 'e' :
                        state = 3240;
                        return true;
                    case 'f' :
                        state = 3241;
                        return true;
                    case 'g' :
                        state = 3242;
                        return true;
                    case 'h' :
                        state = 3243;
                        return true;
                }
                break;
            case 3236:
                // angmsdaa;
                if (';' == current) {
                    match = "\u29A8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3237:
                // angmsdab;
                if (';' == current) {
                    match = "\u29A9";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3238:
                // angmsdac;
                if (';' == current) {
                    match = "\u29AA";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3239:
                // angmsdad;
                if (';' == current) {
                    match = "\u29AB";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3240:
                // angmsdae;
                if (';' == current) {
                    match = "\u29AC";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3241:
                // angmsdaf;
                if (';' == current) {
                    match = "\u29AD";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3242:
                // angmsdag;
                if (';' == current) {
                    match = "\u29AE";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3243:
                // angmsdah;
                if (';' == current) {
                    match = "\u29AF";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3244:
                if ('t' == current) {
                    state = 3245;
                    return true;
                }
                break;
            case 3245:
                // angrt;
                if (';' == current) {
                    match = "\u221F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('v' == current) {
                    state = 3246;
                    return true;
                }
                break;
            case 3246:
                if ('b' == current) {
                    state = 3247;
                    return true;
                }
                break;
            case 3247:
                // angrtvb;
                if (';' == current) {
                    match = "\u22BE";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('d' == current) {
                    state = 3248;
                    return true;
                }
                break;
            case 3248:
                // angrtvbd;
                if (';' == current) {
                    match = "\u299D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3249:
                if ('p' == current) {
                    state = 3250;
                    return true;
                }
                else if ('t' == current) {
                    state = 3252;
                    return true;
                }
                break;
            case 3250:
                if ('h' == current) {
                    state = 3251;
                    return true;
                }
                break;
            case 3251:
                // angsph;
                if (';' == current) {
                    match = "\u2222";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3252:
                // angst;
                if (';' == current) {
                    match = "\u00C5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3253:
                if ('a' == current) {
                    state = 3254;
                    return true;
                }
                break;
            case 3254:
                if ('r' == current) {
                    state = 3255;
                    return true;
                }
                break;
            case 3255:
                if ('r' == current) {
                    state = 3256;
                    return true;
                }
                break;
            case 3256:
                // angzarr;
                if (';' == current) {
                    match = "\u237C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3257:
                if ('g' == current) {
                    state = 3258;
                    return true;
                }
                else if ('p' == current) {
                    state = 3261;
                    return true;
                }
                break;
            case 3258:
                if ('o' == current) {
                    state = 3259;
                    return true;
                }
                break;
            case 3259:
                if ('n' == current) {
                    state = 3260;
                    return true;
                }
                break;
            case 3260:
                // aogon;
                if (';' == current) {
                    match = "\u0105";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3261:
                if ('f' == current) {
                    state = 3262;
                    return true;
                }
                break;
            case 3262:
                // aopf;
                if (';' == current) {
                    match = "\uD835\uDD52";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3263:
                switch (current) {
                    case ';' : // ap;
                        match = "\u2248";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 3264;
                        return true;
                    case 'a' :
                        state = 3265;
                        return true;
                    case 'e' :
                        state = 3269;
                        return true;
                    case 'i' :
                        state = 3270;
                        return true;
                    case 'o' :
                        state = 3272;
                        return true;
                    case 'p' :
                        state = 3274;
                        return true;
                }
                break;
            case 3264:
                // apE;
                if (';' == current) {
                    match = "\u2A70";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3265:
                if ('c' == current) {
                    state = 3266;
                    return true;
                }
                break;
            case 3266:
                if ('i' == current) {
                    state = 3267;
                    return true;
                }
                break;
            case 3267:
                if ('r' == current) {
                    state = 3268;
                    return true;
                }
                break;
            case 3268:
                // apacir;
                if (';' == current) {
                    match = "\u2A6F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3269:
                // ape;
                if (';' == current) {
                    match = "\u224A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3270:
                if ('d' == current) {
                    state = 3271;
                    return true;
                }
                break;
            case 3271:
                // apid;
                if (';' == current) {
                    match = "\u224B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3272:
                if ('s' == current) {
                    state = 3273;
                    return true;
                }
                break;
            case 3273:
                // apos;
                if (';' == current) {
                    match = "'";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3274:
                if ('r' == current) {
                    state = 3275;
                    return true;
                }
                break;
            case 3275:
                if ('o' == current) {
                    state = 3276;
                    return true;
                }
                break;
            case 3276:
                if ('x' == current) {
                    state = 3277;
                    return true;
                }
                break;
            case 3277:
                // approx;
                if (';' == current) {
                    match = "\u2248";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state = 3278;
                    return true;
                }
                break;
            case 3278:
                if ('q' == current) {
                    state = 3279;
                    return true;
                }
                break;
            case 3279:
                // approxeq;
                if (';' == current) {
                    match = "\u224A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3280:
                if ('i' == current) {
                    state = 3281;
                    return true;
                }
                break;
            case 3281:
                if ('n' == current) {
                    state = 3282;
                    return true;
                }
                break;
            case 3282:
                // aring
                if ('g' == current) {
                    match = "\u00E5";
                    matchLength = consumedCount;
                    state = 3283;
                    return true;
                }
                break;
            case 3283:
                // aring;
                if (';' == current) {
                    match = "\u00E5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3284:
                switch (current) {
                    case 'c' :
                        state = 3285;
                        return true;
                    case 't' :
                        state = 3287;
                        return true;
                    case 'y' :
                        state = 3288;
                        return true;
                }
                break;
            case 3285:
                if ('r' == current) {
                    state = 3286;
                    return true;
                }
                break;
            case 3286:
                // ascr;
                if (';' == current) {
                    match = "\uD835\uDCB6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3287:
                // ast;
                if (';' == current) {
                    match = "*";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3288:
                if ('m' == current) {
                    state = 3289;
                    return true;
                }
                break;
            case 3289:
                if ('p' == current) {
                    state = 3290;
                    return true;
                }
                break;
            case 3290:
                // asymp;
                if (';' == current) {
                    match = "\u2248";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state = 3291;
                    return true;
                }
                break;
            case 3291:
                if ('q' == current) {
                    state = 3292;
                    return true;
                }
                break;
            case 3292:
                // asympeq;
                if (';' == current) {
                    match = "\u224D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3293:
                if ('i' == current) {
                    state = 3294;
                    return true;
                }
                break;
            case 3294:
                if ('l' == current) {
                    state = 3295;
                    return true;
                }
                break;
            case 3295:
                if ('d' == current) {
                    state = 3296;
                    return true;
                }
                break;
            case 3296:
                // atilde
                if ('e' == current) {
                    match = "\u00E3";
                    matchLength = consumedCount;
                    state = 3297;
                    return true;
                }
                break;
            case 3297:
                // atilde;
                if (';' == current) {
                    match = "\u00E3";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3298:
                if ('m' == current) {
                    state = 3299;
                    return true;
                }
                break;
            case 3299:
                // auml
                if ('l' == current) {
                    match = "\u00E4";
                    matchLength = consumedCount;
                    state = 3300;
                    return true;
                }
                break;
            case 3300:
                // auml;
                if (';' == current) {
                    match = "\u00E4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3301:
                if ('c' == current) {
                    state = 3302;
                    return true;
                }
                else if ('i' == current) {
                    state = 3308;
                    return true;
                }
                break;
            case 3302:
                if ('o' == current) {
                    state = 3303;
                    return true;
                }
                break;
            case 3303:
                if ('n' == current) {
                    state = 3304;
                    return true;
                }
                break;
            case 3304:
                if ('i' == current) {
                    state = 3305;
                    return true;
                }
                break;
            case 3305:
                if ('n' == current) {
                    state = 3306;
                    return true;
                }
                break;
            case 3306:
                if ('t' == current) {
                    state = 3307;
                    return true;
                }
                break;
            case 3307:
                // awconint;
                if (';' == current) {
                    match = "\u2233";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3308:
                if ('n' == current) {
                    state = 3309;
                    return true;
                }
                break;
            case 3309:
                if ('t' == current) {
                    state = 3310;
                    return true;
                }
                break;
            case 3310:
                // awint;
                if (';' == current) {
                    match = "\u2A11";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3311:
                switch (current) {
                    case 'N' :
                        state = 3312;
                        return true;
                    case 'a' :
                        state = 3315;
                        return true;
                    case 'b' :
                        state = 3348;
                        return true;
                    case 'c' :
                        state = 3355;
                        return true;
                    case 'd' :
                        state = 3360;
                        return true;
                    case 'e' :
                        state = 3364;
                        return true;
                    case 'f' :
                        state = 3389;
                        return true;
                    case 'i' :
                        state = 3391;
                        return true;
                    case 'k' :
                        state = 3449;
                        return true;
                    case 'l' :
                        state = 3454;
                        return true;
                    case 'n' :
                        state = 3503;
                        return true;
                    case 'o' :
                        state = 3511;
                        return true;
                    case 'p' :
                        state = 3584;
                        return true;
                    case 'r' :
                        state = 3589;
                        return true;
                    case 's' :
                        state = 3597;
                        return true;
                    case 'u' :
                        state = 3613;
                        return true;
                }
                break;
            case 3312:
                if ('o' == current) {
                    state = 3313;
                    return true;
                }
                break;
            case 3313:
                if ('t' == current) {
                    state = 3314;
                    return true;
                }
                break;
            case 3314:
                // bNot;
                if (';' == current) {
                    match = "\u2AED";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3315:
                if ('c' == current) {
                    state = 3316;
                    return true;
                }
                else if ('r' == current) {
                    state = 3339;
                    return true;
                }
                break;
            case 3316:
                if ('k' == current) {
                    state = 3317;
                    return true;
                }
                break;
            case 3317:
                switch (current) {
                    case 'c' :
                        state = 3318;
                        return true;
                    case 'e' :
                        state = 3322;
                        return true;
                    case 'p' :
                        state = 3329;
                        return true;
                    case 's' :
                        state = 3334;
                        return true;
                }
                break;
            case 3318:
                if ('o' == current) {
                    state = 3319;
                    return true;
                }
                break;
            case 3319:
                if ('n' == current) {
                    state = 3320;
                    return true;
                }
                break;
            case 3320:
                if ('g' == current) {
                    state = 3321;
                    return true;
                }
                break;
            case 3321:
                // backcong;
                if (';' == current) {
                    match = "\u224C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3322:
                if ('p' == current) {
                    state = 3323;
                    return true;
                }
                break;
            case 3323:
                if ('s' == current) {
                    state = 3324;
                    return true;
                }
                break;
            case 3324:
                if ('i' == current) {
                    state = 3325;
                    return true;
                }
                break;
            case 3325:
                if ('l' == current) {
                    state = 3326;
                    return true;
                }
                break;
            case 3326:
                if ('o' == current) {
                    state = 3327;
                    return true;
                }
                break;
            case 3327:
                if ('n' == current) {
                    state = 3328;
                    return true;
                }
                break;
            case 3328:
                // backepsilon;
                if (';' == current) {
                    match = "\u03F6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3329:
                if ('r' == current) {
                    state = 3330;
                    return true;
                }
                break;
            case 3330:
                if ('i' == current) {
                    state = 3331;
                    return true;
                }
                break;
            case 3331:
                if ('m' == current) {
                    state = 3332;
                    return true;
                }
                break;
            case 3332:
                if ('e' == current) {
                    state = 3333;
                    return true;
                }
                break;
            case 3333:
                // backprime;
                if (';' == current) {
                    match = "\u2035";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3334:
                if ('i' == current) {
                    state = 3335;
                    return true;
                }
                break;
            case 3335:
                if ('m' == current) {
                    state = 3336;
                    return true;
                }
                break;
            case 3336:
                // backsim;
                if (';' == current) {
                    match = "\u223D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state = 3337;
                    return true;
                }
                break;
            case 3337:
                if ('q' == current) {
                    state = 3338;
                    return true;
                }
                break;
            case 3338:
                // backsimeq;
                if (';' == current) {
                    match = "\u22CD";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3339:
                if ('v' == current) {
                    state = 3340;
                    return true;
                }
                else if ('w' == current) {
                    state = 3343;
                    return true;
                }
                break;
            case 3340:
                if ('e' == current) {
                    state = 3341;
                    return true;
                }
                break;
            case 3341:
                if ('e' == current) {
                    state = 3342;
                    return true;
                }
                break;
            case 3342:
                // barvee;
                if (';' == current) {
                    match = "\u22BD";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3343:
                if ('e' == current) {
                    state = 3344;
                    return true;
                }
                break;
            case 3344:
                if ('d' == current) {
                    state = 3345;
                    return true;
                }
                break;
            case 3345:
                // barwed;
                if (';' == current) {
                    match = "\u2305";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('g' == current) {
                    state = 3346;
                    return true;
                }
                break;
            case 3346:
                if ('e' == current) {
                    state = 3347;
                    return true;
                }
                break;
            case 3347:
                // barwedge;
                if (';' == current) {
                    match = "\u2305";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3348:
                if ('r' == current) {
                    state = 3349;
                    return true;
                }
                break;
            case 3349:
                if ('k' == current) {
                    state = 3350;
                    return true;
                }
                break;
            case 3350:
                // bbrk;
                if (';' == current) {
                    match = "\u23B5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('t' == current) {
                    state = 3351;
                    return true;
                }
                break;
            case 3351:
                if ('b' == current) {
                    state = 3352;
                    return true;
                }
                break;
            case 3352:
                if ('r' == current) {
                    state = 3353;
                    return true;
                }
                break;
            case 3353:
                if ('k' == current) {
                    state = 3354;
                    return true;
                }
                break;
            case 3354:
                // bbrktbrk;
                if (';' == current) {
                    match = "\u23B6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3355:
                if ('o' == current) {
                    state = 3356;
                    return true;
                }
                else if ('y' == current) {
                    state = 3359;
                    return true;
                }
                break;
            case 3356:
                if ('n' == current) {
                    state = 3357;
                    return true;
                }
                break;
            case 3357:
                if ('g' == current) {
                    state = 3358;
                    return true;
                }
                break;
            case 3358:
                // bcong;
                if (';' == current) {
                    match = "\u224C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3359:
                // bcy;
                if (';' == current) {
                    match = "\u0431";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3360:
                if ('q' == current) {
                    state = 3361;
                    return true;
                }
                break;
            case 3361:
                if ('u' == current) {
                    state = 3362;
                    return true;
                }
                break;
            case 3362:
                if ('o' == current) {
                    state = 3363;
                    return true;
                }
                break;
            case 3363:
                // bdquo;
                if (';' == current) {
                    match = "\u201E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3364:
                switch (current) {
                    case 'c' :
                        state = 3365;
                        return true;
                    case 'm' :
                        state = 3370;
                        return true;
                    case 'p' :
                        state = 3375;
                        return true;
                    case 'r' :
                        state = 3378;
                        return true;
                    case 't' :
                        state = 3382;
                        return true;
                }
                break;
            case 3365:
                if ('a' == current) {
                    state = 3366;
                    return true;
                }
                break;
            case 3366:
                if ('u' == current) {
                    state = 3367;
                    return true;
                }
                break;
            case 3367:
                if ('s' == current) {
                    state = 3368;
                    return true;
                }
                break;
            case 3368:
                // becaus;
                if (';' == current) {
                    match = "\u2235";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state = 3369;
                    return true;
                }
                break;
            case 3369:
                // because;
                if (';' == current) {
                    match = "\u2235";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3370:
                if ('p' == current) {
                    state = 3371;
                    return true;
                }
                break;
            case 3371:
                if ('t' == current) {
                    state = 3372;
                    return true;
                }
                break;
            case 3372:
                if ('y' == current) {
                    state = 3373;
                    return true;
                }
                break;
            case 3373:
                if ('v' == current) {
                    state = 3374;
                    return true;
                }
                break;
            case 3374:
                // bemptyv;
                if (';' == current) {
                    match = "\u29B0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3375:
                if ('s' == current) {
                    state = 3376;
                    return true;
                }
                break;
            case 3376:
                if ('i' == current) {
                    state = 3377;
                    return true;
                }
                break;
            case 3377:
                // bepsi;
                if (';' == current) {
                    match = "\u03F6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3378:
                if ('n' == current) {
                    state = 3379;
                    return true;
                }
                break;
            case 3379:
                if ('o' == current) {
                    state = 3380;
                    return true;
                }
                break;
            case 3380:
                if ('u' == current) {
                    state = 3381;
                    return true;
                }
                break;
            case 3381:
                // bernou;
                if (';' == current) {
                    match = "\u212C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3382:
                switch (current) {
                    case 'a' :
                        state = 3383;
                        return true;
                    case 'h' :
                        state = 3384;
                        return true;
                    case 'w' :
                        state = 3385;
                        return true;
                }
                break;
            case 3383:
                // beta;
                if (';' == current) {
                    match = "\u03B2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3384:
                // beth;
                if (';' == current) {
                    match = "\u2136";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3385:
                if ('e' == current) {
                    state = 3386;
                    return true;
                }
                break;
            case 3386:
                if ('e' == current) {
                    state = 3387;
                    return true;
                }
                break;
            case 3387:
                if ('n' == current) {
                    state = 3388;
                    return true;
                }
                break;
            case 3388:
                // between;
                if (';' == current) {
                    match = "\u226C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3389:
                if ('r' == current) {
                    state = 3390;
                    return true;
                }
                break;
            case 3390:
                // bfr;
                if (';' == current) {
                    match = "\uD835\uDD1F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3391:
                if ('g' == current) {
                    state = 3392;
                    return true;
                }
                break;
            case 3392:
                switch (current) {
                    case 'c' :
                        state = 3393;
                        return true;
                    case 'o' :
                        state = 3401;
                        return true;
                    case 's' :
                        state = 3414;
                        return true;
                    case 't' :
                        state = 3422;
                        return true;
                    case 'u' :
                        state = 3436;
                        return true;
                    case 'v' :
                        state = 3441;
                        return true;
                    case 'w' :
                        state = 3444;
                        return true;
                }
                break;
            case 3393:
                switch (current) {
                    case 'a' :
                        state = 3394;
                        return true;
                    case 'i' :
                        state = 3396;
                        return true;
                    case 'u' :
                        state = 3399;
                        return true;
                }
                break;
            case 3394:
                if ('p' == current) {
                    state = 3395;
                    return true;
                }
                break;
            case 3395:
                // bigcap;
                if (';' == current) {
                    match = "\u22C2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3396:
                if ('r' == current) {
                    state = 3397;
                    return true;
                }
                break;
            case 3397:
                if ('c' == current) {
                    state = 3398;
                    return true;
                }
                break;
            case 3398:
                // bigcirc;
                if (';' == current) {
                    match = "\u25EF";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3399:
                if ('p' == current) {
                    state = 3400;
                    return true;
                }
                break;
            case 3400:
                // bigcup;
                if (';' == current) {
                    match = "\u22C3";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3401:
                switch (current) {
                    case 'd' :
                        state = 3402;
                        return true;
                    case 'p' :
                        state = 3405;
                        return true;
                    case 't' :
                        state = 3409;
                        return true;
                }
                break;
            case 3402:
                if ('o' == current) {
                    state = 3403;
                    return true;
                }
                break;
            case 3403:
                if ('t' == current) {
                    state = 3404;
                    return true;
                }
                break;
            case 3404:
                // bigodot;
                if (';' == current) {
                    match = "\u2A00";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3405:
                if ('l' == current) {
                    state = 3406;
                    return true;
                }
                break;
            case 3406:
                if ('u' == current) {
                    state = 3407;
                    return true;
                }
                break;
            case 3407:
                if ('s' == current) {
                    state = 3408;
                    return true;
                }
                break;
            case 3408:
                // bigoplus;
                if (';' == current) {
                    match = "\u2A01";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3409:
                if ('i' == current) {
                    state = 3410;
                    return true;
                }
                break;
            case 3410:
                if ('m' == current) {
                    state = 3411;
                    return true;
                }
                break;
            case 3411:
                if ('e' == current) {
                    state = 3412;
                    return true;
                }
                break;
            case 3412:
                if ('s' == current) {
                    state = 3413;
                    return true;
                }
                break;
            case 3413:
                // bigotimes;
                if (';' == current) {
                    match = "\u2A02";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3414:
                if ('q' == current) {
                    state = 3415;
                    return true;
                }
                else if ('t' == current) {
                    state = 3419;
                    return true;
                }
                break;
            case 3415:
                if ('c' == current) {
                    state = 3416;
                    return true;
                }
                break;
            case 3416:
                if ('u' == current) {
                    state = 3417;
                    return true;
                }
                break;
            case 3417:
                if ('p' == current) {
                    state = 3418;
                    return true;
                }
                break;
            case 3418:
                // bigsqcup;
                if (';' == current) {
                    match = "\u2A06";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3419:
                if ('a' == current) {
                    state = 3420;
                    return true;
                }
                break;
            case 3420:
                if ('r' == current) {
                    state = 3421;
                    return true;
                }
                break;
            case 3421:
                // bigstar;
                if (';' == current) {
                    match = "\u2605";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3422:
                if ('r' == current) {
                    state = 3423;
                    return true;
                }
                break;
            case 3423:
                if ('i' == current) {
                    state = 3424;
                    return true;
                }
                break;
            case 3424:
                if ('a' == current) {
                    state = 3425;
                    return true;
                }
                break;
            case 3425:
                if ('n' == current) {
                    state = 3426;
                    return true;
                }
                break;
            case 3426:
                if ('g' == current) {
                    state = 3427;
                    return true;
                }
                break;
            case 3427:
                if ('l' == current) {
                    state = 3428;
                    return true;
                }
                break;
            case 3428:
                if ('e' == current) {
                    state = 3429;
                    return true;
                }
                break;
            case 3429:
                if ('d' == current) {
                    state = 3430;
                    return true;
                }
                else if ('u' == current) {
                    state = 3434;
                    return true;
                }
                break;
            case 3430:
                if ('o' == current) {
                    state = 3431;
                    return true;
                }
                break;
            case 3431:
                if ('w' == current) {
                    state = 3432;
                    return true;
                }
                break;
            case 3432:
                if ('n' == current) {
                    state = 3433;
                    return true;
                }
                break;
            case 3433:
                // bigtriangledown;
                if (';' == current) {
                    match = "\u25BD";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3434:
                if ('p' == current) {
                    state = 3435;
                    return true;
                }
                break;
            case 3435:
                // bigtriangleup;
                if (';' == current) {
                    match = "\u25B3";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3436:
                if ('p' == current) {
                    state = 3437;
                    return true;
                }
                break;
            case 3437:
                if ('l' == current) {
                    state = 3438;
                    return true;
                }
                break;
            case 3438:
                if ('u' == current) {
                    state = 3439;
                    return true;
                }
                break;
            case 3439:
                if ('s' == current) {
                    state = 3440;
                    return true;
                }
                break;
            case 3440:
                // biguplus;
                if (';' == current) {
                    match = "\u2A04";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3441:
                if ('e' == current) {
                    state = 3442;
                    return true;
                }
                break;
            case 3442:
                if ('e' == current) {
                    state = 3443;
                    return true;
                }
                break;
            case 3443:
                // bigvee;
                if (';' == current) {
                    match = "\u22C1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3444:
                if ('e' == current) {
                    state = 3445;
                    return true;
                }
                break;
            case 3445:
                if ('d' == current) {
                    state = 3446;
                    return true;
                }
                break;
            case 3446:
                if ('g' == current) {
                    state = 3447;
                    return true;
                }
                break;
            case 3447:
                if ('e' == current) {
                    state = 3448;
                    return true;
                }
                break;
            case 3448:
                // bigwedge;
                if (';' == current) {
                    match = "\u22C0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3449:
                if ('a' == current) {
                    state = 3450;
                    return true;
                }
                break;
            case 3450:
                if ('r' == current) {
                    state = 3451;
                    return true;
                }
                break;
            case 3451:
                if ('o' == current) {
                    state = 3452;
                    return true;
                }
                break;
            case 3452:
                if ('w' == current) {
                    state = 3453;
                    return true;
                }
                break;
            case 3453:
                // bkarow;
                if (';' == current) {
                    match = "\u290D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3454:
                switch (current) {
                    case 'a' :
                        state = 3455;
                        return true;
                    case 'k' :
                        state = 3494;
                        return true;
                    case 'o' :
                        state = 3500;
                        return true;
                }
                break;
            case 3455:
                if ('c' == current) {
                    state = 3456;
                    return true;
                }
                else if ('n' == current) {
                    state = 3492;
                    return true;
                }
                break;
            case 3456:
                if ('k' == current) {
                    state = 3457;
                    return true;
                }
                break;
            case 3457:
                switch (current) {
                    case 'l' :
                        state = 3458;
                        return true;
                    case 's' :
                        state = 3465;
                        return true;
                    case 't' :
                        state = 3471;
                        return true;
                }
                break;
            case 3458:
                if ('o' == current) {
                    state = 3459;
                    return true;
                }
                break;
            case 3459:
                if ('z' == current) {
                    state = 3460;
                    return true;
                }
                break;
            case 3460:
                if ('e' == current) {
                    state = 3461;
                    return true;
                }
                break;
            case 3461:
                if ('n' == current) {
                    state = 3462;
                    return true;
                }
                break;
            case 3462:
                if ('g' == current) {
                    state = 3463;
                    return true;
                }
                break;
            case 3463:
                if ('e' == current) {
                    state = 3464;
                    return true;
                }
                break;
            case 3464:
                // blacklozenge;
                if (';' == current) {
                    match = "\u29EB";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3465:
                if ('q' == current) {
                    state = 3466;
                    return true;
                }
                break;
            case 3466:
                if ('u' == current) {
                    state = 3467;
                    return true;
                }
                break;
            case 3467:
                if ('a' == current) {
                    state = 3468;
                    return true;
                }
                break;
            case 3468:
                if ('r' == current) {
                    state = 3469;
                    return true;
                }
                break;
            case 3469:
                if ('e' == current) {
                    state = 3470;
                    return true;
                }
                break;
            case 3470:
                // blacksquare;
                if (';' == current) {
                    match = "\u25AA";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3471:
                if ('r' == current) {
                    state = 3472;
                    return true;
                }
                break;
            case 3472:
                if ('i' == current) {
                    state = 3473;
                    return true;
                }
                break;
            case 3473:
                if ('a' == current) {
                    state = 3474;
                    return true;
                }
                break;
            case 3474:
                if ('n' == current) {
                    state = 3475;
                    return true;
                }
                break;
            case 3475:
                if ('g' == current) {
                    state = 3476;
                    return true;
                }
                break;
            case 3476:
                if ('l' == current) {
                    state = 3477;
                    return true;
                }
                break;
            case 3477:
                if ('e' == current) {
                    state = 3478;
                    return true;
                }
                break;
            case 3478:
                switch (current) {
                    case ';' : // blacktriangle;
                        match = "\u25B4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'd' :
                        state = 3479;
                        return true;
                    case 'l' :
                        state = 3483;
                        return true;
                    case 'r' :
                        state = 3487;
                        return true;
                }
                break;
            case 3479:
                if ('o' == current) {
                    state = 3480;
                    return true;
                }
                break;
            case 3480:
                if ('w' == current) {
                    state = 3481;
                    return true;
                }
                break;
            case 3481:
                if ('n' == current) {
                    state = 3482;
                    return true;
                }
                break;
            case 3482:
                // blacktriangledown;
                if (';' == current) {
                    match = "\u25BE";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3483:
                if ('e' == current) {
                    state = 3484;
                    return true;
                }
                break;
            case 3484:
                if ('f' == current) {
                    state = 3485;
                    return true;
                }
                break;
            case 3485:
                if ('t' == current) {
                    state = 3486;
                    return true;
                }
                break;
            case 3486:
                // blacktriangleleft;
                if (';' == current) {
                    match = "\u25C2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3487:
                if ('i' == current) {
                    state = 3488;
                    return true;
                }
                break;
            case 3488:
                if ('g' == current) {
                    state = 3489;
                    return true;
                }
                break;
            case 3489:
                if ('h' == current) {
                    state = 3490;
                    return true;
                }
                break;
            case 3490:
                if ('t' == current) {
                    state = 3491;
                    return true;
                }
                break;
            case 3491:
                // blacktriangleright;
                if (';' == current) {
                    match = "\u25B8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3492:
                if ('k' == current) {
                    state = 3493;
                    return true;
                }
                break;
            case 3493:
                // blank;
                if (';' == current) {
                    match = "\u2423";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3494:
                if ('1' == current) {
                    state = 3495;
                    return true;
                }
                else if ('3' == current) {
                    state = 3498;
                    return true;
                }
                break;
            case 3495:
                if ('2' == current) {
                    state = 3496;
                    return true;
                }
                else if ('4' == current) {
                    state = 3497;
                    return true;
                }
                break;
            case 3496:
                // blk12;
                if (';' == current) {
                    match = "\u2592";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3497:
                // blk14;
                if (';' == current) {
                    match = "\u2591";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3498:
                if ('4' == current) {
                    state = 3499;
                    return true;
                }
                break;
            case 3499:
                // blk34;
                if (';' == current) {
                    match = "\u2593";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3500:
                if ('c' == current) {
                    state = 3501;
                    return true;
                }
                break;
            case 3501:
                if ('k' == current) {
                    state = 3502;
                    return true;
                }
                break;
            case 3502:
                // block;
                if (';' == current) {
                    match = "\u2588";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3503:
                if ('e' == current) {
                    state = 3504;
                    return true;
                }
                else if ('o' == current) {
                    state = 3509;
                    return true;
                }
                break;
            case 3504:
                // bne;
                if (';' == current) {
                    match = "=\u20E5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('q' == current) {
                    state = 3505;
                    return true;
                }
                break;
            case 3505:
                if ('u' == current) {
                    state = 3506;
                    return true;
                }
                break;
            case 3506:
                if ('i' == current) {
                    state = 3507;
                    return true;
                }
                break;
            case 3507:
                if ('v' == current) {
                    state = 3508;
                    return true;
                }
                break;
            case 3508:
                // bnequiv;
                if (';' == current) {
                    match = "\u2261\u20E5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3509:
                if ('t' == current) {
                    state = 3510;
                    return true;
                }
                break;
            case 3510:
                // bnot;
                if (';' == current) {
                    match = "\u2310";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3511:
                switch (current) {
                    case 'p' :
                        state = 3512;
                        return true;
                    case 't' :
                        state = 3514;
                        return true;
                    case 'w' :
                        state = 3518;
                        return true;
                    case 'x' :
                        state = 3522;
                        return true;
                }
                break;
            case 3512:
                if ('f' == current) {
                    state = 3513;
                    return true;
                }
                break;
            case 3513:
                // bopf;
                if (';' == current) {
                    match = "\uD835\uDD53";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3514:
                // bot;
                if (';' == current) {
                    match = "\u22A5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('t' == current) {
                    state = 3515;
                    return true;
                }
                break;
            case 3515:
                if ('o' == current) {
                    state = 3516;
                    return true;
                }
                break;
            case 3516:
                if ('m' == current) {
                    state = 3517;
                    return true;
                }
                break;
            case 3517:
                // bottom;
                if (';' == current) {
                    match = "\u22A5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3518:
                if ('t' == current) {
                    state = 3519;
                    return true;
                }
                break;
            case 3519:
                if ('i' == current) {
                    state = 3520;
                    return true;
                }
                break;
            case 3520:
                if ('e' == current) {
                    state = 3521;
                    return true;
                }
                break;
            case 3521:
                // bowtie;
                if (';' == current) {
                    match = "\u22C8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3522:
                switch (current) {
                    case 'D' :
                        state = 3523;
                        return true;
                    case 'H' :
                        state = 3528;
                        return true;
                    case 'U' :
                        state = 3533;
                        return true;
                    case 'V' :
                        state = 3538;
                        return true;
                    case 'b' :
                        state = 3545;
                        return true;
                    case 'd' :
                        state = 3548;
                        return true;
                    case 'h' :
                        state = 3553;
                        return true;
                    case 'm' :
                        state = 3558;
                        return true;
                    case 'p' :
                        state = 3563;
                        return true;
                    case 't' :
                        state = 3567;
                        return true;
                    case 'u' :
                        state = 3572;
                        return true;
                    case 'v' :
                        state = 3577;
                        return true;
                }
                break;
            case 3523:
                switch (current) {
                    case 'L' :
                        state = 3524;
                        return true;
                    case 'R' :
                        state = 3525;
                        return true;
                    case 'l' :
                        state = 3526;
                        return true;
                    case 'r' :
                        state = 3527;
                        return true;
                }
                break;
            case 3524:
                // boxDL;
                if (';' == current) {
                    match = "\u2557";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3525:
                // boxDR;
                if (';' == current) {
                    match = "\u2554";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3526:
                // boxDl;
                if (';' == current) {
                    match = "\u2556";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3527:
                // boxDr;
                if (';' == current) {
                    match = "\u2553";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3528:
                switch (current) {
                    case ';' : // boxH;
                        match = "\u2550";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'D' :
                        state = 3529;
                        return true;
                    case 'U' :
                        state = 3530;
                        return true;
                    case 'd' :
                        state = 3531;
                        return true;
                    case 'u' :
                        state = 3532;
                        return true;
                }
                break;
            case 3529:
                // boxHD;
                if (';' == current) {
                    match = "\u2566";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3530:
                // boxHU;
                if (';' == current) {
                    match = "\u2569";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3531:
                // boxHd;
                if (';' == current) {
                    match = "\u2564";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3532:
                // boxHu;
                if (';' == current) {
                    match = "\u2567";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3533:
                switch (current) {
                    case 'L' :
                        state = 3534;
                        return true;
                    case 'R' :
                        state = 3535;
                        return true;
                    case 'l' :
                        state = 3536;
                        return true;
                    case 'r' :
                        state = 3537;
                        return true;
                }
                break;
            case 3534:
                // boxUL;
                if (';' == current) {
                    match = "\u255D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3535:
                // boxUR;
                if (';' == current) {
                    match = "\u255A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3536:
                // boxUl;
                if (';' == current) {
                    match = "\u255C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3537:
                // boxUr;
                if (';' == current) {
                    match = "\u2559";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3538:
                switch (current) {
                    case ';' : // boxV;
                        match = "\u2551";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'H' :
                        state = 3539;
                        return true;
                    case 'L' :
                        state = 3540;
                        return true;
                    case 'R' :
                        state = 3541;
                        return true;
                    case 'h' :
                        state = 3542;
                        return true;
                    case 'l' :
                        state = 3543;
                        return true;
                    case 'r' :
                        state = 3544;
                        return true;
                }
                break;
            case 3539:
                // boxVH;
                if (';' == current) {
                    match = "\u256C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3540:
                // boxVL;
                if (';' == current) {
                    match = "\u2563";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3541:
                // boxVR;
                if (';' == current) {
                    match = "\u2560";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3542:
                // boxVh;
                if (';' == current) {
                    match = "\u256B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3543:
                // boxVl;
                if (';' == current) {
                    match = "\u2562";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3544:
                // boxVr;
                if (';' == current) {
                    match = "\u255F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3545:
                if ('o' == current) {
                    state = 3546;
                    return true;
                }
                break;
            case 3546:
                if ('x' == current) {
                    state = 3547;
                    return true;
                }
                break;
            case 3547:
                // boxbox;
                if (';' == current) {
                    match = "\u29C9";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3548:
                switch (current) {
                    case 'L' :
                        state = 3549;
                        return true;
                    case 'R' :
                        state = 3550;
                        return true;
                    case 'l' :
                        state = 3551;
                        return true;
                    case 'r' :
                        state = 3552;
                        return true;
                }
                break;
            case 3549:
                // boxdL;
                if (';' == current) {
                    match = "\u2555";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3550:
                // boxdR;
                if (';' == current) {
                    match = "\u2552";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3551:
                // boxdl;
                if (';' == current) {
                    match = "\u2510";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3552:
                // boxdr;
                if (';' == current) {
                    match = "\u250C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3553:
                switch (current) {
                    case ';' : // boxh;
                        match = "\u2500";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'D' :
                        state = 3554;
                        return true;
                    case 'U' :
                        state = 3555;
                        return true;
                    case 'd' :
                        state = 3556;
                        return true;
                    case 'u' :
                        state = 3557;
                        return true;
                }
                break;
            case 3554:
                // boxhD;
                if (';' == current) {
                    match = "\u2565";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3555:
                // boxhU;
                if (';' == current) {
                    match = "\u2568";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3556:
                // boxhd;
                if (';' == current) {
                    match = "\u252C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3557:
                // boxhu;
                if (';' == current) {
                    match = "\u2534";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3558:
                if ('i' == current) {
                    state = 3559;
                    return true;
                }
                break;
            case 3559:
                if ('n' == current) {
                    state = 3560;
                    return true;
                }
                break;
            case 3560:
                if ('u' == current) {
                    state = 3561;
                    return true;
                }
                break;
            case 3561:
                if ('s' == current) {
                    state = 3562;
                    return true;
                }
                break;
            case 3562:
                // boxminus;
                if (';' == current) {
                    match = "\u229F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3563:
                if ('l' == current) {
                    state = 3564;
                    return true;
                }
                break;
            case 3564:
                if ('u' == current) {
                    state = 3565;
                    return true;
                }
                break;
            case 3565:
                if ('s' == current) {
                    state = 3566;
                    return true;
                }
                break;
            case 3566:
                // boxplus;
                if (';' == current) {
                    match = "\u229E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3567:
                if ('i' == current) {
                    state = 3568;
                    return true;
                }
                break;
            case 3568:
                if ('m' == current) {
                    state = 3569;
                    return true;
                }
                break;
            case 3569:
                if ('e' == current) {
                    state = 3570;
                    return true;
                }
                break;
            case 3570:
                if ('s' == current) {
                    state = 3571;
                    return true;
                }
                break;
            case 3571:
                // boxtimes;
                if (';' == current) {
                    match = "\u22A0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3572:
                switch (current) {
                    case 'L' :
                        state = 3573;
                        return true;
                    case 'R' :
                        state = 3574;
                        return true;
                    case 'l' :
                        state = 3575;
                        return true;
                    case 'r' :
                        state = 3576;
                        return true;
                }
                break;
            case 3573:
                // boxuL;
                if (';' == current) {
                    match = "\u255B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3574:
                // boxuR;
                if (';' == current) {
                    match = "\u2558";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3575:
                // boxul;
                if (';' == current) {
                    match = "\u2518";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3576:
                // boxur;
                if (';' == current) {
                    match = "\u2514";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3577:
                switch (current) {
                    case ';' : // boxv;
                        match = "\u2502";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'H' :
                        state = 3578;
                        return true;
                    case 'L' :
                        state = 3579;
                        return true;
                    case 'R' :
                        state = 3580;
                        return true;
                    case 'h' :
                        state = 3581;
                        return true;
                    case 'l' :
                        state = 3582;
                        return true;
                    case 'r' :
                        state = 3583;
                        return true;
                }
                break;
            case 3578:
                // boxvH;
                if (';' == current) {
                    match = "\u256A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3579:
                // boxvL;
                if (';' == current) {
                    match = "\u2561";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3580:
                // boxvR;
                if (';' == current) {
                    match = "\u255E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3581:
                // boxvh;
                if (';' == current) {
                    match = "\u253C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3582:
                // boxvl;
                if (';' == current) {
                    match = "\u2524";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3583:
                // boxvr;
                if (';' == current) {
                    match = "\u251C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3584:
                if ('r' == current) {
                    state = 3585;
                    return true;
                }
                break;
            case 3585:
                if ('i' == current) {
                    state = 3586;
                    return true;
                }
                break;
            case 3586:
                if ('m' == current) {
                    state = 3587;
                    return true;
                }
                break;
            case 3587:
                if ('e' == current) {
                    state = 3588;
                    return true;
                }
                break;
            case 3588:
                // bprime;
                if (';' == current) {
                    match = "\u2035";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3589:
                if ('e' == current) {
                    state = 3590;
                    return true;
                }
                else if ('v' == current) {
                    state = 3593;
                    return true;
                }
                break;
            case 3590:
                if ('v' == current) {
                    state = 3591;
                    return true;
                }
                break;
            case 3591:
                if ('e' == current) {
                    state = 3592;
                    return true;
                }
                break;
            case 3592:
                // breve;
                if (';' == current) {
                    match = "\u02D8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3593:
                if ('b' == current) {
                    state = 3594;
                    return true;
                }
                break;
            case 3594:
                if ('a' == current) {
                    state = 3595;
                    return true;
                }
                break;
            case 3595:
                // brvbar
                if ('r' == current) {
                    match = "\u00A6";
                    matchLength = consumedCount;
                    state = 3596;
                    return true;
                }
                break;
            case 3596:
                // brvbar;
                if (';' == current) {
                    match = "\u00A6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3597:
                switch (current) {
                    case 'c' :
                        state = 3598;
                        return true;
                    case 'e' :
                        state = 3600;
                        return true;
                    case 'i' :
                        state = 3603;
                        return true;
                    case 'o' :
                        state = 3606;
                        return true;
                }
                break;
            case 3598:
                if ('r' == current) {
                    state = 3599;
                    return true;
                }
                break;
            case 3599:
                // bscr;
                if (';' == current) {
                    match = "\uD835\uDCB7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3600:
                if ('m' == current) {
                    state = 3601;
                    return true;
                }
                break;
            case 3601:
                if ('i' == current) {
                    state = 3602;
                    return true;
                }
                break;
            case 3602:
                // bsemi;
                if (';' == current) {
                    match = "\u204F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3603:
                if ('m' == current) {
                    state = 3604;
                    return true;
                }
                break;
            case 3604:
                // bsim;
                if (';' == current) {
                    match = "\u223D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state = 3605;
                    return true;
                }
                break;
            case 3605:
                // bsime;
                if (';' == current) {
                    match = "\u22CD";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3606:
                if ('l' == current) {
                    state = 3607;
                    return true;
                }
                break;
            case 3607:
                switch (current) {
                    case ';' : // bsol;
                        match = "\\";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'b' :
                        state = 3608;
                        return true;
                    case 'h' :
                        state = 3609;
                        return true;
                }
                break;
            case 3608:
                // bsolb;
                if (';' == current) {
                    match = "\u29C5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3609:
                if ('s' == current) {
                    state = 3610;
                    return true;
                }
                break;
            case 3610:
                if ('u' == current) {
                    state = 3611;
                    return true;
                }
                break;
            case 3611:
                if ('b' == current) {
                    state = 3612;
                    return true;
                }
                break;
            case 3612:
                // bsolhsub;
                if (';' == current) {
                    match = "\u27C8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3613:
                if ('l' == current) {
                    state = 3614;
                    return true;
                }
                else if ('m' == current) {
                    state = 3618;
                    return true;
                }
                break;
            case 3614:
                if ('l' == current) {
                    state = 3615;
                    return true;
                }
                break;
            case 3615:
                // bull;
                if (';' == current) {
                    match = "\u2022";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state = 3616;
                    return true;
                }
                break;
            case 3616:
                if ('t' == current) {
                    state = 3617;
                    return true;
                }
                break;
            case 3617:
                // bullet;
                if (';' == current) {
                    match = "\u2022";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3618:
                if ('p' == current) {
                    state = 3619;
                    return true;
                }
                break;
            case 3619:
                switch (current) {
                    case ';' : // bump;
                        match = "\u224E";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 3620;
                        return true;
                    case 'e' :
                        state = 3621;
                        return true;
                }
                break;
            case 3620:
                // bumpE;
                if (';' == current) {
                    match = "\u2AAE";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3621:
                // bumpe;
                if (';' == current) {
                    match = "\u224F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('q' == current) {
                    state = 3622;
                    return true;
                }
                break;
            case 3622:
                // bumpeq;
                if (';' == current) {
                    match = "\u224F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3623:
                switch (current) {
                    case 'a' :
                        state = 3624;
                        return true;
                    case 'c' :
                        state = 3652;
                        return true;
                    case 'd' :
                        state = 3671;
                        return true;
                    case 'e' :
                        state = 3674;
                        return true;
                    case 'f' :
                        state = 3690;
                        return true;
                    case 'h' :
                        state = 3692;
                        return true;
                    case 'i' :
                        state = 3703;
                        return true;
                    case 'l' :
                        state = 3752;
                        return true;
                    case 'o' :
                        state = 3759;
                        return true;
                    case 'r' :
                        state = 3797;
                        return true;
                    case 's' :
                        state = 3804;
                        return true;
                    case 't' :
                        state = 3812;
                        return true;
                    case 'u' :
                        state = 3816;
                        return true;
                    case 'w' :
                        state = 3900;
                        return true;
                    case 'y' :
                        state = 3910;
                        return true;
                }
                break;
            case 3624:
                switch (current) {
                    case 'c' :
                        state = 3625;
                        return true;
                    case 'p' :
                        state = 3629;
                        return true;
                    case 'r' :
                        state = 3647;
                        return true;
                }
                break;
            case 3625:
                if ('u' == current) {
                    state = 3626;
                    return true;
                }
                break;
            case 3626:
                if ('t' == current) {
                    state = 3627;
                    return true;
                }
                break;
            case 3627:
                if ('e' == current) {
                    state = 3628;
                    return true;
                }
                break;
            case 3628:
                // cacute;
                if (';' == current) {
                    match = "\u0107";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3629:
                switch (current) {
                    case ';' : // cap;
                        match = "\u2229";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'a' :
                        state = 3630;
                        return true;
                    case 'b' :
                        state = 3633;
                        return true;
                    case 'c' :
                        state = 3638;
                        return true;
                    case 'd' :
                        state = 3643;
                        return true;
                    case 's' :
                        state = 3646;
                        return true;
                }
                break;
            case 3630:
                if ('n' == current) {
                    state = 3631;
                    return true;
                }
                break;
            case 3631:
                if ('d' == current) {
                    state = 3632;
                    return true;
                }
                break;
            case 3632:
                // capand;
                if (';' == current) {
                    match = "\u2A44";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3633:
                if ('r' == current) {
                    state = 3634;
                    return true;
                }
                break;
            case 3634:
                if ('c' == current) {
                    state = 3635;
                    return true;
                }
                break;
            case 3635:
                if ('u' == current) {
                    state = 3636;
                    return true;
                }
                break;
            case 3636:
                if ('p' == current) {
                    state = 3637;
                    return true;
                }
                break;
            case 3637:
                // capbrcup;
                if (';' == current) {
                    match = "\u2A49";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3638:
                if ('a' == current) {
                    state = 3639;
                    return true;
                }
                else if ('u' == current) {
                    state = 3641;
                    return true;
                }
                break;
            case 3639:
                if ('p' == current) {
                    state = 3640;
                    return true;
                }
                break;
            case 3640:
                // capcap;
                if (';' == current) {
                    match = "\u2A4B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3641:
                if ('p' == current) {
                    state = 3642;
                    return true;
                }
                break;
            case 3642:
                // capcup;
                if (';' == current) {
                    match = "\u2A47";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3643:
                if ('o' == current) {
                    state = 3644;
                    return true;
                }
                break;
            case 3644:
                if ('t' == current) {
                    state = 3645;
                    return true;
                }
                break;
            case 3645:
                // capdot;
                if (';' == current) {
                    match = "\u2A40";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3646:
                // caps;
                if (';' == current) {
                    match = "\u2229\uFE00";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3647:
                if ('e' == current) {
                    state = 3648;
                    return true;
                }
                else if ('o' == current) {
                    state = 3650;
                    return true;
                }
                break;
            case 3648:
                if ('t' == current) {
                    state = 3649;
                    return true;
                }
                break;
            case 3649:
                // caret;
                if (';' == current) {
                    match = "\u2041";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3650:
                if ('n' == current) {
                    state = 3651;
                    return true;
                }
                break;
            case 3651:
                // caron;
                if (';' == current) {
                    match = "\u02C7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3652:
                switch (current) {
                    case 'a' :
                        state = 3653;
                        return true;
                    case 'e' :
                        state = 3659;
                        return true;
                    case 'i' :
                        state = 3663;
                        return true;
                    case 'u' :
                        state = 3666;
                        return true;
                }
                break;
            case 3653:
                if ('p' == current) {
                    state = 3654;
                    return true;
                }
                else if ('r' == current) {
                    state = 3656;
                    return true;
                }
                break;
            case 3654:
                if ('s' == current) {
                    state = 3655;
                    return true;
                }
                break;
            case 3655:
                // ccaps;
                if (';' == current) {
                    match = "\u2A4D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3656:
                if ('o' == current) {
                    state = 3657;
                    return true;
                }
                break;
            case 3657:
                if ('n' == current) {
                    state = 3658;
                    return true;
                }
                break;
            case 3658:
                // ccaron;
                if (';' == current) {
                    match = "\u010D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3659:
                if ('d' == current) {
                    state = 3660;
                    return true;
                }
                break;
            case 3660:
                if ('i' == current) {
                    state = 3661;
                    return true;
                }
                break;
            case 3661:
                // ccedil
                if ('l' == current) {
                    match = "\u00E7";
                    matchLength = consumedCount;
                    state = 3662;
                    return true;
                }
                break;
            case 3662:
                // ccedil;
                if (';' == current) {
                    match = "\u00E7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3663:
                if ('r' == current) {
                    state = 3664;
                    return true;
                }
                break;
            case 3664:
                if ('c' == current) {
                    state = 3665;
                    return true;
                }
                break;
            case 3665:
                // ccirc;
                if (';' == current) {
                    match = "\u0109";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3666:
                if ('p' == current) {
                    state = 3667;
                    return true;
                }
                break;
            case 3667:
                if ('s' == current) {
                    state = 3668;
                    return true;
                }
                break;
            case 3668:
                // ccups;
                if (';' == current) {
                    match = "\u2A4C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('s' == current) {
                    state = 3669;
                    return true;
                }
                break;
            case 3669:
                if ('m' == current) {
                    state = 3670;
                    return true;
                }
                break;
            case 3670:
                // ccupssm;
                if (';' == current) {
                    match = "\u2A50";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3671:
                if ('o' == current) {
                    state = 3672;
                    return true;
                }
                break;
            case 3672:
                if ('t' == current) {
                    state = 3673;
                    return true;
                }
                break;
            case 3673:
                // cdot;
                if (';' == current) {
                    match = "\u010B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3674:
                switch (current) {
                    case 'd' :
                        state = 3675;
                        return true;
                    case 'm' :
                        state = 3678;
                        return true;
                    case 'n' :
                        state = 3683;
                        return true;
                }
                break;
            case 3675:
                if ('i' == current) {
                    state = 3676;
                    return true;
                }
                break;
            case 3676:
                // cedil
                if ('l' == current) {
                    match = "\u00B8";
                    matchLength = consumedCount;
                    state = 3677;
                    return true;
                }
                break;
            case 3677:
                // cedil;
                if (';' == current) {
                    match = "\u00B8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3678:
                if ('p' == current) {
                    state = 3679;
                    return true;
                }
                break;
            case 3679:
                if ('t' == current) {
                    state = 3680;
                    return true;
                }
                break;
            case 3680:
                if ('y' == current) {
                    state = 3681;
                    return true;
                }
                break;
            case 3681:
                if ('v' == current) {
                    state = 3682;
                    return true;
                }
                break;
            case 3682:
                // cemptyv;
                if (';' == current) {
                    match = "\u29B2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3683:
                // cent
                if ('t' == current) {
                    match = "\u00A2";
                    matchLength = consumedCount;
                    state = 3684;
                    return true;
                }
                break;
            case 3684:
                // cent;
                if (';' == current) {
                    match = "\u00A2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state = 3685;
                    return true;
                }
                break;
            case 3685:
                if ('r' == current) {
                    state = 3686;
                    return true;
                }
                break;
            case 3686:
                if ('d' == current) {
                    state = 3687;
                    return true;
                }
                break;
            case 3687:
                if ('o' == current) {
                    state = 3688;
                    return true;
                }
                break;
            case 3688:
                if ('t' == current) {
                    state = 3689;
                    return true;
                }
                break;
            case 3689:
                // centerdot;
                if (';' == current) {
                    match = "\u00B7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3690:
                if ('r' == current) {
                    state = 3691;
                    return true;
                }
                break;
            case 3691:
                // cfr;
                if (';' == current) {
                    match = "\uD835\uDD20";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3692:
                switch (current) {
                    case 'c' :
                        state = 3693;
                        return true;
                    case 'e' :
                        state = 3695;
                        return true;
                    case 'i' :
                        state = 3702;
                        return true;
                }
                break;
            case 3693:
                if ('y' == current) {
                    state = 3694;
                    return true;
                }
                break;
            case 3694:
                // chcy;
                if (';' == current) {
                    match = "\u0447";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3695:
                if ('c' == current) {
                    state = 3696;
                    return true;
                }
                break;
            case 3696:
                if ('k' == current) {
                    state = 3697;
                    return true;
                }
                break;
            case 3697:
                // check;
                if (';' == current) {
                    match = "\u2713";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('m' == current) {
                    state = 3698;
                    return true;
                }
                break;
            case 3698:
                if ('a' == current) {
                    state = 3699;
                    return true;
                }
                break;
            case 3699:
                if ('r' == current) {
                    state = 3700;
                    return true;
                }
                break;
            case 3700:
                if ('k' == current) {
                    state = 3701;
                    return true;
                }
                break;
            case 3701:
                // checkmark;
                if (';' == current) {
                    match = "\u2713";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3702:
                // chi;
                if (';' == current) {
                    match = "\u03C7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3703:
                if ('r' == current) {
                    state = 3704;
                    return true;
                }
                break;
            case 3704:
                switch (current) {
                    case ';' : // cir;
                        match = "\u25CB";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E' :
                        state = 3705;
                        return true;
                    case 'c' :
                        state = 3706;
                        return true;
                    case 'e' :
                        state = 3739;
                        return true;
                    case 'f' :
                        state = 3740;
                        return true;
                    case 'm' :
                        state = 3745;
                        return true;
                    case 's' :
                        state = 3748;
                        return true;
                }
                break;
            case 3705:
                // cirE;
                if (';' == current) {
                    match = "\u29C3";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3706:
                switch (current) {
                    case ';' : // circ;
                        match = "\u02C6";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 3707;
                        return true;
                    case 'l' :
                        state = 3709;
                        return true;
                }
                break;
            case 3707:
                if ('q' == current) {
                    state = 3708;
                    return true;
                }
                break;
            case 3708:
                // circeq;
                if (';' == current) {
                    match = "\u2257";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3709:
                if ('e' == current) {
                    state = 3710;
                    return true;
                }
                break;
            case 3710:
                if ('a' == current) {
                    state = 3711;
                    return true;
                }
                else if ('d' == current) {
                    state = 3725;
                    return true;
                }
                break;
            case 3711:
                if ('r' == current) {
                    state = 3712;
                    return true;
                }
                break;
            case 3712:
                if ('r' == current) {
                    state = 3713;
                    return true;
                }
                break;
            case 3713:
                if ('o' == current) {
                    state = 3714;
                    return true;
                }
                break;
            case 3714:
                if ('w' == current) {
                    state = 3715;
                    return true;
                }
                break;
            case 3715:
                if ('l' == current) {
                    state = 3716;
                    return true;
                }
                else if ('r' == current) {
                    state = 3720;
                    return true;
                }
                break;
            case 3716:
                if ('e' == current) {
                    state = 3717;
                    return true;
                }
                break;
            case 3717:
                if ('f' == current) {
                    state = 3718;
                    return true;
                }
                break;
            case 3718:
                if ('t' == current) {
                    state = 3719;
                    return true;
                }
                break;
            case 3719:
                // circlearrowleft;
                if (';' == current) {
                    match = "\u21BA";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3720:
                if ('i' == current) {
                    state = 3721;
                    return true;
                }
                break;
            case 3721:
                if ('g' == current) {
                    state = 3722;
                    return true;
                }
                break;
            case 3722:
                if ('h' == current) {
                    state = 3723;
                    return true;
                }
                break;
            case 3723:
                if ('t' == current) {
                    state = 3724;
                    return true;
                }
                break;
            case 3724:
                // circlearrowright;
                if (';' == current) {
                    match = "\u21BB";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3725:
                switch (current) {
                    case 'R' :
                        state = 3726;
                        return true;
                    case 'S' :
                        state = 3727;
                        return true;
                    case 'a' :
                        state = 3728;
                        return true;
                    case 'c' :
                        state = 3731;
                        return true;
                    case 'd' :
                        state = 3735;
                        return true;
                }
                break;
            case 3726:
                // circledR;
                if (';' == current) {
                    match = "\u00AE";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3727:
                // circledS;
                if (';' == current) {
                    match = "\u24C8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3728:
                if ('s' == current) {
                    state = 3729;
                    return true;
                }
                break;
            case 3729:
                if ('t' == current) {
                    state = 3730;
                    return true;
                }
                break;
            case 3730:
                // circledast;
                if (';' == current) {
                    match = "\u229B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3731:
                if ('i' == current) {
                    state = 3732;
                    return true;
                }
                break;
            case 3732:
                if ('r' == current) {
                    state = 3733;
                    return true;
                }
                break;
            case 3733:
                if ('c' == current) {
                    state = 3734;
                    return true;
                }
                break;
            case 3734:
                // circledcirc;
                if (';' == current) {
                    match = "\u229A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3735:
                if ('a' == current) {
                    state = 3736;
                    return true;
                }
                break;
            case 3736:
                if ('s' == current) {
                    state = 3737;
                    return true;
                }
                break;
            case 3737:
                if ('h' == current) {
                    state = 3738;
                    return true;
                }
                break;
            case 3738:
                // circleddash;
                if (';' == current) {
                    match = "\u229D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3739:
                // cire;
                if (';' == current) {
                    match = "\u2257";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3740:
                if ('n' == current) {
                    state = 3741;
                    return true;
                }
                break;
            case 3741:
                if ('i' == current) {
                    state = 3742;
                    return true;
                }
                break;
            case 3742:
                if ('n' == current) {
                    state = 3743;
                    return true;
                }
                break;
            case 3743:
                if ('t' == current) {
                    state = 3744;
                    return true;
                }
                break;
            case 3744:
                // cirfnint;
                if (';' == current) {
                    match = "\u2A10";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3745:
                if ('i' == current) {
                    state = 3746;
                    return true;
                }
                break;
            case 3746:
                if ('d' == current) {
                    state = 3747;
                    return true;
                }
                break;
            case 3747:
                // cirmid;
                if (';' == current) {
                    match = "\u2AEF";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3748:
                if ('c' == current) {
                    state = 3749;
                    return true;
                }
                break;
            case 3749:
                if ('i' == current) {
                    state = 3750;
                    return true;
                }
                break;
            case 3750:
                if ('r' == current) {
                    state = 3751;
                    return true;
                }
                break;
            case 3751:
                // cirscir;
                if (';' == current) {
                    match = "\u29C2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3752:
                if ('u' == current) {
                    state = 3753;
                    return true;
                }
                break;
            case 3753:
                if ('b' == current) {
                    state = 3754;
                    return true;
                }
                break;
            case 3754:
                if ('s' == current) {
                    state = 3755;
                    return true;
                }
                break;
            case 3755:
                // clubs;
                if (';' == current) {
                    match = "\u2663";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('u' == current) {
                    state = 3756;
                    return true;
                }
                break;
            case 3756:
                if ('i' == current) {
                    state = 3757;
                    return true;
                }
                break;
            case 3757:
                if ('t' == current) {
                    state = 3758;
                    return true;
                }
                break;
            case 3758:
                // clubsuit;
                if (';' == current) {
                    match = "\u2663";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3759:
                switch (current) {
                    case 'l' :
                        state = 3760;
                        return true;
                    case 'm' :
                        state = 3765;
                        return true;
                    case 'n' :
                        state = 3781;
                        return true;
                    case 'p' :
                        state = 3789;
                        return true;
                }
                break;
            case 3760:
                if ('o' == current) {
                    state = 3761;
                    return true;
                }
                break;
            case 3761:
                if ('n' == current) {
                    state = 3762;
                    return true;
                }
                break;
            case 3762:
                // colon;
                if (';' == current) {
                    match = ":";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state = 3763;
                    return true;
                }
                break;
            case 3763:
                // colone;
                if (';' == current) {
                    match = "\u2254";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('q' == current) {
                    state = 3764;
                    return true;
                }
                break;
            case 3764:
                // coloneq;
                if (';' == current) {
                    match = "\u2254";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3765:
                if ('m' == current) {
                    state = 3766;
                    return true;
                }
                else if ('p' == current) {
                    state = 3769;
                    return true;
                }
                break;
            case 3766:
                if ('a' == current) {
                    state = 3767;
                    return true;
                }
                break;
            case 3767:
                // comma;
                if (';' == current) {
                    match = ",";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('t' == current) {
                    state = 3768;
                    return true;
                }
                break;
            case 3768:
                // commat;
                if (';' == current) {
                    match = "@";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3769:
                switch (current) {
                    case ';' : // comp;
                        match = "\u2201";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'f' :
                        state = 3770;
                        return true;
                    case 'l' :
                        state = 3772;
                        return true;
                }
                break;
            case 3770:
                if ('n' == current) {
                    state = 3771;
                    return true;
                }
                break;
            case 3771:
                // compfn;
                if (';' == current) {
                    match = "\u2218";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3772:
                if ('e' == current) {
                    state = 3773;
                    return true;
                }
                break;
            case 3773:
                if ('m' == current) {
                    state = 3774;
                    return true;
                }
                else if ('x' == current) {
                    state = 3778;
                    return true;
                }
                break;
            case 3774:
                if ('e' == current) {
                    state = 3775;
                    return true;
                }
                break;
            case 3775:
                if ('n' == current) {
                    state = 3776;
                    return true;
                }
                break;
            case 3776:
                if ('t' == current) {
                    state = 3777;
                    return true;
                }
                break;
            case 3777:
                // complement;
                if (';' == current) {
                    match = "\u2201";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3778:
                if ('e' == current) {
                    state = 3779;
                    return true;
                }
                break;
            case 3779:
                if ('s' == current) {
                    state = 3780;
                    return true;
                }
                break;
            case 3780:
                // complexes;
                if (';' == current) {
                    match = "\u2102";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3781:
                if ('g' == current) {
                    state = 3782;
                    return true;
                }
                else if ('i' == current) {
                    state = 3786;
                    return true;
                }
                break;
            case 3782:
                // cong;
                if (';' == current) {
                    match = "\u2245";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('d' == current) {
                    state = 3783;
                    return true;
                }
                break;
            case 3783:
                if ('o' == current) {
                    state = 3784;
                    return true;
                }
                break;
            case 3784:
                if ('t' == current) {
                    state = 3785;
                    return true;
                }
                break;
            case 3785:
                // congdot;
                if (';' == current) {
                    match = "\u2A6D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3786:
                if ('n' == current) {
                    state = 3787;
                    return true;
                }
                break;
            case 3787:
                if ('t' == current) {
                    state = 3788;
                    return true;
                }
                break;
            case 3788:
                // conint;
                if (';' == current) {
                    match = "\u222E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3789:
                switch (current) {
                    case 'f' :
                        state = 3790;
                        return true;
                    case 'r' :
                        state = 3791;
                        return true;
                    case 'y' : // copy
                        match = "\u00A9";
                        matchLength = consumedCount;
                        state = 3794;
                        return true;
                }
                break;
            case 3790:
                // copf;
                if (';' == current) {
                    match = "\uD835\uDD54";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3791:
                if ('o' == current) {
                    state = 3792;
                    return true;
                }
                break;
            case 3792:
                if ('d' == current) {
                    state = 3793;
                    return true;
                }
                break;
            case 3793:
                // coprod;
                if (';' == current) {
                    match = "\u2210";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3794:
                // copy;
                if (';' == current) {
                    match = "\u00A9";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('s' == current) {
                    state = 3795;
                    return true;
                }
                break;
            case 3795:
                if ('r' == current) {
                    state = 3796;
                    return true;
                }
                break;
            case 3796:
                // copysr;
                if (';' == current) {
                    match = "\u2117";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3797:
                if ('a' == current) {
                    state = 3798;
                    return true;
                }
                else if ('o' == current) {
                    state = 3801;
                    return true;
                }
                break;
            case 3798:
                if ('r' == current) {
                    state = 3799;
                    return true;
                }
                break;
            case 3799:
                if ('r' == current) {
                    state = 3800;
                    return true;
                }
                break;
            case 3800:
                // crarr;
                if (';' == current) {
                    match = "\u21B5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3801:
                if ('s' == current) {
                    state = 3802;
                    return true;
                }
                break;
            case 3802:
                if ('s' == current) {
                    state = 3803;
                    return true;
                }
                break;
            case 3803:
                // cross;
                if (';' == current) {
                    match = "\u2717";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3804:
                if ('c' == current) {
                    state = 3805;
                    return true;
                }
                else if ('u' == current) {
                    state = 3807;
                    return true;
                }
                break;
            case 3805:
                if ('r' == current) {
                    state = 3806;
                    return true;
                }
                break;
            case 3806:
                // cscr;
                if (';' == current) {
                    match = "\uD835\uDCB8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3807:
                if ('b' == current) {
                    state = 3808;
                    return true;
                }
                else if ('p' == current) {
                    state = 3810;
                    return true;
                }
                break;
            case 3808:
                // csub;
                if (';' == current) {
                    match = "\u2ACF";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state = 3809;
                    return true;
                }
                break;
            case 3809:
                // csube;
                if (';' == current) {
                    match = "\u2AD1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3810:
                // csup;
                if (';' == current) {
                    match = "\u2AD0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state = 3811;
                    return true;
                }
                break;
            case 3811:
                // csupe;
                if (';' == current) {
                    match = "\u2AD2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3812:
                if ('d' == current) {
                    state = 3813;
                    return true;
                }
                break;
            case 3813:
                if ('o' == current) {
                    state = 3814;
                    return true;
                }
                break;
            case 3814:
                if ('t' == current) {
                    state = 3815;
                    return true;
                }
                break;
            case 3815:
                // ctdot;
                if (';' == current) {
                    match = "\u22EF";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3816:
                switch (current) {
                    case 'd' :
                        state = 3817;
                        return true;
                    case 'e' :
                        state = 3823;
                        return true;
                    case 'l' :
                        state = 3828;
                        return true;
                    case 'p' :
                        state = 3833;
                        return true;
                    case 'r' :
                        state = 3850;
                        return true;
                    case 'v' :
                        state = 3894;
                        return true;
                    case 'w' :
                        state = 3897;
                        return true;
                }
                break;
            case 3817:
                if ('a' == current) {
                    state = 3818;
                    return true;
                }
                break;
            case 3818:
                if ('r' == current) {
                    state = 3819;
                    return true;
                }
                break;
            case 3819:
                if ('r' == current) {
                    state = 3820;
                    return true;
                }
                break;
            case 3820:
                if ('l' == current) {
                    state = 3821;
                    return true;
                }
                else if ('r' == current) {
                    state = 3822;
                    return true;
                }
                break;
            case 3821:
                // cudarrl;
                if (';' == current) {
                    match = "\u2938";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3822:
                // cudarrr;
                if (';' == current) {
                    match = "\u2935";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3823:
                if ('p' == current) {
                    state = 3824;
                    return true;
                }
                else if ('s' == current) {
                    state = 3826;
                    return true;
                }
                break;
            case 3824:
                if ('r' == current) {
                    state = 3825;
                    return true;
                }
                break;
            case 3825:
                // cuepr;
                if (';' == current) {
                    match = "\u22DE";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3826:
                if ('c' == current) {
                    state = 3827;
                    return true;
                }
                break;
            case 3827:
                // cuesc;
                if (';' == current) {
                    match = "\u22DF";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3828:
                if ('a' == current) {
                    state = 3829;
                    return true;
                }
                break;
            case 3829:
                if ('r' == current) {
                    state = 3830;
                    return true;
                }
                break;
            case 3830:
                if ('r' == current) {
                    state = 3831;
                    return true;
                }
                break;
            case 3831:
                // cularr;
                if (';' == current) {
                    match = "\u21B6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('p' == current) {
                    state = 3832;
                    return true;
                }
                break;
            case 3832:
                // cularrp;
                if (';' == current) {
                    match = "\u293D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3833:
                switch (current) {
                    case ';' : // cup;
                        match = "\u222A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'b' :
                        state = 3834;
                        return true;
                    case 'c' :
                        state = 3839;
                        return true;
                    case 'd' :
                        state = 3844;
                        return true;
                    case 'o' :
                        state = 3847;
                        return true;
                    case 's' :
                        state = 3849;
                        return true;
                }
                break;
            case 3834:
                if ('r' == current) {
                    state = 3835;
                    return true;
                }
                break;
            case 3835:
                if ('c' == current) {
                    state = 3836;
                    return true;
                }
                break;
            case 3836:
                if ('a' == current) {
                    state = 3837;
                    return true;
                }
                break;
            case 3837:
                if ('p' == current) {
                    state = 3838;
                    return true;
                }
                break;
            case 3838:
                // cupbrcap;
                if (';' == current) {
                    match = "\u2A48";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3839:
                if ('a' == current) {
                    state = 3840;
                    return true;
                }
                else if ('u' == current) {
                    state = 3842;
                    return true;
                }
                break;
            case 3840:
                if ('p' == current) {
                    state = 3841;
                    return true;
                }
                break;
            case 3841:
                // cupcap;
                if (';' == current) {
                    match = "\u2A46";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3842:
                if ('p' == current) {
                    state = 3843;
                    return true;
                }
                break;
            case 3843:
                // cupcup;
                if (';' == current) {
                    match = "\u2A4A";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3844:
                if ('o' == current) {
                    state = 3845;
                    return true;
                }
                break;
            case 3845:
                if ('t' == current) {
                    state = 3846;
                    return true;
                }
                break;
            case 3846:
                // cupdot;
                if (';' == current) {
                    match = "\u228D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3847:
                if ('r' == current) {
                    state = 3848;
                    return true;
                }
                break;
            case 3848:
                // cupor;
                if (';' == current) {
                    match = "\u2A45";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3849:
                // cups;
                if (';' == current) {
                    match = "\u222A\uFE00";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3850:
                switch (current) {
                    case 'a' :
                        state = 3851;
                        return true;
                    case 'l' :
                        state = 3855;
                        return true;
                    case 'r' :
                        state = 3875;
                        return true;
                    case 'v' :
                        state = 3878;
                        return true;
                }
                break;
            case 3851:
                if ('r' == current) {
                    state = 3852;
                    return true;
                }
                break;
            case 3852:
                if ('r' == current) {
                    state = 3853;
                    return true;
                }
                break;
            case 3853:
                // curarr;
                if (';' == current) {
                    match = "\u21B7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('m' == current) {
                    state = 3854;
                    return true;
                }
                break;
            case 3854:
                // curarrm;
                if (';' == current) {
                    match = "\u293C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3855:
                if ('y' == current) {
                    state = 3856;
                    return true;
                }
                break;
            case 3856:
                switch (current) {
                    case 'e' :
                        state = 3857;
                        return true;
                    case 'v' :
                        state = 3867;
                        return true;
                    case 'w' :
                        state = 3870;
                        return true;
                }
                break;
            case 3857:
                if ('q' == current) {
                    state = 3858;
                    return true;
                }
                break;
            case 3858:
                if ('p' == current) {
                    state = 3859;
                    return true;
                }
                else if ('s' == current) {
                    state = 3863;
                    return true;
                }
                break;
            case 3859:
                if ('r' == current) {
                    state = 3860;
                    return true;
                }
                break;
            case 3860:
                if ('e' == current) {
                    state = 3861;
                    return true;
                }
                break;
            case 3861:
                if ('c' == current) {
                    state = 3862;
                    return true;
                }
                break;
            case 3862:
                // curlyeqprec;
                if (';' == current) {
                    match = "\u22DE";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3863:
                if ('u' == current) {
                    state = 3864;
                    return true;
                }
                break;
            case 3864:
                if ('c' == current) {
                    state = 3865;
                    return true;
                }
                break;
            case 3865:
                if ('c' == current) {
                    state = 3866;
                    return true;
                }
                break;
            case 3866:
                // curlyeqsucc;
                if (';' == current) {
                    match = "\u22DF";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3867:
                if ('e' == current) {
                    state = 3868;
                    return true;
                }
                break;
            case 3868:
                if ('e' == current) {
                    state = 3869;
                    return true;
                }
                break;
            case 3869:
                // curlyvee;
                if (';' == current) {
                    match = "\u22CE";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3870:
                if ('e' == current) {
                    state = 3871;
                    return true;
                }
                break;
            case 3871:
                if ('d' == current) {
                    state = 3872;
                    return true;
                }
                break;
            case 3872:
                if ('g' == current) {
                    state = 3873;
                    return true;
                }
                break;
            case 3873:
                if ('e' == current) {
                    state = 3874;
                    return true;
                }
                break;
            case 3874:
                // curlywedge;
                if (';' == current) {
                    match = "\u22CF";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3875:
                if ('e' == current) {
                    state = 3876;
                    return true;
                }
                break;
            case 3876:
                // curren
                if ('n' == current) {
                    match = "\u00A4";
                    matchLength = consumedCount;
                    state = 3877;
                    return true;
                }
                break;
            case 3877:
                // curren;
                if (';' == current) {
                    match = "\u00A4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3878:
                if ('e' == current) {
                    state = 3879;
                    return true;
                }
                break;
            case 3879:
                if ('a' == current) {
                    state = 3880;
                    return true;
                }
                break;
            case 3880:
                if ('r' == current) {
                    state = 3881;
                    return true;
                }
                break;
            case 3881:
                if ('r' == current) {
                    state = 3882;
                    return true;
                }
                break;
            case 3882:
                if ('o' == current) {
                    state = 3883;
                    return true;
                }
                break;
            case 3883:
                if ('w' == current) {
                    state = 3884;
                    return true;
                }
                break;
            case 3884:
                if ('l' == current) {
                    state = 3885;
                    return true;
                }
                else if ('r' == current) {
                    state = 3889;
                    return true;
                }
                break;
            case 3885:
                if ('e' == current) {
                    state = 3886;
                    return true;
                }
                break;
            case 3886:
                if ('f' == current) {
                    state = 3887;
                    return true;
                }
                break;
            case 3887:
                if ('t' == current) {
                    state = 3888;
                    return true;
                }
                break;
            case 3888:
                // curvearrowleft;
                if (';' == current) {
                    match = "\u21B6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3889:
                if ('i' == current) {
                    state = 3890;
                    return true;
                }
                break;
            case 3890:
                if ('g' == current) {
                    state = 3891;
                    return true;
                }
                break;
            case 3891:
                if ('h' == current) {
                    state = 3892;
                    return true;
                }
                break;
            case 3892:
                if ('t' == current) {
                    state = 3893;
                    return true;
                }
                break;
            case 3893:
                // curvearrowright;
                if (';' == current) {
                    match = "\u21B7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3894:
                if ('e' == current) {
                    state = 3895;
                    return true;
                }
                break;
            case 3895:
                if ('e' == current) {
                    state = 3896;
                    return true;
                }
                break;
            case 3896:
                // cuvee;
                if (';' == current) {
                    match = "\u22CE";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3897:
                if ('e' == current) {
                    state = 3898;
                    return true;
                }
                break;
            case 3898:
                if ('d' == current) {
                    state = 3899;
                    return true;
                }
                break;
            case 3899:
                // cuwed;
                if (';' == current) {
                    match = "\u22CF";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3900:
                if ('c' == current) {
                    state = 3901;
                    return true;
                }
                else if ('i' == current) {
                    state = 3907;
                    return true;
                }
                break;
            case 3901:
                if ('o' == current) {
                    state = 3902;
                    return true;
                }
                break;
            case 3902:
                if ('n' == current) {
                    state = 3903;
                    return true;
                }
                break;
            case 3903:
                if ('i' == current) {
                    state = 3904;
                    return true;
                }
                break;
            case 3904:
                if ('n' == current) {
                    state = 3905;
                    return true;
                }
                break;
            case 3905:
                if ('t' == current) {
                    state = 3906;
                    return true;
                }
                break;
            case 3906:
                // cwconint;
                if (';' == current) {
                    match = "\u2232";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3907:
                if ('n' == current) {
                    state = 3908;
                    return true;
                }
                break;
            case 3908:
                if ('t' == current) {
                    state = 3909;
                    return true;
                }
                break;
            case 3909:
                // cwint;
                if (';' == current) {
                    match = "\u2231";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3910:
                if ('l' == current) {
                    state = 3911;
                    return true;
                }
                break;
            case 3911:
                if ('c' == current) {
                    state = 3912;
                    return true;
                }
                break;
            case 3912:
                if ('t' == current) {
                    state = 3913;
                    return true;
                }
                break;
            case 3913:
                if ('y' == current) {
                    state = 3914;
                    return true;
                }
                break;
            case 3914:
                // cylcty;
                if (';' == current) {
                    match = "\u232D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3915:
                switch (current) {
                    case 'A' :
                        state = 3916;
                        return true;
                    case 'H' :
                        state = 3919;
                        return true;
                    case 'a' :
                        state = 3922;
                        return true;
                    case 'b' :
                        state = 3936;
                        return true;
                    case 'c' :
                        state = 3945;
                        return true;
                    case 'd' :
                        state = 3951;
                        return true;
                    case 'e' :
                        state = 3964;
                        return true;
                    case 'f' :
                        state = 3974;
                        return true;
                    case 'h' :
                        state = 3980;
                        return true;
                    case 'i' :
                        state = 3985;
                        return true;
                    case 'j' :
                        state = 4019;
                        return true;
                    case 'l' :
                        state = 4022;
                        return true;
                    case 'o' :
                        state = 4030;
                        return true;
                    case 'r' :
                        state = 4103;
                        return true;
                    case 's' :
                        state = 4117;
                        return true;
                    case 't' :
                        state = 4127;
                        return true;
                    case 'u' :
                        state = 4134;
                        return true;
                    case 'w' :
                        state = 4141;
                        return true;
                    case 'z' :
                        state = 4147;
                        return true;
                }
                break;
            case 3916:
                if ('r' == current) {
                    state = 3917;
                    return true;
                }
                break;
            case 3917:
                if ('r' == current) {
                    state = 3918;
                    return true;
                }
                break;
            case 3918:
                // dArr;
                if (';' == current) {
                    match = "\u21D3";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3919:
                if ('a' == current) {
                    state = 3920;
                    return true;
                }
                break;
            case 3920:
                if ('r' == current) {
                    state = 3921;
                    return true;
                }
                break;
            case 3921:
                // dHar;
                if (';' == current) {
                    match = "\u2965";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3922:
                switch (current) {
                    case 'g' :
                        state = 3923;
                        return true;
                    case 'l' :
                        state = 3927;
                        return true;
                    case 'r' :
                        state = 3931;
                        return true;
                    case 's' :
                        state = 3933;
                        return true;
                }
                break;
            case 3923:
                if ('g' == current) {
                    state = 3924;
                    return true;
                }
                break;
            case 3924:
                if ('e' == current) {
                    state = 3925;
                    return true;
                }
                break;
            case 3925:
                if ('r' == current) {
                    state = 3926;
                    return true;
                }
                break;
            case 3926:
                // dagger;
                if (';' == current) {
                    match = "\u2020";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3927:
                if ('e' == current) {
                    state = 3928;
                    return true;
                }
                break;
            case 3928:
                if ('t' == current) {
                    state = 3929;
                    return true;
                }
                break;
            case 3929:
                if ('h' == current) {
                    state = 3930;
                    return true;
                }
                break;
            case 3930:
                // daleth;
                if (';' == current) {
                    match = "\u2138";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3931:
                if ('r' == current) {
                    state = 3932;
                    return true;
                }
                break;
            case 3932:
                // darr;
                if (';' == current) {
                    match = "\u2193";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3933:
                if ('h' == current) {
                    state = 3934;
                    return true;
                }
                break;
            case 3934:
                // dash;
                if (';' == current) {
                    match = "\u2010";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('v' == current) {
                    state = 3935;
                    return true;
                }
                break;
            case 3935:
                // dashv;
                if (';' == current) {
                    match = "\u22A3";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3936:
                if ('k' == current) {
                    state = 3937;
                    return true;
                }
                else if ('l' == current) {
                    state = 3942;
                    return true;
                }
                break;
            case 3937:
                if ('a' == current) {
                    state = 3938;
                    return true;
                }
                break;
            case 3938:
                if ('r' == current) {
                    state = 3939;
                    return true;
                }
                break;
            case 3939:
                if ('o' == current) {
                    state = 3940;
                    return true;
                }
                break;
            case 3940:
                if ('w' == current) {
                    state = 3941;
                    return true;
                }
                break;
            case 3941:
                // dbkarow;
                if (';' == current) {
                    match = "\u290F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3942:
                if ('a' == current) {
                    state = 3943;
                    return true;
                }
                break;
            case 3943:
                if ('c' == current) {
                    state = 3944;
                    return true;
                }
                break;
            case 3944:
                // dblac;
                if (';' == current) {
                    match = "\u02DD";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3945:
                if ('a' == current) {
                    state = 3946;
                    return true;
                }
                else if ('y' == current) {
                    state = 3950;
                    return true;
                }
                break;
            case 3946:
                if ('r' == current) {
                    state = 3947;
                    return true;
                }
                break;
            case 3947:
                if ('o' == current) {
                    state = 3948;
                    return true;
                }
                break;
            case 3948:
                if ('n' == current) {
                    state = 3949;
                    return true;
                }
                break;
            case 3949:
                // dcaron;
                if (';' == current) {
                    match = "\u010F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3950:
                // dcy;
                if (';' == current) {
                    match = "\u0434";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3951:
                switch (current) {
                    case ';' : // dd;
                        match = "\u2146";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'a' :
                        state = 3952;
                        return true;
                    case 'o' :
                        state = 3959;
                        return true;
                }
                break;
            case 3952:
                if ('g' == current) {
                    state = 3953;
                    return true;
                }
                else if ('r' == current) {
                    state = 3957;
                    return true;
                }
                break;
            case 3953:
                if ('g' == current) {
                    state = 3954;
                    return true;
                }
                break;
            case 3954:
                if ('e' == current) {
                    state = 3955;
                    return true;
                }
                break;
            case 3955:
                if ('r' == current) {
                    state = 3956;
                    return true;
                }
                break;
            case 3956:
                // ddagger;
                if (';' == current) {
                    match = "\u2021";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3957:
                if ('r' == current) {
                    state = 3958;
                    return true;
                }
                break;
            case 3958:
                // ddarr;
                if (';' == current) {
                    match = "\u21CA";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3959:
                if ('t' == current) {
                    state = 3960;
                    return true;
                }
                break;
            case 3960:
                if ('s' == current) {
                    state = 3961;
                    return true;
                }
                break;
            case 3961:
                if ('e' == current) {
                    state = 3962;
                    return true;
                }
                break;
            case 3962:
                if ('q' == current) {
                    state = 3963;
                    return true;
                }
                break;
            case 3963:
                // ddotseq;
                if (';' == current) {
                    match = "\u2A77";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3964:
                switch (current) {
                    case 'g' : // deg
                        match = "\u00B0";
                        matchLength = consumedCount;
                        state = 3965;
                        return true;
                    case 'l' :
                        state = 3966;
                        return true;
                    case 'm' :
                        state = 3969;
                        return true;
                }
                break;
            case 3965:
                // deg;
                if (';' == current) {
                    match = "\u00B0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3966:
                if ('t' == current) {
                    state = 3967;
                    return true;
                }
                break;
            case 3967:
                if ('a' == current) {
                    state = 3968;
                    return true;
                }
                break;
            case 3968:
                // delta;
                if (';' == current) {
                    match = "\u03B4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3969:
                if ('p' == current) {
                    state = 3970;
                    return true;
                }
                break;
            case 3970:
                if ('t' == current) {
                    state = 3971;
                    return true;
                }
                break;
            case 3971:
                if ('y' == current) {
                    state = 3972;
                    return true;
                }
                break;
            case 3972:
                if ('v' == current) {
                    state = 3973;
                    return true;
                }
                break;
            case 3973:
                // demptyv;
                if (';' == current) {
                    match = "\u29B1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3974:
                if ('i' == current) {
                    state = 3975;
                    return true;
                }
                else if ('r' == current) {
                    state = 3979;
                    return true;
                }
                break;
            case 3975:
                if ('s' == current) {
                    state = 3976;
                    return true;
                }
                break;
            case 3976:
                if ('h' == current) {
                    state = 3977;
                    return true;
                }
                break;
            case 3977:
                if ('t' == current) {
                    state = 3978;
                    return true;
                }
                break;
            case 3978:
                // dfisht;
                if (';' == current) {
                    match = "\u297F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3979:
                // dfr;
                if (';' == current) {
                    match = "\uD835\uDD21";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3980:
                if ('a' == current) {
                    state = 3981;
                    return true;
                }
                break;
            case 3981:
                if ('r' == current) {
                    state = 3982;
                    return true;
                }
                break;
            case 3982:
                if ('l' == current) {
                    state = 3983;
                    return true;
                }
                else if ('r' == current) {
                    state = 3984;
                    return true;
                }
                break;
            case 3983:
                // dharl;
                if (';' == current) {
                    match = "\u21C3";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3984:
                // dharr;
                if (';' == current) {
                    match = "\u21C2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3985:
                switch (current) {
                    case 'a' :
                        state = 3986;
                        return true;
                    case 'e' :
                        state = 3996;
                        return true;
                    case 'g' :
                        state = 3997;
                        return true;
                    case 's' :
                        state = 4002;
                        return true;
                    case 'v' :
                        state = 4005;
                        return true;
                }
                break;
            case 3986:
                if ('m' == current) {
                    state = 3987;
                    return true;
                }
                break;
            case 3987:
                switch (current) {
                    case ';' : // diam;
                        match = "\u22C4";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'o' :
                        state = 3988;
                        return true;
                    case 's' :
                        state = 3995;
                        return true;
                }
                break;
            case 3988:
                if ('n' == current) {
                    state = 3989;
                    return true;
                }
                break;
            case 3989:
                if ('d' == current) {
                    state = 3990;
                    return true;
                }
                break;
            case 3990:
                // diamond;
                if (';' == current) {
                    match = "\u22C4";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('s' == current) {
                    state = 3991;
                    return true;
                }
                break;
            case 3991:
                if ('u' == current) {
                    state = 3992;
                    return true;
                }
                break;
            case 3992:
                if ('i' == current) {
                    state = 3993;
                    return true;
                }
                break;
            case 3993:
                if ('t' == current) {
                    state = 3994;
                    return true;
                }
                break;
            case 3994:
                // diamondsuit;
                if (';' == current) {
                    match = "\u2666";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3995:
                // diams;
                if (';' == current) {
                    match = "\u2666";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3996:
                // die;
                if (';' == current) {
                    match = "\u00A8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3997:
                if ('a' == current) {
                    state = 3998;
                    return true;
                }
                break;
            case 3998:
                if ('m' == current) {
                    state = 3999;
                    return true;
                }
                break;
            case 3999:
                if ('m' == current) {
                    state = 4000;
                    return true;
                }
                break;
        }
        return false;
    }

    private boolean parse5(final int current) {
        consumedCount++;
        switch (state) {
            case 4000:
                if ('a' == current) {
                    state = 4001;
                    return true;
                }
                break;
            case 4001:
                // digamma;
                if (';' == current) {
                    match = "\u03DD";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4002:
                if ('i' == current) {
                    state = 4003;
                    return true;
                }
                break;
            case 4003:
                if ('n' == current) {
                    state = 4004;
                    return true;
                }
                break;
            case 4004:
                // disin;
                if (';' == current) {
                    match = "\u22F2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4005:
                switch (current) {
                    case ';' : // div;
                        match = "\u00F7";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'i' :
                        state = 4006;
                        return true;
                    case 'o' :
                        state = 4016;
                        return true;
                }
                break;
            case 4006:
                if ('d' == current) {
                    state = 4007;
                    return true;
                }
                break;
            case 4007:
                // divide
                if ('e' == current) {
                    match = "\u00F7";
                    matchLength = consumedCount;
                    state = 4008;
                    return true;
                }
                break;
            case 4008:
                // divide;
                if (';' == current) {
                    match = "\u00F7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('o' == current) {
                    state = 4009;
                    return true;
                }
                break;
            case 4009:
                if ('n' == current) {
                    state = 4010;
                    return true;
                }
                break;
            case 4010:
                if ('t' == current) {
                    state = 4011;
                    return true;
                }
                break;
            case 4011:
                if ('i' == current) {
                    state = 4012;
                    return true;
                }
                break;
            case 4012:
                if ('m' == current) {
                    state = 4013;
                    return true;
                }
                break;
            case 4013:
                if ('e' == current) {
                    state = 4014;
                    return true;
                }
                break;
            case 4014:
                if ('s' == current) {
                    state = 4015;
                    return true;
                }
                break;
            case 4015:
                // divideontimes;
                if (';' == current) {
                    match = "\u22C7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4016:
                if ('n' == current) {
                    state = 4017;
                    return true;
                }
                break;
            case 4017:
                if ('x' == current) {
                    state = 4018;
                    return true;
                }
                break;
            case 4018:
                // divonx;
                if (';' == current) {
                    match = "\u22C7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4019:
                if ('c' == current) {
                    state = 4020;
                    return true;
                }
                break;
            case 4020:
                if ('y' == current) {
                    state = 4021;
                    return true;
                }
                break;
            case 4021:
                // djcy;
                if (';' == current) {
                    match = "\u0452";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4022:
                if ('c' == current) {
                    state = 4023;
                    return true;
                }
                break;
            case 4023:
                if ('o' == current) {
                    state = 4024;
                    return true;
                }
                else if ('r' == current) {
                    state = 4027;
                    return true;
                }
                break;
            case 4024:
                if ('r' == current) {
                    state = 4025;
                    return true;
                }
                break;
            case 4025:
                if ('n' == current) {
                    state = 4026;
                    return true;
                }
                break;
            case 4026:
                // dlcorn;
                if (';' == current) {
                    match = "\u231E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4027:
                if ('o' == current) {
                    state = 4028;
                    return true;
                }
                break;
            case 4028:
                if ('p' == current) {
                    state = 4029;
                    return true;
                }
                break;
            case 4029:
                // dlcrop;
                if (';' == current) {
                    match = "\u230D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4030:
                switch (current) {
                    case 'l' :
                        state = 4031;
                        return true;
                    case 'p' :
                        state = 4035;
                        return true;
                    case 't' :
                        state = 4037;
                        return true;
                    case 'u' :
                        state = 4058;
                        return true;
                    case 'w' :
                        state = 4070;
                        return true;
                }
                break;
            case 4031:
                if ('l' == current) {
                    state = 4032;
                    return true;
                }
                break;
            case 4032:
                if ('a' == current) {
                    state = 4033;
                    return true;
                }
                break;
            case 4033:
                if ('r' == current) {
                    state = 4034;
                    return true;
                }
                break;
            case 4034:
                // dollar;
                if (';' == current) {
                    match = "$";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4035:
                if ('f' == current) {
                    state = 4036;
                    return true;
                }
                break;
            case 4036:
                // dopf;
                if (';' == current) {
                    match = "\uD835\uDD55";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4037:
                switch (current) {
                    case ';' : // dot;
                        match = "\u02D9";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e' :
                        state = 4038;
                        return true;
                    case 'm' :
                        state = 4043;
                        return true;
                    case 'p' :
                        state = 4048;
                        return true;
                    case 's' :
                        state = 4052;
                        return true;
                }
                break;
            case 4038:
                if ('q' == current) {
                    state = 4039;
                    return true;
                }
                break;
            case 4039:
                // doteq;
                if (';' == current) {
                    match = "\u2250";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('d' == current) {
                    state = 4040;
                    return true;
                }
                break;
            case 4040:
                if ('o' == current) {
                    state = 4041;
                    return true;
                }
                break;
            case 4041:
                if ('t' == current) {
                    state = 4042;
                    return true;
                }
                break;
            case 4042:
                // doteqdot;
                if (';' == current) {
                    match = "\u2251";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4043:
                if ('i' == current) {
                    state = 4044;
                    return true;
                }
                break;
            case 4044:
                if ('n' == current) {
                    state = 4045;
                    return true;
                }
                break;
            case 4045:
                if ('u' == current) {
                    state = 4046;
                    return true;
                }
                break;
            case 4046:
                if ('s' == current) {
                    state = 4047;
                    return true;
                }
                break;
            case 4047:
                // dotminus;
                if (';' == current) {
                    match = "\u2238";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4048:
                if ('l' == current) {
                    state = 4049;
                    return true;
                }
                break;
            case 4049:
                if ('u' == current) {
                    state = 4050;
                    return true;
                }
                break;
            case 4050:
                if ('s' == current) {
                    state = 4051;
                    return true;
                }
                break;
            case 4051:
                // dotplus;
                if (';' == current) {
                    match = "\u2214";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4052:
                if ('q' == current) {
                    state = 4053;
                    return true;
                }
                break;
            case 4053:
                if ('u' == current) {
                    state = 4054;
                    return true;
                }
                break;
            case 4054:
                if ('a' == current) {
                    state = 4055;
                    return true;
                }
                break;
            case 4055:
                if ('r' == current) {
                    state = 4056;
                    return true;
                }
                break;
            case 4056:
                if ('e' == current) {
                    state = 4057;
                    return true;
                }
                break;
            case 4057:
                // dotsquare;
                if (';' == current) {
                    match = "\u22A1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4058:
                if ('b' == current) {
                    state = 4059;
                    return true;
                }
                break;
            case 4059:
                if ('l' == current) {
                    state = 4060;
                    return true;
                }
                break;
            case 4060:
                if ('e' == current) {
                    state = 4061;
                    return true;
                }
                break;
            case 4061:
                if ('b' == current) {
                    state = 4062;
                    return true;
                }
                break;
            case 4062:
                if ('a' == current) {
                    state = 4063;
                    return true;
                }
                break;
            case 4063:
                if ('r' == current) {
                    state = 4064;
                    return true;
                }
                break;
            case 4064:
                if ('w' == current) {
                    state = 4065;
                    return true;
                }
                break;
            case 4065:
                if ('e' == current) {
                    state = 4066;
                    return true;
                }
                break;
            case 4066:
                if ('d' == current) {
                    state = 4067;
                    return true;
                }
                break;
            case 4067:
                if ('g' == current) {
                    state = 4068;
                    return true;
                }
                break;
            case 4068:
                if ('e' == current) {
                    state = 4069;
                    return true;
                }
                break;
            case 4069:
                // doublebarwedge;
                if (';' == current) {
                    match = "\u2306";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4070:
                if ('n' == current) {
                    state = 4071;
                    return true;
                }
                break;
            case 4071:
                switch (current) {
                    case 'a' :
                        state = 4072;
                        return true;
                    case 'd' :
                        state = 4077;
                        return true;
                    case 'h' :
                        state = 4087;
                        return true;
                }
                break;
            case 4072:
                if ('r' == current) {
                    state = 4073;
                    return true;
                }
                break;
            case 4073:
                if ('r' == current) {
                    state = 4074;
                    return true;
                }
                break;
            case 4074:
                if ('o' == current) {
                    state = 4075;
                    return true;
                }
                break;
            case 4075:
                if ('w' == current) {
                    state = 4076;
                    return true;
                }
                break;
            case 4076:
                // downarrow;
                if (';' == current) {
                    match = "\u2193";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4077:
                if ('o' == current) {
                    state = 4078;
                    return true;
                }
                break;
            case 4078:
                if ('w' == current) {
                    state = 4079;
                    return true;
                }
                break;
            case 4079:
                if ('n' == current) {
                    state = 4080;
                    return true;
                }
                break;
            case 4080:
                if ('a' == current) {
                    state = 4081;
                    return true;
                }
                break;
            case 4081:
                if ('r' == current) {
                    state = 4082;
                    return true;
                }
                break;
            case 4082:
                if ('r' == current) {
                    state = 4083;
                    return true;
                }
                break;
            case 4083:
                if ('o' == current) {
                    state = 4084;
                    return true;
                }
                break;
            case 4084:
                if ('w' == current) {
                    state = 4085;
                    return true;
                }
                break;
            case 4085:
                if ('s' == current) {
                    state = 4086;
                    return true;
                }
                break;
            case 4086:
                // downdownarrows;
                if (';' == current) {
                    match = "\u21CA";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4087:
                if ('a' == current) {
                    state = 4088;
                    return true;
                }
                break;
            case 4088:
                if ('r' == current) {
                    state = 4089;
                    return true;
                }
                break;
            case 4089:
                if ('p' == current) {
                    state = 4090;
                    return true;
                }
                break;
            case 4090:
                if ('o' == current) {
                    state = 4091;
                    return true;
                }
                break;
            case 4091:
                if ('o' == current) {
                    state = 4092;
                    return true;
                }
                break;
            case 4092:
                if ('n' == current) {
                    state = 4093;
                    return true;
                }
                break;
            case 4093:
                if ('l' == current) {
                    state = 4094;
                    return true;
                }
                else if ('r' == current) {
                    state = 4098;
                    return true;
                }
                break;
            case 4094:
                if ('e' == current) {
                    state = 4095;
                    return true;
                }
                break;
            case 4095:
                if ('f' == current) {
                    state = 4096;
                    return true;
                }
                break;
            case 4096:
                if ('t' == current) {
                    state = 4097;
                    return true;
                }
                break;
            case 4097:
                // downharpoonleft;
                if (';' == current) {
                    match = "\u21C3";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4098:
                if ('i' == current) {
                    state = 4099;
                    return true;
                }
                break;
            case 4099:
                if ('g' == current) {
                    state = 4100;
                    return true;
                }
                break;
            case 4100:
                if ('h' == current) {
                    state = 4101;
                    return true;
                }
                break;
            case 4101:
                if ('t' == current) {
                    state = 4102;
                    return true;
                }
                break;
            case 4102:
                // downharpoonright;
                if (';' == current) {
                    match = "\u21C2";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4103:
                if ('b' == current) {
                    state = 4104;
                    return true;
                }
                else if ('c' == current) {
                    state = 4110;
                    return true;
                }
                break;
            case 4104:
                if ('k' == current) {
                    state = 4105;
                    return true;
                }
                break;
            case 4105:
                if ('a' == current) {
                    state = 4106;
                    return true;
                }
                break;
            case 4106:
                if ('r' == current) {
                    state = 4107;
                    return true;
                }
                break;
            case 4107:
                if ('o' == current) {
                    state = 4108;
                    return true;
                }
                break;
            case 4108:
                if ('w' == current) {
                    state = 4109;
                    return true;
                }
                break;
            case 4109:
                // drbkarow;
                if (';' == current) {
                    match = "\u2910";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4110:
                if ('o' == current) {
                    state = 4111;
                    return true;
                }
                else if ('r' == current) {
                    state = 4114;
                    return true;
                }
                break;
            case 4111:
                if ('r' == current) {
                    state = 4112;
                    return true;
                }
                break;
            case 4112:
                if ('n' == current) {
                    state = 4113;
                    return true;
                }
                break;
            case 4113:
                // drcorn;
                if (';' == current) {
                    match = "\u231F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4114:
                if ('o' == current) {
                    state = 4115;
                    return true;
                }
                break;
            case 4115:
                if ('p' == current) {
                    state = 4116;
                    return true;
                }
                break;
            case 4116:
                // drcrop;
                if (';' == current) {
                    match = "\u230C";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4117:
                switch (current) {
                    case 'c' :
                        state = 4118;
                        return true;
                    case 'o' :
                        state = 4121;
                        return true;
                    case 't' :
                        state = 4123;
                        return true;
                }
                break;
            case 4118:
                if ('r' == current) {
                    state = 4119;
                    return true;
                }
                else if ('y' == current) {
                    state = 4120;
                    return true;
                }
                break;
            case 4119:
                // dscr;
                if (';' == current) {
                    match = "\uD835\uDCB9";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4120:
                // dscy;
                if (';' == current) {
                    match = "\u0455";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4121:
                if ('l' == current) {
                    state = 4122;
                    return true;
                }
                break;
            case 4122:
                // dsol;
                if (';' == current) {
                    match = "\u29F6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4123:
                if ('r' == current) {
                    state = 4124;
                    return true;
                }
                break;
            case 4124:
                if ('o' == current) {
                    state = 4125;
                    return true;
                }
                break;
            case 4125:
                if ('k' == current) {
                    state = 4126;
                    return true;
                }
                break;
            case 4126:
                // dstrok;
                if (';' == current) {
                    match = "\u0111";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4127:
                if ('d' == current) {
                    state = 4128;
                    return true;
                }
                else if ('r' == current) {
                    state = 4131;
                    return true;
                }
                break;
            case 4128:
                if ('o' == current) {
                    state = 4129;
                    return true;
                }
                break;
            case 4129:
                if ('t' == current) {
                    state = 4130;
                    return true;
                }
                break;
            case 4130:
                // dtdot;
                if (';' == current) {
                    match = "\u22F1";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4131:
                if ('i' == current) {
                    state = 4132;
                    return true;
                }
                break;
            case 4132:
                // dtri;
                if (';' == current) {
                    match = "\u25BF";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('f' == current) {
                    state = 4133;
                    return true;
                }
                break;
            case 4133:
                // dtrif;
                if (';' == current) {
                    match = "\u25BE";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4134:
                if ('a' == current) {
                    state = 4135;
                    return true;
                }
                else if ('h' == current) {
                    state = 4138;
                    return true;
                }
                break;
            case 4135:
                if ('r' == current) {
                    state = 4136;
                    return true;
                }
                break;
            case 4136:
                if ('r' == current) {
                    state = 4137;
                    return true;
                }
                break;
            case 4137:
                // duarr;
                if (';' == current) {
                    match = "\u21F5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4138:
                if ('a' == current) {
                    state = 4139;
                    return true;
                }
                break;
            case 4139:
                if ('r' == current) {
                    state = 4140;
                    return true;
                }
                break;
            case 4140:
                // duhar;
                if (';' == current) {
                    match = "\u296F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4141:
                if ('a' == current) {
                    state = 4142;
                    return true;
                }
                break;
            case 4142:
                if ('n' == current) {
                    state = 4143;
                    return true;
                }
                break;
            case 4143:
                if ('g' == current) {
                    state = 4144;
                    return true;
                }
                break;
            case 4144:
                if ('l' == current) {
                    state = 4145;
                    return true;
                }
                break;
            case 4145:
                if ('e' == current) {
                    state = 4146;
                    return true;
                }
                break;
            case 4146:
                // dwangle;
                if (';' == current) {
                    match = "\u29A6";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4147:
                if ('c' == current) {
                    state = 4148;
                    return true;
                }
                else if ('i' == current) {
                    state = 4150;
                    return true;
                }
                break;
            case 4148:
                if ('y' == current) {
                    state = 4149;
                    return true;
                }
                break;
            case 4149:
                // dzcy;
                if (';' == current) {
                    match = "\u045F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4150:
                if ('g' == current) {
                    state = 4151;
                    return true;
                }
                break;
            case 4151:
                if ('r' == current) {
                    state = 4152;
                    return true;
                }
                break;
            case 4152:
                if ('a' == current) {
                    state = 4153;
                    return true;
                }
                break;
            case 4153:
                if ('r' == current) {
                    state = 4154;
                    return true;
                }
                break;
            case 4154:
                if ('r' == current) {
                    state = 4155;
                    return true;
                }
                break;
            case 4155:
                // dzigrarr;
                if (';' == current) {
                    match = "\u27FF";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4156:
                switch (current) {
                    case 'D' :
                        state = 4157;
                        return true;
                    case 'a' :
                        state = 4163;
                        return true;
                    case 'c' :
                        state = 4172;
                        return true;
                    case 'd' :
                        state = 4185;
                        return true;
                    case 'e' :
                        state = 4188;
                        return true;
                    case 'f' :
                        state = 4189;
                        return true;
                    case 'g' :
                        state = 4194;
                        return true;
                    case 'l' :
                        state = 4203;
                        return true;
                    case 'm' :
                        state = 4215;
                        return true;
                    case 'n' :
                        state = 4231;
                        return true;
                    case 'o' :
                        state = 4235;
                        return true;
                    case 'p' :
                        state = 4241;
                        return true;
                    case 'q' :
                        state = 4255;
                        return true;
                    case 'r' :
                        state = 4295;
                        return true;
                    case 's' :
                        state = 4302;
                        return true;
                    case 't' :
                        state = 4310;
                        return true;
                    case 'u' :
                        state = 4313;
                        return true;
                    case 'x' :
                        state = 4318;
                        return true;
                }
                break;
            case 4157:
                if ('D' == current) {
                    state = 4158;
                    return true;
                }
                else if ('o' == current) {
                    state = 4161;
                    return true;
                }
                break;
            case 4158:
                if ('o' == current) {
                    state = 4159;
                    return true;
                }
                break;
            case 4159:
                if ('t' == current) {
                    state = 4160;
                    return true;
                }
                break;
            case 4160:
                // eDDot;
                if (';' == current) {
                    match = "\u2A77";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4161:
                if ('t' == current) {
                    state = 4162;
                    return true;
                }
                break;
            case 4162:
                // eDot;
                if (';' == current) {
                    match = "\u2251";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4163:
                if ('c' == current) {
                    state = 4164;
                    return true;
                }
                else if ('s' == current) {
                    state = 4168;
                    return true;
                }
                break;
            case 4164:
                if ('u' == current) {
                    state = 4165;
                    return true;
                }
                break;
            case 4165:
                if ('t' == current) {
                    state = 4166;
                    return true;
                }
                break;
            case 4166:
                // eacute
                if ('e' == current) {
                    match = "\u00E9";
                    matchLength = consumedCount;
                    state = 4167;
                    return true;
                }
                break;
            case 4167:
                // eacute;
                if (';' == current) {
                    match = "\u00E9";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4168:
                if ('t' == current) {
                    state = 4169;
                    return true;
                }
                break;
            case 4169:
                if ('e' == current) {
                    state = 4170;
                    return true;
                }
                break;
            case 4170:
                if ('r' == current) {
                    state = 4171;
                    return true;
                }
                break;
            case 4171:
                // easter;
                if (';' == current) {
                    match = "\u2A6E";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4172:
                switch (current) {
                    case 'a' :
                        state = 4173;
                        return true;
                    case 'i' :
                        state = 4177;
                        return true;
                    case 'o' :
                        state = 4180;
                        return true;
                    case 'y' :
                        state = 4184;
                        return true;
                }
                break;
            case 4173:
                if ('r' == current) {
                    state = 4174;
                    return true;
                }
                break;
            case 4174:
                if ('o' == current) {
                    state = 4175;
                    return true;
                }
                break;
            case 4175:
                if ('n' == current) {
                    state = 4176;
                    return true;
                }
                break;
            case 4176:
                // ecaron;
                if (';' == current) {
                    match = "\u011B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4177:
                if ('r' == current) {
                    state = 4178;
                    return true;
                }
                break;
            case 4178:
                // ecir;
                if (';' == current) {
                    match = "\u2256";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                // ecirc
                else if ('c' == current) {
                    match = "\u00EA";
                    matchLength = consumedCount;
                    state = 4179;
                    return true;
                }
                break;
            case 4179:
                // ecirc;
                if (';' == current) {
                    match = "\u00EA";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4180:
                if ('l' == current) {
                    state = 4181;
                    return true;
                }
                break;
            case 4181:
                if ('o' == current) {
                    state = 4182;
                    return true;
                }
                break;
            case 4182:
                if ('n' == current) {
                    state = 4183;
                    return true;
                }
                break;
            case 4183:
                // ecolon;
                if (';' == current) {
                    match = "\u2255";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4184:
                // ecy;
                if (';' == current) {
                    match = "\u044D";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4185:
                if ('o' == current) {
                    state = 4186;
                    return true;
                }
                break;
            case 4186:
                if ('t' == current) {
                    state = 4187;
                    return true;
                }
                break;
            case 4187:
                // edot;
                if (';' == current) {
                    match = "\u0117";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4188:
                // ee;
                if (';' == current) {
                    match = "\u2147";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4189:
                if ('D' == current) {
                    state = 4190;
                    return true;
                }
                else if ('r' == current) {
                    state = 4193;
                    return true;
                }
                break;
            case 4190:
                if ('o' == current) {
                    state = 4191;
                    return true;
                }
                break;
            case 4191:
                if ('t' == current) {
                    state = 4192;
                    return true;
                }
                break;
            case 4192:
                // efDot;
                if (';' == current) {
                    match = "\u2252";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4193:
                // efr;
                if (';' == current) {
                    match = "\uD835\uDD22";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4194:
                switch (current) {
                    case ';' : // eg;
                        match = "\u2A9A";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'r' :
                        state = 4195;
                        return true;
                    case 's' :
                        state = 4199;
                        return true;
                }
                break;
            case 4195:
                if ('a' == current) {
                    state = 4196;
                    return true;
                }
                break;
            case 4196:
                if ('v' == current) {
                    state = 4197;
                    return true;
                }
                break;
            case 4197:
                // egrave
                if ('e' == current) {
                    match = "\u00E8";
                    matchLength = consumedCount;
                    state = 4198;
                    return true;
                }
                break;
            case 4198:
                // egrave;
                if (';' == current) {
                    match = "\u00E8";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4199:
                // egs;
                if (';' == current) {
                    match = "\u2A96";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('d' == current) {
                    state = 4200;
                    return true;
                }
                break;
            case 4200:
                if ('o' == current) {
                    state = 4201;
                    return true;
                }
                break;
            case 4201:
                if ('t' == current) {
                    state = 4202;
                    return true;
                }
                break;
            case 4202:
                // egsdot;
                if (';' == current) {
                    match = "\u2A98";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4203:
                switch (current) {
                    case ';' : // el;
                        match = "\u2A99";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'i' :
                        state = 4204;
                        return true;
                    case 'l' :
                        state = 4210;
                        return true;
                    case 's' :
                        state = 4211;
                        return true;
                }
                break;
            case 4204:
                if ('n' == current) {
                    state = 4205;
                    return true;
                }
                break;
            case 4205:
                if ('t' == current) {
                    state = 4206;
                    return true;
                }
                break;
            case 4206:
                if ('e' == current) {
                    state = 4207;
                    return true;
                }
                break;
            case 4207:
                if ('r' == current) {
                    state = 4208;
                    return true;
                }
                break;
            case 4208:
                if ('s' == current) {
                    state = 4209;
                    return true;
                }
                break;
            case 4209:
                // elinters;
                if (';' == current) {
                    match = "\u23E7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4210:
                // ell;
                if (';' == current) {
                    match = "\u2113";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4211:
                // els;
                if (';' == current) {
                    match = "\u2A95";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('d' == current) {
                    state = 4212;
                    return true;
                }
                break;
            case 4212:
                if ('o' == current) {
                    state = 4213;
                    return true;
                }
                break;
            case 4213:
                if ('t' == current) {
                    state = 4214;
                    return true;
                }
                break;
            case 4214:
                // elsdot;
                if (';' == current) {
                    match = "\u2A97";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4215:
                switch (current) {
                    case 'a' :
                        state = 4216;
                        return true;
                    case 'p' :
                        state = 4219;
                        return true;
                    case 's' :
                        state = 4226;
                        return true;
                }
                break;
            case 4216:
                if ('c' == current) {
                    state = 4217;
                    return true;
                }
                break;
            case 4217:
                if ('r' == current) {
                    state = 4218;
                    return true;
                }
                break;
            case 4218:
                // emacr;
                if (';' == current) {
                    match = "\u0113";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4219:
                if ('t' == current) {
                    state = 4220;
                    return true;
                }
                break;
            case 4220:
                if ('y' == current) {
                    state = 4221;
                    return true;
                }
                break;
            case 4221:
                switch (current) {
                    case ';' : // empty;
                        match = "\u2205";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 's' :
                        state = 4222;
                        return true;
                    case 'v' :
                        state = 4225;
                        return true;
                }
                break;
            case 4222:
                if ('e' == current) {
                    state = 4223;
                    return true;
                }
                break;
            case 4223:
                if ('t' == current) {
                    state = 4224;
                    return true;
                }
                break;
            case 4224:
                // emptyset;
                if (';' == current) {
                    match = "\u2205";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4225:
                // emptyv;
                if (';' == current) {
                    match = "\u2205";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4226:
                if ('p' == current) {
                    state = 4227;
                    return true;
                }
                break;
            case 4227:
                if ('1' == current) {
                    state = 4228;
                    return true;
                }
                // emsp;
                else if (';' == current) {
                    match = "\u2003";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4228:
                if ('3' == current) {
                    state = 4229;
                    return true;
                }
                else if ('4' == current) {
                    state = 4230;
                    return true;
                }
                break;
            case 4229:
                // emsp13;
                if (';' == current) {
                    match = "\u2004";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4230:
                // emsp14;
                if (';' == current) {
                    match = "\u2005";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4231:
                if ('g' == current) {
                    state = 4232;
                    return true;
                }
                else if ('s' == current) {
                    state = 4233;
                    return true;
                }
                break;
            case 4232:
                // eng;
                if (';' == current) {
                    match = "\u014B";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4233:
                if ('p' == current) {
                    state = 4234;
                    return true;
                }
                break;
            case 4234:
                // ensp;
                if (';' == current) {
                    match = "\u2002";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4235:
                if ('g' == current) {
                    state = 4236;
                    return true;
                }
                else if ('p' == current) {
                    state = 4239;
                    return true;
                }
                break;
            case 4236:
                if ('o' == current) {
                    state = 4237;
                    return true;
                }
                break;
            case 4237:
                if ('n' == current) {
                    state = 4238;
                    return true;
                }
                break;
            case 4238:
                // eogon;
                if (';' == current) {
                    match = "\u0119";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4239:
                if ('f' == current) {
                    state = 4240;
                    return true;
                }
                break;
            case 4240:
                // eopf;
                if (';' == current) {
                    match = "\uD835\uDD56";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4241:
                switch (current) {
                    case 'a' :
                        state = 4242;
                        return true;
                    case 'l' :
                        state = 4246;
                        return true;
                    case 's' :
                        state = 4249;
                        return true;
                }
                break;
            case 4242:
                if ('r' == current) {
                    state = 4243;
                    return true;
                }
                break;
            case 4243:
                // epar;
                if (';' == current) {
                    match = "\u22D5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('s' == current) {
                    state = 4244;
                    return true;
                }
                break;
            case 4244:
                if ('l' == current) {
                    state = 4245;
                    return true;
                }
                break;
            case 4245:
                // eparsl;
                if (';' == current) {
                    match = "\u29E3";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4246:
                if ('u' == current) {
                    state = 4247;
                    return true;
                }
                break;
            case 4247:
                if ('s' == current) {
                    state = 4248;
                    return true;
                }
                break;
            case 4248:
                // eplus;
                if (';' == current) {
                    match = "\u2A71";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4249:
                if ('i' == current) {
                    state = 4250;
                    return true;
                }
                break;
            case 4250:
                switch (current) {
                    case ';' : // epsi;
                        match = "\u03B5";
                        matchLength = consumedCount;
                        state = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'l' :
                        state = 4251;
                        return true;
                    case 'v' :
                        state = 4254;
                        return true;
                }
                break;
            case 4251:
                if ('o' == current) {
                    state = 4252;
                    return true;
                }
                break;
            case 4252:
                if ('n' == current) {
                    state = 4253;
                    return true;
                }
                break;
            case 4253:
                // epsilon;
                if (';' == current) {
                    match = "\u03B5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4254:
                // epsiv;
                if (';' == current) {
                    match = "\u03F5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4255:
                switch (current) {
                    case 'c' :
                        state = 4256;
                        return true;
                    case 's' :
                        state = 4264;
                        return true;
                    case 'u' :
                        state = 4278;
                        return true;
                    case 'v' :
                        state = 4289;
                        return true;
                }
                break;
            case 4256:
                if ('i' == current) {
                    state = 4257;
                    return true;
                }
                else if ('o' == current) {
                    state = 4260;
                    return true;
                }
                break;
            case 4257:
                if ('r' == current) {
                    state = 4258;
                    return true;
                }
                break;
            case 4258:
                if ('c' == current) {
                    state = 4259;
                    return true;
                }
                break;
            case 4259:
                // eqcirc;
                if (';' == current) {
                    match = "\u2256";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4260:
                if ('l' == current) {
                    state = 4261;
                    return true;
                }
                break;
            case 4261:
                if ('o' == current) {
                    state = 4262;
                    return true;
                }
                break;
            case 4262:
                if ('n' == current) {
                    state = 4263;
                    return true;
                }
                break;
            case 4263:
                // eqcolon;
                if (';' == current) {
                    match = "\u2255";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4264:
                if ('i' == current) {
                    state = 4265;
                    return true;
                }
                else if ('l' == current) {
                    state = 4267;
                    return true;
                }
                break;
            case 4265:
                if ('m' == current) {
                    state = 4266;
                    return true;
                }
                break;
            case 4266:
                // eqsim;
                if (';' == current) {
                    match = "\u2242";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4267:
                if ('a' == current) {
                    state = 4268;
                    return true;
                }
                break;
            case 4268:
                if ('n' == current) {
                    state = 4269;
                    return true;
                }
                break;
            case 4269:
                if ('t' == current) {
                    state = 4270;
                    return true;
                }
                break;
            case 4270:
                if ('g' == current) {
                    state = 4271;
                    return true;
                }
                else if ('l' == current) {
                    state = 4274;
                    return true;
                }
                break;
            case 4271:
                if ('t' == current) {
                    state = 4272;
                    return true;
                }
                break;
            case 4272:
                if ('r' == current) {
                    state = 4273;
                    return true;
                }
                break;
            case 4273:
                // eqslantgtr;
                if (';' == current) {
                    match = "\u2A96";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4274:
                if ('e' == current) {
                    state = 4275;
                    return true;
                }
                break;
            case 4275:
                if ('s' == current) {
                    state = 4276;
                    return true;
                }
                break;
            case 4276:
                if ('s' == current) {
                    state = 4277;
                    return true;
                }
                break;
            case 4277:
                // eqslantless;
                if (';' == current) {
                    match = "\u2A95";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4278:
                switch (current) {
                    case 'a' :
                        state = 4279;
                        return true;
                    case 'e' :
                        state = 4282;
                        return true;
                    case 'i' :
                        state = 4285;
                        return true;
                }
                break;
            case 4279:
                if ('l' == current) {
                    state = 4280;
                    return true;
                }
                break;
            case 4280:
                if ('s' == current) {
                    state = 4281;
                    return true;
                }
                break;
            case 4281:
                // equals;
                if (';' == current) {
                    match = "=";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4282:
                if ('s' == current) {
                    state = 4283;
                    return true;
                }
                break;
            case 4283:
                if ('t' == current) {
                    state = 4284;
                    return true;
                }
                break;
            case 4284:
                // equest;
                if (';' == current) {
                    match = "\u225F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4285:
                if ('v' == current) {
                    state = 4286;
                    return true;
                }
                break;
            case 4286:
                // equiv;
                if (';' == current) {
                    match = "\u2261";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('D' == current) {
                    state = 4287;
                    return true;
                }
                break;
            case 4287:
                if ('D' == current) {
                    state = 4288;
                    return true;
                }
                break;
            case 4288:
                // equivDD;
                if (';' == current) {
                    match = "\u2A78";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4289:
                if ('p' == current) {
                    state = 4290;
                    return true;
                }
                break;
            case 4290:
                if ('a' == current) {
                    state = 4291;
                    return true;
                }
                break;
            case 4291:
                if ('r' == current) {
                    state = 4292;
                    return true;
                }
                break;
            case 4292:
                if ('s' == current) {
                    state = 4293;
                    return true;
                }
                break;
            case 4293:
                if ('l' == current) {
                    state = 4294;
                    return true;
                }
                break;
            case 4294:
                // eqvparsl;
                if (';' == current) {
                    match = "\u29E5";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4295:
                if ('D' == current) {
                    state = 4296;
                    return true;
                }
                else if ('a' == current) {
                    state = 4299;
                    return true;
                }
                break;
            case 4296:
                if ('o' == current) {
                    state = 4297;
                    return true;
                }
                break;
            case 4297:
                if ('t' == current) {
                    state = 4298;
                    return true;
                }
                break;
            case 4298:
                // erDot;
                if (';' == current) {
                    match = "\u2253";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4299:
                if ('r' == current) {
                    state = 4300;
                    return true;
                }
                break;
            case 4300:
                if ('r' == current) {
                    state = 4301;
                    return true;
                }
                break;
            case 4301:
                // erarr;
                if (';' == current) {
                    match = "\u2971";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4302:
                switch (current) {
                    case 'c' :
                        state = 4303;
                        return true;
                    case 'd' :
                        state = 4305;
                        return true;
                    case 'i' :
                        state = 4308;
                        return true;
                }
                break;
            case 4303:
                if ('r' == current) {
                    state = 4304;
                    return true;
                }
                break;
            case 4304:
                // escr;
                if (';' == current) {
                    match = "\u212F";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4305:
                if ('o' == current) {
                    state = 4306;
                    return true;
                }
                break;
            case 4306:
                if ('t' == current) {
                    state = 4307;
                    return true;
                }
                break;
            case 4307:
                // esdot;
                if (';' == current) {
                    match = "\u2250";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4308:
                if ('m' == current) {
                    state = 4309;
                    return true;
                }
                break;
            case 4309:
                // esim;
                if (';' == current) {
                    match = "\u2242";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4310:
                if ('a' == current) {
                    state = 4311;
                    return true;
                }
                // eth
                else if ('h' == current) {
                    match = "\u00F0";
                    matchLength = consumedCount;
                    state = 4312;
                    return true;
                }
                break;
            case 4311:
                // eta;
                if (';' == current) {
                    match = "\u03B7";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4312:
                // eth;
                if (';' == current) {
                    match = "\u00F0";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4313:
                if ('m' == current) {
                    state = 4314;
                    return true;
                }
                else if ('r' == current) {
                    state = 4316;
                    return true;
                }
                break;
            case 4314:
                // euml
                if ('l' == current) {
                    match = "\u00EB";
                    matchLength = consumedCount;
                    state = 4315;
                    return true;
                }
                break;
            case 4315:
                // euml;
                if (';' == current) {
                    match = "\u00EB";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4316:
                if ('o' == current) {
                    state = 4317;
                    return true;
                }
                break;
            case 4317:
                // euro;
                if (';' == current) {
                    match = "\u20AC";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4318:
                switch (current) {
                    case 'c' :
                        state = 4319;
                        return true;
                    case 'i' :
                        state = 4321;
                        return true;
                    case 'p' :
                        state = 4324;
                        return true;
                }
                break;
            case 4319:
                if ('l' == current) {
                    state = 4320;
                    return true;
                }
                break;
            case 4320:
                // excl;
                if (';' == current) {
                    match = "!";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4321:
                if ('s' == current) {
                    state = 4322;
                    return true;
                }
                break;
            case 4322:
                if ('t' == current) {
                    state = 4323;
                    return true;
                }
                break;
            case 4323:
                // exist;
                if (';' == current) {
                    match = "\u2203";
                    matchLength = consumedCount;
                    state = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4324:
                if ('e' == current) {
                    state = 4325;
                    return true;
                }
                else if ('o' == current) {
                    state = 4333;
                    return true;
                }
                break;
            case 4325:
                if ('c' == current) {
                    state = 4326;
                    return true;
                }
                break;
            case 4326:
                if ('t' == current) {
                    state = 4327;
                    return true;
                }
                break;
            case 4327:
                if ('a' == current) {
                    state = 4328;
                    return true;
                }
                break;
            case 4328:
                if ('t' == current) {
                    state = 4329;
                    return true;
                }
                break;
            case 4329:
                if ('i' == current) {
                    state = 4330;
                    return true;
                }
                break;
            case 4330:
                if ('o' == current) {
                    state = 4331;
               