/*
 * Copyright (c) 2012-2017, FOSS Nova Software foundation (FNSF),
 * and individual contributors as indicated by the @author tags.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.fossnova.json;

import java.io.IOException;
import java.io.OutputStream;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.List;

import org.fossnova.json.stream.JsonException;
import org.fossnova.json.stream.JsonWriter;

/**
 * JSON array.
 * @author <a href="mailto:opalka.richard@gmail.com">Richard Opalka</a>
 * @see JsonValue
 * @see JsonValueFactory
 */
public interface JsonArray extends JsonValue, List< JsonValue > {

    /**
     * Wraps passed value with JsonString and delegates the call to
     * {@link java.util.List#add(Object)} method.
     * @param value string to wrap
     * @return <tt>true</tt> if this JSON array contained the specified JSON string
     */
    boolean add( String value );

    /**
     * Wraps passed value with JsonNumber and delegates the call to
     * {@link java.util.List#add(Object)} method.
     * @param value number to wrap
     * @return <tt>true</tt> if this JSON array contained the specified JSON number
     */
    boolean add( Number value );

    /**
     * Wraps passed value with JsonBoolean and delegates the call to
     * {@link java.util.List#add(Object)} method.
     * @param value boolean to wrap
     * @return <tt>true</tt> if this JSON array contained the specified JSON boolean
     */
    boolean add( Boolean value );

    /**
     * Delegates the call to {@link java.util.List#add(Object)} method.
     * @param value JSON value
     * @return <tt>true</tt> if this JSON array contained the specified JSON value
     */
    @Override
    boolean add( JsonValue value );

    /**
     * Delegates the call to
     * {@link java.util.List#add(Object)} method.
     * @return <tt>true</tt> if this JSON array contained <tt>null</tt>
     */
    boolean addNull();

    /**
     * Wraps passed value with JsonString and delegates the call to
     * {@link java.util.List#add(int, Object)} method.
     * @param index index at which the specified JSON string is to be inserted
     * @param value string to wrap
     */
    void add( int index, String value );

    /**
     * Wraps passed value with JsonNumber and delegates the call to
     * {@link java.util.List#add(int, Object)} method.
     * @param index index at which the specified JSON number is to be inserted
     * @param value number to wrap
     */
    void add( int index, Number value );

    /**
     * Wraps passed value with JsonBoolean and delegates the call to
     * {@link java.util.List#add(int, Object)} method.
     * @param index index at which the specified JSON boolean is to be inserted
     * @param value boolean to wrap
     */
    void add( int index, Boolean value );

    /**
     * Delegates the call to {@link java.util.List#add(int, Object)} method.
     * @param index index at which the specified JSON value is to be inserted
     * @param value JSON value
     */
    @Override
    void add( int index, JsonValue value );

    /**
     * Delegates the call to {@link java.util.List#add(int, Object)} method.
     * @param index index at which <tt>null</tt> value is to be inserted
     */
    void addNull( int index );

    /**
     * Wraps passed value with JsonString and delegates the call to
     * {@link java.util.List#contains(Object)} method.
     * @param value string to wrap
     * @return <tt>true</tt> if this JSON array contains the specified JSON string
     */
    boolean contains( String value );

    /**
     * Wraps passed value with JsonNumber and delegates the call to
     * {@link java.util.List#contains(Object)} method.
     * @param value number to wrap
     * @return <tt>true</tt> if this JSON array contains the specified JSON number
     */
    boolean contains( Number value );

    /**
     * Wraps passed value with JsonBoolean and delegates the call to
     * {@link java.util.List#contains(Object)} method.
     * @param value boolean to wrap
     * @return <tt>true</tt> if this JSON array contains the specified JSON boolean
     */
    boolean contains( Boolean value );

    /**
     * Delegates the call to {@link java.util.List#contains(Object)} method.
     * @param value JSON value
     * @return <tt>true</tt> if this JSON array contains the specified JSON value
     */
    boolean contains( JsonValue value );

    /**
     * Delegates the call to {@link java.util.List#contains(Object)} method.
     * @return <tt>true</tt> if this JSON array contains <tt>null</tt>
     */
    boolean containsNull();

    /**
     * Wraps passed value with JsonString and delegates the call to
     * {@link java.util.List#indexOf(Object)} method.
     * @param value string to wrap
     * @return the index of the first occurrence of the specified JSON string in
     *         this JSON array, or -1 if this JSON array does not contain the JSON string
     */
    int indexOf( String value );

    /**
     * Wraps passed value with JsonNumber and delegates the call to
     * {@link java.util.List#indexOf(Object)} method.
     * @param value number to wrap
     * @return the index of the first occurrence of the specified JSON number in
     *         this JSON array, or -1 if this JSON array does not contain the JSON number
     */
    int indexOf( Number value );

    /**
     * Wraps passed value with JsonBoolean and delegates the call to
     * {@link java.util.List#indexOf(Object)} method.
     * @param value boolean to wrap
     * @return the index of the first occurrence of the specified JSON boolean in
     *         this JSON array, or -1 if this JSON array does not contain the JSON boolean
     */
    int indexOf( Boolean value );

    /**
     * Delegates the call to {@link java.util.List#indexOf(Object)} method.
     * @param value JSON value
     * @return the index of the first occurrence of the specified JSON value in
     *         this JSON array, or -1 if this JSON array does not contain the JSON value
     */
    int indexOf( JsonValue value );

    /**
     * Delegates the call to {@link java.util.List#indexOf(Object)} method.
     * @return the index of the first occurrence of <tt>null</tt> value in
     *         this JSON array, or -1 if this JSON array does not contain the <tt>null</tt> value
     */
    int indexOfNull();

    /**
     * Wraps passed value with JsonString and delegates the call to
     * {@link java.util.List#lastIndexOf(Object)} method.
     * @param value string to wrap
     * @return he index of the last occurrence of the specified JSON string in
     *         this JSON array, or -1 if this JSON array does not contain the JSON string
     */
    int lastIndexOf( String value );

    /**
     * Wraps passed value with JsonNumber and delegates the call to
     * {@link java.util.List#lastIndexOf(Object)} method.
     * @param value string to wrap
     * @return he index of the last occurrence of the specified JSON number in
     *         this JSON array, or -1 if this JSON array does not contain the JSON number
     */
    int lastIndexOf( Number value );

    /**
     * Wraps passed value with JsonBoolean and delegates the call to
     * {@link java.util.List#lastIndexOf(Object)} method.
     * @param value string to wrap
     * @return he index of the last occurrence of the specified JSON boolean in
     *         this JSON array, or -1 if this JSON array does not contain the JSON boolean
     */
    int lastIndexOf( Boolean value );

    /**
     * Delegates the call to {@link java.util.List#lastIndexOf(Object)} method.
     * @param value JSON value
     * @return he index of the last occurrence of the specified JSON value in
     *         this JSON array, or -1 if this JSON array does not contain the JSON value
     */
    int lastIndexOf( JsonValue value );

    /**
     * Delegates the call to {@link java.util.List#lastIndexOf(Object)} method.
     * @return he index of the last occurrence of the <tt>null</tt> value in
     *         this JSON array, or -1 if this JSON array does not contain the <tt>null</tt> value
     */
    int lastIndexOfNull();

    /**
     * Wraps passed value with JsonString and delegates the call to
     * {@link java.util.List#remove(Object)} method.
     * @param value string to wrap
     * @return <tt>true</tt> if this JSON array contained the specified JSON string
     */
    boolean remove( String value );

    /**
     * Wraps passed value with JsonNumber and delegates the call to
     * {@link java.util.List#remove(Object)} method.
     * @param value number to wrap
     * @return <tt>true</tt> if this JSON array contained the specified JSON number
     */
    boolean remove( Number value );

    /**
     * Wraps passed value with JsonBoolean and delegates the call to
     * {@link java.util.List#remove(Object)} method.
     * @param value boolean to wrap
     * @return <tt>true</tt> if this JSON array contained the specified JSON boolean
     */
    boolean remove( Boolean value );

    /**
     * Delegates the call to {@link java.util.List#remove(Object)} method.
     * @param value JSON value
     * @return <tt>true</tt> if this JSON array contained the specified JSON value
     */
    boolean remove( JsonValue value );

    /**
     * Delegates the call to {@link java.util.List#remove(Object)} method.
     * @return <tt>true</tt> if this JSON array contained the <tt>null</tt> value
     */
    boolean removeNull();

    /**
     * Wraps passed value with JsonString and delegates the call to
     * {@link java.util.List#set(int, Object)} method.
     * @param index index of the JSON value to replace 
     * @param value string to wrap
     * @return the JSON value previously held at the specified position
     */
    JsonValue set( int index, String value );

    /**
     * Wraps passed value with JsonNumber and delegates the call to
     * {@link java.util.List#set(int, Object)} method.
     * @param index index of the JSON value to replace 
     * @param value number to wrap
     * @return the JSON value previously held at the specified position
     */
    JsonValue set( int index, Number value );

    /**
     * Wraps passed value with JsonBoolean and delegates the call to
     * {@link java.util.List#set(int, Object)} method.
     * @param index index of the JSON value to replace 
     * @param value boolean to wrap
     * @return the JSON value previously held at the specified position
     */
    JsonValue set( int index, Boolean value );

    /**
     * Delegates the call to {@link java.util.List#set(int, Object)} method.
     * @param index index of the JSON value to replace 
     * @param value JSON value
     * @return the JSON value previously held at the specified position
     */
    @Override
    JsonValue set( int index, JsonValue value );

    /**
     * Delegates the call to
     * {@link java.util.List#set(int, Object)} method.
     * @param index index of the JSON value to replace with <tt>null</tt>
     * @return the JSON value previously held at the specified position
     */
    JsonValue setNull( int index );

    /**
     * Translates this JSON array to Java array.
     * @return java array 
     */
    @Override
    JsonValue[] toArray();

    /**
     * Serializes this JSON array to the writer.
     * @param output to write to
     * @throws IOException if some I/O error occurs
     * @throws JsonException if wrong JSON is detected
     */
    void writeTo( JsonWriter output ) throws IOException, JsonException;

    /**
     * Serializes this JSON array to the writer.
     * @param output to write to
     * @throws IOException if some I/O error occurs
     * @throws JsonException if wrong JSON is detected
     */
    void writeTo( Writer output ) throws IOException, JsonException;

    /**
     * Serializes this JSON array to the stream using <code>UTF-8</code> character set.
     * @param output to write to
     * @throws IOException if some I/O error occurs
     * @throws JsonException if wrong JSON is detected
     */
    void writeTo( OutputStream output ) throws IOException, JsonException;

    /**
     * Serializes this JSON array to the writer using specified character set.
     * @param output to write to
     * @param charset character set
     * @throws IOException if some I/O error occurs
     * @throws JsonException if wrong JSON is detected
     */
    void writeTo( OutputStream output, Charset charset ) throws IOException, JsonException;

    /**
     * Clones this JSON array.
     * @return new JSON array clone.
     */
    @Override
    JsonArray clone();

}
