/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.tez.runtime.api.events;

import org.apache.hadoop.classification.InterfaceAudience.Public;
import org.apache.tez.runtime.api.Event;

import java.util.Objects;

/**
 * Event generated by an Input to indicate error when trying to retrieve data.
 * This is not necessarily a fatal event - it's an indication to the AM to retry
 * source data generation.
 */
@Public
public final class InputReadErrorEvent extends Event {

  /**
   * Diagnostics/trace of the error that occurred on the Input's edge.
   */
  private final String diagnostics;

  /**
   * Index of the physical edge on which the error occurred.
   */
  private final int index;

  /**
   * Version of the data on which the error occurred.
   */
  private final int version;

  /**
   * Number of failures.
   */
  private final int numFailures;

  private InputReadErrorEvent(final String diagnostics, final int index,
                              final int version, final int numFailures) {
    super();
    this.diagnostics = diagnostics;
    this.index = index;
    this.version = version;
    this.numFailures = numFailures;
  }

  public static InputReadErrorEvent create(String diagnostics, int index,
                                           int version) {
    return create(diagnostics, index, version, 1);
  }

  /**
   * Create an InputReadErrorEvent.
   */
  public static InputReadErrorEvent create(final String diagnostics, final int index,
      final int version, final int numFailures) {
    return new InputReadErrorEvent(diagnostics, index, version, numFailures);
  }

  public String getDiagnostics() {
    return diagnostics;
  }

  public int getIndex() {
    return index;
  }

  public int getVersion() {
    return version;
  }

  /**
   * @return number of failures
   */
  public int getNumFailures() {
    return numFailures;
  }

  @Override
  public int hashCode() {
    return Objects.hash(index, version);
  }

  @Override
  public boolean equals(final Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    InputReadErrorEvent that = (InputReadErrorEvent) o;
    return index == that.index && version == that.version;
  }
}
