/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.core.query.lucene;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.jcr.ItemNotFoundException;
import javax.jcr.RepositoryException;
import org.apache.commons.io.IOExceptionWithCause;
import org.apache.jackrabbit.core.HierarchyManager;
import org.apache.jackrabbit.core.RepositoryImpl;
import org.apache.jackrabbit.core.cluster.ClusterException;
import org.apache.jackrabbit.core.cluster.ClusterNode;
import org.apache.jackrabbit.core.id.NodeId;
import org.apache.jackrabbit.core.persistence.IterablePersistenceManager;
import org.apache.jackrabbit.core.persistence.PersistenceManager;
import org.apache.jackrabbit.core.query.lucene.CachingMultiIndexReader;
import org.apache.jackrabbit.core.query.lucene.ConsistencyCheckError;
import org.apache.jackrabbit.core.query.lucene.FieldNames;
import org.apache.jackrabbit.core.query.lucene.FieldSelectors;
import org.apache.jackrabbit.core.query.lucene.MultiIndex;
import org.apache.jackrabbit.core.query.lucene.SearchIndex;
import org.apache.jackrabbit.core.state.ChildNodeEntry;
import org.apache.jackrabbit.core.state.ItemState;
import org.apache.jackrabbit.core.state.ItemStateException;
import org.apache.jackrabbit.core.state.ItemStateManager;
import org.apache.jackrabbit.core.state.NoSuchItemStateException;
import org.apache.jackrabbit.core.state.NodeState;
import org.apache.jackrabbit.spi.Path;
import org.apache.lucene.document.Document;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConsistencyCheck {
    private static final Logger log = LoggerFactory.getLogger(ConsistencyCheck.class);
    private static final int NODESATONCE = Integer.getInteger("org.apache.jackrabbit.checker.nodesatonce", 8192);
    private final SearchIndex handler;
    private final ItemStateManager stateMgr;
    private IterablePersistenceManager pm;
    private final MultiIndex index;
    private Map<NodeId, Boolean> nodeIds;
    private Set<Path> excludedPaths;
    private final List<ConsistencyCheckError> errors = new ArrayList<ConsistencyCheckError>();

    private ConsistencyCheck(MultiIndex index, SearchIndex handler, Set<NodeId> excludedIds) {
        this.index = index;
        this.handler = handler;
        HierarchyManager hierarchyManager = handler.getContext().getHierarchyManager();
        this.excludedPaths = new HashSet<Path>(excludedIds.size());
        for (NodeId excludedId : excludedIds) {
            try {
                Path path = hierarchyManager.getPath(excludedId);
                this.excludedPaths.add(path);
            }
            catch (ItemNotFoundException e) {
                log.warn("Excluded node does not exist");
            }
            catch (RepositoryException e) {
                log.error("Failed to get excluded path", (Throwable)e);
            }
        }
        this.stateMgr = handler.getContext().getItemStateManager();
        PersistenceManager pm = handler.getContext().getPersistenceManager();
        if (pm instanceof IterablePersistenceManager) {
            this.pm = (IterablePersistenceManager)pm;
        }
    }

    static ConsistencyCheck run(MultiIndex index, SearchIndex handler, Set<NodeId> excludedIds) throws IOException {
        ConsistencyCheck check = new ConsistencyCheck(index, handler, excludedIds);
        check.run();
        return check;
    }

    public void repair(boolean ignoreFailure) throws IOException {
        if (this.errors.size() == 0) {
            log.info("No errors found.");
            return;
        }
        int notRepairable = 0;
        for (ConsistencyCheckError error : this.errors) {
            try {
                if (error.repairable()) {
                    error.repair();
                    continue;
                }
                log.warn("Not repairable: " + error);
                ++notRepairable;
            }
            catch (Exception e) {
                if (ignoreFailure) {
                    log.warn("Exception while repairing: " + error, (Throwable)e);
                    continue;
                }
                if (e instanceof IOException) {
                    throw (IOException)e;
                }
                throw new IOExceptionWithCause((Throwable)e);
            }
        }
        log.info("Repaired " + (this.errors.size() - notRepairable) + " errors.");
        if (notRepairable > 0) {
            log.warn("" + notRepairable + " error(s) not repairable.");
        }
    }

    public List<ConsistencyCheckError> getErrors() {
        return new ArrayList<ConsistencyCheckError>(this.errors);
    }

    private void run() throws IOException {
        log.info("Checking index of workspace " + this.handler.getContext().getWorkspace());
        this.loadNodes();
        if (this.nodeIds != null) {
            this.checkIndexConsistency();
            this.checkIndexCompleteness();
        }
    }

    public void doubleCheckErrors() {
        if (!this.errors.isEmpty()) {
            log.info("Double checking errors");
            ClusterNode clusterNode = this.handler.getContext().getClusterNode();
            if (clusterNode != null) {
                try {
                    clusterNode.sync();
                }
                catch (ClusterException e) {
                    log.error("Could not sync cluster node for double checking errors");
                }
            }
            Iterator<ConsistencyCheckError> iterator = this.errors.iterator();
            while (iterator.hasNext()) {
                try {
                    ConsistencyCheckError error = iterator.next();
                    if (error.doubleCheck(this.handler, this.stateMgr)) continue;
                    log.info("False positive: " + error.toString());
                    iterator.remove();
                }
                catch (RepositoryException e) {
                    log.error("Failed to double check consistency error", (Throwable)e);
                }
                catch (IOException e) {
                    log.error("Failed to double check consistency error", (Throwable)e);
                }
            }
        }
    }

    private void loadNodes() {
        log.info("Loading nodes");
        try {
            int count = 0;
            HashMap<NodeId, Boolean> nodeIds = new HashMap<NodeId, Boolean>();
            List<NodeId> batch = this.pm.getAllNodeIds(null, NODESATONCE);
            NodeId lastId = null;
            while (!batch.isEmpty()) {
                Iterator<NodeId> i$ = batch.iterator();
                while (i$.hasNext()) {
                    NodeId nodeId;
                    lastId = nodeId = i$.next();
                    if (++count % 1000 == 0) {
                        log.info(this.pm + ": loaded " + count + " node ids...");
                    }
                    nodeIds.put(nodeId, Boolean.FALSE);
                }
                batch = this.pm.getAllNodeIds(lastId, NODESATONCE);
            }
            if (this.pm.exists(lastId)) {
                this.nodeIds = nodeIds;
            } else {
                log.info("Failed to read all nodes, starting over");
                this.loadNodes();
            }
        }
        catch (ItemStateException e) {
            log.error("Exception while loading items to check", (Throwable)e);
        }
        catch (RepositoryException e) {
            log.error("Exception while loading items to check", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void checkIndexConsistency() throws IOException {
        NodeId id;
        log.info("Checking index consistency");
        HashSet<NodeId> multipleEntries = new HashSet<NodeId>();
        CachingMultiIndexReader reader = this.index.getIndexReader();
        try {
            for (int i = 0; i < reader.maxDoc(); ++i) {
                if (i > 10 && i % (reader.maxDoc() / 5) == 0) {
                    long progress = Math.round(100.0 * (double)i / (double)((float)reader.maxDoc() * 2.0f));
                    log.info("progress: " + progress + "%");
                }
                if (reader.isDeleted(i)) continue;
                Document d = reader.document(i, FieldSelectors.UUID);
                id = new NodeId(d.get(FieldNames.UUID));
                boolean nodeExists = this.nodeIds.containsKey(id);
                if (nodeExists) {
                    Boolean alreadyIndexed = this.nodeIds.put(id, Boolean.TRUE);
                    if (!alreadyIndexed.booleanValue()) continue;
                    multipleEntries.add(id);
                    continue;
                }
                this.errors.add(new NodeDeleted(id));
            }
        }
        finally {
            reader.release();
        }
        for (NodeId id2 : multipleEntries) {
            this.errors.add(new MultipleEntries(id2));
        }
        reader = this.index.getIndexReader();
        try {
            for (int i = 0; i < reader.maxDoc(); ++i) {
                boolean parentIndexed;
                String parent;
                Document d;
                if (i > 10 && i % (reader.maxDoc() / 5) == 0) {
                    long progress = Math.round(100.0 * (double)i / (double)((float)reader.maxDoc() * 2.0f));
                    log.info("progress: " + (progress + 50L) + "%");
                }
                if (reader.isDeleted(i) || !this.nodeIds.containsKey(id = new NodeId((d = reader.document(i, FieldSelectors.UUID_AND_PARENT)).get(FieldNames.UUID))) || (parent = d.get(FieldNames.PARENT)) == null || parent.isEmpty()) continue;
                NodeId parentId = new NodeId(parent);
                boolean parentExists = this.nodeIds.containsKey(parentId);
                boolean bl = parentIndexed = parentExists && this.nodeIds.get(parentId) != false;
                if (parentIndexed || id.equals(RepositoryImpl.SYSTEM_ROOT_NODE_ID) && parentId.equals(RepositoryImpl.ROOT_NODE_ID)) continue;
                if (parentExists) {
                    this.errors.add(new MissingAncestor(id, parentId));
                    continue;
                }
                try {
                    ItemState itemState = this.stateMgr.getItemState(id);
                    if (parentId.equals(itemState.getParentId())) {
                        this.errors.add(new UnknownParent(id, parentId));
                        continue;
                    }
                    this.errors.add(new WrongParent(id, parentId, itemState.getParentId()));
                    continue;
                }
                catch (ItemStateException ignored) {
                    // empty catch block
                }
            }
        }
        finally {
            reader.release();
        }
    }

    private void checkIndexCompleteness() {
        log.info("Checking index completeness");
        int i = 0;
        int size = this.nodeIds.size();
        for (Map.Entry<NodeId, Boolean> entry : this.nodeIds.entrySet()) {
            NodeId nodeId = entry.getKey();
            boolean indexed = entry.getValue();
            try {
                NodeState nodeState;
                if (++i > 10 && i % (size / 10) == 0) {
                    long progress = Math.round(100.0 * (double)i / (double)size);
                    log.info("progress: " + progress + "%");
                }
                if (indexed || this.isExcluded(nodeId) || (nodeState = this.getNodeState(nodeId)) == null || this.isBrokenNode(nodeId, nodeState)) continue;
                this.errors.add(new NodeAdded(nodeId));
            }
            catch (ItemStateException e) {
                log.error("Failed to check node: " + nodeId, (Throwable)e);
            }
        }
    }

    private boolean isExcluded(NodeId id) {
        try {
            HierarchyManager hierarchyManager = this.handler.getContext().getHierarchyManager();
            Path path = hierarchyManager.getPath(id);
            for (Path excludedPath : this.excludedPaths) {
                if (!excludedPath.isEquivalentTo(path) && !excludedPath.isAncestorOf(path)) continue;
                return true;
            }
        }
        catch (RepositoryException repositoryException) {
            // empty catch block
        }
        return false;
    }

    private NodeState getNodeState(NodeId nodeId) throws ItemStateException {
        try {
            return (NodeState)this.stateMgr.getItemState(nodeId);
        }
        catch (NoSuchItemStateException e) {
            return null;
        }
    }

    private boolean isBrokenNode(NodeId nodeId, NodeState nodeState) throws ItemStateException {
        NodeId parentId = nodeState.getParentId();
        if (parentId != null) {
            NodeState parentState = this.getNodeState(parentId);
            if (parentState == null) {
                log.warn("Node missing from index is orphaned node: " + nodeId);
                return true;
            }
            if (!parentState.hasChildNodeEntry(nodeId)) {
                log.warn("Node missing from index is abandoned node: " + nodeId);
                return true;
            }
        }
        return false;
    }

    private String getPath(NodeState node) {
        String uuid = node.getNodeId().toString();
        StringBuilder path = new StringBuilder();
        ArrayList<ChildNodeEntry> elements = new ArrayList<ChildNodeEntry>();
        try {
            while (node.getParentId() != null) {
                NodeId parentId = node.getParentId();
                NodeState parent = (NodeState)this.stateMgr.getItemState(parentId);
                ChildNodeEntry entry = parent.getChildNodeEntry(node.getNodeId());
                elements.add(entry);
                node = parent;
            }
            for (int i = elements.size() - 1; i > -1; --i) {
                ChildNodeEntry entry = (ChildNodeEntry)elements.get(i);
                path.append('/').append(entry.getName().getLocalName());
                if (entry.getIndex() <= 1) continue;
                path.append('[').append(entry.getIndex()).append(']');
            }
            if (path.length() == 0) {
                path.append('/');
            }
            return path.toString();
        }
        catch (ItemStateException e) {
            return uuid;
        }
    }

    private class NodeAdded
    extends ConsistencyCheckError {
        NodeAdded(NodeId id) {
            super("Node " + id + " is missing.", id);
        }

        @Override
        public boolean repairable() {
            return true;
        }

        @Override
        void repair() throws Exception {
            try {
                NodeState nodeState = (NodeState)ConsistencyCheck.this.stateMgr.getItemState(this.id);
                log.info("Adding missing node to index: " + ConsistencyCheck.this.getPath(nodeState));
                Iterator<NodeId> remove = Collections.emptyList().iterator();
                Iterator<NodeState> add = Collections.singletonList(nodeState).iterator();
                ConsistencyCheck.this.handler.updateNodes(remove, add);
            }
            catch (NoSuchItemStateException e) {
                log.info("Not adding missing node because node no longer exists");
            }
        }

        @Override
        boolean doubleCheck(SearchIndex handler, ItemStateManager stateManager) throws RepositoryException, IOException {
            List<Document> documents = handler.getNodeDocuments(this.id);
            return documents.isEmpty() && stateManager.hasItemState(this.id);
        }
    }

    private class NodeDeleted
    extends ConsistencyCheckError {
        NodeDeleted(NodeId id) {
            super("Node " + id + " no longer exists.", id);
        }

        @Override
        public boolean repairable() {
            return true;
        }

        @Override
        public void repair() throws IOException {
            log.info("Removing deleted node from index: " + this.id);
            ConsistencyCheck.this.index.removeDocument(this.id);
        }

        @Override
        boolean doubleCheck(SearchIndex handler, ItemStateManager stateManager) throws RepositoryException, IOException {
            List<Document> documents = handler.getNodeDocuments(this.id);
            return !documents.isEmpty() && !stateManager.hasItemState(this.id);
        }
    }

    private class MultipleEntries
    extends ConsistencyCheckError {
        MultipleEntries(NodeId id) {
            super("Multiple entries found for node " + id, id);
        }

        @Override
        public boolean repairable() {
            return true;
        }

        @Override
        public void repair() throws Exception {
            ConsistencyCheck.this.index.removeAllDocuments(this.id);
            try {
                NodeState node = (NodeState)ConsistencyCheck.this.stateMgr.getItemState(this.id);
                log.info("Re-indexing duplicate node occurrences in index: " + ConsistencyCheck.this.getPath(node));
                Document d = ConsistencyCheck.this.index.createDocument(node);
                ConsistencyCheck.this.index.addDocument(d);
                ConsistencyCheck.this.nodeIds.put(node.getNodeId(), Boolean.TRUE);
            }
            catch (NoSuchItemStateException e) {
                log.info("Not re-indexing node with multiple occurrences because node no longer exists");
            }
        }

        @Override
        boolean doubleCheck(SearchIndex handler, ItemStateManager stateManager) throws RepositoryException, IOException {
            return handler.getNodeDocuments(this.id).size() > 1;
        }
    }

    private class WrongParent
    extends ConsistencyCheckError {
        private NodeId indexedParentId;

        private WrongParent(NodeId id, NodeId indexedParentId, NodeId actualParentId) {
            super("Node " + id + " has wrong parent: " + indexedParentId + ", should be : " + actualParentId, id);
            this.indexedParentId = indexedParentId;
        }

        @Override
        public boolean repairable() {
            return true;
        }

        @Override
        void repair() throws Exception {
            ConsistencyCheck.this.index.removeAllDocuments(this.id);
            try {
                NodeState node = (NodeState)ConsistencyCheck.this.stateMgr.getItemState(this.id);
                log.info("Re-indexing node with wrong parent in index: " + ConsistencyCheck.this.getPath(node));
                Document d = ConsistencyCheck.this.index.createDocument(node);
                ConsistencyCheck.this.index.addDocument(d);
                ConsistencyCheck.this.nodeIds.put(node.getNodeId(), Boolean.TRUE);
            }
            catch (NoSuchItemStateException e) {
                log.info("Not re-indexing node with wrong parent because node no longer exists");
            }
        }

        @Override
        boolean doubleCheck(SearchIndex handler, ItemStateManager stateManager) throws RepositoryException, IOException {
            List<Document> documents = handler.getNodeDocuments(this.id);
            for (Document document : documents) {
                NodeId parentId;
                String parent = document.get(FieldNames.PARENT);
                if (parent == null || parent.isEmpty() || !(parentId = new NodeId(parent)).equals(this.indexedParentId) || stateManager.hasItemState(parentId)) continue;
                return true;
            }
            return false;
        }
    }

    private static class UnknownParent
    extends ConsistencyCheckError {
        private NodeId parentId;

        private UnknownParent(NodeId id, NodeId parentId) {
            super("Node " + id + " has unknown parent: " + parentId, id);
            this.parentId = parentId;
        }

        @Override
        public boolean repairable() {
            return false;
        }

        @Override
        public void repair() {
            log.warn("Unknown parent for " + this.id + " cannot be repaired");
        }

        @Override
        boolean doubleCheck(SearchIndex handler, ItemStateManager stateManager) throws IOException, RepositoryException {
            List<Document> documents = handler.getNodeDocuments(this.id);
            for (Document document : documents) {
                NodeId parentId;
                String parent = document.get(FieldNames.PARENT);
                if (parent == null || parent.isEmpty() || !(parentId = new NodeId(parent)).equals(this.parentId) || stateManager.hasItemState(parentId)) continue;
                return true;
            }
            return false;
        }
    }

    private class MissingAncestor
    extends ConsistencyCheckError {
        private final NodeId parentId;

        private MissingAncestor(NodeId id, NodeId parentId) {
            super("Parent of " + id + " missing in index. Parent: " + parentId, id);
            this.parentId = parentId;
        }

        @Override
        public boolean repairable() {
            return true;
        }

        @Override
        public void repair() throws Exception {
            NodeId ancestorId = this.parentId;
            while (ancestorId != null && ConsistencyCheck.this.nodeIds.containsKey(ancestorId) && ((Boolean)ConsistencyCheck.this.nodeIds.get(ancestorId)).booleanValue()) {
                NodeState n = (NodeState)ConsistencyCheck.this.stateMgr.getItemState(ancestorId);
                log.info("Repairing missing node " + ConsistencyCheck.this.getPath(n) + " (" + ancestorId + ")");
                Document d = ConsistencyCheck.this.index.createDocument(n);
                ConsistencyCheck.this.index.addDocument(d);
                ConsistencyCheck.this.nodeIds.put(n.getNodeId(), Boolean.TRUE);
                ancestorId = n.getParentId();
            }
        }

        @Override
        boolean doubleCheck(SearchIndex handler, ItemStateManager stateManager) throws RepositoryException, IOException {
            List<Document> documents = handler.getNodeDocuments(this.id);
            for (Document document : documents) {
                NodeId parentId;
                String parent = document.get(FieldNames.PARENT);
                if (parent == null || parent.isEmpty() || !handler.getNodeDocuments(parentId = new NodeId(parent)).isEmpty()) continue;
                return true;
            }
            return false;
        }
    }
}

