/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.common.utils;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.RateLimiter;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileDescriptor;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.ArchiveOutputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorOutputStream;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;

public class TarGzCompressionUtils {
    public static final long NO_DISK_WRITE_RATE_LIMIT = -1L;
    public static final long SYNC_DISK_WRITE_WITH_UPSTREAM_RATE = 0L;
    private static final int DEFAULT_BUFFER_SIZE = 0x400000;
    public static final String TAR_GZ_FILE_EXTENSION = ".tar.gz";
    private static final char ENTRY_NAME_SEPARATOR = '/';

    private TarGzCompressionUtils() {
    }

    public static void createTarGzFile(File inputFile, File outputFile) throws IOException {
        TarGzCompressionUtils.createTarGzFile(new File[]{inputFile}, outputFile);
    }

    public static void createTarGzFile(File[] inputFiles, File outputFile) throws IOException {
        Preconditions.checkArgument((boolean)outputFile.getName().endsWith(TAR_GZ_FILE_EXTENSION), (String)"Output file: %s does not have '.tar.gz' file extension", (Object)outputFile);
        try (OutputStream fileOut = Files.newOutputStream(outputFile.toPath(), new OpenOption[0]);
             BufferedOutputStream bufferedOut = new BufferedOutputStream(fileOut);
             GzipCompressorOutputStream gzipOut = new GzipCompressorOutputStream((OutputStream)bufferedOut);
             TarArchiveOutputStream tarGzOut = new TarArchiveOutputStream((OutputStream)gzipOut);){
            tarGzOut.setBigNumberMode(1);
            tarGzOut.setLongFileMode(2);
            for (File inputFile : inputFiles) {
                TarGzCompressionUtils.addFileToTarGz((ArchiveOutputStream)tarGzOut, inputFile, "");
            }
        }
    }

    private static void addFileToTarGz(ArchiveOutputStream tarGzOut, File file, String baseEntryName) throws IOException {
        String entryName = baseEntryName + file.getName();
        TarArchiveEntry entry = new TarArchiveEntry(file, entryName);
        tarGzOut.putArchiveEntry((ArchiveEntry)entry);
        if (file.isFile()) {
            try (InputStream in = Files.newInputStream(file.toPath(), new OpenOption[0]);){
                IOUtils.copy((InputStream)in, (OutputStream)tarGzOut);
            }
            tarGzOut.closeArchiveEntry();
        } else {
            tarGzOut.closeArchiveEntry();
            File[] children = file.listFiles();
            assert (children != null);
            String baseEntryNameForChildren = entryName + '/';
            for (File child : children) {
                TarGzCompressionUtils.addFileToTarGz(tarGzOut, child, baseEntryNameForChildren);
            }
        }
    }

    public static List<File> untar(File inputFile, File outputDir) throws IOException {
        try (InputStream fileIn = Files.newInputStream(inputFile.toPath(), new OpenOption[0]);){
            List<File> list = TarGzCompressionUtils.untar(fileIn, outputDir);
            return list;
        }
    }

    public static List<File> untar(InputStream inputStream, File outputDir) throws IOException {
        return TarGzCompressionUtils.untarWithRateLimiter(inputStream, outputDir, -1L);
    }

    public static List<File> untarWithRateLimiter(InputStream inputStream, File outputDir, long maxStreamRateInByte) throws IOException {
        String outputDirCanonicalPath = outputDir.getCanonicalPath();
        if (!outputDirCanonicalPath.endsWith(File.separator)) {
            outputDirCanonicalPath = outputDirCanonicalPath + File.separator;
        }
        ArrayList<File> untarredFiles = new ArrayList<File>();
        try (BufferedInputStream bufferedIn = new BufferedInputStream(inputStream);
             GzipCompressorInputStream gzipIn = new GzipCompressorInputStream((InputStream)bufferedIn);
             TarArchiveInputStream tarGzIn = new TarArchiveInputStream((InputStream)gzipIn);){
            ArchiveEntry entry;
            while ((entry = tarGzIn.getNextEntry()) != null) {
                String entryName = entry.getName();
                String[] parts = StringUtils.split((String)entryName, (char)'/');
                File outputFile = outputDir;
                for (String part : parts) {
                    outputFile = new File(outputFile, part);
                }
                if (entry.isDirectory()) {
                    if (!outputFile.getCanonicalPath().startsWith(outputDirCanonicalPath)) {
                        throw new IOException(String.format("Trying to create directory: %s outside of the output directory: %s", outputFile, outputDir));
                    }
                    if (!outputFile.isDirectory() && !outputFile.mkdirs()) {
                        throw new IOException(String.format("Failed to create directory: %s", outputFile));
                    }
                } else {
                    File parentFile = outputFile.getParentFile();
                    String parentFileCanonicalPath = parentFile.getCanonicalPath();
                    if (!parentFileCanonicalPath.endsWith(File.separator)) {
                        parentFileCanonicalPath = parentFileCanonicalPath + File.separator;
                    }
                    if (!parentFileCanonicalPath.startsWith(outputDirCanonicalPath)) {
                        throw new IOException(String.format("Trying to create directory: %s outside of the output directory: %s", parentFile, outputDir));
                    }
                    if (!parentFile.isDirectory() && !parentFile.mkdirs()) {
                        throw new IOException(String.format("Failed to create directory: %s", parentFile));
                    }
                    try (FileOutputStream out = new FileOutputStream(outputFile.toPath().toString());){
                        if (maxStreamRateInByte != -1L) {
                            TarGzCompressionUtils.copyWithRateLimiter((InputStream)tarGzIn, out, maxStreamRateInByte);
                        } else {
                            IOUtils.copy((InputStream)tarGzIn, (OutputStream)out);
                        }
                    }
                }
                untarredFiles.add(outputFile);
            }
        }
        return untarredFiles;
    }

    public static void untarOneFile(File inputFile, String fileName, File outputFile) throws IOException {
        try (InputStream fileIn = Files.newInputStream(inputFile.toPath(), new OpenOption[0]);
             BufferedInputStream bufferedIn = new BufferedInputStream(fileIn);
             GzipCompressorInputStream gzipIn = new GzipCompressorInputStream((InputStream)bufferedIn);
             TarArchiveInputStream tarGzIn = new TarArchiveInputStream((InputStream)gzipIn);){
            ArchiveEntry entry;
            while ((entry = tarGzIn.getNextEntry()) != null) {
                if (entry.isDirectory()) continue;
                String entryName = entry.getName();
                String[] parts = StringUtils.split((String)entryName, (char)'/');
                if (parts.length <= 0 || !parts[parts.length - 1].equals(fileName)) continue;
                try (OutputStream out = Files.newOutputStream(outputFile.toPath(), new OpenOption[0]);){
                    IOUtils.copy((InputStream)tarGzIn, (OutputStream)out);
                }
                return;
            }
            throw new IOException(String.format("Failed to find file: %s in: %s", fileName, inputFile));
        }
    }

    public static long copyWithRateLimiter(InputStream inputStream, FileOutputStream outputStream, long maxStreamRateInByte) throws IOException {
        long count;
        Preconditions.checkState((inputStream != null ? 1 : 0) != 0, (Object)"inputStream is null");
        Preconditions.checkState((outputStream != null ? 1 : 0) != 0, (Object)"outputStream is null");
        FileDescriptor fd = outputStream.getFD();
        byte[] buffer = new byte[0x400000];
        if (maxStreamRateInByte == 0L) {
            int n;
            count = 0L;
            while (-1 != (n = inputStream.read(buffer))) {
                outputStream.write(buffer, 0, n);
                fd.sync();
                count += (long)n;
            }
        } else {
            int n;
            RateLimiter rateLimiter = RateLimiter.create((double)maxStreamRateInByte);
            count = 0L;
            while (-1 != (n = inputStream.read(buffer))) {
                rateLimiter.acquire(n);
                outputStream.write(buffer, 0, n);
                fd.sync();
                count += (long)n;
            }
        }
        return count;
    }
}

