/*
 * Decompiled with CFR 0.152.
 */
package org.apache.parquet.io.api;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.Arrays;
import org.apache.parquet.io.api.Binary;
import org.junit.Assert;
import org.junit.Test;

public class TestBinary {
    private static final String testString = "test-123";
    private static final String UTF8 = "UTF-8";
    private static final BinaryFactory BYTE_ARRAY_BACKED_BF = new BinaryFactory(){

        @Override
        public BinaryFactory.BinaryAndOriginal get(byte[] bytes, boolean reused) throws Exception {
            byte[] orig = Arrays.copyOf(bytes, bytes.length);
            if (reused) {
                return new BinaryFactory.BinaryAndOriginal(Binary.fromReusedByteArray((byte[])orig), orig);
            }
            return new BinaryFactory.BinaryAndOriginal(Binary.fromConstantByteArray((byte[])orig), orig);
        }
    };
    private static final BinaryFactory BYTE_ARRAY_SLICE_BACKED_BF = new BinaryFactory(){

        @Override
        public BinaryFactory.BinaryAndOriginal get(byte[] bytes, boolean reused) throws Exception {
            byte[] orig = TestBinary.padded(bytes);
            Binary b = reused ? Binary.fromReusedByteArray((byte[])orig, (int)5, (int)bytes.length) : Binary.fromConstantByteArray((byte[])orig, (int)5, (int)bytes.length);
            Assert.assertArrayEquals((byte[])bytes, (byte[])b.getBytes());
            return new BinaryFactory.BinaryAndOriginal(b, orig);
        }
    };
    private static final BinaryFactory BUFFER_BF = new BinaryFactory(){

        @Override
        public BinaryFactory.BinaryAndOriginal get(byte[] bytes, boolean reused) throws Exception {
            byte[] orig = TestBinary.padded(bytes);
            ByteBuffer buff = ByteBuffer.wrap(orig, 5, bytes.length);
            Binary b = reused ? Binary.fromReusedByteBuffer((ByteBuffer)buff) : Binary.fromConstantByteBuffer((ByteBuffer)buff);
            buff.mark();
            Assert.assertArrayEquals((byte[])bytes, (byte[])b.getBytes());
            buff.reset();
            return new BinaryFactory.BinaryAndOriginal(b, orig);
        }
    };
    private static final BinaryFactory STRING_BF = new BinaryFactory(){

        @Override
        public BinaryFactory.BinaryAndOriginal get(byte[] bytes, boolean reused) throws Exception {
            Binary b = Binary.fromString((String)new String(bytes, TestBinary.UTF8));
            return new BinaryFactory.BinaryAndOriginal(b, b.getBytesUnsafe());
        }
    };

    private static void mutate(byte[] bytes) {
        for (int i = 0; i < bytes.length; ++i) {
            bytes[i] = (byte)(bytes[i] + 1);
        }
    }

    private static byte[] padded(byte[] bytes) {
        int i;
        byte[] padded = new byte[bytes.length + 10];
        for (i = 0; i < 5; ++i) {
            padded[i] = (byte)i;
        }
        System.arraycopy(bytes, 0, padded, 5, bytes.length);
        for (i = 0; i < 5; ++i) {
            padded[i + 5 + bytes.length] = (byte)i;
        }
        return padded;
    }

    @Test
    public void testByteArrayBackedBinary() throws Exception {
        this.testBinary(BYTE_ARRAY_BACKED_BF, true);
        this.testBinary(BYTE_ARRAY_BACKED_BF, false);
    }

    @Test
    public void testByteArraySliceBackedBinary() throws Exception {
        this.testBinary(BYTE_ARRAY_SLICE_BACKED_BF, true);
        this.testBinary(BYTE_ARRAY_SLICE_BACKED_BF, false);
    }

    @Test
    public void testByteBufferBackedBinary() throws Exception {
        this.testBinary(BUFFER_BF, true);
        this.testBinary(BUFFER_BF, false);
    }

    @Test
    public void testEqualityMethods() throws Exception {
        Binary bin1 = Binary.fromConstantByteArray((byte[])"alice".getBytes(), (int)1, (int)3);
        Binary bin2 = Binary.fromConstantByteBuffer((ByteBuffer)ByteBuffer.wrap("alice".getBytes(), 1, 3));
        Assert.assertEquals((Object)bin1, (Object)bin2);
    }

    @Test
    public void testWriteAllTo() throws Exception {
        byte[] orig = new byte[]{10, 9, 8, 7, 6, 5, 4, 3, 2, 1};
        this.testWriteAllToHelper(Binary.fromConstantByteBuffer((ByteBuffer)ByteBuffer.wrap(orig)), orig);
        ByteBuffer buf = ByteBuffer.allocateDirect(orig.length);
        buf.put(orig);
        buf.flip();
        this.testWriteAllToHelper(Binary.fromConstantByteBuffer((ByteBuffer)buf), orig);
    }

    private void testWriteAllToHelper(Binary binary, byte[] orig) throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream(orig.length);
        binary.writeTo((OutputStream)out);
        Assert.assertArrayEquals((byte[])orig, (byte[])out.toByteArray());
    }

    @Test
    public void testFromStringBinary() throws Exception {
        this.testBinary(STRING_BF, false);
    }

    private void testSlice(BinaryFactory bf, boolean reused) throws Exception {
        BinaryFactory.BinaryAndOriginal bao = bf.get(testString.getBytes(UTF8), reused);
        Assert.assertArrayEquals((byte[])testString.getBytes(UTF8), (byte[])bao.binary.slice(0, testString.length()).getBytesUnsafe());
        Assert.assertArrayEquals((byte[])"123".getBytes(UTF8), (byte[])bao.binary.slice(5, 3).getBytesUnsafe());
    }

    private void testConstantCopy(BinaryFactory bf) throws Exception {
        BinaryFactory.BinaryAndOriginal bao = bf.get(testString.getBytes(UTF8), false);
        Assert.assertEquals((Object)false, (Object)bao.binary.isBackingBytesReused());
        Assert.assertArrayEquals((byte[])testString.getBytes(UTF8), (byte[])bao.binary.getBytes());
        Assert.assertArrayEquals((byte[])testString.getBytes(UTF8), (byte[])bao.binary.getBytesUnsafe());
        Assert.assertArrayEquals((byte[])testString.getBytes(UTF8), (byte[])bao.binary.copy().getBytesUnsafe());
        Assert.assertArrayEquals((byte[])testString.getBytes(UTF8), (byte[])bao.binary.copy().getBytes());
        bao = bf.get(testString.getBytes(UTF8), false);
        Assert.assertEquals((Object)false, (Object)bao.binary.isBackingBytesReused());
        Binary copy = bao.binary.copy();
        Assert.assertSame((Object)copy, (Object)bao.binary);
    }

    private void testReusedCopy(BinaryFactory bf) throws Exception {
        BinaryFactory.BinaryAndOriginal bao = bf.get(testString.getBytes(UTF8), true);
        Assert.assertEquals((Object)true, (Object)bao.binary.isBackingBytesReused());
        Assert.assertArrayEquals((byte[])testString.getBytes(UTF8), (byte[])bao.binary.getBytes());
        Assert.assertArrayEquals((byte[])testString.getBytes(UTF8), (byte[])bao.binary.getBytesUnsafe());
        Assert.assertArrayEquals((byte[])testString.getBytes(UTF8), (byte[])bao.binary.copy().getBytesUnsafe());
        Assert.assertArrayEquals((byte[])testString.getBytes(UTF8), (byte[])bao.binary.copy().getBytes());
        bao = bf.get(testString.getBytes(UTF8), true);
        Assert.assertEquals((Object)true, (Object)bao.binary.isBackingBytesReused());
        Binary copy = bao.binary.copy();
        TestBinary.mutate(bao.original);
        Assert.assertArrayEquals((byte[])testString.getBytes(UTF8), (byte[])copy.getBytes());
        Assert.assertArrayEquals((byte[])testString.getBytes(UTF8), (byte[])copy.getBytesUnsafe());
        Assert.assertArrayEquals((byte[])testString.getBytes(UTF8), (byte[])copy.copy().getBytesUnsafe());
        Assert.assertArrayEquals((byte[])testString.getBytes(UTF8), (byte[])copy.copy().getBytes());
    }

    private void testSerializable(BinaryFactory bf, boolean reused) throws Exception {
        BinaryFactory.BinaryAndOriginal bao = bf.get("polygon".getBytes(UTF8), reused);
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        ObjectOutputStream out = new ObjectOutputStream(baos);
        out.writeObject(bao.binary);
        out.close();
        baos.close();
        ObjectInputStream in = new ObjectInputStream(new ByteArrayInputStream(baos.toByteArray()));
        Object object = in.readObject();
        Assert.assertTrue((boolean)(object instanceof Binary));
        Assert.assertEquals((Object)bao.binary, (Object)object);
    }

    private void testBinary(BinaryFactory bf, boolean reused) throws Exception {
        this.testSlice(bf, reused);
        if (reused) {
            this.testReusedCopy(bf);
        } else {
            this.testConstantCopy(bf);
        }
        this.testSerializable(bf, reused);
    }

    @Test
    public void testCompare() {
        Binary b1 = Binary.fromCharSequence((CharSequence)"aaaaaaaa");
        Binary b2 = Binary.fromString((String)"aaaaaaab");
        Binary b3 = Binary.fromReusedByteArray((byte[])"aaaaaaaaaaa".getBytes(), (int)1, (int)8);
        Binary b4 = Binary.fromConstantByteBuffer((ByteBuffer)ByteBuffer.wrap("aaaaaaac".getBytes()));
        Assert.assertTrue((b1.compareTo(b2) < 0 ? 1 : 0) != 0);
        Assert.assertTrue((b2.compareTo(b1) > 0 ? 1 : 0) != 0);
        Assert.assertTrue((b3.compareTo(b4) < 0 ? 1 : 0) != 0);
        Assert.assertTrue((b4.compareTo(b3) > 0 ? 1 : 0) != 0);
        Assert.assertTrue((b1.compareTo(b4) < 0 ? 1 : 0) != 0);
        Assert.assertTrue((b4.compareTo(b1) > 0 ? 1 : 0) != 0);
        Assert.assertTrue((b2.compareTo(b4) < 0 ? 1 : 0) != 0);
        Assert.assertTrue((b4.compareTo(b2) > 0 ? 1 : 0) != 0);
        Assert.assertTrue((b1.compareTo(b3) == 0 ? 1 : 0) != 0);
        Assert.assertTrue((b3.compareTo(b1) == 0 ? 1 : 0) != 0);
    }

    static interface BinaryFactory {
        public BinaryAndOriginal get(byte[] var1, boolean var2) throws Exception;

        public static class BinaryAndOriginal {
            public Binary binary;
            public byte[] original;

            public BinaryAndOriginal(Binary binary, byte[] original) {
                this.binary = binary;
                this.original = original;
            }
        }
    }
}

