/*
 * Decompiled with CFR 0.152.
 */
package org.apache.maven.buildcache;

import java.io.File;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import javax.inject.Inject;
import javax.inject.Named;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.maven.SessionScoped;
import org.apache.maven.buildcache.CacheContext;
import org.apache.maven.buildcache.CacheResult;
import org.apache.maven.buildcache.CacheUtils;
import org.apache.maven.buildcache.LocalCacheRepository;
import org.apache.maven.buildcache.RemoteCacheRepository;
import org.apache.maven.buildcache.xml.CacheConfig;
import org.apache.maven.buildcache.xml.CacheSource;
import org.apache.maven.buildcache.xml.XmlService;
import org.apache.maven.buildcache.xml.build.Artifact;
import org.apache.maven.buildcache.xml.build.Build;
import org.apache.maven.buildcache.xml.build.Scm;
import org.apache.maven.buildcache.xml.report.CacheReport;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.model.Dependency;
import org.apache.maven.project.MavenProject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SessionScoped
@Named
public class LocalCacheRepositoryImpl
implements LocalCacheRepository {
    private static final String BUILDINFO_XML = "buildinfo.xml";
    private static final String LOOKUPINFO_XML = "lookupinfo.xml";
    private static final long ONE_HOUR_MILLIS = TimeUnit.HOURS.toMillis(1L);
    private static final long ONE_MINUTE_MILLIS = TimeUnit.MINUTES.toMillis(1L);
    private static final long ONE_DAY_MILLIS = TimeUnit.DAYS.toMillis(1L);
    private static final String EMPTY = "";
    private static final Logger LOGGER = LoggerFactory.getLogger(LocalCacheRepositoryImpl.class);
    private final RemoteCacheRepository remoteRepository;
    private final XmlService xmlService;
    private final CacheConfig cacheConfig;
    private final Map<Pair<MavenSession, Dependency>, Optional<org.apache.maven.buildcache.xml.Build>> bestBuildCache = new ConcurrentHashMap<Pair<MavenSession, Dependency>, Optional<org.apache.maven.buildcache.xml.Build>>();

    @Inject
    public LocalCacheRepositoryImpl(RemoteCacheRepository remoteRepository, XmlService xmlService, CacheConfig cacheConfig) {
        this.remoteRepository = remoteRepository;
        this.xmlService = xmlService;
        this.cacheConfig = cacheConfig;
    }

    @Override
    @Nonnull
    public Optional<org.apache.maven.buildcache.xml.Build> findLocalBuild(CacheContext context) throws IOException {
        Path localBuildInfoPath = this.localBuildPath(context, BUILDINFO_XML, false);
        LOGGER.debug("Checking local build info: {}", (Object)localBuildInfoPath);
        if (Files.exists(localBuildInfoPath, new LinkOption[0])) {
            LOGGER.info("Local build found by checksum {}", (Object)context.getInputInfo().getChecksum());
            try {
                Build dto = this.xmlService.loadBuild(localBuildInfoPath.toFile());
                return Optional.of(new org.apache.maven.buildcache.xml.Build(dto, CacheSource.LOCAL));
            }
            catch (Exception e) {
                LOGGER.info("Local build info is not valid, deleting: {}", (Object)localBuildInfoPath, (Object)e);
                Files.delete(localBuildInfoPath);
            }
        }
        return Optional.empty();
    }

    @Override
    @Nonnull
    public Optional<org.apache.maven.buildcache.xml.Build> findBuild(CacheContext context) throws IOException {
        Path buildInfoPath = this.remoteBuildPath(context, BUILDINFO_XML);
        LOGGER.debug("Checking if build is already downloaded: {}", (Object)buildInfoPath);
        if (Files.exists(buildInfoPath, new LinkOption[0])) {
            LOGGER.info("Downloaded build found by checksum {}", (Object)context.getInputInfo().getChecksum());
            try {
                Build dto = this.xmlService.loadBuild(buildInfoPath.toFile());
                return Optional.of(new org.apache.maven.buildcache.xml.Build(dto, CacheSource.REMOTE));
            }
            catch (Exception e) {
                LOGGER.info("Downloaded build info is not valid, deleting: {}", (Object)buildInfoPath, (Object)e);
                Files.delete(buildInfoPath);
            }
        }
        if (!this.cacheConfig.isRemoteCacheEnabled()) {
            return Optional.empty();
        }
        try {
            Optional<org.apache.maven.buildcache.xml.Build> build;
            Path lookupInfoPath = this.remoteBuildPath(context, LOOKUPINFO_XML);
            if (Files.exists(lookupInfoPath, new LinkOption[0])) {
                BasicFileAttributes fileAttributes = Files.readAttributes(lookupInfoPath, BasicFileAttributes.class, new LinkOption[0]);
                long lastModified = fileAttributes.lastModifiedTime().toMillis();
                long created = fileAttributes.creationTime().toMillis();
                long now = System.currentTimeMillis();
                if (now < created + ONE_HOUR_MILLIS && now < lastModified + ONE_MINUTE_MILLIS) {
                    LOGGER.info("Skipping remote lookup, last unsuccessful lookup less than 1m ago.");
                    return Optional.empty();
                }
                if (now < created + ONE_DAY_MILLIS && now < lastModified + ONE_HOUR_MILLIS) {
                    LOGGER.info("Skipping remote lookup, last unsuccessful lookup less than 1h ago.");
                    return Optional.empty();
                }
                if (now > created + ONE_DAY_MILLIS && now < lastModified + ONE_DAY_MILLIS) {
                    LOGGER.info("Skipping remote lookup, last unsuccessful lookup less than 1d ago.");
                    return Optional.empty();
                }
            }
            if ((build = this.remoteRepository.findBuild(context)).isPresent()) {
                LOGGER.info("Build info downloaded from remote repo, saving to: {}", (Object)buildInfoPath);
                Files.createDirectories(buildInfoPath.getParent(), new FileAttribute[0]);
                Files.write(buildInfoPath, this.xmlService.toBytes(build.get().getDto()), StandardOpenOption.CREATE_NEW);
            } else {
                FileUtils.touch((File)lookupInfoPath.toFile());
            }
            return build;
        }
        catch (Exception e) {
            LOGGER.error("Remote build info is not valid, cached data is not compatible", (Throwable)e);
            return Optional.empty();
        }
    }

    @Override
    public void clearCache(CacheContext context) {
        try {
            Path path;
            Path buildCacheDir = this.buildCacheDir(context);
            Path artifactCacheDir = buildCacheDir.getParent();
            if (!Files.exists(artifactCacheDir, new LinkOption[0])) {
                return;
            }
            ArrayList<Path> cacheDirs = new ArrayList<Path>();
            try (DirectoryStream<Path> paths = Files.newDirectoryStream(artifactCacheDir);){
                for (Path dir : paths) {
                    if (!Files.isDirectory(dir, new LinkOption[0])) continue;
                    cacheDirs.add(dir);
                }
            }
            int maxLocalBuildsCached = this.cacheConfig.getMaxLocalBuildsCached() - 1;
            if (cacheDirs.size() > maxLocalBuildsCached) {
                cacheDirs.sort(Comparator.comparing(LocalCacheRepositoryImpl::lastModifiedTime));
                for (Path dir : cacheDirs.subList(0, cacheDirs.size() - maxLocalBuildsCached)) {
                    FileUtils.deleteDirectory((File)dir.toFile());
                }
            }
            if (Files.exists(path = this.localBuildDir(context), new LinkOption[0])) {
                FileUtils.deleteDirectory((File)path.toFile());
            }
        }
        catch (IOException e) {
            String artifactId = context.getProject().getArtifactId();
            throw new RuntimeException("Failed to cleanup local cache of " + artifactId + " on build failure, it might be inconsistent", e);
        }
    }

    @Override
    @Nonnull
    public Optional<org.apache.maven.buildcache.xml.Build> findBestMatchingBuild(MavenSession session, Dependency dependency) {
        return this.bestBuildCache.computeIfAbsent((Pair<MavenSession, Dependency>)Pair.of((Object)session, (Object)dependency), this::findBestMatchingBuildImpl);
    }

    @Nonnull
    private Optional<org.apache.maven.buildcache.xml.Build> findBestMatchingBuildImpl(Pair<MavenSession, Dependency> dependencySession) {
        try {
            MavenSession session = (MavenSession)dependencySession.getLeft();
            Dependency dependency = (Dependency)dependencySession.getRight();
            Path artifactCacheDir = this.artifactCacheDir(session, dependency.getGroupId(), dependency.getArtifactId());
            final HashMap filesByVersion = new HashMap();
            Files.walkFileTree(artifactCacheDir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path path, BasicFileAttributes basicFileAttributes) {
                    File file = path.toFile();
                    if (file.getName().equals(LocalCacheRepositoryImpl.BUILDINFO_XML)) {
                        try {
                            Build dto = LocalCacheRepositoryImpl.this.xmlService.loadBuild(file);
                            Pair buildInfoAndFile = Pair.of((Object)new org.apache.maven.buildcache.xml.Build(dto, CacheSource.LOCAL), (Object)path);
                            String cachedVersion = dto.getArtifact().getVersion();
                            String cachedBranch = LocalCacheRepositoryImpl.this.getScmRef(dto.getScm());
                            LocalCacheRepositoryImpl.add(filesByVersion, Pair.of((Object)cachedVersion, (Object)cachedBranch), buildInfoAndFile);
                            if (StringUtils.isNotBlank((CharSequence)cachedBranch)) {
                                LocalCacheRepositoryImpl.add(filesByVersion, Pair.of((Object)LocalCacheRepositoryImpl.EMPTY, (Object)cachedBranch), buildInfoAndFile);
                            }
                            if (StringUtils.isNotBlank((CharSequence)cachedVersion)) {
                                LocalCacheRepositoryImpl.add(filesByVersion, Pair.of((Object)cachedVersion, (Object)LocalCacheRepositoryImpl.EMPTY), buildInfoAndFile);
                            }
                        }
                        catch (Exception e) {
                            LOGGER.info("Build info is not compatible to current maven implementation: {}", (Object)file, (Object)e);
                        }
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
            if (filesByVersion.isEmpty()) {
                return Optional.empty();
            }
            String currentRef = this.getScmRef(CacheUtils.readGitInfo(session));
            Collection bestMatched = new LinkedList();
            if (StringUtils.isNotBlank((CharSequence)currentRef)) {
                bestMatched = (Collection)filesByVersion.get(Pair.of((Object)dependency.getVersion(), (Object)currentRef));
            }
            if (bestMatched.isEmpty()) {
                bestMatched = (Collection)filesByVersion.get(Pair.of((Object)dependency.getVersion(), (Object)EMPTY));
            }
            if (bestMatched.isEmpty() && StringUtils.isNotBlank((CharSequence)currentRef)) {
                bestMatched = (Collection)filesByVersion.get(Pair.of((Object)EMPTY, (Object)currentRef));
            }
            if (bestMatched.isEmpty()) {
                bestMatched = filesByVersion.values().stream().flatMap(Collection::stream).collect(Collectors.toList());
            }
            return bestMatched.stream().max(Comparator.comparing(p -> LocalCacheRepositoryImpl.lastModifiedTime((Path)p.getRight()))).map(Pair::getLeft);
        }
        catch (IOException e) {
            LOGGER.info("Cannot find dependency in cache", (Throwable)e);
            return Optional.empty();
        }
    }

    private String getScmRef(Scm scm) {
        if (scm != null) {
            return scm.getSourceBranch() != null ? scm.getSourceBranch() : scm.getRevision();
        }
        return EMPTY;
    }

    @Override
    public Path getArtifactFile(CacheContext context, CacheSource source, Artifact artifact) throws IOException {
        if (source == CacheSource.LOCAL) {
            return this.localBuildPath(context, artifact.getFileName(), false);
        }
        Path cachePath = this.remoteBuildPath(context, artifact.getFileName());
        if (!Files.exists(cachePath, new LinkOption[0]) && this.cacheConfig.isRemoteCacheEnabled() && !this.remoteRepository.getArtifactContent(context, artifact, cachePath)) {
            Files.deleteIfExists(cachePath);
        }
        return cachePath;
    }

    @Override
    public void beforeSave(CacheContext environment) {
        this.clearCache(environment);
    }

    @Override
    public void saveBuildInfo(CacheResult cacheResult, org.apache.maven.buildcache.xml.Build build) throws IOException {
        Path path = this.localBuildPath(cacheResult.getContext(), BUILDINFO_XML, true);
        Files.write(path, this.xmlService.toBytes(build.getDto()), StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.CREATE);
        if (this.cacheConfig.isRemoteCacheEnabled() && this.cacheConfig.isSaveToRemote() && !cacheResult.isFinal()) {
            this.remoteRepository.saveBuildInfo(cacheResult, build);
        }
    }

    @Override
    public void saveCacheReport(String buildId, MavenSession session, CacheReport cacheReport) throws IOException {
        Path path = CacheUtils.getMultimoduleRoot(session).resolve("target").resolve("maven-incremental");
        Files.createDirectories(path, new FileAttribute[0]);
        Files.write(path.resolve("cache-report." + buildId + ".xml"), this.xmlService.toBytes(cacheReport), StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.CREATE);
        if (this.cacheConfig.isRemoteCacheEnabled() && this.cacheConfig.isSaveToRemote()) {
            LOGGER.info("Saving cache report on build completion");
            this.remoteRepository.saveCacheReport(buildId, session, cacheReport);
        }
    }

    @Override
    public void saveArtifactFile(CacheResult cacheResult, org.apache.maven.artifact.Artifact artifact) throws IOException {
        File artifactFile = artifact.getFile();
        Path cachePath = this.localBuildPath(cacheResult.getContext(), CacheUtils.normalizedName(artifact), true);
        Files.copy(artifactFile.toPath(), cachePath, StandardCopyOption.REPLACE_EXISTING);
        if (this.cacheConfig.isRemoteCacheEnabled() && this.cacheConfig.isSaveToRemote() && !cacheResult.isFinal()) {
            this.remoteRepository.saveArtifactFile(cacheResult, artifact);
        }
    }

    private Path buildCacheDir(CacheContext context) throws IOException {
        MavenProject project = context.getProject();
        Path artifactCacheDir = this.artifactCacheDir(context.getSession(), project.getGroupId(), project.getArtifactId());
        return artifactCacheDir.resolve(context.getInputInfo().getChecksum());
    }

    private Path artifactCacheDir(MavenSession session, String groupId, String artifactId) throws IOException {
        Path vga = Paths.get("v1", groupId, artifactId);
        Path path = this.baseDir(session).resolve(vga);
        Files.createDirectories(path, new FileAttribute[0]);
        return path;
    }

    private Path baseDir(MavenSession session) {
        String loc = this.cacheConfig.getLocalRepositoryLocation();
        if (loc != null) {
            return Paths.get(loc, new String[0]);
        }
        return Paths.get(session.getLocalRepository().getBasedir(), new String[0]).getParent().resolve("build-cache");
    }

    private Path remoteBuildPath(CacheContext context, String filename) throws IOException {
        return this.remoteBuildDir(context).resolve(filename);
    }

    private Path localBuildPath(CacheContext context, String filename, boolean createDir) throws IOException {
        Path localBuildDir = this.localBuildDir(context);
        if (createDir) {
            Files.createDirectories(localBuildDir, new FileAttribute[0]);
        }
        return localBuildDir.resolve(filename);
    }

    private Path remoteBuildDir(CacheContext context) throws IOException {
        return this.buildCacheDir(context).resolve(this.cacheConfig.getId());
    }

    private Path localBuildDir(CacheContext context) throws IOException {
        return this.buildCacheDir(context).resolve("local");
    }

    private static FileTime lastModifiedTime(Path p) {
        try {
            return Files.getLastModifiedTime(p, new LinkOption[0]);
        }
        catch (IOException e) {
            return FileTime.fromMillis(0L);
        }
    }

    private static <K, V> void add(Map<K, Collection<V>> map, K key, V value) {
        map.computeIfAbsent(key, k -> new ArrayList()).add(value);
    }
}

