/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller.metrics;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.kafka.common.Cell;
import org.apache.kafka.common.CellState;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.config.ConfigResource;
import org.apache.kafka.common.metadata.BrokerRegistrationChangeRecord;
import org.apache.kafka.common.metadata.CellRecord;
import org.apache.kafka.common.metadata.ConfigRecord;
import org.apache.kafka.common.metadata.FenceBrokerRecord;
import org.apache.kafka.common.metadata.MetadataRecordType;
import org.apache.kafka.common.metadata.PartitionChangeRecord;
import org.apache.kafka.common.metadata.PartitionRecord;
import org.apache.kafka.common.metadata.RegisterBrokerRecord;
import org.apache.kafka.common.metadata.RemoveCellRecord;
import org.apache.kafka.common.metadata.RemoveTenantRecord;
import org.apache.kafka.common.metadata.RemoveTopicRecord;
import org.apache.kafka.common.metadata.TenantRecord;
import org.apache.kafka.common.metadata.TopicRecord;
import org.apache.kafka.common.metadata.UnfenceBrokerRecord;
import org.apache.kafka.common.metadata.UnregisterBrokerRecord;
import org.apache.kafka.common.protocol.ApiMessage;
import org.apache.kafka.controller.ConfigurationControlManager;
import org.apache.kafka.controller.metrics.CellMetrics;
import org.apache.kafka.controller.metrics.ControllerMetrics;
import org.apache.kafka.metadata.BrokerRegistrationFencingChange;
import org.apache.kafka.metadata.placement.CellAssignor;
import org.apache.kafka.server.common.ApiMessageAndVersion;

public final class ControllerMetricsManager {
    private static final String TOPIC_MIN_IN_SYNC_REPLICAS_PROP = "min.insync.replicas";
    private static final String NODE_MIN_IN_SYNC_REPLICAS_PROP = "min.insync.replicas";
    private final Set<Integer> registeredBrokers = new HashSet<Integer>();
    private final Set<Integer> fencedBrokers = new HashSet<Integer>();
    private final Set<Integer> degradedBrokers = new HashSet<Integer>();
    private final Map<String, ConfigState> dynamicTopicMinIsrConfigs = new HashMap<String, ConfigState>();
    private OptionalInt dynamicClusterMinIsrConfig = OptionalInt.empty();
    private final Map<Uuid, String> topicIdToName = new HashMap<Uuid, String>();
    private final Map<TopicIdPartition, PartitionState> topicPartitions = new HashMap<TopicIdPartition, PartitionState>();
    private final Set<TopicIdPartition> offlineTopicPartitions = new HashSet<TopicIdPartition>();
    private final Set<TopicIdPartition> imbalancedTopicPartitions = new HashSet<TopicIdPartition>();
    private final Set<TopicIdPartition> underMinIsrTopicPartitions = new HashSet<TopicIdPartition>();
    private final Map<String, TenantState> tenants = new HashMap<String, TenantState>();
    private final ControllerMetrics controllerMetrics;
    private final CellMetrics cellMetrics;
    private final Optional<Function<String, String>> topicNameToTenant;
    private final int defaultMinInSyncReplicas;
    private final short defaultReplicationFactor;

    public ControllerMetricsManager(ControllerMetrics controllerMetrics, CellMetrics cellMetrics, Optional<Function<String, String>> topicNameToTenant, int defaultMinInSyncReplicas, short defaultReplicationFactor) {
        this.controllerMetrics = controllerMetrics;
        this.cellMetrics = cellMetrics;
        this.topicNameToTenant = topicNameToTenant;
        this.defaultMinInSyncReplicas = defaultMinInSyncReplicas;
        this.defaultReplicationFactor = defaultReplicationFactor;
    }

    public void replayBatch(long baseOffset, List<ApiMessageAndVersion> messages) {
        int i = 1;
        for (ApiMessageAndVersion message : messages) {
            try {
                this.replay(message.message());
            }
            catch (Exception e) {
                String failureMessage = String.format("Unable to update controller metrics for %s record, it was %d of %d record(s) in the batch with baseOffset %d.", message.message().getClass().getSimpleName(), i, messages.size(), baseOffset);
                throw new IllegalArgumentException(failureMessage, e);
            }
            ++i;
        }
    }

    public void replay(ApiMessage message) {
        MetadataRecordType type = MetadataRecordType.fromId(message.apiKey());
        switch (type) {
            case REGISTER_BROKER_RECORD: {
                this.replay((RegisterBrokerRecord)message);
                break;
            }
            case UNREGISTER_BROKER_RECORD: {
                this.replay((UnregisterBrokerRecord)message);
                break;
            }
            case FENCE_BROKER_RECORD: {
                this.replay((FenceBrokerRecord)message);
                break;
            }
            case UNFENCE_BROKER_RECORD: {
                this.replay((UnfenceBrokerRecord)message);
                break;
            }
            case BROKER_REGISTRATION_CHANGE_RECORD: {
                this.replay((BrokerRegistrationChangeRecord)message);
                break;
            }
            case TOPIC_RECORD: {
                this.replay((TopicRecord)message);
                break;
            }
            case PARTITION_RECORD: {
                this.replay((PartitionRecord)message);
                break;
            }
            case PARTITION_CHANGE_RECORD: {
                this.replay((PartitionChangeRecord)message);
                break;
            }
            case REMOVE_TOPIC_RECORD: {
                this.replay((RemoveTopicRecord)message);
                break;
            }
            case CONFIG_RECORD: {
                this.replay((ConfigRecord)message);
                break;
            }
            case CELL_RECORD: {
                this.replay((CellRecord)message);
                break;
            }
            case REMOVE_CELL_RECORD: {
                this.replay((RemoveCellRecord)message);
                break;
            }
            case TENANT_RECORD: {
                this.replay((TenantRecord)message);
                break;
            }
            case REMOVE_TENANT_RECORD: {
                this.replay((RemoveTenantRecord)message);
                break;
            }
            case FEATURE_LEVEL_RECORD: 
            case CLIENT_QUOTA_RECORD: 
            case PRODUCER_IDS_RECORD: 
            case ACCESS_CONTROL_ENTRY_RECORD: 
            case REMOVE_ACCESS_CONTROL_ENTRY_RECORD: 
            case USER_SCRAM_CREDENTIAL_RECORD: 
            case REMOVE_USER_SCRAM_CREDENTIAL_RECORD: 
            case NO_OP_RECORD: 
            case ZK_MIGRATION_STATE_RECORD: {
                break;
            }
            case BROKER_REPLICA_EXCLUSION_RECORD: 
            case CLUSTER_LINK_RECORD: 
            case ENCRYPTED_ENVELOPE_RECORD: 
            case INSTALL_METADATA_ENCRYPTOR_RECORD: 
            case MIRROR_TOPIC_CHANGE_RECORD: 
            case MIRROR_TOPIC_RECORD: 
            case REMOVE_CLUSTER_LINK_RECORD: {
                break;
            }
            default: {
                throw new RuntimeException("Unhandled record type " + (Object)((Object)type));
            }
        }
    }

    private void replay(RegisterBrokerRecord record) {
        Integer brokerId = record.brokerId();
        this.registeredBrokers.add(brokerId);
        if (record.fenced()) {
            this.fencedBrokers.add(brokerId);
        } else {
            this.fencedBrokers.remove(brokerId);
        }
        if (record.degradedComponents() == null) {
            this.degradedBrokers.remove(brokerId);
        } else if (record.degradedComponents().isEmpty()) {
            this.degradedBrokers.remove(brokerId);
        } else {
            this.degradedBrokers.add(brokerId);
        }
        this.updateBrokerStateMetrics();
        this.updateUnhealthyCellMetrics();
    }

    private void replay(UnregisterBrokerRecord record) {
        Integer brokerId = record.brokerId();
        this.registeredBrokers.remove(brokerId);
        this.fencedBrokers.remove(brokerId);
        this.degradedBrokers.remove(brokerId);
        this.updateBrokerStateMetrics();
        this.updateUnhealthyCellMetrics();
    }

    private void replay(FenceBrokerRecord record) {
        this.handleFencingChange(record.id(), BrokerRegistrationFencingChange.FENCE);
        this.updateBrokerStateMetrics();
    }

    private void replay(UnfenceBrokerRecord record) {
        this.handleFencingChange(record.id(), BrokerRegistrationFencingChange.UNFENCE);
        this.updateBrokerStateMetrics();
    }

    private void replay(BrokerRegistrationChangeRecord record) {
        BrokerRegistrationFencingChange fencingChange = BrokerRegistrationFencingChange.fromValue(record.fenced()).orElseThrow(() -> new IllegalArgumentException(String.format("Registration change record for %d has unknown value for fenced field: %x", record.brokerId(), record.fenced())));
        this.handleFencingChange(record.brokerId(), fencingChange);
        if (record.degradedComponents() != null) {
            if (record.degradedComponents().isEmpty()) {
                this.degradedBrokers.remove(record.brokerId());
            } else {
                this.degradedBrokers.add(record.brokerId());
            }
        }
        this.updateBrokerStateMetrics();
    }

    private void handleFencingChange(Integer brokerId, BrokerRegistrationFencingChange fencingChange) {
        if (!this.registeredBrokers.contains(brokerId)) {
            throw new IllegalArgumentException(String.format("Broker with id %s is not registered", brokerId));
        }
        if (fencingChange == BrokerRegistrationFencingChange.FENCE) {
            this.fencedBrokers.add(brokerId);
        } else if (fencingChange == BrokerRegistrationFencingChange.UNFENCE) {
            this.fencedBrokers.remove(brokerId);
        }
    }

    private void updateBrokerStateMetrics() {
        this.controllerMetrics.setFencedBrokerCount(this.fencedBrokers.size());
        HashSet<Integer> activeBrokers = new HashSet<Integer>(this.registeredBrokers);
        activeBrokers.removeAll(this.fencedBrokers);
        this.controllerMetrics.setActiveBrokerCount(activeBrokers.size());
        this.controllerMetrics.setBrokersWithDegradedHealthCount(this.degradedBrokers.size());
    }

    private void replay(TopicRecord record) {
        this.topicIdToName.put(record.topicId(), record.name());
        this.dynamicTopicMinIsrConfigs.compute(record.name(), (topicName, configState) -> {
            if (configState == null) {
                return new ConfigState(Optional.of(record.topicId()), OptionalInt.empty());
            }
            configState.setTopicId(record.topicId());
            return configState;
        });
        this.tenant(record.name()).ifPresent(lkc -> this.tenants.computeIfAbsent((String)lkc, key -> new TenantState()));
        this.controllerMetrics.setGlobalTopicCount(this.topicIdToName.size());
    }

    private void replay(PartitionRecord record) {
        TopicIdPartition tp = new TopicIdPartition(record.topicId(), record.partitionId());
        PartitionState partitionState = new PartitionState(record.leader(), record.replicas().get(0), record.isr().size(), record.replicas());
        PartitionState prevPartition = this.topicPartitions.get(tp);
        this.topicPartitions.put(tp, partitionState);
        String topicName = Objects.requireNonNull(this.topicIdToName.get(tp.topicId()), () -> String.format("Found a partition %s without a matching topic", tp));
        if (prevPartition == null) {
            this.updateCellReplicaCounts(topicName, record.partitionId(), Collections.emptyList(), record.replicas());
        } else {
            this.updateCellReplicaCounts(topicName, record.partitionId(), prevPartition.replicas, record.replicas() == null ? prevPartition.replicas : record.replicas());
        }
        this.tenant(topicName).ifPresent(lkc -> this.tenants.computeIfPresent((String)lkc, (key, tenantState) -> {
            tenantState.topicPartitions().add(tp);
            return tenantState;
        }));
        Optional<String> tenant = this.tenant(topicName);
        this.updateBasedOnPartitionState(tp, tenant, partitionState, this.minIsrConfig(topicName));
        this.updateTopicAndPartitionMetrics(tenant);
    }

    private void replay(PartitionChangeRecord record) {
        TopicIdPartition tp = new TopicIdPartition(record.topicId(), record.partitionId());
        if (!this.topicPartitions.containsKey(tp)) {
            throw new IllegalArgumentException(String.format("Unknown topic partitions %s", tp));
        }
        PartitionState prevPartition = this.topicPartitions.get(tp);
        List<Integer> newReplicas = record.replicas() == null ? prevPartition.replicas() : record.replicas();
        PartitionState partitionState = this.topicPartitions.computeIfPresent(tp, (key, oldValue) -> {
            PartitionState newValue = oldValue;
            if (record.replicas() != null) {
                newValue = new PartitionState(newValue.leader(), record.replicas().get(0), newValue.isrSize(), newReplicas);
            }
            if (record.leader() != -2) {
                newValue = new PartitionState(record.leader(), newValue.preferredReplica(), newValue.isrSize(), newReplicas);
            }
            if (record.isr() != null) {
                newValue = new PartitionState(newValue.leader(), newValue.preferredReplica(), record.isr().size(), newReplicas);
            }
            return newValue;
        });
        String topicName = Objects.requireNonNull(this.topicIdToName.get(tp.topicId()), () -> String.format("Found a partition change for %s without a matching topic", tp));
        this.updateCellReplicaCounts(topicName, record.partitionId(), prevPartition.replicas, newReplicas);
        int minIsr = this.minIsrConfig(topicName);
        Optional<String> tenant = this.tenant(topicName);
        if (this.updateBasedOnPartitionState(tp, tenant, partitionState, minIsr)) {
            this.updateTopicAndPartitionMetrics(tenant);
        }
    }

    private void replay(RemoveTopicRecord record) {
        Uuid topicId = record.topicId();
        Predicate<TopicIdPartition> matchesTopic = tp -> tp.topicId().equals((Object)topicId);
        List<Map.Entry<TopicIdPartition, PartitionState>> partitionsRemoved = this.topicPartitions.entrySet().stream().filter(entry -> matchesTopic.test((TopicIdPartition)entry.getKey())).collect(Collectors.toList());
        for (Map.Entry entry2 : partitionsRemoved) {
            TopicIdPartition topicIdPartition = (TopicIdPartition)entry2.getKey();
            this.topicPartitions.remove(topicIdPartition);
            this.offlineTopicPartitions.remove(topicIdPartition);
            this.imbalancedTopicPartitions.remove(topicIdPartition);
            this.underMinIsrTopicPartitions.remove(topicIdPartition);
        }
        String topicName = this.topicIdToName.remove(record.topicId());
        this.dynamicTopicMinIsrConfigs.remove(topicName);
        Optional<String> optional = this.tenant(topicName);
        optional.ifPresent(lkc -> this.tenants.computeIfPresent((String)lkc, (key, tenantState) -> {
            tenantState.topicPartitions().removeIf(matchesTopic);
            tenantState.offlineTopicPartitions().removeIf(matchesTopic);
            tenantState.underMinIsrTopicPartitions().removeIf(matchesTopic);
            if (tenantState.topicPartitions().isEmpty()) {
                return null;
            }
            return tenantState;
        }));
        this.updateTopicAndPartitionMetrics(optional);
        this.updateCellMetricsForPartitionsRemoved(topicName, partitionsRemoved);
    }

    private void replay(ConfigRecord record) {
        if (ConfigResource.Type.forId((byte)record.resourceType()).equals((Object)ConfigResource.Type.TOPIC) && "min.insync.replicas".equals(record.name())) {
            String topicName = record.resourceName();
            OptionalInt configMinIsr = record.value() == null ? OptionalInt.empty() : OptionalInt.of(Integer.valueOf(record.value()));
            ConfigState configState = this.dynamicTopicMinIsrConfigs.compute(topicName, (key, currentConfigState) -> {
                if (currentConfigState == null) {
                    return new ConfigState(Optional.empty(), configMinIsr);
                }
                currentConfigState.setMinIsr(configMinIsr);
                return currentConfigState;
            });
            int minIsr = this.minIsrConfig(topicName);
            Optional<String> tenant = this.tenant(topicName);
            this.topicPartitions.entrySet().stream().filter(entry -> Optional.of(((TopicIdPartition)entry.getKey()).topicId()).equals(configState.topicId())).forEach(entry -> this.updateBasedOnPartitionState((TopicIdPartition)entry.getKey(), tenant, (PartitionState)entry.getValue(), minIsr));
            this.updateTopicAndPartitionMetrics(tenant);
        } else {
            ConfigResource configResource = new ConfigResource(ConfigResource.Type.forId((byte)record.resourceType()), record.resourceName());
            if (configResource.equals((Object)ConfigurationControlManager.DEFAULT_NODE) && "min.insync.replicas".equals(record.name())) {
                String value = record.value();
                this.dynamicClusterMinIsrConfig = value == null ? OptionalInt.empty() : OptionalInt.of(Integer.valueOf(value));
                HashSet<String> updatedTenants = new HashSet<String>();
                this.topicPartitions.entrySet().stream().forEach(entry -> {
                    String topicName = Objects.requireNonNull(this.topicIdToName.get(((TopicIdPartition)entry.getKey()).topicId()), () -> String.format("Found a partition %s without a matching topic", entry.getKey()));
                    int minIsr = this.minIsrConfig(topicName);
                    Optional<String> tenant = this.tenant(topicName);
                    if (this.updateBasedOnPartitionState((TopicIdPartition)entry.getKey(), tenant, (PartitionState)entry.getValue(), minIsr)) {
                        tenant.ifPresent(updatedTenants::add);
                    }
                });
                this.updateTopicAndPartitionMetrics(updatedTenants);
            }
        }
    }

    private void replay(CellRecord record) {
        this.cellMetrics.updateCell(new Cell(record.cellId(), new HashSet<Integer>(record.brokers()), CellState.toEnum((byte)record.state()), record.minSize(), record.maxSize()));
        this.updateUnhealthyCellMetrics();
    }

    private void replay(RemoveCellRecord removeCellRecord) {
        this.cellMetrics.deleteCell(removeCellRecord.cellId());
        this.updateUnhealthyCellMetrics();
    }

    private void replay(TenantRecord tenantRecord) {
        this.cellMetrics.updateTenantIdToCell(tenantRecord.tenantId(), tenantRecord.cellId());
    }

    private void replay(RemoveTenantRecord removeTenantRecord) {
        this.cellMetrics.deleteTenant(removeTenantRecord.tenantId());
    }

    private boolean updateBasedOnPartitionState(TopicIdPartition tp, Optional<String> tenant, PartitionState partitionState, int minIsr) {
        boolean updated = partitionState.leader() == partitionState.preferredReplica() ? this.imbalancedTopicPartitions.remove(tp) : this.imbalancedTopicPartitions.add(tp);
        tenant.ifPresent(lkc -> {
            TenantState tenantState = Objects.requireNonNull(this.tenants.get(lkc), () -> String.format("Missing tenant metrics for %s", lkc));
            ControllerMetricsManager.updateAvailabilityFromPartitionState(tp, partitionState, minIsr, tenantState.offlineTopicPartitions(), tenantState.underMinIsrTopicPartitions());
        });
        return updated |= ControllerMetricsManager.updateAvailabilityFromPartitionState(tp, partitionState, minIsr, this.offlineTopicPartitions, this.underMinIsrTopicPartitions);
    }

    private static boolean updateAvailabilityFromPartitionState(TopicIdPartition tp, PartitionState partitionState, int minIsr, Set<TopicIdPartition> offlinePartitions, Set<TopicIdPartition> underMinIsrPartitions) {
        boolean updated = false;
        if (partitionState.leader() == -1) {
            updated |= offlinePartitions.add(tp);
            updated |= underMinIsrPartitions.remove(tp);
        } else {
            updated |= offlinePartitions.remove(tp);
            updated = partitionState.isrSize() < minIsr ? (updated |= underMinIsrPartitions.add(tp)) : (updated |= underMinIsrPartitions.remove(tp));
        }
        return updated;
    }

    private void updateTopicAndPartitionMetrics(Optional<String> updatedTenant) {
        Set<String> updatedTenants = updatedTenant.isPresent() ? Collections.singleton(updatedTenant.get()) : Collections.emptySet();
        this.updateTopicAndPartitionMetrics(updatedTenants);
    }

    private void updateCellReplicaCounts(String topicName, int partitionId, List<Integer> replicas, List<Integer> newReplicas) {
        if (replicas.equals(newReplicas)) {
            return;
        }
        List addingReplicas = newReplicas.stream().filter(replica -> !replicas.contains(replica)).collect(Collectors.toList());
        List removingReplicas = replicas.stream().filter(replica -> !newReplicas.contains(replica)).collect(Collectors.toList());
        this.cellMetrics.updateReplicaCounts(topicName, Collections.singletonMap(partitionId, addingReplicas), Collections.singletonMap(partitionId, removingReplicas));
    }

    private void updateCellMetricsForPartitionsRemoved(String topicName, List<Map.Entry<TopicIdPartition, PartitionState>> partitionsRemoved) {
        HashMap<Integer, List<Integer>> partitionIdToReplicasRemoved = new HashMap<Integer, List<Integer>>();
        for (Map.Entry<TopicIdPartition, PartitionState> partitionEntry : partitionsRemoved) {
            TopicIdPartition topicIdPartition = partitionEntry.getKey();
            PartitionState partitionState = partitionEntry.getValue();
            if (partitionState.replicas() == null) continue;
            partitionIdToReplicasRemoved.put(topicIdPartition.partitionId(), partitionState.replicas());
        }
        this.cellMetrics.updateReplicaCounts(topicName, Collections.emptyMap(), partitionIdToReplicasRemoved);
    }

    private void updateUnhealthyCellMetrics() {
        Set cellBrokers = this.cellMetrics.cells().stream().flatMap(cell -> cell.brokers().stream()).collect(Collectors.toSet());
        int numStrayBrokers = (int)this.registeredBrokers.stream().filter(broker -> !cellBrokers.contains(broker)).count();
        int numCellsNotOpenForTenantAssignment = (int)this.cellMetrics.cells().stream().filter(cell -> !CellAssignor.isCellOpenForAssignment(cell, this.registeredBrokers, this.defaultReplicationFactor)).count();
        this.cellMetrics.updateUnhealthyCellStats(numCellsNotOpenForTenantAssignment, numStrayBrokers);
    }

    private void updateTopicAndPartitionMetrics(Set<String> updatedTenants) {
        this.controllerMetrics.setGlobalTopicCount(this.topicIdToName.size());
        this.controllerMetrics.setGlobalPartitionCount(this.topicPartitions.size());
        this.controllerMetrics.setGlobalOfflinePartitionCount(this.offlineTopicPartitions.size());
        this.controllerMetrics.setPreferredReplicaImbalanceCount(this.imbalancedTopicPartitions.size());
        this.controllerMetrics.setGlobalUnderMinIsrCount(this.underMinIsrTopicPartitions.size());
        updatedTenants.forEach(tenant -> {
            TenantState tenantState = this.tenants.get(tenant);
            if (tenantState != null) {
                this.controllerMetrics.setTenantPartitionCount((String)tenant, tenantState.topicPartitions().size());
                this.controllerMetrics.setTenantOfflinePartitionCount((String)tenant, tenantState.offlineTopicPartitions().size());
                this.controllerMetrics.setTenantUnderMinIsrCount((String)tenant, tenantState.underMinIsrTopicPartitions().size());
            } else {
                this.controllerMetrics.removeTenant((String)tenant);
            }
        });
    }

    private Optional<String> tenant(String name) {
        return this.topicNameToTenant.flatMap(fun -> Optional.ofNullable(fun.apply(name)));
    }

    private int minIsrConfig(String topicName) {
        return this.dynamicTopicMinIsrConfigs.get(topicName).minIsr.orElseGet(() -> this.dynamicClusterMinIsrConfig.orElse(this.defaultMinInSyncReplicas));
    }

    public void reset() {
        this.registeredBrokers.clear();
        this.fencedBrokers.clear();
        this.topicIdToName.clear();
        this.topicPartitions.clear();
        this.offlineTopicPartitions.clear();
        this.imbalancedTopicPartitions.clear();
        this.dynamicTopicMinIsrConfigs.clear();
        this.dynamicClusterMinIsrConfig = OptionalInt.empty();
        this.underMinIsrTopicPartitions.clear();
        HashSet<String> removedTenants = new HashSet<String>(this.tenants.keySet());
        this.tenants.clear();
        this.cellMetrics.clear();
        this.updateBrokerStateMetrics();
        this.updateTopicAndPartitionMetrics(removedTenants);
    }

    private static final class ConfigState {
        private OptionalInt minIsr;
        private Optional<Uuid> topicId;

        ConfigState(Optional<Uuid> topicId, OptionalInt minIsr) {
            this.topicId = topicId;
            this.minIsr = minIsr;
        }

        OptionalInt minIsr() {
            return this.minIsr;
        }

        void setMinIsr(OptionalInt minIsr) {
            this.minIsr = minIsr;
        }

        Optional<Uuid> topicId() {
            return this.topicId;
        }

        void setTopicId(Uuid topicId) {
            if (this.topicId.isPresent()) {
                throw new IllegalStateException(String.format("Trying to change the value of an existing topic id %s to %s", this.topicId, topicId));
            }
            this.topicId = Optional.of(topicId);
        }
    }

    private static final class TenantState {
        private final Set<TopicIdPartition> topicPartitions = new HashSet<TopicIdPartition>();
        private final Set<TopicIdPartition> offlineTopicPartitions = new HashSet<TopicIdPartition>();
        private final Set<TopicIdPartition> underMinIsrTopicPartitions = new HashSet<TopicIdPartition>();

        private TenantState() {
        }

        Set<TopicIdPartition> topicPartitions() {
            return this.topicPartitions;
        }

        Set<TopicIdPartition> offlineTopicPartitions() {
            return this.offlineTopicPartitions;
        }

        Set<TopicIdPartition> underMinIsrTopicPartitions() {
            return this.underMinIsrTopicPartitions;
        }
    }

    static final class TopicIdPartition {
        private final Uuid topicId;
        private final int partitionId;

        TopicIdPartition(Uuid topicId, int partitionId) {
            this.topicId = topicId;
            this.partitionId = partitionId;
        }

        public Uuid topicId() {
            return this.topicId;
        }

        public int partitionId() {
            return this.partitionId;
        }

        public boolean equals(Object o) {
            if (!(o instanceof TopicIdPartition)) {
                return false;
            }
            TopicIdPartition other = (TopicIdPartition)o;
            return other.topicId.equals((Object)this.topicId) && other.partitionId == this.partitionId;
        }

        public int hashCode() {
            return Objects.hash(this.topicId, this.partitionId);
        }

        public String toString() {
            return this.topicId + ":" + this.partitionId;
        }
    }

    private static final class PartitionState {
        final int leader;
        final int preferredReplica;
        final int isrSize;
        final List<Integer> replicas;

        PartitionState(int leader, int preferredReplica, int isrSize, List<Integer> replicas) {
            this.leader = leader;
            this.preferredReplica = preferredReplica;
            this.isrSize = isrSize;
            this.replicas = replicas;
        }

        int leader() {
            return this.leader;
        }

        int preferredReplica() {
            return this.preferredReplica;
        }

        int isrSize() {
            return this.isrSize;
        }

        List<Integer> replicas() {
            return this.replicas;
        }
    }
}

