/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.common.cloud;

import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeoutException;
import org.apache.solr.common.SolrException;
import org.apache.solr.common.cloud.OnReconnect;
import org.apache.solr.common.cloud.SolrZkClient;
import org.apache.solr.common.cloud.SolrZooKeeper;
import org.apache.solr.common.cloud.ZkClientConnectionStrategy;
import org.apache.solr.common.cloud.ZooKeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectionManager
implements Watcher {
    protected static final Logger log = LoggerFactory.getLogger(ConnectionManager.class);
    private final String name;
    private CountDownLatch clientConnected = new CountDownLatch(1);
    private boolean connected = false;
    private final ZkClientConnectionStrategy connectionStrategy;
    private final String zkServerAddress;
    private final SolrZkClient client;
    private final OnReconnect onReconnect;
    private volatile Watcher.Event.KeeperState state = Watcher.Event.KeeperState.Disconnected;
    private volatile boolean isClosed = false;
    private volatile boolean likelyExpired = true;
    private volatile Timer disconnectedTimer;

    public ConnectionManager(String name, SolrZkClient client, String zkServerAddress, int zkClientTimeout, ZkClientConnectionStrategy strat, OnReconnect onConnect) {
        this.name = name;
        this.client = client;
        this.connectionStrategy = strat;
        this.zkServerAddress = zkServerAddress;
        this.onReconnect = onConnect;
    }

    private synchronized void connected() {
        if (this.disconnectedTimer != null) {
            this.disconnectedTimer.cancel();
            this.disconnectedTimer = null;
        }
        this.connected = true;
        this.likelyExpired = false;
        this.notifyAll();
    }

    private synchronized void disconnected() {
        if (this.disconnectedTimer != null) {
            this.disconnectedTimer.cancel();
            this.disconnectedTimer = null;
        }
        if (!this.isClosed) {
            this.disconnectedTimer = new Timer(true);
            this.disconnectedTimer.schedule(new TimerTask(){

                @Override
                public void run() {
                    ConnectionManager.this.likelyExpired = true;
                }
            }, (long)((double)this.client.getZkClientTimeout() * 0.9));
            if (this.isClosed) {
                this.disconnectedTimer.cancel();
                this.disconnectedTimer = null;
            }
        }
        this.connected = false;
        this.notifyAll();
    }

    @Override
    public void process(WatchedEvent event) {
        if (log.isInfoEnabled()) {
            log.info("Watcher " + this + " name:" + this.name + " got event " + event + " path:" + event.getPath() + " type:" + (Object)((Object)event.getType()));
        }
        if (this.isClosed) {
            log.info("Client->ZooKeeper status change trigger but we are already closed");
            return;
        }
        this.state = event.getState();
        if (this.state == Watcher.Event.KeeperState.SyncConnected) {
            this.connected();
            this.clientConnected.countDown();
            this.connectionStrategy.connected();
        } else if (this.state == Watcher.Event.KeeperState.Expired) {
            if (this.disconnectedTimer != null) {
                this.disconnectedTimer.cancel();
                this.disconnectedTimer = null;
            }
            this.connected = false;
            this.likelyExpired = true;
            log.info("Our previous ZooKeeper session was expired. Attempting to reconnect to recover relationship with ZooKeeper...");
            try {
                this.connectionStrategy.reconnect(this.zkServerAddress, this.client.getZkClientTimeout(), this, new ZkClientConnectionStrategy.ZkUpdate(){

                    @Override
                    public void update(SolrZooKeeper keeper) {
                        try {
                            ConnectionManager.this.waitForConnected(Long.MAX_VALUE);
                        }
                        catch (Exception e1) {
                            ConnectionManager.this.closeKeeper(keeper);
                            throw new RuntimeException(e1);
                        }
                        log.info("Connection with ZooKeeper reestablished.");
                        try {
                            ConnectionManager.this.client.updateKeeper(keeper);
                        }
                        catch (InterruptedException e2) {
                            ConnectionManager.this.closeKeeper(keeper);
                            Thread.currentThread().interrupt();
                            throw new RuntimeException(e2);
                        }
                        catch (Throwable t) {
                            ConnectionManager.this.closeKeeper(keeper);
                            throw new RuntimeException(t);
                        }
                        ConnectionManager.this.connected();
                        if (ConnectionManager.this.onReconnect != null) {
                            Thread thread = new Thread(){

                                @Override
                                public void run() {
                                    try {
                                        ConnectionManager.this.onReconnect.command();
                                    }
                                    catch (Exception e2) {
                                        log.warn("Exception running onReconnect command", e2);
                                    }
                                }
                            };
                            thread.start();
                        }
                    }
                });
            }
            catch (Exception e2) {
                SolrException.log(log, "", e2);
            }
            log.info("Connected:" + this.connected);
        } else if (this.state == Watcher.Event.KeeperState.Disconnected) {
            log.info("zkClient has disconnected");
            this.disconnected();
            this.connectionStrategy.disconnected();
        } else {
            this.disconnected();
        }
    }

    public synchronized boolean isConnected() {
        return !this.isClosed && this.connected;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void close() {
        this.isClosed = true;
        try {
            this.disconnectedTimer.cancel();
        }
        catch (NullPointerException nullPointerException) {
        }
        finally {
            this.disconnectedTimer = null;
        }
    }

    public boolean isLikelyExpired() {
        return this.likelyExpired;
    }

    public synchronized void waitForConnected(long waitForConnection) throws TimeoutException {
        log.info("Waiting for client to connect to ZooKeeper");
        long expire = System.currentTimeMillis() + waitForConnection;
        long left = 1L;
        while (!this.connected && left > 0L && !this.isClosed) {
            try {
                this.wait(500L);
            }
            catch (InterruptedException e2) {
                Thread.currentThread().interrupt();
                throw new RuntimeException(e2);
            }
            left = expire - System.currentTimeMillis();
        }
        if (!this.connected) {
            throw new TimeoutException("Could not connect to ZooKeeper " + this.zkServerAddress + " within " + waitForConnection + " ms");
        }
        log.info("Client is connected to ZooKeeper");
    }

    public synchronized void waitForDisconnected(long timeout) throws InterruptedException, TimeoutException {
        long expire = System.currentTimeMillis() + timeout;
        long left = timeout;
        while (this.connected && left > 0L) {
            this.wait(left);
            left = expire - System.currentTimeMillis();
        }
        if (this.connected) {
            throw new TimeoutException("Did not disconnect");
        }
    }

    private void closeKeeper(SolrZooKeeper keeper) {
        try {
            keeper.close();
        }
        catch (InterruptedException e2) {
            Thread.currentThread().interrupt();
            log.error("", e2);
            throw new ZooKeeperException(SolrException.ErrorCode.SERVER_ERROR, "", (Throwable)e2);
        }
    }
}

