/****************************************************************
 * Licensed to the Apache Software Foundation (ASF) under one   *
 * or more contributor license agreements.  See the NOTICE file *
 * distributed with this work for additional information        *
 * regarding copyright ownership.  The ASF licenses this file   *
 * to you under the Apache License, Version 2.0 (the            *
 * "License"); you may not use this file except in compliance   *
 * with the License.  You may obtain a copy of the License at   *
 *                                                              *
 *   http://www.apache.org/licenses/LICENSE-2.0                 *
 *                                                              *
 * Unless required by applicable law or agreed to in writing,   *
 * software distributed under the License is distributed on an  *
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY       *
 * KIND, either express or implied.  See the License for the    *
 * specific language governing permissions and limitations      *
 * under the License.                                           *
 ****************************************************************/
package org.apache.james.mailbox.store.mail;

import java.util.List;

import org.apache.james.mailbox.MailboxException;
import org.apache.james.mailbox.MailboxNotFoundException;
import org.apache.james.mailbox.MailboxPath;
import org.apache.james.mailbox.store.mail.model.Mailbox;
import org.apache.james.mailbox.store.transaction.Mapper;

/**
 * Mapper for {@link Mailbox} actions. A {@link MailboxMapper} has a lifecycle from the start of a request 
 * to the end of the request.
 *
 */
public interface MailboxMapper<Id> extends Mapper {
    
    /**
     * Save the give {@link Mailbox} to the underlying storage
     * 
     * @param mailbox
     * @throws MailboxException
     */
    public abstract void save(Mailbox<Id> mailbox) throws MailboxException;
    
    /**
     * Delete the given {@link Mailbox} from the underlying storage
     * 
     * @param mailbox
     * @throws MailboxException
     */
    public abstract void delete(Mailbox<Id> mailbox) throws MailboxException;

    /**
     * Delete all {@link Mailbox} objects from the underlying storage
     * 
     * @throws MailboxException
     */
    public abstract void deleteAll() throws MailboxException;

    /**
     * Return the {@link Mailbox} for the given name
     * 
     * @param name 
     * @return mailbox
     * @throws MailboxException
     * @throws MailboxNotFoundException
     */
    public abstract Mailbox<Id> findMailboxByPath(MailboxPath mailboxName)
            throws MailboxException, MailboxNotFoundException;

    /**
     * Return a List of {@link Mailbox} which name is like the given name
     * 
     * @param name
     * @return mailboxList
     * @throws MailboxException
     */
    public abstract List<Mailbox<Id>> findMailboxWithPathLike(MailboxPath mailboxPath)
            throws MailboxException;

    /**
     * Return the {@link Mailbox} for the given id
     * 
     * @param mailboxId
     * @return mailbox
     * @throws MailboxException
     * @throws MailboxNotFoundException
     */
    public abstract Mailbox<Id> findMailboxById(Id mailboxId)
            throws MailboxException, MailboxNotFoundException;

    /**
     * Return if the given {@link Mailbox} has children
     * 
     * @param mailbox not null
     * @return true when the mailbox has children, false otherwise
     * @throws MailboxException
     * @throws MailboxNotFoundException
     */
    public abstract boolean hasChildren(Mailbox<Id> mailbox)
            throws MailboxException, MailboxNotFoundException;
}