/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.quotas;

import java.io.IOException;
import java.time.Duration;
import java.util.EnumSet;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.ClusterMetrics;
import org.apache.hadoop.hbase.ScheduledChore;
import org.apache.hadoop.hbase.Stoppable;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.RegionStatesCount;
import org.apache.hadoop.hbase.ipc.RpcCall;
import org.apache.hadoop.hbase.ipc.RpcServer;
import org.apache.hadoop.hbase.quotas.NoopQuotaLimiter;
import org.apache.hadoop.hbase.quotas.QuotaLimiter;
import org.apache.hadoop.hbase.quotas.QuotaState;
import org.apache.hadoop.hbase.quotas.QuotaUtil;
import org.apache.hadoop.hbase.quotas.UserQuotaState;
import org.apache.hadoop.hbase.regionserver.RegionServerServices;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hbase.thirdparty.com.google.common.cache.CacheBuilder;
import org.apache.hbase.thirdparty.com.google.common.cache.CacheLoader;
import org.apache.hbase.thirdparty.com.google.common.cache.LoadingCache;
import org.apache.yetus.audience.InterfaceAudience;
import org.apache.yetus.audience.InterfaceStability;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class QuotaCache
implements Stoppable {
    private static final Logger LOG = LoggerFactory.getLogger(QuotaCache.class);
    public static final String REFRESH_CONF_KEY = "hbase.quota.refresh.period";
    public static final String TABLE_REGION_STATES_CACHE_TTL_MS = "hbase.quota.cache.ttl.region.states.ms";
    public static final String REGION_SERVERS_SIZE_CACHE_TTL_MS = "hbase.quota.cache.ttl.servers.size.ms";
    public static final String QUOTA_USER_REQUEST_ATTRIBUTE_OVERRIDE_KEY = "hbase.quota.user.override.key";
    private static final int REFRESH_DEFAULT_PERIOD = 43200000;
    private final Object initializerLock = new Object();
    private volatile boolean initialized = false;
    private volatile Map<String, QuotaState> namespaceQuotaCache = new ConcurrentHashMap<String, QuotaState>();
    private volatile Map<TableName, QuotaState> tableQuotaCache = new ConcurrentHashMap<TableName, QuotaState>();
    private volatile Map<String, UserQuotaState> userQuotaCache = new ConcurrentHashMap<String, UserQuotaState>();
    private volatile Map<String, QuotaState> regionServerQuotaCache = new ConcurrentHashMap<String, QuotaState>();
    private volatile boolean exceedThrottleQuotaEnabled = false;
    private volatile double machineQuotaFactor = 1.0;
    private final ConcurrentHashMap<TableName, Double> tableMachineQuotaFactors = new ConcurrentHashMap();
    private final RegionServerServices rsServices;
    private final String userOverrideRequestAttributeKey;
    private QuotaRefresherChore refreshChore;
    private boolean stopped = true;

    public QuotaCache(RegionServerServices rsServices) {
        this.rsServices = rsServices;
        this.userOverrideRequestAttributeKey = rsServices.getConfiguration().get(QUOTA_USER_REQUEST_ATTRIBUTE_OVERRIDE_KEY);
    }

    public void start() throws IOException {
        this.stopped = false;
        Configuration conf = this.rsServices.getConfiguration();
        int period = conf.getInt(REFRESH_CONF_KEY, 43200000);
        this.refreshChore = new QuotaRefresherChore(conf, period, this);
        this.rsServices.getChoreService().scheduleChore((ScheduledChore)this.refreshChore);
    }

    public void stop(String why) {
        if (this.refreshChore != null) {
            LOG.debug("Stopping QuotaRefresherChore chore.");
            this.refreshChore.shutdown(true);
        }
        this.stopped = true;
    }

    public boolean isStopped() {
        return this.stopped;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void ensureInitialized() {
        if (!this.initialized) {
            Object object = this.initializerLock;
            synchronized (object) {
                if (!this.initialized) {
                    this.refreshChore.chore();
                    this.initialized = true;
                }
            }
        }
    }

    private Map<String, UserQuotaState> fetchUserQuotaStateEntries() throws IOException {
        return QuotaUtil.fetchUserQuotas(this.rsServices.getConfiguration(), this.rsServices.getConnection(), this.tableMachineQuotaFactors, this.machineQuotaFactor);
    }

    private Map<String, QuotaState> fetchRegionServerQuotaStateEntries() throws IOException {
        return QuotaUtil.fetchRegionServerQuotas(this.rsServices.getConfiguration(), this.rsServices.getConnection());
    }

    private Map<TableName, QuotaState> fetchTableQuotaStateEntries() throws IOException {
        return QuotaUtil.fetchTableQuotas(this.rsServices.getConfiguration(), this.rsServices.getConnection(), this.tableMachineQuotaFactors);
    }

    private Map<String, QuotaState> fetchNamespaceQuotaStateEntries() throws IOException {
        return QuotaUtil.fetchNamespaceQuotas(this.rsServices.getConfiguration(), this.rsServices.getConnection(), this.machineQuotaFactor);
    }

    public QuotaLimiter getUserLimiter(UserGroupInformation ugi, TableName table) {
        if (table.isSystemTable()) {
            return NoopQuotaLimiter.get();
        }
        return this.getUserQuotaState(ugi).getTableLimiter(table);
    }

    public UserQuotaState getUserQuotaState(UserGroupInformation ugi) {
        String user = this.getQuotaUserName(ugi);
        this.ensureInitialized();
        Map<String, UserQuotaState> cache = this.userQuotaCache;
        if (!cache.containsKey(user)) {
            cache.put(user, QuotaUtil.buildDefaultUserQuotaState(this.rsServices.getConfiguration()));
        }
        return cache.get(user);
    }

    public QuotaLimiter getTableLimiter(TableName table) {
        this.ensureInitialized();
        Map<TableName, QuotaState> cache = this.tableQuotaCache;
        if (!cache.containsKey(table)) {
            cache.put(table, new QuotaState());
        }
        return cache.get(table).getGlobalLimiter();
    }

    public QuotaLimiter getNamespaceLimiter(String namespace) {
        this.ensureInitialized();
        Map<String, QuotaState> cache = this.namespaceQuotaCache;
        if (!cache.containsKey(namespace)) {
            cache.put(namespace, new QuotaState());
        }
        return cache.get(namespace).getGlobalLimiter();
    }

    public QuotaLimiter getRegionServerQuotaLimiter(String regionServer) {
        this.ensureInitialized();
        Map<String, QuotaState> cache = this.regionServerQuotaCache;
        if (!cache.containsKey(regionServer)) {
            cache.put(regionServer, new QuotaState());
        }
        return cache.get(regionServer).getGlobalLimiter();
    }

    protected boolean isExceedThrottleQuotaEnabled() {
        return this.exceedThrottleQuotaEnabled;
    }

    String getQuotaUserName(UserGroupInformation ugi) {
        if (this.userOverrideRequestAttributeKey == null) {
            return ugi.getShortUserName();
        }
        Optional<RpcCall> rpcCall = RpcServer.getCurrentCall();
        if (!rpcCall.isPresent()) {
            return ugi.getShortUserName();
        }
        byte[] override = rpcCall.get().getRequestAttribute(this.userOverrideRequestAttributeKey);
        if (override == null) {
            return ugi.getShortUserName();
        }
        return Bytes.toString((byte[])override);
    }

    void triggerCacheRefresh() {
        this.refreshChore.triggerNow();
    }

    void forceSynchronousCacheRefresh() {
        this.refreshChore.chore();
    }

    Map<String, QuotaState> getNamespaceQuotaCache() {
        return this.namespaceQuotaCache;
    }

    Map<String, QuotaState> getRegionServerQuotaCache() {
        return this.regionServerQuotaCache;
    }

    Map<TableName, QuotaState> getTableQuotaCache() {
        return this.tableQuotaCache;
    }

    Map<String, UserQuotaState> getUserQuotaCache() {
        return this.userQuotaCache;
    }

    static <K, V extends QuotaState> void updateNewCacheFromOld(Map<K, V> oldCache, Map<K, V> newCache) {
        for (Map.Entry<K, V> entry : oldCache.entrySet()) {
            K key = entry.getKey();
            if (!newCache.containsKey(key)) continue;
            QuotaState newState = (QuotaState)newCache.get(key);
            QuotaState oldState = (QuotaState)entry.getValue();
            oldState.update(newState);
            newCache.put(key, oldState);
        }
    }

    @FunctionalInterface
    static interface ThrowingSupplier<T> {
        public T get() throws Exception;
    }

    static class RefreshableExpiringValueCache<T> {
        private final String name;
        private final LoadingCache<String, Optional<T>> cache;

        RefreshableExpiringValueCache(final String name, Duration refreshPeriod, final ThrowingSupplier<T> supplier) {
            this.name = name;
            this.cache = CacheBuilder.newBuilder().expireAfterWrite(refreshPeriod.toMillis(), TimeUnit.MILLISECONDS).build(new CacheLoader<String, Optional<T>>(){

                public Optional<T> load(String key) {
                    try {
                        return Optional.of(supplier.get());
                    }
                    catch (Exception e) {
                        LOG.warn("Failed to refresh cache {}", (Object)name, (Object)e);
                        return Optional.empty();
                    }
                }
            });
        }

        Optional<T> get() {
            return (Optional)this.cache.getUnchecked((Object)this.name);
        }

        void refresh() {
            this.cache.refresh((Object)this.name);
        }

        void invalidate() {
            this.cache.invalidate((Object)this.name);
        }
    }

    private class QuotaRefresherChore
    extends ScheduledChore {
        private final RefreshableExpiringValueCache<ClusterMetrics> tableRegionStatesClusterMetrics;
        private final RefreshableExpiringValueCache<Integer> regionServersSize;

        public QuotaRefresherChore(Configuration conf, int period, Stoppable stoppable) {
            super("QuotaRefresherChore", stoppable, period);
            Duration tableRegionStatesCacheTtl = Duration.ofMillis(conf.getLong(QuotaCache.TABLE_REGION_STATES_CACHE_TTL_MS, (long)period));
            this.tableRegionStatesClusterMetrics = new RefreshableExpiringValueCache<ClusterMetrics>("tableRegionStatesClusterMetrics", tableRegionStatesCacheTtl, () -> QuotaCache.this.rsServices.getConnection().getAdmin().getClusterMetrics(EnumSet.of(ClusterMetrics.Option.SERVERS_NAME, ClusterMetrics.Option.TABLE_TO_REGIONS_COUNT)));
            Duration regionServersSizeCacheTtl = Duration.ofMillis(conf.getLong(QuotaCache.REGION_SERVERS_SIZE_CACHE_TTL_MS, (long)period));
            this.regionServersSize = new RefreshableExpiringValueCache<Integer>("regionServersSize", regionServersSizeCacheTtl, () -> QuotaCache.this.rsServices.getConnection().getAdmin().getRegionServers().size());
        }

        public synchronized boolean triggerNow() {
            this.tableRegionStatesClusterMetrics.invalidate();
            this.regionServersSize.invalidate();
            return super.triggerNow();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected void chore() {
            QuotaRefresherChore quotaRefresherChore = this;
            synchronized (quotaRefresherChore) {
                LOG.info("Reloading quota cache from hbase:quota table");
                this.updateQuotaFactors();
                try {
                    ConcurrentHashMap newUserQuotaCache = new ConcurrentHashMap(QuotaCache.this.fetchUserQuotaStateEntries());
                    QuotaCache.updateNewCacheFromOld(QuotaCache.this.userQuotaCache, newUserQuotaCache);
                    QuotaCache.this.userQuotaCache = newUserQuotaCache;
                }
                catch (IOException e) {
                    LOG.error("Error while fetching user quotas", (Throwable)e);
                }
                try {
                    ConcurrentHashMap newRegionServerQuotaCache = new ConcurrentHashMap(QuotaCache.this.fetchRegionServerQuotaStateEntries());
                    QuotaCache.updateNewCacheFromOld(QuotaCache.this.regionServerQuotaCache, newRegionServerQuotaCache);
                    QuotaCache.this.regionServerQuotaCache = newRegionServerQuotaCache;
                }
                catch (IOException e) {
                    LOG.error("Error while fetching region server quotas", (Throwable)e);
                }
                try {
                    ConcurrentHashMap newTableQuotaCache = new ConcurrentHashMap(QuotaCache.this.fetchTableQuotaStateEntries());
                    QuotaCache.updateNewCacheFromOld(QuotaCache.this.tableQuotaCache, newTableQuotaCache);
                    QuotaCache.this.tableQuotaCache = newTableQuotaCache;
                }
                catch (IOException e) {
                    LOG.error("Error while refreshing table quotas", (Throwable)e);
                }
                try {
                    ConcurrentHashMap newNamespaceQuotaCache = new ConcurrentHashMap(QuotaCache.this.fetchNamespaceQuotaStateEntries());
                    QuotaCache.updateNewCacheFromOld(QuotaCache.this.namespaceQuotaCache, newNamespaceQuotaCache);
                    QuotaCache.this.namespaceQuotaCache = newNamespaceQuotaCache;
                }
                catch (IOException e) {
                    LOG.error("Error while refreshing namespace quotas", (Throwable)e);
                }
                this.fetchExceedThrottleQuota();
            }
        }

        private void fetchExceedThrottleQuota() {
            try {
                QuotaCache.this.exceedThrottleQuotaEnabled = QuotaUtil.isExceedThrottleQuotaEnabled(QuotaCache.this.rsServices.getConnection());
            }
            catch (IOException e) {
                LOG.warn("Unable to read if exceed throttle quota enabled from quota table", (Throwable)e);
            }
        }

        private void updateQuotaFactors() {
            boolean hasTableQuotas;
            boolean bl = hasTableQuotas = !QuotaCache.this.tableQuotaCache.entrySet().isEmpty() || QuotaCache.this.userQuotaCache.values().stream().anyMatch(UserQuotaState::hasTableLimiters);
            if (hasTableQuotas) {
                this.updateTableMachineQuotaFactors();
            } else {
                this.updateOnlyMachineQuotaFactors();
            }
        }

        private void updateOnlyMachineQuotaFactors() {
            Optional<Integer> rsSize = this.regionServersSize.get();
            if (rsSize.isPresent()) {
                this.updateMachineQuotaFactors(rsSize.get());
            } else {
                this.regionServersSize.refresh();
            }
        }

        private void updateTableMachineQuotaFactors() {
            Optional<ClusterMetrics> clusterMetricsMaybe = this.tableRegionStatesClusterMetrics.get();
            if (!clusterMetricsMaybe.isPresent()) {
                this.tableRegionStatesClusterMetrics.refresh();
                return;
            }
            ClusterMetrics clusterMetrics = clusterMetricsMaybe.get();
            this.updateMachineQuotaFactors(clusterMetrics.getServersName().size());
            Map tableRegionStatesCount = clusterMetrics.getTableRegionStatesCount();
            for (TableName tableName : QuotaCache.this.tableQuotaCache.keySet()) {
                if (tableRegionStatesCount.containsKey(tableName)) {
                    double factor = 1.0;
                    try {
                        long regionSize = ((RegionStatesCount)tableRegionStatesCount.get(tableName)).getOpenRegions();
                        if (regionSize == 0L) {
                            factor = 0.0;
                        } else {
                            int localRegionSize = QuotaCache.this.rsServices.getRegions(tableName).size();
                            factor = 1.0 * (double)localRegionSize / (double)regionSize;
                        }
                    }
                    catch (IOException e) {
                        LOG.warn("Get table regions failed: {}", (Object)tableName, (Object)e);
                    }
                    QuotaCache.this.tableMachineQuotaFactors.put(tableName, factor);
                    continue;
                }
                QuotaCache.this.tableMachineQuotaFactors.remove(tableName);
            }
        }

        private void updateMachineQuotaFactors(int rsSize) {
            if (rsSize != 0) {
                QuotaCache.this.machineQuotaFactor = 1.0 / (double)rsSize;
            }
        }
    }
}

