/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.replication.regionserver;

import edu.umd.cs.findbugs.annotations.SuppressWarnings;
import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.OptionalLong;
import java.util.concurrent.PriorityBlockingQueue;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.regionserver.wal.ProtobufLogReader;
import org.apache.hadoop.hbase.replication.regionserver.MetricsSource;
import org.apache.hadoop.hbase.replication.regionserver.ReplicationSourceLogQueue;
import org.apache.hadoop.hbase.replication.regionserver.WALFileLengthProvider;
import org.apache.hadoop.hbase.util.CancelableProgressable;
import org.apache.hadoop.hbase.util.CommonFSUtils;
import org.apache.hadoop.hbase.util.LeaseNotRecoveredException;
import org.apache.hadoop.hbase.util.RecoverLeaseFSUtils;
import org.apache.hadoop.hbase.wal.AbstractFSWALProvider;
import org.apache.hadoop.hbase.wal.WAL;
import org.apache.hadoop.hbase.wal.WALFactory;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.yetus.audience.InterfaceAudience;
import org.apache.yetus.audience.InterfaceStability;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Evolving
class WALEntryStream
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(WALEntryStream.class);
    private WAL.Reader reader;
    private Path currentPath;
    private WAL.Entry currentEntry;
    private long currentPositionOfEntry = 0L;
    private long currentPositionOfReader = 0L;
    private final ReplicationSourceLogQueue logQueue;
    private final String walGroupId;
    private final FileSystem fs;
    private final Configuration conf;
    private final WALFileLengthProvider walFileLengthProvider;
    private final ServerName serverName;
    private final MetricsSource metrics;

    public WALEntryStream(ReplicationSourceLogQueue logQueue, Configuration conf, long startPosition, WALFileLengthProvider walFileLengthProvider, ServerName serverName, MetricsSource metrics, String walGroupId) throws IOException {
        this.logQueue = logQueue;
        this.fs = CommonFSUtils.getWALFileSystem((Configuration)conf);
        this.conf = conf;
        this.currentPositionOfEntry = startPosition;
        this.walFileLengthProvider = walFileLengthProvider;
        this.serverName = serverName;
        this.metrics = metrics;
        this.walGroupId = walGroupId;
    }

    public boolean hasNext() throws IOException {
        if (this.currentEntry == null) {
            this.tryAdvanceEntry();
        }
        return this.currentEntry != null;
    }

    public WAL.Entry peek() throws IOException {
        return this.hasNext() ? this.currentEntry : null;
    }

    public WAL.Entry next() throws IOException {
        WAL.Entry save = this.peek();
        this.currentPositionOfEntry = this.currentPositionOfReader;
        this.currentEntry = null;
        return save;
    }

    @Override
    public void close() throws IOException {
        this.closeReader();
    }

    public long getPosition() {
        return this.currentPositionOfEntry;
    }

    public Path getCurrentPath() {
        return this.currentPath;
    }

    private String getCurrentPathStat() {
        StringBuilder sb = new StringBuilder();
        if (this.currentPath != null) {
            sb.append("currently replicating from: ").append(this.currentPath).append(" at position: ").append(this.currentPositionOfEntry).append("\n");
        } else {
            sb.append("no replication ongoing, waiting for new log");
        }
        return sb.toString();
    }

    public void reset() throws IOException {
        if (this.reader != null && this.currentPath != null) {
            this.resetReader();
        }
    }

    private void setPosition(long position) {
        this.currentPositionOfEntry = position;
    }

    private void setCurrentPath(Path path) {
        this.currentPath = path;
    }

    private void tryAdvanceEntry() throws IOException {
        if (this.checkReader()) {
            boolean beingWritten = this.readNextEntryAndRecordReaderPosition();
            LOG.trace("Reading WAL {}; currently open for write={}", (Object)this.currentPath, (Object)beingWritten);
            if (this.currentEntry == null && !beingWritten) {
                this.resetReader();
                this.readNextEntryAndRecordReaderPosition();
                if (this.currentEntry == null && this.checkAllBytesParsed()) {
                    this.dequeueCurrentLog();
                    if (this.openNextLog()) {
                        this.readNextEntryAndRecordReaderPosition();
                    }
                }
            }
        }
    }

    private boolean checkAllBytesParsed() throws IOException {
        long trailerSize = this.currentTrailerSize();
        FileStatus stat = null;
        try {
            stat = this.fs.getFileStatus(this.currentPath);
        }
        catch (IOException exception) {
            LOG.warn("Couldn't get file length information about log {}, it {} closed cleanly {}", new Object[]{this.currentPath, trailerSize < 0L ? "was not" : "was", this.getCurrentPathStat()});
            this.metrics.incrUnknownFileLengthForClosedWAL();
        }
        if (stat != null) {
            if (trailerSize < 0L) {
                if (this.currentPositionOfReader < stat.getLen()) {
                    long skippedBytes = stat.getLen() - this.currentPositionOfReader;
                    LOG.warn("Reached the end of WAL {}. It was not closed cleanly, so we did not parse {} bytes of data.", (Object)this.currentPath, (Object)skippedBytes);
                    this.metrics.incrUncleanlyClosedWALs();
                    this.metrics.incrBytesSkippedInUncleanlyClosedWALs(skippedBytes);
                }
            } else if (this.currentPositionOfReader + trailerSize < stat.getLen()) {
                LOG.warn("Processing end of WAL {} at position {}, which is too far away from reported file length {}. Restarting WAL reading (see HBASE-15983 for details). {}", new Object[]{this.currentPath, this.currentPositionOfReader, stat.getLen(), this.getCurrentPathStat()});
                this.setPosition(0L);
                this.resetReader();
                this.metrics.incrRestartedWALReading();
                this.metrics.incrRepeatedFileBytes(this.currentPositionOfReader);
                return false;
            }
        }
        if (LOG.isTraceEnabled()) {
            LOG.trace("Reached the end of " + this.currentPath + " and length of the file is " + (stat == null ? "N/A" : Long.valueOf(stat.getLen())));
        }
        this.metrics.incrCompletedWAL();
        return true;
    }

    private void dequeueCurrentLog() throws IOException {
        LOG.debug("EOF, closing {}", (Object)this.currentPath);
        this.closeReader();
        this.logQueue.remove(this.walGroupId);
        this.setCurrentPath(null);
        this.setPosition(0L);
    }

    private boolean readNextEntryAndRecordReaderPosition() throws IOException {
        WAL.Entry readEntry = this.reader.next();
        long readerPos = this.reader.getPosition();
        OptionalLong fileLength = this.walFileLengthProvider.getLogFileSizeIfBeingWritten(this.currentPath);
        if (fileLength.isPresent() && readerPos > fileLength.getAsLong()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("The provider tells us the valid length for " + this.currentPath + " is " + fileLength.getAsLong() + ", but we have advanced to " + readerPos);
            }
            this.resetReader();
            return true;
        }
        if (readEntry != null) {
            LOG.trace("reading entry: {} ", (Object)readEntry);
            this.metrics.incrLogEditsRead();
            this.metrics.incrLogReadInBytes(readerPos - this.currentPositionOfEntry);
        }
        this.currentEntry = readEntry;
        this.currentPositionOfReader = readerPos;
        return fileLength.isPresent();
    }

    private void closeReader() throws IOException {
        if (this.reader != null) {
            this.reader.close();
            this.reader = null;
        }
    }

    private boolean checkReader() throws IOException {
        if (this.reader == null) {
            return this.openNextLog();
        }
        return true;
    }

    private boolean openNextLog() throws IOException {
        PriorityBlockingQueue<Path> queue = this.logQueue.getQueue(this.walGroupId);
        Path nextPath = queue.peek();
        if (nextPath != null) {
            this.openReader(nextPath);
            if (this.reader != null) {
                return true;
            }
        } else {
            this.setCurrentPath(null);
        }
        return false;
    }

    private void handleFileNotFound(Path path, FileNotFoundException fnfe) throws IOException {
        Path archivedLog = AbstractFSWALProvider.findArchivedLog(path, this.conf);
        if (archivedLog == null) {
            throw fnfe;
        }
        this.openReader(archivedLog);
    }

    @SuppressWarnings(value={"DCN_NULLPOINTER_EXCEPTION"}, justification="HDFS-4380")
    private void openReader(Path path) throws IOException {
        try {
            if (this.reader == null || !this.getCurrentPath().equals((Object)path)) {
                this.closeReader();
                this.reader = WALFactory.createReader(this.fs, path, this.conf);
                this.seek();
                this.setCurrentPath(path);
            } else {
                this.resetReader();
            }
        }
        catch (FileNotFoundException fnfe) {
            this.handleFileNotFound(path, fnfe);
        }
        catch (RemoteException re) {
            IOException ioe = re.unwrapRemoteException(new Class[]{FileNotFoundException.class});
            if (!(ioe instanceof FileNotFoundException)) {
                throw ioe;
            }
            this.handleFileNotFound(path, (FileNotFoundException)ioe);
        }
        catch (LeaseNotRecoveredException lnre) {
            LOG.warn("Try to recover the WAL lease " + path, (Throwable)((Object)lnre));
            this.recoverLease(this.conf, path);
            this.reader = null;
        }
        catch (NullPointerException npe) {
            LOG.warn("Got NPE opening reader, will retry.");
            this.reader = null;
        }
    }

    private void recoverLease(Configuration conf, final Path path) {
        try {
            FileSystem dfs = CommonFSUtils.getWALFileSystem((Configuration)conf);
            RecoverLeaseFSUtils.recoverFileLease((FileSystem)dfs, (Path)path, (Configuration)conf, (CancelableProgressable)new CancelableProgressable(){

                public boolean progress() {
                    LOG.debug("recover WAL lease: " + path);
                    return true;
                }
            });
        }
        catch (IOException e) {
            LOG.warn("unable to recover lease for WAL: " + path, (Throwable)e);
        }
    }

    @SuppressWarnings(value={"DCN_NULLPOINTER_EXCEPTION"}, justification="HDFS-4380")
    private void resetReader() throws IOException {
        try {
            this.currentEntry = null;
            this.reader.reset();
            this.seek();
        }
        catch (FileNotFoundException fnfe) {
            Path archivedLog = AbstractFSWALProvider.findArchivedLog(this.currentPath, this.conf);
            if (archivedLog != null) {
                this.openReader(archivedLog);
            }
            throw fnfe;
        }
        catch (NullPointerException npe) {
            throw new IOException("NPE resetting reader, likely HDFS-4380", npe);
        }
    }

    private void seek() throws IOException {
        if (this.currentPositionOfEntry != 0L) {
            this.reader.seek(this.currentPositionOfEntry);
        }
    }

    private long currentTrailerSize() {
        long size = -1L;
        if (this.reader instanceof ProtobufLogReader) {
            ProtobufLogReader pblr = (ProtobufLogReader)this.reader;
            size = pblr.trailerSize();
        }
        return size;
    }
}

