/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.regionserver;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InterruptedIOException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hbase.backup.HFileArchiver;
import org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hadoop.hbase.regionserver.HRegionFileSystem;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
final class HRegionFileSystemHelper {
    private static final Logger LOG = LoggerFactory.getLogger(HRegionFileSystemHelper.class);
    private final int hdfsClientRetriesNumber;
    private final int baseSleepBeforeRetries;
    private static final int DEFAULT_HDFS_CLIENT_RETRIES_NUMBER = 10;
    private static final int DEFAULT_BASE_SLEEP_BEFORE_RETRIES = 1000;
    private final Configuration conf;
    private final Path tableDir;
    private final Path regionDir;
    private final FileSystem fs;
    private final RegionInfo regionInfo;
    private final RegionInfo regionInfoForFs;

    HRegionFileSystemHelper(Configuration conf, FileSystem fs, Path tableDir, Path regionDir, RegionInfo regionInfo, RegionInfo regionInfoForFs) {
        this.conf = conf;
        this.fs = fs;
        this.regionDir = regionDir;
        this.tableDir = tableDir;
        this.regionInfo = regionInfo;
        this.regionInfoForFs = regionInfoForFs;
        this.hdfsClientRetriesNumber = conf.getInt("hdfs.client.retries.number", 10);
        this.baseSleepBeforeRetries = conf.getInt("hdfs.client.sleep.before.retries", 1000);
    }

    static byte[] getRegionInfoFileContent(RegionInfo hri) throws IOException {
        return RegionInfo.toDelimitedByteArray((RegionInfo)hri);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static void writeRegionInfoFileContent(Configuration conf, FileSystem fs, Path regionInfoFile, byte[] content) throws IOException {
        FsPermission perms = FSUtils.getFilePermissions((FileSystem)fs, (Configuration)conf, (String)"hbase.data.umask");
        try (FSDataOutputStream out = FSUtils.create(conf, fs, regionInfoFile, perms, null);){
            out.write(content);
        }
    }

    void checkRegionInfoOnFilesystem() throws IOException {
        byte[] content = HRegionFileSystemHelper.getRegionInfoFileContent(this.regionInfoForFs);
        try {
            FileStatus fileStatus = this.fs.getFileStatus(this.getRegionDir());
        }
        catch (FileNotFoundException e) {
            LOG.warn(this.getRegionDir() + " doesn't exist for region: " + this.regionInfoForFs.getEncodedName() + " on table " + this.regionInfo.getTable());
        }
        try {
            Path regionInfoFile = new Path(this.getRegionDir(), ".regioninfo");
            FileStatus status = this.fs.getFileStatus(regionInfoFile);
            if (status != null && status.getLen() == (long)content.length) {
                return;
            }
            LOG.info("Rewriting .regioninfo file at: " + regionInfoFile);
            if (!this.fs.delete(regionInfoFile, false)) {
                throw new IOException("Unable to remove existing " + regionInfoFile);
            }
        }
        catch (FileNotFoundException e) {
            LOG.warn(".regioninfo file not found for region: " + this.regionInfoForFs.getEncodedName() + " on table " + this.regionInfo.getTable());
        }
        this.writeRegionInfoOnFilesystem(content, true);
    }

    void writeRegionInfoOnFilesystem(boolean useTempDir) throws IOException {
        byte[] content = HRegionFileSystemHelper.getRegionInfoFileContent(this.regionInfoForFs);
        this.writeRegionInfoOnFilesystem(content, useTempDir);
    }

    private void writeRegionInfoOnFilesystem(byte[] regionInfoContent, boolean useTempDir) throws IOException {
        Path regionInfoFile = new Path(this.getRegionDir(), ".regioninfo");
        if (useTempDir) {
            Path tmpPath = new Path(this.getTempDir(), ".regioninfo");
            if (FSUtils.isExists((FileSystem)this.fs, (Path)tmpPath)) {
                FSUtils.delete((FileSystem)this.fs, (Path)tmpPath, (boolean)true);
            }
            HRegionFileSystemHelper.writeRegionInfoFileContent(this.conf, this.fs, tmpPath, regionInfoContent);
            if (this.fs.exists(tmpPath) && !this.rename(tmpPath, regionInfoFile)) {
                throw new IOException("Unable to rename " + tmpPath + " to " + regionInfoFile);
            }
        } else {
            HRegionFileSystemHelper.writeRegionInfoFileContent(this.conf, this.fs, regionInfoFile, regionInfoContent);
        }
    }

    public static HRegionFileSystem createRegionOnFileSystem(Configuration conf, FileSystem fs, Path tableDir, RegionInfo regionInfo) throws IOException {
        HRegionFileSystem regionFs = new HRegionFileSystem(conf, fs, tableDir, regionInfo);
        if (regionInfo.getReplicaId() == 0) {
            Path regionDir = regionFs.getRegionDir();
            if (fs.exists(regionDir)) {
                LOG.warn("Trying to create a region that already exists on disk: " + regionDir);
            } else if (!HRegionFileSystemHelper.createDirOnFileSystem(fs, conf, regionDir)) {
                LOG.warn("Unable to create the region directory: " + regionDir);
                throw new IOException("Unable to create region directory: " + regionDir);
            }
            regionFs.writeRegionInfoOnFilesystem(false);
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("Skipping creation of .regioninfo file for " + regionInfo);
        }
        return regionFs;
    }

    public static HRegionFileSystem openRegionFromFileSystem(Configuration conf, FileSystem fs, Path tableDir, RegionInfo regionInfo, boolean readOnly) throws IOException {
        HRegionFileSystem regionFs = new HRegionFileSystem(conf, fs, tableDir, regionInfo);
        Path regionDir = regionFs.getRegionDir();
        if (!fs.exists(regionDir)) {
            LOG.warn("Trying to open a region that do not exists on disk: " + regionDir);
            throw new IOException("The specified region do not exists on disk: " + regionDir);
        }
        if (!readOnly) {
            regionFs.cleanupTempDir();
            regionFs.cleanupSplitsDir();
            regionFs.cleanupMergesDir();
            if (regionInfo.getReplicaId() == 0) {
                regionFs.checkRegionInfoOnFilesystem();
            } else if (LOG.isDebugEnabled()) {
                LOG.debug("Skipping creation of .regioninfo file for " + regionInfo);
            }
        }
        return regionFs;
    }

    public static void deleteRegionFromFileSystem(Configuration conf, FileSystem fs, Path tableDir, RegionInfo regionInfo) throws IOException {
        HRegionFileSystem regionFs = new HRegionFileSystem(conf, fs, tableDir, regionInfo);
        Path regionDir = regionFs.getRegionDir();
        if (!fs.exists(regionDir)) {
            LOG.warn("Trying to delete a region that do not exists on disk: " + regionDir);
            return;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("DELETING region " + regionDir);
        }
        Path rootDir = FSUtils.getRootDir((Configuration)conf);
        HFileArchiver.archiveRegion(fs, rootDir, tableDir, regionDir);
        if (!fs.delete(regionDir, true)) {
            LOG.warn("Failed delete of " + regionDir);
        }
    }

    boolean createDir(Path dir) throws IOException {
        int i = 0;
        IOException lastIOE = null;
        while (true) {
            try {
                return HRegionFileSystemHelper.mkdirs(this.fs, this.conf, dir);
            }
            catch (IOException ioe) {
                lastIOE = ioe;
                if (this.fs.exists(dir)) {
                    return true;
                }
                try {
                    HRegionFileSystemHelper.sleepBeforeRetry("Create Directory", i + 1, this.baseSleepBeforeRetries, this.hdfsClientRetriesNumber);
                    continue;
                }
                catch (InterruptedException e) {
                    throw (InterruptedIOException)new InterruptedIOException().initCause(e);
                }
                if (++i <= this.hdfsClientRetriesNumber) continue;
                throw new IOException("Exception in createDir", lastIOE);
            }
            break;
        }
    }

    boolean rename(Path srcpath, Path dstPath) throws IOException {
        IOException lastIOE = null;
        int i = 0;
        while (true) {
            try {
                return this.fs.rename(srcpath, dstPath);
            }
            catch (IOException ioe) {
                lastIOE = ioe;
                if (!this.fs.exists(srcpath) && this.fs.exists(dstPath)) {
                    return true;
                }
                try {
                    HRegionFileSystemHelper.sleepBeforeRetry("Rename Directory", i + 1, this.baseSleepBeforeRetries, this.hdfsClientRetriesNumber);
                    continue;
                }
                catch (InterruptedException e) {
                    throw (InterruptedIOException)new InterruptedIOException().initCause(e);
                }
                if (++i <= this.hdfsClientRetriesNumber) continue;
                throw new IOException("Exception in rename", lastIOE);
            }
            break;
        }
    }

    boolean deleteDir(Path dir) throws IOException {
        IOException lastIOE = null;
        int i = 0;
        while (true) {
            try {
                return this.fs.delete(dir, true);
            }
            catch (IOException ioe) {
                lastIOE = ioe;
                if (!this.fs.exists(dir)) {
                    return true;
                }
                try {
                    HRegionFileSystemHelper.sleepBeforeRetry("Delete Directory", i + 1, this.baseSleepBeforeRetries, this.hdfsClientRetriesNumber);
                    continue;
                }
                catch (InterruptedException e) {
                    throw (InterruptedIOException)new InterruptedIOException().initCause(e);
                }
                if (++i <= this.hdfsClientRetriesNumber) continue;
                throw new IOException("Exception in DeleteDir", lastIOE);
            }
            break;
        }
    }

    private static boolean createDirOnFileSystem(FileSystem fs, Configuration conf, Path dir) throws IOException {
        int i = 0;
        IOException lastIOE = null;
        int hdfsClientRetriesNumber = conf.getInt("hdfs.client.retries.number", 10);
        int baseSleepBeforeRetries = conf.getInt("hdfs.client.sleep.before.retries", 1000);
        while (true) {
            try {
                return fs.mkdirs(dir);
            }
            catch (IOException ioe) {
                lastIOE = ioe;
                if (fs.exists(dir)) {
                    return true;
                }
                try {
                    HRegionFileSystemHelper.sleepBeforeRetry("Create Directory", i + 1, baseSleepBeforeRetries, hdfsClientRetriesNumber);
                    continue;
                }
                catch (InterruptedException e) {
                    throw (InterruptedIOException)new InterruptedIOException().initCause(e);
                }
                if (++i <= hdfsClientRetriesNumber) continue;
                throw new IOException("Exception in createDir", lastIOE);
            }
            break;
        }
    }

    static boolean mkdirs(FileSystem fs, Configuration conf, Path dir) throws IOException {
        if (FSUtils.isDistributedFileSystem(fs) || !conf.getBoolean("hbase.data.umask.enable", false)) {
            return fs.mkdirs(dir);
        }
        FsPermission perms = FSUtils.getFilePermissions((FileSystem)fs, (Configuration)conf, (String)"hbase.data.umask");
        return fs.mkdirs(dir, perms);
    }

    private static void sleepBeforeRetry(String msg, int sleepMultiplier, int baseSleepBeforeRetries, int hdfsClientRetriesNumber) throws InterruptedException {
        if (sleepMultiplier > hdfsClientRetriesNumber) {
            if (LOG.isDebugEnabled()) {
                LOG.debug(msg + ", retries exhausted");
            }
            return;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug(msg + ", sleeping " + baseSleepBeforeRetries + " times " + sleepMultiplier);
        }
        Thread.sleep((long)baseSleepBeforeRetries * (long)sleepMultiplier);
    }

    Path getTempDir() {
        return new Path(this.getRegionDir(), ".tmp");
    }

    Configuration getConf() {
        return this.conf;
    }

    FileSystem getFileSystem() {
        return this.fs;
    }

    RegionInfo getRegionInfo() {
        return this.regionInfo;
    }

    Path getRegionDir() {
        return this.regionDir;
    }

    Path getTableDir() {
        return this.tableDir;
    }

    RegionInfo getRegionInfoForFS() {
        return this.regionInfoForFs;
    }
}

