/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.client;

import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentNavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.hadoop.hbase.HBaseIOException;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.MetaTableAccessor;
import org.apache.hadoop.hbase.RegionLocations;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.TableNotFoundException;
import org.apache.hadoop.hbase.client.AdvancedScanResultConsumer;
import org.apache.hadoop.hbase.client.AsyncConnectionImpl;
import org.apache.hadoop.hbase.client.AsyncRegionLocatorHelper;
import org.apache.hadoop.hbase.client.CatalogReplicaLoadBalanceSelector;
import org.apache.hadoop.hbase.client.CatalogReplicaLoadBalanceSelectorFactory;
import org.apache.hadoop.hbase.client.CatalogReplicaLoadBalanceSimpleSelector;
import org.apache.hadoop.hbase.client.CatalogReplicaMode;
import org.apache.hadoop.hbase.client.ConnectionUtils;
import org.apache.hadoop.hbase.client.Consistency;
import org.apache.hadoop.hbase.client.MetricsConnection;
import org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hadoop.hbase.client.RegionLocateType;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.ConcurrentMapUtils;
import org.apache.hbase.thirdparty.com.google.common.base.Objects;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
class AsyncNonMetaRegionLocator {
    private static final Logger LOG = LoggerFactory.getLogger(AsyncNonMetaRegionLocator.class);
    static final String MAX_CONCURRENT_LOCATE_REQUEST_PER_TABLE = "hbase.client.meta.max.concurrent.locate.per.table";
    private static final int DEFAULT_MAX_CONCURRENT_LOCATE_REQUEST_PER_TABLE = 8;
    static String LOCATE_PREFETCH_LIMIT = "hbase.client.locate.prefetch.limit";
    private static final int DEFAULT_LOCATE_PREFETCH_LIMIT = 10;
    private final AsyncConnectionImpl conn;
    private final int maxConcurrentLocateRequestPerTable;
    private final int locatePrefetchLimit;
    private CatalogReplicaMode metaReplicaMode;
    private CatalogReplicaLoadBalanceSelector metaReplicaSelector;
    private final ConcurrentMap<TableName, TableCache> cache = new ConcurrentHashMap<TableName, TableCache>();

    AsyncNonMetaRegionLocator(AsyncConnectionImpl conn) {
        this.conn = conn;
        this.maxConcurrentLocateRequestPerTable = conn.getConfiguration().getInt(MAX_CONCURRENT_LOCATE_REQUEST_PER_TABLE, 8);
        this.locatePrefetchLimit = conn.getConfiguration().getInt(LOCATE_PREFETCH_LIMIT, 10);
        this.metaReplicaMode = CatalogReplicaMode.fromString(conn.getConfiguration().get("hbase.locator.meta.replicas.mode", CatalogReplicaMode.NONE.toString()));
        switch (this.metaReplicaMode) {
            case LOAD_BALANCE: {
                String replicaSelectorClass = conn.getConfiguration().get("hbase.locator.meta.replicas.mode.loadbalance.selector", CatalogReplicaLoadBalanceSimpleSelector.class.getName());
                this.metaReplicaSelector = CatalogReplicaLoadBalanceSelectorFactory.createSelector(replicaSelectorClass, TableName.META_TABLE_NAME, conn.getChoreService(), () -> {
                    int numOfReplicas = 1;
                    try {
                        RegionLocations metaLocations = conn.registry.getMetaRegionLocations().get(conn.connConf.getReadRpcTimeoutNs(), TimeUnit.NANOSECONDS);
                        numOfReplicas = metaLocations.size();
                    }
                    catch (Exception e) {
                        LOG.error("Failed to get table {}'s region replication, ", (Object)TableName.META_TABLE_NAME, (Object)e);
                    }
                    return numOfReplicas;
                });
                break;
            }
            case NONE: {
                boolean useMetaReplicas = conn.getConfiguration().getBoolean("hbase.meta.replicas.use", false);
                if (!useMetaReplicas) break;
                this.metaReplicaMode = CatalogReplicaMode.HEDGED_READ;
                break;
            }
        }
    }

    private TableCache getTableCache(TableName tableName) {
        return (TableCache)ConcurrentMapUtils.computeIfAbsent(this.cache, (Object)tableName, () -> new TableCache());
    }

    private boolean isEqual(RegionLocations locs1, RegionLocations locs2) {
        HRegionLocation[] locArr2;
        HRegionLocation[] locArr1 = locs1.getRegionLocations();
        if (locArr1.length != (locArr2 = locs2.getRegionLocations()).length) {
            return false;
        }
        for (int i = 0; i < locArr1.length; ++i) {
            HRegionLocation loc1 = locArr1[i];
            HRegionLocation loc2 = locArr2[i];
            if (loc1 == null) {
                if (loc2 == null) continue;
                return false;
            }
            if (loc2 == null) {
                return false;
            }
            if (loc1.getSeqNum() != loc2.getSeqNum()) {
                return false;
            }
            if (Objects.equal((Object)loc1.getServerName(), (Object)loc2.getServerName())) continue;
            return false;
        }
        return true;
    }

    private RegionLocations addToCache(TableCache tableCache, RegionLocations locs) {
        LOG.trace("Try adding {} to cache", (Object)locs);
        byte[] startKey = locs.getRegionLocation().getRegion().getStartKey();
        while (true) {
            RegionLocations oldLocs;
            if ((oldLocs = tableCache.cache.putIfAbsent(startKey, locs)) == null) {
                return locs;
            }
            RegionInfo region = locs.getRegionLocation().getRegion();
            RegionInfo oldRegion = oldLocs.getRegionLocation().getRegion();
            if (region.getEncodedName().equals(oldRegion.getEncodedName())) {
                RegionLocations mergedLocs = oldLocs.mergeLocations(locs);
                if (this.isEqual(mergedLocs, oldLocs)) {
                    LOG.trace("Will not add {} to cache because the old value {}  is newer than us or has the same server name. Maybe it is updated before we replace it", (Object)locs, (Object)oldLocs);
                    return oldLocs;
                }
                if (!tableCache.cache.replace(startKey, oldLocs, mergedLocs)) continue;
                return mergedLocs;
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("The newnly fetch region {} is different from the old one {} for row '{}', try replaing the old one...", new Object[]{region, oldRegion, Bytes.toStringBinary((byte[])startKey)});
            }
            if (tableCache.cache.replace(startKey, oldLocs, locs)) break;
        }
        return locs;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void complete(TableName tableName, LocateRequest req, RegionLocations locs, Throwable error) {
        if (error != null) {
            LOG.warn("Failed to locate region in '" + tableName + "', row='" + Bytes.toStringBinary((byte[])req.row) + "', locateType=" + (Object)((Object)req.locateType), error);
        }
        Optional<Object> toSend = Optional.empty();
        TableCache tableCache = this.getTableCache(tableName);
        if (locs != null) {
            RegionLocations addedLocs = this.addToCache(tableCache, locs);
            TableCache tableCache2 = tableCache;
            synchronized (tableCache2) {
                tableCache.pendingRequests.remove(req);
                tableCache.clearCompletedRequests(addedLocs);
                toSend = tableCache.getCandidate();
                toSend.ifPresent(r -> tableCache.send((LocateRequest)r));
            }
            toSend.ifPresent(r -> this.locateInMeta(tableName, (LocateRequest)r));
        } else {
            assert (error != null);
            TableCache tableCache3 = tableCache;
            synchronized (tableCache3) {
                tableCache.pendingRequests.remove(req);
                CompletableFuture future = (CompletableFuture)tableCache.allRequests.remove(req);
                if (future != null) {
                    future.completeExceptionally(error);
                }
                tableCache.clearCompletedRequests(null);
                toSend = tableCache.getCandidate();
                toSend.ifPresent(r -> tableCache.send((LocateRequest)r));
            }
            toSend.ifPresent(r -> this.locateInMeta(tableName, (LocateRequest)r));
        }
    }

    private boolean onScanNext(TableName tableName, LocateRequest req, Result result) {
        RegionLocations locs = MetaTableAccessor.getRegionLocations(result);
        if (LOG.isDebugEnabled()) {
            LOG.debug("The fetched location of '{}', row='{}', locateType={} is {}", new Object[]{tableName, Bytes.toStringBinary((byte[])req.row), req.locateType, locs});
        }
        if (locs != null) {
            locs = locs.removeElementsWithNullLocation();
        }
        if (locs == null || locs.getDefaultRegionLocation() == null) {
            this.complete(tableName, req, null, (Throwable)new HBaseIOException(String.format("No location found for '%s', row='%s', locateType=%s", new Object[]{tableName, Bytes.toStringBinary((byte[])req.row), req.locateType})));
            return true;
        }
        HRegionLocation loc = locs.getDefaultRegionLocation();
        RegionInfo info = loc.getRegion();
        if (info == null) {
            this.complete(tableName, req, null, (Throwable)new HBaseIOException(String.format("HRegionInfo is null for '%s', row='%s', locateType=%s", new Object[]{tableName, Bytes.toStringBinary((byte[])req.row), req.locateType})));
            return true;
        }
        if (info.isSplitParent()) {
            return false;
        }
        this.complete(tableName, req, locs, null);
        return true;
    }

    private void recordCacheHit() {
        this.conn.getConnectionMetrics().ifPresent(MetricsConnection::incrMetaCacheHit);
    }

    private void recordCacheMiss() {
        this.conn.getConnectionMetrics().ifPresent(MetricsConnection::incrMetaCacheMiss);
    }

    private RegionLocations locateRowInCache(TableCache tableCache, TableName tableName, byte[] row, int replicaId) {
        Map.Entry entry = tableCache.cache.floorEntry(row);
        if (entry == null) {
            this.recordCacheMiss();
            return null;
        }
        RegionLocations locs = (RegionLocations)entry.getValue();
        HRegionLocation loc = locs.getRegionLocation(replicaId);
        if (loc == null) {
            this.recordCacheMiss();
            return null;
        }
        byte[] endKey = loc.getRegion().getEndKey();
        if (ConnectionUtils.isEmptyStopRow(endKey) || Bytes.compareTo((byte[])row, (byte[])endKey) < 0) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("Found {} in cache for {}, row='{}', locateType={}, replicaId={}", new Object[]{loc, tableName, Bytes.toStringBinary((byte[])row), RegionLocateType.CURRENT, replicaId});
            }
            this.recordCacheHit();
            return locs;
        }
        this.recordCacheMiss();
        return null;
    }

    private RegionLocations locateRowBeforeInCache(TableCache tableCache, TableName tableName, byte[] row, int replicaId) {
        Map.Entry entry;
        boolean isEmptyStopRow = ConnectionUtils.isEmptyStopRow(row);
        Map.Entry entry2 = entry = isEmptyStopRow ? tableCache.cache.lastEntry() : tableCache.cache.lowerEntry(row);
        if (entry == null) {
            this.recordCacheMiss();
            return null;
        }
        RegionLocations locs = (RegionLocations)entry.getValue();
        HRegionLocation loc = locs.getRegionLocation(replicaId);
        if (loc == null) {
            this.recordCacheMiss();
            return null;
        }
        if (ConnectionUtils.isEmptyStopRow(loc.getRegion().getEndKey()) || !isEmptyStopRow && Bytes.compareTo((byte[])loc.getRegion().getEndKey(), (byte[])row) >= 0) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("Found {} in cache for {}, row='{}', locateType={}, replicaId={}", new Object[]{loc, tableName, Bytes.toStringBinary((byte[])row), RegionLocateType.BEFORE, replicaId});
            }
            this.recordCacheHit();
            return locs;
        }
        this.recordCacheMiss();
        return null;
    }

    private void locateInMeta(final TableName tableName, final LocateRequest req) {
        byte[] metaStartKey;
        if (LOG.isTraceEnabled()) {
            LOG.trace("Try locate '" + tableName + "', row='" + Bytes.toStringBinary((byte[])req.row) + "', locateType=" + (Object)((Object)req.locateType) + " in meta");
        }
        if (req.locateType.equals((Object)RegionLocateType.BEFORE)) {
            if (ConnectionUtils.isEmptyStopRow(req.row)) {
                byte[] binaryTableName = tableName.getName();
                metaStartKey = Arrays.copyOf(binaryTableName, binaryTableName.length + 1);
            } else {
                metaStartKey = RegionInfo.createRegionName(tableName, req.row, "00000000000000", false);
            }
        } else {
            metaStartKey = RegionInfo.createRegionName(tableName, req.row, "99999999999999", false);
        }
        byte[] metaStopKey = RegionInfo.createRegionName(tableName, HConstants.EMPTY_START_ROW, "", false);
        Scan scan = new Scan().withStartRow(metaStartKey).withStopRow(metaStopKey, true).addFamily(HConstants.CATALOG_FAMILY).setReversed(true).setCaching(this.locatePrefetchLimit).setReadType(Scan.ReadType.PREAD);
        switch (this.metaReplicaMode) {
            case LOAD_BALANCE: {
                int metaReplicaId = this.metaReplicaSelector.select(tableName, req.row, req.locateType);
                if (metaReplicaId == 0) break;
                scan.setConsistency(Consistency.TIMELINE);
                scan.setReplicaId(metaReplicaId);
                break;
            }
            case HEDGED_READ: {
                scan.setConsistency(Consistency.TIMELINE);
                break;
            }
        }
        this.conn.getTable(TableName.META_TABLE_NAME).scan(scan, new AdvancedScanResultConsumer(){
            private boolean completeNormally = false;
            private boolean tableNotFound = true;

            @Override
            public void onError(Throwable error) {
                AsyncNonMetaRegionLocator.this.complete(tableName, req, null, error);
            }

            @Override
            public void onComplete() {
                if (this.tableNotFound) {
                    AsyncNonMetaRegionLocator.this.complete(tableName, req, null, (Throwable)((Object)new TableNotFoundException(tableName)));
                } else if (!this.completeNormally) {
                    AsyncNonMetaRegionLocator.this.complete(tableName, req, null, new IOException("Unable to find region for '" + Bytes.toStringBinary((byte[])req.row) + "' in " + tableName));
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void onNext(Result[] results, AdvancedScanResultConsumer.ScanController controller) {
                int i;
                if (results.length == 0) {
                    return;
                }
                this.tableNotFound = false;
                for (i = 0; i < results.length; ++i) {
                    if (!AsyncNonMetaRegionLocator.this.onScanNext(tableName, req, results[i])) continue;
                    this.completeNormally = true;
                    controller.terminate();
                    ++i;
                    break;
                }
                if (i < results.length) {
                    TableCache tableCache = AsyncNonMetaRegionLocator.this.getTableCache(tableName);
                    while (i < results.length) {
                        RegionInfo info;
                        HRegionLocation loc;
                        RegionLocations locs = MetaTableAccessor.getRegionLocations(results[i]);
                        if (locs != null && (loc = locs.getDefaultRegionLocation()) != null && (info = loc.getRegion()) != null && !info.isOffline() && !info.isSplitParent()) {
                            RegionLocations addedLocs = AsyncNonMetaRegionLocator.this.addToCache(tableCache, locs);
                            TableCache tableCache2 = tableCache;
                            synchronized (tableCache2) {
                                tableCache.clearCompletedRequests(addedLocs);
                            }
                        }
                        ++i;
                    }
                }
            }
        });
    }

    private RegionLocations locateInCache(TableCache tableCache, TableName tableName, byte[] row, int replicaId, RegionLocateType locateType) {
        return locateType.equals((Object)RegionLocateType.BEFORE) ? this.locateRowBeforeInCache(tableCache, tableName, row, replicaId) : this.locateRowInCache(tableCache, tableName, row, replicaId);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CompletableFuture<RegionLocations> getRegionLocationsInternal(TableName tableName, byte[] row, int replicaId, RegionLocateType locateType, boolean reload) {
        CompletableFuture<RegionLocations> future;
        LocateRequest req;
        RegionLocations locs;
        assert (!locateType.equals((Object)RegionLocateType.AFTER));
        TableCache tableCache = this.getTableCache(tableName);
        if (!reload && AsyncRegionLocatorHelper.isGood(locs = this.locateInCache(tableCache, tableName, row, replicaId, locateType), replicaId)) {
            return CompletableFuture.completedFuture(locs);
        }
        boolean sendRequest = false;
        TableCache tableCache2 = tableCache;
        synchronized (tableCache2) {
            RegionLocations locs2;
            if (!reload && AsyncRegionLocatorHelper.isGood(locs2 = this.locateInCache(tableCache, tableName, row, replicaId, locateType), replicaId)) {
                return CompletableFuture.completedFuture(locs2);
            }
            req = new LocateRequest(row, locateType);
            future = (CompletableFuture<RegionLocations>)tableCache.allRequests.get(req);
            if (future == null) {
                future = new CompletableFuture<RegionLocations>();
                tableCache.allRequests.put(req, future);
                if (tableCache.hasQuota(this.maxConcurrentLocateRequestPerTable) && !tableCache.isPending(req)) {
                    tableCache.send(req);
                    sendRequest = true;
                }
            }
        }
        if (sendRequest) {
            this.locateInMeta(tableName, req);
        }
        return future;
    }

    CompletableFuture<RegionLocations> getRegionLocations(TableName tableName, byte[] row, int replicaId, RegionLocateType locateType, boolean reload) {
        if (locateType.equals((Object)RegionLocateType.AFTER)) {
            row = ConnectionUtils.createClosestRowAfter(row);
            locateType = RegionLocateType.CURRENT;
        }
        return this.getRegionLocationsInternal(tableName, row, replicaId, locateType, reload);
    }

    private void recordClearRegionCache() {
        this.conn.getConnectionMetrics().ifPresent(MetricsConnection::incrMetaCacheNumClearRegion);
    }

    private void removeLocationFromCache(HRegionLocation loc) {
        TableCache tableCache = (TableCache)this.cache.get(loc.getRegion().getTable());
        if (tableCache == null) {
            return;
        }
        byte[] startKey = loc.getRegion().getStartKey();
        while (true) {
            RegionLocations newLocs;
            RegionLocations oldLocs;
            if ((oldLocs = (RegionLocations)tableCache.cache.get(startKey)) == null) {
                return;
            }
            HRegionLocation oldLoc = oldLocs.getRegionLocation(loc.getRegion().getReplicaId());
            if (!AsyncRegionLocatorHelper.canUpdateOnError(loc, oldLoc)) {
                return;
            }
            if (this.metaReplicaMode == CatalogReplicaMode.LOAD_BALANCE) {
                this.metaReplicaSelector.onError(loc);
            }
            if ((newLocs = AsyncRegionLocatorHelper.removeRegionLocation(oldLocs, loc.getRegion().getReplicaId())) == null) {
                if (!tableCache.cache.remove(startKey, oldLocs)) continue;
                this.recordClearRegionCache();
                return;
            }
            if (tableCache.cache.replace(startKey, oldLocs, newLocs)) break;
        }
        this.recordClearRegionCache();
    }

    private void addLocationToCache(HRegionLocation loc) {
        this.addToCache(this.getTableCache(loc.getRegion().getTable()), AsyncRegionLocatorHelper.createRegionLocations(loc));
    }

    private HRegionLocation getCachedLocation(HRegionLocation loc) {
        TableCache tableCache = (TableCache)this.cache.get(loc.getRegion().getTable());
        if (tableCache == null) {
            return null;
        }
        RegionLocations locs = (RegionLocations)tableCache.cache.get(loc.getRegion().getStartKey());
        return locs != null ? locs.getRegionLocation(loc.getRegion().getReplicaId()) : null;
    }

    void updateCachedLocationOnError(HRegionLocation loc, Throwable exception) {
        Optional<MetricsConnection> connectionMetrics = this.conn.getConnectionMetrics();
        AsyncRegionLocatorHelper.updateCachedLocationOnError(loc, exception, this::getCachedLocation, this::addLocationToCache, this::removeLocationFromCache, connectionMetrics.orElse(null));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void clearCache(TableName tableName) {
        TableCache tableCache = (TableCache)this.cache.remove(tableName);
        if (tableCache == null) {
            return;
        }
        TableCache tableCache2 = tableCache;
        synchronized (tableCache2) {
            if (!tableCache.allRequests.isEmpty()) {
                IOException error = new IOException("Cache cleared");
                tableCache.allRequests.values().forEach(f -> f.completeExceptionally(error));
            }
        }
        this.conn.getConnectionMetrics().ifPresent(metrics -> metrics.incrMetaCacheNumClearRegion(tableCache.cache.size()));
    }

    void clearCache() {
        this.cache.clear();
    }

    void clearCache(ServerName serverName) {
        for (TableCache tableCache : this.cache.values()) {
            for (Map.Entry entry : tableCache.cache.entrySet()) {
                RegionLocations newLocs;
                byte[] regionName = (byte[])entry.getKey();
                RegionLocations locs = (RegionLocations)entry.getValue();
                if (locs == (newLocs = locs.removeByServer(serverName))) continue;
                if (newLocs.isEmpty()) {
                    tableCache.cache.remove(regionName, locs);
                    continue;
                }
                tableCache.cache.replace(regionName, locs, newLocs);
            }
        }
    }

    RegionLocations getRegionLocationInCache(TableName tableName, byte[] row) {
        TableCache tableCache = (TableCache)this.cache.get(tableName);
        if (tableCache == null) {
            return null;
        }
        return this.locateRowInCache(tableCache, tableName, row, 0);
    }

    private static final class TableCache {
        private final ConcurrentNavigableMap<byte[], RegionLocations> cache = new ConcurrentSkipListMap<byte[], RegionLocations>(Bytes.BYTES_COMPARATOR);
        private final Set<LocateRequest> pendingRequests = new HashSet<LocateRequest>();
        private final Map<LocateRequest, CompletableFuture<RegionLocations>> allRequests = new LinkedHashMap<LocateRequest, CompletableFuture<RegionLocations>>();

        private TableCache() {
        }

        public boolean hasQuota(int max) {
            return this.pendingRequests.size() < max;
        }

        public boolean isPending(LocateRequest req) {
            return this.pendingRequests.contains(req);
        }

        public void send(LocateRequest req) {
            this.pendingRequests.add(req);
        }

        public Optional<LocateRequest> getCandidate() {
            return this.allRequests.keySet().stream().filter(r -> !this.isPending((LocateRequest)r)).findFirst();
        }

        public void clearCompletedRequests(RegionLocations locations) {
            Iterator<Map.Entry<LocateRequest, CompletableFuture<RegionLocations>>> iter = this.allRequests.entrySet().iterator();
            while (iter.hasNext()) {
                Map.Entry<LocateRequest, CompletableFuture<RegionLocations>> entry = iter.next();
                if (!this.tryComplete(entry.getKey(), entry.getValue(), locations)) continue;
                iter.remove();
            }
        }

        private boolean tryComplete(LocateRequest req, CompletableFuture<RegionLocations> future, RegionLocations locations) {
            byte[] endKey;
            int c;
            if (future.isDone()) {
                return true;
            }
            if (locations == null) {
                return false;
            }
            HRegionLocation loc = (HRegionLocation)ObjectUtils.firstNonNull((Object[])locations.getRegionLocations());
            assert (loc != null);
            boolean completed = req.locateType.equals((Object)RegionLocateType.BEFORE) ? (c = Bytes.compareTo((byte[])(endKey = loc.getRegion().getEndKey()), (byte[])req.row)) == 0 || (c > 0 || Bytes.equals((byte[])HConstants.EMPTY_END_ROW, (byte[])endKey)) && Bytes.compareTo((byte[])loc.getRegion().getStartKey(), (byte[])req.row) < 0 : loc.getRegion().containsRow(req.row);
            if (completed) {
                future.complete(locations);
                return true;
            }
            return false;
        }
    }

    private static final class LocateRequest {
        private final byte[] row;
        private final RegionLocateType locateType;

        public LocateRequest(byte[] row, RegionLocateType locateType) {
            this.row = row;
            this.locateType = locateType;
        }

        public int hashCode() {
            return Bytes.hashCode((byte[])this.row) ^ this.locateType.hashCode();
        }

        public boolean equals(Object obj) {
            if (obj == null || obj.getClass() != LocateRequest.class) {
                return false;
            }
            LocateRequest that = (LocateRequest)obj;
            return this.locateType.equals((Object)that.locateType) && Bytes.equals((byte[])this.row, (byte[])that.row);
        }
    }
}

