/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.tools.offlineImageViewer;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.protobuf.CodedInputStream;
import com.google.protobuf.InvalidProtocolBufferException;
import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.XAttr;
import org.apache.hadoop.fs.permission.AclEntry;
import org.apache.hadoop.fs.permission.AclStatus;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.fs.permission.PermissionStatus;
import org.apache.hadoop.hdfs.XAttrHelper;
import org.apache.hadoop.hdfs.protocol.proto.HdfsProtos;
import org.apache.hadoop.hdfs.server.namenode.FSImageFormatPBINode;
import org.apache.hadoop.hdfs.server.namenode.FSImageFormatProtobuf;
import org.apache.hadoop.hdfs.server.namenode.FSImageUtil;
import org.apache.hadoop.hdfs.server.namenode.FsImageProto;
import org.apache.hadoop.hdfs.server.namenode.SerialNumberManager;
import org.apache.hadoop.hdfs.tools.offlineImageViewer.FSImageHandler;
import org.apache.hadoop.hdfs.web.JsonUtil;
import org.apache.hadoop.hdfs.web.resources.XAttrEncodingParam;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.util.LimitInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class FSImageLoader {
    public static final Logger LOG = LoggerFactory.getLogger(FSImageHandler.class);
    private final SerialNumberManager.StringTable stringTable;
    private final byte[][] inodes;
    private final Map<Long, long[]> dirmap;
    private static final Comparator<byte[]> INODE_BYTES_COMPARATOR = new Comparator<byte[]>(){

        @Override
        public int compare(byte[] o1, byte[] o2) {
            try {
                FsImageProto.INodeSection.INode l = FsImageProto.INodeSection.INode.parseFrom(o1);
                FsImageProto.INodeSection.INode r = FsImageProto.INodeSection.INode.parseFrom(o2);
                if (l.getId() < r.getId()) {
                    return -1;
                }
                if (l.getId() > r.getId()) {
                    return 1;
                }
                return 0;
            }
            catch (InvalidProtocolBufferException e) {
                throw new RuntimeException(e);
            }
        }
    };

    private FSImageLoader(SerialNumberManager.StringTable stringTable, byte[][] inodes, Map<Long, long[]> dirmap) {
        this.stringTable = stringTable;
        this.inodes = inodes;
        this.dirmap = dirmap;
    }

    static FSImageLoader load(String inputFile) throws IOException {
        Configuration conf = new Configuration();
        RandomAccessFile file = new RandomAccessFile(inputFile, "r");
        if (!FSImageUtil.checkFileFormat(file)) {
            throw new IOException("Unrecognized FSImage");
        }
        FsImageProto.FileSummary summary = FSImageUtil.loadSummary(file);
        try (FileInputStream fin = new FileInputStream(file.getFD());){
            ImmutableList<Long> refIdList = null;
            SerialNumberManager.StringTable stringTable = null;
            byte[][] inodes = null;
            Map<Long, long[]> dirmap = null;
            ArrayList sections = Lists.newArrayList(summary.getSectionsList());
            Collections.sort(sections, new Comparator<FsImageProto.FileSummary.Section>(){

                @Override
                public int compare(FsImageProto.FileSummary.Section s1, FsImageProto.FileSummary.Section s2) {
                    FSImageFormatProtobuf.SectionName n1 = FSImageFormatProtobuf.SectionName.fromString(s1.getName());
                    FSImageFormatProtobuf.SectionName n2 = FSImageFormatProtobuf.SectionName.fromString(s2.getName());
                    if (n1 == null) {
                        return n2 == null ? 0 : -1;
                    }
                    if (n2 == null) {
                        return -1;
                    }
                    return n1.ordinal() - n2.ordinal();
                }
            });
            for (FsImageProto.FileSummary.Section s : sections) {
                fin.getChannel().position(s.getOffset());
                InputStream is = FSImageUtil.wrapInputStreamForCompression(conf, summary.getCodec(), new BufferedInputStream((InputStream)new LimitInputStream((InputStream)fin, s.getLength())));
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Loading section " + s.getName() + " length: " + s.getLength());
                }
                switch (FSImageFormatProtobuf.SectionName.fromString(s.getName())) {
                    case STRING_TABLE: {
                        stringTable = FSImageLoader.loadStringTable(is);
                        break;
                    }
                    case INODE: {
                        inodes = FSImageLoader.loadINodeSection(is);
                        break;
                    }
                    case INODE_REFERENCE: {
                        refIdList = FSImageLoader.loadINodeReferenceSection(is);
                        break;
                    }
                    case INODE_DIR: {
                        dirmap = FSImageLoader.loadINodeDirectorySection(is, refIdList);
                        break;
                    }
                }
            }
            FSImageLoader fSImageLoader = new FSImageLoader(stringTable, inodes, dirmap);
            return fSImageLoader;
        }
    }

    private static Map<Long, long[]> loadINodeDirectorySection(InputStream in, List<Long> refIdList) throws IOException {
        FsImageProto.INodeDirectorySection.DirEntry e;
        LOG.info("Loading inode directory section");
        HashMap dirs = Maps.newHashMap();
        long counter = 0L;
        while ((e = FsImageProto.INodeDirectorySection.DirEntry.parseDelimitedFrom(in)) != null) {
            int i;
            ++counter;
            long[] l = new long[e.getChildrenCount() + e.getRefChildrenCount()];
            for (i = 0; i < e.getChildrenCount(); ++i) {
                l[i] = e.getChildren(i);
            }
            for (i = e.getChildrenCount(); i < l.length; ++i) {
                int refId = e.getRefChildren(i - e.getChildrenCount());
                l[i] = refIdList.get(refId);
            }
            dirs.put(e.getParent(), l);
        }
        LOG.info("Loaded " + counter + " directories");
        return dirs;
    }

    static ImmutableList<Long> loadINodeReferenceSection(InputStream in) throws IOException {
        FsImageProto.INodeReferenceSection.INodeReference e;
        LOG.info("Loading inode references");
        ImmutableList.Builder builder = ImmutableList.builder();
        long counter = 0L;
        while ((e = FsImageProto.INodeReferenceSection.INodeReference.parseDelimitedFrom(in)) != null) {
            ++counter;
            builder.add((Object)e.getReferredId());
        }
        LOG.info("Loaded " + counter + " inode references");
        return builder.build();
    }

    private static byte[][] loadINodeSection(InputStream in) throws IOException {
        FsImageProto.INodeSection s = FsImageProto.INodeSection.parseDelimitedFrom(in);
        LOG.info("Loading " + s.getNumInodes() + " inodes.");
        byte[][] inodes = new byte[(int)s.getNumInodes()][];
        int i = 0;
        while ((long)i < s.getNumInodes()) {
            int size = CodedInputStream.readRawVarint32((int)in.read(), (InputStream)in);
            byte[] bytes = new byte[size];
            IOUtils.readFully((InputStream)in, (byte[])bytes, (int)0, (int)size);
            inodes[i] = bytes;
            ++i;
        }
        LOG.debug("Sorting inodes");
        Arrays.sort(inodes, INODE_BYTES_COMPARATOR);
        LOG.debug("Finished sorting inodes");
        return inodes;
    }

    static SerialNumberManager.StringTable loadStringTable(InputStream in) throws IOException {
        FsImageProto.StringTableSection s = FsImageProto.StringTableSection.parseDelimitedFrom(in);
        LOG.info("Loading " + s.getNumEntry() + " strings");
        SerialNumberManager.StringTable stringTable = SerialNumberManager.newStringTable(s.getNumEntry(), s.getMaskBits());
        for (int i = 0; i < s.getNumEntry(); ++i) {
            FsImageProto.StringTableSection.Entry e = FsImageProto.StringTableSection.Entry.parseDelimitedFrom(in);
            stringTable.put(e.getId(), e.getStr());
        }
        return stringTable;
    }

    String getFileStatus(String path) throws IOException {
        FsImageProto.INodeSection.INode inode = this.fromINodeId(this.lookup(path));
        return "{\"FileStatus\":\n" + JsonUtil.toJsonString(this.getFileStatus(inode, false)) + "\n}\n";
    }

    String listStatus(String path) throws IOException {
        StringBuilder sb = new StringBuilder();
        List<Map<String, Object>> fileStatusList = this.getFileStatusList(path);
        sb.append("{\"FileStatuses\":{\"FileStatus\":[\n");
        int i = 0;
        for (Map<String, Object> fileStatusMap : fileStatusList) {
            if (i++ != 0) {
                sb.append(',');
            }
            sb.append(JsonUtil.toJsonString(fileStatusMap));
        }
        sb.append("\n]}}\n");
        return sb.toString();
    }

    private List<Map<String, Object>> getFileStatusList(String path) throws IOException {
        ArrayList<Map<String, Object>> list = new ArrayList<Map<String, Object>>();
        long id = this.lookup(path);
        FsImageProto.INodeSection.INode inode = this.fromINodeId(id);
        if (inode.getType() == FsImageProto.INodeSection.INode.Type.DIRECTORY) {
            long[] children;
            if (!this.dirmap.containsKey(id)) {
                return list;
            }
            for (long cid : children = this.dirmap.get(id)) {
                list.add(this.getFileStatus(this.fromINodeId(cid), true));
            }
        } else {
            list.add(this.getFileStatus(inode, false));
        }
        return list;
    }

    String getContentSummary(String path) throws IOException {
        return "{\"ContentSummary\":\n" + JsonUtil.toJsonString(this.getContentSummaryMap(path)) + "\n}\n";
    }

    private Map<String, Object> getContentSummaryMap(String path) throws IOException {
        long id = this.lookup(path);
        FsImageProto.INodeSection.INode inode = this.fromINodeId(id);
        long spaceQuota = 0L;
        long nsQuota = 0L;
        long[] data = new long[4];
        FsImageProto.INodeSection.INodeFile f = inode.getFile();
        switch (inode.getType()) {
            case FILE: {
                data[0] = 0L;
                data[1] = 1L;
                data[2] = FSImageLoader.getFileSize(f);
                nsQuota = -1L;
                data[3] = data[2] * (long)f.getReplication();
                spaceQuota = -1L;
                return this.fillSummaryMap(spaceQuota, nsQuota, data);
            }
            case DIRECTORY: {
                this.fillDirSummary(id, data);
                nsQuota = inode.getDirectory().getNsQuota();
                spaceQuota = inode.getDirectory().getDsQuota();
                return this.fillSummaryMap(spaceQuota, nsQuota, data);
            }
            case SYMLINK: {
                data[0] = 0L;
                data[1] = 1L;
                data[2] = 0L;
                nsQuota = -1L;
                data[3] = 0L;
                spaceQuota = -1L;
                return this.fillSummaryMap(spaceQuota, nsQuota, data);
            }
        }
        return null;
    }

    private Map<String, Object> fillSummaryMap(long spaceQuota, long nsQuota, long[] data) {
        HashMap map = Maps.newHashMap();
        map.put("directoryCount", data[0]);
        map.put("fileCount", data[1]);
        map.put("length", data[2]);
        map.put("quota", nsQuota);
        map.put("spaceConsumed", data[3]);
        map.put("spaceQuota", spaceQuota);
        return map;
    }

    private void fillDirSummary(long id, long[] data) throws IOException {
        data[0] = data[0] + 1L;
        long[] children = this.dirmap.get(id);
        if (children == null) {
            return;
        }
        block5: for (long cid : children) {
            FsImageProto.INodeSection.INode node = this.fromINodeId(cid);
            switch (node.getType()) {
                case DIRECTORY: {
                    this.fillDirSummary(cid, data);
                    continue block5;
                }
                case FILE: {
                    FsImageProto.INodeSection.INodeFile f = node.getFile();
                    long curLength = FSImageLoader.getFileSize(f);
                    data[1] = data[1] + 1L;
                    data[2] = data[2] + curLength;
                    data[3] = data[3] + curLength * (long)f.getReplication();
                    continue block5;
                }
                case SYMLINK: {
                    data[1] = data[1] + 1L;
                    continue block5;
                }
            }
        }
    }

    String listXAttrs(String path) throws IOException {
        return JsonUtil.toJsonString(this.getXAttrList(path));
    }

    String getXAttrs(String path, List<String> names, String encoder) throws IOException {
        ArrayList filtered;
        ArrayList xAttrs = this.getXAttrList(path);
        if (names == null || names.size() == 0) {
            filtered = xAttrs;
        } else {
            filtered = Lists.newArrayListWithCapacity((int)names.size());
            for (String name : names) {
                XAttr search = XAttrHelper.buildXAttr((String)name);
                boolean found = false;
                for (XAttr aXAttr : xAttrs) {
                    if (aXAttr.getNameSpace() != search.getNameSpace() || !aXAttr.getName().equals(search.getName())) continue;
                    filtered.add(aXAttr);
                    found = true;
                    break;
                }
                if (found) continue;
                throw new IOException("At least one of the attributes provided was not found.");
            }
        }
        return JsonUtil.toJsonString(filtered, new XAttrEncodingParam(encoder).getEncoding());
    }

    private List<XAttr> getXAttrList(String path) throws IOException {
        long id = this.lookup(path);
        FsImageProto.INodeSection.INode inode = this.fromINodeId(id);
        switch (inode.getType()) {
            case FILE: {
                return FSImageFormatPBINode.Loader.loadXAttrs(inode.getFile().getXAttrs(), this.stringTable);
            }
            case DIRECTORY: {
                return FSImageFormatPBINode.Loader.loadXAttrs(inode.getDirectory().getXAttrs(), this.stringTable);
            }
        }
        return null;
    }

    String getAclStatus(String path) throws IOException {
        PermissionStatus p = this.getPermissionStatus(path);
        List<AclEntry> aclEntryList = this.getAclEntryList(path);
        FsPermission permission = p.getPermission();
        AclStatus.Builder builder = new AclStatus.Builder();
        builder.owner(p.getUserName()).group(p.getGroupName()).addEntries(aclEntryList).setPermission(permission).stickyBit(permission.getStickyBit());
        AclStatus aclStatus = builder.build();
        return JsonUtil.toJsonString(aclStatus);
    }

    private List<AclEntry> getAclEntryList(String path) throws IOException {
        long id = this.lookup(path);
        FsImageProto.INodeSection.INode inode = this.fromINodeId(id);
        switch (inode.getType()) {
            case FILE: {
                FsImageProto.INodeSection.INodeFile f = inode.getFile();
                return FSImageFormatPBINode.Loader.loadAclEntries(f.getAcl(), this.stringTable);
            }
            case DIRECTORY: {
                FsImageProto.INodeSection.INodeDirectory d = inode.getDirectory();
                return FSImageFormatPBINode.Loader.loadAclEntries(d.getAcl(), this.stringTable);
            }
        }
        return new ArrayList<AclEntry>();
    }

    private PermissionStatus getPermissionStatus(String path) throws IOException {
        long id = this.lookup(path);
        FsImageProto.INodeSection.INode inode = this.fromINodeId(id);
        switch (inode.getType()) {
            case FILE: {
                FsImageProto.INodeSection.INodeFile f = inode.getFile();
                return FSImageFormatPBINode.Loader.loadPermission(f.getPermission(), this.stringTable);
            }
            case DIRECTORY: {
                FsImageProto.INodeSection.INodeDirectory d = inode.getDirectory();
                return FSImageFormatPBINode.Loader.loadPermission(d.getPermission(), this.stringTable);
            }
            case SYMLINK: {
                FsImageProto.INodeSection.INodeSymlink s = inode.getSymlink();
                return FSImageFormatPBINode.Loader.loadPermission(s.getPermission(), this.stringTable);
            }
        }
        return null;
    }

    private long lookup(String path) throws IOException {
        Preconditions.checkArgument((boolean)path.startsWith("/"));
        long id = 16385L;
        int offset = 0;
        while (offset < path.length()) {
            int next = path.indexOf(47, offset + 1);
            if (next == -1) {
                next = path.length();
            }
            if (offset + 1 > next) break;
            String component = path.substring(offset + 1, next);
            if (!component.isEmpty()) {
                long[] children = this.dirmap.get(id);
                if (children == null) {
                    throw new FileNotFoundException(path);
                }
                boolean found = false;
                for (long cid : children) {
                    FsImageProto.INodeSection.INode child = this.fromINodeId(cid);
                    if (!component.equals(child.getName().toStringUtf8())) continue;
                    found = true;
                    id = child.getId();
                    break;
                }
                if (!found) {
                    throw new FileNotFoundException(path);
                }
            }
            offset = next;
        }
        return id;
    }

    private Map<String, Object> getFileStatus(FsImageProto.INodeSection.INode inode, boolean printSuffix) {
        HashMap map = Maps.newHashMap();
        switch (inode.getType()) {
            case FILE: {
                FsImageProto.INodeSection.INodeFile f = inode.getFile();
                PermissionStatus p = FSImageFormatPBINode.Loader.loadPermission(f.getPermission(), this.stringTable);
                map.put("accessTime", f.getAccessTime());
                map.put("blockSize", f.getPreferredBlockSize());
                map.put("group", p.getGroupName());
                map.put("length", FSImageLoader.getFileSize(f));
                map.put("modificationTime", f.getModificationTime());
                map.put("owner", p.getUserName());
                map.put("pathSuffix", printSuffix ? inode.getName().toStringUtf8() : "");
                map.put("permission", this.toString(p.getPermission()));
                if (f.hasErasureCodingPolicyID()) {
                    map.put("replication", (short)1);
                } else {
                    map.put("replication", f.getReplication());
                }
                map.put("type", inode.getType());
                map.put("fileId", inode.getId());
                map.put("childrenNum", 0);
                return map;
            }
            case DIRECTORY: {
                FsImageProto.INodeSection.INodeDirectory d = inode.getDirectory();
                PermissionStatus p = FSImageFormatPBINode.Loader.loadPermission(d.getPermission(), this.stringTable);
                map.put("accessTime", 0);
                map.put("blockSize", 0);
                map.put("group", p.getGroupName());
                map.put("length", 0);
                map.put("modificationTime", d.getModificationTime());
                map.put("owner", p.getUserName());
                map.put("pathSuffix", printSuffix ? inode.getName().toStringUtf8() : "");
                map.put("permission", this.toString(p.getPermission()));
                map.put("replication", 0);
                map.put("type", inode.getType());
                map.put("fileId", inode.getId());
                map.put("childrenNum", this.dirmap.containsKey(inode.getId()) ? this.dirmap.get(inode.getId()).length : 0);
                return map;
            }
            case SYMLINK: {
                FsImageProto.INodeSection.INodeSymlink d = inode.getSymlink();
                PermissionStatus p = FSImageFormatPBINode.Loader.loadPermission(d.getPermission(), this.stringTable);
                map.put("accessTime", d.getAccessTime());
                map.put("blockSize", 0);
                map.put("group", p.getGroupName());
                map.put("length", 0);
                map.put("modificationTime", d.getModificationTime());
                map.put("owner", p.getUserName());
                map.put("pathSuffix", printSuffix ? inode.getName().toStringUtf8() : "");
                map.put("permission", this.toString(p.getPermission()));
                map.put("replication", 0);
                map.put("type", inode.getType());
                map.put("symlink", d.getTarget().toStringUtf8());
                map.put("fileId", inode.getId());
                map.put("childrenNum", 0);
                return map;
            }
        }
        return null;
    }

    static long getFileSize(FsImageProto.INodeSection.INodeFile f) {
        long size = 0L;
        for (HdfsProtos.BlockProto p : f.getBlocksList()) {
            size += p.getNumBytes();
        }
        return size;
    }

    private String toString(FsPermission permission) {
        return String.format("%o", permission.toShort());
    }

    private FsImageProto.INodeSection.INode fromINodeId(long id) throws IOException {
        int l = 0;
        int r = this.inodes.length;
        while (l < r) {
            int mid = l + (r - l) / 2;
            FsImageProto.INodeSection.INode n = FsImageProto.INodeSection.INode.parseFrom(this.inodes[mid]);
            long nid = n.getId();
            if (id > nid) {
                l = mid + 1;
                continue;
            }
            if (id < nid) {
                r = mid;
                continue;
            }
            return n;
        }
        return null;
    }
}

