/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.util;

import com.google.common.annotations.VisibleForTesting;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.Timer;
import org.slf4j.Logger;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public class InstrumentedLock
implements Lock {
    private final Lock lock;
    private final Logger logger;
    private final String name;
    private final Timer clock;
    private final long minLoggingGap;
    private final long lockWarningThreshold;
    private volatile long lockAcquireTimestamp;
    private final AtomicLong lastHoldLogTimestamp;
    private final AtomicLong lastWaitLogTimestamp;
    private final SuppressedStats holdStats = new SuppressedStats();
    private final SuppressedStats waitStats = new SuppressedStats();

    public InstrumentedLock(String name, Logger logger, long minLoggingGapMs, long lockWarningThresholdMs) {
        this(name, logger, new ReentrantLock(), minLoggingGapMs, lockWarningThresholdMs);
    }

    public InstrumentedLock(String name, Logger logger, Lock lock, long minLoggingGapMs, long lockWarningThresholdMs) {
        this(name, logger, lock, minLoggingGapMs, lockWarningThresholdMs, new Timer());
    }

    @VisibleForTesting
    InstrumentedLock(String name, Logger logger, Lock lock, long minLoggingGapMs, long lockWarningThresholdMs, Timer clock) {
        this.name = name;
        this.lock = lock;
        this.clock = clock;
        this.logger = logger;
        this.minLoggingGap = minLoggingGapMs;
        this.lockWarningThreshold = lockWarningThresholdMs;
        this.lastHoldLogTimestamp = new AtomicLong(clock.monotonicNow() - Math.max(this.minLoggingGap, this.lockWarningThreshold));
        this.lastWaitLogTimestamp = new AtomicLong(this.lastHoldLogTimestamp.get());
    }

    @Override
    public void lock() {
        long waitStart = this.clock.monotonicNow();
        this.lock.lock();
        this.check(waitStart, this.clock.monotonicNow(), false);
        this.startLockTiming();
    }

    @Override
    public void lockInterruptibly() throws InterruptedException {
        long waitStart = this.clock.monotonicNow();
        this.lock.lockInterruptibly();
        this.check(waitStart, this.clock.monotonicNow(), false);
        this.startLockTiming();
    }

    @Override
    public boolean tryLock() {
        if (this.lock.tryLock()) {
            this.startLockTiming();
            return true;
        }
        return false;
    }

    @Override
    public boolean tryLock(long time, TimeUnit unit) throws InterruptedException {
        long waitStart = this.clock.monotonicNow();
        boolean retval = false;
        if (this.lock.tryLock(time, unit)) {
            this.startLockTiming();
            retval = true;
        }
        this.check(waitStart, this.clock.monotonicNow(), false);
        return retval;
    }

    @Override
    public void unlock() {
        long localLockReleaseTime = this.clock.monotonicNow();
        long localLockAcquireTime = this.lockAcquireTimestamp;
        this.lock.unlock();
        this.check(localLockAcquireTime, localLockReleaseTime, true);
    }

    @Override
    public Condition newCondition() {
        return this.lock.newCondition();
    }

    @VisibleForTesting
    void logWarning(long lockHeldTime, SuppressedSnapshot stats) {
        this.logger.warn(String.format("Lock held time above threshold: lock identifier: %s lockHeldTimeMs=%d ms. Suppressed %d lock warnings. Longest suppressed LockHeldTimeMs=%d. The stack trace is: %s", this.name, lockHeldTime, stats.getSuppressedCount(), stats.getMaxSuppressedWait(), StringUtils.getStackTrace(Thread.currentThread())));
    }

    @VisibleForTesting
    void logWaitWarning(long lockWaitTime, SuppressedSnapshot stats) {
        this.logger.warn(String.format("Waited above threshold to acquire lock: lock identifier: %s waitTimeMs=%d ms. Suppressed %d lock wait warnings. Longest suppressed WaitTimeMs=%d. The stack trace is: %s", this.name, lockWaitTime, stats.getSuppressedCount(), stats.getMaxSuppressedWait(), StringUtils.getStackTrace(Thread.currentThread())));
    }

    protected void startLockTiming() {
        this.lockAcquireTimestamp = this.clock.monotonicNow();
    }

    protected void check(long acquireTime, long releaseTime, boolean checkLockHeld) {
        if (!this.logger.isWarnEnabled()) {
            return;
        }
        long lockHeldTime = releaseTime - acquireTime;
        if (this.lockWarningThreshold - lockHeldTime < 0L) {
            long now;
            long localLastLogTs;
            SuppressedStats stats;
            AtomicLong lastLogTime;
            if (checkLockHeld) {
                lastLogTime = this.lastHoldLogTimestamp;
                stats = this.holdStats;
            } else {
                lastLogTime = this.lastWaitLogTimestamp;
                stats = this.waitStats;
            }
            do {
                long deltaSinceLastLog;
                if ((deltaSinceLastLog = (now = this.clock.monotonicNow()) - (localLastLogTs = lastLogTime.get())) - this.minLoggingGap >= 0L) continue;
                stats.incrementSuppressed(lockHeldTime);
                return;
            } while (!lastLogTime.compareAndSet(localLastLogTs, now));
            SuppressedSnapshot statsSnapshot = stats.snapshot();
            if (checkLockHeld) {
                this.logWarning(lockHeldTime, statsSnapshot);
            } else {
                this.logWaitWarning(lockHeldTime, statsSnapshot);
            }
        }
    }

    protected Lock getLock() {
        return this.lock;
    }

    protected Timer getTimer() {
        return this.clock;
    }

    protected static class SuppressedSnapshot {
        private long suppressedCount = 0L;
        private long maxSuppressedWait = 0L;

        public SuppressedSnapshot(long suppressedCount, long maxWait) {
            this.suppressedCount = suppressedCount;
            this.maxSuppressedWait = maxWait;
        }

        public long getMaxSuppressedWait() {
            return this.maxSuppressedWait;
        }

        public long getSuppressedCount() {
            return this.suppressedCount;
        }
    }

    private static class SuppressedStats {
        private long suppressedCount = 0L;
        private long maxSuppressedWait = 0L;

        private SuppressedStats() {
        }

        public synchronized void incrementSuppressed(long wait) {
            ++this.suppressedCount;
            if (wait > this.maxSuppressedWait) {
                this.maxSuppressedWait = wait;
            }
        }

        public synchronized SuppressedSnapshot snapshot() {
            SuppressedSnapshot snap = new SuppressedSnapshot(this.suppressedCount, this.maxSuppressedWait);
            this.suppressedCount = 0L;
            this.maxSuppressedWait = 0L;
            return snap;
        }
    }
}

