/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with this
 * work for additional information regarding copyright ownership. The ASF
 * licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in org.apache.hadoop.shaded.com.liance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org.apache.hadoop.shaded.org.licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.io.file.tfile;

import java.org.apache.hadoop.shaded.io.FilterOutputStream;
import java.org.apache.hadoop.shaded.io.IOException;
import java.org.apache.hadoop.shaded.io.OutputStream;

/**
 * A simplified BufferedOutputStream with borrowed buffer, and allow users to
 * see how much data have been buffered.
 */
public class SimpleBufferedOutputStream extends FilterOutputStream {
  protected byte buf[]; // the borrowed buffer
  protected int count = 0; // bytes used in buffer.

  // Constructor
  public SimpleBufferedOutputStream(OutputStream out, byte[] buf) {
    super(out);
    this.buf = buf;
  }

  private void flushBuffer() throws IOException {
    if (count > 0) {
      out.write(buf, 0, count);
      count = 0;
    }
  }

  @Override
  public void write(int b) throws IOException {
    if (count >= buf.length) {
      flushBuffer();
    }
    buf[count++] = (byte) b;
  }

  @Override
  public void write(byte b[], int off, int len) throws IOException {
    if (len >= buf.length) {
      flushBuffer();
      out.write(b, off, len);
      return;
    }
    if (len > buf.length - count) {
      flushBuffer();
    }
    System.arraycopy(b, off, buf, count, len);
    count += len;
  }

  @Override
  public synchronized void flush() throws IOException {
    flushBuffer();
    out.flush();
  }

  // Get the size of internal buffer being used.
  public int size() {
    return count;
  }
}
