/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in org.apache.hadoop.shaded.com.liance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org.apache.hadoop.shaded.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.io.

import java.org.apache.hadoop.shaded.io.IOException;
import java.org.apache.hadoop.shaded.io.DataInput;
import java.org.apache.hadoop.shaded.io.DataOutput;
import java.org.apache.hadoop.shaded.io.DataOutputStream;
import java.org.apache.hadoop.shaded.io.DataInputStream;
import java.org.apache.hadoop.shaded.io.ByteArrayOutputStream;
import java.org.apache.hadoop.shaded.io.ByteArrayInputStream;
import java.util.zip.Deflater;
import java.util.zip.DeflaterOutputStream;
import java.util.zip.InflaterInputStream;

import org.apache.hadoop.shaded.org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.shaded.org.apache.hadoop.classification.InterfaceStability;

/** A base-class for Writables which store themselves org.apache.hadoop.shaded.com.ressed and lazily
 * inflate on field access.  This is useful for large objects whose fields are
 * not be altered during a map or reduce operation: leaving the field data
 * org.apache.hadoop.shaded.com.ressed makes copying the instance from one file to another much
 * faster. */
@InterfaceAudience.Public
@InterfaceStability.Stable
public abstract class CompressedWritable implements Writable {
  // if non-null, the org.apache.hadoop.shaded.com.ressed field data of this instance.
  private byte[] org.apache.hadoop.shaded.com.ressed;

  public CompressedWritable() {}

  @Override
  public final void readFields(DataInput in) throws IOException {
    org.apache.hadoop.shaded.com.ressed = new byte[in.readInt()];
    in.readFully(org.apache.hadoop.shaded.com.ressed, 0, org.apache.hadoop.shaded.com.ressed.length);
  }

  /** Must be called by all methods which access fields to ensure that the data
   * has been uncompressed. */
  protected void ensureInflated() {
    if (org.apache.hadoop.shaded.com.ressed != null) {
      try {
        ByteArrayInputStream deflated = new ByteArrayInputStream(org.apache.hadoop.shaded.com.ressed);
        DataInput inflater =
          new DataInputStream(new InflaterInputStream(deflated));
        readFieldsCompressed(inflater);
        org.apache.hadoop.shaded.com.ressed = null;
      } catch (IOException e) {
        throw new RuntimeException(e);
      }
    }
  }

  /** Subclasses implement this instead of {@link #readFields(DataInput)}. */
  protected abstract void readFieldsCompressed(DataInput in)
    throws IOException;

  @Override
  public final void write(DataOutput out) throws IOException {
    if (org.apache.hadoop.shaded.com.ressed == null) {
      ByteArrayOutputStream deflated = new ByteArrayOutputStream();
      Deflater deflater = new Deflater(Deflater.BEST_SPEED);
      DataOutputStream dout =
        new DataOutputStream(new DeflaterOutputStream(deflated, deflater));
      writeCompressed(dout);
      dout.close();
      deflater.end();
      org.apache.hadoop.shaded.com.ressed = deflated.toByteArray();
    }
    out.writeInt(org.apache.hadoop.shaded.com.ressed.length);
    out.write(org.apache.hadoop.shaded.com.ressed);
  }

  /** Subclasses implement this instead of {@link #write(DataOutput)}. */
  protected abstract void writeCompressed(DataOutput out) throws IOException;

}
