/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a.impl;

import com.amazonaws.AmazonClientException;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.transfer.model.CopyResult;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.s3a.RenameFailedException;
import org.apache.hadoop.fs.s3a.S3AFileStatus;
import org.apache.hadoop.fs.s3a.S3ALocatedFileStatus;
import org.apache.hadoop.fs.s3a.S3AReadOpContext;
import org.apache.hadoop.fs.s3a.S3AUtils;
import org.apache.hadoop.fs.s3a.S3ObjectAttributes;
import org.apache.hadoop.fs.s3a.Tristate;
import org.apache.hadoop.fs.s3a.impl.CallableSupplier;
import org.apache.hadoop.fs.s3a.impl.DirMarkerTracker;
import org.apache.hadoop.fs.s3a.impl.ExecutingStoreOperation;
import org.apache.hadoop.fs.s3a.impl.OperationCallbacks;
import org.apache.hadoop.fs.s3a.impl.StoreContext;
import org.apache.hadoop.fs.s3a.s3guard.MetadataStore;
import org.apache.hadoop.fs.s3a.s3guard.RenameTracker;
import org.apache.hadoop.fs.store.audit.AuditSpan;
import org.apache.hadoop.fs.store.audit.AuditingFunctions;
import org.apache.hadoop.thirdparty.com.google.common.base.Preconditions;
import org.apache.hadoop.thirdparty.com.google.common.collect.Lists;
import org.apache.hadoop.util.DurationInfo;
import org.apache.hadoop.util.OperationDuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RenameOperation
extends ExecutingStoreOperation<Long> {
    private static final Logger LOG = LoggerFactory.getLogger(RenameOperation.class);
    private final Path sourcePath;
    private final String sourceKey;
    private final S3AFileStatus sourceStatus;
    private final Path destPath;
    private final String destKey;
    private final S3AFileStatus destStatus;
    private final OperationCallbacks callbacks;
    private final AtomicLong bytesCopied = new AtomicLong();
    private final int pageSize;
    private RenameTracker renameTracker;
    private final List<CompletableFuture<Path>> activeCopies = new ArrayList<CompletableFuture<Path>>(10);
    private final List<DeleteObjectsRequest.KeyVersion> keysToDelete = new ArrayList<DeleteObjectsRequest.KeyVersion>();
    private final List<Path> pathsToDelete = new ArrayList<Path>();
    private final long blocksize;

    public RenameOperation(StoreContext storeContext, Path sourcePath, String sourceKey, S3AFileStatus sourceStatus, Path destPath, String destKey, S3AFileStatus destStatus, OperationCallbacks callbacks, int pageSize) {
        super(storeContext);
        this.sourcePath = sourcePath;
        this.sourceKey = sourceKey;
        this.sourceStatus = sourceStatus;
        this.destPath = destPath;
        this.destKey = destKey;
        this.destStatus = destStatus;
        this.callbacks = callbacks;
        this.blocksize = storeContext.getConfiguration().getLongBytes("fs.s3a.block.size", 0x2000000L);
        this.pageSize = pageSize;
    }

    private void completeActiveCopies(String reason) throws IOException {
        LOG.debug("Waiting for {} active copies to complete: {}", (Object)this.activeCopies.size(), (Object)reason);
        CallableSupplier.waitForCompletion(this.activeCopies);
        this.activeCopies.clear();
    }

    private void queueToDelete(Path path, String key) {
        LOG.debug("Queueing to delete {}", (Object)path);
        if (path != null) {
            this.pathsToDelete.add(path);
        }
        this.keysToDelete.add(new DeleteObjectsRequest.KeyVersion(key));
    }

    private void queueToDelete(List<DirMarkerTracker.Marker> markersToDelete) {
        markersToDelete.forEach(m -> this.queueToDelete(null, m.getKey()));
    }

    private void queueToDelete(DirMarkerTracker.Marker marker) {
        this.queueToDelete(marker.getPath(), marker.getKey());
    }

    private void completeActiveCopiesAndDeleteSources(String reason) throws IOException {
        this.completeActiveCopies(reason);
        this.removeSourceObjects(this.keysToDelete, this.pathsToDelete);
        this.keysToDelete.clear();
        this.pathsToDelete.clear();
    }

    @Override
    public Long execute() throws IOException {
        this.executeOnlyOnce();
        StoreContext storeContext = this.getStoreContext();
        MetadataStore metadataStore = (MetadataStore)Preconditions.checkNotNull((Object)storeContext.getMetadataStore(), (Object)"No metadata store in context");
        this.renameTracker = metadataStore.initiateRenameOperation(storeContext, this.sourcePath, this.sourceStatus, this.destPath);
        Path destCreated = this.destPath;
        try {
            if (this.sourceStatus.isFile()) {
                destCreated = this.renameFileToDest();
            } else {
                this.recursiveDirectoryRename();
            }
        }
        catch (AmazonClientException | IOException ex) {
            try {
                this.completeActiveCopies("failure handling");
            }
            catch (IOException e) {
                LOG.warn("While completing all active copies", (Throwable)e);
            }
            throw this.renameTracker.renameFailed((Exception)ex);
        }
        this.renameTracker.completeRename();
        this.callbacks.finishRename(this.sourcePath, destCreated);
        return this.bytesCopied.get();
    }

    protected Path renameFileToDest() throws IOException {
        StoreContext storeContext = this.getStoreContext();
        Path copyDestinationPath = this.destPath;
        String copyDestinationKey = this.destKey;
        S3ObjectAttributes sourceAttributes = this.callbacks.createObjectAttributes(this.sourceStatus);
        S3AReadOpContext readContext = this.callbacks.createReadContext(this.sourceStatus);
        if (this.destStatus != null && this.destStatus.isDirectory()) {
            String newDestKey = this.maybeAddTrailingSlash(this.destKey);
            String filename = this.sourceKey.substring(storeContext.pathToKey(this.sourcePath.getParent()).length() + 1);
            copyDestinationKey = newDestKey = newDestKey + filename;
            copyDestinationPath = storeContext.keyToPath(newDestKey);
        }
        LOG.debug("rename: renaming file {} to {}", (Object)this.sourcePath, (Object)copyDestinationPath);
        this.copySourceAndUpdateTracker(this.sourcePath, this.sourceKey, sourceAttributes, readContext, copyDestinationPath, copyDestinationKey, false);
        this.bytesCopied.addAndGet(this.sourceStatus.getLen());
        this.callbacks.deleteObjectAtPath(this.sourcePath, this.sourceKey, true, null);
        this.renameTracker.sourceObjectsDeleted(Lists.newArrayList((Object[])new Path[]{this.sourcePath}));
        return copyDestinationPath;
    }

    protected void recursiveDirectoryRename() throws IOException {
        StoreContext storeContext = this.getStoreContext();
        LOG.debug("rename: renaming directory {} to {}", (Object)this.sourcePath, (Object)this.destPath);
        String dstKey = this.maybeAddTrailingSlash(this.destKey);
        String srcKey = this.maybeAddTrailingSlash(this.sourceKey);
        if (dstKey.startsWith(srcKey)) {
            throw new RenameFailedException(srcKey, dstKey, "cannot rename a directory to a subdirectory of itself ");
        }
        if (this.destStatus != null && this.destStatus.isEmptyDirectory() == Tristate.TRUE) {
            LOG.debug("Deleting fake directory marker at destination {}", (Object)this.destStatus.getPath());
            this.callbacks.deleteObjectAtPath(this.destStatus.getPath(), dstKey, false, null);
        }
        Path parentPath = storeContext.keyToPath(srcKey);
        DirMarkerTracker dirMarkerTracker = new DirMarkerTracker(parentPath, false);
        RemoteIterator<S3ALocatedFileStatus> iterator = this.callbacks.listFilesAndDirectoryMarkers(parentPath, this.sourceStatus, true, true);
        while (iterator.hasNext()) {
            List<DirMarkerTracker.Marker> markersToDelete;
            S3ALocatedFileStatus child = (S3ALocatedFileStatus)((Object)iterator.next());
            LOG.debug("To rename {}", (Object)child);
            String k = storeContext.pathToKey(child.getPath());
            String key = child.isDirectory() && !k.endsWith("/") ? k + "/" : k;
            Path childSourcePath = storeContext.keyToPath(key);
            boolean isMarker = key.endsWith("/");
            if (isMarker) {
                markersToDelete = dirMarkerTracker.markerFound(childSourcePath, key, child);
            } else {
                markersToDelete = dirMarkerTracker.fileFound(childSourcePath, key, child);
                String newDestKey = dstKey + key.substring(srcKey.length());
                Path childDestPath = storeContext.keyToPath(newDestKey);
                this.queueToDelete(childSourcePath, key);
                CompletableFuture<Path> copy = this.initiateCopy(child, key, childSourcePath, newDestKey, childDestPath);
                this.activeCopies.add(copy);
                this.bytesCopied.addAndGet(this.sourceStatus.getLen());
            }
            this.queueToDelete(markersToDelete);
            this.endOfLoopActions();
        }
        this.copyEmptyDirectoryMarkers(srcKey, dstKey, dirMarkerTracker);
        this.completeActiveCopiesAndDeleteSources("final copy and delete");
        this.renameTracker.moveSourceDirectory();
    }

    private void endOfLoopActions() throws IOException {
        if (this.keysToDelete.size() == this.pageSize) {
            this.completeActiveCopiesAndDeleteSources("paged delete");
        } else if (this.activeCopies.size() == 10) {
            LOG.debug("Waiting for active copies to complete");
            this.completeActiveCopies("batch threshold reached");
        }
    }

    private OperationDuration copyEmptyDirectoryMarkers(String srcKey, String dstKey, DirMarkerTracker dirMarkerTracker) throws IOException {
        LOG.debug("Copying markers from {}", (Object)dirMarkerTracker);
        StoreContext storeContext = this.getStoreContext();
        Map<Path, DirMarkerTracker.Marker> leafMarkers = dirMarkerTracker.getLeafMarkers();
        Map<Path, DirMarkerTracker.Marker> surplus = dirMarkerTracker.getSurplusMarkers();
        DurationInfo duration = new DurationInfo(LOG, false, "copying %d leaf markers with %d surplus not copied", new Object[]{leafMarkers.size(), surplus.size()});
        for (DirMarkerTracker.Marker entry : leafMarkers.values()) {
            Path source = entry.getPath();
            String key = entry.getKey();
            String newDestKey = dstKey + key.substring(srcKey.length());
            Path childDestPath = storeContext.keyToPath(newDestKey);
            LOG.debug("copying dir marker from {} to {}", (Object)key, (Object)newDestKey);
            this.activeCopies.add(this.initiateCopy(entry.getStatus(), key, source, newDestKey, childDestPath));
            this.queueToDelete(entry);
            this.endOfLoopActions();
        }
        duration.close();
        return duration;
    }

    protected CompletableFuture<Path> initiateCopy(S3ALocatedFileStatus source, String key, Path childSourcePath, String newDestKey, Path childDestPath) {
        S3ObjectAttributes sourceAttributes = this.callbacks.createObjectAttributes(source.getPath(), source.getEtag(), source.getVersionId(), source.getLen());
        return CallableSupplier.submit(this.getStoreContext().getExecutor(), AuditingFunctions.callableWithinAuditSpan((AuditSpan)this.getAuditSpan(), () -> this.copySourceAndUpdateTracker(childSourcePath, key, sourceAttributes, this.callbacks.createReadContext((FileStatus)source), childDestPath, newDestKey, true)));
    }

    private Path copySourceAndUpdateTracker(Path sourceFile, String srcKey, S3ObjectAttributes srcAttributes, S3AReadOpContext readContext, Path destination, String destinationKey, boolean addAncestors) throws IOException {
        CopyResult copyResult;
        long len = srcAttributes.getLen();
        try (DurationInfo ignored = new DurationInfo(LOG, false, "Copy file from %s to %s (length=%d)", new Object[]{srcKey, destinationKey, len});){
            copyResult = this.callbacks.copyFile(srcKey, destinationKey, srcAttributes, readContext);
        }
        if (S3AUtils.objectRepresentsDirectory(srcKey)) {
            this.renameTracker.directoryMarkerCopied(sourceFile, destination, addAncestors);
        } else {
            S3ObjectAttributes destAttributes = new S3ObjectAttributes(destination, copyResult, srcAttributes.getServerSideEncryptionAlgorithm(), srcAttributes.getServerSideEncryptionKey(), len);
            this.renameTracker.fileCopied(sourceFile, srcAttributes, destAttributes, destination, this.blocksize, addAncestors);
        }
        return destination;
    }

    private void removeSourceObjects(List<DeleteObjectsRequest.KeyVersion> keys, List<Path> paths) throws IOException {
        ArrayList<Path> undeletedObjects = new ArrayList<Path>();
        try {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Initiating delete operation for {} objects", (Object)keys.size());
                for (DeleteObjectsRequest.KeyVersion key : keys) {
                    LOG.debug(" {} {}", (Object)key.getKey(), (Object)(key.getVersion() != null ? key.getVersion() : ""));
                }
            }
            this.callbacks.removeKeys(keys, false, undeletedObjects, this.renameTracker.getOperationState(), true);
        }
        catch (AmazonClientException | IOException e) {
            throw this.renameTracker.deleteFailed((Exception)e, paths, undeletedObjects);
        }
        this.renameTracker.sourceObjectsDeleted(paths);
    }

    private String maybeAddTrailingSlash(String key) {
        if (!key.isEmpty() && !key.endsWith("/")) {
            return key + '/';
        }
        return key;
    }
}

