/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a;

import com.amazonaws.services.s3.model.CompleteMultipartUploadResult;
import com.amazonaws.services.s3.model.PartETag;
import com.amazonaws.services.s3.model.UploadPartRequest;
import com.amazonaws.services.s3.model.UploadPartResult;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BBPartHandle;
import org.apache.hadoop.fs.BBUploadHandle;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.MultipartUploader;
import org.apache.hadoop.fs.MultipartUploaderFactory;
import org.apache.hadoop.fs.PartHandle;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathHandle;
import org.apache.hadoop.fs.UploadHandle;
import org.apache.hadoop.fs.s3a.S3AFileSystem;
import org.apache.hadoop.fs.s3a.WriteOperationHelper;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public class S3AMultipartUploader
extends MultipartUploader {
    private final S3AFileSystem s3a;
    public static final String HEADER = "S3A-part01";

    public S3AMultipartUploader(FileSystem fs, Configuration conf) {
        Preconditions.checkArgument((boolean)(fs instanceof S3AFileSystem), (String)"Wrong filesystem: expected S3A but got %s", (Object[])new Object[]{fs});
        this.s3a = (S3AFileSystem)fs;
    }

    public UploadHandle initialize(Path filePath) throws IOException {
        WriteOperationHelper writeHelper = this.s3a.getWriteOperationHelper();
        String key = this.s3a.pathToKey(filePath);
        String uploadId = writeHelper.initiateMultiPartUpload(key);
        return BBUploadHandle.from((ByteBuffer)ByteBuffer.wrap(uploadId.getBytes(Charsets.UTF_8)));
    }

    public PartHandle putPart(Path filePath, InputStream inputStream, int partNumber, UploadHandle uploadId, long lengthInBytes) throws IOException {
        this.checkPutArguments(filePath, inputStream, partNumber, uploadId, lengthInBytes);
        byte[] uploadIdBytes = uploadId.toByteArray();
        this.checkUploadId(uploadIdBytes);
        String key = this.s3a.pathToKey(filePath);
        WriteOperationHelper writeHelper = this.s3a.getWriteOperationHelper();
        String uploadIdString = new String(uploadIdBytes, 0, uploadIdBytes.length, Charsets.UTF_8);
        UploadPartRequest request = writeHelper.newUploadPartRequest(key, uploadIdString, partNumber, (int)lengthInBytes, inputStream, null, 0L);
        UploadPartResult result = writeHelper.uploadPart(request);
        String eTag = result.getETag();
        return BBPartHandle.from((ByteBuffer)ByteBuffer.wrap(S3AMultipartUploader.buildPartHandlePayload(eTag, lengthInBytes)));
    }

    public PathHandle complete(Path filePath, Map<Integer, PartHandle> handleMap, UploadHandle uploadId) throws IOException {
        byte[] uploadIdBytes = uploadId.toByteArray();
        this.checkUploadId(uploadIdBytes);
        this.checkPartHandles(handleMap);
        ArrayList<Map.Entry<Integer, PartHandle>> handles = new ArrayList<Map.Entry<Integer, PartHandle>>(handleMap.entrySet());
        handles.sort(Comparator.comparingInt(Map.Entry::getKey));
        WriteOperationHelper writeHelper = this.s3a.getWriteOperationHelper();
        String key = this.s3a.pathToKey(filePath);
        String uploadIdStr = new String(uploadIdBytes, 0, uploadIdBytes.length, Charsets.UTF_8);
        ArrayList<PartETag> eTags = new ArrayList<PartETag>();
        eTags.ensureCapacity(handles.size());
        long totalLength = 0L;
        for (Map.Entry entry : handles) {
            byte[] payload = ((PartHandle)entry.getValue()).toByteArray();
            Pair<Long, String> result = S3AMultipartUploader.parsePartHandlePayload(payload);
            totalLength += ((Long)result.getLeft()).longValue();
            eTags.add(new PartETag(((Integer)entry.getKey()).intValue(), (String)result.getRight()));
        }
        AtomicInteger errorCount = new AtomicInteger(0);
        CompleteMultipartUploadResult completeMultipartUploadResult = writeHelper.completeMPUwithRetries(key, uploadIdStr, eTags, totalLength, errorCount);
        byte[] eTag = completeMultipartUploadResult.getETag().getBytes(Charsets.UTF_8);
        return (PathHandle & Serializable)() -> ByteBuffer.wrap(eTag);
    }

    public void abort(Path filePath, UploadHandle uploadId) throws IOException {
        byte[] uploadIdBytes = uploadId.toByteArray();
        this.checkUploadId(uploadIdBytes);
        WriteOperationHelper writeHelper = this.s3a.getWriteOperationHelper();
        String key = this.s3a.pathToKey(filePath);
        String uploadIdString = new String(uploadIdBytes, 0, uploadIdBytes.length, Charsets.UTF_8);
        writeHelper.abortMultipartCommit(key, uploadIdString);
    }

    @VisibleForTesting
    static byte[] buildPartHandlePayload(String eTag, long len) throws IOException {
        Preconditions.checkArgument((boolean)StringUtils.isNotEmpty((CharSequence)eTag), (Object)"Empty etag");
        Preconditions.checkArgument((len >= 0L ? 1 : 0) != 0, (Object)"Invalid length");
        ByteArrayOutputStream bytes = new ByteArrayOutputStream();
        try (DataOutputStream output = new DataOutputStream(bytes);){
            output.writeUTF(HEADER);
            output.writeLong(len);
            output.writeUTF(eTag);
        }
        return bytes.toByteArray();
    }

    @VisibleForTesting
    static Pair<Long, String> parsePartHandlePayload(byte[] data) throws IOException {
        try (DataInputStream input = new DataInputStream(new ByteArrayInputStream(data));){
            String header = input.readUTF();
            if (!HEADER.equals(header)) {
                throw new IOException("Wrong header string: \"" + header + "\"");
            }
            long len = input.readLong();
            String etag = input.readUTF();
            if (len < 0L) {
                throw new IOException("Negative length");
            }
            Pair pair = Pair.of((Object)len, (Object)etag);
            return pair;
        }
    }

    public static class Factory
    extends MultipartUploaderFactory {
        protected MultipartUploader createMultipartUploader(FileSystem fs, Configuration conf) {
            if ("s3a".equals(fs.getScheme())) {
                return new S3AMultipartUploader(fs, conf);
            }
            return null;
        }
    }
}

