/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.connector.source.lookup.cache.trigger;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalTime;
import java.time.OffsetTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoUnit;
import java.time.temporal.Temporal;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.ReadableConfig;
import org.apache.flink.core.testutils.ScheduledTask;
import org.apache.flink.table.connector.source.lookup.LookupOptions;
import org.apache.flink.table.connector.source.lookup.cache.trigger.CacheReloadTrigger;
import org.apache.flink.table.connector.source.lookup.cache.trigger.ScheduleStrategyExecutorService;
import org.apache.flink.table.connector.source.lookup.cache.trigger.TestTriggerContext;
import org.apache.flink.table.connector.source.lookup.cache.trigger.TimedCacheReloadTrigger;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

class TimedCacheReloadTriggerTest {
    private static final long MILLIS_DIFF = 2L;
    private static final int RELOAD_INTERVAL_IN_DAYS = 7;
    private static final Clock CONSTANT_CLOCK = Clock.fixed(Instant.ofEpochMilli(10L), ZoneId.systemDefault());
    private final ScheduleStrategyExecutorService scheduledExecutor = new ScheduleStrategyExecutorService();
    private final TestTriggerContext context = new TestTriggerContext();

    TimedCacheReloadTriggerTest() {
    }

    @ParameterizedTest
    @MethodSource(value={"normalReloadTimes"})
    void testNormalReloadTime(Temporal reloadTime) throws Exception {
        try (TimedCacheReloadTrigger trigger = new TimedCacheReloadTrigger(reloadTime, 7, (ScheduledExecutorService)((Object)this.scheduledExecutor), CONSTANT_CLOCK);){
            trigger.open((CacheReloadTrigger.Context)this.context);
            Assertions.assertThat((int)this.scheduledExecutor.numQueuedRunnables()).isEqualTo(1);
            Assertions.assertThat((int)this.scheduledExecutor.getNumPeriodicTasksWithFixedDelay()).isEqualTo(0);
            Assertions.assertThat((int)this.scheduledExecutor.getNumPeriodicTasksWithFixedRate()).isEqualTo(1);
            ScheduledTask scheduledLoadTask = (ScheduledTask)this.scheduledExecutor.getAllPeriodicScheduledTask().iterator().next();
            Assertions.assertThat((long)scheduledLoadTask.getDelay(TimeUnit.MILLISECONDS)).isEqualTo(2L);
            Assertions.assertThat((long)scheduledLoadTask.getPeriod()).isEqualTo(Duration.ofDays(7L).toMillis());
            this.scheduledExecutor.trigger();
            Assertions.assertThat((int)this.context.getReloadTask().getNumLoads()).isEqualTo(1);
            this.scheduledExecutor.triggerPeriodicScheduledTasks();
            Assertions.assertThat((int)this.context.getReloadTask().getNumLoads()).isEqualTo(2);
        }
        Assertions.assertThat((boolean)this.scheduledExecutor.isTerminated()).isTrue();
    }

    @ParameterizedTest
    @MethodSource(value={"nextDayReloadTimes"})
    void testNextDayReloadTime(Temporal reloadTime) throws Exception {
        try (TimedCacheReloadTrigger trigger = new TimedCacheReloadTrigger(reloadTime, 7, (ScheduledExecutorService)((Object)this.scheduledExecutor), CONSTANT_CLOCK);){
            trigger.open((CacheReloadTrigger.Context)this.context);
            Assertions.assertThat((int)this.scheduledExecutor.numQueuedRunnables()).isEqualTo(1);
            Assertions.assertThat((int)this.scheduledExecutor.getNumPeriodicTasksWithFixedDelay()).isEqualTo(0);
            Assertions.assertThat((int)this.scheduledExecutor.getNumPeriodicTasksWithFixedRate()).isEqualTo(1);
            ScheduledTask scheduledLoadTask = (ScheduledTask)this.scheduledExecutor.getAllPeriodicScheduledTask().iterator().next();
            Assertions.assertThat((long)scheduledLoadTask.getDelay(TimeUnit.MILLISECONDS)).isEqualTo(Duration.ofDays(1L).minus(2L, ChronoUnit.MILLIS).toMillis());
            Assertions.assertThat((long)scheduledLoadTask.getPeriod()).isEqualTo(Duration.ofDays(7L).toMillis());
            this.scheduledExecutor.trigger();
            Assertions.assertThat((int)this.context.getReloadTask().getNumLoads()).isEqualTo(1);
            this.scheduledExecutor.triggerPeriodicScheduledTasks();
            Assertions.assertThat((int)this.context.getReloadTask().getNumLoads()).isEqualTo(2);
        }
        Assertions.assertThat((boolean)this.scheduledExecutor.isTerminated()).isTrue();
    }

    @Test
    void testBadReloadIntervalInDays() {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> new TimedCacheReloadTrigger(LocalTime.now(), 0)).isInstanceOf(IllegalArgumentException.class)).hasMessageContaining("at least 1 day");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> new TimedCacheReloadTrigger(LocalTime.now(), -1)).isInstanceOf(IllegalArgumentException.class)).hasMessageContaining("at least 1 day");
    }

    @ParameterizedTest
    @MethodSource(value={"reloadTimeStrings"})
    void testParseReloadTimeFromConf(String reloadTimeStr, Temporal reloadTime) {
        Configuration configuration = TimedCacheReloadTriggerTest.createValidConf();
        configuration.set(LookupOptions.FULL_CACHE_TIMED_RELOAD_ISO_TIME, (Object)reloadTimeStr);
        TimedCacheReloadTrigger trigger = TimedCacheReloadTrigger.fromConfig((ReadableConfig)configuration);
        Assertions.assertThat((Object)trigger.getReloadTime()).isEqualTo((Object)reloadTime);
    }

    @Test
    void testCreateFromConfig() {
        Assertions.assertThat((Object)TimedCacheReloadTrigger.fromConfig((ReadableConfig)TimedCacheReloadTriggerTest.createValidConf())).isNotNull();
        Configuration conf1 = TimedCacheReloadTriggerTest.createValidConf().set(LookupOptions.CACHE_TYPE, (Object)LookupOptions.LookupCacheType.PARTIAL);
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> TimedCacheReloadTrigger.fromConfig((ReadableConfig)conf1)).isInstanceOf(IllegalArgumentException.class)).hasMessageContaining("should be 'FULL'");
        Configuration conf2 = TimedCacheReloadTriggerTest.createValidConf().set(LookupOptions.FULL_CACHE_RELOAD_STRATEGY, (Object)LookupOptions.ReloadStrategy.PERIODIC);
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> TimedCacheReloadTrigger.fromConfig((ReadableConfig)conf2)).isInstanceOf(IllegalArgumentException.class)).hasMessageContaining("should be 'TIMED'");
        Configuration conf3 = TimedCacheReloadTriggerTest.createValidConf();
        conf3.removeConfig(LookupOptions.FULL_CACHE_TIMED_RELOAD_ISO_TIME);
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> TimedCacheReloadTrigger.fromConfig((ReadableConfig)conf3)).isInstanceOf(IllegalArgumentException.class)).hasMessageContaining("Missing '" + LookupOptions.FULL_CACHE_TIMED_RELOAD_ISO_TIME.key() + "'");
        Configuration conf4 = TimedCacheReloadTriggerTest.createValidConf().set(LookupOptions.FULL_CACHE_TIMED_RELOAD_ISO_TIME, (Object)"10");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> TimedCacheReloadTrigger.fromConfig((ReadableConfig)conf4)).isInstanceOf(DateTimeParseException.class)).hasMessageContaining("could not be parsed");
    }

    static Stream<Arguments> normalReloadTimes() {
        return Stream.of(Arguments.of((Object[])new Object[]{OffsetTime.now(CONSTANT_CLOCK).plus(2L, ChronoUnit.MILLIS)}), Arguments.of((Object[])new Object[]{LocalTime.now(CONSTANT_CLOCK).plus(2L, ChronoUnit.MILLIS)}));
    }

    static Stream<Arguments> nextDayReloadTimes() {
        return Stream.of(Arguments.of((Object[])new Object[]{OffsetTime.now(CONSTANT_CLOCK).minus(2L, ChronoUnit.MILLIS)}), Arguments.of((Object[])new Object[]{LocalTime.now(CONSTANT_CLOCK).minus(2L, ChronoUnit.MILLIS)}));
    }

    static Stream<Arguments> reloadTimeStrings() {
        LocalTime localTime = LocalTime.of(10, 15);
        return Stream.of(Arguments.of((Object[])new Object[]{"10:15", localTime}), Arguments.of((Object[])new Object[]{"10:15Z", OffsetTime.of(localTime, ZoneOffset.UTC)}), Arguments.of((Object[])new Object[]{"10:15+07:00", OffsetTime.of(localTime, ZoneOffset.ofHours(7))}));
    }

    private static Configuration createValidConf() {
        Configuration configuration = new Configuration();
        configuration.set(LookupOptions.CACHE_TYPE, (Object)LookupOptions.LookupCacheType.FULL);
        configuration.set(LookupOptions.FULL_CACHE_RELOAD_STRATEGY, (Object)LookupOptions.ReloadStrategy.TIMED);
        configuration.set(LookupOptions.FULL_CACHE_TIMED_RELOAD_ISO_TIME, (Object)"10:15");
        return configuration;
    }
}

