/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.memory;

import java.nio.ByteBuffer;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Set;
import org.apache.flink.core.memory.HybridMemorySegment;
import org.apache.flink.core.memory.MemorySegment;
import org.apache.flink.core.memory.MemorySegmentFactory;
import org.apache.flink.core.memory.MemoryType;
import org.apache.flink.runtime.memory.MemoryAllocationException;
import org.apache.flink.util.MathUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MemoryManager {
    private static final Logger LOG = LoggerFactory.getLogger(MemoryManager.class);
    public static final int DEFAULT_PAGE_SIZE = 32768;
    public static final int MIN_PAGE_SIZE = 4096;
    private final Object lock = new Object();
    private final MemoryPool memoryPool;
    private final HashMap<Object, Set<MemorySegment>> allocatedSegments;
    private final MemoryType memoryType;
    private final long roundingMask;
    private final int pageSize;
    private final int totalNumPages;
    private final long memorySize;
    private final int numberOfSlots;
    private final boolean isPreAllocated;
    private int numNonAllocatedPages;
    private boolean isShutDown;

    public MemoryManager(long memorySize, int numberOfSlots) {
        this(memorySize, numberOfSlots, 32768, MemoryType.HEAP, true);
    }

    public MemoryManager(long memorySize, int numberOfSlots, int pageSize, MemoryType memoryType, boolean preAllocateMemory) {
        if (memoryType == null) {
            throw new NullPointerException();
        }
        if (memorySize <= 0L) {
            throw new IllegalArgumentException("Size of total memory must be positive.");
        }
        if (pageSize < 4096) {
            throw new IllegalArgumentException("The page size must be at least 4096 bytes.");
        }
        if (!MathUtils.isPowerOf2((long)pageSize)) {
            throw new IllegalArgumentException("The given page size is not a power of two.");
        }
        this.memoryType = memoryType;
        this.memorySize = memorySize;
        this.numberOfSlots = numberOfSlots;
        this.pageSize = pageSize;
        this.roundingMask = (long)(pageSize - 1) ^ 0xFFFFFFFFFFFFFFFFL;
        long numPagesLong = memorySize / (long)pageSize;
        if (numPagesLong > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("The given number of memory bytes (" + memorySize + ") corresponds to more than MAX_INT pages.");
        }
        this.totalNumPages = (int)numPagesLong;
        if (this.totalNumPages < 1) {
            throw new IllegalArgumentException("The given amount of memory amounted to less than one page.");
        }
        this.allocatedSegments = new HashMap();
        this.isPreAllocated = preAllocateMemory;
        this.numNonAllocatedPages = preAllocateMemory ? 0 : this.totalNumPages;
        int memToAllocate = preAllocateMemory ? this.totalNumPages : 0;
        switch (memoryType) {
            case HEAP: {
                this.memoryPool = new HybridHeapMemoryPool(memToAllocate, pageSize);
                break;
            }
            case OFF_HEAP: {
                if (!preAllocateMemory) {
                    LOG.warn("It is advisable to set 'taskmanager.memory.preallocate' to true when the memory type 'taskmanager.memory.off-heap' is set to true.");
                }
                this.memoryPool = new HybridOffHeapMemoryPool(memToAllocate, pageSize);
                break;
            }
            default: {
                throw new IllegalArgumentException("unrecognized memory type: " + memoryType);
            }
        }
        LOG.debug("Initialized MemoryManager with total memory size {}, number of slots {}, page size {}, memory type {}, pre allocate memory {} and number of non allocated pages {}.", new Object[]{memorySize, numberOfSlots, pageSize, memoryType, preAllocateMemory, this.numNonAllocatedPages});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown() {
        Object object = this.lock;
        synchronized (object) {
            if (!this.isShutDown) {
                this.isShutDown = true;
                this.numNonAllocatedPages = 0;
                for (Set<MemorySegment> segments : this.allocatedSegments.values()) {
                    for (MemorySegment seg : segments) {
                        seg.free();
                    }
                }
                this.memoryPool.clear();
            }
        }
    }

    public boolean isShutdown() {
        return this.isShutDown;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean verifyEmpty() {
        Object object = this.lock;
        synchronized (object) {
            return this.isPreAllocated ? this.memoryPool.getNumberOfAvailableMemorySegments() == this.totalNumPages : this.numNonAllocatedPages == this.totalNumPages;
        }
    }

    public List<MemorySegment> allocatePages(Object owner, int numPages) throws MemoryAllocationException {
        ArrayList<MemorySegment> segs = new ArrayList<MemorySegment>(numPages);
        this.allocatePages(owner, segs, numPages);
        return segs;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void allocatePages(Object owner, List<MemorySegment> target, int numPages) throws MemoryAllocationException {
        if (owner == null) {
            throw new IllegalArgumentException("The memory owner must not be null.");
        }
        if (target instanceof ArrayList) {
            ((ArrayList)target).ensureCapacity(numPages);
        }
        Object object = this.lock;
        synchronized (object) {
            if (this.isShutDown) {
                throw new IllegalStateException("Memory manager has been shut down.");
            }
            if (numPages > this.memoryPool.getNumberOfAvailableMemorySegments() + this.numNonAllocatedPages) {
                throw new MemoryAllocationException("Could not allocate " + numPages + " pages. Only " + (this.memoryPool.getNumberOfAvailableMemorySegments() + this.numNonAllocatedPages) + " pages are remaining.");
            }
            Set<MemorySegment> segmentsForOwner = this.allocatedSegments.get(owner);
            if (segmentsForOwner == null) {
                segmentsForOwner = new HashSet<MemorySegment>(numPages);
                this.allocatedSegments.put(owner, segmentsForOwner);
            }
            if (this.isPreAllocated) {
                for (int i = numPages; i > 0; --i) {
                    MemorySegment segment = this.memoryPool.requestSegmentFromPool(owner);
                    target.add(segment);
                    segmentsForOwner.add(segment);
                }
            } else {
                for (int i = numPages; i > 0; --i) {
                    MemorySegment segment = this.memoryPool.allocateNewSegment(owner);
                    target.add(segment);
                    segmentsForOwner.add(segment);
                }
                this.numNonAllocatedPages -= numPages;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void release(MemorySegment segment) {
        if (segment == null || segment.getOwner() == null) {
            return;
        }
        Object owner = segment.getOwner();
        Object object = this.lock;
        synchronized (object) {
            if (segment.isFreed()) {
                return;
            }
            if (this.isShutDown) {
                throw new IllegalStateException("Memory manager has been shut down.");
            }
            try {
                Set<MemorySegment> segsForOwner = this.allocatedSegments.get(owner);
                if (segsForOwner != null) {
                    segsForOwner.remove(segment);
                    if (segsForOwner.isEmpty()) {
                        this.allocatedSegments.remove(owner);
                    }
                }
                if (this.isPreAllocated) {
                    this.memoryPool.returnSegmentToPool(segment);
                } else {
                    segment.free();
                    ++this.numNonAllocatedPages;
                }
            }
            catch (Throwable t) {
                throw new RuntimeException("Error removing book-keeping reference to allocated memory segment.", t);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void release(Collection<MemorySegment> segments) {
        if (segments == null) {
            return;
        }
        Object object = this.lock;
        synchronized (object) {
            if (this.isShutDown) {
                throw new IllegalStateException("Memory manager has been shut down.");
            }
            boolean successfullyReleased = false;
            do {
                Iterator<MemorySegment> segmentsIterator = segments.iterator();
                Object lastOwner = null;
                Set<MemorySegment> segsForOwner = null;
                try {
                    while (segmentsIterator.hasNext()) {
                        MemorySegment seg = segmentsIterator.next();
                        if (seg == null || seg.isFreed()) continue;
                        Object owner = seg.getOwner();
                        try {
                            if (lastOwner != owner) {
                                lastOwner = owner;
                                segsForOwner = this.allocatedSegments.get(owner);
                            }
                            if (segsForOwner != null) {
                                segsForOwner.remove(seg);
                                if (segsForOwner.isEmpty()) {
                                    this.allocatedSegments.remove(owner);
                                }
                            }
                            if (this.isPreAllocated) {
                                this.memoryPool.returnSegmentToPool(seg);
                                continue;
                            }
                            seg.free();
                            ++this.numNonAllocatedPages;
                        }
                        catch (Throwable t) {
                            throw new RuntimeException("Error removing book-keeping reference to allocated memory segment.", t);
                        }
                    }
                    segments.clear();
                    return;
                }
                catch (ConcurrentModificationException | NoSuchElementException runtimeException) {
                    // empty catch block
                }
            } while (!successfullyReleased);
            return;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void releaseAll(Object owner) {
        if (owner == null) {
            return;
        }
        Object object = this.lock;
        synchronized (object) {
            if (this.isShutDown) {
                throw new IllegalStateException("Memory manager has been shut down.");
            }
            Set<MemorySegment> segments = this.allocatedSegments.remove(owner);
            if (segments == null || segments.isEmpty()) {
                return;
            }
            if (this.isPreAllocated) {
                for (MemorySegment seg : segments) {
                    this.memoryPool.returnSegmentToPool(seg);
                }
            } else {
                for (MemorySegment seg : segments) {
                    seg.free();
                }
                this.numNonAllocatedPages += segments.size();
            }
            segments.clear();
        }
    }

    public MemoryType getMemoryType() {
        return this.memoryType;
    }

    public boolean isPreAllocated() {
        return this.isPreAllocated;
    }

    public int getPageSize() {
        return this.pageSize;
    }

    public long getMemorySize() {
        return this.memorySize;
    }

    public int getTotalNumPages() {
        return this.totalNumPages;
    }

    public int computeNumberOfPages(double fraction) {
        if (fraction <= 0.0 || fraction > 1.0) {
            throw new IllegalArgumentException("The fraction of memory to allocate must within (0, 1].");
        }
        return (int)((double)this.totalNumPages * fraction / (double)this.numberOfSlots);
    }

    public long computeMemorySize(double fraction) {
        return (long)this.pageSize * (long)this.computeNumberOfPages(fraction);
    }

    public long roundDownToPageSizeMultiple(long numBytes) {
        return numBytes & this.roundingMask;
    }

    static final class HybridOffHeapMemoryPool
    extends MemoryPool {
        private final ArrayDeque<ByteBuffer> availableMemory;
        private final int segmentSize;

        HybridOffHeapMemoryPool(int numInitialSegments, int segmentSize) {
            this.availableMemory = new ArrayDeque(numInitialSegments);
            this.segmentSize = segmentSize;
            for (int i = 0; i < numInitialSegments; ++i) {
                this.availableMemory.add(ByteBuffer.allocateDirect(segmentSize));
            }
        }

        @Override
        MemorySegment allocateNewSegment(Object owner) {
            ByteBuffer memory = ByteBuffer.allocateDirect(this.segmentSize);
            return MemorySegmentFactory.wrapPooledOffHeapMemory((ByteBuffer)memory, (Object)owner);
        }

        @Override
        MemorySegment requestSegmentFromPool(Object owner) {
            ByteBuffer buf = this.availableMemory.remove();
            return MemorySegmentFactory.wrapPooledOffHeapMemory((ByteBuffer)buf, (Object)owner);
        }

        @Override
        void returnSegmentToPool(MemorySegment segment) {
            if (segment.getClass() != HybridMemorySegment.class) {
                throw new IllegalArgumentException("Memory segment is not a " + HybridMemorySegment.class.getSimpleName());
            }
            HybridMemorySegment hybridSegment = (HybridMemorySegment)segment;
            ByteBuffer buf = hybridSegment.getOffHeapBuffer();
            this.availableMemory.add(buf);
            hybridSegment.free();
        }

        @Override
        protected int getNumberOfAvailableMemorySegments() {
            return this.availableMemory.size();
        }

        @Override
        void clear() {
            this.availableMemory.clear();
        }
    }

    static final class HybridHeapMemoryPool
    extends MemoryPool {
        private final ArrayDeque<byte[]> availableMemory;
        private final int segmentSize;

        HybridHeapMemoryPool(int numInitialSegments, int segmentSize) {
            this.availableMemory = new ArrayDeque(numInitialSegments);
            this.segmentSize = segmentSize;
            for (int i = 0; i < numInitialSegments; ++i) {
                this.availableMemory.add(new byte[segmentSize]);
            }
        }

        @Override
        MemorySegment allocateNewSegment(Object owner) {
            return MemorySegmentFactory.allocateUnpooledSegment((int)this.segmentSize, (Object)owner);
        }

        @Override
        MemorySegment requestSegmentFromPool(Object owner) {
            byte[] buf = this.availableMemory.remove();
            return MemorySegmentFactory.wrapPooledHeapMemory((byte[])buf, (Object)owner);
        }

        @Override
        void returnSegmentToPool(MemorySegment segment) {
            if (segment.getClass() != HybridMemorySegment.class) {
                throw new IllegalArgumentException("Memory segment is not a " + HybridMemorySegment.class.getSimpleName());
            }
            HybridMemorySegment heapSegment = (HybridMemorySegment)segment;
            this.availableMemory.add(heapSegment.getArray());
            heapSegment.free();
        }

        @Override
        protected int getNumberOfAvailableMemorySegments() {
            return this.availableMemory.size();
        }

        @Override
        void clear() {
            this.availableMemory.clear();
        }
    }

    static abstract class MemoryPool {
        MemoryPool() {
        }

        abstract int getNumberOfAvailableMemorySegments();

        abstract MemorySegment allocateNewSegment(Object var1);

        abstract MemorySegment requestSegmentFromPool(Object var1);

        abstract void returnSegmentToPool(MemorySegment var1);

        abstract void clear();
    }
}

