/*
 * Decompiled with CFR 0.152.
 */
package org.apache.directory.mavibot.btree;

import java.io.Closeable;
import java.io.EOFException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.directory.mavibot.btree.AbstractBTree;
import org.apache.directory.mavibot.btree.Addition;
import org.apache.directory.mavibot.btree.BTreeHeader;
import org.apache.directory.mavibot.btree.BTreeTypeEnum;
import org.apache.directory.mavibot.btree.DeleteResult;
import org.apache.directory.mavibot.btree.Deletion;
import org.apache.directory.mavibot.btree.InMemoryBTreeConfiguration;
import org.apache.directory.mavibot.btree.InMemoryLeaf;
import org.apache.directory.mavibot.btree.InMemoryNode;
import org.apache.directory.mavibot.btree.InsertResult;
import org.apache.directory.mavibot.btree.Modification;
import org.apache.directory.mavibot.btree.ModifyResult;
import org.apache.directory.mavibot.btree.NotPresentResult;
import org.apache.directory.mavibot.btree.Page;
import org.apache.directory.mavibot.btree.RemoveResult;
import org.apache.directory.mavibot.btree.SplitResult;
import org.apache.directory.mavibot.btree.Tuple;
import org.apache.directory.mavibot.btree.TupleCursor;
import org.apache.directory.mavibot.btree.exception.InitializationException;
import org.apache.directory.mavibot.btree.exception.KeyNotFoundException;
import org.apache.directory.mavibot.btree.exception.MissingSerializerException;
import org.apache.directory.mavibot.btree.serializer.BufferHandler;
import org.apache.directory.mavibot.btree.serializer.LongSerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class InMemoryBTree<K, V>
extends AbstractBTree<K, V>
implements Closeable {
    protected static final Logger LOG = LoggerFactory.getLogger(InMemoryBTree.class);
    public static final String DEFAULT_JOURNAL = "mavibot.log";
    public static final String DATA_SUFFIX = ".db";
    public static final String JOURNAL_SUFFIX = ".log";
    private File file;
    private boolean withJournal;
    private File journal;
    private File envDir;
    private FileChannel journalChannel = null;

    InMemoryBTree() {
        this.btreeHeader = new BTreeHeader();
        this.setType(BTreeTypeEnum.IN_MEMORY);
    }

    InMemoryBTree(InMemoryBTreeConfiguration<K, V> configuration) {
        String name = configuration.getName();
        if (name == null) {
            throw new IllegalArgumentException("BTree name cannot be null");
        }
        String filePath = configuration.getFilePath();
        if (filePath != null) {
            this.envDir = new File(filePath);
        }
        this.btreeHeader = new BTreeHeader();
        this.btreeHeader.setName(name);
        this.btreeHeader.setPageSize(configuration.getPageSize());
        this.keySerializer = configuration.getKeySerializer();
        this.btreeHeader.setKeySerializerFQCN(this.keySerializer.getClass().getName());
        this.valueSerializer = configuration.getValueSerializer();
        this.btreeHeader.setValueSerializerFQCN(this.valueSerializer.getClass().getName());
        this.readTimeOut = configuration.getReadTimeOut();
        this.writeBufferSize = configuration.getWriteBufferSize();
        this.btreeHeader.setAllowDuplicates(configuration.isAllowDuplicates());
        this.setType(configuration.getType());
        if (this.keySerializer.getComparator() == null) {
            throw new IllegalArgumentException("Comparator should not be null");
        }
        this.rootPage = new InMemoryLeaf(this);
        try {
            this.init();
        }
        catch (IOException ioe) {
            throw new InitializationException(ioe.getMessage());
        }
    }

    @Override
    public void init() throws IOException {
        if (this.envDir != null) {
            boolean created;
            if (!this.envDir.exists() && !(created = this.envDir.mkdirs())) {
                throw new IllegalStateException("Could not create the directory " + this.envDir + " for storing data");
            }
            this.file = new File(this.envDir, this.btreeHeader.getName() + DATA_SUFFIX);
            this.journal = new File(this.envDir, this.file.getName() + JOURNAL_SUFFIX);
            this.setType(BTreeTypeEnum.BACKED_ON_DISK);
        }
        this.readTransactions = new ConcurrentLinkedQueue();
        this.writeLock = new ReentrantLock();
        if (this.getType() == BTreeTypeEnum.BACKED_ON_DISK) {
            if (this.file.length() > 0L) {
                this.load(this.file);
            }
            this.withJournal = true;
            FileOutputStream stream = new FileOutputStream(this.journal);
            this.journalChannel = stream.getChannel();
            if (this.journal.length() > 0L) {
                this.applyJournal();
            }
        } else {
            this.setType(BTreeTypeEnum.IN_MEMORY);
        }
    }

    @Override
    public void close() throws IOException {
        if (this.getType() == BTreeTypeEnum.BACKED_ON_DISK) {
            this.flush();
            this.journalChannel.close();
        }
        this.rootPage = null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected Tuple<K, V> delete(K key, V value, long revision) throws IOException {
        this.writeLock.lock();
        try {
            Tuple tuple = null;
            DeleteResult<K, V> result = this.rootPage.delete(revision, key, value, null, -1);
            if (result instanceof NotPresentResult) {
                Tuple<K, V> tuple2 = null;
                return tuple2;
            }
            Page oldRootPage = this.rootPage;
            if (result instanceof RemoveResult) {
                Page modifiedPage;
                RemoveResult removeResult = (RemoveResult)result;
                this.rootPage = modifiedPage = removeResult.getModifiedPage();
                tuple = removeResult.getRemovedElement();
            }
            if (this.withJournal) {
                this.writeToJournal(new Deletion(key));
            }
            if (tuple != null) {
                this.btreeHeader.decrementNbElems();
            }
            Tuple tuple3 = tuple;
            return tuple3;
        }
        finally {
            this.writeLock.unlock();
        }
    }

    @Override
    InsertResult<K, V> insert(K key, V value, long revision) throws IOException {
        if (key == null) {
            throw new IllegalArgumentException("Key must not be null");
        }
        Object modifiedValue = null;
        InsertResult<K, V> result = this.rootPage.insert(revision, key, value);
        if (result instanceof ModifyResult) {
            Page modifiedPage;
            ModifyResult modifyResult = (ModifyResult)result;
            this.rootPage = modifiedPage = modifyResult.getModifiedPage();
            modifiedValue = modifyResult.getModifiedValue();
        } else {
            SplitResult splitResult = (SplitResult)result;
            Object pivot = splitResult.getPivot();
            Page leftPage = splitResult.getLeftPage();
            Page rightPage = splitResult.getRightPage();
            InMemoryNode newRootPage = null;
            this.rootPage = newRootPage = new InMemoryNode(this, revision, pivot, leftPage, rightPage);
        }
        if (this.withJournal) {
            this.writeToJournal(new Addition<K, V>(key, value));
        }
        if (modifiedValue == null) {
            this.btreeHeader.incrementNbElems();
        }
        return result;
    }

    private void writeBuffer(FileChannel channel, ByteBuffer bb, byte[] buffer) throws IOException {
        int size = buffer.length;
        int pos = 0;
        do {
            if (bb.remaining() >= size) {
                bb.put(buffer, pos, size);
                size = 0;
                continue;
            }
            int len = bb.remaining();
            size -= len;
            bb.put(buffer, pos, len);
            pos += len;
            bb.flip();
            channel.write(bb);
            bb.clear();
        } while (size > 0);
    }

    public void flush(File file) throws IOException {
        File parentFile = file.getParentFile();
        File baseDirectory = null;
        baseDirectory = parentFile != null ? new File(file.getParentFile().getAbsolutePath()) : new File(".");
        File tmpFileFD = File.createTempFile("mavibot", null, baseDirectory);
        FileOutputStream stream = new FileOutputStream(tmpFileFD);
        FileChannel ch = stream.getChannel();
        ByteBuffer bb = ByteBuffer.allocateDirect(this.writeBufferSize);
        TupleCursor cursor = this.browse();
        if (this.keySerializer == null) {
            throw new MissingSerializerException("Cannot flush the btree without a Key serializer");
        }
        if (this.valueSerializer == null) {
            throw new MissingSerializerException("Cannot flush the btree without a Value serializer");
        }
        bb.putLong(this.btreeHeader.getNbElems());
        while (cursor.hasNext()) {
            Tuple tuple = cursor.next();
            byte[] keyBuffer = this.keySerializer.serialize(tuple.getKey());
            this.writeBuffer(ch, bb, keyBuffer);
            byte[] valueBuffer = this.valueSerializer.serialize(tuple.getValue());
            this.writeBuffer(ch, bb, valueBuffer);
        }
        if (bb.position() > 0) {
            bb.flip();
            ch.write(bb);
        }
        ch.force(true);
        ch.close();
        File backupFile = File.createTempFile("mavibot", null, baseDirectory);
        file.renameTo(backupFile);
        tmpFileFD.renameTo(file);
        backupFile.delete();
    }

    private void applyJournal() throws IOException {
        long revision = this.generateRevision();
        if (!this.journal.exists()) {
            throw new IOException("The journal does not exist");
        }
        FileChannel channel = new RandomAccessFile(this.journal, "rw").getChannel();
        ByteBuffer buffer = ByteBuffer.allocate(65536);
        BufferHandler bufferHandler = new BufferHandler(channel, buffer);
        try {
            while (true) {
                Object key;
                byte[] type;
                if ((type = bufferHandler.read(1))[0] == 0) {
                    key = this.keySerializer.deserialize(bufferHandler);
                    Object value = this.valueSerializer.deserialize(bufferHandler);
                    this.insert(key, value, revision);
                    continue;
                }
                key = this.keySerializer.deserialize(bufferHandler);
                this.delete(key, revision);
            }
        }
        catch (EOFException eofe) {
            eofe.printStackTrace();
            this.journalChannel.truncate(0L);
            return;
        }
    }

    public void load(File file) throws IOException {
        long revision = this.generateRevision();
        if (!file.exists()) {
            throw new IOException("The file does not exist");
        }
        FileChannel channel = new RandomAccessFile(file, "rw").getChannel();
        ByteBuffer buffer = ByteBuffer.allocate(65536);
        BufferHandler bufferHandler = new BufferHandler(channel, buffer);
        long nbElems = LongSerializer.deserialize(bufferHandler.read(8));
        this.btreeHeader.setNbElems(nbElems);
        boolean isJournalActivated = this.withJournal;
        this.withJournal = false;
        for (long i = 0L; i < nbElems; ++i) {
            Object key = this.keySerializer.deserialize(bufferHandler);
            Object value = this.valueSerializer.deserialize(bufferHandler);
            this.insert(key, value, revision);
        }
        this.withJournal = isJournalActivated;
    }

    @Override
    public Page<K, V> getRootPage(long revision) throws IOException, KeyNotFoundException {
        return this.rootPage;
    }

    @Override
    public void flush() throws IOException {
        if (this.getType() == BTreeTypeEnum.BACKED_ON_DISK) {
            this.flush(this.file);
            this.journalChannel.truncate(0L);
        }
    }

    public File getFile() {
        return this.file;
    }

    public File getJournal() {
        return this.journal;
    }

    public boolean isInMemory() {
        return this.getType() == BTreeTypeEnum.IN_MEMORY;
    }

    public boolean isPersistent() {
        return this.getType() == BTreeTypeEnum.IN_MEMORY;
    }

    private void writeToJournal(Modification<K, V> modification) throws IOException {
        if (modification instanceof Addition) {
            byte[] keyBuffer = this.keySerializer.serialize(modification.getKey());
            ByteBuffer bb = ByteBuffer.allocateDirect(keyBuffer.length + 1);
            bb.put((byte)0);
            bb.put(keyBuffer);
            bb.flip();
            this.journalChannel.write(bb);
            byte[] valueBuffer = this.valueSerializer.serialize(modification.getValue());
            bb = ByteBuffer.allocateDirect(valueBuffer.length);
            bb.put(valueBuffer);
            bb.flip();
            this.journalChannel.write(bb);
        } else if (modification instanceof Deletion) {
            byte[] keyBuffer = this.keySerializer.serialize(modification.getKey());
            ByteBuffer bb = ByteBuffer.allocateDirect(keyBuffer.length + 1);
            bb.put((byte)1);
            bb.put(keyBuffer);
            bb.flip();
            this.journalChannel.write(bb);
        }
        this.journalChannel.force(true);
    }

    @Override
    public void beginTransaction() {
    }

    @Override
    public void commit() {
    }

    @Override
    public void rollback() {
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        switch (this.getType()) {
            case IN_MEMORY: {
                sb.append("In-memory ");
                break;
            }
            case BACKED_ON_DISK: {
                sb.append("Persistent ");
            }
        }
        sb.append("BTree");
        sb.append("[").append(this.btreeHeader.getName()).append("]");
        sb.append("( pageSize:").append(this.btreeHeader.getPageSize());
        if (this.rootPage != null) {
            sb.append(", nbEntries:").append(this.btreeHeader.getNbElems());
        } else {
            sb.append(", nbEntries:").append(0);
        }
        sb.append(", comparator:");
        if (this.keySerializer.getComparator() == null) {
            sb.append("null");
        } else {
            sb.append(this.keySerializer.getComparator().getClass().getSimpleName());
        }
        sb.append(", DuplicatesAllowed: ").append(this.btreeHeader.isAllowDuplicates());
        if (this.getType() == BTreeTypeEnum.BACKED_ON_DISK) {
            try {
                sb.append(", file : ");
                if (this.file != null) {
                    sb.append(this.file.getCanonicalPath());
                } else {
                    sb.append("Unknown");
                }
                sb.append(", journal : ");
                if (this.journal != null) {
                    sb.append(this.journal.getCanonicalPath());
                } else {
                    sb.append("Unkown");
                }
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        sb.append(") : \n");
        sb.append(this.rootPage.dumpPage(""));
        return sb.toString();
    }
}

