/*
 * Decompiled with CFR 0.152.
 */
package org.apache.directory.server.operations.search;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import javax.naming.InvalidNameException;
import javax.naming.NameNotFoundException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.BasicAttributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.LdapContext;
import org.apache.directory.junit.tools.MultiThreadedMultiInvoker;
import org.apache.directory.ldap.client.api.LdapConnection;
import org.apache.directory.ldap.client.api.LdapNetworkConnection;
import org.apache.directory.server.annotations.CreateLdapServer;
import org.apache.directory.server.annotations.CreateTransport;
import org.apache.directory.server.core.annotations.ApplyLdifs;
import org.apache.directory.server.core.integ.AbstractLdapTestUnit;
import org.apache.directory.server.core.integ.FrameworkRunner;
import org.apache.directory.server.integ.ServerIntegrationUtils;
import org.apache.directory.server.ldap.LdapServer;
import org.apache.directory.shared.ldap.codec.api.LdapApiService;
import org.apache.directory.shared.ldap.model.cursor.EntryCursor;
import org.apache.directory.shared.ldap.model.cursor.SearchCursor;
import org.apache.directory.shared.ldap.model.entry.DefaultEntry;
import org.apache.directory.shared.ldap.model.entry.Entry;
import org.apache.directory.shared.ldap.model.exception.LdapException;
import org.apache.directory.shared.ldap.model.message.Control;
import org.apache.directory.shared.ldap.model.message.Response;
import org.apache.directory.shared.ldap.model.message.SearchRequest;
import org.apache.directory.shared.ldap.model.message.SearchRequestImpl;
import org.apache.directory.shared.ldap.model.message.SearchScope;
import org.apache.directory.shared.ldap.model.message.controls.SubentriesImpl;
import org.apache.directory.shared.ldap.model.name.Dn;
import org.apache.directory.shared.ldap.util.JndiUtils;
import org.junit.Assert;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;

@RunWith(value=FrameworkRunner.class)
@CreateLdapServer(transports={@CreateTransport(protocol="LDAP")})
@ApplyLdifs(value={"dn: cn=Kate Bush,ou=system", "objectClass: person", "objectClass: organizationalPerson", "objectClass: inetOrgPerson", "objectClass: strongAuthenticationUser", "objectClass: top", "userCertificate:: NFZOXw==", "cn: Kate Bush", "description: this is a person", "sn: Bush", "jpegPhoto:: /9j/4AAQSkZJRgABAQEASABIAAD/4QAWRXhpZgAATU0AKgAAAAgAAAAAAAD//gAX", " Q3JlYXRlZCB3aXRoIFRoZSBHSU1Q/9sAQwAQCwwODAoQDg0OEhEQExgoGhgWFhgxIyUdKDozPTw", " 5Mzg3QEhcTkBEV0U3OFBtUVdfYmdoZz5NcXlwZHhcZWdj/9sAQwEREhIYFRgvGhovY0I4QmNjY2", " NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2Nj/8AAEQgAAQABA", " wEiAAIRAQMRAf/EABUAAQEAAAAAAAAAAAAAAAAAAAAF/8QAFBABAAAAAAAAAAAAAAAAAAAAAP/E", " ABUBAQEAAAAAAAAAAAAAAAAAAAUG/8QAFBEBAAAAAAAAAAAAAAAAAAAAAP/aAAwDAQACEQMRAD8", " AigC14//Z", "dn: cn=Tori Amos,ou=system", "objectClass: person", "objectClass: organizationalPerson", "objectClass: inetOrgPerson", "objectClass: strongAuthenticationUser", "objectClass: top", "userCertificate:: NFZOXw==", "cn: Tori Amos", "description: an American singer-songwriter", "sn: Amos", "jpegPhoto:: /9j/4AAQSkZJRgABAQEASABIAAD/4QAWRXhpZgAATU0AKgAAAAgAAAAAAAD//gAX", " Q3JlYXRlZCB3aXRoIFRoZSBHSU1Q/9sAQwAQCwwODAoQDg0OEhEQExgoGhgWFhgxIyUdKDozPTw", " 5Mzg3QEhcTkBEV0U3OFBtUVdfYmdoZz5NcXlwZHhcZWdj/9sAQwEREhIYFRgvGhovY0I4QmNjY2", " NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2Nj/8AAEQgAAQABA", " wEiAAIRAQMRAf/EABUAAQEAAAAAAAAAAAAAAAAAAAAF/8QAFBABAAAAAAAAAAAAAAAAAAAAAP/E", " ABUBAQEAAAAAAAAAAAAAAAAAAAUG/8QAFBEBAAAAAAAAAAAAAAAAAAAAAP/aAAwDAQACEQMRAD8", " AigC14//Z", "dn: cn=Rolling-Stones,ou=system", "objectClass: person", "objectClass: organizationalPerson", "objectClass: inetOrgPerson", "objectClass: strongAuthenticationUser", "objectClass: top", "userCertificate:: NFZOXw==", "cn: Rolling-Stones", "description: an English singer-songwriter", "sn: Jagger", "jpegPhoto:: /9j/4AAQSkZJRgABAQEASABIAAD/4QAWRXhpZgAATU0AKgAAAAgAAAAAAAD//gAX", " Q3JlYXRlZCB3aXRoIFRoZSBHSU1Q/9sAQwAQCwwODAoQDg0OEhEQExgoGhgWFhgxIyUdKDozPTw", " 5Mzg3QEhcTkBEV0U3OFBtUVdfYmdoZz5NcXlwZHhcZWdj/9sAQwEREhIYFRgvGhovY0I4QmNjY2", " NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2Nj/8AAEQgAAQABA", " wEiAAIRAQMRAf/EABUAAQEAAAAAAAAAAAAAAAAAAAAF/8QAFBABAAAAAAAAAAAAAAAAAAAAAP/E", " ABUBAQEAAAAAAAAAAAAAAAAAAAUG/8QAFBEBAAAAAAAAAAAAAAAAAAAAAP/aAAwDAQACEQMRAD8", " AigC14//Z", "dn: cn=Heather Nova,ou=system", "objectClass: person", "objectClass: organizationalPerson", "objectClass: inetOrgPerson", "objectClass: strongAuthenticationUser", "objectClass: top", "userCertificate:: NFZOXw==", "cn: Heather Nova", "sn: Nova", "jpegPhoto:: /9j/4AAQSkZJRgABAQEASABIAAD/4QAWRXhpZgAATU0AKgAAAAgAAAAAAAD//gAX", " Q3JlYXRlZCB3aXRoIFRoZSBHSU1Q/9sAQwAQCwwODAoQDg0OEhEQExgoGhgWFhgxIyUdKDozPTw", " 5Mzg3QEhcTkBEV0U3OFBtUVdfYmdoZz5NcXlwZHhcZWdj/9sAQwEREhIYFRgvGhovY0I4QmNjY2", " NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2Nj/8AAEQgAAQABA", " wEiAAIRAQMRAf/EABUAAQEAAAAAAAAAAAAAAAAAAAAF/8QAFBABAAAAAAAAAAAAAAAAAAAAAP/E", " ABUBAQEAAAAAAAAAAAAAAAAAAAUG/8QAFBEBAAAAAAAAAAAAAAAAAAAAAP/aAAwDAQACEQMRAD8", " AigC14//Z", "dn: cn=Janis Joplin,ou=system", "objectClass: person", "objectClass: organizationalPerson", "objectClass: inetOrgPerson", "objectClass: top", "objectClass: strongAuthenticationUser", "cn: Janis Joplin", "sn: Joplin", "userCertificate:: ", "dn: cn=Kim Wilde,ou=system", "objectClass: person", "objectClass: top", "cn: Kim Wilde", "sn: Wilde", "description: an American singer-songwriter+sexy blond"})
public class SearchIT
extends AbstractLdapTestUnit {
    @Rule
    public MultiThreadedMultiInvoker i = new MultiThreadedMultiInvoker(false);
    private static final String BASE = "ou=system";
    private static final String RDN = "cn=Tori Amos";
    private static final String RDN2 = "cn=Rolling-Stones";
    private static final String HEATHER_RDN = "cn=Heather Nova";
    private static final String FILTER = "(objectclass=*)";
    private static final byte[] JPEG = new byte[]{-1, -40, -1, -32, 0, 16, 74, 70, 73, 70, 0, 1, 1, 1, 0, 72, 0, 72, 0, 0, -1, -31, 0, 22, 69, 120, 105, 102, 0, 0, 77, 77, 0, 42, 0, 0, 0, 8, 0, 0, 0, 0, 0, 0, -1, -2, 0, 23, 67, 114, 101, 97, 116, 101, 100, 32, 119, 105, 116, 104, 32, 84, 104, 101, 32, 71, 73, 77, 80, -1, -37, 0, 67, 0, 16, 11, 12, 14, 12, 10, 16, 14, 13, 14, 18, 17, 16, 19, 24, 40, 26, 24, 22, 22, 24, 49, 35, 37, 29, 40, 58, 51, 61, 60, 57, 51, 56, 55, 64, 72, 92, 78, 64, 68, 87, 69, 55, 56, 80, 109, 81, 87, 95, 98, 103, 104, 103, 62, 77, 113, 121, 112, 100, 120, 92, 101, 103, 99, -1, -37, 0, 67, 1, 17, 18, 18, 24, 21, 24, 47, 26, 26, 47, 99, 66, 56, 66, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, -1, -64, 0, 17, 8, 0, 1, 0, 1, 3, 1, 34, 0, 2, 17, 1, 3, 17, 1, -1, -60, 0, 21, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, -1, -60, 0, 20, 16, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -1, -60, 0, 21, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 6, -1, -60, 0, 20, 17, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -1, -38, 0, 12, 3, 1, 0, 2, 17, 3, 17, 0, 63, 0, -118, 0, -75, -29, -1, -39};

    private Attributes getPersonAttributes(String sn, String cn) {
        BasicAttributes attributes = new BasicAttributes(true);
        BasicAttribute attribute = new BasicAttribute("objectClass");
        attribute.add("top");
        attribute.add("person");
        attribute.add("organizationalPerson");
        attribute.add("inetOrgPerson");
        attributes.put(attribute);
        attributes.put("cn", cn);
        attributes.put("sn", sn);
        attributes.put("jpegPhoto", JPEG);
        return attributes;
    }

    private void checkForAttributes(Attributes attrs, String[] attrNames) {
        for (String attrName : attrNames) {
            Assert.assertNotNull((String)("Check if attr " + attrName + " is present"), (Object)attrs.get(attrName));
        }
    }

    @Test
    public void testSearchByBinaryAttribute() throws Exception {
        DirContext ctx = (DirContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        byte[] certData = new byte[]{52, 86, 78, 95};
        SearchControls controls = new SearchControls();
        controls.setSearchScope(1);
        NamingEnumeration<SearchResult> enm = ctx.search("", "(cn=Kate Bush)", controls);
        Assert.assertTrue((boolean)enm.hasMore());
        SearchResult sr = enm.next();
        Assert.assertNotNull((Object)sr);
        Assert.assertFalse((boolean)enm.hasMore());
        Assert.assertEquals((Object)"cn=Kate Bush", (Object)sr.getName());
        enm = ctx.search("", "(&(cn=Kate Bush)(userCertificate={0}))", new Object[]{certData}, controls);
        Assert.assertTrue((boolean)enm.hasMore());
        sr = enm.next();
        Assert.assertNotNull((Object)sr);
        Assert.assertFalse((boolean)enm.hasMore());
        Assert.assertEquals((Object)"cn=Kate Bush", (Object)sr.getName());
        enm = ctx.search("", "(userCertificate=\\34\\56\\4E\\5F)", controls);
        Assert.assertTrue((boolean)enm.hasMore());
        int count = 0;
        HashSet<String> expected = new HashSet<String>();
        expected.add("cn=Kate Bush");
        expected.add(RDN);
        expected.add(RDN2);
        expected.add(HEATHER_RDN);
        while (enm.hasMore()) {
            ++count;
            sr = enm.next();
            Assert.assertNotNull((Object)sr);
            Assert.assertTrue((boolean)expected.contains(sr.getName()));
            expected.remove(sr.getName());
        }
        Assert.assertEquals((long)4L, (long)count);
        Assert.assertFalse((boolean)enm.hasMore());
        Assert.assertEquals((long)0L, (long)expected.size());
    }

    @Test
    public void testSearch() throws Exception {
        LdapContext ctx = ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer());
        SearchControls controls = new SearchControls();
        controls.setSearchScope(0);
        controls.setTimeLimit(10);
        try {
            ctx.search("myBadDN", "(objectClass=*)", controls);
            Assert.fail();
        }
        catch (InvalidNameException ine) {
        }
        catch (NamingException ne) {
            Assert.fail();
        }
        catch (Exception e) {
            Assert.fail();
        }
        try {
            controls = new SearchControls();
            controls.setSearchScope(1);
            controls.setTimeLimit(10);
            NamingEnumeration<SearchResult> result = ctx.search(BASE, "(objectClass=*)", controls);
            Assert.assertTrue((boolean)result.hasMore());
        }
        catch (InvalidNameException ine) {
            Assert.fail();
        }
        catch (NamingException ne) {
            Assert.fail();
        }
    }

    @Test
    public void testSearchEmptyBaseDn() throws Exception {
        LdapContext ctx = ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer());
        SearchControls controls = new SearchControls();
        controls.setSearchScope(0);
        controls.setTimeLimit(10);
        try {
            controls = new SearchControls();
            controls.setSearchScope(2);
            controls.setTimeLimit(10);
            NamingEnumeration<SearchResult> result = ctx.search("", "(cn=Janis Joplin)", controls);
            Assert.assertTrue((boolean)result.hasMore());
            SearchResult entry = result.next();
            Assert.assertEquals((Object)"cn=Janis Joplin,ou=system", (Object)entry.getName());
            Assert.assertFalse((boolean)result.hasMore());
        }
        catch (InvalidNameException ine) {
            Assert.fail();
        }
        catch (NamingException ne) {
            Assert.fail();
        }
    }

    private Set<String> search(String filter) throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls controls = new SearchControls();
        controls.setSearchScope(1);
        NamingEnumeration<SearchResult> ii = ctx.search("", filter, controls);
        HashSet<String> results = new HashSet<String>();
        while (ii.hasMore()) {
            SearchResult result = ii.next();
            results.add(result.getName());
        }
        return results;
    }

    @Test
    public void testDirserver635() throws Exception {
        Set<String> results = this.search("(|(cn=Kate*)(cn=Tori*))");
        Assert.assertEquals((String)"returned size of results", (long)2L, (long)results.size());
        Assert.assertTrue((String)"contains cn=Tori Amos", (boolean)results.contains(RDN));
        Assert.assertTrue((String)"contains cn=Kate Bush", (boolean)results.contains("cn=Kate Bush"));
        results = this.search("(|(cn=*Amos)(cn=Kate*))");
        Assert.assertEquals((String)"returned size of results", (long)2L, (long)results.size());
        Assert.assertTrue((String)"contains cn=Tori Amos", (boolean)results.contains(RDN));
        Assert.assertTrue((String)"contains cn=Kate Bush", (boolean)results.contains("cn=Kate Bush"));
        results = this.search("(|(cn=Kate Bush)(cn=Tori*))");
        Assert.assertEquals((String)"returned size of results", (long)2L, (long)results.size());
        Assert.assertTrue((String)"contains cn=Tori Amos", (boolean)results.contains(RDN));
        Assert.assertTrue((String)"contains cn=Kate Bush", (boolean)results.contains("cn=Kate Bush"));
        results = this.search("(|(cn=*Amos))");
        Assert.assertEquals((String)"returned size of results", (long)1L, (long)results.size());
        Assert.assertTrue((String)"contains cn=Tori Amos", (boolean)results.contains(RDN));
    }

    @Test
    public void testSearchWithBackslashEscapedBase() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        Attributes attributes = this.getPersonAttributes("Ferry", "Bryan Ferry");
        ctx.createSubcontext("sn=Ferry", attributes);
        SearchControls sctls = new SearchControls();
        sctls.setSearchScope(0);
        String filter = "(cn=Bryan Ferry)";
        String base = "sn=\\46\\65\\72\\72\\79";
        try {
            NamingEnumeration<SearchResult> enm = ctx.search(base, filter, sctls);
            Assert.assertTrue((boolean)enm.hasMore());
            while (enm.hasMore()) {
                SearchResult sr = enm.next();
                Attributes attrs = sr.getAttributes();
                Attribute sn = attrs.get("sn");
                Assert.assertNotNull((Object)sn);
                Assert.assertTrue((boolean)sn.contains("Ferry"));
            }
        }
        catch (Exception e) {
            Assert.fail((String)e.getMessage());
        }
    }

    @Test
    public void testSearchValue() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls ctls = new SearchControls();
        ctls.setReturningAttributes(new String[]{"*"});
        ctls.setSearchScope(0);
        NamingEnumeration<SearchResult> results = ctx.search(RDN, "(cn=*)", ctls);
        Assert.assertTrue((boolean)results.hasMore());
        results = ctx.search(RDN2, "(cn=*)", ctls);
        Assert.assertTrue((boolean)results.hasMore());
        results = ctx.search(RDN, "(cn=*Amos)", ctls);
        Assert.assertTrue((boolean)results.hasMore());
        results = ctx.search(RDN2, "(cn=*Amos)", ctls);
        Assert.assertFalse((boolean)results.hasMore());
        results = ctx.search(RDN, "(cn=*amos)", ctls);
        Assert.assertTrue((boolean)results.hasMore());
        results = ctx.search(RDN2, "(cn=*amos)", ctls);
        Assert.assertFalse((boolean)results.hasMore());
        results = ctx.search(RDN, "(cn=Tori*)", ctls);
        Assert.assertTrue((boolean)results.hasMore());
        results = ctx.search(RDN2, "(cn=Tori*)", ctls);
        Assert.assertFalse((boolean)results.hasMore());
        results = ctx.search(RDN, "(cn=tori*)", ctls);
        Assert.assertTrue((boolean)results.hasMore());
        results = ctx.search(RDN2, "(cn=tori*)", ctls);
        Assert.assertFalse((boolean)results.hasMore());
        results = ctx.search(RDN, "(cn=*ori*)", ctls);
        Assert.assertTrue((boolean)results.hasMore());
        results = ctx.search(RDN2, "(cn=*ori*)", ctls);
        Assert.assertFalse((boolean)results.hasMore());
        results = ctx.search(RDN, "(cn=*o*i*)", ctls);
        Assert.assertTrue((boolean)results.hasMore());
        results = ctx.search(RDN2, "(cn=*o*i*)", ctls);
        Assert.assertTrue((boolean)results.hasMore());
        results = ctx.search(RDN, "(cn=*o* *o*)", ctls);
        Assert.assertTrue((boolean)results.hasMore());
        results = ctx.search(RDN2, "(cn=*o* *o*)", ctls);
        Assert.assertFalse((boolean)results.hasMore());
        results = ctx.search(RDN2, "(cn=*o*-*o*)", ctls);
        Assert.assertTrue((boolean)results.hasMore());
        results = ctx.search(RDN, "(cn=To*A*)", ctls);
        Assert.assertTrue((boolean)results.hasMore());
        results = ctx.search(RDN2, "(cn=To*A*)", ctls);
        Assert.assertFalse((boolean)results.hasMore());
        results = ctx.search(RDN, "(cn=*ri*os)", ctls);
        Assert.assertTrue((boolean)results.hasMore());
        results = ctx.search(RDN2, "(cn=*ri*os)", ctls);
        Assert.assertFalse((boolean)results.hasMore());
    }

    @Test
    public void testUndefinedAvaInBranchFilters() throws Exception {
        Set<String> results = this.search("(|(sn=Bush)(numberOfOctaves=4))");
        Assert.assertEquals((String)"returned size of results", (long)1L, (long)results.size());
        Assert.assertTrue((String)"contains cn=Kate Bush", (boolean)results.contains("cn=Kate Bush"));
        results = this.search("(&(sn=Bush)(numberOfOctaves=4))");
        Assert.assertEquals((String)"returned size of results", (long)0L, (long)results.size());
    }

    @Test
    public void testSearchSchema() throws Exception {
        SearchControls controls = new SearchControls();
        controls.setSearchScope(0);
        controls.setReturningAttributes(new String[]{"objectClasses"});
        LdapContext ctx = ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer());
        NamingEnumeration<SearchResult> results = ctx.search("cn=schema", "objectClass=subschema", controls);
        Assert.assertTrue((boolean)results.hasMore());
        SearchResult result = results.next();
        Assert.assertNotNull((Object)result);
        Assert.assertFalse((boolean)results.hasMore());
        NamingEnumeration<? extends Attribute> attrs = result.getAttributes().getAll();
        while (attrs.hasMoreElements()) {
            Attribute attr = attrs.next();
            String ID = attr.getID();
            Assert.assertEquals((Object)"objectClasses", (Object)ID);
        }
        Assert.assertNotNull((Object)result.getAttributes().get("objectClasses"));
        Assert.assertEquals((long)1L, (long)result.getAttributes().size());
    }

    private void createAccessControlSubentry(String cn, String subtree, String aciItem) throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        LdapContext adminCtx = ctx;
        Attributes ap = adminCtx.getAttributes("", new String[]{"administrativeRole"});
        Attribute administrativeRole = ap.get("administrativeRole");
        if (administrativeRole == null || !administrativeRole.contains("accessControlSpecificArea")) {
            BasicAttributes changes = new BasicAttributes("administrativeRole", "accessControlSpecificArea", true);
            adminCtx.modifyAttributes("", 1, (Attributes)changes);
        }
        BasicAttributes subentry = new BasicAttributes("cn", cn, true);
        BasicAttribute objectClass = new BasicAttribute("objectClass");
        subentry.put(objectClass);
        objectClass.add("top");
        objectClass.add("subentry");
        objectClass.add("accessControlSubentry");
        subentry.put("subtreeSpecification", subtree);
        subentry.put("prescriptiveACI", aciItem);
        adminCtx.createSubcontext("cn=" + cn, (Attributes)subentry);
    }

    @Test
    public void testAddWithObjectclasses() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls ctls = new SearchControls();
        ctls.setSearchScope(0);
        ctls.setReturningAttributes(new String[]{"objectclass"});
        String filter = FILTER;
        String rdn = "cn=Kim Wilde";
        NamingEnumeration<SearchResult> result = ctx.search(rdn, filter, ctls);
        if (result.hasMore()) {
            SearchResult entry = result.next();
            Attributes heatherReloaded = entry.getAttributes();
            Attribute loadedOcls = heatherReloaded.get("objectClass");
            Assert.assertNotNull((Object)loadedOcls);
            Assert.assertTrue((boolean)loadedOcls.contains("person"));
            Assert.assertTrue((boolean)loadedOcls.contains("top"));
        } else {
            Assert.fail((String)("entry " + rdn + " not found"));
        }
        ctx.destroySubcontext(rdn);
    }

    @Test
    public void testAddWithMissingObjectclasses() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        String rdn = "cn=Kate Bush";
        SearchControls ctls = new SearchControls();
        ctls.setSearchScope(0);
        ctls.setReturningAttributes(new String[]{"objectclass"});
        String filter = FILTER;
        NamingEnumeration<SearchResult> result = ctx.search(rdn, filter, ctls);
        if (result.hasMore()) {
            SearchResult entry = result.next();
            Attributes kateReloaded = entry.getAttributes();
            Attribute loadedOcls = kateReloaded.get("objectClass");
            Assert.assertNotNull((Object)loadedOcls);
            Assert.assertTrue((boolean)loadedOcls.contains("top"));
            Assert.assertTrue((boolean)loadedOcls.contains("person"));
            Assert.assertTrue((boolean)loadedOcls.contains("organizationalPerson"));
        } else {
            Assert.fail((String)("entry " + rdn + " not found"));
        }
        ctx.destroySubcontext(rdn);
    }

    @Test
    public void testSubentryControl() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        this.createAccessControlSubentry("anyBodyAdd", "{}", "{   identificationTag \"addAci\",   precedence 14,   authenticationLevel none,   itemOrUserFirst userFirst:   {     userClasses     {       allUsers     },     userPermissions     {       {         protectedItems         {           entry, allUserAttributeTypesAndValues        },         grantsAndDenials         {           grantAdd, grantBrowse         }       }     }   } }");
        SubentriesImpl ctl = new SubentriesImpl();
        ctl.setVisibility(true);
        Control[] reqControls = new Control[]{ctl};
        SearchControls searchControls = new SearchControls();
        searchControls.setSearchScope(1);
        ctx.setRequestControls(JndiUtils.toJndiControls((LdapApiService)SearchIT.getLdapServer().getDirectoryService().getLdapCodecService(), (Control[])reqControls));
        NamingEnumeration<SearchResult> enm = ctx.search("", "(objectClass=*)", searchControls);
        HashSet<String> results = new HashSet<String>();
        while (enm.hasMore()) {
            SearchResult result = enm.next();
            results.add(result.getName());
        }
        Assert.assertEquals((String)"expected results size of", (long)1L, (long)results.size());
        Assert.assertTrue((boolean)results.contains("cn=anyBodyAdd"));
    }

    @Test
    public void testMultiValuedRdnContent() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        Attributes attrs = this.getPersonAttributes("Bush", "Kate Bush");
        String rdn = "cn=Kate Bush+sn=Bush";
        ctx.createSubcontext(rdn, attrs);
        SearchControls sctls = new SearchControls();
        sctls.setSearchScope(2);
        String filter = "(sn=Bush)";
        String base = "";
        NamingEnumeration<SearchResult> enm = ctx.search(base, filter, sctls);
        while (enm.hasMore()) {
            SearchResult sr = enm.next();
            attrs = sr.getAttributes();
            Attribute cn = sr.getAttributes().get("cn");
            Assert.assertNotNull((Object)cn);
            Assert.assertTrue((boolean)cn.contains("Kate Bush"));
            Attribute sn = sr.getAttributes().get("sn");
            Assert.assertNotNull((Object)sn);
            Assert.assertTrue((boolean)sn.contains("Bush"));
        }
        ctx.destroySubcontext(rdn);
    }

    @Test
    public void testMultiValuedRdnName() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        Attributes attrs = this.getPersonAttributes("Bush", "Kate Bush");
        String rdn = "cn=Kate Bush+sn=Bush";
        DirContext entry = ctx.createSubcontext(rdn, attrs);
        String nameInNamespace = entry.getNameInNamespace();
        SearchControls sctls = new SearchControls();
        sctls.setSearchScope(0);
        String filter = "(sn=Bush)";
        NamingEnumeration<SearchResult> enm = ctx.search(rdn, filter, sctls);
        if (enm.hasMore()) {
            SearchResult sr = enm.next();
            Assert.assertNotNull((Object)sr);
            Assert.assertEquals((String)"Name in namespace", (Object)nameInNamespace, (Object)sr.getNameInNamespace());
        } else {
            Assert.fail((String)("Entry not found:" + nameInNamespace));
        }
        enm.close();
        String mixedRdn = "sn=Bush+cn=Kate Bush";
        enm = ctx.search(mixedRdn, filter, sctls);
        if (enm.hasMore()) {
            SearchResult sr = enm.next();
            Assert.assertNotNull((Object)sr);
            Dn expectedDn = new Dn(new String[]{nameInNamespace});
            Assert.assertEquals((String)"Name in namespace", (Object)expectedDn, (Object)sr.getNameInNamespace());
        } else {
            Assert.fail((String)("Entry not found:" + nameInNamespace));
        }
        ctx.destroySubcontext(rdn);
    }

    @Test
    public void testSearchJpeg() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls controls = new SearchControls();
        controls.setSearchScope(1);
        NamingEnumeration<SearchResult> res = ctx.search("", "(cn=Tori*)", controls);
        while (res.hasMore()) {
            SearchResult result = res.next();
            Attributes attrs = result.getAttributes();
            NamingEnumeration<? extends Attribute> all = attrs.getAll();
            while (all.hasMoreElements()) {
                Attribute attr = all.next();
                if (!"jpegPhoto".equalsIgnoreCase(attr.getID())) continue;
                byte[] jpegVal = (byte[])attr.get();
                Assert.assertTrue((boolean)Arrays.equals(jpegVal, JPEG));
            }
        }
    }

    @Test
    public void testSearchOID() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls controls = new SearchControls();
        controls.setSearchScope(1);
        NamingEnumeration<SearchResult> res = ctx.search("", "(2.5.4.3=Tori*)", controls);
        Assert.assertTrue((boolean)res.hasMore());
        SearchResult result = res.next();
        Assert.assertNotNull((Object)result);
        String rdn = result.getName();
        Assert.assertEquals((Object)RDN, (Object)rdn);
        Assert.assertFalse((boolean)res.hasMore());
    }

    @Test
    public void testSearchAttrCN() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls controls = new SearchControls();
        controls.setSearchScope(1);
        controls.setReturningAttributes(new String[]{"cn"});
        NamingEnumeration<SearchResult> res = ctx.search("", "(commonName=Tori*)", controls);
        Assert.assertTrue((boolean)res.hasMore());
        SearchResult result = res.next();
        Assert.assertNotNull((Object)result);
        Attributes attrs = result.getAttributes();
        Assert.assertEquals((long)1L, (long)attrs.size());
        Assert.assertNotNull((Object)attrs.get("cn"));
        Assert.assertEquals((long)1L, (long)attrs.get("cn").size());
        Assert.assertEquals((Object)"Tori Amos", (Object)attrs.get("cn").get());
    }

    @Test
    public void testSearchAttrName() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls controls = new SearchControls();
        controls.setSearchScope(1);
        controls.setReturningAttributes(new String[]{"name"});
        NamingEnumeration<SearchResult> res = ctx.search("", "(commonName=Tori*)", controls);
        Assert.assertTrue((boolean)res.hasMore());
        SearchResult result = res.next();
        Assert.assertNotNull((Object)result);
        Attributes attrs = result.getAttributes();
        Assert.assertEquals((long)2L, (long)attrs.size());
        Assert.assertNotNull((Object)attrs.get("cn"));
        Assert.assertEquals((long)1L, (long)attrs.get("cn").size());
        Assert.assertEquals((Object)"Tori Amos", (Object)attrs.get("cn").get());
        Assert.assertNotNull((Object)attrs.get("sn"));
        Assert.assertEquals((long)1L, (long)attrs.get("sn").size());
        Assert.assertEquals((Object)"Amos", (Object)attrs.get("sn").get());
    }

    @Test
    public void testSearchAttrCommonName() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls controls = new SearchControls();
        controls.setSearchScope(1);
        controls.setReturningAttributes(new String[]{"commonName"});
        NamingEnumeration<SearchResult> res = ctx.search("", "(commonName=Tori*)", controls);
        Assert.assertTrue((boolean)res.hasMore());
        SearchResult result = res.next();
        Assert.assertNotNull((Object)result);
        Attributes attrs = result.getAttributes();
        Assert.assertEquals((long)1L, (long)attrs.size());
        Assert.assertNotNull((Object)attrs.get("cn"));
        Assert.assertEquals((long)1L, (long)attrs.get("cn").size());
        Assert.assertEquals((Object)"Tori Amos", (Object)attrs.get("cn").get());
    }

    @Test
    public void testSearchAttrOID() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls controls = new SearchControls();
        controls.setSearchScope(1);
        controls.setReturningAttributes(new String[]{"2.5.4.3"});
        NamingEnumeration<SearchResult> res = ctx.search("", "(commonName=Tori*)", controls);
        Assert.assertTrue((boolean)res.hasMore());
        SearchResult result = res.next();
        Assert.assertNotNull((Object)result);
        Attributes attrs = result.getAttributes();
        Assert.assertEquals((long)1L, (long)attrs.size());
        Assert.assertNotNull((Object)attrs.get("cn"));
        Assert.assertEquals((long)1L, (long)attrs.get("cn").size());
        Assert.assertEquals((Object)"Tori Amos", (Object)attrs.get("cn").get());
    }

    @Test
    public void testSearchAttrC_L() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        BasicAttributes aaAttrs = new BasicAttributes(true);
        BasicAttribute aaObjectClass = new BasicAttribute("objectClass");
        aaObjectClass.add("top");
        aaObjectClass.add("organizationalUnit");
        aaObjectClass.add("extensibleObject");
        aaAttrs.put(aaObjectClass);
        aaAttrs.put("ou", "Collective Area");
        aaAttrs.put("administrativeRole", "collectiveAttributeSpecificArea");
        DirContext aaCtx = ctx.createSubcontext("ou=Collective Area", (Attributes)aaAttrs);
        BasicAttributes subentry = new BasicAttributes(true);
        BasicAttribute objectClass = new BasicAttribute("objectClass");
        objectClass.add("top");
        objectClass.add("subentry");
        objectClass.add("collectiveAttributeSubentry");
        subentry.put(objectClass);
        subentry.put("c-l", "Munich");
        subentry.put("cn", "Collective Subentry");
        subentry.put("subtreeSpecification", "{ }");
        aaCtx.createSubcontext("cn=Collective Subentry", (Attributes)subentry);
        Attributes attributes = this.getPersonAttributes("Bush", "Kate Bush");
        aaCtx.createSubcontext("cn=Kate Bush", attributes);
        SearchControls controls = new SearchControls();
        controls.setSearchScope(1);
        controls.setReturningAttributes(new String[]{"c-l"});
        NamingEnumeration<SearchResult> res = aaCtx.search("", "(cn=Kate Bush)", controls);
        Assert.assertTrue((boolean)res.hasMore());
        SearchResult result = res.next();
        Assert.assertNotNull((Object)result);
        Attributes attrs = result.getAttributes();
        Assert.assertEquals((long)1L, (long)attrs.size());
        Assert.assertNotNull((Object)attrs.get("c-l"));
        Assert.assertEquals((long)1L, (long)attrs.get("c-l").size());
        Assert.assertEquals((Object)"Munich", (Object)attrs.get("c-l").get());
    }

    @Test
    public void testSearchUsersAttrs() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls controls = new SearchControls();
        controls.setSearchScope(1);
        controls.setReturningAttributes(new String[]{"*"});
        NamingEnumeration<SearchResult> res = ctx.search("", "(commonName=Tori Amos)", controls);
        Assert.assertTrue((boolean)res.hasMore());
        SearchResult result = res.next();
        Assert.assertNotNull((Object)result);
        Attributes attrs = result.getAttributes();
        Assert.assertEquals((long)6L, (long)attrs.size());
        Assert.assertNotNull((Object)attrs.get("cn"));
        Assert.assertNotNull((Object)attrs.get("sn"));
        Assert.assertNotNull((Object)attrs.get("objectClass"));
        Assert.assertNotNull((Object)attrs.get("jpegPhoto"));
        Assert.assertNotNull((Object)attrs.get("description"));
        Assert.assertNotNull((Object)attrs.get("userCertificate"));
        Assert.assertNull((Object)attrs.get("createtimestamp"));
        Assert.assertNull((Object)attrs.get("creatorsname"));
    }

    @Test
    public void testSearchOperationalAttrs() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls controls = new SearchControls();
        controls.setSearchScope(1);
        controls.setReturningAttributes(new String[]{"+"});
        NamingEnumeration<SearchResult> res = ctx.search("", "(commonName=Tori Amos)", controls);
        Assert.assertTrue((boolean)res.hasMore());
        SearchResult result = res.next();
        Assert.assertNotNull((Object)result);
        Attributes attrs = result.getAttributes();
        Assert.assertEquals((long)5L, (long)attrs.size());
        Assert.assertNull((Object)attrs.get("cn"));
        Assert.assertNull((Object)attrs.get("sn"));
        Assert.assertNull((Object)attrs.get("objectClass"));
        Assert.assertNull((Object)attrs.get("jpegPhoto"));
        Assert.assertNull((Object)attrs.get("description"));
        Assert.assertNotNull((Object)attrs.get("createtimestamp"));
        Assert.assertNotNull((Object)attrs.get("creatorsname"));
        Assert.assertNotNull((Object)attrs.get("entryuuid"));
        Assert.assertNotNull((Object)attrs.get("entrycsn"));
    }

    @Test
    public void testSearchAllAttrs() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls controls = new SearchControls();
        controls.setSearchScope(1);
        controls.setReturningAttributes(new String[]{"+", "*"});
        NamingEnumeration<SearchResult> res = ctx.search("", "(commonName=Tori Amos)", controls);
        Assert.assertTrue((boolean)res.hasMore());
        SearchResult result = res.next();
        Assert.assertNotNull((Object)result);
        Attributes attrs = result.getAttributes();
        Assert.assertEquals((long)11L, (long)attrs.size());
        Assert.assertNotNull((Object)attrs.get("cn"));
        Assert.assertNotNull((Object)attrs.get("sn"));
        Assert.assertNotNull((Object)attrs.get("objectClass"));
        Assert.assertNotNull((Object)attrs.get("jpegPhoto"));
        Assert.assertNotNull((Object)attrs.get("userCertificate"));
        Assert.assertNotNull((Object)attrs.get("description"));
        Assert.assertNotNull((Object)attrs.get("createtimestamp"));
        Assert.assertNotNull((Object)attrs.get("creatorsname"));
        Assert.assertNotNull((Object)attrs.get("entryuuid"));
        Assert.assertNotNull((Object)attrs.get("entrycsn"));
    }

    @Test
    public void testSearchBadDN() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls controls = new SearchControls();
        controls.setSearchScope(1);
        try {
            ctx.search("cn=admin", "(objectClass=*)", controls);
        }
        catch (NameNotFoundException nnfe) {
            Assert.assertTrue((boolean)true);
        }
    }

    @Test
    public void testSearchInvalidDN() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls controls = new SearchControls();
        controls.setSearchScope(1);
        try {
            ctx.search("myBadDN", "(objectClass=*)", controls);
            Assert.fail();
        }
        catch (NamingException ne) {
            Assert.assertTrue((boolean)true);
        }
    }

    @Test
    public void testSearchOperationalAttributes() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls ctls = new SearchControls();
        ctls.setSearchScope(0);
        ctls.setReturningAttributes(new String[]{"+"});
        NamingEnumeration<SearchResult> result = ctx.search(HEATHER_RDN, FILTER, ctls);
        if (result.hasMore()) {
            SearchResult entry = result.next();
            String[] opAttrNames = new String[]{"creatorsName", "createTimestamp"};
            this.checkForAttributes(entry.getAttributes(), opAttrNames);
        } else {
            Assert.fail((String)"entry cn=Heather Nova not found");
        }
        result.close();
    }

    @Test
    public void testSearchUserAttributes() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls ctls = new SearchControls();
        ctls.setSearchScope(0);
        ctls.setReturningAttributes(new String[]{"*"});
        NamingEnumeration<SearchResult> result = ctx.search(HEATHER_RDN, FILTER, ctls);
        if (result.hasMore()) {
            SearchResult entry = result.next();
            String[] userAttrNames = new String[]{"objectClass", "sn", "cn"};
            this.checkForAttributes(entry.getAttributes(), userAttrNames);
        } else {
            Assert.fail((String)"entry cn=Heather Nova not found");
        }
        result.close();
    }

    @Test
    public void testSearchUserAttributes_Space() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls ctls = new SearchControls();
        ctls.setSearchScope(0);
        ctls.setReturningAttributes(new String[]{" "});
        NamingEnumeration<SearchResult> result = ctx.search(HEATHER_RDN, FILTER, ctls);
        result.close();
    }

    @Test
    public void testSearchUserAttributes_EmptyAttrs() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls ctls = new SearchControls();
        ctls.setSearchScope(0);
        ctls.setReturningAttributes(new String[]{""});
        NamingEnumeration<SearchResult> result = ctx.search(HEATHER_RDN, FILTER, ctls);
        result.close();
    }

    @Test
    public void testSearchUserAttributes_NullAttrs() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls ctls = new SearchControls();
        ctls.setSearchScope(0);
        ctls.setReturningAttributes(new String[0]);
        NamingEnumeration<SearchResult> result = ctx.search(HEATHER_RDN, FILTER, ctls);
        result.close();
    }

    @Test
    public void testSearchOperationalAndUserAttributes() throws Exception {
        Attributes attrs;
        SearchResult entry;
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls ctls = new SearchControls();
        ctls.setSearchScope(0);
        ctls.setReturningAttributes(new String[]{"+", "*"});
        String[] userAttrNames = new String[]{"objectClass", "sn", "cn"};
        String[] opAttrNames = new String[]{"creatorsName", "createTimestamp"};
        NamingEnumeration<SearchResult> result = ctx.search(HEATHER_RDN, FILTER, ctls);
        if (result.hasMore()) {
            entry = result.next();
            attrs = entry.getAttributes();
            Assert.assertNotNull((Object)attrs);
            this.checkForAttributes(attrs, userAttrNames);
            this.checkForAttributes(attrs, opAttrNames);
        } else {
            Assert.fail((String)"entry cn=Heather Nova not found");
        }
        result.close();
        ctls.setReturningAttributes(new String[]{"*", "+"});
        result = ctx.search(HEATHER_RDN, FILTER, ctls);
        if (result.hasMore()) {
            entry = result.next();
            attrs = entry.getAttributes();
            Assert.assertNotNull((Object)attrs);
            this.checkForAttributes(attrs, userAttrNames);
            this.checkForAttributes(attrs, opAttrNames);
        } else {
            Assert.fail((String)"entry cn=Heather Nova not found");
        }
        result.close();
    }

    @Test
    public void testSubstringSearchWithEscapedCharsInFilter() throws Exception {
        String[] filters;
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        Attributes attrs = new BasicAttributes("objectClass", "inetOrgPerson", true);
        attrs.get("objectClass").add("organizationalPerson");
        attrs.get("objectClass").add("person");
        attrs.put("givenName", "Jim");
        attrs.put("sn", "Bean");
        attrs.put("cn", "jimbean");
        attrs.put("description", "(sex*pis\\tols)");
        ctx.createSubcontext("cn=jimbean", attrs);
        SearchControls controls = new SearchControls();
        controls.setSearchScope(1);
        controls.setReturningAttributes(new String[]{"cn"});
        for (String filter : filters = new String[]{"(description=*\\28*)", "(description=*\\29*)", "(description=*\\2A*)", "(description=*\\5C*)"}) {
            NamingEnumeration<SearchResult> res = ctx.search("", filter, controls);
            Assert.assertTrue((boolean)res.hasMore());
            SearchResult result = res.next();
            Assert.assertNotNull((Object)result);
            attrs = result.getAttributes();
            Assert.assertEquals((long)1L, (long)attrs.size());
            Assert.assertNotNull((Object)attrs.get("cn"));
            Assert.assertEquals((long)1L, (long)attrs.get("cn").size());
            Assert.assertEquals((Object)"jimbean", (Object)attrs.get("cn").get());
            Assert.assertFalse((boolean)res.hasMore());
        }
    }

    @Test
    public void testMissingAnyInSubstring_DIRSERVER_1180() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        BasicAttributes attrs = new BasicAttributes("objectClass", "inetOrgPerson", true);
        attrs.get("objectClass").add("organizationalPerson");
        attrs.get("objectClass").add("person");
        attrs.put("givenName", "Jim");
        attrs.put("sn", "Bean");
        attrs.put("cn", "jimbean");
        ctx.createSubcontext("cn=jimbean", (Attributes)attrs);
        try {
            ctx.search("", "(cn=**)", new SearchControls());
            Assert.fail();
        }
        catch (Exception e) {
            Assert.assertTrue((boolean)true);
        }
    }

    @Test
    public void testSubstringSearchWithEscapedAsterisksInFilter_DIRSERVER_1181() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        BasicAttributes vicious = new BasicAttributes(true);
        BasicAttribute ocls = new BasicAttribute("objectClass");
        ocls.add("top");
        ocls.add("person");
        vicious.put(ocls);
        vicious.put("cn", "x*y*z*");
        vicious.put("sn", "x*y*z*");
        ctx.createSubcontext("cn=x*y*z*", (Attributes)vicious);
        SearchControls controls = new SearchControls();
        controls.setSearchScope(1);
        controls.setReturningAttributes(new String[]{"cn"});
        NamingEnumeration<SearchResult> res = ctx.search("", "(cn=*x\\2Ay\\2Az\\2A*)", controls);
        Assert.assertTrue((boolean)res.hasMore());
        Assert.assertEquals((Object)"x*y*z*", (Object)res.next().getAttributes().get("cn").get());
        Assert.assertFalse((boolean)res.hasMore());
        res = ctx.search("", "(cn=*{0}*)", (Object[])new String[]{"x*y*z*"}, controls);
        Assert.assertTrue((boolean)res.hasMore());
        Assert.assertEquals((Object)"x*y*z*", (Object)res.next().getAttributes().get("cn").get());
        Assert.assertFalse((boolean)res.hasMore());
    }

    @Test
    public void testUnicodeFilter_DIRSERVER_1347() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        BasicAttributes groupOfNames = new BasicAttributes(true);
        BasicAttribute groupOfNamesOC = new BasicAttribute("objectClass");
        groupOfNamesOC.add("top");
        groupOfNamesOC.add("groupOfNames");
        groupOfNames.put(groupOfNamesOC);
        groupOfNames.put("cn", "groupOfNames");
        BasicAttribute member = new BasicAttribute("member");
        member.add("uid=test,ou=system");
        member.add("uid=r\u00e9dacteur1,ou=system");
        groupOfNames.put(member);
        ctx.createSubcontext("cn=groupOfNames", (Attributes)groupOfNames);
        BasicAttributes groupOfUniqueNames = new BasicAttributes(true);
        BasicAttribute groupOfUniqueNamesOC = new BasicAttribute("objectClass");
        groupOfUniqueNamesOC.add("top");
        groupOfUniqueNamesOC.add("groupOfUniqueNames");
        groupOfUniqueNames.put(groupOfUniqueNamesOC);
        groupOfUniqueNames.put("cn", "groupOfUniqueNames");
        BasicAttribute uniqueMember = new BasicAttribute("uniqueMember");
        uniqueMember.add("uid=test,ou=system");
        uniqueMember.add("uid=r\u00e9dacteur1,ou=system");
        groupOfUniqueNames.put(uniqueMember);
        ctx.createSubcontext("cn=groupOfUniqueNames", (Attributes)groupOfUniqueNames);
        SearchControls controls = new SearchControls();
        NamingEnumeration<SearchResult> res = ctx.search("", "(member=uid=r\u00e9dacteur1,ou=system)", controls);
        Assert.assertTrue((boolean)res.hasMore());
        Assert.assertEquals((Object)"groupOfNames", (Object)res.next().getAttributes().get("cn").get());
        Assert.assertFalse((boolean)res.hasMore());
        res = ctx.search("", "(member=uid=r\\c3\\a9dacteur1,ou=system)", controls);
        Assert.assertTrue((boolean)res.hasMore());
        Assert.assertEquals((Object)"groupOfNames", (Object)res.next().getAttributes().get("cn").get());
        Assert.assertFalse((boolean)res.hasMore());
        res = ctx.search("", "(uniqueMember=uid=r\u00e9dacteur1,ou=system)", controls);
        Assert.assertTrue((boolean)res.hasMore());
        Assert.assertEquals((Object)"groupOfUniqueNames", (Object)res.next().getAttributes().get("cn").get());
        Assert.assertFalse((boolean)res.hasMore());
        res = ctx.search("", "(uniqueMember=uid=r\\c3\\a9dacteur1,ou=system)", controls);
        Assert.assertTrue((boolean)res.hasMore());
        Assert.assertEquals((Object)"groupOfUniqueNames", (Object)res.next().getAttributes().get("cn").get());
        Assert.assertFalse((boolean)res.hasMore());
    }

    @Test
    public void testSearchUserAttributes_1_1() throws Exception {
        LdapContext ctx = (LdapContext)ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer()).lookup(BASE);
        SearchControls ctls = new SearchControls();
        ctls.setSearchScope(0);
        ctls.setReturningAttributes(new String[]{"1.1"});
        NamingEnumeration<SearchResult> result = ctx.search(HEATHER_RDN, FILTER, ctls);
        if (result.hasMore()) {
            SearchResult entry = result.next();
            Assert.assertEquals((String)"No user attributes expected when requesting attribute 1.1", (long)0L, (long)entry.getAttributes().size());
        } else {
            Assert.fail((String)"entry cn=Heather Nova not found");
        }
        result.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testAbandonnedRequest() throws Exception {
        LdapNetworkConnection asyncCnx = new LdapNetworkConnection("localhost", SearchIT.getLdapServer().getPort());
        EntryCursor cursor = null;
        try {
            asyncCnx.bind("uid=admin,ou=system", "secret");
            for (int i = 0; i < 1000; ++i) {
                String dn = "cn=user" + i + "," + BASE;
                DefaultEntry kate = new DefaultEntry(dn);
                kate.add("objectclass", new String[]{"top", "person"});
                kate.add("sn", new String[]{"Bush"});
                kate.add("cn", new String[]{"user" + i});
                asyncCnx.add((Entry)kate);
            }
            cursor = asyncCnx.search(BASE, "(ObjectClass=*)", SearchScope.SUBTREE, new String[]{"*"});
            int count = 0;
            while (cursor.next()) {
                if (++count != 10) continue;
                asyncCnx.abandon(1002);
            }
            Assert.assertEquals((long)10L, (long)count);
        }
        catch (LdapException e) {
            e.printStackTrace();
            Assert.fail((String)"Should not have caught exception.");
        }
        finally {
            asyncCnx.unBind();
            asyncCnx.close();
            cursor.close();
        }
    }

    @Test
    public void testSearchSubstringWithPlus() throws Exception {
        LdapContext ctx = ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer());
        SearchControls controls = new SearchControls();
        controls.setSearchScope(2);
        controls.setTimeLimit(10);
        NamingEnumeration<SearchResult> result = ctx.search(BASE, "(description=*+*)", controls);
        Assert.assertTrue((boolean)result.hasMore());
        SearchResult entry = result.next();
        Assert.assertEquals((Object)"Kim Wilde", (Object)entry.getAttributes().get("cn").get());
    }

    @Test
    public void testSearchSizeLimit() throws Exception {
        long sizeLimit = 7L;
        LdapConnection connection = ServerIntegrationUtils.getAdminConnection((LdapServer)SearchIT.getLdapServer());
        SearchRequestImpl req = new SearchRequestImpl();
        req.setBase(new Dn(new String[]{BASE}));
        req.setFilter("(ou=*)");
        req.setScope(SearchScope.SUBTREE);
        req.setSizeLimit(sizeLimit);
        SearchCursor cursor = connection.search((SearchRequest)req);
        long i = 0L;
        for (Response response : cursor) {
            ++i;
        }
        cursor.close();
        Assert.assertEquals((long)sizeLimit, (long)i);
        connection.close();
    }

    @Test
    @Ignore(value="This test is failing because of the timing issue. Note that the SearchHandler handles time based searches correctly, this is just the below test's problem")
    public void testSearchTimeLimit() throws Exception, InterruptedException {
        LdapConnection connection = ServerIntegrationUtils.getAdminConnection((LdapServer)SearchIT.getLdapServer());
        SearchRequestImpl req = new SearchRequestImpl();
        req.setBase(new Dn(new String[]{"ou=schema"}));
        req.setFilter("(objectClass=*)");
        req.setScope(SearchScope.SUBTREE);
        SearchCursor cursor = connection.search((SearchRequest)req);
        int count = 0;
        while (cursor.next()) {
            ++count;
        }
        cursor.close();
        req.setTimeLimit(1);
        cursor = connection.search((SearchRequest)req);
        int newCount = 0;
        while (cursor.next()) {
            ++newCount;
        }
        Assert.assertTrue((newCount < count ? 1 : 0) != 0);
        connection.close();
    }

    @Test
    public void testSearchComplexFilter() throws Exception {
        LdapContext ctx = ServerIntegrationUtils.getWiredContext((LdapServer)SearchIT.getLdapServer());
        SearchControls controls = new SearchControls();
        controls.setSearchScope(0);
        controls.setTimeLimit(10);
        NamingEnumeration<SearchResult> result = ctx.search("cn=Kim Wilde,ou=system", "(&(&(ObjectClass=person)(!(ObjectClass=strongAuthenticationUser))(sn=Wilde)))", controls);
        Assert.assertTrue((boolean)result.hasMore());
        SearchResult sr = result.next();
        Assert.assertNotNull((Object)sr);
        Assert.assertEquals((Object)"Kim Wilde", (Object)sr.getAttributes().get("cn").get());
        result = ctx.search("cn=Kim Wilde,ou=system", "(&(sn=Wilde)(&(objectClass=person)(!(objectClass=strongAuthenticationUser))))", controls);
        Assert.assertTrue((boolean)result.hasMore());
        sr = result.next();
        Assert.assertNotNull((Object)sr);
        Assert.assertEquals((Object)"Kim Wilde", (Object)sr.getAttributes().get("cn").get());
    }
}

