/*
 * Decompiled with CFR 0.152.
 */
package org.apache.datasketches.kll;

import java.util.Arrays;
import java.util.Random;
import org.apache.datasketches.ByteArrayUtil;
import org.apache.datasketches.Family;
import org.apache.datasketches.SketchesArgumentException;
import org.apache.datasketches.Util;
import org.apache.datasketches.kll.KllFloatsQuantileCalculator;
import org.apache.datasketches.kll.KllFloatsSketchIterator;
import org.apache.datasketches.kll.KllHelper;
import org.apache.datasketches.memory.Memory;

public class KllFloatsSketch {
    public static final int DEFAULT_K = 200;
    private static final int DEFAULT_M = 8;
    static final int MIN_K = 8;
    static final int MAX_K = 65535;
    private static final int PREAMBLE_INTS_BYTE = 0;
    private static final int SER_VER_BYTE = 1;
    private static final int FAMILY_BYTE = 2;
    private static final int FLAGS_BYTE = 3;
    private static final int K_SHORT = 4;
    private static final int M_BYTE = 6;
    private static final int N_LONG = 8;
    private static final int MIN_K_SHORT = 16;
    private static final int NUM_LEVELS_BYTE = 18;
    private static final int DATA_START = 20;
    private static final int DATA_START_SINGLE_ITEM = 8;
    private static final byte serialVersionUID1 = 1;
    private static final byte serialVersionUID2 = 2;
    private static final int PREAMBLE_INTS_SMALL = 2;
    private static final int PREAMBLE_INTS_FULL = 5;
    private final int k_;
    private final int m_;
    private int minK_;
    private long n_;
    private int numLevels_;
    private int[] levels_;
    private boolean isLevelZeroSorted_;
    private float[] items_;
    private float minValue_;
    private float maxValue_;
    private final boolean compatible;
    private static final Random random = new Random();

    public KllFloatsSketch() {
        this(200);
    }

    public KllFloatsSketch(int k) {
        this(k, 8, true);
    }

    KllFloatsSketch(int k, boolean compatible) {
        this(k, 8, compatible);
    }

    private KllFloatsSketch(int k, int m, boolean compatible) {
        KllFloatsSketch.checkK(k);
        this.k_ = k;
        this.minK_ = k;
        this.m_ = m;
        this.numLevels_ = 1;
        this.levels_ = new int[]{k, k};
        this.items_ = new float[k];
        this.minValue_ = Float.NaN;
        this.maxValue_ = Float.NaN;
        this.isLevelZeroSorted_ = false;
        this.compatible = compatible;
    }

    private KllFloatsSketch(Memory mem) {
        this.m_ = 8;
        this.k_ = mem.getShort(4L) & 0xFFFF;
        int flags = mem.getByte(3L) & 0xFF;
        boolean isEmpty = (flags & 1 << Flags.IS_EMPTY.ordinal()) > 0;
        boolean isSingleItem = (flags & 1 << Flags.IS_SINGLE_ITEM.ordinal()) > 0;
        this.compatible = true;
        if (isEmpty) {
            this.numLevels_ = 1;
            this.levels_ = new int[]{this.k_, this.k_};
            this.isLevelZeroSorted_ = false;
            this.minK_ = this.k_;
            this.items_ = new float[this.k_];
            this.minValue_ = Float.NaN;
            this.maxValue_ = Float.NaN;
        } else {
            if (isSingleItem) {
                this.n_ = 1L;
                this.minK_ = this.k_;
                this.numLevels_ = 1;
            } else {
                this.n_ = mem.getLong(8L);
                this.minK_ = mem.getShort(16L) & 0xFFFF;
                this.numLevels_ = mem.getByte(18L) & 0xFF;
            }
            this.levels_ = new int[this.numLevels_ + 1];
            int offset = isSingleItem ? 8 : 20;
            int capacity = KllHelper.computeTotalCapacity(this.k_, this.m_, this.numLevels_);
            if (isSingleItem) {
                this.levels_[0] = capacity - 1;
            } else {
                mem.getIntArray((long)offset, this.levels_, 0, this.numLevels_);
                offset += this.numLevels_ * 4;
            }
            this.levels_[this.numLevels_] = capacity;
            if (!isSingleItem) {
                this.minValue_ = mem.getFloat((long)offset);
                this.maxValue_ = mem.getFloat((long)(offset += 4));
                offset += 4;
            }
            this.items_ = new float[capacity];
            mem.getFloatArray((long)offset, this.items_, this.levels_[0], this.getNumRetained());
            if (isSingleItem) {
                this.minValue_ = this.items_[this.levels_[0]];
                this.maxValue_ = this.items_[this.levels_[0]];
            }
            this.isLevelZeroSorted_ = (flags & 1 << Flags.IS_LEVEL_ZERO_SORTED.ordinal()) > 0;
        }
    }

    public static KllFloatsSketch heapify(Memory mem) {
        boolean isSingleItem;
        int preambleInts = mem.getByte(0L) & 0xFF;
        int serialVersion = mem.getByte(1L) & 0xFF;
        int family = mem.getByte(2L) & 0xFF;
        int flags = mem.getByte(3L) & 0xFF;
        int m = mem.getByte(6L) & 0xFF;
        if (m != 8) {
            throw new SketchesArgumentException("Possible corruption: M must be 8: " + m);
        }
        boolean isEmpty = (flags & 1 << Flags.IS_EMPTY.ordinal()) > 0;
        boolean bl = isSingleItem = (flags & 1 << Flags.IS_SINGLE_ITEM.ordinal()) > 0;
        if (isEmpty || isSingleItem) {
            if (preambleInts != 2) {
                throw new SketchesArgumentException("Possible corruption: preambleInts must be 2 for an empty or single item sketch: " + preambleInts);
            }
        } else if (preambleInts != 5) {
            throw new SketchesArgumentException("Possible corruption: preambleInts must be 5 for a sketch with more than one item: " + preambleInts);
        }
        if (serialVersion != 1 && serialVersion != 2) {
            throw new SketchesArgumentException("Possible corruption: serial version mismatch: expected 1 or 2, got " + serialVersion);
        }
        if (family != Family.KLL.getID()) {
            throw new SketchesArgumentException("Possible corruption: family mismatch: expected " + Family.KLL.getID() + ", got " + family);
        }
        return new KllFloatsSketch(mem);
    }

    public double[] getCDF(float[] splitPoints) {
        return this.getPmfOrCdf(splitPoints, true);
    }

    public int getK() {
        return this.k_;
    }

    public static int getKFromEpsilon(double epsilon, boolean pmf) {
        double eps = Math.max(epsilon, 4.7634E-5);
        double kdbl = pmf ? Math.exp(Math.log(2.446 / eps) / 0.9433) : Math.exp(Math.log(2.296 / eps) / 0.9723);
        double krnd = Math.round(kdbl);
        double del = Math.abs(krnd - kdbl);
        int k = (int)(del < 1.0E-6 ? krnd : Math.ceil(kdbl));
        return Math.max(8, Math.min(65535, k));
    }

    public float getMaxValue() {
        return this.maxValue_;
    }

    public float getMinValue() {
        return this.minValue_;
    }

    public long getN() {
        return this.n_;
    }

    public double getNormalizedRankError(boolean pmf) {
        return KllFloatsSketch.getNormalizedRankError(this.minK_, pmf);
    }

    public static double getNormalizedRankError(int k, boolean pmf) {
        return pmf ? 2.446 / Math.pow(k, 0.9433) : 2.296 / Math.pow(k, 0.9723);
    }

    public int getNumRetained() {
        return this.levels_[this.numLevels_] - this.levels_[0];
    }

    public static int getMaxSerializedSizeBytes(int k, long n) {
        int numLevels = KllHelper.ubOnNumLevels(n);
        int maxNumItems = KllHelper.computeTotalCapacity(k, 8, numLevels);
        return KllFloatsSketch.getSerializedSizeBytes(numLevels, maxNumItems);
    }

    public double[] getPMF(float[] splitPoints) {
        return this.getPmfOrCdf(splitPoints, false);
    }

    public float getQuantile(double fraction) {
        if (this.isEmpty()) {
            return Float.NaN;
        }
        if (this.compatible) {
            if (fraction == 0.0) {
                return this.minValue_;
            }
            if (fraction == 1.0) {
                return this.maxValue_;
            }
        }
        if (fraction < 0.0 || fraction > 1.0) {
            throw new SketchesArgumentException("Fraction cannot be less than zero or greater than 1.0");
        }
        KllFloatsQuantileCalculator quant = this.getQuantileCalculator();
        return quant.getQuantile(fraction);
    }

    public float getQuantileUpperBound(double fraction) {
        return this.getQuantile(Math.min(1.0, fraction + KllFloatsSketch.getNormalizedRankError(this.minK_, false)));
    }

    public float getQuantileLowerBound(double fraction) {
        return this.getQuantile(Math.max(0.0, fraction - KllFloatsSketch.getNormalizedRankError(this.minK_, false)));
    }

    public float[] getQuantiles(double[] fractions) {
        if (this.isEmpty()) {
            return null;
        }
        KllFloatsQuantileCalculator quant = null;
        float[] quantiles = new float[fractions.length];
        for (int i = 0; i < fractions.length; ++i) {
            double fraction = fractions[i];
            if (fraction < 0.0 || fraction > 1.0) {
                throw new SketchesArgumentException("Fraction cannot be less than zero or greater than 1.0");
            }
            if (fraction == 0.0 && this.compatible) {
                quantiles[i] = this.minValue_;
                continue;
            }
            if (fraction == 1.0 && this.compatible) {
                quantiles[i] = this.maxValue_;
                continue;
            }
            if (quant == null) {
                quant = this.getQuantileCalculator();
            }
            quantiles[i] = quant.getQuantile(fraction);
        }
        return quantiles;
    }

    public float[] getQuantiles(int numEvenlySpaced) {
        if (this.isEmpty()) {
            return null;
        }
        return this.getQuantiles(Util.evenlySpaced(0.0, 1.0, numEvenlySpaced));
    }

    public double getRank(float value) {
        if (this.isEmpty()) {
            return Double.NaN;
        }
        int level = 0;
        int weight = 1;
        long total = 0L;
        while (level < this.numLevels_) {
            int fromIndex = this.levels_[level];
            int toIndex = this.levels_[level + 1];
            for (int i = fromIndex; i < toIndex; ++i) {
                if (this.items_[i] < value) {
                    total += (long)weight;
                    continue;
                }
                if (level > 0 || this.isLevelZeroSorted_) break;
            }
            ++level;
            weight *= 2;
        }
        return (double)total / (double)this.n_;
    }

    public int getSerializedSizeBytes() {
        if (this.isEmpty()) {
            return 8;
        }
        return KllFloatsSketch.getSerializedSizeBytes(this.numLevels_, this.getNumRetained());
    }

    public boolean isEmpty() {
        return this.n_ == 0L;
    }

    public boolean isEstimationMode() {
        return this.numLevels_ > 1;
    }

    public KllFloatsSketchIterator iterator() {
        return new KllFloatsSketchIterator(this.items_, this.levels_, this.numLevels_);
    }

    public void merge(KllFloatsSketch other) {
        if (other == null || other.isEmpty()) {
            return;
        }
        if (this.m_ != other.m_) {
            throw new SketchesArgumentException("incompatible M: " + this.m_ + " and " + other.m_);
        }
        long finalN = this.n_ + other.n_;
        for (int i = other.levels_[0]; i < other.levels_[1]; ++i) {
            this.update(other.items_[i]);
        }
        if (other.numLevels_ >= 2) {
            this.mergeHigherLevels(other, finalN);
        }
        if (Float.isNaN(this.minValue_) || other.minValue_ < this.minValue_) {
            this.minValue_ = other.minValue_;
        }
        if (Float.isNaN(this.maxValue_) || other.maxValue_ > this.maxValue_) {
            this.maxValue_ = other.maxValue_;
        }
        this.n_ = finalN;
        this.assertCorrectTotalWeight();
        if (other.isEstimationMode()) {
            this.minK_ = Math.min(this.minK_, other.minK_);
        }
    }

    public byte[] toByteArray() {
        byte[] bytes = new byte[this.getSerializedSizeBytes()];
        boolean isSingleItem = this.n_ == 1L;
        bytes[0] = (byte)(this.isEmpty() || isSingleItem ? 2 : 5);
        bytes[1] = isSingleItem ? 2 : 1;
        bytes[2] = (byte)Family.KLL.getID();
        bytes[3] = (byte)((this.isEmpty() ? 1 << Flags.IS_EMPTY.ordinal() : 0) | (this.isLevelZeroSorted_ ? 1 << Flags.IS_LEVEL_ZERO_SORTED.ordinal() : 0) | (isSingleItem ? 1 << Flags.IS_SINGLE_ITEM.ordinal() : 0));
        ByteArrayUtil.putShortLE(bytes, 4, (short)this.k_);
        bytes[6] = (byte)this.m_;
        if (this.isEmpty()) {
            return bytes;
        }
        int offset = 8;
        if (!isSingleItem) {
            ByteArrayUtil.putLongLE(bytes, 8, this.n_);
            ByteArrayUtil.putShortLE(bytes, 16, (short)this.minK_);
            bytes[18] = (byte)this.numLevels_;
            offset = 20;
            for (int i = 0; i < this.numLevels_; ++i) {
                ByteArrayUtil.putIntLE(bytes, offset, this.levels_[i]);
                offset += 4;
            }
            ByteArrayUtil.putFloatLE(bytes, offset, this.minValue_);
            ByteArrayUtil.putFloatLE(bytes, offset += 4, this.maxValue_);
            offset += 4;
        }
        int numItems = this.getNumRetained();
        for (int i = 0; i < numItems; ++i) {
            ByteArrayUtil.putFloatLE(bytes, offset, this.items_[this.levels_[0] + i]);
            offset += 4;
        }
        return bytes;
    }

    public String toString() {
        return this.toString(false, false);
    }

    public String toString(boolean withLevels, boolean withData) {
        String epsPct = String.format("%.3f%%", this.getNormalizedRankError(false) * 100.0);
        String epsPMFPct = String.format("%.3f%%", this.getNormalizedRankError(true) * 100.0);
        StringBuilder sb = new StringBuilder();
        sb.append(Util.LS).append("### KLL sketch summary:").append(Util.LS);
        sb.append("   K                    : ").append(this.k_).append(Util.LS);
        sb.append("   min K                : ").append(this.minK_).append(Util.LS);
        sb.append("   M                    : ").append(this.m_).append(Util.LS);
        sb.append("   N                    : ").append(this.n_).append(Util.LS);
        sb.append("   Epsilon              : ").append(epsPct).append(Util.LS);
        sb.append("   Epsison PMF          : ").append(epsPMFPct).append(Util.LS);
        sb.append("   Empty                : ").append(this.isEmpty()).append(Util.LS);
        sb.append("   Estimation Mode      : ").append(this.isEstimationMode()).append(Util.LS);
        sb.append("   Levels               : ").append(this.numLevels_).append(Util.LS);
        sb.append("   Sorted               : ").append(this.isLevelZeroSorted_).append(Util.LS);
        sb.append("   Buffer Capacity Items: ").append(this.items_.length).append(Util.LS);
        sb.append("   Retained Items       : ").append(this.getNumRetained()).append(Util.LS);
        sb.append("   Storage Bytes        : ").append(this.getSerializedSizeBytes()).append(Util.LS);
        sb.append("   Min Value            : ").append(this.minValue_).append(Util.LS);
        sb.append("   Max Value            : ").append(this.maxValue_).append(Util.LS);
        sb.append("### End sketch summary").append(Util.LS);
        if (withLevels) {
            sb.append("### KLL sketch levels:").append(Util.LS).append("   level, offset: nominal capacity, actual size").append(Util.LS);
            for (int i = 0; i < this.numLevels_; ++i) {
                sb.append("   ").append(i).append(", ").append(this.levels_[i]).append(": ").append(KllHelper.levelCapacity(this.k_, this.numLevels_, i, this.m_)).append(", ").append(this.safeLevelSize(i)).append(Util.LS);
            }
            sb.append("### End sketch levels").append(Util.LS);
        }
        if (withData) {
            sb.append("### KLL sketch data:").append(Util.LS);
            for (int level = 0; level < this.numLevels_; ++level) {
                int fromIndex = this.levels_[level];
                int toIndex = this.levels_[level + 1];
                if (fromIndex < toIndex) {
                    sb.append(" level ").append(level).append(":").append(Util.LS);
                }
                for (int i = fromIndex; i < toIndex; ++i) {
                    sb.append("   ").append(this.items_[i]).append(Util.LS);
                }
            }
            sb.append("### End sketch data").append(Util.LS);
        }
        return sb.toString();
    }

    public void update(float value) {
        if (Float.isNaN(value)) {
            return;
        }
        if (this.isEmpty()) {
            this.minValue_ = value;
            this.maxValue_ = value;
        } else {
            if (value < this.minValue_) {
                this.minValue_ = value;
            }
            if (value > this.maxValue_) {
                this.maxValue_ = value;
            }
        }
        if (this.levels_[0] == 0) {
            this.compressWhileUpdating();
        }
        ++this.n_;
        this.isLevelZeroSorted_ = false;
        int nextPos = this.levels_[0] - 1;
        assert (this.levels_[0] >= 0);
        this.levels_[0] = nextPos;
        this.items_[nextPos] = value;
    }

    private static void checkK(int k) {
        if (k < 8 || k > 65535) {
            throw new SketchesArgumentException("K must be >= 8 and <= 65535: " + k);
        }
    }

    private KllFloatsQuantileCalculator getQuantileCalculator() {
        this.sortLevelZero();
        return new KllFloatsQuantileCalculator(this.items_, this.levels_, this.numLevels_, this.n_);
    }

    private double[] getPmfOrCdf(float[] splitPoints, boolean isCdf) {
        if (this.isEmpty()) {
            return null;
        }
        Util.validateValues(splitPoints);
        double[] buckets = new double[splitPoints.length + 1];
        int level = 0;
        int weight = 1;
        while (level < this.numLevels_) {
            int fromIndex = this.levels_[level];
            int toIndex = this.levels_[level + 1];
            if (level == 0 && !this.isLevelZeroSorted_) {
                this.incrementBucketsUnsortedLevel(fromIndex, toIndex, weight, splitPoints, buckets);
            } else {
                this.incrementBucketsSortedLevel(fromIndex, toIndex, weight, splitPoints, buckets);
            }
            ++level;
            weight *= 2;
        }
        if (isCdf) {
            double subtotal = 0.0;
            for (int i = 0; i < buckets.length; ++i) {
                buckets[i] = (subtotal += buckets[i]) / (double)this.n_;
            }
        } else {
            int i = 0;
            while (i < buckets.length) {
                int n = i++;
                buckets[n] = buckets[n] / (double)this.n_;
            }
        }
        return buckets;
    }

    private void incrementBucketsUnsortedLevel(int fromIndex, int toIndex, int weight, float[] splitPoints, double[] buckets) {
        for (int i = fromIndex; i < toIndex; ++i) {
            int j;
            for (j = 0; j < splitPoints.length && !(this.items_[i] < splitPoints[j]); ++j) {
            }
            int n = j;
            buckets[n] = buckets[n] + (double)weight;
        }
    }

    private void incrementBucketsSortedLevel(int fromIndex, int toIndex, int weight, float[] splitPoints, double[] buckets) {
        int i = fromIndex;
        int j = 0;
        while (i < toIndex && j < splitPoints.length) {
            if (this.items_[i] < splitPoints[j]) {
                int n = j;
                buckets[n] = buckets[n] + (double)weight;
                ++i;
                continue;
            }
            ++j;
        }
        if (j == splitPoints.length) {
            int n = j;
            buckets[n] = buckets[n] + (double)(weight * (toIndex - i));
        }
    }

    private void compressWhileUpdating() {
        int level = this.findLevelToCompact();
        if (level == this.numLevels_ - 1) {
            this.addEmptyTopLevelToCompletelyFullSketch();
        }
        int rawBeg = this.levels_[level];
        int rawLim = this.levels_[level + 1];
        int popAbove = this.levels_[level + 2] - rawLim;
        int rawPop = rawLim - rawBeg;
        boolean oddPop = KllHelper.isOdd(rawPop);
        int adjBeg = oddPop ? rawBeg + 1 : rawBeg;
        int adjPop = oddPop ? rawPop - 1 : rawPop;
        int halfAdjPop = adjPop / 2;
        if (level == 0) {
            Arrays.sort(this.items_, adjBeg, adjBeg + adjPop);
        }
        if (popAbove == 0) {
            KllHelper.randomlyHalveUp(this.items_, adjBeg, adjPop, random);
        } else {
            KllHelper.randomlyHalveDown(this.items_, adjBeg, adjPop, random);
            KllHelper.mergeSortedArrays(this.items_, adjBeg, halfAdjPop, this.items_, rawLim, popAbove, this.items_, adjBeg + halfAdjPop);
        }
        int n = level + 1;
        this.levels_[n] = this.levels_[n] - halfAdjPop;
        if (oddPop) {
            this.levels_[level] = this.levels_[level + 1] - 1;
            this.items_[this.levels_[level]] = this.items_[rawBeg];
        } else {
            this.levels_[level] = this.levels_[level + 1];
        }
        assert (this.levels_[level] == rawBeg + halfAdjPop);
        if (level > 0) {
            int amount = rawBeg - this.levels_[0];
            System.arraycopy(this.items_, this.levels_[0], this.items_, this.levels_[0] + halfAdjPop, amount);
            int lvl = 0;
            while (lvl < level) {
                int n2 = lvl++;
                this.levels_[n2] = this.levels_[n2] + halfAdjPop;
            }
        }
    }

    private int findLevelToCompact() {
        int level = 0;
        while (true) {
            assert (level < this.numLevels_);
            int pop = this.levels_[level + 1] - this.levels_[level];
            int cap = KllHelper.levelCapacity(this.k_, this.numLevels_, level, this.m_);
            if (pop >= cap) {
                return level;
            }
            ++level;
        }
    }

    private void addEmptyTopLevelToCompletelyFullSketch() {
        int curTotalCap = this.levels_[this.numLevels_];
        assert (this.levels_[0] == 0);
        assert (this.items_.length == curTotalCap);
        if (this.levels_.length < this.numLevels_ + 2) {
            this.levels_ = KllHelper.growIntArray(this.levels_, this.numLevels_ + 2);
        }
        int deltaCap = KllHelper.levelCapacity(this.k_, this.numLevels_ + 1, 0, this.m_);
        int newTotalCap = curTotalCap + deltaCap;
        float[] newBuf = new float[newTotalCap];
        System.arraycopy(this.items_, this.levels_[0], newBuf, this.levels_[0] + deltaCap, curTotalCap);
        this.items_ = newBuf;
        int i = 0;
        while (i <= this.numLevels_) {
            int n = i++;
            this.levels_[n] = this.levels_[n] + deltaCap;
        }
        assert (this.levels_[this.numLevels_] == newTotalCap);
        ++this.numLevels_;
        this.levels_[this.numLevels_] = newTotalCap;
    }

    private void sortLevelZero() {
        if (!this.isLevelZeroSorted_) {
            Arrays.sort(this.items_, this.levels_[0], this.levels_[1]);
            this.isLevelZeroSorted_ = true;
        }
    }

    private void mergeHigherLevels(KllFloatsSketch other, long finalN) {
        int tmpSpaceNeeded = this.getNumRetained() + other.getNumRetainedAboveLevelZero();
        float[] workbuf = new float[tmpSpaceNeeded];
        int ub = KllHelper.ubOnNumLevels(finalN);
        int[] worklevels = new int[ub + 2];
        int[] outlevels = new int[ub + 2];
        int provisionalNumLevels = Math.max(this.numLevels_, other.numLevels_);
        this.populateWorkArrays(other, workbuf, worklevels, provisionalNumLevels);
        int[] result = KllHelper.generalCompress(this.k_, this.m_, provisionalNumLevels, workbuf, worklevels, workbuf, outlevels, this.isLevelZeroSorted_, random);
        int finalNumLevels = result[0];
        int finalCapacity = result[1];
        int finalPop = result[2];
        assert (finalNumLevels <= ub);
        float[] newbuf = finalCapacity == this.items_.length ? this.items_ : new float[finalCapacity];
        int freeSpaceAtBottom = finalCapacity - finalPop;
        System.arraycopy(workbuf, outlevels[0], newbuf, freeSpaceAtBottom, finalPop);
        int theShift = freeSpaceAtBottom - outlevels[0];
        if (this.levels_.length < finalNumLevels + 1) {
            this.levels_ = new int[finalNumLevels + 1];
        }
        for (int lvl = 0; lvl < finalNumLevels + 1; ++lvl) {
            this.levels_[lvl] = outlevels[lvl] + theShift;
        }
        this.items_ = newbuf;
        this.numLevels_ = finalNumLevels;
    }

    private void populateWorkArrays(KllFloatsSketch other, float[] workbuf, int[] worklevels, int provisionalNumLevels) {
        worklevels[0] = 0;
        int selfPopZero = this.safeLevelSize(0);
        System.arraycopy(this.items_, this.levels_[0], workbuf, worklevels[0], selfPopZero);
        worklevels[1] = worklevels[0] + selfPopZero;
        for (int lvl = 1; lvl < provisionalNumLevels; ++lvl) {
            int selfPop = this.safeLevelSize(lvl);
            int otherPop = other.safeLevelSize(lvl);
            worklevels[lvl + 1] = worklevels[lvl] + selfPop + otherPop;
            if (selfPop > 0 && otherPop == 0) {
                System.arraycopy(this.items_, this.levels_[lvl], workbuf, worklevels[lvl], selfPop);
                continue;
            }
            if (selfPop == 0 && otherPop > 0) {
                System.arraycopy(other.items_, other.levels_[lvl], workbuf, worklevels[lvl], otherPop);
                continue;
            }
            if (selfPop <= 0 || otherPop <= 0) continue;
            KllHelper.mergeSortedArrays(this.items_, this.levels_[lvl], selfPop, other.items_, other.levels_[lvl], otherPop, workbuf, worklevels[lvl]);
        }
    }

    private int safeLevelSize(int level) {
        if (level >= this.numLevels_) {
            return 0;
        }
        return this.levels_[level + 1] - this.levels_[level];
    }

    private int getNumRetainedAboveLevelZero() {
        if (this.numLevels_ == 1) {
            return 0;
        }
        return this.levels_[this.numLevels_] - this.levels_[1];
    }

    private void assertCorrectTotalWeight() {
        long total = KllHelper.sumTheSampleWeights(this.numLevels_, this.levels_);
        assert (total == this.n_);
    }

    private static int getSerializedSizeBytes(int numLevels, int numRetained) {
        if (numLevels == 1 && numRetained == 1) {
            return 12;
        }
        return 20 + numLevels * 4 + (numRetained + 2) * 4;
    }

    float[] getItems() {
        return this.items_;
    }

    int[] getLevels() {
        return this.levels_;
    }

    int getNumLevels() {
        return this.numLevels_;
    }

    private static enum Flags {
        IS_EMPTY,
        IS_LEVEL_ZERO_SORTED,
        IS_SINGLE_ITEM;

    }
}

