/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.ctakes.constituency.parser.ae;

import java.io.FileNotFoundException;

import org.apache.ctakes.constituency.parser.MaxentParserWrapper;
import org.apache.ctakes.constituency.parser.ParserWrapper;
import org.apache.ctakes.core.resource.FileLocator;
import org.apache.log4j.Logger;
import org.apache.uima.UimaContext;
import org.apache.uima.analysis_engine.AnalysisEngineProcessException;
import org.apache.uima.jcas.JCas;
import org.apache.uima.resource.ResourceInitializationException;
import org.uimafit.component.JCasAnnotator_ImplBase;
import org.uimafit.descriptor.ConfigurationParameter;


public class ConstituencyParser extends JCasAnnotator_ImplBase {
	public static final String PARAM_MODEL_FILENAME = "MODEL_FILENAME";
	public static final String PARAM_USE_POS = "USE_POS";
	
	@ConfigurationParameter(
			name = PARAM_MODEL_FILENAME,
			description = "File containing the opennlp-trained parser model",
			mandatory = false,
			defaultValue = "org/apache/ctakes/constituency/parser/models/sharpacq-3.1.bin"
	) private String modelFilename;
	
	@ConfigurationParameter(
			name = PARAM_USE_POS,
			description = "Whether to use the POS tags generated by cTAKES pos tagger or generate our own",
			mandatory = false,
			defaultValue = "false"
	) private boolean usePos;
	
	private ParserWrapper parser = null;
	private Logger logger = Logger.getLogger(this.getClass());

	@Override
	public void initialize(UimaContext aContext)
			throws ResourceInitializationException {
		super.initialize(aContext);
		try {
			logger.info("Initializing parser...");		
			parser = new MaxentParserWrapper(FileLocator.locateFile(modelFilename).getAbsolutePath(), usePos);
		} catch (FileNotFoundException e) {
			e.printStackTrace();
			logger.error("Error reading parser model file/directory: " + e.getMessage());
			throw new ResourceInitializationException(e);
		}
	}


	@Override
	public void process(JCas jcas) throws AnalysisEngineProcessException {
		parser.createAnnotations(jcas);
	}
}
