/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.cocoon.xml.sax;

import org.xml.sax.Attributes;

/**
 * A helper Class creating SAX Attributes
 *
 * @version $Id: AttributesImpl.java 737106 2009-01-23 17:33:27Z cziegeler $
 */
public class AttributesImpl extends org.xml.sax.helpers.AttributesImpl {

    /**
     * Constructor
     */
    public AttributesImpl() {
        super();
    }

    /**
     *  Constructor
     */
    public AttributesImpl(Attributes attr) {
        super(attr);
    }

	/**
	 * Add an attribute of type CDATA with empty Namespace to the end of the list.
	 *
	 * <p>For the sake of speed, this method does no checking
	 * to see if the attribute is already in the list: that is
	 * the responsibility of the application.</p>
	 *
	 * @param localName The local name.
	 * @param value The attribute value.
	 */
	public void addCDATAAttribute(String localName, String value) {
		addAttribute("", localName, localName, AttributeTypes.CDATA, value);
	}

    /**
     * Add an attribute of type CDATA with the namespace to the end of the list.
     *
     * <p>For the sake of speed, this method does no checking
     * to see if the attribute is already in the list: that is
     * the responsibility of the application.</p>
     *
     * @param namespace The namespace.
     * @param localName The local name.
     * @param value The attribute value.
     */
    public void addCDATAAttribute(String namespace, String localName, String value) {
        addAttribute(namespace, localName, localName, AttributeTypes.CDATA, value);
    }

    /**
	 * Add an attribute of type CDATA to the end of the list.
	 *
	 * <p>For the sake of speed, this method does no checking
	 * to see if the attribute is already in the list: that is
	 * the responsibility of the application.</p>
	 *
	 * @param uri The Namespace URI, or the empty string if
	 *        none is available or Namespace processing is not
	 *        being performed.
	 * @param localName The local name, or the empty string if
	 *        Namespace processing is not being performed.
	 * @param qName The qualified (prefixed) name, or the empty string
	 *        if qualified names are not available.
	 * @param value The attribute value.
	 */
	public void addCDATAAttribute(String uri,
                            		String localName,
                            		String qName,
                            		String value) {
		addAttribute(uri, localName, qName, AttributeTypes.CDATA, value);
	}

    /**
     * Remove an attribute
     */
    public void removeAttribute(String localName) {
        final int index = this.getIndex(localName);
        if ( index != -1 ) {
            this.removeAttribute(index);
        }
    }

    /**
     * Remove an attribute
     */
    public void removeAttribute(String uri, String localName) {
        final int index = this.getIndex(uri, localName);
        if ( index != -1 ) {
            this.removeAttribute(index);
        }
    }

    /**
     * Utility method to update the value of the named attribute. Returns
     * an updated set of attributes instead of modifying the given attribute
     * set as the given value may be read-only.
     *
     * @param attributes original set of attributes
     * @param name attribute name
     * @param value new attribute value
     * @return updated set of attributes
     */
    public static Attributes update(Attributes attributes, String name, String value) {
        AttributesImpl update = new AttributesImpl(attributes);
        int index = update.getIndex(name);
        if (index != -1) {
            update.setValue(index, value);
        } else {
            update.addCDATAAttribute(name, value);
        }
        return update;
    }

    /**
     * Update or add an attribute of type CDATA with empty Namespace.
     *
     * @param localName The local name.
     * @param value The attribute value.
     */
    public void updateCDATAAttribute(String localName, String value) {
        int index = this.getIndex(localName);
        if (index != -1) {
            this.setValue(index, value);
        } else {
            this.addCDATAAttribute(localName, value);
        }
    }
}
