/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.base.Predicates;
import com.google.common.collect.Iterables;
import com.google.common.util.concurrent.RateLimiter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.Memtable;
import org.apache.cassandra.db.PartitionPosition;
import org.apache.cassandra.db.RowIndexEntry;
import org.apache.cassandra.db.lifecycle.SSTableIntervalTree;
import org.apache.cassandra.db.partitions.Partition;
import org.apache.cassandra.db.rows.UnfilteredRowIterator;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.io.util.FileDataInput;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.schema.CompactionParams;
import org.apache.cassandra.utils.AlwaysPresentFilter;
import org.apache.cassandra.utils.OverlapIterator;
import org.apache.cassandra.utils.concurrent.Refs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompactionController
implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(CompactionController.class);
    private static final String NEVER_PURGE_TOMBSTONES_PROPERTY = "cassandra.never_purge_tombstones";
    static final boolean NEVER_PURGE_TOMBSTONES = Boolean.getBoolean("cassandra.never_purge_tombstones");
    public final ColumnFamilyStore cfs;
    private final boolean compactingRepaired;
    private Refs<SSTableReader> overlappingSSTables;
    private OverlapIterator<PartitionPosition, SSTableReader> overlapIterator;
    private final Iterable<SSTableReader> compacting;
    private final RateLimiter limiter;
    private final long minTimestamp;
    final CompactionParams.TombstoneOption tombstoneOption;
    final Map<SSTableReader, FileDataInput> openDataFiles = new HashMap<SSTableReader, FileDataInput>();
    public final int gcBefore;

    protected CompactionController(ColumnFamilyStore cfs, int maxValue) {
        this(cfs, null, maxValue);
    }

    public CompactionController(ColumnFamilyStore cfs, Set<SSTableReader> compacting, int gcBefore) {
        this(cfs, compacting, gcBefore, null, cfs.getCompactionStrategyManager().getCompactionParams().tombstoneOption());
    }

    public CompactionController(ColumnFamilyStore cfs, Set<SSTableReader> compacting, int gcBefore, RateLimiter limiter, CompactionParams.TombstoneOption tombstoneOption) {
        assert (cfs != null);
        this.cfs = cfs;
        this.gcBefore = gcBefore;
        this.compacting = compacting;
        this.limiter = limiter;
        this.compactingRepaired = compacting != null && compacting.stream().allMatch(SSTableReader::isRepaired);
        this.tombstoneOption = tombstoneOption;
        this.minTimestamp = compacting != null && !compacting.isEmpty() ? compacting.stream().mapToLong(SSTableReader::getMinTimestamp).min().getAsLong() : 0L;
        this.refreshOverlaps();
        if (NEVER_PURGE_TOMBSTONES) {
            logger.warn("You are running with -Dcassandra.never_purge_tombstones=true, this is dangerous!");
        }
    }

    public void maybeRefreshOverlaps() {
        if (NEVER_PURGE_TOMBSTONES) {
            logger.debug("not refreshing overlaps - running with -D{}=true", (Object)NEVER_PURGE_TOMBSTONES_PROPERTY);
            return;
        }
        if (this.ignoreOverlaps()) {
            logger.debug("not refreshing overlaps - running with ignoreOverlaps activated");
            return;
        }
        for (SSTableReader reader : this.overlappingSSTables) {
            if (!reader.isMarkedCompacted()) continue;
            this.refreshOverlaps();
            return;
        }
    }

    private void refreshOverlaps() {
        if (NEVER_PURGE_TOMBSTONES) {
            return;
        }
        if (this.overlappingSSTables != null) {
            this.close();
        }
        this.overlappingSSTables = this.compacting == null || this.ignoreOverlaps() ? Refs.tryRef(Collections.emptyList()) : this.cfs.getAndReferenceOverlappingLiveSSTables(this.compacting);
        this.overlapIterator = new OverlapIterator(SSTableIntervalTree.buildIntervals(this.overlappingSSTables));
    }

    public Set<SSTableReader> getFullyExpiredSSTables() {
        return CompactionController.getFullyExpiredSSTables(this.cfs, this.compacting, this.overlappingSSTables, this.gcBefore, this.ignoreOverlaps());
    }

    public static Set<SSTableReader> getFullyExpiredSSTables(ColumnFamilyStore cfStore, Iterable<SSTableReader> compacting, Iterable<SSTableReader> overlapping, int gcBefore, boolean ignoreOverlaps) {
        logger.trace("Checking droppable sstables in {}", (Object)cfStore);
        if (NEVER_PURGE_TOMBSTONES || compacting == null) {
            return Collections.emptySet();
        }
        if (cfStore.getCompactionStrategyManager().onlyPurgeRepairedTombstones() && !Iterables.all(compacting, SSTableReader::isRepaired)) {
            return Collections.emptySet();
        }
        if (ignoreOverlaps) {
            HashSet<SSTableReader> fullyExpired = new HashSet<SSTableReader>();
            for (SSTableReader candidate : compacting) {
                if (candidate.getSSTableMetadata().maxLocalDeletionTime >= gcBefore) continue;
                fullyExpired.add(candidate);
                logger.trace("Dropping overlap ignored expired SSTable {} (maxLocalDeletionTime={}, gcBefore={})", new Object[]{candidate, candidate.getSSTableMetadata().maxLocalDeletionTime, gcBefore});
            }
            return fullyExpired;
        }
        ArrayList<SSTableReader> candidates = new ArrayList<SSTableReader>();
        long minTimestamp = Long.MAX_VALUE;
        for (SSTableReader sstable : overlapping) {
            if (sstable.getSSTableMetadata().maxLocalDeletionTime < gcBefore) continue;
            minTimestamp = Math.min(minTimestamp, sstable.getMinTimestamp());
        }
        for (SSTableReader candidate : compacting) {
            if (candidate.getSSTableMetadata().maxLocalDeletionTime < gcBefore) {
                candidates.add(candidate);
                continue;
            }
            minTimestamp = Math.min(minTimestamp, candidate.getMinTimestamp());
        }
        for (Memtable memtable : cfStore.getTracker().getView().getAllMemtables()) {
            if (memtable.getMinTimestamp() == -1L) continue;
            minTimestamp = Math.min(minTimestamp, memtable.getMinTimestamp());
        }
        Iterator iterator = candidates.iterator();
        while (iterator.hasNext()) {
            SSTableReader candidate;
            candidate = (SSTableReader)iterator.next();
            if (candidate.getMaxTimestamp() >= minTimestamp) {
                iterator.remove();
                continue;
            }
            logger.trace("Dropping expired SSTable {} (maxLocalDeletionTime={}, gcBefore={})", new Object[]{candidate, candidate.getSSTableMetadata().maxLocalDeletionTime, gcBefore});
        }
        return new HashSet<SSTableReader>(candidates);
    }

    public static Set<SSTableReader> getFullyExpiredSSTables(ColumnFamilyStore cfStore, Iterable<SSTableReader> compacting, Iterable<SSTableReader> overlapping, int gcBefore) {
        return CompactionController.getFullyExpiredSSTables(cfStore, compacting, overlapping, gcBefore, false);
    }

    public String getKeyspace() {
        return this.cfs.keyspace.getName();
    }

    public String getColumnFamily() {
        return this.cfs.name;
    }

    public Predicate<Long> getPurgeEvaluator(DecoratedKey key) {
        if (NEVER_PURGE_TOMBSTONES || !this.compactingRepaired()) {
            return time -> false;
        }
        this.overlapIterator.update(key);
        Set<SSTableReader> filteredSSTables = this.overlapIterator.overlaps();
        Iterable<Memtable> memtables = this.cfs.getTracker().getView().getAllMemtables();
        long minTimestampSeen = Long.MAX_VALUE;
        boolean hasTimestamp = false;
        for (SSTableReader sstable : filteredSSTables) {
            if ((!(sstable.getBloomFilter() instanceof AlwaysPresentFilter) || sstable.getPosition((PartitionPosition)key, SSTableReader.Operator.EQ, false) == null) && !sstable.getBloomFilter().isPresent(key)) continue;
            minTimestampSeen = Math.min(minTimestampSeen, sstable.getMinTimestamp());
            hasTimestamp = true;
        }
        for (Memtable memtable : memtables) {
            Partition partition;
            if (memtable.getMinTimestamp() == -1L || (partition = memtable.getPartition(key)) == null) continue;
            minTimestampSeen = Math.min(minTimestampSeen, partition.stats().minTimestamp);
            hasTimestamp = true;
        }
        if (!hasTimestamp) {
            return time -> true;
        }
        long finalTimestamp = minTimestampSeen;
        return time -> time < finalTimestamp;
    }

    @Override
    public void close() {
        if (this.overlappingSSTables != null) {
            this.overlappingSSTables.release();
        }
        FileUtils.closeQuietly(this.openDataFiles.values());
        this.openDataFiles.clear();
    }

    public boolean compactingRepaired() {
        return !this.cfs.getCompactionStrategyManager().onlyPurgeRepairedTombstones() || this.compactingRepaired;
    }

    boolean provideTombstoneSources() {
        return this.tombstoneOption != CompactionParams.TombstoneOption.NONE;
    }

    public Iterable<UnfilteredRowIterator> shadowSources(DecoratedKey key, boolean tombstoneOnly) {
        if (!this.provideTombstoneSources() || !this.compactingRepaired() || NEVER_PURGE_TOMBSTONES) {
            return null;
        }
        this.overlapIterator.update(key);
        return Iterables.filter((Iterable)Iterables.transform(this.overlapIterator.overlaps(), reader -> this.getShadowIterator((SSTableReader)reader, key, tombstoneOnly)), (com.google.common.base.Predicate)Predicates.notNull());
    }

    private UnfilteredRowIterator getShadowIterator(SSTableReader reader, DecoratedKey key, boolean tombstoneOnly) {
        if (reader.isMarkedSuspect() || reader.getMaxTimestamp() <= this.minTimestamp || tombstoneOnly && !reader.mayHaveTombstones()) {
            return null;
        }
        RowIndexEntry position = reader.getPosition(key, SSTableReader.Operator.EQ);
        if (position == null) {
            return null;
        }
        FileDataInput dfile = this.openDataFiles.computeIfAbsent(reader, this::openDataFile);
        return reader.simpleIterator(dfile, key, position, tombstoneOnly);
    }

    protected boolean ignoreOverlaps() {
        return false;
    }

    private FileDataInput openDataFile(SSTableReader reader) {
        return this.limiter != null ? reader.openDataReader(this.limiter) : reader.openDataReader();
    }
}

