/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import com.google.common.collect.PeekingIterator;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.compaction.AbstractCompactionStrategy;
import org.apache.cassandra.db.compaction.AbstractCompactionTask;
import org.apache.cassandra.db.compaction.CompactionController;
import org.apache.cassandra.db.compaction.CompactionTask;
import org.apache.cassandra.db.compaction.DateTieredCompactionStrategyOptions;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.db.lifecycle.LifecycleTransaction;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DateTieredCompactionStrategy
extends AbstractCompactionStrategy {
    private static final Logger logger = LoggerFactory.getLogger(DateTieredCompactionStrategy.class);
    private final DateTieredCompactionStrategyOptions options;
    protected volatile int estimatedRemainingTasks = 0;
    private final Set<SSTableReader> sstables = new HashSet<SSTableReader>();
    private long lastExpiredCheck;

    public DateTieredCompactionStrategy(ColumnFamilyStore cfs, Map<String, String> options) {
        super(cfs, options);
        this.options = new DateTieredCompactionStrategyOptions(options);
        if (!options.containsKey("tombstone_compaction_interval") && !options.containsKey("tombstone_threshold")) {
            this.disableTombstoneCompactions = true;
            logger.trace("Disabling tombstone compactions for DTCS");
        } else {
            logger.trace("Enabling tombstone compactions for DTCS");
        }
    }

    @Override
    public synchronized AbstractCompactionTask getNextBackgroundTask(int gcBefore) {
        List<SSTableReader> latestBucket;
        LifecycleTransaction modifier;
        do {
            if (!(latestBucket = this.getNextBackgroundSSTables(gcBefore)).isEmpty()) continue;
            return null;
        } while ((modifier = this.cfs.getTracker().tryModify(latestBucket, OperationType.COMPACTION)) == null);
        return new CompactionTask(this.cfs, modifier, gcBefore, false);
    }

    private List<SSTableReader> getNextBackgroundSSTables(int gcBefore) {
        if (this.cfs.getSSTables().isEmpty()) {
            return Collections.emptyList();
        }
        Sets.SetView uncompacting = Sets.intersection(this.sstables, this.cfs.getUncompactingSSTables());
        Set<Object> expired = Collections.emptySet();
        if (System.currentTimeMillis() - this.lastExpiredCheck > this.options.expiredSSTableCheckFrequency) {
            expired = CompactionController.getFullyExpiredSSTables(this.cfs, (Iterable<SSTableReader>)uncompacting, this.cfs.getOverlappingSSTables((Iterable<SSTableReader>)uncompacting), gcBefore);
            this.lastExpiredCheck = System.currentTimeMillis();
        }
        HashSet candidates = Sets.newHashSet(DateTieredCompactionStrategy.filterSuspectSSTables((Iterable<SSTableReader>)uncompacting));
        ArrayList<SSTableReader> compactionCandidates = new ArrayList<SSTableReader>(this.getNextNonExpiredSSTables((Iterable<SSTableReader>)Sets.difference((Set)candidates, expired), gcBefore));
        if (!expired.isEmpty()) {
            logger.trace("Including expired sstables: {}", expired);
            compactionCandidates.addAll(expired);
        }
        return compactionCandidates;
    }

    private List<SSTableReader> getNextNonExpiredSSTables(Iterable<SSTableReader> nonExpiringSSTables, int gcBefore) {
        int base = this.cfs.getMinimumCompactionThreshold();
        long now = this.getNow();
        List<SSTableReader> mostInteresting = this.getCompactionCandidates(nonExpiringSSTables, now, base);
        if (mostInteresting != null) {
            return mostInteresting;
        }
        ArrayList sstablesWithTombstones = Lists.newArrayList();
        for (SSTableReader sstable : nonExpiringSSTables) {
            if (!this.worthDroppingTombstones(sstable, gcBefore)) continue;
            sstablesWithTombstones.add(sstable);
        }
        if (sstablesWithTombstones.isEmpty()) {
            return Collections.emptyList();
        }
        return Collections.singletonList(Collections.min(sstablesWithTombstones, new SSTableReader.SizeComparator()));
    }

    private List<SSTableReader> getCompactionCandidates(Iterable<SSTableReader> candidateSSTables, long now, int base) {
        Iterable<SSTableReader> candidates = DateTieredCompactionStrategy.filterOldSSTables(Lists.newArrayList(candidateSSTables), this.options.maxSSTableAge, now);
        List<List<SSTableReader>> buckets = DateTieredCompactionStrategy.getBuckets(DateTieredCompactionStrategy.createSSTableAndMinTimestampPairs(candidates), this.options.baseTime, base, now);
        logger.trace("Compaction buckets are {}", buckets);
        this.updateEstimatedCompactionsByTasks(buckets);
        List<SSTableReader> mostInteresting = DateTieredCompactionStrategy.newestBucket(buckets, this.cfs.getMinimumCompactionThreshold(), this.cfs.getMaximumCompactionThreshold(), now, this.options.baseTime);
        if (!mostInteresting.isEmpty()) {
            return mostInteresting;
        }
        return null;
    }

    private long getNow() {
        return Collections.max(this.cfs.getSSTables(), new Comparator<SSTableReader>(){

            @Override
            public int compare(SSTableReader o1, SSTableReader o2) {
                return Long.compare(o1.getMaxTimestamp(), o2.getMaxTimestamp());
            }
        }).getMaxTimestamp();
    }

    @VisibleForTesting
    static Iterable<SSTableReader> filterOldSSTables(List<SSTableReader> sstables, long maxSSTableAge, long now) {
        if (maxSSTableAge == 0L) {
            return sstables;
        }
        final long cutoff = now - maxSSTableAge;
        return Iterables.filter(sstables, (Predicate)new Predicate<SSTableReader>(){

            public boolean apply(SSTableReader sstable) {
                return sstable.getMaxTimestamp() >= cutoff;
            }
        });
    }

    public static List<Pair<SSTableReader, Long>> createSSTableAndMinTimestampPairs(Iterable<SSTableReader> sstables) {
        ArrayList sstableMinTimestampPairs = Lists.newArrayListWithCapacity((int)Iterables.size(sstables));
        for (SSTableReader sstable : sstables) {
            sstableMinTimestampPairs.add(Pair.create(sstable, sstable.getMinTimestamp()));
        }
        return sstableMinTimestampPairs;
    }

    @Override
    public void addSSTable(SSTableReader sstable) {
        this.sstables.add(sstable);
    }

    @Override
    public void removeSSTable(SSTableReader sstable) {
        this.sstables.remove(sstable);
    }

    @VisibleForTesting
    static <T> List<List<T>> getBuckets(Collection<Pair<T, Long>> files, long timeUnit, int base, long now) {
        ArrayList sortedFiles = Lists.newArrayList(files);
        Collections.sort(sortedFiles, Collections.reverseOrder(new Comparator<Pair<T, Long>>(){

            @Override
            public int compare(Pair<T, Long> p1, Pair<T, Long> p2) {
                return ((Long)p1.right).compareTo((Long)p2.right);
            }
        }));
        ArrayList buckets = Lists.newArrayList();
        Target target = DateTieredCompactionStrategy.getInitialTarget(now, timeUnit);
        PeekingIterator it = Iterators.peekingIterator(sortedFiles.iterator());
        block0: while (it.hasNext()) {
            while (!target.onTarget((Long)((Pair)it.peek()).right)) {
                if (target.compareToTimestamp((Long)((Pair)it.peek()).right) < 0) {
                    it.next();
                    if (it.hasNext()) continue;
                    break block0;
                }
                target = target.nextTarget(base);
            }
            ArrayList bucket = Lists.newArrayList();
            while (target.onTarget((Long)((Pair)it.peek()).right)) {
                bucket.add(((Pair)it.next()).left);
                if (it.hasNext()) continue;
            }
            buckets.add(bucket);
        }
        return buckets;
    }

    @VisibleForTesting
    static Target getInitialTarget(long now, long timeUnit) {
        return new Target(timeUnit, now / timeUnit);
    }

    private void updateEstimatedCompactionsByTasks(List<List<SSTableReader>> tasks) {
        int n = 0;
        for (List<SSTableReader> bucket : tasks) {
            if (bucket.size() < this.cfs.getMinimumCompactionThreshold()) continue;
            n = (int)((double)n + Math.ceil((double)bucket.size() / (double)this.cfs.getMaximumCompactionThreshold()));
        }
        this.estimatedRemainingTasks = n;
    }

    @VisibleForTesting
    static List<SSTableReader> newestBucket(List<List<SSTableReader>> buckets, int minThreshold, int maxThreshold, long now, long baseTime) {
        Target incomingWindow = DateTieredCompactionStrategy.getInitialTarget(now, baseTime);
        for (List<SSTableReader> bucket : buckets) {
            if (bucket.size() < minThreshold && (bucket.size() < 2 || incomingWindow.onTarget(bucket.get(0).getMinTimestamp()))) continue;
            return DateTieredCompactionStrategy.trimToThreshold(bucket, maxThreshold);
        }
        return Collections.emptyList();
    }

    @VisibleForTesting
    static List<SSTableReader> trimToThreshold(List<SSTableReader> bucket, int maxThreshold) {
        return bucket.subList(0, Math.min(bucket.size(), maxThreshold));
    }

    @Override
    public synchronized Collection<AbstractCompactionTask> getMaximalTask(int gcBefore, boolean splitOutput) {
        LifecycleTransaction modifier = this.cfs.markAllCompacting(OperationType.COMPACTION);
        if (modifier == null) {
            return null;
        }
        return Arrays.asList(new CompactionTask(this.cfs, modifier, gcBefore, false));
    }

    @Override
    public synchronized AbstractCompactionTask getUserDefinedTask(Collection<SSTableReader> sstables, int gcBefore) {
        assert (!sstables.isEmpty());
        LifecycleTransaction modifier = this.cfs.getTracker().tryModify(sstables, OperationType.COMPACTION);
        if (modifier == null) {
            logger.trace("Unable to mark {} for compaction; probably a background compaction got to it first.  You can disable background compactions temporarily if this is a problem", sstables);
            return null;
        }
        return new CompactionTask(this.cfs, modifier, gcBefore, false).setUserDefined(true);
    }

    @Override
    public int getEstimatedRemainingTasks() {
        return this.estimatedRemainingTasks;
    }

    @Override
    public long getMaxSSTableBytes() {
        return Long.MAX_VALUE;
    }

    @Override
    public Collection<Collection<SSTableReader>> groupSSTablesForAntiCompaction(Collection<SSTableReader> sstablesToGroup) {
        ArrayList<Collection<SSTableReader>> groups = new ArrayList<Collection<SSTableReader>>();
        for (SSTableReader sstable : sstablesToGroup) {
            groups.add(Collections.singleton(sstable));
        }
        return groups;
    }

    public static Map<String, String> validateOptions(Map<String, String> options) throws ConfigurationException {
        Map<String, String> uncheckedOptions = AbstractCompactionStrategy.validateOptions(options);
        uncheckedOptions = DateTieredCompactionStrategyOptions.validateOptions(options, uncheckedOptions);
        uncheckedOptions.remove("min_threshold");
        uncheckedOptions.remove("max_threshold");
        return uncheckedOptions;
    }

    public String toString() {
        return String.format("DateTieredCompactionStrategy[%s/%s]", this.cfs.getMinimumCompactionThreshold(), this.cfs.getMaximumCompactionThreshold());
    }

    private static class Target {
        public final long size;
        public final long divPosition;

        public Target(long size, long divPosition) {
            this.size = size;
            this.divPosition = divPosition;
        }

        public int compareToTimestamp(long timestamp) {
            return Long.compare(this.divPosition, timestamp / this.size);
        }

        public boolean onTarget(long timestamp) {
            return this.compareToTimestamp(timestamp) == 0;
        }

        public Target nextTarget(int base) {
            if (this.divPosition % (long)base > 0L) {
                return new Target(this.size, this.divPosition - 1L);
            }
            return new Target(this.size * (long)base, this.divPosition / (long)base - 1L);
        }
    }
}

