/*
 * Decompiled with CFR 0.152.
 */
package org.apache.drill.exec.physical.impl.xsort;

import org.apache.drill.exec.physical.impl.xsort.SortConfig;
import org.apache.drill.shaded.guava.com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SortMemoryManager {
    private static final Logger logger = LoggerFactory.getLogger(SortMemoryManager.class);
    public static final double INTERNAL_FRAGMENTATION_ESTIMATE = 0.25;
    public static final double PAYLOAD_FROM_BUFFER = 0.75;
    public static final double BUFFER_FROM_PAYLOAD = 1.5;
    public static final double WORST_CASE_BUFFER_RATIO = 2.0;
    public static final int MIN_ROWS_PER_SORT_BATCH = 100;
    public static final double LOW_MEMORY_MERGE_BATCH_RATIO = 0.25;
    private final long memoryLimit;
    private int estimatedRowWidth;
    private final BatchSizeEstimate mergeBatchSize = new BatchSizeEstimate();
    private final BatchSizeEstimate inputBatchSize = new BatchSizeEstimate();
    private long bufferMemoryLimit;
    private long mergeMemoryLimit;
    private int preferredMergeBatchSize;
    private int preferredSpillBatchSize;
    private int spillBatchRowCount;
    private final BatchSizeEstimate spillBatchSize = new BatchSizeEstimate();
    private int mergeBatchRowCount;
    private SortConfig config;
    private boolean potentialOverflow;
    private boolean isLowMemory;
    private boolean performanceWarning;

    public SortMemoryManager(SortConfig config, long opMemoryLimit) {
        this.config = config;
        long configMemoryLimit = config.maxMemory();
        this.memoryLimit = configMemoryLimit == 0L ? opMemoryLimit : Math.min(opMemoryLimit, configMemoryLimit);
        this.preferredSpillBatchSize = config.spillBatchSize();
        this.preferredMergeBatchSize = config.mergeBatchSize();
        this.bufferMemoryLimit = (this.memoryLimit - (long)config.spillBatchSize()) / 2L;
        if (this.bufferMemoryLimit < 0L) {
            this.bufferMemoryLimit = this.memoryLimit / 2L;
        }
        if (this.memoryLimit == opMemoryLimit) {
            logger.debug("Memory config: Allocator limit = {}", (Object)this.memoryLimit);
        } else {
            logger.debug("Memory config: Allocator limit = {}, Configured limit: {}", (Object)opMemoryLimit, (Object)this.memoryLimit);
        }
    }

    public boolean updateEstimates(int batchDataSize, int batchRowWidth, int batchRowCount) {
        if (batchRowCount == 0) {
            return false;
        }
        if (!this.updateInputEstimates(batchDataSize, batchRowWidth, batchRowCount)) {
            return false;
        }
        this.updateSpillSettings();
        this.updateMergeSettings();
        this.adjustForLowMemory();
        this.logSettings(batchRowCount);
        return true;
    }

    private boolean updateInputEstimates(int batchDataSize, int batchRowWidth, int batchRowCount) {
        if (batchRowWidth == 0) {
            batchRowWidth = 10;
        }
        int origRowEstimate = this.estimatedRowWidth;
        this.estimatedRowWidth = Math.max(this.estimatedRowWidth, batchRowWidth);
        long origInputBatchSize = this.inputBatchSize.dataSize;
        this.inputBatchSize.setFromData(Math.max(this.inputBatchSize.dataSize, batchDataSize));
        return this.estimatedRowWidth > origRowEstimate || (long)this.inputBatchSize.dataSize > origInputBatchSize;
    }

    private void updateSpillSettings() {
        this.spillBatchRowCount = this.rowsPerBatch(this.preferredSpillBatchSize);
        this.spillBatchRowCount = Math.max(this.spillBatchRowCount, 100);
        this.spillBatchSize.setFromData(this.spillBatchRowCount * this.estimatedRowWidth);
        this.bufferMemoryLimit = this.memoryLimit - (long)(2 * this.spillBatchSize.maxBufferSize);
    }

    private void updateMergeSettings() {
        this.mergeBatchRowCount = this.rowsPerBatch(this.preferredMergeBatchSize);
        this.mergeBatchRowCount = Math.max(this.mergeBatchRowCount, 100);
        this.mergeBatchSize.setFromData(this.mergeBatchRowCount * this.estimatedRowWidth);
        this.mergeMemoryLimit = this.memoryLimit - (long)this.mergeBatchSize.maxBufferSize;
    }

    private void adjustForLowMemory() {
        this.potentialOverflow = false;
        this.performanceWarning = false;
        long loadHeadroom = this.bufferMemoryLimit - (long)(2 * this.inputBatchSize.expectedBufferSize);
        long mergeHeadroom = this.mergeMemoryLimit - (long)(2 * this.spillBatchSize.maxBufferSize);
        this.isLowMemory = loadHeadroom < 0L | mergeHeadroom < 0L;
        if (!this.isLowMemory) {
            return;
        }
        this.lowMemoryInternalBatchSizes();
        long minNeeds = 2 * this.inputBatchSize.expectedBufferSize + this.spillBatchSize.maxBufferSize;
        if (minNeeds > this.memoryLimit) {
            logger.warn("Potential memory overflow during load phase! Minimum needed = {} bytes, actual available = {} bytes", (Object)minNeeds, (Object)this.memoryLimit);
            this.bufferMemoryLimit = 0L;
            this.potentialOverflow = true;
        }
        if ((minNeeds = (long)(2 * this.spillBatchSize.expectedBufferSize + this.mergeBatchSize.expectedBufferSize)) > this.memoryLimit) {
            logger.warn("Potential memory overflow during merge phase! Minimum needed = {} bytes, actual available = {} bytes", (Object)minNeeds, (Object)this.memoryLimit);
            this.mergeMemoryLimit = 0L;
            this.potentialOverflow = true;
        }
        if (this.potentialOverflow) {
            return;
        }
        if (this.spillBatchSize.dataSize < this.config.spillBatchSize() && this.spillBatchRowCount < 65535) {
            logger.warn("Potential performance degredation due to low memory. Preferred spill batch size: {}, actual: {}, rows per batch: {}", new Object[]{this.config.spillBatchSize(), this.spillBatchSize.dataSize, this.spillBatchRowCount});
            this.performanceWarning = true;
        }
        if (this.mergeBatchSize.dataSize < this.config.mergeBatchSize() && this.mergeBatchRowCount < 65535) {
            logger.warn("Potential performance degredation due to low memory. Preferred merge batch size: {}, actual: {}, rows per batch: {}", new Object[]{this.config.mergeBatchSize(), this.mergeBatchSize.dataSize, this.mergeBatchRowCount});
            this.performanceWarning = true;
        }
    }

    private void lowMemoryInternalBatchSizes() {
        int spillBufferSize = (int)(this.memoryLimit - (long)(2 * this.inputBatchSize.maxBufferSize)) / 2;
        spillBufferSize = (int)Math.min((long)spillBufferSize, this.memoryLimit / 4L);
        this.spillBatchSize.setFromWorstCaseBuffer(spillBufferSize);
        int spillDataSize = Math.min(this.spillBatchSize.dataSize, this.config.spillBatchSize());
        spillDataSize = Math.max(spillDataSize, this.estimatedRowWidth);
        if (spillDataSize != this.spillBatchSize.dataSize) {
            this.spillBatchSize.setFromData(spillDataSize);
        }
        this.spillBatchRowCount = this.rowsPerBatch(this.spillBatchSize.dataSize);
        this.bufferMemoryLimit = this.memoryLimit - (long)(2 * this.spillBatchSize.maxBufferSize);
        this.bufferMemoryLimit = Math.max(this.bufferMemoryLimit, 0L);
        long mergeBufferSize = this.memoryLimit - (long)(2 * this.spillBatchSize.maxBufferSize);
        this.mergeBatchSize.setFromBuffer((int)mergeBufferSize);
        int mergeDataSize = Math.min(this.mergeBatchSize.dataSize, this.config.mergeBatchSize());
        mergeDataSize = Math.max(mergeDataSize, this.estimatedRowWidth);
        if (mergeDataSize != this.mergeBatchSize.dataSize) {
            this.mergeBatchSize.setFromData(spillDataSize);
        }
        this.mergeBatchRowCount = this.rowsPerBatch(this.mergeBatchSize.dataSize);
        this.mergeMemoryLimit = Math.max((long)(2 * this.spillBatchSize.expectedBufferSize), this.memoryLimit - (long)this.mergeBatchSize.maxBufferSize);
    }

    private void logSettings(int actualRecordCount) {
        logger.debug("Input Batch Estimates: record size = {} bytes; net = {} bytes, gross = {}, records = {}", new Object[]{this.estimatedRowWidth, this.inputBatchSize.dataSize, this.inputBatchSize.expectedBufferSize, actualRecordCount});
        logger.debug("Spill batch size: net = {} bytes, gross = {} bytes, records = {}; spill file = {} bytes", new Object[]{this.spillBatchSize.dataSize, this.spillBatchSize.expectedBufferSize, this.spillBatchRowCount, this.config.spillFileSize()});
        logger.debug("Output batch size: net = {} bytes, gross = {} bytes, records = {}", new Object[]{this.mergeBatchSize.dataSize, this.mergeBatchSize.expectedBufferSize, this.mergeBatchRowCount});
        logger.debug("Available memory: {}, buffer memory = {}, merge memory = {}", new Object[]{this.memoryLimit, this.bufferMemoryLimit, this.mergeMemoryLimit});
        if (this.spillBatchRowCount < 100) {
            logger.warn("Potential performance degredation due to low memory or large input row. Preferred spill batch row count: {}, actual: {}", (Object)100, (Object)this.spillBatchRowCount);
            this.performanceWarning = true;
        }
        if (this.mergeBatchRowCount < 100) {
            logger.warn("Potential performance degredation due to low memory or large input row. Preferred merge batch row count: {}, actual: {}", (Object)100, (Object)this.mergeBatchRowCount);
            this.performanceWarning = true;
        }
    }

    public MergeTask consolidateBatches(long allocMemory, int inMemCount, int spilledRunsCount) {
        assert (allocMemory == 0L || inMemCount > 0);
        assert (inMemCount + spilledRunsCount > 0);
        if (inMemCount == 0 && spilledRunsCount <= 1) {
            return new MergeTask(MergeAction.NONE, 0);
        }
        if (allocMemory > this.mergeMemoryLimit) {
            return new MergeTask(MergeAction.SPILL, 0);
        }
        int memMergeLimit = (int)((this.mergeMemoryLimit - allocMemory) / (long)this.spillBatchSize.maxBufferSize);
        memMergeLimit = Math.max(0, memMergeLimit);
        if (inMemCount > 0 && (inMemCount + spilledRunsCount > this.config.mergeLimit() || memMergeLimit < spilledRunsCount)) {
            return new MergeTask(MergeAction.SPILL, 0);
        }
        int mergeRunCount = spilledRunsCount - (memMergeLimit = Math.min(memMergeLimit, this.config.mergeLimit()));
        if (mergeRunCount <= 0) {
            return new MergeTask(MergeAction.NONE, 0);
        }
        ++mergeRunCount;
        memMergeLimit = (int)(this.memoryLimit / (long)this.spillBatchSize.maxBufferSize) - 1;
        mergeRunCount = Math.min(mergeRunCount, memMergeLimit);
        mergeRunCount = Math.max(mergeRunCount, 2);
        mergeRunCount = Math.min(mergeRunCount, this.config.mergeLimit());
        return new MergeTask(MergeAction.MERGE, mergeRunCount);
    }

    private int rowsPerBatch(int batchSize) {
        int rowCount = batchSize / this.estimatedRowWidth;
        return Math.max(1, Math.min(rowCount, 65535));
    }

    public static int multiply(int byteSize, double multiplier) {
        return (int)Math.floor((double)byteSize * multiplier);
    }

    public boolean isSpillNeeded(long allocatedBytes, long incomingSize) {
        return allocatedBytes + incomingSize >= this.bufferMemoryLimit;
    }

    public boolean hasMemoryMergeCapacity(long allocatedBytes, long neededForInMemorySort) {
        return this.freeMemory(allocatedBytes) >= neededForInMemorySort;
    }

    public long freeMemory(long allocatedBytes) {
        return this.memoryLimit - allocatedBytes;
    }

    public long getMergeMemoryLimit() {
        return this.mergeMemoryLimit;
    }

    public int getSpillBatchRowCount() {
        return this.spillBatchRowCount;
    }

    public int getMergeBatchRowCount() {
        return this.mergeBatchRowCount;
    }

    @VisibleForTesting
    public long getMemoryLimit() {
        return this.memoryLimit;
    }

    @VisibleForTesting
    public int getRowWidth() {
        return this.estimatedRowWidth;
    }

    @VisibleForTesting
    public BatchSizeEstimate getInputBatchSize() {
        return this.inputBatchSize;
    }

    @VisibleForTesting
    public int getPreferredSpillBatchSize() {
        return this.preferredSpillBatchSize;
    }

    @VisibleForTesting
    public int getPreferredMergeBatchSize() {
        return this.preferredMergeBatchSize;
    }

    @VisibleForTesting
    public BatchSizeEstimate getSpillBatchSize() {
        return this.spillBatchSize;
    }

    @VisibleForTesting
    public BatchSizeEstimate getMergeBatchSize() {
        return this.mergeBatchSize;
    }

    @VisibleForTesting
    public long getBufferMemoryLimit() {
        return this.bufferMemoryLimit;
    }

    @VisibleForTesting
    public boolean mayOverflow() {
        return this.potentialOverflow;
    }

    @VisibleForTesting
    public boolean isLowMemory() {
        return this.isLowMemory;
    }

    @VisibleForTesting
    public boolean hasPerformanceWarning() {
        return this.performanceWarning;
    }

    public static class BatchSizeEstimate {
        int dataSize;
        int expectedBufferSize;
        int maxBufferSize;

        public void setFromData(int dataSize) {
            this.dataSize = dataSize;
            this.expectedBufferSize = SortMemoryManager.multiply(dataSize, 1.5);
            this.maxBufferSize = SortMemoryManager.multiply(dataSize, 2.0);
        }

        public void setFromBuffer(int bufferSize) {
            this.expectedBufferSize = bufferSize;
            this.dataSize = SortMemoryManager.multiply(bufferSize, 0.75);
            this.maxBufferSize = SortMemoryManager.multiply(this.dataSize, 2.0);
        }

        public void setFromWorstCaseBuffer(int bufferSize) {
            this.maxBufferSize = bufferSize;
            this.dataSize = SortMemoryManager.multiply(this.maxBufferSize, 0.5);
            this.expectedBufferSize = SortMemoryManager.multiply(this.dataSize, 1.5);
        }
    }

    public static class MergeTask {
        public MergeAction action;
        public int count;

        public MergeTask(MergeAction action, int count) {
            this.action = action;
            this.count = count;
        }
    }

    public static enum MergeAction {
        SPILL,
        MERGE,
        NONE;

    }
}

