/*
 * Decompiled with CFR 0.152.
 */
package org.apache.drill.exec.physical.impl.xsort;

import java.io.IOException;
import java.util.List;
import org.apache.drill.exec.expr.TypeHelper;
import org.apache.drill.exec.memory.BufferAllocator;
import org.apache.drill.exec.ops.OperatorContext;
import org.apache.drill.exec.physical.impl.xsort.BatchGroup;
import org.apache.drill.exec.physical.impl.xsort.BufferedBatches;
import org.apache.drill.exec.physical.impl.xsort.InputBatch;
import org.apache.drill.exec.physical.impl.xsort.MSortTemplate;
import org.apache.drill.exec.physical.impl.xsort.MergeSortWrapper;
import org.apache.drill.exec.physical.impl.xsort.SortConfig;
import org.apache.drill.exec.physical.impl.xsort.SortMemoryManager;
import org.apache.drill.exec.physical.impl.xsort.SortMetrics;
import org.apache.drill.exec.physical.impl.xsort.SpilledRuns;
import org.apache.drill.exec.record.BatchSchema;
import org.apache.drill.exec.record.MaterializedField;
import org.apache.drill.exec.record.RecordBatch;
import org.apache.drill.exec.record.RecordBatchSizer;
import org.apache.drill.exec.record.VectorAccessible;
import org.apache.drill.exec.record.VectorAccessibleUtilities;
import org.apache.drill.exec.record.VectorContainer;
import org.apache.drill.exec.record.VectorInitializer;
import org.apache.drill.exec.record.VectorWrapper;
import org.apache.drill.exec.record.selection.SelectionVector2;
import org.apache.drill.exec.record.selection.SelectionVector4;
import org.apache.drill.exec.vector.ValueVector;
import org.apache.drill.shaded.guava.com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SortImpl {
    private static final Logger logger = LoggerFactory.getLogger(SortImpl.class);
    private final SortConfig config;
    private final SortMetrics metrics;
    private final SortMemoryManager memManager;
    private final VectorContainer outputBatch;
    private final OperatorContext context;
    private final BufferAllocator allocator;
    private final SpilledRuns spilledRuns;
    private final BufferedBatches bufferedBatches;
    private RecordBatchSizer sizer;
    private VectorInitializer allocHelper;

    public SortImpl(OperatorContext opContext, SortConfig sortConfig, SpilledRuns spilledRuns, VectorContainer batch) {
        this.context = opContext;
        this.outputBatch = batch;
        this.spilledRuns = spilledRuns;
        this.allocator = opContext.getAllocator();
        this.config = sortConfig;
        this.memManager = new SortMemoryManager(this.config, this.allocator.getLimit());
        this.metrics = new SortMetrics(opContext.getStats());
        this.bufferedBatches = new BufferedBatches(opContext);
        boolean allowed = this.allocator.setLenient();
        logger.debug("Config: Is allocator lenient? {}", (Object)allowed);
    }

    @VisibleForTesting
    public OperatorContext opContext() {
        return this.context;
    }

    public void setSchema(BatchSchema schema) {
        this.bufferedBatches.setSchema(schema);
        this.spilledRuns.setSchema(schema);
    }

    public boolean forceSpill() {
        if (this.bufferedBatches.size() < 2) {
            return false;
        }
        this.spillFromMemory();
        return true;
    }

    public void addBatch(VectorAccessible incoming) {
        if (incoming.getRecordCount() == 0) {
            VectorAccessibleUtilities.clear(incoming);
            return;
        }
        this.analyzeIncomingBatch(incoming);
        if (this.isSpillNeeded(this.sizer.getActualSize())) {
            this.spillFromMemory();
        }
        long startMem = this.allocator.getAllocatedMemory();
        this.bufferedBatches.add(incoming, this.sizer.getNetBatchSize());
        long endMem = this.allocator.getAllocatedMemory();
        long batchSize = endMem - startMem;
        this.metrics.updateInputMetrics(this.sizer.rowCount(), this.sizer.getActualSize());
        this.metrics.updateMemory(this.memManager.freeMemory(endMem));
        this.metrics.updatePeakBatches(this.bufferedBatches.size());
        this.validateBatchSize(this.sizer.getActualSize(), batchSize);
        if (this.memManager.updateEstimates((int)batchSize, this.sizer.getNetRowWidth(), this.sizer.rowCount())) {
            this.allocHelper = null;
        }
    }

    private void analyzeIncomingBatch(VectorAccessible incoming) {
        this.sizer = new RecordBatchSizer(incoming);
        this.sizer.applySv2();
        if (this.metrics.getInputBatchCount() == 0L) {
            logger.debug("{}", (Object)this.sizer.toString());
        }
    }

    private boolean isSpillNeeded(long incomingSize) {
        if (this.bufferedBatches.size() >= this.config.getBufferedBatchLimit()) {
            return true;
        }
        boolean spillNeeded = this.memManager.isSpillNeeded(this.allocator.getAllocatedMemory(), incomingSize);
        if (this.bufferedBatches.size() < 2) {
            if (spillNeeded) {
                logger.error("Insufficient memory to merge two batches. Incoming batch size: {}, available memory: {}", (Object)incomingSize, (Object)this.memManager.freeMemory(this.allocator.getAllocatedMemory()));
            }
            return false;
        }
        return spillNeeded;
    }

    private void validateBatchSize(long actualBatchSize, long memoryDelta) {
        if (actualBatchSize != memoryDelta) {
            logger.debug("Memory delta: {}, actual batch size: {}, Diff: {}", new Object[]{memoryDelta, actualBatchSize, memoryDelta - actualBatchSize});
        }
    }

    private void spillFromMemory() {
        int startCount = this.bufferedBatches.size();
        List<BatchGroup> batchesToSpill = this.bufferedBatches.prepareSpill(this.config.spillFileSize());
        logger.trace("Spilling {} of {} batches, allocated memory = {} bytes", new Object[]{batchesToSpill.size(), startCount, this.allocator.getAllocatedMemory()});
        int spillBatchRowCount = this.memManager.getSpillBatchRowCount();
        this.spilledRuns.mergeAndSpill(batchesToSpill, spillBatchRowCount, this.allocHelper());
        this.metrics.incrSpillCount();
    }

    private VectorInitializer allocHelper() {
        if (this.allocHelper == null) {
            this.allocHelper = this.sizer.buildVectorInitializer();
        }
        return this.allocHelper;
    }

    public SortMetrics getMetrics() {
        return this.metrics;
    }

    public SortResults startMerge() {
        if (this.metrics.getInputRowCount() == 0) {
            return new EmptyResults(this.outputBatch);
        }
        logger.debug("Completed load phase: read {} batches, spilled {} times, total input bytes: {}", new Object[]{this.metrics.getInputBatchCount(), this.spilledRuns.size(), this.metrics.getInputBytes()});
        if (this.canUseMemoryMerge()) {
            return this.mergeInMemory();
        }
        return this.mergeSpilledRuns();
    }

    private SortResults singleBatchResult() {
        List<InputBatch> batches = this.bufferedBatches.removeAll();
        return new SingleBatchResults(batches.get(0), this.outputBatch);
    }

    private boolean canUseMemoryMerge() {
        if (this.spilledRuns.hasSpilled()) {
            return false;
        }
        if (!this.memManager.hasMemoryMergeCapacity(this.allocator.getAllocatedMemory(), MSortTemplate.memoryNeeded(this.metrics.getInputRowCount()))) {
            return false;
        }
        return this.bufferedBatches.size() <= 65535;
    }

    private SortResults mergeInMemory() {
        logger.debug("Starting in-memory sort. Batches = {}, Records = {}, Memory = {}", new Object[]{this.bufferedBatches.size(), this.metrics.getInputRowCount(), this.allocator.getAllocatedMemory()});
        MergeSortWrapper memoryMerge = new MergeSortWrapper(this.context, this.outputBatch);
        try {
            memoryMerge.merge(this.bufferedBatches.removeAll(), this.config.getMSortBatchSize());
        }
        catch (Throwable t) {
            memoryMerge.close();
            throw t;
        }
        logger.debug("Completed in-memory sort. Memory = {}", (Object)this.allocator.getAllocatedMemory());
        return memoryMerge;
    }

    private SortResults mergeSpilledRuns() {
        logger.debug("Starting consolidate phase. Batches = {}, Records = {}, Memory = {}, In-memory batches {}, spilled runs {}", new Object[]{this.metrics.getInputBatchCount(), this.metrics.getInputRowCount(), this.allocator.getAllocatedMemory(), this.bufferedBatches.size(), this.spilledRuns.size()});
        block5: while (true) {
            SortMemoryManager.MergeTask task = this.memManager.consolidateBatches(this.allocator.getAllocatedMemory(), this.bufferedBatches.size(), this.spilledRuns.size());
            switch (task.action) {
                case SPILL: {
                    logger.debug("Consolidate: spill");
                    this.spillFromMemory();
                    break;
                }
                case MERGE: {
                    logger.debug("Consolidate: merge {} batches", (Object)task.count);
                    this.mergeRuns(task.count);
                    break;
                }
                case NONE: {
                    break block5;
                }
                default: {
                    throw new IllegalStateException("Unexpected action: " + (Object)((Object)task.action));
                }
            }
        }
        int mergeRowCount = this.memManager.getMergeBatchRowCount();
        return this.spilledRuns.finalMerge(this.bufferedBatches.removeAll(), this.outputBatch, mergeRowCount, this.allocHelper);
    }

    private void mergeRuns(int targetCount) {
        long mergeMemoryPool = this.memManager.getMergeMemoryLimit();
        int spillBatchRowCount = this.memManager.getSpillBatchRowCount();
        this.spilledRuns.mergeRuns(targetCount, mergeMemoryPool, spillBatchRowCount, this.allocHelper);
        this.metrics.incrMergeCount();
    }

    public void close() {
        this.metrics.updateWriteBytes(this.spilledRuns.getWriteBytes());
        RuntimeException ex = null;
        try {
            this.spilledRuns.close();
        }
        catch (RuntimeException e) {
            ex = e;
        }
        try {
            this.bufferedBatches.close();
        }
        catch (RuntimeException e) {
            RuntimeException runtimeException = ex = ex == null ? e : ex;
        }
        if (ex != null) {
            throw ex;
        }
    }

    public String toString() {
        return "SortImpl[config=" + this.config + ", outputBatch=" + this.outputBatch + ", sizer=" + this.sizer + "]";
    }

    public static class EmptyResults
    implements SortResults {
        private final VectorContainer dest;

        public EmptyResults(VectorContainer dest) {
            dest.setRecordCount(0);
            dest.buildSchema(BatchSchema.SelectionVectorMode.NONE);
            this.dest = dest;
        }

        @Override
        public boolean next() {
            return false;
        }

        @Override
        public void close() {
        }

        @Override
        public int getBatchCount() {
            return 0;
        }

        @Override
        public int getRecordCount() {
            return 0;
        }

        @Override
        public SelectionVector4 getSv4() {
            return null;
        }

        @Override
        public SelectionVector2 getSv2() {
            return null;
        }

        @Override
        public VectorContainer getContainer() {
            return this.dest;
        }

        @Override
        public void updateOutputContainer(VectorContainer container, SelectionVector4 sv4, RecordBatch.IterOutcome outcome, BatchSchema schema) {
            if (container.getNumberOfColumns() == 0) {
                for (MaterializedField field : schema) {
                    ValueVector vv = TypeHelper.getNewVector(field, container.getAllocator());
                    vv.clear();
                    ValueVector[] hyperVector = new ValueVector[]{vv};
                    container.add(hyperVector, true);
                }
                container.buildSchema(BatchSchema.SelectionVectorMode.FOUR_BYTE);
            }
            sv4.clear();
            container.zeroVectors();
            container.setRecordCount(0);
        }
    }

    public static interface SortResults {
        public VectorContainer getContainer();

        public boolean next();

        public void close();

        public int getBatchCount();

        public int getRecordCount();

        public SelectionVector2 getSv2();

        public SelectionVector4 getSv4();

        public void updateOutputContainer(VectorContainer var1, SelectionVector4 var2, RecordBatch.IterOutcome var3, BatchSchema var4);
    }

    public static class SingleBatchResults
    implements SortResults {
        private boolean done;
        private final VectorContainer outputContainer;
        private final InputBatch batch;

        public SingleBatchResults(InputBatch batch, VectorContainer outputContainer) {
            this.batch = batch;
            this.outputContainer = outputContainer;
        }

        @Override
        public boolean next() {
            if (this.done) {
                return false;
            }
            for (VectorWrapper<?> vw : this.batch.getContainer()) {
                this.outputContainer.add((ValueVector)vw.getValueVector());
            }
            this.outputContainer.buildSchema(BatchSchema.SelectionVectorMode.TWO_BYTE);
            this.outputContainer.setRecordCount(this.batch.getRecordCount());
            this.done = true;
            return true;
        }

        @Override
        public void close() {
            try {
                this.batch.close();
            }
            catch (IOException e) {
                throw new IllegalStateException(e);
            }
        }

        @Override
        public int getBatchCount() {
            return 1;
        }

        @Override
        public int getRecordCount() {
            return this.outputContainer.getRecordCount();
        }

        @Override
        public SelectionVector4 getSv4() {
            return null;
        }

        @Override
        public SelectionVector2 getSv2() {
            return this.batch.getSv2();
        }

        @Override
        public VectorContainer getContainer() {
            return this.outputContainer;
        }

        @Override
        public void updateOutputContainer(VectorContainer container, SelectionVector4 sv4, RecordBatch.IterOutcome outcome, BatchSchema schema) {
            if (outcome == RecordBatch.IterOutcome.EMIT) {
                throw new UnsupportedOperationException("SingleBatchResults for sort with SV2 is currently not supported with EMIT outcome");
            }
        }
    }
}

