/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.hdfs;

import java.io.Closeable;
import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.security.auth.login.Configuration;
import org.apache.camel.Endpoint;
import org.apache.camel.Exchange;
import org.apache.camel.Expression;
import org.apache.camel.RuntimeCamelException;
import org.apache.camel.component.hdfs.HdfsComponent;
import org.apache.camel.component.hdfs.HdfsConfiguration;
import org.apache.camel.component.hdfs.HdfsEndpoint;
import org.apache.camel.component.hdfs.HdfsHeader;
import org.apache.camel.component.hdfs.HdfsInfoFactory;
import org.apache.camel.component.hdfs.HdfsOutputStream;
import org.apache.camel.support.DefaultProducer;
import org.apache.camel.util.IOHelper;
import org.apache.camel.util.StringHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HdfsProducer
extends DefaultProducer {
    private static final Logger LOG = LoggerFactory.getLogger(HdfsProducer.class);
    private final HdfsConfiguration config;
    private final StringBuilder hdfsPath;
    private final AtomicBoolean idle = new AtomicBoolean();
    private volatile ScheduledExecutorService scheduler;
    private volatile HdfsOutputStream oStream;

    public HdfsProducer(HdfsEndpoint endpoint, HdfsConfiguration config) {
        super((Endpoint)endpoint);
        this.config = config;
        this.hdfsPath = config.getFileSystemType().getHdfsPath(config);
    }

    public HdfsEndpoint getEndpoint() {
        return (HdfsEndpoint)super.getEndpoint();
    }

    protected void doStart() {
        Configuration auth = HdfsComponent.getJAASConfiguration();
        try {
            Optional<SplitStrategy> idleStrategy;
            super.doStart();
            if (this.getEndpoint().getConfig().isConnectOnStartup()) {
                this.oStream = this.setupHdfs(true);
            }
            if ((idleStrategy = this.tryFindIdleStrategy(this.config.getSplitStrategies())).isPresent()) {
                this.scheduler = this.getEndpoint().getCamelContext().getExecutorServiceManager().newSingleThreadScheduledExecutor((Object)this, "HdfsIdleCheck");
                LOG.debug("Creating IdleCheck task scheduled to run every {} millis", (Object)this.config.getCheckIdleInterval());
                this.scheduler.scheduleAtFixedRate(new IdleCheck(idleStrategy.get()), this.config.getCheckIdleInterval(), this.config.getCheckIdleInterval(), TimeUnit.MILLISECONDS);
            }
        }
        catch (Exception e) {
            LOG.warn("Failed to start the HDFS producer. Caused by: [{}]", (Object)e.getMessage());
            LOG.debug("", (Throwable)e);
            throw new RuntimeCamelException((Throwable)e);
        }
        finally {
            HdfsComponent.setJAASConfiguration(auth);
        }
    }

    private synchronized HdfsOutputStream setupHdfs(boolean onStartup) throws IOException {
        if (this.oStream != null) {
            return this.oStream;
        }
        StringBuilder actualPath = new StringBuilder(this.hdfsPath);
        if (this.config.hasSplitStrategies()) {
            actualPath = this.newFileName();
        }
        String hdfsFsDescription = this.config.getFileSystemLabel(actualPath.toString());
        if (onStartup) {
            LOG.info("Connecting to hdfs file-system {} (may take a while if connection is not available)", (Object)hdfsFsDescription);
        } else {
            LOG.debug("Connecting to hdfs file-system {} (may take a while if connection is not available)", (Object)hdfsFsDescription);
        }
        HdfsInfoFactory hdfsInfoFactory = new HdfsInfoFactory(this.config);
        HdfsOutputStream answer = HdfsOutputStream.createOutputStream(actualPath.toString(), hdfsInfoFactory);
        if (onStartup) {
            LOG.info("Connected to hdfs file-system {}", (Object)hdfsFsDescription);
        } else {
            LOG.debug("Connected to hdfs file-system {}", (Object)hdfsFsDescription);
        }
        return answer;
    }

    private Optional<SplitStrategy> tryFindIdleStrategy(List<SplitStrategy> strategies) {
        for (SplitStrategy strategy : strategies) {
            if (strategy.type != SplitStrategyType.IDLE) continue;
            return Optional.of(strategy);
        }
        return Optional.empty();
    }

    protected void doStop() throws Exception {
        super.doStop();
        if (this.scheduler != null) {
            this.getEndpoint().getCamelContext().getExecutorServiceManager().shutdown((ExecutorService)this.scheduler);
            this.scheduler = null;
        }
        if (this.oStream != null) {
            IOHelper.close((Closeable)this.oStream, (String)"output stream", (Logger)LOG);
            this.oStream = null;
        }
    }

    public void process(Exchange exchange) throws Exception {
        Configuration auth = HdfsComponent.getJAASConfiguration();
        try {
            this.doProcess(exchange);
        }
        finally {
            HdfsComponent.setJAASConfiguration(auth);
        }
    }

    void doProcess(Exchange exchange) throws IOException {
        StringBuilder actualPath;
        Object body = exchange.getIn().getBody();
        Object key = exchange.getIn().getHeader(HdfsHeader.KEY.name());
        HdfsInfoFactory hdfsInfoFactory = new HdfsInfoFactory(this.config);
        if (exchange.getIn().getHeader("CamelFileName") != null) {
            if (this.oStream != null) {
                IOHelper.close((Closeable)this.oStream, (String)"output stream", (Logger)LOG);
            }
            actualPath = this.getHdfsPathUsingFileNameHeader(exchange);
            this.oStream = HdfsOutputStream.createOutputStream(actualPath.toString(), hdfsInfoFactory);
        } else if (this.oStream == null) {
            this.oStream = this.setupHdfs(false);
        }
        if (this.isSplitRequired(this.config.getSplitStrategies())) {
            if (this.oStream != null) {
                IOHelper.close((Closeable)this.oStream, (String)"output stream", (Logger)LOG);
            }
            actualPath = this.newFileName();
            this.oStream = HdfsOutputStream.createOutputStream(actualPath.toString(), hdfsInfoFactory);
        }
        String path = this.oStream.getActualPath();
        LOG.trace("Writing body to hdfs-file {}", (Object)path);
        this.oStream.append(key, body, exchange);
        this.idle.set(false);
        boolean close = this.scheduler == null;
        Boolean closeHeader = (Boolean)exchange.getIn().getHeader("CamelHdfsClose", Boolean.class);
        if (closeHeader != null) {
            close = closeHeader;
        }
        if (close) {
            try {
                LOG.trace("Closing stream");
                this.oStream.close();
                this.oStream = null;
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        LOG.debug("Wrote body to hdfs-file {}", (Object)path);
    }

    private StringBuilder getHdfsPathUsingFileNameHeader(Exchange exchange) {
        StringBuilder actualPath = new StringBuilder(this.hdfsPath);
        String fileName = "";
        Object value = exchange.getIn().getHeader("CamelFileName");
        if (value instanceof String) {
            fileName = (String)exchange.getContext().getTypeConverter().convertTo(String.class, exchange, value);
        } else if (value instanceof Expression) {
            fileName = (String)((Expression)value).evaluate(exchange, String.class);
        }
        return actualPath.append(fileName);
    }

    private boolean isSplitRequired(List<SplitStrategy> strategies) {
        boolean split = false;
        for (SplitStrategy splitStrategy : strategies) {
            split |= splitStrategy.getType().split(this.oStream, splitStrategy.value, this);
        }
        return split;
    }

    private StringBuilder newFileName() {
        StringBuilder actualPath = new StringBuilder(this.hdfsPath);
        actualPath.append(StringHelper.sanitize((String)this.getEndpoint().getCamelContext().getUuidGenerator().generateUuid()));
        return actualPath;
    }

    private final class IdleCheck
    implements Runnable {
        private final SplitStrategy strategy;

        private IdleCheck(SplitStrategy strategy) {
            this.strategy = strategy;
        }

        @Override
        public void run() {
            if (HdfsProducer.this.oStream == null) {
                return;
            }
            LOG.trace("IdleCheck running");
            if (System.currentTimeMillis() - HdfsProducer.this.oStream.getLastAccess() > this.strategy.value && !HdfsProducer.this.idle.get() && !HdfsProducer.this.oStream.isBusy().get()) {
                HdfsProducer.this.idle.set(true);
                try {
                    LOG.trace("Closing stream as idle");
                    HdfsProducer.this.oStream.close();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
        }

        public String toString() {
            return "IdleCheck";
        }
    }

    public static enum SplitStrategyType {
        BYTES{

            @Override
            public boolean split(HdfsOutputStream oldOstream, long value, HdfsProducer producer) {
                return oldOstream.getNumOfWrittenBytes() >= value;
            }
        }
        ,
        MESSAGES{

            @Override
            public boolean split(HdfsOutputStream oldOstream, long value, HdfsProducer producer) {
                return oldOstream.getNumOfWrittenMessages() >= value;
            }
        }
        ,
        IDLE{

            @Override
            public boolean split(HdfsOutputStream oldOstream, long value, HdfsProducer producer) {
                return producer.idle.get();
            }
        };


        public abstract boolean split(HdfsOutputStream var1, long var2, HdfsProducer var4);
    }

    public static final class SplitStrategy {
        private SplitStrategyType type;
        private long value;

        public SplitStrategy(SplitStrategyType type, long value) {
            this.type = type;
            this.value = value;
        }

        public SplitStrategyType getType() {
            return this.type;
        }

        public long getValue() {
            return this.value;
        }
    }
}

