/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.test.patterns;

import org.apache.camel.EndpointInject;
import org.apache.camel.Produce;
import org.apache.camel.ProducerTemplate;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.mock.MockEndpoint;
import org.apache.camel.test.junit4.CamelTestSupport;
import org.junit.Test;

public class MockEndpointFailNoHeaderTest extends CamelTestSupport {

    @EndpointInject(uri = "mock:result")
    protected MockEndpoint resultEndpoint;

    @Produce(uri = "direct:start")
    protected ProducerTemplate template;

    @Override
    public boolean isDumpRouteCoverage() {
        return true;
    }

    @Test
    public void withHeaderTestCase() throws InterruptedException {
        String expectedBody = "<matched/>";
        resultEndpoint.expectedHeaderReceived("foo", "bar");
        template.sendBodyAndHeader(expectedBody, "foo", "bar");
        resultEndpoint.assertIsSatisfied();
    }

    @Test
    public void withHeaderAndTwoMessagesTestCase() throws InterruptedException {
        String expectedBody = "<matched/>";
        resultEndpoint.setExpectedCount(2);
        resultEndpoint.expectedHeaderReceived("foo", "bar");
        template.sendBodyAndHeader(expectedBody, "foo", "bar");
        template.sendBodyAndHeader(expectedBody, "foo", "bar");
        resultEndpoint.assertIsSatisfied();
    }

    @Test
    public void withHeaderAndMinTwoMessagesTestCase() throws InterruptedException {
        String expectedBody = "<matched/>";
        resultEndpoint.setMinimumExpectedMessageCount(2);
        resultEndpoint.expectedHeaderReceived("foo", "bar");
        template.sendBodyAndHeader(expectedBody, "foo", "bar");
        template.sendBodyAndHeader(expectedBody, "foo", "bar");
        template.sendBodyAndHeader(expectedBody, "foo", "bar");
        resultEndpoint.assertIsSatisfied();
    }

    @Test
    public void noHeaderTestCase() throws InterruptedException {
        resultEndpoint.expectedHeaderReceived("foo", "bar");
        resultEndpoint.setResultWaitTime(1); // be quicker instead of default 10 sec
        // should fail as there is no message sent
        resultEndpoint.assertIsNotSatisfied();
    }

    @Override
    protected RouteBuilder createRouteBuilder() {
        return new RouteBuilder() {
            public void configure() {
                from("direct:start").filter(header("foo").isEqualTo("bar")).to("mock:result");
            }
        };
    }
}
