/*
 * Decompiled with CFR 0.152.
 */
package org.apache.beam.sdk.io.range;

import java.io.Serializable;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import org.apache.beam.sdk.io.range.ByteKey;
import org.apache.beam.sdk.transforms.splittabledofn.ByteKeyRangeTracker;
import org.apache.beam.sdk.transforms.splittabledofn.HasDefaultTracker;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.base.MoreObjects;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.base.Preconditions;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.base.Verify;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.collect.ImmutableList;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ByteKeyRange
implements Serializable,
HasDefaultTracker<ByteKeyRange, ByteKeyRangeTracker> {
    private static final Logger LOG = LoggerFactory.getLogger(ByteKeyRange.class);
    public static final ByteKeyRange ALL_KEYS = ByteKeyRange.of(ByteKey.EMPTY, ByteKey.EMPTY);
    private final ByteKey startKey;
    private final ByteKey endKey;

    public static ByteKeyRange of(ByteKey startKey, ByteKey endKey) {
        return new ByteKeyRange(startKey, endKey);
    }

    public ByteKey getStartKey() {
        return this.startKey;
    }

    public ByteKey getEndKey() {
        return this.endKey;
    }

    public Boolean containsKey(ByteKey key) {
        return key.compareTo(this.startKey) >= 0 && this.endsAfterKey(key);
    }

    public Boolean overlaps(ByteKeyRange other) {
        return this.endsAfterKey(other.startKey) && other.endsAfterKey(this.startKey);
    }

    public List<ByteKey> split(int numSplits) {
        Preconditions.checkArgument(numSplits > 0, "numSplits %s must be a positive integer", numSplits);
        try {
            ImmutableList.Builder ret = ImmutableList.builder();
            ret.add(this.startKey);
            for (int i = 1; i < numSplits; ++i) {
                ret.add(this.interpolateKey((double)i / (double)numSplits));
            }
            ret.add(this.endKey);
            return ret.build();
        }
        catch (IllegalStateException e) {
            return ImmutableList.of(this.startKey, this.endKey);
        }
    }

    public double estimateFractionForKey(ByteKey key) {
        Preconditions.checkNotNull(key, "key");
        Preconditions.checkArgument(!key.isEmpty(), "Cannot compute fraction for an empty key");
        Preconditions.checkArgument(key.compareTo(this.startKey) >= 0, "Expected key %s >= range start key %s", (Object)key, (Object)this.startKey);
        if (key.equals(this.endKey)) {
            return 1.0;
        }
        Preconditions.checkArgument((boolean)this.containsKey(key), "Cannot compute fraction for %s outside this %s", (Object)key, (Object)this);
        byte[] startBytes = this.startKey.getBytes();
        byte[] endBytes = this.endKey.getBytes();
        byte[] keyBytes = key.getBytes();
        if (this.endKey.isEmpty()) {
            startBytes = ByteKeyRange.addHeadByte(startBytes, (byte)0);
            endBytes = ByteKeyRange.addHeadByte(endBytes, (byte)1);
            keyBytes = ByteKeyRange.addHeadByte(keyBytes, (byte)0);
        }
        int paddedKeyLength = Math.max(Math.max(startBytes.length, endBytes.length), keyBytes.length);
        BigInteger rangeStartInt = ByteKeyRange.paddedPositiveInt(startBytes, paddedKeyLength);
        BigInteger rangeEndInt = ByteKeyRange.paddedPositiveInt(endBytes, paddedKeyLength);
        BigInteger keyInt = ByteKeyRange.paddedPositiveInt(keyBytes, paddedKeyLength);
        BigInteger range = rangeEndInt.subtract(rangeStartInt);
        if (range.equals(BigInteger.ZERO)) {
            LOG.warn("Using 0.0 as the default fraction for this near-empty range {} where start and end keys differ only by trailing zeros.", (Object)this);
            return 0.0;
        }
        BigInteger progressScaled = keyInt.subtract(rangeStartInt).shiftLeft(64);
        return progressScaled.divide(range).doubleValue() / Math.pow(2.0, 64.0);
    }

    public ByteKey interpolateKey(double fraction) {
        Preconditions.checkArgument(fraction >= 0.0 && fraction < 1.0, "Fraction %s must be in the range [0, 1)", (Object)fraction);
        byte[] startBytes = this.startKey.getBytes();
        byte[] endBytes = this.endKey.getBytes();
        if (this.endKey.isEmpty()) {
            startBytes = ByteKeyRange.addHeadByte(startBytes, (byte)0);
            endBytes = ByteKeyRange.addHeadByte(endBytes, (byte)1);
        }
        int paddedKeyLength = Math.max(startBytes.length, endBytes.length);
        BigInteger rangeStartInt = ByteKeyRange.paddedPositiveInt(startBytes, paddedKeyLength);
        BigInteger rangeEndInt = ByteKeyRange.paddedPositiveInt(endBytes, paddedKeyLength);
        BigInteger range = rangeEndInt.subtract(rangeStartInt);
        Preconditions.checkState(!range.equals(BigInteger.ZERO), "Refusing to interpolate for near-empty %s where start and end keys differ only by trailing zero bytes.", (Object)this);
        int bytesNeeded = (53 - range.bitLength() + 7) / 8;
        if (bytesNeeded > 0) {
            range = range.shiftLeft(bytesNeeded * 8);
            rangeStartInt = rangeStartInt.shiftLeft(bytesNeeded * 8);
            paddedKeyLength += bytesNeeded;
        }
        BigInteger interpolatedOffset = new BigDecimal(range).multiply(BigDecimal.valueOf(fraction)).toBigInteger();
        int outputKeyLength = this.endKey.isEmpty() ? paddedKeyLength - 1 : paddedKeyLength;
        return ByteKey.copyFrom(ByteKeyRange.fixupHeadZeros(rangeStartInt.add(interpolatedOffset).toByteArray(), outputKeyLength));
    }

    public ByteKeyRange withStartKey(ByteKey startKey) {
        return new ByteKeyRange(startKey, this.endKey);
    }

    public ByteKeyRange withEndKey(ByteKey endKey) {
        return new ByteKeyRange(this.startKey, endKey);
    }

    private ByteKeyRange(ByteKey startKey, ByteKey endKey) {
        this.startKey = Preconditions.checkNotNull(startKey, "startKey");
        this.endKey = Preconditions.checkNotNull(endKey, "endKey");
        Preconditions.checkArgument(endKey.isEmpty() || startKey.compareTo(endKey) <= 0, "Start %s must be less than or equal to end %s", (Object)startKey, (Object)endKey);
    }

    public String toString() {
        return MoreObjects.toStringHelper(ByteKeyRange.class).add("startKey", this.startKey).add("endKey", this.endKey).toString();
    }

    public boolean equals(@Nullable Object o) {
        if (o == this) {
            return true;
        }
        if (!(o instanceof ByteKeyRange)) {
            return false;
        }
        ByteKeyRange other = (ByteKeyRange)o;
        return Objects.equals(this.startKey, other.startKey) && Objects.equals(this.endKey, other.endKey);
    }

    public int hashCode() {
        return Objects.hash(this.startKey, this.endKey);
    }

    private static byte[] addHeadByte(byte[] array, byte b) {
        byte[] ret = new byte[array.length + 1];
        ret[0] = b;
        System.arraycopy(array, 0, ret, 1, array.length);
        return ret;
    }

    private static byte[] fixupHeadZeros(byte[] array, int size) {
        int padding = size - array.length;
        if (padding == 0) {
            return array;
        }
        if (padding < 0) {
            Verify.verify(padding == -1, "key %s: expected length %s with exactly one byte of padding, found %s", (Object)ByteKey.copyFrom(array), (Object)size, (Object)(-padding));
            Verify.verify(array[0] == 0 && (array[1] & 0x80) == 128, "key %s: is 1 byte longer than expected, indicating BigInteger padding. Expect first byte to be zero with set MSB in second byte.", (Object)ByteKey.copyFrom(array));
            return Arrays.copyOfRange(array, 1, array.length);
        }
        byte[] ret = new byte[size];
        System.arraycopy(array, 0, ret, padding, array.length);
        return ret;
    }

    boolean endsAfterKey(ByteKey key) {
        return this.endKey.isEmpty() || key.compareTo(this.endKey) < 0;
    }

    private static BigInteger paddedPositiveInt(byte[] bytes, int length) {
        int bytePaddingNeeded = length - bytes.length;
        Preconditions.checkArgument(bytePaddingNeeded >= 0, "Required bytes.length {} < length {}", bytes.length, length);
        BigInteger ret = new BigInteger(1, bytes);
        return bytePaddingNeeded == 0 ? ret : ret.shiftLeft(8 * bytePaddingNeeded);
    }

    @Override
    public ByteKeyRangeTracker newTracker() {
        return ByteKeyRangeTracker.of(this);
    }
}

