

/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.arrow.vector.complex;


import static org.apache.arrow.util.Preconditions.checkArgument;
import static org.apache.arrow.util.Preconditions.checkState;

import com.google.flatbuffers.FlatBufferBuilder;

import io.netty.buffer.*;

import org.apache.arrow.memory.*;
import org.apache.arrow.util.Preconditions;
import org.apache.arrow.vector.types.Types;
import org.apache.arrow.vector.types.Types.*;
import org.apache.arrow.vector.types.pojo.*;
import org.apache.arrow.vector.types.pojo.ArrowType.*;
import org.apache.arrow.vector.types.*;
import org.apache.arrow.vector.*;
import org.apache.arrow.vector.holders.*;
import org.apache.arrow.vector.util.*;
import org.apache.arrow.vector.complex.*;
import org.apache.arrow.vector.complex.reader.*;
import org.apache.arrow.vector.complex.impl.*;
import org.apache.arrow.vector.complex.writer.*;
import org.apache.arrow.vector.complex.writer.BaseWriter.StructWriter;
import org.apache.arrow.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.arrow.vector.util.JsonStringArrayList;

import java.util.Arrays;
import java.util.Random;
import java.util.List;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;

import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.Period;
import java.time.ZonedDateTime;


import io.netty.buffer.ArrowBuf;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import org.apache.arrow.vector.compare.VectorVisitor;
import org.apache.arrow.vector.complex.impl.ComplexCopier;
import org.apache.arrow.vector.util.CallBack;
import org.apache.arrow.vector.ipc.message.ArrowFieldNode;
import org.apache.arrow.memory.BaseAllocator;
import org.apache.arrow.vector.BaseValueVector;
import org.apache.arrow.vector.util.OversizedAllocationException;
import org.apache.arrow.util.Preconditions;

import static org.apache.arrow.vector.types.UnionMode.Sparse;



/*
 * This class is generated using freemarker and the UnionVector.java template.
 */
@SuppressWarnings("unused")


/**
 * A vector which can hold values of different types. It does so by using a StructVector which contains a vector for each
 * primitive type that is stored. StructVector is used in order to take advantage of its serialization/deserialization methods,
 * as well as the addOrGet method.
 *
 * For performance reasons, UnionVector stores a cached reference to each subtype vector, to avoid having to do the struct lookup
 * each time the vector is accessed.
 * Source code generated using FreeMarker template UnionVector.java
 */
public class UnionVector implements FieldVector {

  private String name;
  private BufferAllocator allocator;
  int valueCount;

  NonNullableStructVector internalStruct;
  protected ArrowBuf typeBuffer;

  private StructVector structVector;
  private ListVector listVector;

  private FieldReader reader;

  private int singleType = 0;
  private ValueVector singleVector;

  private final CallBack callBack;
  private int typeBufferAllocationSizeInBytes;

  private final FieldType fieldType;

  private static final byte TYPE_WIDTH = 1;
  private static final FieldType INTERNAL_STRUCT_TYPE = new FieldType(false /*nullable*/,
      ArrowType.Struct.INSTANCE, null /*dictionary*/, null /*metadata*/);

  public static UnionVector empty(String name, BufferAllocator allocator) {
    FieldType fieldType = FieldType.nullable(new ArrowType.Union(
        UnionMode.Sparse, null));
    return new UnionVector(name, allocator, fieldType, null);
  }

  @Deprecated
  public UnionVector(String name, BufferAllocator allocator, CallBack callBack) {
    this(name, allocator, null, callBack);
  }

  public UnionVector(String name, BufferAllocator allocator, FieldType fieldType, CallBack callBack) {
    this.name = name;
    this.allocator = allocator;
    this.fieldType = fieldType;
    this.internalStruct = new NonNullableStructVector("internal", allocator, INTERNAL_STRUCT_TYPE,
        callBack);
    this.typeBuffer = allocator.getEmpty();
    this.callBack = callBack;
    this.typeBufferAllocationSizeInBytes = BaseValueVector.INITIAL_VALUE_ALLOCATION * TYPE_WIDTH;
  }

  public BufferAllocator getAllocator() {
    return allocator;
  }

  @Override
  public MinorType getMinorType() {
    return MinorType.UNION;
  }

  @Override
  public void initializeChildrenFromFields(List<Field> children) {
    internalStruct.initializeChildrenFromFields(children);
  }

  @Override
  public List<FieldVector> getChildrenFromFields() {
    return internalStruct.getChildrenFromFields();
  }

  @Override
  public void loadFieldBuffers(ArrowFieldNode fieldNode, List<ArrowBuf> ownBuffers) {
    if (ownBuffers.size() != 1) {
      throw new IllegalArgumentException("Illegal buffer count, expected " + 1 + ", got: " + ownBuffers.size());
    }

    ArrowBuf buffer = ownBuffers.get(0);
    typeBuffer.getReferenceManager().release();
    typeBuffer = buffer.getReferenceManager().retain(buffer, allocator);
    typeBufferAllocationSizeInBytes = typeBuffer.capacity();
    this.valueCount = fieldNode.getLength();
  }

  @Override
  public List<ArrowBuf> getFieldBuffers() {
    List<ArrowBuf> result = new ArrayList<>(1);
    setReaderAndWriterIndex();
    result.add(typeBuffer);

    return result;
  }

  private void setReaderAndWriterIndex() {
    typeBuffer.readerIndex(0);
    typeBuffer.writerIndex(valueCount * TYPE_WIDTH);
  }

  @Override
  @Deprecated
  public List<BufferBacked> getFieldInnerVectors() {
     throw new UnsupportedOperationException("There are no inner vectors. Use geFieldBuffers");
  }

  private String fieldName(MinorType type) {
    return type.name().toLowerCase();
  }

  private FieldType fieldType(MinorType type) {
    return FieldType.nullable(type.getType());
  }

  private <T extends FieldVector> T addOrGet(MinorType minorType, Class<T> c) {
    return internalStruct.addOrGet(fieldName(minorType), fieldType(minorType), c);
  }

  @Override
  public long getValidityBufferAddress() {
    return typeBuffer.memoryAddress();
  }

  @Override
  public long getDataBufferAddress() {
    throw new UnsupportedOperationException();
  }

  @Override
  public long getOffsetBufferAddress() {
    throw new UnsupportedOperationException();
  }

  @Override
  public ArrowBuf getValidityBuffer() { return typeBuffer; }

  @Override
  public ArrowBuf getDataBuffer() { throw new UnsupportedOperationException(); }

  @Override
  public ArrowBuf getOffsetBuffer() { throw new UnsupportedOperationException(); }

  public StructVector getStruct() {
    if (structVector == null) {
      int vectorCount = internalStruct.size();
      structVector = addOrGet(MinorType.STRUCT, StructVector.class);
      if (internalStruct.size() > vectorCount) {
        structVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return structVector;
  }

  private TinyIntVector tinyIntVector;

  public TinyIntVector getTinyIntVector() {
    if (tinyIntVector == null) {
      int vectorCount = internalStruct.size();
      tinyIntVector = addOrGet(MinorType.TINYINT, TinyIntVector.class);
      if (internalStruct.size() > vectorCount) {
        tinyIntVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return tinyIntVector;
  }

  private UInt1Vector uInt1Vector;

  public UInt1Vector getUInt1Vector() {
    if (uInt1Vector == null) {
      int vectorCount = internalStruct.size();
      uInt1Vector = addOrGet(MinorType.UINT1, UInt1Vector.class);
      if (internalStruct.size() > vectorCount) {
        uInt1Vector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return uInt1Vector;
  }

  private UInt2Vector uInt2Vector;

  public UInt2Vector getUInt2Vector() {
    if (uInt2Vector == null) {
      int vectorCount = internalStruct.size();
      uInt2Vector = addOrGet(MinorType.UINT2, UInt2Vector.class);
      if (internalStruct.size() > vectorCount) {
        uInt2Vector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return uInt2Vector;
  }

  private SmallIntVector smallIntVector;

  public SmallIntVector getSmallIntVector() {
    if (smallIntVector == null) {
      int vectorCount = internalStruct.size();
      smallIntVector = addOrGet(MinorType.SMALLINT, SmallIntVector.class);
      if (internalStruct.size() > vectorCount) {
        smallIntVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return smallIntVector;
  }

  private IntVector intVector;

  public IntVector getIntVector() {
    if (intVector == null) {
      int vectorCount = internalStruct.size();
      intVector = addOrGet(MinorType.INT, IntVector.class);
      if (internalStruct.size() > vectorCount) {
        intVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return intVector;
  }

  private UInt4Vector uInt4Vector;

  public UInt4Vector getUInt4Vector() {
    if (uInt4Vector == null) {
      int vectorCount = internalStruct.size();
      uInt4Vector = addOrGet(MinorType.UINT4, UInt4Vector.class);
      if (internalStruct.size() > vectorCount) {
        uInt4Vector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return uInt4Vector;
  }

  private Float4Vector float4Vector;

  public Float4Vector getFloat4Vector() {
    if (float4Vector == null) {
      int vectorCount = internalStruct.size();
      float4Vector = addOrGet(MinorType.FLOAT4, Float4Vector.class);
      if (internalStruct.size() > vectorCount) {
        float4Vector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return float4Vector;
  }

  private DateDayVector dateDayVector;

  public DateDayVector getDateDayVector() {
    if (dateDayVector == null) {
      int vectorCount = internalStruct.size();
      dateDayVector = addOrGet(MinorType.DATEDAY, DateDayVector.class);
      if (internalStruct.size() > vectorCount) {
        dateDayVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return dateDayVector;
  }

  private IntervalYearVector intervalYearVector;

  public IntervalYearVector getIntervalYearVector() {
    if (intervalYearVector == null) {
      int vectorCount = internalStruct.size();
      intervalYearVector = addOrGet(MinorType.INTERVALYEAR, IntervalYearVector.class);
      if (internalStruct.size() > vectorCount) {
        intervalYearVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return intervalYearVector;
  }

  private TimeSecVector timeSecVector;

  public TimeSecVector getTimeSecVector() {
    if (timeSecVector == null) {
      int vectorCount = internalStruct.size();
      timeSecVector = addOrGet(MinorType.TIMESEC, TimeSecVector.class);
      if (internalStruct.size() > vectorCount) {
        timeSecVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return timeSecVector;
  }

  private TimeMilliVector timeMilliVector;

  public TimeMilliVector getTimeMilliVector() {
    if (timeMilliVector == null) {
      int vectorCount = internalStruct.size();
      timeMilliVector = addOrGet(MinorType.TIMEMILLI, TimeMilliVector.class);
      if (internalStruct.size() > vectorCount) {
        timeMilliVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return timeMilliVector;
  }

  private BigIntVector bigIntVector;

  public BigIntVector getBigIntVector() {
    if (bigIntVector == null) {
      int vectorCount = internalStruct.size();
      bigIntVector = addOrGet(MinorType.BIGINT, BigIntVector.class);
      if (internalStruct.size() > vectorCount) {
        bigIntVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return bigIntVector;
  }

  private UInt8Vector uInt8Vector;

  public UInt8Vector getUInt8Vector() {
    if (uInt8Vector == null) {
      int vectorCount = internalStruct.size();
      uInt8Vector = addOrGet(MinorType.UINT8, UInt8Vector.class);
      if (internalStruct.size() > vectorCount) {
        uInt8Vector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return uInt8Vector;
  }

  private Float8Vector float8Vector;

  public Float8Vector getFloat8Vector() {
    if (float8Vector == null) {
      int vectorCount = internalStruct.size();
      float8Vector = addOrGet(MinorType.FLOAT8, Float8Vector.class);
      if (internalStruct.size() > vectorCount) {
        float8Vector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return float8Vector;
  }

  private DateMilliVector dateMilliVector;

  public DateMilliVector getDateMilliVector() {
    if (dateMilliVector == null) {
      int vectorCount = internalStruct.size();
      dateMilliVector = addOrGet(MinorType.DATEMILLI, DateMilliVector.class);
      if (internalStruct.size() > vectorCount) {
        dateMilliVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return dateMilliVector;
  }

  private TimeStampSecVector timeStampSecVector;

  public TimeStampSecVector getTimeStampSecVector() {
    if (timeStampSecVector == null) {
      int vectorCount = internalStruct.size();
      timeStampSecVector = addOrGet(MinorType.TIMESTAMPSEC, TimeStampSecVector.class);
      if (internalStruct.size() > vectorCount) {
        timeStampSecVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return timeStampSecVector;
  }

  private TimeStampMilliVector timeStampMilliVector;

  public TimeStampMilliVector getTimeStampMilliVector() {
    if (timeStampMilliVector == null) {
      int vectorCount = internalStruct.size();
      timeStampMilliVector = addOrGet(MinorType.TIMESTAMPMILLI, TimeStampMilliVector.class);
      if (internalStruct.size() > vectorCount) {
        timeStampMilliVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return timeStampMilliVector;
  }

  private TimeStampMicroVector timeStampMicroVector;

  public TimeStampMicroVector getTimeStampMicroVector() {
    if (timeStampMicroVector == null) {
      int vectorCount = internalStruct.size();
      timeStampMicroVector = addOrGet(MinorType.TIMESTAMPMICRO, TimeStampMicroVector.class);
      if (internalStruct.size() > vectorCount) {
        timeStampMicroVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return timeStampMicroVector;
  }

  private TimeStampNanoVector timeStampNanoVector;

  public TimeStampNanoVector getTimeStampNanoVector() {
    if (timeStampNanoVector == null) {
      int vectorCount = internalStruct.size();
      timeStampNanoVector = addOrGet(MinorType.TIMESTAMPNANO, TimeStampNanoVector.class);
      if (internalStruct.size() > vectorCount) {
        timeStampNanoVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return timeStampNanoVector;
  }

  private TimeMicroVector timeMicroVector;

  public TimeMicroVector getTimeMicroVector() {
    if (timeMicroVector == null) {
      int vectorCount = internalStruct.size();
      timeMicroVector = addOrGet(MinorType.TIMEMICRO, TimeMicroVector.class);
      if (internalStruct.size() > vectorCount) {
        timeMicroVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return timeMicroVector;
  }

  private TimeNanoVector timeNanoVector;

  public TimeNanoVector getTimeNanoVector() {
    if (timeNanoVector == null) {
      int vectorCount = internalStruct.size();
      timeNanoVector = addOrGet(MinorType.TIMENANO, TimeNanoVector.class);
      if (internalStruct.size() > vectorCount) {
        timeNanoVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return timeNanoVector;
  }

  private IntervalDayVector intervalDayVector;

  public IntervalDayVector getIntervalDayVector() {
    if (intervalDayVector == null) {
      int vectorCount = internalStruct.size();
      intervalDayVector = addOrGet(MinorType.INTERVALDAY, IntervalDayVector.class);
      if (internalStruct.size() > vectorCount) {
        intervalDayVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return intervalDayVector;
  }

  private VarBinaryVector varBinaryVector;

  public VarBinaryVector getVarBinaryVector() {
    if (varBinaryVector == null) {
      int vectorCount = internalStruct.size();
      varBinaryVector = addOrGet(MinorType.VARBINARY, VarBinaryVector.class);
      if (internalStruct.size() > vectorCount) {
        varBinaryVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return varBinaryVector;
  }

  private VarCharVector varCharVector;

  public VarCharVector getVarCharVector() {
    if (varCharVector == null) {
      int vectorCount = internalStruct.size();
      varCharVector = addOrGet(MinorType.VARCHAR, VarCharVector.class);
      if (internalStruct.size() > vectorCount) {
        varCharVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return varCharVector;
  }

  private BitVector bitVector;

  public BitVector getBitVector() {
    if (bitVector == null) {
      int vectorCount = internalStruct.size();
      bitVector = addOrGet(MinorType.BIT, BitVector.class);
      if (internalStruct.size() > vectorCount) {
        bitVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return bitVector;
  }

  public ListVector getList() {
    if (listVector == null) {
      int vectorCount = internalStruct.size();
      listVector = addOrGet(MinorType.LIST, ListVector.class);
      if (internalStruct.size() > vectorCount) {
        listVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return listVector;
  }

  public int getTypeValue(int index) {
    return typeBuffer.getByte(index * TYPE_WIDTH);
  }

  @Override
  public void allocateNew() throws OutOfMemoryException {
    /* new allocation -- clear the current buffers */
    clear();
    internalStruct.allocateNew();
    try {
      allocateTypeBuffer();
    } catch (Exception e) {
      clear();
      throw e;
    }
  }

  @Override
  public boolean allocateNewSafe() {
    /* new allocation -- clear the current buffers */
    clear();
    boolean safe = internalStruct.allocateNewSafe();
    if (!safe) { return false; }
    try {
      allocateTypeBuffer();
    } catch (Exception e) {
      clear();
      return  false;
    }

    return true;
  }

  private void allocateTypeBuffer() {
    typeBuffer = allocator.buffer(typeBufferAllocationSizeInBytes);
    typeBuffer.readerIndex(0);
    typeBuffer.setZero(0, typeBuffer.capacity());
  }

  @Override
  public void reAlloc() {
    internalStruct.reAlloc();
    reallocTypeBuffer();
  }

  private void reallocTypeBuffer() {
    final int currentBufferCapacity = typeBuffer.capacity();
    long baseSize  = typeBufferAllocationSizeInBytes;

    if (baseSize < (long)currentBufferCapacity) {
      baseSize = (long)currentBufferCapacity;
    }

    long newAllocationSize = baseSize * 2L;
    newAllocationSize = BaseAllocator.nextPowerOfTwo(newAllocationSize);
    assert newAllocationSize >= 1;

    if (newAllocationSize > BaseValueVector.MAX_ALLOCATION_SIZE) {
      throw new OversizedAllocationException("Unable to expand the buffer");
    }

    final ArrowBuf newBuf = allocator.buffer((int)newAllocationSize);
    newBuf.setBytes(0, typeBuffer, 0, currentBufferCapacity);
    newBuf.setZero(currentBufferCapacity, newBuf.capacity() - currentBufferCapacity);
    typeBuffer.getReferenceManager().release(1);
    typeBuffer = newBuf;
    typeBufferAllocationSizeInBytes = (int)newAllocationSize;
  }

  @Override
  public void setInitialCapacity(int numRecords) { }

  @Override
  public int getValueCapacity() {
    return Math.min(getTypeBufferValueCapacity(), internalStruct.getValueCapacity());
  }

  @Override
  public void close() {
    clear();
  }

  @Override
  public void clear() {
    valueCount = 0;
    typeBuffer.getReferenceManager().release();
    typeBuffer = allocator.getEmpty();
    internalStruct.clear();
  }

  @Override
  public void reset() {
    valueCount = 0;
    typeBuffer.setZero(0, typeBuffer.capacity());
    internalStruct.reset();
  }

  @Override
  public Field getField() {
    List<org.apache.arrow.vector.types.pojo.Field> childFields = new ArrayList<>();
    List<FieldVector> children = internalStruct.getChildren();
    int[] typeIds = new int[children.size()];
    for (ValueVector v : children) {
      typeIds[childFields.size()] = v.getMinorType().ordinal();
      childFields.add(v.getField());
    }

    FieldType fieldType;
    if (this.fieldType == null) {
      fieldType = FieldType.nullable(new ArrowType.Union(Sparse, typeIds));
    } else {
      final UnionMode mode = ((ArrowType.Union)this.fieldType.getType()).getMode();
      fieldType = new FieldType(this.fieldType.isNullable(), new ArrowType.Union(mode, typeIds),
          this.fieldType.getDictionary(), this.fieldType.getMetadata());
    }

    return new Field(name, fieldType, childFields);
  }

  @Override
  public TransferPair getTransferPair(BufferAllocator allocator) {
    return getTransferPair(name, allocator);
  }

  @Override
  public TransferPair getTransferPair(String ref, BufferAllocator allocator) {
    return getTransferPair(ref, allocator, null);
  }

  @Override
  public TransferPair getTransferPair(String ref, BufferAllocator allocator, CallBack callBack) {
    return new org.apache.arrow.vector.complex.UnionVector.TransferImpl(ref, allocator, callBack);
  }

  @Override
  public TransferPair makeTransferPair(ValueVector target) {
    return new TransferImpl((UnionVector) target);
  }

  @Override
  public void copyFrom(int inIndex, int outIndex, ValueVector from) {
    Preconditions.checkArgument(this.getMinorType() == from.getMinorType());
    UnionVector fromCast = (UnionVector) from;
    fromCast.getReader().setPosition(inIndex);
    getWriter().setPosition(outIndex);
    ComplexCopier.copy(fromCast.reader, writer);
  }

  @Override
  public void copyFromSafe(int inIndex, int outIndex, ValueVector from) {
    copyFrom(inIndex, outIndex, from);
  }

  public FieldVector addVector(FieldVector v) {
    String name = v.getMinorType().name().toLowerCase();
    Preconditions.checkState(internalStruct.getChild(name) == null, String.format("%s vector already exists", name));
    final FieldVector newVector = internalStruct.addOrGet(name, v.getField().getFieldType(), v.getClass());
    v.makeTransferPair(newVector).transfer();
    internalStruct.putChild(name, newVector);
    if (callBack != null) {
      callBack.doWork();
    }
    return newVector;
  }

  /**
   * Directly put a vector to internalStruct without creating a new one with same type.
   */
  public void directAddVector(FieldVector v) {
    String name = v.getMinorType().name().toLowerCase();
    Preconditions.checkState(internalStruct.getChild(name) == null, String.format("%s vector already exists", name));
    internalStruct.putChild(name, v);
    if (callBack != null) {
      callBack.doWork();
    }
  }

  private class TransferImpl implements TransferPair {
    private final TransferPair internalStructVectorTransferPair;
    private final UnionVector to;

    public TransferImpl(String name, BufferAllocator allocator, CallBack callBack) {
      to = new UnionVector(name, allocator, callBack);
      internalStructVectorTransferPair = internalStruct.makeTransferPair(to.internalStruct);
    }

    public TransferImpl(UnionVector to) {
      this.to = to;
      internalStructVectorTransferPair = internalStruct.makeTransferPair(to.internalStruct);
    }

    @Override
    public void transfer() {
      to.clear();
      final ReferenceManager refManager = typeBuffer.getReferenceManager();
      to.typeBuffer = refManager.transferOwnership(typeBuffer, to.allocator).getTransferredBuffer();
      internalStructVectorTransferPair.transfer();
      to.valueCount = valueCount;
      clear();
    }

    @Override
    public void splitAndTransfer(int startIndex, int length) {
      Preconditions.checkArgument(startIndex + length <= valueCount);
      to.clear();
      internalStructVectorTransferPair.splitAndTransfer(startIndex, length);
      final int startPoint = startIndex * TYPE_WIDTH;
      final int sliceLength = length * TYPE_WIDTH;
      final ArrowBuf slicedBuffer = typeBuffer.slice(startPoint, sliceLength);
      final ReferenceManager refManager = slicedBuffer.getReferenceManager();
      to.typeBuffer = refManager.transferOwnership(slicedBuffer, to.allocator).getTransferredBuffer();
      to.setValueCount(length);
    }

    @Override
    public ValueVector getTo() {
      return to;
    }

    @Override
    public void copyValueSafe(int from, int to) {
      this.to.copyFrom(from, to, UnionVector.this);
    }
  }

  @Override
  public FieldReader getReader() {
    if (reader == null) {
      reader = new UnionReader(this);
    }
    return reader;
  }

  public FieldWriter getWriter() {
    if (writer == null) {
      writer = new UnionWriter(this);
    }
    return writer;
  }

  @Override
  public int getBufferSize() {
    if (valueCount == 0) { return 0; }

    return (valueCount * TYPE_WIDTH) + internalStruct.getBufferSize();
  }

  @Override
  public int getBufferSizeFor(final int valueCount) {
    if (valueCount == 0) {
      return 0;
    }

    long bufferSize = 0;
    for (final ValueVector v : (Iterable<ValueVector>) this) {
      bufferSize += v.getBufferSizeFor(valueCount);
    }

    return (int) bufferSize + (valueCount * TYPE_WIDTH);
  }

  @Override
  public ArrowBuf[] getBuffers(boolean clear) {
    List<ArrowBuf> list = new java.util.ArrayList<>();
    setReaderAndWriterIndex();
    if (getBufferSize() != 0) {
      list.add(typeBuffer);
      list.addAll(java.util.Arrays.asList(internalStruct.getBuffers(clear)));
    }
    if (clear) {
      valueCount = 0;
      typeBuffer.getReferenceManager().retain();
      typeBuffer.getReferenceManager().release();
      typeBuffer = allocator.getEmpty();
    }
    return list.toArray(new ArrowBuf[list.size()]);
  }

  @Override
  public Iterator<ValueVector> iterator() {
    List<ValueVector> vectors = org.apache.arrow.util.Collections2.toList(internalStruct.iterator());
    return vectors.iterator();
  }

    private ValueVector getVector(int index) {
      int type = typeBuffer.getByte(index * TYPE_WIDTH);
      switch (MinorType.values()[type]) {
        case NULL:
          return null;
        case TINYINT:
        return getTinyIntVector();
        case UINT1:
        return getUInt1Vector();
        case UINT2:
        return getUInt2Vector();
        case SMALLINT:
        return getSmallIntVector();
        case INT:
        return getIntVector();
        case UINT4:
        return getUInt4Vector();
        case FLOAT4:
        return getFloat4Vector();
        case DATEDAY:
        return getDateDayVector();
        case INTERVALYEAR:
        return getIntervalYearVector();
        case TIMESEC:
        return getTimeSecVector();
        case TIMEMILLI:
        return getTimeMilliVector();
        case BIGINT:
        return getBigIntVector();
        case UINT8:
        return getUInt8Vector();
        case FLOAT8:
        return getFloat8Vector();
        case DATEMILLI:
        return getDateMilliVector();
        case TIMESTAMPSEC:
        return getTimeStampSecVector();
        case TIMESTAMPMILLI:
        return getTimeStampMilliVector();
        case TIMESTAMPMICRO:
        return getTimeStampMicroVector();
        case TIMESTAMPNANO:
        return getTimeStampNanoVector();
        case TIMEMICRO:
        return getTimeMicroVector();
        case TIMENANO:
        return getTimeNanoVector();
        case INTERVALDAY:
        return getIntervalDayVector();
        case VARBINARY:
        return getVarBinaryVector();
        case VARCHAR:
        return getVarCharVector();
        case BIT:
        return getBitVector();
        case STRUCT:
          return getStruct();
        case LIST:
          return getList();
        default:
          throw new UnsupportedOperationException("Cannot support type: " + MinorType.values()[type]);
      }
    }

    public Object getObject(int index) {
      ValueVector vector = getVector(index);
      if (vector != null) {
        return vector.getObject(index);
      }
      return null;
    }

    public byte[] get(int index) {
      return null;
    }

    public void get(int index, ComplexHolder holder) {
    }

    public void get(int index, UnionHolder holder) {
      FieldReader reader = new UnionReader(UnionVector.this);
      reader.setPosition(index);
      holder.reader = reader;
    }

    public int getValueCount() {
      return valueCount;
    }

    public boolean isNull(int index) {
      return (typeBuffer.getByte(index * TYPE_WIDTH) == 0);
    }

    @Override
    public int getNullCount() {
      int nullCount = 0;
      for (int i = 0; i < getValueCount(); i++) {
        if (isNull(i)) {
          nullCount++;
        }
      }
      return nullCount;
    }

    public int isSet(int index) {
      return isNull(index) ? 0 : 1;
    }

    UnionWriter writer;

    public void setValueCount(int valueCount) {
      this.valueCount = valueCount;
      while (valueCount > getTypeBufferValueCapacity()) {
        reallocTypeBuffer();
      }
      internalStruct.setValueCount(valueCount);
    }

    public void setSafe(int index, UnionHolder holder) {
      FieldReader reader = holder.reader;
      if (writer == null) {
        writer = new UnionWriter(UnionVector.this);
      }
      writer.setPosition(index);
      MinorType type = reader.getMinorType();
      switch (type) {
      case TINYINT:
        NullableTinyIntHolder tinyIntHolder = new NullableTinyIntHolder();
        reader.read(tinyIntHolder);
        setSafe(index, tinyIntHolder);
        break;
      case UINT1:
        NullableUInt1Holder uInt1Holder = new NullableUInt1Holder();
        reader.read(uInt1Holder);
        setSafe(index, uInt1Holder);
        break;
      case UINT2:
        NullableUInt2Holder uInt2Holder = new NullableUInt2Holder();
        reader.read(uInt2Holder);
        setSafe(index, uInt2Holder);
        break;
      case SMALLINT:
        NullableSmallIntHolder smallIntHolder = new NullableSmallIntHolder();
        reader.read(smallIntHolder);
        setSafe(index, smallIntHolder);
        break;
      case INT:
        NullableIntHolder intHolder = new NullableIntHolder();
        reader.read(intHolder);
        setSafe(index, intHolder);
        break;
      case UINT4:
        NullableUInt4Holder uInt4Holder = new NullableUInt4Holder();
        reader.read(uInt4Holder);
        setSafe(index, uInt4Holder);
        break;
      case FLOAT4:
        NullableFloat4Holder float4Holder = new NullableFloat4Holder();
        reader.read(float4Holder);
        setSafe(index, float4Holder);
        break;
      case DATEDAY:
        NullableDateDayHolder dateDayHolder = new NullableDateDayHolder();
        reader.read(dateDayHolder);
        setSafe(index, dateDayHolder);
        break;
      case INTERVALYEAR:
        NullableIntervalYearHolder intervalYearHolder = new NullableIntervalYearHolder();
        reader.read(intervalYearHolder);
        setSafe(index, intervalYearHolder);
        break;
      case TIMESEC:
        NullableTimeSecHolder timeSecHolder = new NullableTimeSecHolder();
        reader.read(timeSecHolder);
        setSafe(index, timeSecHolder);
        break;
      case TIMEMILLI:
        NullableTimeMilliHolder timeMilliHolder = new NullableTimeMilliHolder();
        reader.read(timeMilliHolder);
        setSafe(index, timeMilliHolder);
        break;
      case BIGINT:
        NullableBigIntHolder bigIntHolder = new NullableBigIntHolder();
        reader.read(bigIntHolder);
        setSafe(index, bigIntHolder);
        break;
      case UINT8:
        NullableUInt8Holder uInt8Holder = new NullableUInt8Holder();
        reader.read(uInt8Holder);
        setSafe(index, uInt8Holder);
        break;
      case FLOAT8:
        NullableFloat8Holder float8Holder = new NullableFloat8Holder();
        reader.read(float8Holder);
        setSafe(index, float8Holder);
        break;
      case DATEMILLI:
        NullableDateMilliHolder dateMilliHolder = new NullableDateMilliHolder();
        reader.read(dateMilliHolder);
        setSafe(index, dateMilliHolder);
        break;
      case TIMESTAMPSEC:
        NullableTimeStampSecHolder timeStampSecHolder = new NullableTimeStampSecHolder();
        reader.read(timeStampSecHolder);
        setSafe(index, timeStampSecHolder);
        break;
      case TIMESTAMPMILLI:
        NullableTimeStampMilliHolder timeStampMilliHolder = new NullableTimeStampMilliHolder();
        reader.read(timeStampMilliHolder);
        setSafe(index, timeStampMilliHolder);
        break;
      case TIMESTAMPMICRO:
        NullableTimeStampMicroHolder timeStampMicroHolder = new NullableTimeStampMicroHolder();
        reader.read(timeStampMicroHolder);
        setSafe(index, timeStampMicroHolder);
        break;
      case TIMESTAMPNANO:
        NullableTimeStampNanoHolder timeStampNanoHolder = new NullableTimeStampNanoHolder();
        reader.read(timeStampNanoHolder);
        setSafe(index, timeStampNanoHolder);
        break;
      case TIMEMICRO:
        NullableTimeMicroHolder timeMicroHolder = new NullableTimeMicroHolder();
        reader.read(timeMicroHolder);
        setSafe(index, timeMicroHolder);
        break;
      case TIMENANO:
        NullableTimeNanoHolder timeNanoHolder = new NullableTimeNanoHolder();
        reader.read(timeNanoHolder);
        setSafe(index, timeNanoHolder);
        break;
      case INTERVALDAY:
        NullableIntervalDayHolder intervalDayHolder = new NullableIntervalDayHolder();
        reader.read(intervalDayHolder);
        setSafe(index, intervalDayHolder);
        break;
      case VARBINARY:
        NullableVarBinaryHolder varBinaryHolder = new NullableVarBinaryHolder();
        reader.read(varBinaryHolder);
        setSafe(index, varBinaryHolder);
        break;
      case VARCHAR:
        NullableVarCharHolder varCharHolder = new NullableVarCharHolder();
        reader.read(varCharHolder);
        setSafe(index, varCharHolder);
        break;
      case BIT:
        NullableBitHolder bitHolder = new NullableBitHolder();
        reader.read(bitHolder);
        setSafe(index, bitHolder);
        break;
      case STRUCT: {
        ComplexCopier.copy(reader, writer);
        break;
      }
      case LIST: {
        ComplexCopier.copy(reader, writer);
        break;
      }
      default:
        throw new UnsupportedOperationException();
      }
    }
    public void setSafe(int index, NullableTinyIntHolder holder) {
      setType(index, MinorType.TINYINT);
      getTinyIntVector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableUInt1Holder holder) {
      setType(index, MinorType.UINT1);
      getUInt1Vector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableUInt2Holder holder) {
      setType(index, MinorType.UINT2);
      getUInt2Vector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableSmallIntHolder holder) {
      setType(index, MinorType.SMALLINT);
      getSmallIntVector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableIntHolder holder) {
      setType(index, MinorType.INT);
      getIntVector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableUInt4Holder holder) {
      setType(index, MinorType.UINT4);
      getUInt4Vector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableFloat4Holder holder) {
      setType(index, MinorType.FLOAT4);
      getFloat4Vector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableDateDayHolder holder) {
      setType(index, MinorType.DATEDAY);
      getDateDayVector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableIntervalYearHolder holder) {
      setType(index, MinorType.INTERVALYEAR);
      getIntervalYearVector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableTimeSecHolder holder) {
      setType(index, MinorType.TIMESEC);
      getTimeSecVector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableTimeMilliHolder holder) {
      setType(index, MinorType.TIMEMILLI);
      getTimeMilliVector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableBigIntHolder holder) {
      setType(index, MinorType.BIGINT);
      getBigIntVector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableUInt8Holder holder) {
      setType(index, MinorType.UINT8);
      getUInt8Vector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableFloat8Holder holder) {
      setType(index, MinorType.FLOAT8);
      getFloat8Vector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableDateMilliHolder holder) {
      setType(index, MinorType.DATEMILLI);
      getDateMilliVector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableTimeStampSecHolder holder) {
      setType(index, MinorType.TIMESTAMPSEC);
      getTimeStampSecVector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableTimeStampMilliHolder holder) {
      setType(index, MinorType.TIMESTAMPMILLI);
      getTimeStampMilliVector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableTimeStampMicroHolder holder) {
      setType(index, MinorType.TIMESTAMPMICRO);
      getTimeStampMicroVector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableTimeStampNanoHolder holder) {
      setType(index, MinorType.TIMESTAMPNANO);
      getTimeStampNanoVector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableTimeMicroHolder holder) {
      setType(index, MinorType.TIMEMICRO);
      getTimeMicroVector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableTimeNanoHolder holder) {
      setType(index, MinorType.TIMENANO);
      getTimeNanoVector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableIntervalDayHolder holder) {
      setType(index, MinorType.INTERVALDAY);
      getIntervalDayVector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableVarBinaryHolder holder) {
      setType(index, MinorType.VARBINARY);
      getVarBinaryVector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableVarCharHolder holder) {
      setType(index, MinorType.VARCHAR);
      getVarCharVector().setSafe(index, holder);
    }

    public void setSafe(int index, NullableBitHolder holder) {
      setType(index, MinorType.BIT);
      getBitVector().setSafe(index, holder);
    }


    public void setType(int index, MinorType type) {
      while (index >= getTypeBufferValueCapacity()) {
        reallocTypeBuffer();
      }
      typeBuffer.setByte(index * TYPE_WIDTH , (byte) type.ordinal());
    }

    private int getTypeBufferValueCapacity() {
      return typeBuffer.capacity() / TYPE_WIDTH;
    }

    @Override
    public int hashCode(int index) {
      return getVector(index).hashCode(index);
    }

    @Override
    public <OUT, IN> OUT accept(VectorVisitor<OUT, IN> visitor, IN value) {
      return visitor.visit(this, value);
    }
}
