/*******************************************************************************

 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
package org.apache.arrow.vector.types.pojo;

import com.google.flatbuffers.FlatBufferBuilder;
import org.apache.arrow.flatbuf.Type;

import java.io.IOException;
import java.util.Objects;

import org.apache.arrow.flatbuf.Precision;
import org.apache.arrow.flatbuf.UnionMode;
import org.apache.arrow.flatbuf.TimeUnit;
import org.apache.arrow.flatbuf.IntervalUnit;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonSubTypes;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;

/**
 * Arrow types
 **/
@JsonTypeInfo(
  use = JsonTypeInfo.Id.NAME,
  include = JsonTypeInfo.As.PROPERTY,
  property = "name")
@JsonSubTypes({
  @JsonSubTypes.Type(value = ArrowType.Null.class, name = "null"),
  @JsonSubTypes.Type(value = ArrowType.Struct_.class, name = "struct"),
  @JsonSubTypes.Type(value = ArrowType.List.class, name = "list"),
  @JsonSubTypes.Type(value = ArrowType.Union.class, name = "union"),
  @JsonSubTypes.Type(value = ArrowType.Int.class, name = "int"),
  @JsonSubTypes.Type(value = ArrowType.FloatingPoint.class, name = "floatingpoint"),
  @JsonSubTypes.Type(value = ArrowType.Utf8.class, name = "utf8"),
  @JsonSubTypes.Type(value = ArrowType.Binary.class, name = "binary"),
  @JsonSubTypes.Type(value = ArrowType.Bool.class, name = "bool"),
  @JsonSubTypes.Type(value = ArrowType.Decimal.class, name = "decimal"),
  @JsonSubTypes.Type(value = ArrowType.Date.class, name = "date"),
  @JsonSubTypes.Type(value = ArrowType.Time.class, name = "time"),
  @JsonSubTypes.Type(value = ArrowType.Timestamp.class, name = "timestamp"),
  @JsonSubTypes.Type(value = ArrowType.Interval.class, name = "interval"),
})
public abstract class ArrowType {

  private static class FloatingPointPrecisionSerializer extends JsonSerializer<Short> {
    @Override
    public void serialize(Short precision,
        JsonGenerator jsonGenerator,
        SerializerProvider serializerProvider)
            throws IOException, JsonProcessingException {
      jsonGenerator.writeObject(Precision.name(precision));
    }
  }

  private static class FloatingPointPrecisionDeserializer extends JsonDeserializer<Short> {
    @Override
    public Short deserialize(JsonParser p, DeserializationContext ctxt) throws IOException, JsonProcessingException {
      String name = p.getText();
      switch(name) {
        case "HALF":
          return Precision.HALF;
        case "SINGLE":
          return Precision.SINGLE;
        case "DOUBLE":
          return Precision.DOUBLE;
        default:
          throw new IllegalArgumentException("unknown precision: " + name);
      }
    }
  }

  private static class UnionModeSerializer extends JsonSerializer<Short> {
    @Override
    public void serialize(Short mode,
        JsonGenerator jsonGenerator,
        SerializerProvider serializerProvider)
            throws IOException, JsonProcessingException {
      jsonGenerator.writeObject(UnionMode.name(mode));
    }
  }

  private static class UnionModeDeserializer extends JsonDeserializer<Short> {
    @Override
    public Short deserialize(JsonParser p, DeserializationContext ctxt) throws IOException, JsonProcessingException {
      String name = p.getText();
      switch(name) {
        case "Sparse":
          return UnionMode.Sparse;
        case "Dense":
          return UnionMode.Dense;
        default:
          throw new IllegalArgumentException("unknown union mode: " + name);
      }
    }
  }

  private static class TimestampUnitSerializer extends JsonSerializer<Short> {
    @Override
    public void serialize(Short unit,
        JsonGenerator jsonGenerator,
        SerializerProvider serializerProvider)
            throws IOException, JsonProcessingException {
      jsonGenerator.writeObject(TimeUnit.name(unit));
    }
  }

  private static class TimestampUnitDeserializer extends JsonDeserializer<Short> {
    @Override
    public Short deserialize(JsonParser p, DeserializationContext ctxt) throws IOException, JsonProcessingException {
      String name = p.getText();
      switch(name) {
        case "SECOND":
          return TimeUnit.SECOND;
        case "MILLISECOND":
          return TimeUnit.MILLISECOND;
        case "MICROSECOND":
          return TimeUnit.MICROSECOND;
        case "NANOSECOND":
          return TimeUnit.NANOSECOND;
        default:
          throw new IllegalArgumentException("unknown time unit: " + name);
      }
    }
  }

  private static class IntervalUnitSerializer extends JsonSerializer<Short> {
    @Override
    public void serialize(Short unit,
        JsonGenerator jsonGenerator,
        SerializerProvider serializerProvider)
            throws IOException, JsonProcessingException {
      jsonGenerator.writeObject(IntervalUnit.name(unit));
    }
  }

  private static class IntervalUnitDeserializer extends JsonDeserializer<Short> {
    @Override
    public Short deserialize(JsonParser p, DeserializationContext ctxt) throws IOException, JsonProcessingException {
      String name = p.getText();
      switch(name) {
        case "YEAR_MONTH":
          return IntervalUnit.YEAR_MONTH;
        case "DAY_TIME":
          return IntervalUnit.DAY_TIME;
        default:
          throw new IllegalArgumentException("unknown interval unit: " + name);
      }
    }
  }

  @JsonIgnore
  public abstract byte getTypeType();
  public abstract int getType(FlatBufferBuilder builder);
  public abstract <T> T accept(ArrowTypeVisitor<T> visitor);

  /**
   * to visit the ArrowTypes
   * <code>
   *   type.accept(new ArrowTypeVisitor<Type>() {
   *   ...
   *   });
   * </code>
   */
  public static interface ArrowTypeVisitor<T> {
    T visit(Null type);
    T visit(Struct_ type);
    T visit(List type);
    T visit(Union type);
    T visit(Int type);
    T visit(FloatingPoint type);
    T visit(Utf8 type);
    T visit(Binary type);
    T visit(Bool type);
    T visit(Decimal type);
    T visit(Date type);
    T visit(Time type);
    T visit(Timestamp type);
    T visit(Interval type);
  }

  public static class Null extends ArrowType {
    public static final byte TYPE_TYPE = Type.Null;
    public static final Null INSTANCE = new Null();



    @Override
    public byte getTypeType() {
      return TYPE_TYPE;
    }

    @Override
    public int getType(FlatBufferBuilder builder) {
      org.apache.arrow.flatbuf.Null.startNull(builder);
      return org.apache.arrow.flatbuf.Null.endNull(builder);
    }


    public String toString() {
      return "Null{"
      + "}";
    }

    @Override
    public int hashCode() {
      return Objects.hash();
    }

    @Override
    public boolean equals(Object obj) {
      if (!(obj instanceof Null)) {
        return false;
      }
      return true;
    }

    @Override
    public <T> T accept(ArrowTypeVisitor<T> visitor) {
      return visitor.visit(this);
    }
  }
  public static class Struct_ extends ArrowType {
    public static final byte TYPE_TYPE = Type.Struct_;
    public static final Struct_ INSTANCE = new Struct_();



    @Override
    public byte getTypeType() {
      return TYPE_TYPE;
    }

    @Override
    public int getType(FlatBufferBuilder builder) {
      org.apache.arrow.flatbuf.Struct_.startStruct_(builder);
      return org.apache.arrow.flatbuf.Struct_.endStruct_(builder);
    }


    public String toString() {
      return "Struct_{"
      + "}";
    }

    @Override
    public int hashCode() {
      return Objects.hash();
    }

    @Override
    public boolean equals(Object obj) {
      if (!(obj instanceof Struct_)) {
        return false;
      }
      return true;
    }

    @Override
    public <T> T accept(ArrowTypeVisitor<T> visitor) {
      return visitor.visit(this);
    }
  }
  public static class List extends ArrowType {
    public static final byte TYPE_TYPE = Type.List;
    public static final List INSTANCE = new List();



    @Override
    public byte getTypeType() {
      return TYPE_TYPE;
    }

    @Override
    public int getType(FlatBufferBuilder builder) {
      org.apache.arrow.flatbuf.List.startList(builder);
      return org.apache.arrow.flatbuf.List.endList(builder);
    }


    public String toString() {
      return "List{"
      + "}";
    }

    @Override
    public int hashCode() {
      return Objects.hash();
    }

    @Override
    public boolean equals(Object obj) {
      if (!(obj instanceof List)) {
        return false;
      }
      return true;
    }

    @Override
    public <T> T accept(ArrowTypeVisitor<T> visitor) {
      return visitor.visit(this);
    }
  }
  public static class Union extends ArrowType {
    public static final byte TYPE_TYPE = Type.Union;

    short mode;
    int[] typeIds;

    @JsonCreator
    public Union(
       @JsonDeserialize(using = UnionModeDeserializer.class) @JsonProperty("mode") short mode, 
      @JsonProperty("typeIds") int[] typeIds
    ) {
      this.mode = mode;
      this.typeIds = typeIds;
    }

    @Override
    public byte getTypeType() {
      return TYPE_TYPE;
    }

    @Override
    public int getType(FlatBufferBuilder builder) {
      int typeIds = this.typeIds == null ? -1 : org.apache.arrow.flatbuf.Union.createTypeIdsVector(builder, this.typeIds);
      org.apache.arrow.flatbuf.Union.startUnion(builder);
      org.apache.arrow.flatbuf.Union.addMode(builder, this.mode);
      if (this.typeIds != null) {
        org.apache.arrow.flatbuf.Union.addTypeIds(builder, typeIds);
      }
      return org.apache.arrow.flatbuf.Union.endUnion(builder);
    }

    @JsonSerialize(using = UnionModeSerializer.class)
    public short getMode() {
      return mode;
    }
    public int[] getTypeIds() {
      return typeIds;
    }

    public String toString() {
      return "Union{"
        + mode + ", " 
        + java.util.Arrays.toString(typeIds)
      + "}";
    }

    @Override
    public int hashCode() {
      return Objects.hash(mode, typeIds);
    }

    @Override
    public boolean equals(Object obj) {
      if (!(obj instanceof Union)) {
        return false;
      }
      Union that = (Union) obj;
      return Objects.deepEquals(this.mode, that.mode) &&
Objects.deepEquals(this.typeIds, that.typeIds) ;
    }

    @Override
    public <T> T accept(ArrowTypeVisitor<T> visitor) {
      return visitor.visit(this);
    }
  }
  public static class Int extends ArrowType {
    public static final byte TYPE_TYPE = Type.Int;

    int bitWidth;
    boolean isSigned;

    @JsonCreator
    public Int(
      @JsonProperty("bitWidth") int bitWidth, 
      @JsonProperty("isSigned") boolean isSigned
    ) {
      this.bitWidth = bitWidth;
      this.isSigned = isSigned;
    }

    @Override
    public byte getTypeType() {
      return TYPE_TYPE;
    }

    @Override
    public int getType(FlatBufferBuilder builder) {
      org.apache.arrow.flatbuf.Int.startInt(builder);
      org.apache.arrow.flatbuf.Int.addBitWidth(builder, this.bitWidth);
      org.apache.arrow.flatbuf.Int.addIsSigned(builder, this.isSigned);
      return org.apache.arrow.flatbuf.Int.endInt(builder);
    }

    public int getBitWidth() {
      return bitWidth;
    }
    public boolean getIsSigned() {
      return isSigned;
    }

    public String toString() {
      return "Int{"
        + bitWidth + ", " 
        + isSigned
      + "}";
    }

    @Override
    public int hashCode() {
      return Objects.hash(bitWidth, isSigned);
    }

    @Override
    public boolean equals(Object obj) {
      if (!(obj instanceof Int)) {
        return false;
      }
      Int that = (Int) obj;
      return Objects.deepEquals(this.bitWidth, that.bitWidth) &&
Objects.deepEquals(this.isSigned, that.isSigned) ;
    }

    @Override
    public <T> T accept(ArrowTypeVisitor<T> visitor) {
      return visitor.visit(this);
    }
  }
  public static class FloatingPoint extends ArrowType {
    public static final byte TYPE_TYPE = Type.FloatingPoint;

    short precision;

    @JsonCreator
    public FloatingPoint(
       @JsonDeserialize(using = FloatingPointPrecisionDeserializer.class) @JsonProperty("precision") short precision
    ) {
      this.precision = precision;
    }

    @Override
    public byte getTypeType() {
      return TYPE_TYPE;
    }

    @Override
    public int getType(FlatBufferBuilder builder) {
      org.apache.arrow.flatbuf.FloatingPoint.startFloatingPoint(builder);
      org.apache.arrow.flatbuf.FloatingPoint.addPrecision(builder, this.precision);
      return org.apache.arrow.flatbuf.FloatingPoint.endFloatingPoint(builder);
    }

    @JsonSerialize(using = FloatingPointPrecisionSerializer.class)
    public short getPrecision() {
      return precision;
    }

    public String toString() {
      return "FloatingPoint{"
        + precision
      + "}";
    }

    @Override
    public int hashCode() {
      return Objects.hash(precision);
    }

    @Override
    public boolean equals(Object obj) {
      if (!(obj instanceof FloatingPoint)) {
        return false;
      }
      FloatingPoint that = (FloatingPoint) obj;
      return Objects.deepEquals(this.precision, that.precision) ;
    }

    @Override
    public <T> T accept(ArrowTypeVisitor<T> visitor) {
      return visitor.visit(this);
    }
  }
  public static class Utf8 extends ArrowType {
    public static final byte TYPE_TYPE = Type.Utf8;
    public static final Utf8 INSTANCE = new Utf8();



    @Override
    public byte getTypeType() {
      return TYPE_TYPE;
    }

    @Override
    public int getType(FlatBufferBuilder builder) {
      org.apache.arrow.flatbuf.Utf8.startUtf8(builder);
      return org.apache.arrow.flatbuf.Utf8.endUtf8(builder);
    }


    public String toString() {
      return "Utf8{"
      + "}";
    }

    @Override
    public int hashCode() {
      return Objects.hash();
    }

    @Override
    public boolean equals(Object obj) {
      if (!(obj instanceof Utf8)) {
        return false;
      }
      return true;
    }

    @Override
    public <T> T accept(ArrowTypeVisitor<T> visitor) {
      return visitor.visit(this);
    }
  }
  public static class Binary extends ArrowType {
    public static final byte TYPE_TYPE = Type.Binary;
    public static final Binary INSTANCE = new Binary();



    @Override
    public byte getTypeType() {
      return TYPE_TYPE;
    }

    @Override
    public int getType(FlatBufferBuilder builder) {
      org.apache.arrow.flatbuf.Binary.startBinary(builder);
      return org.apache.arrow.flatbuf.Binary.endBinary(builder);
    }


    public String toString() {
      return "Binary{"
      + "}";
    }

    @Override
    public int hashCode() {
      return Objects.hash();
    }

    @Override
    public boolean equals(Object obj) {
      if (!(obj instanceof Binary)) {
        return false;
      }
      return true;
    }

    @Override
    public <T> T accept(ArrowTypeVisitor<T> visitor) {
      return visitor.visit(this);
    }
  }
  public static class Bool extends ArrowType {
    public static final byte TYPE_TYPE = Type.Bool;
    public static final Bool INSTANCE = new Bool();



    @Override
    public byte getTypeType() {
      return TYPE_TYPE;
    }

    @Override
    public int getType(FlatBufferBuilder builder) {
      org.apache.arrow.flatbuf.Bool.startBool(builder);
      return org.apache.arrow.flatbuf.Bool.endBool(builder);
    }


    public String toString() {
      return "Bool{"
      + "}";
    }

    @Override
    public int hashCode() {
      return Objects.hash();
    }

    @Override
    public boolean equals(Object obj) {
      if (!(obj instanceof Bool)) {
        return false;
      }
      return true;
    }

    @Override
    public <T> T accept(ArrowTypeVisitor<T> visitor) {
      return visitor.visit(this);
    }
  }
  public static class Decimal extends ArrowType {
    public static final byte TYPE_TYPE = Type.Decimal;

    int precision;
    int scale;

    @JsonCreator
    public Decimal(
      @JsonProperty("precision") int precision, 
      @JsonProperty("scale") int scale
    ) {
      this.precision = precision;
      this.scale = scale;
    }

    @Override
    public byte getTypeType() {
      return TYPE_TYPE;
    }

    @Override
    public int getType(FlatBufferBuilder builder) {
      org.apache.arrow.flatbuf.Decimal.startDecimal(builder);
      org.apache.arrow.flatbuf.Decimal.addPrecision(builder, this.precision);
      org.apache.arrow.flatbuf.Decimal.addScale(builder, this.scale);
      return org.apache.arrow.flatbuf.Decimal.endDecimal(builder);
    }

    public int getPrecision() {
      return precision;
    }
    public int getScale() {
      return scale;
    }

    public String toString() {
      return "Decimal{"
        + precision + ", " 
        + scale
      + "}";
    }

    @Override
    public int hashCode() {
      return Objects.hash(precision, scale);
    }

    @Override
    public boolean equals(Object obj) {
      if (!(obj instanceof Decimal)) {
        return false;
      }
      Decimal that = (Decimal) obj;
      return Objects.deepEquals(this.precision, that.precision) &&
Objects.deepEquals(this.scale, that.scale) ;
    }

    @Override
    public <T> T accept(ArrowTypeVisitor<T> visitor) {
      return visitor.visit(this);
    }
  }
  public static class Date extends ArrowType {
    public static final byte TYPE_TYPE = Type.Date;
    public static final Date INSTANCE = new Date();



    @Override
    public byte getTypeType() {
      return TYPE_TYPE;
    }

    @Override
    public int getType(FlatBufferBuilder builder) {
      org.apache.arrow.flatbuf.Date.startDate(builder);
      return org.apache.arrow.flatbuf.Date.endDate(builder);
    }


    public String toString() {
      return "Date{"
      + "}";
    }

    @Override
    public int hashCode() {
      return Objects.hash();
    }

    @Override
    public boolean equals(Object obj) {
      if (!(obj instanceof Date)) {
        return false;
      }
      return true;
    }

    @Override
    public <T> T accept(ArrowTypeVisitor<T> visitor) {
      return visitor.visit(this);
    }
  }
  public static class Time extends ArrowType {
    public static final byte TYPE_TYPE = Type.Time;
    public static final Time INSTANCE = new Time();



    @Override
    public byte getTypeType() {
      return TYPE_TYPE;
    }

    @Override
    public int getType(FlatBufferBuilder builder) {
      org.apache.arrow.flatbuf.Time.startTime(builder);
      return org.apache.arrow.flatbuf.Time.endTime(builder);
    }


    public String toString() {
      return "Time{"
      + "}";
    }

    @Override
    public int hashCode() {
      return Objects.hash();
    }

    @Override
    public boolean equals(Object obj) {
      if (!(obj instanceof Time)) {
        return false;
      }
      return true;
    }

    @Override
    public <T> T accept(ArrowTypeVisitor<T> visitor) {
      return visitor.visit(this);
    }
  }
  public static class Timestamp extends ArrowType {
    public static final byte TYPE_TYPE = Type.Timestamp;

    short unit;

    @JsonCreator
    public Timestamp(
       @JsonDeserialize(using = TimestampUnitDeserializer.class) @JsonProperty("unit") short unit
    ) {
      this.unit = unit;
    }

    @Override
    public byte getTypeType() {
      return TYPE_TYPE;
    }

    @Override
    public int getType(FlatBufferBuilder builder) {
      org.apache.arrow.flatbuf.Timestamp.startTimestamp(builder);
      org.apache.arrow.flatbuf.Timestamp.addUnit(builder, this.unit);
      return org.apache.arrow.flatbuf.Timestamp.endTimestamp(builder);
    }

    @JsonSerialize(using = TimestampUnitSerializer.class)
    public short getUnit() {
      return unit;
    }

    public String toString() {
      return "Timestamp{"
        + unit
      + "}";
    }

    @Override
    public int hashCode() {
      return Objects.hash(unit);
    }

    @Override
    public boolean equals(Object obj) {
      if (!(obj instanceof Timestamp)) {
        return false;
      }
      Timestamp that = (Timestamp) obj;
      return Objects.deepEquals(this.unit, that.unit) ;
    }

    @Override
    public <T> T accept(ArrowTypeVisitor<T> visitor) {
      return visitor.visit(this);
    }
  }
  public static class Interval extends ArrowType {
    public static final byte TYPE_TYPE = Type.Interval;

    short unit;

    @JsonCreator
    public Interval(
       @JsonDeserialize(using = IntervalUnitDeserializer.class) @JsonProperty("unit") short unit
    ) {
      this.unit = unit;
    }

    @Override
    public byte getTypeType() {
      return TYPE_TYPE;
    }

    @Override
    public int getType(FlatBufferBuilder builder) {
      org.apache.arrow.flatbuf.Interval.startInterval(builder);
      org.apache.arrow.flatbuf.Interval.addUnit(builder, this.unit);
      return org.apache.arrow.flatbuf.Interval.endInterval(builder);
    }

    @JsonSerialize(using = IntervalUnitSerializer.class)
    public short getUnit() {
      return unit;
    }

    public String toString() {
      return "Interval{"
        + unit
      + "}";
    }

    @Override
    public int hashCode() {
      return Objects.hash(unit);
    }

    @Override
    public boolean equals(Object obj) {
      if (!(obj instanceof Interval)) {
        return false;
      }
      Interval that = (Interval) obj;
      return Objects.deepEquals(this.unit, that.unit) ;
    }

    @Override
    public <T> T accept(ArrowTypeVisitor<T> visitor) {
      return visitor.visit(this);
    }
  }

  public static org.apache.arrow.vector.types.pojo.ArrowType getTypeForField(org.apache.arrow.flatbuf.Field field) {
    switch(field.typeType()) {
    case Type.Null: {
      org.apache.arrow.flatbuf.Null nullType = (org.apache.arrow.flatbuf.Null) field.type(new org.apache.arrow.flatbuf.Null());
      return new Null();
    }
    case Type.Struct_: {
      org.apache.arrow.flatbuf.Struct_ struct_Type = (org.apache.arrow.flatbuf.Struct_) field.type(new org.apache.arrow.flatbuf.Struct_());
      return new Struct_();
    }
    case Type.List: {
      org.apache.arrow.flatbuf.List listType = (org.apache.arrow.flatbuf.List) field.type(new org.apache.arrow.flatbuf.List());
      return new List();
    }
    case Type.Union: {
      org.apache.arrow.flatbuf.Union unionType = (org.apache.arrow.flatbuf.Union) field.type(new org.apache.arrow.flatbuf.Union());
      short mode = unionType.mode();
      int[] typeIds = new int[unionType.typeIdsLength()];
      for (int i = 0; i< typeIds.length; ++i) {
        typeIds[i] = unionType.typeIds(i);
      }
      return new Union(mode, typeIds);
    }
    case Type.Int: {
      org.apache.arrow.flatbuf.Int intType = (org.apache.arrow.flatbuf.Int) field.type(new org.apache.arrow.flatbuf.Int());
      int bitWidth = intType.bitWidth();
      boolean isSigned = intType.isSigned();
      return new Int(bitWidth, isSigned);
    }
    case Type.FloatingPoint: {
      org.apache.arrow.flatbuf.FloatingPoint floatingpointType = (org.apache.arrow.flatbuf.FloatingPoint) field.type(new org.apache.arrow.flatbuf.FloatingPoint());
      short precision = floatingpointType.precision();
      return new FloatingPoint(precision);
    }
    case Type.Utf8: {
      org.apache.arrow.flatbuf.Utf8 utf8Type = (org.apache.arrow.flatbuf.Utf8) field.type(new org.apache.arrow.flatbuf.Utf8());
      return new Utf8();
    }
    case Type.Binary: {
      org.apache.arrow.flatbuf.Binary binaryType = (org.apache.arrow.flatbuf.Binary) field.type(new org.apache.arrow.flatbuf.Binary());
      return new Binary();
    }
    case Type.Bool: {
      org.apache.arrow.flatbuf.Bool boolType = (org.apache.arrow.flatbuf.Bool) field.type(new org.apache.arrow.flatbuf.Bool());
      return new Bool();
    }
    case Type.Decimal: {
      org.apache.arrow.flatbuf.Decimal decimalType = (org.apache.arrow.flatbuf.Decimal) field.type(new org.apache.arrow.flatbuf.Decimal());
      int precision = decimalType.precision();
      int scale = decimalType.scale();
      return new Decimal(precision, scale);
    }
    case Type.Date: {
      org.apache.arrow.flatbuf.Date dateType = (org.apache.arrow.flatbuf.Date) field.type(new org.apache.arrow.flatbuf.Date());
      return new Date();
    }
    case Type.Time: {
      org.apache.arrow.flatbuf.Time timeType = (org.apache.arrow.flatbuf.Time) field.type(new org.apache.arrow.flatbuf.Time());
      return new Time();
    }
    case Type.Timestamp: {
      org.apache.arrow.flatbuf.Timestamp timestampType = (org.apache.arrow.flatbuf.Timestamp) field.type(new org.apache.arrow.flatbuf.Timestamp());
      short unit = timestampType.unit();
      return new Timestamp(unit);
    }
    case Type.Interval: {
      org.apache.arrow.flatbuf.Interval intervalType = (org.apache.arrow.flatbuf.Interval) field.type(new org.apache.arrow.flatbuf.Interval());
      short unit = intervalType.unit();
      return new Interval(unit);
    }
    default:
      throw new UnsupportedOperationException("Unsupported type: " + field.typeType());
    }
  }

  public static Int getInt(org.apache.arrow.flatbuf.Field field) {
    org.apache.arrow.flatbuf.Int intType = (org.apache.arrow.flatbuf.Int) field.type(new org.apache.arrow.flatbuf.Int());
    return new Int(intType.bitWidth(), intType.isSigned());
  }
}


