/*
 * Licensed by the author of Time4J-project.
 *
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership. The copyright owner
 * licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package net.time4j.calendar;

import net.time4j.engine.CalendarEra;
import net.time4j.format.CalendarText;
import net.time4j.format.TextWidth;

import java.util.Locale;


/**
 * <p>The Ethiopian calendar supports two eras. </p>
 *
 * <p>Amete-Mihret is preferred in modern times (although both eras can still be used in parallel)
 * but if mihret-years become smaller than 1 then Amete-Alem is used. </p>
 *
 * @author  Meno Hochschild
 * @since   3.11/4.8
 */
/*[deutsch]
 * <p>Der &auml;tiopische Kalender unterst&uuml;tzt zwei &Auml;ras. </p>
 *
 * <p>Amete-Mihret ist heutzutage bevorzugt (obwohl immer noch beide &Auml;ren parallel benutzbar sind),
 * aber wenn Mihret-Jahre kleiner als 1 zu werden drohen, wird immer Amete-Alem benutzt. </p>
 *
 * @author  Meno Hochschild
 * @since   3.11/4.8
 */
public enum EthiopianEra
    implements CalendarEra {

    //~ Statische Felder/Initialisierungen --------------------------------

    /**
     * The era &quot;Anno Mundi&quot; (English - &quot;Year of the World&quot;) in the version
     * of Panodoros starts at BC-5493-08-29 (Julian calendar).
     */
    /*[deutsch]
     * Die &Auml;ra &quot;Anno Mundi&quot; (Sch&ouml;pfungsdatum) in der Version
     * von Panodoros beginnt zum Datum BC-5493-08-29 (julianischer Kalender).
     */
    AMETE_ALEM,

    /**
     * This era (English - &quot;Year of Grace&quot;) is used for current years {@code >= 1} and
     * starts at AD-8-08-29 (Julian calendar).
     */
    /*[deutsch]
     * Diese &Auml;ra (Jahr der Barmherzigkeit) wird f&uuml;r aktuelle Jahre {@code >= 1} gebraucht und
     * beginnt zum Datum AD-8-08-29 (julianischer Kalender).
     */
    AMETE_MIHRET;

    //~ Methoden ----------------------------------------------------------

    /**
     * <p>Equivalent to the expression {@code getDisplayName(locale, TextWidth.WIDE)}. </p>
     *
     * @param   locale      language setting
     * @return  descriptive text (long form, never {@code null})
     * @see     #getDisplayName(Locale, TextWidth)
     * @since   3.11/4.8
     */
    /*[deutsch]
     * <p>Entspricht dem Ausdruck {@code getDisplayName(locale, TextWidth.WIDE)}. </p>
     *
     * @param   locale      language setting
     * @return  descriptive text (long form, never {@code null})
     * @see     #getDisplayName(Locale, TextWidth)
     * @since   3.11/4.8
     */
    public String getDisplayName(Locale locale) {

        return this.getDisplayName(locale, TextWidth.WIDE);

    }

    /**
     * <p>Gets the description text dependent on the locale and style parameters. </p>
     *
     * <p>The second argument controls the width of description. </p>
     *
     * @param   locale      language setting
     * @param   width       text width
     * @return  descriptive text for given locale and style (never {@code null})
     * @since   3.11/4.8
     */
    /*[deutsch]
     * <p>Liefert den sprachabh&auml;ngigen Beschreibungstext. </p>
     *
     * <p>&Uuml;ber das zweite Argument kann gesteuert werden, ob eine kurze
     * oder eine lange Form des Beschreibungstexts ausgegeben werden soll. Das
     * ist besonders sinnvoll in Benutzeroberfl&auml;chen, wo zwischen der
     * Beschriftung und der detaillierten Erl&auml;uterung einer graphischen
     * Komponente unterschieden wird. </p>
     *
     * @param   locale      language setting
     * @param   width       text width
     * @return  descriptive text for given locale and style (never {@code null})
     * @since   3.11/4.8
     */
    public String getDisplayName(
        Locale locale,
        TextWidth width
    ) {

        CalendarText names = CalendarText.getInstance("ethiopic", locale);
        return names.getEras(width).print(this);

    }

}
