/*
 * Decompiled with CFR 0.152.
 */
package net.time4j.history;

import java.io.IOException;
import java.io.InvalidObjectException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import net.time4j.CalendarUnit;
import net.time4j.PlainDate;
import net.time4j.engine.AttributeKey;
import net.time4j.engine.ChronoElement;
import net.time4j.engine.EpochDays;
import net.time4j.format.TextElement;
import net.time4j.history.CalendarAlgorithm;
import net.time4j.history.CutOverEvent;
import net.time4j.history.HistoricDate;
import net.time4j.history.HistoricEra;
import net.time4j.history.HistoricVariant;
import net.time4j.history.HistoricalEraElement;
import net.time4j.history.HistoricalIntegerElement;
import net.time4j.history.SPX;

public final class ChronoHistory
implements Serializable {
    public static final AttributeKey<HistoricVariant> ATTRIBUTE_HISTORIC_VARIANT = Key.valueOf("HISTORIC_VARIANT", HistoricVariant.class);
    public static final AttributeKey<PlainDate> ATTRIBUTE_CUTOVER_DATE = Key.valueOf("CUTOVER_DATE", PlainDate.class);
    public static final AttributeKey<Boolean> ATTRIBUTE_COMMON_ERA = Key.valueOf("COMMON_ERA", Boolean.class);
    public static final AttributeKey<Boolean> ATTRIBUTE_LATIN_ERA = Key.valueOf("LATIN_ERA", Boolean.class);
    public static final ChronoHistory PROLEPTIC_GREGORIAN = new ChronoHistory(HistoricVariant.PROLEPTIC_GREGORIAN, Collections.singletonList(new CutOverEvent(Long.MIN_VALUE, CalendarAlgorithm.GREGORIAN, CalendarAlgorithm.GREGORIAN)));
    public static final ChronoHistory PROLEPTIC_JULIAN = new ChronoHistory(HistoricVariant.PROLEPTIC_JULIAN, Collections.singletonList(new CutOverEvent(Long.MIN_VALUE, CalendarAlgorithm.JULIAN, CalendarAlgorithm.JULIAN)));
    private static final long EARLIEST_CUTOVER = PlainDate.of(1582, 10, 15).get(EpochDays.MODIFIED_JULIAN_DATE);
    private static final ChronoHistory INTRODUCTION_BY_POPE_GREGOR = ChronoHistory.ofGregorianReform(EARLIEST_CUTOVER);
    private static final ChronoHistory SWEDEN;
    private static final Map<String, ChronoHistory> LOOKUP;
    private static final long serialVersionUID = 4100690610730913643L;
    private final transient HistoricVariant variant;
    private final transient List<CutOverEvent> events;
    private final transient TextElement<HistoricEra> eraElement;
    private final transient TextElement<Integer> yearOfEraElement;
    private final transient TextElement<Integer> monthElement;
    private final transient TextElement<Integer> dayOfMonthElement;
    private final transient Set<ChronoElement<?>> elements;

    private ChronoHistory(HistoricVariant historicVariant, List<CutOverEvent> list) {
        if (list.isEmpty()) {
            throw new IllegalArgumentException("At least one cutover event must be present in chronological history.");
        }
        if (historicVariant == null) {
            throw new NullPointerException("Missing historical variant.");
        }
        this.variant = historicVariant;
        this.events = list;
        this.eraElement = new HistoricalEraElement(this);
        this.yearOfEraElement = HistoricalIntegerElement.forYearOfEra(this);
        this.monthElement = HistoricalIntegerElement.forMonth(this);
        this.dayOfMonthElement = HistoricalIntegerElement.forDayOfMonth(this);
        HashSet<TextElement<Object>> hashSet = new HashSet<TextElement<Object>>();
        hashSet.add(this.eraElement);
        hashSet.add(this.yearOfEraElement);
        hashSet.add(this.monthElement);
        hashSet.add(this.dayOfMonthElement);
        this.elements = Collections.unmodifiableSet(hashSet);
    }

    public static ChronoHistory ofFirstGregorianReform() {
        return INTRODUCTION_BY_POPE_GREGOR;
    }

    public static ChronoHistory ofGregorianReform(PlainDate plainDate) {
        if (plainDate.equals(PlainDate.axis().getMaximum())) {
            return PROLEPTIC_JULIAN;
        }
        if (plainDate.equals(PlainDate.axis().getMinimum())) {
            return PROLEPTIC_GREGORIAN;
        }
        long l = plainDate.get(EpochDays.MODIFIED_JULIAN_DATE);
        ChronoHistory.check(l);
        if (l == EARLIEST_CUTOVER) {
            return INTRODUCTION_BY_POPE_GREGOR;
        }
        return ChronoHistory.ofGregorianReform(l);
    }

    public static ChronoHistory of(Locale locale) {
        ChronoHistory chronoHistory = LOOKUP.get(locale.getCountry());
        if (chronoHistory == null) {
            return ChronoHistory.ofFirstGregorianReform();
        }
        return chronoHistory;
    }

    public static ChronoHistory ofSweden() {
        return SWEDEN;
    }

    public boolean isValid(HistoricDate historicDate) {
        if (historicDate == null) {
            return false;
        }
        CalendarAlgorithm calendarAlgorithm = this.getAlgorithm(historicDate);
        return calendarAlgorithm != null && calendarAlgorithm.isValid(historicDate);
    }

    public PlainDate convert(HistoricDate historicDate) {
        CalendarAlgorithm calendarAlgorithm = this.getAlgorithm(historicDate);
        if (calendarAlgorithm == null) {
            throw new IllegalArgumentException("Invalid historical date: " + historicDate);
        }
        return PlainDate.of(calendarAlgorithm.toMJD(historicDate), EpochDays.MODIFIED_JULIAN_DATE);
    }

    public HistoricDate convert(PlainDate plainDate) {
        long l = plainDate.get(EpochDays.MODIFIED_JULIAN_DATE);
        for (int i = this.events.size() - 1; i >= 0; --i) {
            CutOverEvent cutOverEvent = this.events.get(i);
            if (l < cutOverEvent.start) continue;
            return cutOverEvent.algorithm.fromMJD(l);
        }
        return CalendarAlgorithm.JULIAN.fromMJD(l);
    }

    public HistoricVariant getVariant() {
        return this.variant;
    }

    public PlainDate getGregorianCutOverDate() {
        return PlainDate.of(this.events.get((int)(this.events.size() - 1)).start, EpochDays.MODIFIED_JULIAN_DATE);
    }

    public int getLengthOfYear(HistoricEra historicEra, int n) {
        try {
            HistoricDate historicDate = HistoricDate.of(historicEra, n, 1, 1);
            HistoricDate historicDate2 = HistoricDate.of(historicEra, n, 12, 31);
            return (int)(CalendarUnit.DAYS.between(this.convert(historicDate), this.convert(historicDate2)) + 1L);
        }
        catch (RuntimeException runtimeException) {
            return -1;
        }
    }

    public TextElement<HistoricEra> era() {
        return this.eraElement;
    }

    public TextElement<Integer> yearOfEra() {
        return this.yearOfEraElement;
    }

    public TextElement<Integer> month() {
        return this.monthElement;
    }

    public TextElement<Integer> dayOfMonth() {
        return this.dayOfMonthElement;
    }

    public Set<ChronoElement<?>> getElements() {
        return this.elements;
    }

    public boolean equals(Object object) {
        if (this == object) {
            return true;
        }
        if (object instanceof ChronoHistory) {
            ChronoHistory chronoHistory = (ChronoHistory)object;
            if (this.variant == chronoHistory.variant) {
                return this.variant != HistoricVariant.SINGLE_CUTOVER_DATE || this.events.get((int)0).start == chronoHistory.events.get((int)0).start;
            }
        }
        return false;
    }

    public int hashCode() {
        if (this.variant == HistoricVariant.SINGLE_CUTOVER_DATE) {
            long l = this.events.get((int)0).start;
            return (int)(l ^ l << 32);
        }
        return this.variant.hashCode();
    }

    public String toString() {
        switch (this.variant) {
            case PROLEPTIC_GREGORIAN: {
                return "ChronoHistory[PROLEPTIC-GREGORIAN]";
            }
            case PROLEPTIC_JULIAN: {
                return "ChronoHistory[PROLEPTIC-JULIAN]";
            }
            case SWEDEN: {
                return "ChronoHistory[SWEDEN]";
            }
        }
        PlainDate plainDate = this.getGregorianCutOverDate();
        return "ChronoHistory[" + plainDate + "]";
    }

    CalendarAlgorithm getAlgorithm(HistoricDate historicDate) {
        for (int i = this.events.size() - 1; i >= 0; --i) {
            CutOverEvent cutOverEvent = this.events.get(i);
            if (historicDate.compareTo(cutOverEvent.dateAtCutOver) >= 0) {
                return cutOverEvent.algorithm;
            }
            if (historicDate.compareTo(cutOverEvent.dateBeforeCutOver) <= 0) continue;
            return null;
        }
        return CalendarAlgorithm.JULIAN;
    }

    HistoricDate adjustDayOfMonth(HistoricDate historicDate) {
        CalendarAlgorithm calendarAlgorithm = this.getAlgorithm(historicDate);
        if (calendarAlgorithm == null) {
            return historicDate;
        }
        int n = calendarAlgorithm.getMaximumDayOfMonth(historicDate);
        if (n < historicDate.getDayOfMonth()) {
            return HistoricDate.of(historicDate.getEra(), historicDate.getYearOfEra(), historicDate.getMonth(), n);
        }
        return historicDate;
    }

    List<CutOverEvent> getEvents() {
        return this.events;
    }

    private static void check(long l) {
        if (l < EARLIEST_CUTOVER) {
            throw new IllegalArgumentException("Gregorian calendar did not exist before 1582-10-15");
        }
    }

    private static ChronoHistory ofGregorianReform(long l) {
        return new ChronoHistory(l == EARLIEST_CUTOVER ? HistoricVariant.INTRODUCTION_ON_1582_10_15 : HistoricVariant.SINGLE_CUTOVER_DATE, Collections.singletonList(new CutOverEvent(l, CalendarAlgorithm.JULIAN, CalendarAlgorithm.GREGORIAN)));
    }

    private Object writeReplace() {
        return new SPX(this, 1);
    }

    private void readObject(ObjectInputStream objectInputStream) throws IOException {
        throw new InvalidObjectException("Serialization proxy required.");
    }

    static {
        ArrayList<CutOverEvent> arrayList = new ArrayList<CutOverEvent>();
        arrayList.add(new CutOverEvent(-57959L, CalendarAlgorithm.JULIAN, CalendarAlgorithm.SWEDISH));
        arrayList.add(new CutOverEvent(-53575L, CalendarAlgorithm.SWEDISH, CalendarAlgorithm.JULIAN));
        arrayList.add(new CutOverEvent(-38611L, CalendarAlgorithm.JULIAN, CalendarAlgorithm.GREGORIAN));
        SWEDEN = new ChronoHistory(HistoricVariant.SWEDEN, Collections.unmodifiableList(arrayList));
        HashMap<String, ChronoHistory> hashMap = new HashMap<String, ChronoHistory>();
        hashMap.put("GB", ChronoHistory.ofGregorianReform(PlainDate.of(1752, 9, 14)));
        hashMap.put("RU", ChronoHistory.ofGregorianReform(PlainDate.of(1918, 2, 14)));
        hashMap.put("SE", SWEDEN);
        LOOKUP = Collections.unmodifiableMap(hashMap);
    }

    private static class Key<T>
    implements AttributeKey<T> {
        private final String name;
        private final Class<T> type;

        private Key(String string, Class<T> clazz) {
            this.name = string;
            this.type = clazz;
        }

        static <T> Key<T> valueOf(String string, Class<T> clazz) {
            return new Key<T>(string, clazz);
        }

        @Override
        public String name() {
            return this.name;
        }

        @Override
        public Class<T> type() {
            return this.type;
        }

        public boolean equals(Object object) {
            if (this == object) {
                return true;
            }
            if (object instanceof Key) {
                Key key = (Key)object;
                return this.name.equals(key.name) && this.type.equals(key.type);
            }
            return false;
        }

        public int hashCode() {
            return this.name.hashCode();
        }

        public String toString() {
            return this.type.getName() + "@" + this.name;
        }
    }
}

