/*
 * Copyright (c) 2012-2022 Snowflake Computing Inc. All rights reserved.
 */

package net.snowflake.client.core;

import java.net.MalformedURLException;
import java.net.URL;
import java.security.PrivateKey;
import java.time.Duration;
import java.util.Map;
import net.snowflake.client.jdbc.ErrorCode;

/** A class for holding all information required for login */
public class SFLoginInput {
  private String serverUrl;
  private String databaseName;
  private String schemaName;
  private String warehouse;
  private String role;
  private boolean validateDefaultParameters;
  private String authenticator;
  private String oktaUserName;
  private String accountName;
  private int loginTimeout = -1; // default is invalid
  private int retryTimeout = 300;
  private int authTimeout = 0;
  private String userName;
  private String password;
  private boolean passcodeInPassword;
  private String passcode;
  private String token;
  private Duration connectionTimeout = HttpUtil.getConnectionTimeout();
  private Duration socketTimeout = HttpUtil.getSocketTimeout();
  private String appId;
  private String appVersion;
  private String sessionToken;
  private String masterToken;
  private Map<String, Object> sessionParameters;
  private PrivateKey privateKey;
  private String application;
  private String idToken;
  private String mfaToken;
  private String serviceName;
  private OCSPMode ocspMode;
  private HttpClientSettingsKey httpClientKey;
  private String privateKeyFile;
  private String privateKeyFilePwd;
  private String inFlightCtx; // Opaque string sent for Snowsight account activation

  private boolean disableConsoleLogin = true;
  private boolean disableSamlURLCheck = false;

  // Additional headers to add for Snowsight.
  Map<String, String> additionalHttpHeadersForSnowsight;

  SFLoginInput() {}

  public String getServerUrl() {
    return serverUrl;
  }

  SFLoginInput setServerUrl(String serverUrl) {
    this.serverUrl = serverUrl;
    return this;
  }

  public boolean getDisableConsoleLogin() {
    return disableConsoleLogin;
  }

  SFLoginInput setDisableConsoleLogin(boolean disableConsoleLogin) {
    this.disableConsoleLogin = disableConsoleLogin;
    return this;
  }

  String getDatabaseName() {
    return databaseName;
  }

  SFLoginInput setDatabaseName(String databaseName) {
    this.databaseName = databaseName;
    return this;
  }

  public String getSchemaName() {
    return schemaName;
  }

  public SFLoginInput setSchemaName(String schemaName) {
    this.schemaName = schemaName;
    return this;
  }

  public String getWarehouse() {
    return warehouse;
  }

  public SFLoginInput setWarehouse(String warehouse) {
    this.warehouse = warehouse;
    return this;
  }

  public String getRole() {
    return role;
  }

  public SFLoginInput setRole(String role) {
    this.role = role;
    return this;
  }

  public boolean isValidateDefaultParameters() {
    return validateDefaultParameters;
  }

  public SFLoginInput setValidateDefaultParameters(Object v) {
    validateDefaultParameters = getBooleanValue(v);
    return this;
  }

  public String getAuthenticator() {
    return authenticator;
  }

  public SFLoginInput setAuthenticator(String authenticator) {
    this.authenticator = authenticator;
    return this;
  }

  public String getOKTAUserName() {
    return oktaUserName;
  }

  public SFLoginInput setOKTAUserName(String oktaUserName) {
    this.oktaUserName = oktaUserName;
    return this;
  }

  public String getAccountName() {
    return accountName;
  }

  public SFLoginInput setAccountName(String accountName) {
    this.accountName = accountName;
    return this;
  }

  int getLoginTimeout() {
    return loginTimeout;
  }

  // We want to choose the smaller of the two values between retryTimeout and loginTimeout for the
  // new retry strategy.
  SFLoginInput setLoginTimeout(int loginTimeout) {
    if (loginTimeout > retryTimeout && retryTimeout != 0) {
      this.loginTimeout = retryTimeout;
    } else {
      this.loginTimeout = loginTimeout;
    }
    return this;
  }

  int getRetryTimeout() {
    return retryTimeout;
  }

  SFLoginInput setRetryTimeout(int retryTimeout) {
    this.retryTimeout = retryTimeout;
    return this;
  }

  int getAuthTimeout() {
    return authTimeout;
  }

  SFLoginInput setAuthTimeout(int authTimeout) {
    this.authTimeout = authTimeout;
    return this;
  }

  public String getUserName() {
    return userName;
  }

  SFLoginInput setUserName(String userName) {
    this.userName = userName;
    return this;
  }

  public String getPassword() {
    return password;
  }

  public SFLoginInput setPassword(String password) {
    this.password = password;
    return this;
  }

  String getPasscode() {
    return passcode;
  }

  SFLoginInput setPasscode(String passcode) {
    this.passcode = passcode;
    return this;
  }

  public String getToken() {
    return token;
  }

  public SFLoginInput setToken(String token) {
    this.token = token;
    return this;
  }

  Duration getConnectionTimeout() {
    return connectionTimeout;
  }

  SFLoginInput setConnectionTimeout(Duration connectionTimeout) {
    this.connectionTimeout = connectionTimeout;
    return this;
  }

  int getSocketTimeoutInMillis() {
    return (int) socketTimeout.toMillis();
  }

  SFLoginInput setSocketTimeout(Duration socketTimeout) {
    this.socketTimeout = socketTimeout;
    return this;
  }

  boolean isPasscodeInPassword() {
    return passcodeInPassword;
  }

  SFLoginInput setPasscodeInPassword(boolean passcodeInPassword) {
    this.passcodeInPassword = passcodeInPassword;
    return this;
  }

  String getAppId() {
    return appId;
  }

  SFLoginInput setAppId(String appId) {
    this.appId = appId;
    return this;
  }

  String getAppVersion() {
    return appVersion;
  }

  SFLoginInput setAppVersion(String appVersion) {
    this.appVersion = appVersion;
    return this;
  }

  public String getSessionToken() {
    return sessionToken;
  }

  public SFLoginInput setSessionToken(String sessionToken) {
    this.sessionToken = sessionToken;
    return this;
  }

  String getMasterToken() {
    return masterToken;
  }

  SFLoginInput setMasterToken(String masterToken) {
    this.masterToken = masterToken;
    return this;
  }

  String getIdToken() {
    return idToken;
  }

  SFLoginInput setIdToken(String idToken) {
    this.idToken = idToken;
    return this;
  }

  String getMfaToken() {
    return mfaToken;
  }

  SFLoginInput setMfaToken(String mfaToken) {
    this.mfaToken = mfaToken;
    return this;
  }

  Map<String, Object> getSessionParameters() {
    return sessionParameters;
  }

  SFLoginInput setSessionParameters(Map<String, Object> sessionParameters) {
    this.sessionParameters = sessionParameters;
    return this;
  }

  PrivateKey getPrivateKey() {
    return privateKey;
  }

  SFLoginInput setPrivateKey(PrivateKey privateKey) {
    this.privateKey = privateKey;
    return this;
  }

  SFLoginInput setPrivateKeyFile(String privateKeyFile) {
    this.privateKeyFile = privateKeyFile;
    return this;
  }

  SFLoginInput setPrivateKeyFilePwd(String privateKeyFilePwd) {
    this.privateKeyFilePwd = privateKeyFilePwd;
    return this;
  }

  String getPrivateKeyFile() {
    return privateKeyFile;
  }

  String getPrivateKeyFilePwd() {
    return privateKeyFilePwd;
  }

  public String getApplication() {
    return application;
  }

  public SFLoginInput setApplication(String application) {
    this.application = application;
    return this;
  }

  String getServiceName() {
    return serviceName;
  }

  SFLoginInput setServiceName(String serviceName) {
    this.serviceName = serviceName;
    return this;
  }

  OCSPMode getOCSPMode() {
    return ocspMode;
  }

  SFLoginInput setOCSPMode(OCSPMode ocspMode) {
    this.ocspMode = ocspMode;
    return this;
  }

  HttpClientSettingsKey getHttpClientSettingsKey() {
    return httpClientKey;
  }

  SFLoginInput setHttpClientSettingsKey(HttpClientSettingsKey key) {
    this.httpClientKey = key;
    return this;
  }

  // Opaque string sent for Snowsight account activation
  String getInFlightCtx() {
    return inFlightCtx;
  }

  // Opaque string sent for Snowsight account activation
  SFLoginInput setInFlightCtx(String inFlightCtx) {
    this.inFlightCtx = inFlightCtx;
    return this;
  }

  boolean getDisableSamlURLCheck() {
    return disableSamlURLCheck;
  }

  SFLoginInput setDisableSamlURLCheck(boolean disableSamlURLCheck) {
    this.disableSamlURLCheck = disableSamlURLCheck;
    return this;
  }

  Map<String, String> getAdditionalHttpHeadersForSnowsight() {
    return additionalHttpHeadersForSnowsight;
  }

  /**
   * Set additional http headers to apply to the outgoing request. The additional headers cannot be
   * used to replace or overwrite a header in use by the driver. These will be applied to the
   * outgoing request. Primarily used by Snowsight, as described in {@link
   * HttpUtil#applyAdditionalHeadersForSnowsight(org.apache.http.client.methods.HttpRequestBase,
   * Map)}
   *
   * @param additionalHttpHeaders The new headers to add
   * @return The input object, for chaining
   * @see
   *     HttpUtil#applyAdditionalHeadersForSnowsight(org.apache.http.client.methods.HttpRequestBase,
   *     Map)
   */
  public SFLoginInput setAdditionalHttpHeadersForSnowsight(
      Map<String, String> additionalHttpHeaders) {
    this.additionalHttpHeadersForSnowsight = additionalHttpHeaders;
    return this;
  }

  static boolean getBooleanValue(Object v) {
    if (v instanceof Boolean) {
      return (Boolean) v;
    } else if (v instanceof String) {
      return !Boolean.FALSE.toString().equalsIgnoreCase((String) v)
          && !"off".equalsIgnoreCase((String) v)
          && (Boolean.TRUE.toString().equalsIgnoreCase((String) v)
              || "on".equalsIgnoreCase((String) v));
    }
    return false;
  }

  String getHostFromServerUrl() throws SFException {
    URL url;
    try {
      if (!serverUrl.startsWith("http")) {
        url = new URL("https://" + serverUrl);
      } else {
        url = new URL(serverUrl);
      }
    } catch (MalformedURLException e) {
      throw new SFException(
          e, ErrorCode.INTERNAL_ERROR, "Invalid serverUrl for retrieving host name");
    }
    return url.getHost();
  }
}
