////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2023 Saxonica Limited
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at http://mozilla.org/MPL/2.0/.
// This Source Code Form is "Incompatible With Secondary Licenses", as defined by the Mozilla Public License, v. 2.0.
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

package net.sf.saxon.expr.elab;

import net.sf.saxon.expr.XPathContext;
import net.sf.saxon.om.Item;
import net.sf.saxon.om.Sequence;
import net.sf.saxon.trans.UncheckedXPathException;
import net.sf.saxon.trans.XPathException;
import net.sf.saxon.value.EmptySequence;

/**
 * A SequenceEvaluator for expressions that return a sequence of zero or one items. Unlike the
 * {@link SingleItemEvaluator}, it checks whether the underlying {@link ItemEvaluator} has
 * returned null, and if so, returns an empty sequence rather than a single item.
 */
public class OptionalItemEvaluator implements SequenceEvaluator {

    final ItemEvaluator evaluator;

    public OptionalItemEvaluator(ItemEvaluator eval) {
        this.evaluator = eval;
    }

    /**
     * Evaluate a construct to produce a value (which might be a lazily evaluated Sequence)
     *
     * @param context the evaluation context
     * @return a Sequence (not necessarily grounded). This will be an {@link EmptySequence}
     * if the underlying {@link ItemEvaluator} returns null.
     * @throws XPathException if a dynamic error occurs during the evaluation.
     */
    @Override
    public Sequence evaluate(XPathContext context) throws XPathException {
        try {
            Item result = evaluator.eval(context);
            return result == null ? EmptySequence.getInstance() : result;
        } catch (UncheckedXPathException e) {
            throw e.getXPathException();
        }
    }

}
