/*
 * Copyright (c) 2011-2019 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0, or the Apache License, Version 2.0
 * which is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
 */

package io.vertx.core.impl;

import io.netty.channel.EventLoop;
import io.vertx.core.*;
import io.vertx.core.Future;
import io.vertx.core.impl.logging.Logger;
import io.vertx.core.impl.logging.LoggerFactory;
import io.vertx.core.json.JsonObject;
import io.vertx.core.spi.metrics.PoolMetrics;
import io.vertx.core.spi.tracing.VertxTracer;

import java.util.concurrent.*;

/**
 * A base class for {@link Context} implementations.
 *
 * @author <a href="http://tfox.org">Tim Fox</a>
 * @author <a href="mailto:julien@julienviet.com">Julien Viet</a>
 */
public final class ContextImpl extends ContextBase implements ContextInternal {

  static <T> void setResultHandler(ContextInternal ctx, Future<T> fut, Handler<AsyncResult<T>> resultHandler) {
    if (resultHandler != null) {
      fut.onComplete(resultHandler);
    } else {
      fut.onFailure(ctx::reportException);
    }
  }

  private static final Logger log = LoggerFactory.getLogger(ContextImpl.class);

  private static final String DISABLE_TIMINGS_PROP_NAME = "vertx.disableContextTimings";
  static final boolean DISABLE_TIMINGS = Boolean.getBoolean(DISABLE_TIMINGS_PROP_NAME);

  private final ThreadingModel threadingModel;
  private final VertxImpl owner;
  private final JsonObject config;
  private final Deployment deployment;
  private final CloseFuture closeFuture;
  private final ClassLoader tccl;
  private final EventLoop eventLoop;
  private final EventExecutor executor;
  private ConcurrentMap<Object, Object> data;
  private volatile Handler<Throwable> exceptionHandler;
  final TaskQueue internalOrderedTasks;
  final WorkerPool internalWorkerPool;
  final WorkerPool workerPool;
  final WorkerTaskQueue executeBlockingTasks;

  public ContextImpl(VertxImpl vertx,
                     int localsLength,
                     ThreadingModel threadingModel,
                     EventLoop eventLoop,
                     EventExecutor executor,
                     WorkerPool internalWorkerPool,
                     WorkerPool workerPool,
                     Deployment deployment,
                     CloseFuture closeFuture,
                     ClassLoader tccl) {
    super(localsLength);
    this.threadingModel = threadingModel;
    this.deployment = deployment;
    this.config = deployment != null ? deployment.config() : new JsonObject();
    this.eventLoop = eventLoop;
    this.executor = executor;
    this.tccl = tccl;
    this.owner = vertx;
    this.workerPool = workerPool;
    this.closeFuture = closeFuture;
    this.internalWorkerPool = internalWorkerPool;
    this.executeBlockingTasks = new WorkerTaskQueue();
    this.internalOrderedTasks = new TaskQueue();
  }

  public Future<Void> close() {
    Future<Void> fut;
    if (closeFuture == owner.closeFuture()) {
      fut = Future.succeededFuture();
    } else {
      fut = closeFuture.close();
    }
    fut = fut.eventually(() -> Future.<Void>future(p -> executeBlockingTasks.shutdown(eventLoop, p)));
    if (executor instanceof WorkerExecutor) {
      WorkerExecutor workerExec = (WorkerExecutor) executor;
      fut = fut.eventually(() -> Future.<Void>future(p -> workerExec.taskQueue().shutdown(eventLoop, p)));
    }
    return fut;
  }

  public Deployment getDeployment() {
    return deployment;
  }

  @Override
  public CloseFuture closeFuture() {
    return closeFuture;
  }

  @Override
  public JsonObject config() {
    return config;
  }

  public EventLoop nettyEventLoop() {
    return eventLoop;
  }

  public VertxImpl owner() {
    return owner;
  }

  @Override
  public <T> Future<T> executeBlockingInternal(Handler<Promise<T>> action) {
    return executeBlocking(this, action, internalWorkerPool, internalOrderedTasks);
  }

  @Override
  public <T> Future<T> executeBlockingInternal(Callable<T> action) {
    return executeBlocking(this, action, internalWorkerPool, internalOrderedTasks);
  }

  @Override
  public <T> Future<T> executeBlockingInternal(Handler<Promise<T>> action, boolean ordered) {
    return executeBlocking(this, action, internalWorkerPool, ordered ? internalOrderedTasks : null);
  }

  @Override
  public <T> Future<T> executeBlockingInternal(Callable<T> action, boolean ordered) {
    return executeBlocking(this, action, internalWorkerPool, ordered ? internalOrderedTasks : null);
  }

  @Override
  public <T> Future<T> executeBlocking(Handler<Promise<T>> blockingCodeHandler, boolean ordered) {
    return executeBlocking(this, blockingCodeHandler, workerPool, ordered ? executeBlockingTasks : null);
  }

  @Override
  public <T> Future<T> executeBlocking(Callable<T> blockingCodeHandler, boolean ordered) {
    return executeBlocking(this, blockingCodeHandler, workerPool, ordered ? executeBlockingTasks : null);
  }

  @Override
  public EventExecutor executor() {
    return executor;
  }

  @Override
  public boolean isEventLoopContext() {
    return threadingModel == ThreadingModel.EVENT_LOOP;
  }

  @Override
  public boolean isWorkerContext() {
    return threadingModel == ThreadingModel.WORKER;
  }

  public ThreadingModel threadingModel() {
    return threadingModel;
  }

  @Override
  public boolean inThread() {
    return executor.inThread();
  }

  @Override
  public <T> Future<T> executeBlocking(Handler<Promise<T>> blockingCodeHandler, TaskQueue queue) {
    return executeBlocking(this, blockingCodeHandler, workerPool, queue);
  }

  @Override
  public <T> Future<T> executeBlocking(Callable<T> blockingCodeHandler, TaskQueue queue) {
    return executeBlocking(this, blockingCodeHandler, workerPool, queue);
  }

  static <T> Future<T> executeBlocking(ContextInternal context, Callable<T> blockingCodeHandler,
                                       WorkerPool workerPool, TaskQueue queue) {
    return internalExecuteBlocking(context, promise -> {
      T result;
      try {
        result = blockingCodeHandler.call();
      } catch (Throwable e) {
        promise.fail(e);
        return;
      }
      promise.complete(result);
    }, workerPool, queue);
  }

  static <T> Future<T> executeBlocking(ContextInternal context, Handler<Promise<T>> blockingCodeHandler,
                                       WorkerPool workerPool, TaskQueue queue) {
    return internalExecuteBlocking(context, promise -> {
      try {
        blockingCodeHandler.handle(promise);
      } catch (Throwable e) {
        promise.tryFail(e);
      }
    }, workerPool, queue);
  }

  private static <T> Future<T> internalExecuteBlocking(ContextInternal context, Handler<Promise<T>> blockingCodeHandler,
      WorkerPool workerPool, TaskQueue queue) {
    PoolMetrics metrics = workerPool.metrics();
    Object queueMetric = metrics != null ? metrics.submitted() : null;
    Promise<T> promise = context.promise();
    Future<T> fut = promise.future();
    WorkerTask task = new WorkerTask(metrics, queueMetric) {
      @Override
      protected void execute() {
        context.dispatch(promise, blockingCodeHandler);
      }
      @Override
      void reject() {
        if (metrics != null) {
          metrics.rejected(queueMetric);
        }
        promise.fail(new RejectedExecutionException());
      }
    };
    try {
      Executor exec = workerPool.executor();
      if (queue != null) {
        queue.execute(task, exec);
      } else {
        exec.execute(task);
      }
    } catch (RejectedExecutionException e) {
      // Pool is already shut down
      task.reject();
    }
    return fut;
  }

  @Override
  public VertxTracer tracer() {
    return owner.tracer();
  }

  @Override
  public ClassLoader classLoader() {
    return tccl;
  }

  @Override
  public WorkerPool workerPool() {
    return workerPool;
  }

  @Override
  public synchronized ConcurrentMap<Object, Object> contextData() {
    if (data == null) {
      data = new ConcurrentHashMap<>();
    }
    return data;
  }

  public void reportException(Throwable t) {
    Handler<Throwable> handler = exceptionHandler;
    if (handler == null) {
      handler = owner.exceptionHandler();
    }
    if (handler != null) {
      handler.handle(t);
    } else {
      log.error("Unhandled exception", t);
    }
  }

  @Override
  public Context exceptionHandler(Handler<Throwable> handler) {
    exceptionHandler = handler;
    return this;
  }

  @Override
  public Handler<Throwable> exceptionHandler() {
    return exceptionHandler;
  }

  protected void runOnContext(ContextInternal ctx, Handler<Void> action) {
    try {
      Executor exec = ctx.executor();
      exec.execute(() -> ctx.dispatch(action));
    } catch (RejectedExecutionException ignore) {
      // Pool is already shut down
    }
  }

  @Override
  public void execute(Runnable task) {
    execute(this, task);
  }

  @Override
  public final <T> void execute(T argument, Handler<T> task) {
    execute(this, argument, task);
  }

  protected void execute(ContextInternal ctx, Runnable task) {
    if (inThread()) {
      task.run();
    } else {
      executor.execute(task);
    }
  }

  /**
   * <ul>
   *   <li>When the current thread is event-loop thread of this context the implementation will execute the {@code task} directly</li>
   *   <li>When the current thread is a worker thread of this context the implementation will execute the {@code task} directly</li>
   *   <li>Otherwise the task will be scheduled on the context thread for execution</li>
   * </ul>
   */
  protected <T> void execute(ContextInternal ctx, T argument, Handler<T> task) {
    if (inThread()) {
      task.handle(argument);
    } else {
      executor.execute(() -> task.handle(argument));
    }
  }

  @Override
  public <T> void emit(T argument, Handler<T> task) {
    emit(this, argument, task);
  }

  protected <T> void emit(ContextInternal ctx, T argument, Handler<T> task) {
    if (inThread()) {
      ContextInternal prev = ctx.beginDispatch();
      try {
        task.handle(argument);
      } catch (Throwable t) {
        reportException(t);
      } finally {
        ctx.endDispatch(prev);
      }
    } else {
      executor.execute(() -> emit(ctx, argument, task));
    }
  }

  @Override
  public ContextInternal duplicate() {
    return new DuplicatedContext(this);
  }
}
