/*
 * Decompiled with CFR 0.152.
 */
package io.temporal.internal.testservice;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.Uninterruptibles;
import io.temporal.internal.testservice.LockHandle;
import io.temporal.internal.testservice.RequestContext;
import io.temporal.internal.testservice.SelfAdvancingTimer;
import io.temporal.workflow.Functions;
import java.io.Serializable;
import java.sql.Timestamp;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.PriorityQueue;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.LongSupplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class SelfAdvancingTimerImpl
implements SelfAdvancingTimer {
    private static final Logger log = LoggerFactory.getLogger(SelfAdvancingTimerImpl.class);
    private final Clock systemClock;
    private final LongSupplier clock = this::currentTimeMillis;
    private final Lock lock = new ReentrantLock();
    private final Condition condition = this.lock.newCondition();
    private final ThreadPoolExecutor executor = new ThreadPoolExecutor(5, 5, 1L, TimeUnit.SECONDS, new LinkedBlockingDeque<Runnable>(), r -> new Thread(r, "Timer task"));
    private long currentTimeMs;
    private int lockCount;
    private long systemTimeMsLastAdvancedWhileLocked = -1L;
    private boolean emptyQueue = true;
    private final PriorityQueue<TimerTask> tasks = new PriorityQueue<TimerTask>(Comparator.comparing(TimerTask::getExecutionTime));
    private final Thread timerPump = new Thread((Runnable)new TimerPump(), "SelfAdvancingTimer Pump");
    private LockHandle timeLockOnEmptyQueueHandle;
    private final LinkedList<LockEvent> lockEvents = new LinkedList();

    public SelfAdvancingTimerImpl(long initialTimeMs, Clock systemClock) {
        this.systemClock = systemClock;
        this.currentTimeMs = initialTimeMs == 0L ? systemClock.millis() : initialTimeMs;
        log.trace("Current time on start: {}", (Object)this.currentTimeMs);
        this.executor.setRejectedExecutionHandler(new ThreadPoolExecutor.CallerRunsPolicy());
        this.timeLockOnEmptyQueueHandle = this.lockTimeSkipping("SelfAdvancingTimerImpl constructor empty-queue");
        this.timerPump.start();
    }

    private void updateTimeLocked() {
        if (this.lockCount > 0) {
            if (this.systemTimeMsLastAdvancedWhileLocked < 0L) {
                throw new IllegalStateException("Invalid systemTimeLastLocked");
            }
            long systemTime = this.systemClock.millis();
            this.currentTimeMs += systemTime - this.systemTimeMsLastAdvancedWhileLocked;
            this.systemTimeMsLastAdvancedWhileLocked = systemTime;
        } else {
            TimerTask task = this.tasks.peek();
            if (task != null && !task.isCanceled() && task.getExecutionTime() > this.currentTimeMs) {
                this.currentTimeMs = task.getExecutionTime();
                log.trace("Jumping to the time of the next timer task: {}", (Object)this.currentTimeMs);
            }
        }
    }

    private long currentTimeMillis() {
        this.lock.lock();
        try {
            this.updateTimeLocked();
            long l = this.currentTimeMs;
            return l;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public Functions.Proc schedule(Duration delay, Runnable task) {
        return this.schedule(delay, task, "unknown");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Functions.Proc schedule(Duration delay, Runnable task, String taskInfo) {
        this.lock.lock();
        try {
            this.updateTimeLocked();
            long executionTime = delay.toMillis() + this.currentTimeMs;
            log.trace("Scheduling task <{}> in {} for timestamp {}", new Object[]{taskInfo, delay, this.currentTimeMs});
            Functions.Proc proc = this.scheduleAtLocked(executionTime, task, taskInfo);
            return proc;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Functions.Proc scheduleAt(Instant timestamp, Runnable task, String taskInfo) {
        this.lock.lock();
        try {
            this.updateTimeLocked();
            log.trace("Scheduling task <{}> for timestamp {}", (Object)taskInfo, (Object)timestamp.toEpochMilli());
            Functions.Proc proc = this.scheduleAtLocked(timestamp.toEpochMilli(), task, taskInfo);
            return proc;
        }
        finally {
            this.lock.unlock();
        }
    }

    private Functions.Proc scheduleAtLocked(long timestampMs, Runnable task, String taskInfo) {
        TimerTask timerTask = new TimerTask(timestampMs, task, taskInfo);
        Functions.Proc & Serializable cancellationHandle = timerTask::cancel;
        this.tasks.add(timerTask);
        if (this.tasks.size() == 1 && this.emptyQueue) {
            if (this.timeLockOnEmptyQueueHandle == null) {
                throw new IllegalStateException("SelfAdvancingTimerImpl should take a lock and get a handle when queue is empty, but handle is null");
            }
            this.timeLockOnEmptyQueueHandle.unlock("SelfAdvancingTimerImpl schedule non-empty-queue, task: " + taskInfo);
            this.timeLockOnEmptyQueueHandle = null;
            this.emptyQueue = false;
        }
        this.condition.signal();
        return cancellationHandle;
    }

    @Override
    public LongSupplier getClock() {
        return this.clock;
    }

    @Override
    public LockHandle lockTimeSkipping(String caller) {
        this.lock.lock();
        try {
            LockHandle lockHandle = this.lockTimeSkippingLocked(caller);
            return lockHandle;
        }
        finally {
            this.lock.unlock();
        }
    }

    private LockHandle lockTimeSkippingLocked(String caller) {
        if (this.lockCount++ == 0) {
            this.systemTimeMsLastAdvancedWhileLocked = this.systemClock.millis();
        }
        log.trace("[LOCK] <{}>, new lock counter value: {}", (Object)caller, (Object)this.lockCount);
        LockEvent event = new LockEvent(caller, LockEventType.LOCK);
        this.lockEvents.add(event);
        return new TimerLockHandle(event);
    }

    @Override
    public void unlockTimeSkipping(String caller) {
        this.lock.lock();
        try {
            this.unlockTimeSkippingLocked(caller);
            this.condition.signal();
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public Instant skip(Duration duration) {
        this.lock.lock();
        try {
            this.currentTimeMs += duration.toMillis();
            log.trace("Skipping time by {} to: {}", (Object)duration, (Object)this.currentTimeMs);
            this.condition.signal();
            Instant instant = Instant.ofEpochMilli(this.currentTimeMs);
            return instant;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void skipTo(Instant timestamp) {
        this.lock.lock();
        try {
            if (timestamp.toEpochMilli() > this.currentTimeMs) {
                log.trace("Skipping time from {} to: {}", (Object)this.currentTimeMs, (Object)timestamp.toEpochMilli());
                this.currentTimeMs = timestamp.toEpochMilli();
            } else {
                log.trace("Time Skipping into past with timestamp {} was ignored because the current timestamp is {}", (Object)timestamp.toEpochMilli(), (Object)this.currentTimeMs);
            }
            this.condition.signal();
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void updateLocks(List<RequestContext.TimerLockChange> updates) {
        this.lock.lock();
        try {
            block7: for (RequestContext.TimerLockChange update : updates) {
                switch (update.getChange()) {
                    case 1: {
                        this.lockTimeSkippingLocked(update.getCaller());
                        continue block7;
                    }
                    case -1: {
                        this.unlockTimeSkippingLocked(update.getCaller());
                        continue block7;
                    }
                }
                throw new IllegalStateException("TimerLockChange for not +1/-1 is not implemented");
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void getDiagnostics(StringBuilder result) {
        result.append("Self Advancing Timer Lock Events:\n");
        this.lock.lock();
        try {
            int lockCount = 0;
            for (LockEvent event : this.lockEvents) {
                lockCount = event.lockType == LockEventType.LOCK ? ++lockCount : --lockCount;
                String indent = new String(new char[lockCount * 2]).replace("\u0000", " ");
                result.append(new Timestamp(event.timestamp)).append("\t").append((Object)event.lockType).append("\t").append(indent).append(lockCount).append("\t").append(event.caller).append("\n");
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void shutdown() {
        this.executor.shutdown();
        this.timerPump.interrupt();
        Uninterruptibles.joinUninterruptibly((Thread)this.timerPump);
    }

    private void unlockTimeSkippingLocked(String caller) {
        this.unlockTimeSkippingLockedInternal(caller);
        this.lockEvents.add(new LockEvent(caller, LockEventType.UNLOCK));
    }

    private void unlockTimeSkippingLockedInternal(String caller) {
        if (this.lockCount == 0) {
            throw new IllegalStateException("Unbalanced lock and unlock calls: \n" + this.getDiagnostics());
        }
        --this.lockCount;
        if (caller == null) {
            log.trace("---");
        }
        log.trace("[UNLOCK] <{}>, new lock counter value: {}", (Object)caller, (Object)this.lockCount);
        if (this.lockCount == 0) {
            this.systemTimeMsLastAdvancedWhileLocked = -1L;
        }
    }

    private String getDiagnostics() {
        StringBuilder result = new StringBuilder();
        this.getDiagnostics(result);
        return result.toString();
    }

    @VisibleForTesting
    void pump() {
        this.lock.lock();
        try {
            this.condition.signal();
        }
        finally {
            this.lock.unlock();
        }
    }

    private class TimerLockHandle
    implements LockHandle {
        private final LockEvent event;

        public TimerLockHandle(LockEvent event) {
            this.event = event;
        }

        @Override
        public void unlock() {
            SelfAdvancingTimerImpl.this.lock.lock();
            try {
                this.unlockFromHandleLocked(this.event.caller + "[\u238c]");
                SelfAdvancingTimerImpl.this.condition.signal();
            }
            finally {
                SelfAdvancingTimerImpl.this.lock.unlock();
            }
        }

        @Override
        public void unlock(String caller) {
            SelfAdvancingTimerImpl.this.lock.lock();
            try {
                this.unlockFromHandleLocked("{-" + caller + "} {+" + this.event.caller + "}");
                SelfAdvancingTimerImpl.this.condition.signal();
            }
            finally {
                SelfAdvancingTimerImpl.this.lock.unlock();
            }
        }

        private void unlockFromHandleLocked(String callerInfo) {
            boolean removed = SelfAdvancingTimerImpl.this.lockEvents.remove(this.event);
            if (!removed) {
                throw new IllegalStateException("Unbalanced lock and unlock calls");
            }
            SelfAdvancingTimerImpl.this.unlockTimeSkippingLockedInternal(callerInfo);
        }
    }

    private static enum LockEventType {
        LOCK,
        UNLOCK;


        public String toString() {
            return this == LOCK ? "L" : "U";
        }
    }

    private class LockEvent {
        final String caller;
        final LockEventType lockType;
        final long timestamp;

        public LockEvent(String caller, LockEventType lockType) {
            this.caller = caller;
            this.lockType = lockType;
            this.timestamp = SelfAdvancingTimerImpl.this.systemClock.millis();
        }
    }

    private class TimerPump
    implements Runnable {
        private TimerPump() {
        }

        @Override
        public void run() {
            SelfAdvancingTimerImpl.this.lock.lock();
            try {
                this.runLocked();
            }
            catch (RuntimeException e) {
                log.error("Timer pump failed", (Throwable)e);
            }
            finally {
                SelfAdvancingTimerImpl.this.lock.unlock();
            }
        }

        private void runLocked() {
            while (!Thread.currentThread().isInterrupted()) {
                long timeToAwait;
                TimerTask peekedTask;
                SelfAdvancingTimerImpl.this.updateTimeLocked();
                if (!SelfAdvancingTimerImpl.this.emptyQueue && SelfAdvancingTimerImpl.this.tasks.isEmpty()) {
                    if (SelfAdvancingTimerImpl.this.timeLockOnEmptyQueueHandle != null) {
                        throw new IllegalStateException("SelfAdvancingTimerImpl should have no taken time lock when queue is not empty, but handle is not null");
                    }
                    SelfAdvancingTimerImpl.this.timeLockOnEmptyQueueHandle = SelfAdvancingTimerImpl.this.lockTimeSkippingLocked("SelfAdvancingTimerImpl runLocked empty-queue");
                    SelfAdvancingTimerImpl.this.emptyQueue = true;
                }
                if ((peekedTask = (TimerTask)SelfAdvancingTimerImpl.this.tasks.peek()) != null) {
                    log.trace("peekedTask=" + peekedTask.getTaskInfo() + ", executionTime=" + peekedTask.getExecutionTime() + ", canceled=" + peekedTask.isCanceled());
                }
                if (peekedTask != null && (peekedTask.getExecutionTime() <= SelfAdvancingTimerImpl.this.currentTimeMs || peekedTask.isCanceled())) {
                    try {
                        TimerTask polledTask = (TimerTask)SelfAdvancingTimerImpl.this.tasks.poll();
                        if (polledTask.isCanceled()) {
                            log.trace("Removed canceled task from the task queue: {}", (Object)polledTask.getTaskInfo());
                            continue;
                        }
                        LockHandle lockHandle = SelfAdvancingTimerImpl.this.lockTimeSkippingLocked("[TimerPump] runnable " + peekedTask.getTaskInfo());
                        log.trace("running task=" + polledTask.getTaskInfo() + ", executionTime=" + polledTask.getExecutionTime());
                        Runnable runnable = polledTask.getRunnable();
                        SelfAdvancingTimerImpl.this.executor.execute(() -> {
                            try {
                                runnable.run();
                            }
                            catch (Throwable e) {
                                log.error("Unexpected failure in timer callback", e);
                            }
                            finally {
                                lockHandle.unlock();
                            }
                        });
                    }
                    catch (RuntimeException e) {
                        log.error("Timer task failure", (Throwable)e);
                    }
                    continue;
                }
                if (peekedTask != null) {
                    timeToAwait = peekedTask.getExecutionTime() - SelfAdvancingTimerImpl.this.currentTimeMs;
                    if (log.isTraceEnabled()) {
                        log.trace("Waiting for {} with outstanding tasks:", (Object)Duration.ofMillis(timeToAwait));
                        for (TimerTask task : SelfAdvancingTimerImpl.this.tasks) {
                            log.trace("    +{} <{}>, cancelled: {}", new Object[]{Duration.ofMillis(task.executionTime - SelfAdvancingTimerImpl.this.currentTimeMs), task.taskInfo, task.canceled});
                        }
                    }
                } else {
                    timeToAwait = Long.MAX_VALUE;
                }
                try {
                    SelfAdvancingTimerImpl.this.condition.await(timeToAwait, TimeUnit.MILLISECONDS);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    break;
                }
            }
        }
    }

    private static class TimerTask {
        private final long executionTime;
        private final Runnable runnable;
        private final String taskInfo;
        private boolean canceled;

        TimerTask(long executionTime, Runnable runnable, String taskInfo) {
            this.executionTime = executionTime;
            this.runnable = runnable;
            this.taskInfo = taskInfo;
        }

        long getExecutionTime() {
            return this.executionTime;
        }

        public Runnable getRunnable() {
            return this.runnable;
        }

        String getTaskInfo() {
            return this.taskInfo;
        }

        public String toString() {
            return "TimerTask{executionTime=" + this.executionTime + '}';
        }

        public boolean isCanceled() {
            return this.canceled;
        }

        public void cancel() {
            this.canceled = true;
        }
    }
}

